
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 1994                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

(in-package "USER")

#+lucid
(defun expand ()
  (lcl:change-memory-management :growth-limit 2048
				:growth-rate 128
				:reserved-growth-rate 8
				:expand-reserved 8
				:expand 256)
  (lcl:egc-options :level-sizes '(24 48 96)) ;; 1.5+3+6=10.5
  (room t))

#+lucid
(defun large-expand ()
  (room t)
  (lcl:change-memory-management :growth-limit 2048
				:growth-rate 256
				:reserved-growth-rate 16
				:expand-reserved 16
				:expand 512)
  (lcl:egc-options :level-sizes '(48 96 192)) ;; 3+6+12=21
  ;;(lcl:egc-options :level-sizes '(16 32 64))
  (room t))

#+lucid
(defun very-large-expand ()
  (room t)
  (lcl:change-memory-management :growth-limit 4096
			    :growth-rate 256
			    :reserved-growth-rate 32
			    :expand-reserved 128
			    :expand 1048)
  (lcl:egc-options :level-sizes '(32 64 128))
  (room t))

#+lucid
(defun half-gig-expand () (very-large-expand))

#+lucid
(defun huge-expand ()
  (room t)
  (lcl:change-memory-management :growth-limit 6144
				:growth-rate 512
				:reserved-growth-rate 32
				:expand-reserved 128
				:expand 2096)
  (lcl:egc-options :level-sizes '(32 64 128))
  (room t))

#+cmu
(progn 
  (defun expand-space (o &optional (n 64))
    (let ((onemeg (* 1024 1024)))
      (setf extensions:*bytes-consed-between-gcs* (* n onemeg))))

  (export '(expand-space))
  )

#+allegro
(progn
  (setf excl:*global-gc-behavior* :auto-and-warn)
  
  (defun old-expand ()
    ;;(setq excl:*global-gc-behavior* nil)
    (setf (sys:gsgc-switch :verbose) t
	  (sys:gsgc-switch :stats) t
	  (sys:gsgc-switch :print) t)
    (let ((factor 4))
      (setf excl:*tenured-bytes-limit* (* factor 8 131072))
      (setf (sys:gsgc-parameter :quantum) (* factor 32)
	    (sys:gsgc-parameter :free-bytes-new-other) (* 8 factor 131072)
	    (sys:gsgc-parameter :free-bytes-new-pages) (* 8 factor 131072)
	    ;;(sys:gsgc-parameter :expansion-free-percent-old) 35
	    )))

  (defun old-large-expand ()
    ;;(setq excl:*global-gc-behavior* nil)
    (setf (sys:gsgc-switch :verbose) t
	  (sys:gsgc-switch :stats) t
	  (sys:gsgc-switch :print) t)
    (let ((factor 8))
      (setf (sys:gsgc-parameter :quantum) (* factor 32)
	    (sys:gsgc-parameter :free-bytes-new-other) (* 8 factor 131072)
	    (sys:gsgc-parameter :free-bytes-new-pages) (* 8 factor 131072)
	    (sys:gsgc-parameter :expansion-free-percent-old) 100
	    )))

  (defun expand-space (o n)
    (let ((onemeg (* 1024 1024)))
      (let ((newmegs n)
	    (oldmegs o))

	(cond
	  ((null oldmegs)
	   (sys:resize-areas :new (* newmegs onemeg)
			     :sift-old-areas t
			     :pack-heap t))
	  ((null newmegs)
	   (sys:resize-areas :old (* oldmegs onemeg)
			     :sift-old-areas t
			     :pack-heap t))
	  (t 
	   (sys:resize-areas :old (* oldmegs onemeg)
			     :new (* newmegs onemeg)
			     :sift-old-areas t
			     :pack-heap t)))

	;; slow growth : more gc but might prevent blow up due to intemittent large
	;; space requirements.
	(when oldmegs
	  (setf excl:*tenured-bytes-limit* (* (round (/ oldmegs 2)) onemeg)
		(sys:gsgc-parameter :gc-old-before-expand) nil))

	(room))))

  (defun expand-aux (o n)
    (let ((onemeg (* 1024 1024)))
      (let ((oldmegs o)
	    (newmegs n))

	(sys:resize-areas :new (* newmegs onemeg)
			  :old (* oldmegs onemeg)
			  :sift-old-areas t
			  :pack-heap t)

	;;(setq excl:*global-gc-behavior* :warn)
	(setq excl:*global-gc-behavior* :auto-and-warn)

	(setf (sys:gsgc-switch :verbose) t
	      (sys:gsgc-switch :stats) t
	      (sys:gsgc-switch :print) t)

	(setf
	 (sys:gsgc-parameter :free-bytes-new-other) (* (round (/ newmegs 8)) onemeg)
	 (sys:gsgc-parameter :free-bytes-new-pages) 0
	 (sys:gsgc-parameter :free-percent-new) 0

	 ;; slow growth : more gc but might prevent blow up due to intemittent large
	 ;; space requirements.
	 excl:*tenured-bytes-limit* (* (round (/ oldmegs 8)) onemeg)
	 (sys:gsgc-parameter :expansion-free-percent-old) 16	 
	 (sys:gsgc-parameter :expansion-free-percent-new) 8

	 (sys:gsgc-parameter :gc-old-before-expand) t
	 ))))
  
  ;;(defun three-quarters-gig-expand () (expand-aux 200 200))

  (defun half-gig-expand () (expand-aux 200  96))

  (defun small-expand ()	(expand-aux  64  48))
  (defun expand ()		(expand-aux 128  64))
  (defun large-expand ()	(expand-aux 192  64))
  (defun very-large-expand ()	(expand-aux 200  96))
  (defun huge-expand ()		(expand-aux 200 200))

  (export
   '(expand-space small-expand expand large-expand very-large-expand huge-expand))
  )

#-(or cmu allegro)
(defun expand-space (&rest r) (declare (ignore rest)) (values))




#+cmu
(progn  
  (defun expand ()
    (let ((onemeg (* 1024 1024)))
      (setf extensions:*bytes-consed-between-gcs* (* 12 onemeg))))
   
  (defun large-expand ()
    (let ((onemeg (* 1024 1024)))
       (setf extensions:*bytes-consed-between-gcs* (* 24 onemeg))))
	
  (defun very-large-expand ()
    (let ((onemeg (* 1024 1024)))
      (setf extensions:*bytes-consed-between-gcs* (* 48 onemeg))))

  ;; shoot for .5 G virtual.
  (defun half-gig-expand () (very-large-expand))

  (defun huge-expand ()
    (let ((onemeg (* 1024 1024)))
     (setf extensions:*bytes-consed-between-gcs* (* 64 onemeg))))
)

#-(or allegro lucid cmu)
(progn
  (defun expand ())
  (defun large-expand ())
  (defun very-large-expand ())
  (defun half-gig-expand ())
  (defun huge-expand ()))


(defun ref-expand () (very-large-expand))
(defun edd-expand () (huge-expand))
(defun lib-expand () (huge-expand))
