
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	who-calls : the ML compiler maintains a dynamic xref facility.
;;;;
;;;;	ML identifiers are stamped at parse time. The purpose is to allow
;;;;	re-compile of a function such that old references still call previous
;;;;	version. 
;;;;
;;;;	An ml-identifier consists of three parts:
;;;;
;;;;	<mlid> 		: <id>%<stamp>%<count>
;;;;
;;;;	id : the user assigned id.
;;;;	stamp : a random 4 digit decimal number assigned to ml evaluator at
;;;;	  initialization and unchanged for duration of run. The number
;;;;	  is not actually random, it is the universal time (# of
;;;;	  seconds since midnight 1/1/1900 GMT) modulus 10000.
;;;;	count : initialized to a random number (universal time modulus 128),
;;;;	  and then incremented for each id stamped.
;;;;
;;;;	When loading a previously compiled file, the compile time stamp is used.
;;;;	The current count is not incremented.
;;;;	Thus there is a small, but not impossible, chance of ml-id collisions.
;;;;	Such collisions are not identified or reported. 
;;;;

  
;;;;	
;;;;	need finer grained incremental access to calls-db and
;;;;	desire object_id tracking
;;;;
;;;;	
;;;;	compilex : term -> xref
;;;;	 
;;;;	
;;;;	id : name # dependency   type? refvar-p?
;;;;	
;;;;	recall:
;;;;	<dependency>		: dependency[<stamp{data}>, <stamp{oc}, <oid>]
;;;;	
;;;;	
;;;;	xref : rely on code table to know object or not.
;;;;	
;;;;	<ml-caller>	: (<tok{caller}> . <tok{mltype}> sexpr) 
;;;;	
;;;;	<ml-expr-xref>	: (BOOL{refvar-p} . <ml-caller> list) . <tok{called}> list
;;;;
;;;;	<ml-xref>	: <ml-expr-xref> list
;;;;	
;;;;	
;;;;	with-ml-xref (body)
;;;;	
;;;;	get-ml-xref
;;;;	
;;;;	
;;;;	

(defvar *appns*)
(defvar *appns-p* nil)

(defun note-appn (appn)
  ;;(setf -appn appn) (break (format-string "na ~a" (car (cadr appn))))
  (unless *appns-p*
    (break))
  ;;(setf -appn appn) (break "note-appn")
  (when (eql 'mk-var (car appn))
    (let ((appo (let ((id (car (cadr appn))))
		  ;;(unless (member id (mapcar #'car (cdr %thisdec)))) ; recursive
		  (let ((mldef (find-mldef id)))
		    ;;(unless mldef (setf -appn appn -myid id) (break "note-appn"))
		    (when mldef
		      (cons id mldef))))))
		 
	     ;;(or (eql *ml-runtime-package* (symbol-package (cadr (cdadr appn))))
	     ;; (eql `prim-function (car (cdadr appn))))
	     

      ;;(setf -appn appn) (break "na")
      (when appo
	;;(setf -appn appn -appo appo) (break "note-appn")
		    
	(when *appns-p*
	  (pushnew appo *appns* :test #'equal))))))

(defun force-ml-xref (sid obid)
  (let ((id (intern-system (string sid))))
    (let ((appo (cons id (make-ml-definition :id id :source obid))))
      ;;(setf -appo appo) (break "fmx")
      (when *appns-p*
	(pushnew appo *appns* :test #'(lambda (a b)
					(and (eql (car a) (car b))
					     (equal-oids-p (cdr a) (cdr b)))))
	t))))
  


(defvar *ml-xref*)

(defun mlxref (refp callers called) (cons (cons refp callers) called))
(defun mlxref-caller (ident type) (cons ident type))
(defun mlxref-called (ident dependency) (cons ident dependency))

(defmacro callers-of-mlxref (x) `(cdar ,x))
(defmacro called-of-mlxref (x) `(cdr ,x))
(defmacro refp-of-mlxref (x) `(caar ,x))

(defmacro id-of-mlxref-called (x) `(car ,x))
(defmacro dependency-of-mlxref-called (x) `(cdr ,x))

(defmacro id-of-mlxref-caller (x) `(car ,x))
(defmacro type-of-mlxref-caller (x) `(cdr ,x))

(defmacro with-ml-xref (&body body)
  `(let ((*ml-xref* nil)
	 )
    ,@body))

(defun update-ml-xref (e)
  ;;(setf a e) (break)
  (when (boundp `*ml-xref*) (push e *ml-xref*)))

(defun get-ml-xref ()
  (unless (boundp `*ml-xref*)
    (raise-error (error-message '(ml xref get not))))
  ;;(break "gmx")
  (let ((r (nreverse *ml-xref*)))
    (setf *ml-xref* nil)
    (values r)))

(defun mltype-to-sexpr (mltype)
  ;;(break)
  (let* ((deftype (getdeftype mltype %temt)))
    (if deftype
	(car deftype)
	(if (atom mltype)
	    (case mltype
	      (mk-listyp		'|list|)
	      (mk-prodtyp		'|product|)
	      (mk-sumtyp		'|union|)
	      (mk-funtyp		'|function|)
	      (otherwise	mltype))
	    (or (mkid-to-abstract-type-name (car mltype))
		(case (car mltype)
		  (mk-nulltyp		'|void|)
		  (mk-inttyp		'|int|)
		  (mk-booltyp		'|bool|)
		  (mk-toktyp		'|tok|)
		  (mk-stringtyp		'|string|)
		  (otherwise	(mapcar #'mltype-to-sexpr mltype))))))))


;; : <bool{refp} . ((<tok{id}> . mltype) list)
(defun getvalues ()
  (cond
    ((isdefty) nil)
    ((isdecl)
     (cons (eql 'mk-letref (car %thisdec))
	   (mapcar #'(lambda (decl) (mlxref-caller (car decl) (mltype-to-sexpr (cdr decl))))
		   (cdr %thisdec))))
    (t (cons nil (list (mlxref-caller nil (mltype-to-sexpr %ty)))))))


(defun note-calls (callers called)

  ;;(if *tranpt-xref-p*
  ;;(note-tranpt-xref-p xref))
  ;;(break "nc")

  (update-ml-xref (cons callers called)))


(defmacro with-appns (&body body)
  `(let ((*appns* nil)
	 (*appns-p* t))
    (multiple-value-prog1 (progn ,@body)
      (when (boundp `*ml-xref*)
	;;(break "with-appns")
	(note-calls (getvalues) *appns*)))))



    
#|  
;;;;	At translation time the calls made by a function def or
;;;;	global reference variable assignment will be recorded.
;;;;
;;;;	<calls-db>		: [<calls-who-table>
;;;;				   <who-calls-table>
;;;;				   <mlid-stamp-history>]
;;;;
;;;;	<who-calls-table>	: <who-calls-key> <who-calls-entry> hash table
;;;;
;;;;	<who-calls-key>		: <mlid{called}>
;;;;	<who-calls-entry>	: <mlid{caller}> list
;;;;
;;;;	<calls-who-table>	: <calls-who-key> <calls-who-entry> hash table
;;;;
;;;;	<calls-who-key>		: <mlid{caller}>
;;;;	<calls-who-entry>	: <mlid{called}> list
;;;;
;;;;	<mlid-stamp-history>	: <mlid> <stamp-history> hash table
;;;;
;;;;	<stamp-history>		: (<stamped-mlid> . <bool{ref-variable-p}>) list
;;;;
;;;;
;;;;	note-calls (<mlid{caller}> <id{caller}> <mlid{called}> list)
;;;;	 * updates <calls-db>
;;;;	 * Duplicate entries generated by loading the same compiled file
;;;;	 *   mutiple times are discarded.
;;;;
;;;;	During translation the function applications calls are noted:
;;;;
;;;;	note-appn	: <mlid{called}>
;;;;
;;;;
;;;;	From the who calls table we can produce an alist:
;;;;
;;;;	<who-calls-alist> 	: (<mlid{called}> . <mlid{caller}>) list
;;;;
;;;; 	The alist is not maintained incrementally, it is produced from the hash
;;;;	table upon demand. Once produced, it is saved until hash table is updated.
;;;;
;;;;
;;;;
;;;;	who_calls_alist : unit -> (token # (token list)) list
;;;;	calls_who_alist : unit -> (token # (token list)) list
;;;;	 * returned tokens are all stamped ml-ids.
;;;;
;;;;	who_calls : token -> token list
;;;;	calls_who : token -> token list
;;;;	 * arg should be stamped mlid. List of stamped ml-ids returned.
;;;;
;;;;
;;;;	time_stamp : tok -> tok
;;;;	 * arg should be unstamped mlid.
;;;;
;;;;	untime_stamp : tok -> tok
;;;;	 * arg should be stamped mlid.
;;;;
;;;;	** Note that built-in identifiers, eg +, and primitive identifiers, ie who_calls,
;;;;	**  are not stamped. Both preceding functions will return input arg in such cases. 
;;;;
;;;;	all_time_stamps : tok -> (tok list)
;;;;	 * arg should be unstamped mlid
;;;;	 * The same token may occur in the returned list multiple times if a compiled
;;;;	 *   file was loaded multiple times.
;;;;
;;;;	reference_variable_p : tok -> bool
;;;;	 * arg should be stamped mlid.
;;;;

(defvar *tranpt-xref* )
(defvar *tranpt-xref-p* nil)
(defvar *tranpt-xref-stream* nil)

(defun note-tranpt-xref-p (xref)
  (when *tranpt-xref-p*
    (push xref *tranpt-xref*)))

(defun generate-tranpt-update ()
  (prog1 `(progn ,@(mapcar #'(lambda(xref)
			       `(update-ml-xref (list* ',(car xref) ',(cadr xref) ',(caddr xref) ',(cdddr xref)))
			       ;;`(update-ml-xref ',(car xref) ',(cadr xref) ',(caddr xref) ',(cdddr xref))
			       )
		    *tranpt-xref*))
    (setf *tranpt-xref* nil)))

(defmacro with-tranpt-xref ((stream) &body body)
  `(let ((*tranpt-xref-p* t)
	 (*tranpt-xref* nil)
	 (*tranpt-xref-stream* ,stream))
    (prog1 (progn ,@body)
      ;;(setf a (generate-tranpt-update)) (break)
      (flush-tranpt-xref)
      )))

(defun flush-tranpt-xref ()
  (when *tranpt-xref-p*
    (terpri *tranpt-xref-stream*)
    (write (generate-tranpt-update)
	   :stream *tranpt-xref-stream*
	   :gensym nil
	   :pretty nil
	   :escape t
	   :level nil
	   :length nil)))


(defvar *calls-who-table* (make-hash-table)) 	;; no dups
(defvar *calls-who-alist* nil)	;; no dups

(defvar *mlid-stamp-history* (make-hash-table))

(defvar *who-calls-table* (make-hash-table))
(defvar *who-calls-alist* nil)

(defun calls-who-alist ()
  (or *calls-who-alist*
      (progn
	(format t "updating calls who alist: may take a minute.~%")
	(setf *calls-who-alist*
	      (sort (let ((list nil)
			  (i 0))
		      (maphash #'(lambda (caller called)
				   (incf i)
				   (when (> i 1000)
				     (format t ".")
				     (setf i 0))
				   (push (cons caller called) list))
			       *calls-who-table*)
		      (terpri)
		      list)
		    #'(lambda (a b) (string-lessp (string a) (string b)))
		    :key #'car)))))

(defun who-calls-alist ()
  (or *who-calls-alist*
      (progn
	(format t "updating who calls alist: may take a minute.~%")
	(setf *who-calls-alist*
	      (sort (let ((list nil)
			  (i 0))
		      (maphash #'(lambda (called callers)
				   (incf i)
				   (when (> i 1000)
				     (format t ".")
				     (setf i 0))
				   (push (cons called callers) list))
			       *who-calls-table*)
		      (terpri)
		      list)
		    #'(lambda (a b) (string-lessp (string a) (string b)))
		    :key #'car)))))
				    

(defun incrementally-update-who-calls (caller called-list)
  (dolist (called called-list)
    (setf (gethash called *who-calls-table*)
	  (cons caller (gethash called *who-calls-table*)))
    (setf *who-calls-alist* nil)))

(defun update-calls-db (id history called)
  (let ((caller (car history)))
    (let ((p (gethash caller *calls-who-table*)))
    
      (setf (gethash id *mlid-stamp-history*)
	    (cons history
		  (gethash id *mlid-stamp-history*)))
      (unless p
	(setf (gethash caller *calls-who-table*) called)
	(setf *calls-who-alist* nil)
	(incrementally-update-who-calls caller called)))))
  
(defunml (|who_calls_alist| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (void -> ((tok |#| (tok list)) list))

  (who-calls-alist))


(defunml (|who_calls| (f))
  (tok -> (tok list))

  (let ((f (intern (string f) *ml-runtime-package*)))
    (let ((callers (gethash f *who-calls-table* t)))
      (if (eql callers t)
	  (raise-error (error-message '(id calls-who)))
	  callers))))


;;;;	calls_alist : unit -> (token # (token list)) list
(defunml (|calls_who_alist| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (void -> ((tok |#| (tok list)) list))

  (calls-who-alist))

(defunml (|calls_who| (f))
  (tok -> (tok list))

  (let ((f (intern (string f) *ml-runtime-package*)))
    (let ((called (gethash f *calls-who-table* t)))
      (if (eql called t)
	  (raise-error (error-message '(calls-who id)))
	  called))))


(defun time-stamp (id)
  (caar (gethash id *mlid-stamp-history*)))

(defun time-stamps (id)
  (mapcar #'car (gethash id *mlid-stamp-history*)))

(defun untime-stamp (id)
  (let ((mlid (intern (string id) *ml-runtime-package*)))
    (maphash #'(lambda (k v)
		 (when (member mlid v :key #'car)
		   (return-from untime-stamp k)))
	     *mlid-stamp-history*)
    nil))
  
(defunml (|time_stamp| (id) :error-wrap-p nil)
  (tok -> tok)

  (or (time-stamp id)
      id))

(defunml (|untime_stamp| (id) :error-wrap-p nil)
  (tok -> tok)

  (or (untime-stamp id)
      id))

(defunml (|all_time_stamps| (id) :error-wrap-p nil)
  (tok -> (tok list))

  (time-stamps id))

|#



;;;;
;;;;	There is a facility for displaying backtrace after a failure.
;;;;	
;;;;	There is some overhead 
;;;;
;;;;	show_ml_backtrace_p : unit -> bool
;;;;	set_ml_backtrace : bool -> unit
;;;;
;;;;	backtrace : unit -> tok list
;;;;
;;;;	backtrace returns the list of those functions compiled with the
;;;;	bactrace bit true which were failed out of by the last failure
;;;;	executed which was compiled with the backtrace bit true.
;;;;
;;;;	There is some runtime overhead in functions compiled with
;;;;	the backtrace bit true.
;;;;

(defvar *ml-backtrace* nil)
(defvar *ml-backtrace-p* nil)

(defunml (|show_ml_backtrace_p| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (void -> bool)

  *ml-backtrace-p*)


(defunml (|set_ml_backtrace| (bool) :error-wrap-p nil)
  (bool -> void)

  (setf *ml-backtrace-p* bool)
  nil)

(defun defun-wrap-backtrace-protect (def)
  (if *ml-backtrace-p*
      `(defun ,(car def) ,(cadr def)
	(let ((unwind-p t))
	  (unwind-protect (prog1 (progn ,(caddr def)) (setf unwind-p nil))
	    (when (and *ml-backtrace-p* unwind-p) (push ',(car def) *ml-backtrace*)))))
      `(defun ,(car def) ,(cadr def) ,(caddr def))))



(defunml (|backtrace| (unit) :declare ((declare (ignore unit))) :error-wrap-p nil)
  (void -> (tok list))

  *ml-backtrace*)
