
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


(defvar *null-directive*
  (let ((unit-unit-closure (make-closure #'(lambda (unit)
					     (declare (ignore unit))
					     nil)
					 1)))
    (make-closure #'(lambda (unit)
		      (declare (ignore unit))
		      (cons unit-unit-closure
			    unit-unit-closure))
		  1)))

(defun oc (oid) (objc-of-library-object (library-lookup oid)))

(defunml (|lib_object_contents| (oid) :direct t)
    (object_id -> object_contents)

  (push-io-history (ioid-term oid) 'lookup)

  ;;(format t "~a~%" (tags-of-object-address oa))
  (objc-of-library-object (library-lookup oid)))

(defunml (|lib_object_contents_ephemeral| (oid) :direct t)
    (object_id -> object_contents)

  (push-io-history (ioid-term oid) 'lookup-ephemeral)
  ;;(format t "~a~%" (tags-of-object-address oa))
  (ephemeral-objc-of-library-object (library-lookup oid)))


;;;;	
;;;;	(define-primitive |!definition_update| ((natural . sequence) (oid . oid)) (update))
;;;;	(define-primitive |!definition_activate| ((natural . sequence) (oid . oid)) (update))
;;;;	(define-primitive |!definition_deactivate| ((natural . sequence) (oid . oid)))
;;;;	
;;;;	<update>	: !activate()
;;;;			| !deactivate()
;;;;	
;;;;	but isn't insert just a diff type of update so why the wrapper (ie msg itself implies update).
;;;;	why not include oid parm in insert msg. otoh why dup since it's in def.


(defun map-update-to-broadcast-opid (update)
  (case update
    (insert	*idefinition-insert*)
    (delete	*idefinition-delete*)

    (activate	*idefinition-activate*)
    (deactivate	*idefinition-deactivate*)

    (allow	*idefinition-allow-collection*)
    (disallow	*idefinition-disallow-collection*)

    (undo	*iundo*)
    (commit	*icommit*)

    (otherwise (raise-error (error-message '(broadcast opid))))))


(defunml (|create_object_id| (unit) :declare ((declare (ignore unit))))
    (unit -> object_id)

  (new-object-id))


(defunml (|lib_bound_p| (oid))
    (object_id -> bool)

  (library-oid-bound-p oid))

(defun bind-directive (oid oc &optional oldoc)
  (io-echo "b")
  (set-objc-history oldoc oc)
  (let ((dterm (idefinition-term (let ((stamp (stamp-to-term (stamp-of-data oc))))
				   (idependency-term oid stamp (ivoid-term)))
				 (persist-data oc nil nil))))

    (push-io-history dterm 'bind)
    ;;(setf -oc oc) (break "bind")
    
    (build-directive (definition 'library insert)
		     (oid)
		     *lib-description*
		     ;; seems redundent since dependency contains stamp which is same as data.
		     dterm)))

;; kludge alert : might be better if completion never queued??
(defun bind-sneak (oc)
  ;;(break "bs")
  (format t "~%BindSneak ")
  (let ((oid (new-object-id)))
    (do-sneak-commit (bind-directive oid oc))
    oid))



;; object_id -> object_contents -> (oid # directive) ??
;; or implicitly define new oid and map arg to new.
;; 
;; insert/delete -> bind/unbind

;;;;	
;;;;	PERF TODO : only last bind to oid between transaction commits
;;;;	  needs to be written.  So maybe better not to make objc persist
;;;;	  at bind time but rather at commit time and then only if there 
;;;;	  is not a later commit binding same oid. 
;;;;	

(defunml (|lib_bind| (oid oc))
    (object_id -> (object_contents -> directive))

  (show-telemetry "~%lib_bind ~a"
		  (let ((p (property-of-objc oc 'name))) (when p (token-of-itoken-term p))))
  (when (dummy-object-id-p oid)
    (raise-error (oid-error-message (list oid) '(library insert dummy))))

  (when (library-oid-bound-p oid)
    ;;(break "lb")
    (raise-error (oid-error-message (list oid) '(library insert bound))))

  ;; could check for eq oc (lib-lookup oid) and skip bind if so. but not bound at the moment.
  (make-ml-directive (bind-directive oid oc)))


(defunml (|lib_bind_with_history| (oid oc oldoc))
    (object_id -> (object_contents -> (term -> directive)))

  (show-telemetry "~%lib_bind_with_history ~a "
		  (let ((p (property-of-objc oc 'name))) (when p (token-of-itoken-term p))))
  
  (when (dummy-object-id-p oid)
    (raise-error (oid-error-message (list oid) '(library insert dummy))))

  (when (library-oid-bound-p oid)
    ;;(break "lb")
    ;;(setf -oid oid)
    (raise-error (oid-error-message (list oid) '(library insert bound))))

  ;; could check for eq oc (lib-lookup oid) and skip bind if so. but not bound at the moment.
  (make-ml-directive (bind-directive oid oc oldoc)))


(defunml (|lib_unbind| (oid))
    (object_id -> directive)

  (unless (library-oid-bound-p oid)
    (raise-error (oid-error-message (list oid) '(library delete bound not))))
  (when (library-object-active-p (library-lookup oid))
    (raise-error (oid-error-message (list oid) '(library delete active))))

  (io-echo "u")
  (push-io-history (ioid-term oid) 'unbind)

  ;;(break "lu")
  (make-ml-directive
   (build-directive (oid 'library delete)
		    (oid)
		    *lib-description*
		    )))
;;;;	
;;;;	library makes substance at translation.
;;;;	library exports substance at activation.
;;;;	definition is derived from exported substance at import.
;;;;	
;;;;	
;;;;	activation and the active bit.
;;;;	  - when activating the producer will send definiton_update msg.
;;;;	  - consumer will clone object and set bit.
;;;;	
;;;;	activation : send definition_insert containing parameterized by type and containing exported
;;;;	  substance.
;;;;	
;;;;	



;;;;	
;;;;	activate/deactivate
;;;;	  - bit visibility parameterized by transactions.
;;;;	  - broadcasts substance.
;;;;	


(defunml (|lib_active_p| (oid))
    (object_id ->  bool)
  
  (lib-active-p oid))

(defunml (|lib_collectable_p| (oid))
    (object_id ->  bool)
  
  (library-object-collectable-p (library-lookup oid)))


(defunml (|lib_allow| (oid))
    (object_id -> directive)

  (unless (library-oid-bound-p oid)
    (raise-error (oid-error-message (list oid) '(library allow bound not))))

  (let ((obj (library-lookup oid)))

    (if (library-object-collectable-p obj)
	;; need to do some type if empty directive.
	(progn
	  (message-emit (warn-message '(library allow collectable)))
	  *null-directive*)

	(progn
	  (io-echo "w")

	  (make-ml-directive
	   (build-directive (oid 'library allow)
			    (oid)
			    *lib-description*))))))

(defunml (|lib_disallow| (oid))
    (object_id -> directive)

  (unless (library-oid-bound-p oid)
    (raise-error (oid-error-message (list oid) '(library disallow bound not))))

  (let ((obj (library-lookup oid)))

    (if (library-object-collectable-p obj)
	;; need to do some type if empty directive.
	(progn
	  (io-echo "d")

	  (make-ml-directive
	   (build-directive (oid 'library disallow)
			    (oid)
			    *lib-description*)))

	(progn
	  (message-emit (warn-message '(library disallow collectable not)))
	  *null-directive*))))


;;;;	
;;;;	act/deact dis/allowed state contained  in log.
;;;;	  - should be attribute of oid and rebinding objc should not affect.
;;;;
;;;;	library-object built from bind then act and allow bcasts.
;;;;	binding imports and replaces library object in table losing state bits.
;;;;	but then again nonsensical to activate an unbound oid. However it is
;;;;	sensible to rebind an active oid for cosmetic changes. Needs work.
;;;;	
;;;;	rebindingloses since 
;;;;	
;;;;	
;;;;	need ability to make persistent change to source within a transaction.
;;;;	  - need commit to write to disk.
;;;;	  - need ostate broadcast.
;;;;	  - destructive mod of source ok. destructive mod of data stamp more problemmatical.
;;;;	  - would like to avoid deact/act
;;;;	
;;;;	unbind/ then bind clone  without requiring deact/act.
;;;;	simply binds and rebinds, 
;;;;	 but does not update objc on disk. could bind clone. but then there is some pointer to old
;;;;	 and that is bound then reverts.
;;;;
;;;;	best would be weak-bind new-oc, then undo will restore old oc without update.
;;;;	
;;;;	weak_bind : if ok to bind while active, which is true
;;;;	  - vacuously true if not active
;;;;	  - false if current or new objc requires translation
;;;;	  - true if objc are similar up to non-substantive properties.
;;;;	
;;;;	
;;;;	non-substantive prop mod results in new objc but with same translation
;;;;	allowing similar objc check to succeed.
;;;;	
;;;;	

;;;;	TODO : noticed that multiple calls appear to stack library-objects in lib table??
;;;;	
;;;;	
;;;;	
;;;;	


(defunml (|lib_weak_bind| (oid noc))
    (object_id -> (object_contents -> directive))
  
  ;;(setf -oid oid -noc noc) (break "lwb")
  (if (not (library-oid-bound-p oid))
      (progn
	(io-echo "B")
	(make-ml-directive
	 (build-directive (definition 'library insert)
			  (oid)
			  *lib-description*
			  ;; seems redundent since dependency contains stamp which is same as data.
			  (idefinition-term (let ((stamp (stamp-to-term (stamp-of-data noc))))
					      (idependency-term oid stamp (ivoid-term)))
					    (persist-data noc nil t)))))

      (let* ((obj (library-lookup oid))
	     (oc (objc-of-library-object obj)))
    
	(if (and (library-object-active-p obj)
		 (not (objc-similar-p oc noc)))

	    (raise-error (oid-error-message (list oid) '(library weak_bind similar not)))
	    
	    (progn
	      (io-echo "B")
	      ;;(setf -oid oid -noc noc) (break "lwb")

	      (make-ml-directive
	       (build-sequence-directive 
		(build-directive (oid 'library delete)
				 (oid)
				 *lib-description*)

		(build-directive (definition 'library insert)
				 (oid)
				 *lib-description*
				 ;; seems redundent since dependency contains stamp which is same as data.
				 (idefinition-term (let ((stamp (stamp-to-term (stamp-of-data oc))))
						     (idependency-term oid stamp (ivoid-term)))
						   ;;(persist-data oc nil t)
						   (persist-data (new-lobj obj noc) ))))))))))


(defunml (|lib_activate| (oid))
    (object_id -> directive)

  ;;(setf a oid) (break "la")
  (unless (library-oid-bound-p oid)
    (raise-error (oid-error-message (list oid) '(library activate bound not))))

  (let ((obj (library-lookup oid)))

    (if (library-object-active-p obj)
	;; need to do some type if empty directive.
	(progn
	  (message-emit (warn-message '(library activate active)))
	  *null-directive*)

	(let ((objc (objc-of-library-object obj)))
	  (unless (objc-translated-p objc)
	    (raise-error (oid-error-message (list oid) '(library activate translate not))))

	  ;;(setf a objc b obj) (break "la1")

	  ;; slight kludge to allow failure of activate due to compile errors
	  ;; by sending req prior to broadcast.
	  
	  (io-echo "+")

	  (make-ml-directive
	   (build-sequence-directive

	    (build-directive (oid 'library activate)
			     (oid)
			     *lib-description*)
	
	    ;;(break "la2")

	    ;;(setf b table-type) (break "la3")
	    (let ((table-type (map-objc-to-table-type objc))
		  (desc (description-property-term-of-objc objc)))

	      (when table-type
		(build-directive (definition table-type insert)
				 (oid)
				 desc
				 (let ((data (objc-substance objc)))
				   ;;(setf -data data -oid oid -oc objc) (break "act")
				   (idefinition-term 
				    (idependency-term oid
						      (stamp-to-term (stamp-of-data objc))
						      (stamp-to-term (stamp-of-data data)))
				    (persist-data data nil t)))))

	      ;; not reliable in that not done at start time
	      ;; if to be done at start then order matters.
	      ;; so until can be done right, force user tocall explicit.
	      ;;(when (and table-type (eql 'code table-type))
	      ;;(post-activate-code-object desc oid ))

	   )))))))



(defunml (|lib_deactivate| (oid))
    (object_id -> directive)

  (unless (library-oid-bound-p oid)
    (raise-error (oid-error-message (list oid) '(library deactivate bound not))))

  (let ((obj (library-lookup oid)))

    #|
    (format t "deact ~a ~a ~a~%"
	    (string-of-oid oid)
	    (library-object-active-p obj)
	    (when (library-object-active-p obj)
	      (map-objc-to-table-type (objc-of-library-object obj))))
    |#

    (if (library-object-active-p obj)
	;; need to do some type if empty directive.
	(let ((objc (with-ignore (objc-of-library-object obj))))
	
	  (io-echo "-")

	  (let ((deact-dir (build-directive (oid 'library deactivate)
					    (oid)
					    *lib-description*)))

	    (if (null objc)

		(make-ml-directive deact-dir)
		
		(make-ml-directive
		 (build-sequence-directive

		  deact-dir
	
		  (let ((table-type (map-objc-to-table-type objc)))
		    (when table-type
		      (build-directive (oid table-type delete)
				       (oid)
				       (description-property-term-of-objc objc)))))))))
	(progn
	  (message-emit (warn-message '(library deactivate active not)))
	  *null-directive*))))


#|
;; todo remove map remnants.

(defunml (|lib_map| (oid-old oid-new))
    (object_id -> (object_id -> directive))

  (when (oid-mapped-p oid-old)
    (raise-error (oid-error-message (list oid-old) '(lib insert mapped))))
  (when (library-oid-bound-p oid-old)
    (raise-error (oid-error-message (list oid-old) '(lib insert bound))))

  (make-ml-directive
   (build-directive (oid library insert)
		    (oid-old oid-new)
		    *lib-description*
		    )))
|#



(defunml (|map_library| (f))
    ((object_id -> *) -> unit)

  (without-dependencies
   (definition-table-map (resource 'library)
       (current-transaction-stamp)
     #'(lambda (oid obj)
	 (declare (ignore obj))
	 (ap f oid)))))

(defunml (|library_accumulate| (f))
    (((* -> unit) -> (object_id -> unit)) -> (* list))

  (let ((acc nil))
    (let ((ff (funmlcall f (make-closure #'(lambda (x) (push x acc)) 1))))
      (without-dependencies
       (definition-table-map (resource 'library)
	   (current-transaction-stamp)
	 #'(lambda (oid obj)
	     (declare (ignore obj))
	     (funmlcall ff oid)))))
    acc))

(defunml (|map_lib| (kinds activep f))
    ((tok list) -> (bool -> ((object_id -> (object_contents -> bool)) -> (object_id list))))

  (let ((acc nil))
    (without-dependencies
     (definition-table-map (resource 'library)
	 (current-transaction-stamp)
       #'(lambda (oid obj)
	   (when (or (null activep)
		     (library-object-active-p obj))
	     (with-objc-of-library-object (objc obj)
	       (when (member (kind-of-objc objc) kinds)
		 (when (ap f oid objc)
		   (push oid acc))))))))
      acc))


#| want some quicker map that stops when found
(defunml (|map_library_until_found| (f))
    ((object_id -> *) -> unit)

  (definition-table-map (resource 'library)
      (current-transaction-stamp)
    #'(lambda (oid obj)
	(declare (ignore obj))
	(ap f oid))))
|#


;;;;
;;;; directory object trees contructed to send to edd for printing to files
;;;;

(define-primitive |!object_tree| () (name kind status term children))
(define-primitive |!object_tree_cons| () (car cdr))
(define-primitive |!object_comment| ((oid . oid)) (prefix suffix))

(defvar *null-status* (itoken-term 'status?))
(defun stm-to-prfs (soid)
   (proofs-of-statement-source (source-of-objc (oc soid))))

(defun maybe-stm-to-prfs (soid)
  (with-objc-of-oid (objc soid)
    (when (eql (kind-of-objc objc) 'STM)
      (with-source-of-objc (src objc)
	(proofs-of-statement-source src)))))
	

(defun comment-object-tree-term (comment)
  (iobject-tree-term (itoken-term (intern-system ""))
		     (itoken-term 'INV)
		     *null-status*
		     comment
		     (iobject-tree-nil-term)))

(defun directory-tree (diroid &optional short-p)
  ;;(setf ddd diroid)
  (without-dependencies

   (let ((ohash (make-hash-table :test #'equal)))
	       
     (labels ((visit (noid)	
		(let* ((oid (cdr noid))
		       (directory-p (dag-directory-p oid))
		       (kind (kind-of-oid oid)))		       		       

		  (if directory-p
		     
		      (unless (gethash (stamp-of-oid oid) ohash)
			(setf (gethash (stamp-of-oid oid) ohash) t)
			(let* ((name (itoken-term (car noid)))
			       ;;(kind (kind-of-oid oid))
			       (prop (property-of-objc (oc oid) '|comment|))
			       (comment (when prop (setf -prop prop) (break "dt") (term-of-substance (substance-of-objc (oc (oid-of-ioid-term prop))))))
			       (suffix (when comment (list (comment-object-tree-term (suffix-of-iobject-comment-term comment)))))
			       (prefix (when comment (list (comment-object-tree-term (prefix-of-iobject-comment-term comment))))))
			  (append prefix
				  (cons (iobject-tree-term name
							   (itoken-term 'DIR);;(itoken-term kind)
							   *null-status*
							   (ivoid-term)
							   (map-list-to-isexpr
							    (flatten (mapcar #'visit (dag-directory-children oid)))
							    (iobject-tree-nil-term)))
					suffix))))
			      
		      (if (or short-p
			      (not (equal 'STM kind)))
			  (oid-to-iobject-tree-term-with-comments oid)

			  (let ((prfs (stm-to-prfs oid)))
			    (if prfs
				;;LAL only prints 1st prf, may also want backups, use flag, and extra label fn
				;;no mechanism for adding comment prop to prfs yet
				
				(list (oid-to-iobject-tree-term-with-comments oid)
				      (oid-to-iobject-tree-term-with-comments (car prfs)))

				(oid-to-iobject-tree-term-with-comments oid))))))))		  
				   
       ;;(oid-to-iobject-tree-term-with-comments oid)
       ;;))))))
			  
   
       (iobject-tree-term (itoken-term (name-of-oid diroid))
			  (itoken-term 'DIR);;(kind-of-oid diroid);; why is kind a term?
			  *null-status*
			  (ivoid-term);;(term-of-substance (substance-of-objc (oc diroid)))
			  (map-list-to-isexpr (flatten (mapcar #'visit (if (dag-directory-p diroid)
									   (dag-directory-children diroid)
									   (list (cons (name-of-oid diroid) diroid))
									   )))
					      ;;lal flatten is a hack to group prfs of stms at same level,
					      ;;may want to treat them differently, ie to indent in file, etc
					      (iobject-tree-nil-term)))))))

(defun oid-to-iobject-tree-term (oid)

  (let* ((name (itoken-term (name-of-oid oid)))
	 (kind (kind-of-oid oid)))

    (if (equal 'PRF kind)
	(let* ((itree (inf-tree-of-proof-source-r (source-of-objc (oc oid))))
	       (status (iproof-status-term (if (itree-complete-p itree)
					       `|complete|
					       `|incomplete|)))
	       (proof-term (inf-tree-to-iproof-node-term itree
							 status
							 (ipui-addr-nil-term)
							 (intern-system "d")
							 nil 0 nil)))
	  
	  (iobject-tree-term name (itoken-term kind) status proof-term (iobject-tree-nil-term)))
      
	(progn
	  ;;(setf -kind kind -oid oid) (break "otit")
	  (iobject-tree-term name (itoken-term kind) *null-status*
			     (let ((sub (substance-of-objc (oc oid) t)))
			       (if (and sub (not (member kind '(code))))
				   (term-of-substance sub)
				   (or (term-of-source (source-of-objc (oc oid)))
				       (ivoid-term))))
			     (iobject-tree-nil-term))))))
      
	
(defun oid-to-iobject-tree-term-with-comments (oid)

  (let* ((name (itoken-term (name-of-oid oid)))
	 (prop (property-of-objc (oc oid) '|comment|))
	 (comment (when prop
		    (let ((substance (substance-of-objc (oc (oid-of-ioid-term prop)) t)))
		      (if substance
			  (term-of-substance substance)
			  (or (term-of-source (source-of-objc (oc (oid-of-ioid-term prop))))
			      (ivoid-term))))))
	 (term (oid-to-iobject-tree-term oid)))

    (if comment
	(let ((suffix (when comment (list (comment-object-tree-term (suffix-of-iobject-comment-term comment)))))
	      (prefix (when comment (list (comment-object-tree-term (prefix-of-iobject-comment-term comment))))))
	  (iobject-tree-term name
			     (itoken-term 'GRP)
			     *null-status*
			     (ivoid-term)
			     (map-list-to-isexpr (append prefix (cons term suffix)) (iobject-tree-nil-term))))
	term)))	

(defun object-tree (oid)
  ;;(setf oo oid)
  (without-dependencies
   (oid-to-iobject-tree-term-with-comments oid)))	 
	

(defunml (|directory_object_tree| (oid))
    (object_id -> term)

  (directory-tree oid))

(defunml (|directory_object_tree_short| (oid))
    (object_id -> term)

  (directory-tree oid t))

(defunml (|object_tree| (oid))
    (object_id -> term)

  (object-tree oid))

(defunml (|oid_to_iobject_tree_term_with_comments| (oid))
    (object_id -> term)

  (oid-to-iobject-tree-term-with-comments oid))


