
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	Lib makes some syntax checks, but edd may still reject dform.
;;;;	Lib makes checks which can be caught with a peephole while
;;;;	eddd catches problems requiring a wider view. Eg, lib can
;;;;	catch the floatup display id being used as a parameter but
;;;;	but edd catches floatup var and format being used without iteration.
;;;;
;;;;	
;;;;	Lib maintains a dform table by object.
;;;;	Lib expects edd to maintain a dform table by term group.
;;;;	Updates of dform table cause broadcasts of dform group changes.
;;;;	Group change updates are not broadcast until dform change committed.
;;;;	Thus group changes are atomic in that the change and the commit will
;;;;	always be sent simultaneously.

;;;;	RLE NAP  Disp translations in the lib dform table should reside in memory only on demand,
;;;;	RLE NAP otherwise the disp objects may take up a unwarranted amount of memory.
;;;;	RLE NAP Can probably do this by monkeying with the lib dforms def. Maybe all defs could
;;;;	RLE NAP have some disk/resident options!
;;;;


;;;;	Duplicate group broadcasts can be thinned at dform commmit.
;;;;	maybe wait for some more general broadcast flow control tools.
;;;;	Or add some type of transaction cleanup hook .

;;;;	another possibility is for edd and lib to both maintain dform table.
;;;;	but have lib send over dependencies in order after modification to lib table.
;;;;	fttb prefer first, since in setup where edd connected to multiple libs, probably
;;;;	easier to have group updates. weak reasoning there as can use dependency group
;;;;	to maintain groupings.
;;;;	




;; at commit
;; queue term group, remove prev term group in queue.
;; after commit broadcast term group.
  



;;;
;;;	Makes some rudimentary syntax checks.
;;;
;;;	  - catch known errors asap.
;;;	  - do not otherwise restrict edd input. 
;;;
;;;
;;;	


;;
;; Does syntax checking on known attributes. Ignores unexpected attributes.
;; 
;;

 
(defun disp-translate-attribute-r (term)
  (cond

    ;; !dform_hidden_cond_expr
    ((idform-hidden-cond-expr-term-p term)
     (unless (icond-expr-term-p (expression-of-idform-hidden-cond-expr-term term))
       (raise-error (error-message '(condition-expression hidden) term))))
    
    ;; !dform_cond_expr
    ((idform-cond-expr-term-p term)
     (unless (icond-expr-term-p (expression-of-idform-cond-expr-term term))
       (raise-error (error-message '(condition-expression) term))))
    

    ;; !dform_precedence_injection{}([!dform_address{s}(<oa>) \| !precedence_label{t}()])
    ((idform-precedence-injection-term-p term)
     (unless (idform-precedence-pointer-term-p
	      (pointer-of-idform-precedence-injection-term term))
       (raise-error (error-message '(precedence injection pointer)
				   (pointer-of-idform-precedence-injection-term term)))))
    
    ;; !dform_families{}(!dform_family{s}() !dform_family_cons list)
    ((idform-families-term-p term)
     (map-isexpr (list-of-idform-families-term term)
		 (idform-family-cons-op)
		 #'(lambda (term)
		     (unless (idform-family-term-p term)
		       (raise-error (error-message '(dform families) term)))
		     (unless (real-itext-term-p term)
		       (raise-error (error-message '(dform families family) term))))))
			 
    ;;!dform_family{s}()
    ((idform-family-term-p term)
     (unless (real-itext-term-p term)
       (raise-error (error-message '(dform family) term))))

    ;; !dform_edit_macro{s}()
    ((idform-macro-name-term-p term)
     (unless (real-itext-term-p term)
       (raise-error (error-message '(dform macro-name) term))))

    ;; !dform_name{s}()
    ((idform-name-term-p term)
     (unless (real-itext-term-p term)
       (raise-error (error-message '(dform name) term)))
     (when (every #'digit-char-p (name-of-idform-name-term term))
       (raise-error (error-message '(dform name digits) term))))
    
    ((icondition-sexpr-p term)
     (with-tag '(disp attributes)
       (icondition-sexpr-r term)))
    
    (t nil))

  (values))


;;;;	In the model term,
;;;;	 multiple nonterm slots with the same id are permitted, but they must
;;;;	 be of the same kind (e.g., string, number)
;;;;	 A direct subterm of the model term must be either
;;;;	  a term slot, or
;;;;	  variable{<x>:string}() where <x> is a nonterm slot
;;;;	   and x is not the empty string, or
;;;;	  a term containing no slots.
;;;;	 Further, if a direct subterm contains a free variable, then the
;;;;	  subterm must BE that variable.
;;;;	
;;;;	#TT(3):  This seems to be saying that a possible model is
;;;;	#
;;;;	#        P( variable{<x>:variable} )                     (*)
;;;;	#
;;;;	#which would mean "P applied to some variable". This seems to be a special case
;;;;	#in which there would be a parameter metavariable below the toplevel operator,
;;;;	#which I thought we recently said didn't ever happen.
;;;;	Well, the three conditions in the document you are quoting are
;;;;	correct, about the three kinds of direct subterms, so we made a
;;;;	mistake whenever we said that a parameter metavariable would never
;;;;	occur below the toplevel operator.
;;;;	BTW, this model is not the real target of this display feature.
;;;;	     What we're really after is models such as D(var{<x>};<x>.<e>)
;;;;	     in which one argument is the variable bound in another argument.
;;;;	     For example, suppose D(a;x.f(x)) were defined as the derivitive
;;;;	     of f(x) in x at a. Then by using this display spec:
;;;;	      d<e>/d<x>|<a>for model D(<a>;<x>.<e>)
;;;;	      d<e>/d<x>for model D(var{<x>};<x>.<e>)
;;;;	     we get the usual convention for derivative notations.

;;;;	we allow foo(variable{<v>}) as subterm.
;;;;	we should only allow it when there is a similar binding binding some other term.
;;;;	ie foo(variable{<v>};<v>.<t>) is ok,
;;;;	   foo(<v>.<v>) and foo(variable{<v>}) not ok. 
;;;;	   foo(variable{<v>};<v>.<v>) ?? say no.
;;;;


;;;;	Restrictions on model term :
;;;;	
;;;;	 No display meta variables allowed in subterms of the model term
;;;;	except for the case of a simple variable. If such a display meta
;;;;	variable occurs, the same meta variable must occur in the binding list
;;;;	of some other subterm and must not occur in the binding list of the
;;;;	subterm.
;;;;
;;;;	If a constant term occurs as a subterm, it can not contain free variables
;;;;	unless it is a simple variable itself.
;;;;
;;;;	The same variable may not occur more than once as a subtem.
;;;;
;;;;	Following not currently enforced : 
;;;;	    - a template variable may be used in more than one parameter
;;;;	      only if the types of each parameter in which it is used are the
;;;;	      same or neither one is variable or level-expression.
;;;;	      foo{<x>:n, <x>:tok>} and foo{<x>:n, <x>:n} are allowed.
;;;;	      foo{<x>:n, <x>:v} is not allowed.
;;;;
;;;;	Check other doc for motivation.





(defun display-meta-variable-occurs-p (term)
  (labels ((occurs-p (term)
	     (or (exists-p #'(lambda (p) (display-meta-parameter-p p))
			     (parameters-of-term term))
		 (exists-p #'(lambda (bt)
			       (or (exists-p #'(lambda (b) (display-meta-variable-id-p b))
					     (bindings-of-bound-term bt))
				   (occurs-p (term-of-bound-term bt))))
			   (bound-terms-of-term term)))))
    (occurs-p term)))



;;;
;;;  The error checking is organized semantically.
;;;  A syntactic organization would be more efficient, but also
;;;  more difficult to modify. Historically, there have been many
;;;  modifications to this procedure.
;;;

(defun disp-translate-model-r (term)

  (let ((error-p nil))
    
    (labels ((note-error (msg)
	       (setf error-p t)
	       (message-emit msg)))
	      
  
      (dolist (bound-term (bound-terms-of-term term))
	(let ((bindings (bindings-of-bound-term bound-term))
	      (subterm (term-of-bound-term bound-term)))
	  (cond
	    ((itemplate-term-p subterm)
	     (real-itext-term-p subterm)
	     nil)

	    ((display-meta-variable-term-p subterm)
	     (when (member (id-of-variable-term subterm) bindings)
	       (note-error (error-message '(dform model subterm variable template bound)
					  (id-of-variable-term subterm))))
	     
	     (unless (exists-p #'(lambda (bt)
				   (and (not (eql bt bound-term))
					(member (id-of-variable-term subterm)
						(bindings-of-bound-term bt))))
			       (bound-terms-of-term term))
	       (note-error (error-message '(dform model subterm variable template)
					  (id-of-variable-term subterm)))))

	    ((display-meta-variable-occurs-p subterm)
	     (note-error (error-message '(dform model subterm meta)
					subterm)))

	    (t;; ie constant term
	     (if (variable-p subterm)
		 (when (exists-p #'display-meta-variable-id-p
				 bindings)
		   (note-error (error-message '(dform model subterm constant variable bound meta)
					      subterm)))
		 (when (free-vars subterm)
		   (note-error (error-message '(dform model subterm constant free)
					      subterm)))) ))))

      ;; Check for meta variables used multiple times as subterms.
      (let ((t-vars nil)
	    (noted nil))
	(dolist (bt (bound-terms-of-term term))
	  (let ((subterm (term-of-bound-term bt)))
	    (let ((id (cond
			((itemplate-term-p subterm)
			 (push (meta-id-of-itemplate-term subterm) t-vars))
			((display-meta-variable-term-p subterm)
			 (push (id-of-variable-term subterm) t-vars)))))
	      (when id 
		(cond
		  ((member id noted) nil)
		  ((member id t-vars)
		   (note-error (error-message '(dform model subterm multiple) id))
		   (push id noted))
		  (t (push id t-vars))))))))

      ;; Check for meta variables used to stand for terms and parameter values.
      (let ((p-vars nil)
	    (t-vars nil))
	(dolist (p (parameters-of-term term))
	  (when (display-meta-parameter-p p)
	    (push (value-of-parameter p) p-vars)))
	(dolist (bt (bound-terms-of-term term))
	  (dolist (b (bindings-of-bound-term bt))
	    (when (display-meta-variable-id-p b)
	      (push b p-vars)))
	  (let ((subterm (term-of-bound-term bt)))
	    (cond
	      ((itemplate-term-p subterm)
	       (push (meta-id-of-itemplate-term subterm) t-vars))
	      ((display-meta-variable-term-p subterm)
	       (push (id-of-variable-term subterm) p-vars)))))

	(let ((intersection (intersect-vars t-vars p-vars)))
	  (when intersection
	    (note-error (error-message '(dform model overloaded) intersection)))))

      ;; check for float down without iteration
      (let ((floatdown-p nil)
	    (iterate-p 0))
	(dolist (bt (bound-terms-of-term term))
	  (let ((subterm (term-of-bound-term bt)))
	    (when (itemplate-term-p subterm)
	      (cond
		((floatdown-p (meta-id-of-itemplate-term subterm))
		 (setf floatdown-p t))
		((iterate-p (meta-id-of-itemplate-term subterm))
		 (incf iterate-p))))))
	
	(when (and floatdown-p (zerop iterate-p))
	  (note-error (error-message '(dform model floatdown no-iterate))))

	(when (> iterate-p 1)
	  (note-error (error-message '(dform model iterate multiple))))) )

    (if error-p
	(raise-error '(dform model))
	(values))))


;; RLE TODO
(defun disp-translate-child-format-attribute-r (term)

  (cond

    ;; conditions
    ((icondition-sexpr-p term)
     (with-tag '(disp child attribute)
       (icondition-sexpr-r term)))

    ;; parentheses
    ((idform-child-parentheses-term-p term)
     (let ((error-p nil))
       (labels ((note-error (msg)
		  (setf error-p t)
		  (message-emit msg)))
	 ;; relation
	 (unless (parentheses-relation-value-p (relation-of-idform-child-parentheses-term term))
	   (note-error (error-message '(disp format child attribute parentheses relation)
				      (relation-of-idform-child-parentheses-term term))))
	 ;; pointer
	 (unless (idform-precedence-pointer-term-p
		  (pointer-of-idform-child-parentheses-term term)
		  t)
	   (note-error (error-message '(disp format child attribute parentheses precedence pointer)
				      (pointer-of-idform-child-parentheses-term term))))
	 ;; binding
	 (let ((b (binding-of-formats-of-idform-child-parentheses-term term)))
	   (unless (variable-id-p b)
	     (note-error (error-message '(disp format child attribute parentheses binding)
					(parameter-value-to-pretty-string b
									  *variable-type*))))
	   ;; formats
	   (map-isexpr (formats-of-idform-child-parentheses-term term)
		       (idform-format-cons-op)
		       #'(lambda (iformat)
			   (with-handle-error ((nil) (setf error-p t))
			     (disp-translate-format-r iformat b))))))

       (when error-p
	 (raise-error (error-message '(disp format child attribute parentheses))))))

    (t (raise-error (error-message '(disp format child attribute unknown) term)))))


(defun disp-translate-format-r (term &optional (parens-var nil))

  (labels
      ((translate-child-attributes-r (term)
	 (let ((error-p nil))
	   (map-isexpr term
		       (idform-child-attr-cons-op)
		       #'(lambda (iattr)
			   (with-handle-error ((nil) (setf error-p t))
			     (disp-translate-child-format-attribute-r iattr))))
	   (when error-p
	     (raise-error (error-message '(dform child attribute)))))))



  (cond
    ((variable-p term)
     (unless (eql (id-of-variable-term term) parens-var)
       (raise-error (error-message '(dform format variable) term parens-var))))

    ((itext-term-p term)
     (unless (real-itext-term-p term)
       (raise-error (error-message '(dform format text) term))))

    ((idform-label-wrap-term-p term)

     (let ((formats (do ((term term (formats-of-idform-label-wrap-term term)))
			((not (idform-label-wrap-term-p term)) term)
		      (unless (real-itext-term-p term)
			(raise-error (error-message '(dform format label-wrap label) term))))))
     
     (with-message-accumulator ('(label-wrap))
       (let ((error-p nil))
	 (map-isexpr formats
		     (idform-format-cons-op)
		     #'(lambda (iformat)
			 (with-handle-error ((nil) (setf error-p t))
			   (disp-translate-format-r iformat))))
	 (when error-p (raise-error (error-message '())))))))

    ((iblank-term-p term)
     nil)

    ((idform-depth-term-p term)
     (let ((parameters (parameters-of-term term)))
       (unless (real-parameter-p (car parameters))
	 (raise-error (error-message '(dform format depth type) term)))
       (let ((type (value-of-parameter (car parameters))))
	 (unless (depth-type-value-p type)
	   (raise-error (error-message '(dform format depth type value) type)))
	 (if (eql 'nodepth type)
	     (unless (null (cdr parameters))
	       (raise-error (error-message '(dform format depth nodepth) term)))
	     (unless (real-parameter-p (cadr parameters))
	       (raise-error (error-message '(dform format depth amt) term)))))))
	     
    ((idform-push-term-p term)
     (unless (real-parameter-p (car (parameters-of-term term)))
       (raise-error (error-message '(dform format push) term))))

    ((idform-pop-term-p term)
     nil)
     
    ((idform-break-control-term-p term)
     (unless (real-parameter-p (car (parameters-of-term term)))
       (raise-error (error-message '(dform format break-control real) term)))

     ;;(setf -term term) (break "bcvpok")
     (unless (break-control-value-p (tok-upcase (type-of-idform-break-control-term term)))
       ;;(setf -term term) (break "bcvp")
       (raise-error (error-message '(dform format break-control type)
				   (type-of-idform-break-control-term term)))))

    ((idform-break-term-p term)
     (let ((parameters (parameters-of-term term)))
       (unless (and (real-parameter-p (car parameters))
		    (let ((parameters (cdr parameters)))
		      (or (null parameters)
			  (and (real-parameter-p (car parameters))
			       (let ((parameters (cdr parameters)))
				 (or (null parameters)
				     (and (real-parameter-p (car parameters))
					  (null (cdr parameters)))))))))
	 ;; could do finer discrimination on error and emit narrower msg.
	 (raise-error (error-message '(dform format break) term)))))

    ;; child formats.
    ((idform-variable-child-term-p term)
     (when (dummy-display-meta-variable-id-p (meta-id-of-idform-variable-child-term term))
       (raise-error (error-message '(dform format child variable dummy))))
     (when (floatdown-p (meta-id-of-idform-variable-child-term term))
       (raise-error (error-message '(dform format child variable floatdown))))
     (when parens-var
       (raise-error (error-message '(dform format child variable in-parens))))
     (unless (real-parameter-p (car (parameters-of-term term)))
       ;;(setf a term) (break "vi")
       (raise-error (error-message '(dform format child variable id) term)))
     (unless (real-parameter-p (cadr (parameters-of-term term)))
       (raise-error (error-message '(dform format child variable descriptor) term)))
     (translate-child-attributes-r (attributes-of-idform-child-term term)))
				   
    ((idform-constant-child-term-p term)
     (translate-child-attributes-r (attributes-of-idform-child-term term)))
				   
    ((idform-library-child-term-p term)
     (when parens-var
       (raise-error (error-message '(dform format child variable in-parens))))
     (unless (real-itext-term-p term)
       (raise-error (error-message '(dform format child library) term)))
     (when (string= "" (pointer-of-idform-library-child-term term))
       (raise-error (error-message '(dform format child library empty) term)))

     (translate-child-attributes-r (attributes-of-idform-child-term term)))
				   
    )))


(defun disp-translate-source (term)

  (let ((term-sig nil)
	(error-p nil))
    
    (with-message-accumulator ('(disp translate))
    
      (map-isexpr term
		  (idform-cons-op)
		  #'(lambda (dform)
		      (with-handle-error ((nil) (setf error-p t))
		    
			;; check for !dform(0;0;0) term.
			(unless (idform-term-p dform)
			  (raise-error (error-message '(term) dform)))

			;; check attributes.
			(map-isexpr (attributes-of-idform-term dform)
				    (idform-attr-cons-op)
				    #'(lambda (iattr)
					(with-handle-error ((nil) (setf error-p t))
					  (disp-translate-attribute-r iattr))))

			;; check model
			(with-handle-error ((nil) (setf error-p t))
			  (disp-translate-model-r (model-of-idform-term dform)))

			;; check formats
			(map-isexpr (formats-of-idform-term dform)
				    (idform-format-cons-op)
				    #'(lambda (iformat)
					(with-handle-error ((nil) (setf error-p t))
					  (disp-translate-format-r iformat))))

			;; check models equal.
			(if term-sig
			    (unless (term-sig-of-term-p term-sig (model-of-idform-term dform))
			      (raise-error (error-message '(model)
							  term-sig
							  (term-sig-of-term (model-of-idform-term dform)))))
			    (setf term-sig (term-sig-of-term (model-of-idform-term dform))))

			;; Could check for variables in formats not defined in model.
			;; However, it is forseeable that an edd might allow this, eg,
			;; indirect lib display.
			
			)))

      (cond
	(error-p (raise-error (error-message '())))
	((null term-sig) (raise-error (error-message '(empty))))
	(t (term-sig-to-model-term term-sig))))))





;;;;
;;;;	!precedence_label{t}()
;;;;	!precedence_object{}(oa)
;;;;	!precedence_ordered{}(left right)
;;;;	!precedence_equal{}(left right)
;;;;	!precedence_unrelated{}(left right)
;;;;

;;;;	<prec-tree>		: <prec-unordered-list>
;;;;				| <prec-ordered-list>
;;;;				| <prec-equivalence-list>
;;;;
;;;;	<prec-label>		: [<prec-address>
;;;;			 	   <tok>
;;;;				  ]
;;;;
;;;;	<prec-ordered-list>	: ( <prec-unordered-list> | <prec-equivalence-list> ) list
;;;;
;;;;	<prec-unordered-list>	: ( <prec-ordered-list> | <prec-equivalence-list> ) list
;;;;
;;;;	<prec-equivalence-list>	: <prec-label> list
;;;;				  

(defun prec-translate-source-r (term)
  (let ((error-p nil)
	(labels nil)
	(dups nil))
    
    (with-message-accumulator ('(prec translate))
    
      (labels ((note-error (msg)
		 (setf error-p t)
		 (message-emit msg))
	       
	       (visit-label (term)
		 (if (real-itext-term-p term)
		     (let ((label (token-of-iprecedence-label-term term)))
		       (unless (member label dups)
			 (if (member label labels)
			     (progn
			       (push label dups)
			       (note-error (error-message '(label duplicate) label)))
			     (push label labels))))
		     (note-error (error-message '(label) term))))

	       (visit-obj (term)
		 (unless (oid-p (oid-of-iprecedence-object-term term))
		   (note-error (error-message '(object address) term))))


	       (visit-eq (term)
		 (cond
		   ((iprecedence-label-term-p term)
		    (visit-label term))
		   ((iprecedence-equal-term-p term)
		    (visit-eq (icar term))
		    (visit-eq (icdr term)))
		   (t (note-error (error-message '(equal) term)))))

	       (visit-tree (term)
		 (cond
		   ((iprecedence-label-term-p term)
		    (visit-label term))
		   ((iprecedence-equal-term-p term)
		    (visit-eq (icar term))
		    (visit-eq (icdr term)))
		   ((iprecedence-unrelated-term-p term)
		    (visit-tree (icar term))
		    (visit-tree (icdr term)))
		   ((iprecedence-ordered-term-p term)
		    (visit-tree (icar term))
		    (visit-tree (icdr term)))
		   ((iprecedence-object-term-p term)
		    (visit-obj term))
		   (t (note-error (error-message '(tree) term)))))
	       )

	(visit-tree term)

	(if error-p
	    (raise-error (error-message nil))
	    (values))))))
