
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;	Some motivations for rule interpeter:
;;;;
;;;;	Less code, simpler code.
;;;;
;;;;	Formalizes presentation in book.
;;;;
;;;;	Need precise definition of what interpreter is.
;;;;	Need specification of what interpreter does.
;;;;


;;;; -docs- (mod lib)
;;;;
;;;;	Rule definition:
;;;;
;;;;	Simple abstract rule definition syntax:
;;;;
;;;;	<rule-def>	: PRIMITIVE-RULE(<goal> <rule> <goal{sub}> list)
;;;;
;;;;  -page-
;;;;
;;;;	The basic mechanism of the rule interpreter is to match and then
;;;;	instantiate.  Thus in this simple syntax, the goal and rule would be
;;;;	matched to produce an environment used to instantiate the subgoals.
;;;;	Matching is based solely on syntax, there is no binding of match
;;;;	variables.  Ambiguous matches result in failure.
;;;;	
;;;;  -page-
;;;;
;;;;	In addition to refinement the rule interpreter supports extraction.
;;;;
;;;;	<goal>		: GOAL(<sequent> <extract>)
;;;;
;;;;	Thus to perform extraction the interpreter will create an environment
;;;;	by matching the extracts of the subgoals of a refined proof and then
;;;;	instantiating the extract of the goal.
;;;;
;;;;  -page-
;;;;
;;;;	All user input to the refinement is contained in the rule. The types
;;;;	and numbers of such input is determined by each rule individually.
;;;;
;;;;	<rule>		: RULE(TOKEN <rule-arg> list)
;;;;
;;;;	The various types of rule-args will be incrementally elaborated below.
;;;;
;;;;  -page-
;;;;
;;;;	Many rules require the ability to choose assumptions and to add, move,
;;;;	or remove assumptions in a sequent of a subgoal: 
;;;;
;;;;	<sequent>	: SEQUENT(<assumption> list <conclusion>)
;;;;	<assumption>	: <assumption-list-variable>
;;;;			| ASSUMPTION(<hidden-p> <variable-id> <type>)
;;;;
;;;;	<rule-arg>	: ASSUMPTION-INDEX
;;;;
;;;;	The assumption indices are used to partition the assumptions into
;;;;	segments to match the definition assumption list. Adjacent assumption
;;;;	list variables are allowed in the definition. Thus an assumption index
;;;;	is used to identify a non-variable assumption to be matched or to
;;;;	identify a position used to partition the assumption list.  Thus the
;;;;	number of assumption indices supplied as rule args must equal the number
;;;;	of non-variable assumptions plus the number of assumption list variable
;;;;	adjacencies. The interpreter does allow adjacent non-variable
;;;;	assumptions, however it does still require an index for each.
;;;;
;;;;	In a rule instance, the interpreter requires indices to be in strict
;;;;	increasing order. Thus there is a correspondence between the index args
;;;;	and the non-variable assumptions or assumption list variable
;;;;	adjacencies. Obviously, the indices in a rule instance for such adjacent
;;;;	non-variable assumptions must differ by one.
;;;;
;;;;	The assumptions are numbered beginning with 1 from left to right. The
;;;;	partition points for list adjacencies occur after the indexed
;;;;	assumption.  Thus, an index of 0 may be used to partition off the nil
;;;;	list from the start of the assumption list in the case of an assumption
;;;;	list variable adjacency at the start of the sequent.  Similarly, an
;;;;	index of n in an assumption list of length n may be used to partition
;;;;	off the nil list from the end of the assumption list.
;;;;   
;;;;
;;;;  -page-
;;;;
;;;;	A few rules require the ability to match a term at multiple levels, ie a
;;;;	rule may need to instantiate a term in one place but may also need to
;;;;	instantatiate a subterm of the term in another. One way to accomplish
;;;;	this is to match as finely as needed and then instantiate the larger
;;;;	term from its parts.  Another, method is to allow a destructuring bind
;;;;	as follows:
;;;;
;;;;	<let>		: LET <let-clause> list
;;;;	<let-clause>	: <term{lhs}> <term{rhs}>
;;;;
;;;;	where the lhs is matched against the instantiation of the rhs.
;;;;	The environment is incrementally built so that succeeding let clauses
;;;;	can use preceeding lhs variables on the rhs.
;;;;
;;;;  -page-
;;;;
;;;;	Also, there may be semantic restrictions or computations which
;;;;	are difficult, ugly, or expensive to express syntactically.
;;;;	Thus, there is a hook allowing a rule to invoke a procedure:
;;;;
;;;;	<let-clause>	| <rule-arg> list CALL <function-spec>
;;;;
;;;;	[function(<function-spec>)](<proof-node> <rule>)	: <term> list
;;;;	 * the proof node arg is the proof node being refined while the rule arg
;;;;	   is the rule being interpreted.
;;;;	 * the function will be called at refinement, and may be called at extract. 
;;;;	   Thus it should have no side effects, and should depend only on its 
;;;;	   arguments and not on any global state.
;;;;
;;;;	The <let> is inserted between the rule and subgoals in the rule definition:
;;;;
;;;;	<rule-def>	: PRIMITIVE-RULE(<goal> <rule> <let> <goal> list)
;;;;
;;;;	The interpretation of the let is performebd after the match of the goal and
;;;;	rule within the environment built by those matches.
;;;;
;;;;  -page-
;;;;
;;;;	Most of the data in rule definitions are terms or term components:
;;;;
;;;;	<conclusion>	: <term>
;;;;	<type>		: <term>
;;;;	<extract>	: <var>  | axiom()
;;;;
;;;;
;;;;	Thus it is neccessary for the interpreter to match and instantiate terms
;;;;	and parameter values.  We can choose syntax for the interpreter
;;;;	to recognize as match variables:
;;;;
;;;;	<term>			: <term-aux>
;;;;				| TERM-VARIABLE(TOKEN)
;;;;	<parameter-value>	: PARAMETER-VARIABLE(TOKEN)
;;;;				| REAL-PARAMETER-VALUE
;;;;
;;;;	Due to their ubiguity, parameter values of type variable are treated specially:
;;;;
;;;;	<variable-id>		: VARIABLE-VARIABLE(TOKEN)
;;;;
;;;;	Note that literal variable values are not supported.
;;;;	
;;;;  -page-
;;;;
;;;;	It is important for a rule to be able to discriminate on whether a
;;;;	particular binding is actually binding any variable term occurences.  If
;;;;	the dummy variable id, ie the variable id whose name is null, occurs in
;;;;	a binding position then it will only match an instance which does not
;;;;	bind. In such a match, no value is actually assigned to the dummy id in
;;;;	the match environment, thus dummy variable id can be used to match
;;;;	multiple bindings. As a consequence, the dummy id is treated as a
;;;;	literal when it occurs in an instantiation region.  This is primarily
;;;;	why we opted for the destructuring let to accomplish multi-layer
;;;;	matching, since when the dummy variable id is used to match a binding,
;;;;	it can not be used to instantiate the binding.
;;;;	
;;;;  -page-
;;;;
;;;;	Terms, variables, and parameter values can be passed as rule args:
;;;;	
;;;;	<rule-arg>	| TERM-VARIABLE(TOKEN)
;;;;			| VARIABLE-VARIABLE(TOKEN)
;;;;			| PARAMETER-VARIABLE(TOKEN)
;;;;	
;;;;	As a convenience, a rule may pass in a list of variables and a term as a bound-term:
;;;;
;;;;	<rule-arg>	| BOUND-ID(<variable-id> list <term>)
;;;;
;;;;	Assumption lists can also be passed as rule args:
;;;;
;;;;	<rule-arg>	| <assumption-list-variable>
;;;;	
;;;;  -page-
;;;;
;;;;	The ability to perform substitution during interpretation is also supplied:
;;;;	Refinement time substitution may be specified in the rule definition, the
;;;;	substitution list may be passed as a rule arg:
;;;;
;;;;	<subst>			: SUBST(<term> <substitution-list>)
;;;;
;;;;	<substitution-list>	: SUBSTITUTION_LIST_VARIABLE(TOKEN)
;;;;				| <sub> list
;;;;
;;;;	<sub>			: TERM-SUB(<variable-id> <term>)
;;;;				| PARAMETER-SUB(PARAMETER-VARIABLE(TOKEN) REAL-PARAMETER-VALUE)
;;;;
;;;;	<term-aux>		: <subst>
;;;;				| TERM
;;;;
;;;;  -page-
;;;;
;;;;	Substituting over a list of assumptions is supported by allowing an assumption list
;;;;	variable as a target of substitution.
;;;;
;;;;	<assum-subst>		: ASSUM-SUBST(<assumption-list-variable-aux> <sub> list)
;;;;
;;;;	<assumption-list-variable>	: <assumption-list-variable-aux>
;;;;					| <assum-subst>
;;;;	<assumption-list-variable-aux>	: ASSUMPTION_LIST_VARIABLE(TOKEN)
;;;;
;;;;
;;;;	Substitution args can be passed in as a rule arg:
;;;;
;;;;	<rule-arg>		| <substitution-list>	
;;;;
;;;;	If a subst is encountered during a match, a note is made. Then when the
;;;;	match is otherwise complete; the subst is instantiated, the substitution
;;;;	is performed, and the result is then compared for alpha-equality to the
;;;;	instance.  Note that subst's are not allowed on the rhs of let clauses.
;;;;	However, if the need arose, it would be a simple matter to extend the
;;;;	interpreter.
;;;;
;;;;	When assumption list variable is the target of substitution, the
;;;;	substitution is applied to the ids and the types of all assumptions
;;;;	matched to the variable.
;;;;
;;;;  -page-
;;;;
;;;;	A couple features of dubious value:
;;;;
;;;;	<rule-arg>	| ASSUMPTION_LIST_VARIABLE(TOKEN)
;;;;			| GOAL_LIST_VARIABLE
;;;;
;;;;	These allow some rules not otherwise expressible in this syntax to be
;;;;	coerced into the syntax.
;;;;
;;;;  -page-
;;;;
;;;;	The rule interpreter guarauntees some semantic side effects:
;;;;
;;;;	* Subgoal sequents are closed. This is accomplished through a refinement
;;;;	  time check at instantiation of the sequent. In some instances, it is
;;;;	  be possible for the interpreter to deduce that the check is
;;;;	  unneccessary.
;;;;
;;;;	* It is necessary that an assumption id which is hidden in a sequent instance 
;;;;	  not occur in the sequent's extract instance. 
;;;;
;;;;	* It is desirable that hidden assumption ids in a sequent instance be
;;;;	  unhidden if there is no possibility that the sequent will contribute to
;;;;	  the extract of the larger proof.
;;;;
;;;;	The control of hidden variables is relatively complex.  It is beyond the
;;;;	scope of this documentation to explain the purpose of hidden variables,
;;;;	however their behaviours can be explained.
;;;;
;;;;	An assumption id becomes hidden when an assumption is instantiated
;;;;	from a a spec with the hidden bool true, eg !assumption{t:bool, v:v}(t).
;;;;
;;;;	An assumption id is unhidden upon instantiation of a subgoal when the
;;;;	interpreter can deduce through syntactic analysis of the rule definition
;;;;	that the extract of the subgoal can not contribute to the extract of the
;;;;	goal:
;;;;	 * Subgoal extract is axiom().
;;;;	 * Goal extract is axiom().
;;;;	 * Subgoal extract is variable{v} but variable does not occur free in
;;;;	   goal extract.
;;;;
;;;;	There is also a semantic requirement that any goal whose conclusion is a
;;;;	top level operator which is of equal(0; 0; 0), less-than(0;0), or
;;;;	void(), will extract axiom. Thus when refiner instantiates such a
;;;;	subgoal it can unhide any hidden assumptions.
;;;;	
;;;;	During a refinement, the property of being hidden is associated with the
;;;;	assumption ids.  When a subgoal assumption is being instantiated the
;;;;	assumption inherits the hidden property of the id instance in the
;;;;	assumption. If the hidden flag is on in the definition template for the
;;;;	assumption, then bit is coerced on in the instance. Note that the
;;;;	unhiding takes precedence, ie in either case the assumption will not be
;;;;	hidden if unhiding is in effect for the subgoal.
;;;;	
;;;;	When matching the goal sequent, if the hidden flag is on in an
;;;;	assumption of the rule-definition, then the assumption must be hidden in
;;;;	the instance.
;;;;	
;;;;	To ensure that no hidden id occurs in the extract, the interpreter must
;;;;	check that all terms bound to !term{v} operators occuring free in the
;;;;	extract do not contain free occurences of hidden ids at refinement time.
;;;;	Note that we know extracts of sugoals will not contain variables that
;;;;	are hidden in the subgoal.
;;;;
;;;;	Note that the interpreter can make no other semantic claims of
;;;;	correctness concerning defined rules.
;;;;
;;;;  -page-
;;;;
;;;;	Summary of abstract rule-interpreter syntax: 
;;;;
;;;;	<rule-spec>	: PRIMITIVE-RULE(<goal> <rule> <let> <goal> list)
;;;;
;;;;	<goal>		: GOAL(<sequent> <extract>)
;;;;
;;;;	<rule>		: RULE(TOKEN <rule-arg> list)
;;;;
;;;;	<sequent>	: SEQUENT(<assumption> list <conclusion>)
;;;;
;;;;	<assumption>	: <assumption-list>
;;;;			| ASSUMPTION(<hidden-p> <variable-id> <type>)
;;;;
;;;;	<assumption-list>	: ASSUMPTION_LIST_VARIABLE(TOKEN)
;;;;				| ASSUM-SUBST(<assumption-list> <substitution-list>)
;;;;
;;;;	<rule-arg>		: <term>
;;;;				| ASSUMPTION-INDEX(TOKEN)
;;;;				| TERM-VARIABLE(TOKEN)
;;;;				| VARIABLE-VARIABLE(TOKEN)
;;;;				| PARAMETER-VARIABLE(TOKEN)
;;;;				| BOUND-ID(<varible-id> list <term>)
;;;;				| SUBSTITUTION_LIST_VARIABLE(TOKEN)
;;;;				| ASSUMPTION-LIST-VARIABLE(TOKEN)
;;;;				| GOAL_LIST_VARIABLE(TOKEN)
;;;;
;;;;	<let>			: LET <let-clause> list
;;;;	<let-clause>		: <term{lhs}> <term{rhs}>
;;;;				| <rule-arg> list CALL <function-spec>
;;;;
;;;;	<conclusion>		: <term>
;;;;	<type>			: <term>
;;;;	<extract>		: <term>
;;;;
;;;;	<subst>			: SUBST(<term> <substitution-list>)
;;;;
;;;;	<substitution-list>	: SUBSTITUTION_LIST_VARIABLE(TOKEN)
;;;;				| <sub> list
;;;;
;;;;	<sub>			: TERM-SUB(<variable-id> <term>)
;;;;				| PARAMETER-SUB(PARAMETER-VARIABLE(TOKEN) REAL-PARAMETER-VALUE)
;;;;
;;;;	<term>			: <subst>
;;;;				| TERM-VARIABLE(TOKEN)
;;;;				| TERM
;;;;	<variable-id>		: VARIABLE-VARIABLE(TOKEN)
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	This can be expressed directly as a term: 
;;;; -doct- (mod lib data)
;;;;
;;;;	<rule-spec-term>	: !rule_specification(<sequent>;
;;;;						      <rule>;
;;;;						      <let> !let_cons list;
;;;;						      <subgoals>)
;;;;
;;;;	<sequent>		: !sequent(<assumption> !assum_cons list; <term>; <term>)
;;;;
;;;;	<rule>			: !rule{<tok>:t}(<rule-arg> !rule_arg_cons list)
;;;;
;;;;	<assumption>		: <assumption-list>
;;;;				| !assumption{<hidden>:b,<id>:v}(<type>)
;;;;
;;;;	<assumption-list>	: !assumption_list{<tok>:t}
;;;;				| !subst(<assumption-list>; <sub> !sub_cons list)
;;;;
;;;;	<rule-arg>		: <term>
;;;;				| !assumption_index{<tok>:t}
;;;;				| !term{<id>:v}
;;;;				| !variable{<id>:v}
;;;;				| !parameter{<id>:<type>}
;;;;				| !bound_id(<id> list.<term>)
;;;;				| !substitution_list{<tok>:t}
;;;;				| !assumption_list{<tok>:t}
;;;;
;;;;	<let>			: !let(<term>; <term>)
;;;;				| !let(<let-lhs-arg> !rule_arg_cons list; !call_lisp{<tok>:t})
;;;;
;;;;	<let-lhs-arg>		: <rule-arg>
;;;;				| !goal_list{<tok>:t}
;;;;
;;;;	<subgoals>		: <sequent> !goal_cons list
;;;;				| !goal_list{<tok>:t}
;;;;
;;;;	<term>			: <subst>
;;;;				| !term{<id>:v}
;;;;				| !variable{<id>:v}
;;;;				| term
;;;;
;;;;	<subst>			: !subst(<term>; <substitution-list>)
;;;;
;;;;	<substitution-list>	: !substitution_list{<tok>;t}
;;;;				| <sub> !sub_cons list
;;;;
;;;;	<sub>			: !term_sub{<v>:v}(<term>)
;;;;				| !parameter_sub{<v>:token, <le-value>:l}
;;;;
;;;; -doct- (mod lib)
;;;;	** changed from		| !parameter_sub{<le-var>:l, <le-value>:l}
;;;;	**			| !parameter_sub{$<id>:<type>, <value>:<type>}
;;;;
;;;;	This is the syntax that the rule interpreter expects. 
;;;;
;;;;	Note that it is impossible to pass some terms as data to the interpreter
;;;;	as they are part of the syntax. Eg, a !subst term can never be literally
;;;;	matched against as it will always be interpreted as a substititution to
;;;;	be performed. The syntax operators where chosen such that it is highly
;;;;	unlikely that one would ever desire such a thing.
;;;;
;;;;  -page-
;;;;
;;;;	Though the preceeding syntax is unambiguous, it is not as concise as one
;;;;	might desire. For this reason, the input of rule definition is in a
;;;;	different syntax with much overloading of the variable{<id>:v} terms in
;;;;	match positions.
;;;;
;;;;	This is the source syntax for rule definitions:
;;;;
;;;;	<rule-def>		: !rule_definition(<sequent>; <rule>; <let>; <subgoals>)
;;;;
;;;;	<sequent>		: !sequent(<assumption> !assum_cons list; <subst>; <subst>)
;;;;
;;;;	<rule>			: !rule{<tok>:t}(<rule-arg> !rule_arg_cons list)
;;;;
;;;;	<assumption>		: <assumption-list>
;;;;				| !assumption{<hidden>:b,<id>:v}(<subst>)
;;;;
;;;;	<assumption-list>	: <var>
;;;;				| !subst(<var>; <substitution-list>)
;;;;
;;;;	<rule-arg>		: <term>
;;;;				| <var>
;;;;				| assumption_index{<>:n}
;;;;				| !bound_id(<id> list.<subst>)
;;;;
;;;;	<let>			: !let(<subst>; <subst>)
;;;;				| !let(<rule-arg> !rule_arg_cons list; !call_lisp{<tok>:t})
;;;;				| !cons()
;;;;
;;;;	<subgoals>		: <sequent> !subgoal_cons list
;;;;				| <var>
;;;;
;;;;	<subst>			: !subst(<term>;  <var>.<term> list)
;;;;				: !parameter_subst(<term>; <substitution-list>)
;;;;
;;;;	<substitution-list>	: <var>
;;;;				| <sub> !sub_cons list
;;;;
;;;;	<sub>			: !parameter_sub{<le-var>:l, <le-value>:l}
;;;;				| !parameter_sub{$<id>:<type>, <value>:<type>}
;;;;
;;;;	<var>			: variable{<v>:v}
;;;;
;;;;  -page-
;;;;
;;;;	Note variables are overloaded for use in various syntactic roles.
;;;;	Variable terms, ie variable{x:v}(), will match terms. Except for
;;;;	parameters of type variable, abstraction meta variables will match
;;;;	parameter values. Variable-ids, ie the real values for parameters with
;;;;	type variable, will be used to match parameter values of type variable.
;;;;	This includes parameter values in binding positions.  If a variable
;;;;	term's variable-id is used elsewhere in the rule definition in a
;;;;	parameter value position, then it will be treated as a parameter match
;;;;	and not a term match.  In the case where a variable term is meant
;;;;	literally and the variable id is not used elsewhere in a parameter value
;;;;	position, then the variable term can use the abstraction meta variable,
;;;;	ie variable{$v:v}(). Variable terms will also be used in place of
;;;;	!goal-list and !assumption-list terms.
;;;;
;;;;	Such source is examined and translated to the intepreter syntax. The
;;;;	rule interpreter may then further translate the definition to an
;;;;	internal syntax. Rule instances will be created functionally through the
;;;;	tactic interface to the refiner and will not use the same syntax (see
;;;;	the refiner tactic doc). ML's strict type system provides a measure of
;;;;	validity that the input into the rule interpreter through the rules is
;;;;	proper. It is checked that the argument types match the expected rule
;;;;	arg types.  However there is no need to check that the argument values
;;;;	match the types.
;;;;
;;;;	Some may prefer a different source syntax. One need only provide a
;;;;	translater to the interpreter term syntax.
;;;;
;;;;	TODO : the assumption_index rule-def arg term is shown as:
;;;;				| assumption_index{<tok>:n}
;;;;	 in the implementation it is assumption-index{<i>:n}
;;;;	 neither is appealing. Want ! _ but not to interfere with !assumption_index{<tok>:t}
;;;;     of rule-spec.
;;;;
;;;;  -page-
;;;;
;;;;	rule-definition-translate-source(<term>)		: <term{specification}>
;;;;
;;;;	rule-definition-check(<term{specification>)		: <bool>
;;;;
;;;; -doce-

;;; in lib rule def should be source-reduce'd source term.
;;; in ref there will be more.

;;; RLE NAP might want to add !compute op to rule def to reduce calls for computation??
;;; RLE NAP allow multiple let clauses.


;;;; ***RLE TODO this parameter_sub in rule-def stuff differs from the implmentation.
;;;;	<subst>		: !subst(<term>; <sub> !sub_cons list)
;;;;	<sub>		: !term_sub{<v>:v}(<term>)
;;;;			| !parameter_sub{<var>:<type>, <value>:<type>}()
;;;;
;;;;	;;<op_subst>	: !substitute_in_operator{<psl>:psl}()
;;;;

;; rle note migrate !subst changed.

;; ? change subst form? only in interp syntax??????? NAP in input syntax.
;; ? input syntax? - same except no psl (add something special) and add bound-term
;; ? change subst form in interp syntax? ie subst(Z; z.t) -> subst(z.Z; t) : RLE NAP
;; If interp syntax is tailored to machine then such a change is not desired as current syntax more natural.

;; then normal rule def :
;; check input syntax
;; translate to interp syntax.
;; check interp syntax.

;; or if input directly in interp syntax
;; the just check interp syntax.


;;;	RLE NAP could allow negative assumption indices to be index from right.

;; RLE TODO put rule def term defs in com-ref.

;; rle todo can test this anytime.

;;;;
;;;;	Operators of rule source syntax not used in rule interpreter syntax.
;;;;
;;;; RLE TODO

;; possible types for variables as terms:
;; Term
;; Binding
;; Assumption List
;; Subgoals
;; Assumption Index

;; Abstraction meta vars can be
;; Parameter (type)
;; if type if variable then assigns term version binding.

;;; rle todo doc fact that level expressions are not matched unless def expression is simple variable.
;;; they are otherwise just checked for equality.

(defun rule-definition-translate-source (def)
  ;; coerce vars and type to system symbols.
  (let ((env nil))
    (labels
	((assign-type (v type)
	   (let* ((var (intern-system (string v)))
		  (cur-type (cdr (assoc var env))))

	     (when (display-meta-variable-id-p v)
	       (raise-error (error-message (list 'rule-definition 'translate 'display-meta var))))

	     (cond
	       ((or (null cur-type)
		    (and (eql type *variable*) (eql cur-type 'term)))
		(setf env (acons var type env)))
	       ((or (and (eql cur-type *variable*) (eql type 'term))
		    (eql type cur-type))
		nil)
	       (t (raise-error (error-message (list 'rule-definition 'translate 'type var type cur-type)))))))

	 (visit-parameter (parameter)
	   (let ((value (value-of-parameter-m parameter)))
	     (when (if (level-expression-parameter-p parameter)
		       (level-variable-p value)
		       (abstraction-meta-variable-id-p value))
	       (assign-type value (type-id-of-parameter parameter)))))

	 (visit-old-subst (term)
	   (visit-term term))
	   
	 (visit-subs (subs)
	   (map-isexpr subs
		       (isub-cons-op)
		       #'(lambda (term)
			   (cond
			     ((iparameter-sub-term-p term)
			      (value-of-parameter-r (variable-of-iparameter-sub-term term)))
			     ((iterm-sub-term-p term)
			      (variable-id-p (id-of-iterm-sub-term term))
			      (visit-term (term-of-iterm-sub-term term)))
			     ((variable-p term)
			      (assign-type (id-of-variable-term term) 'substitution-list))
			     (t (raise-error (error-message '(rule-definition translate assumption sub)
							    term)))))))
	 (visit-subst (term)
	   (visit-term (term-of-isubst-term term))
	   (visit-subs (subs-of-isubst-term term)))

	 (visit-rule-arg (term)
	   (cond
	     ((isubst-term-p term)
	      (visit-subst term))
	     (t (dolist (p (parameters-of-operator (operator-of-term term)))
		  (visit-parameter p))
		(dolist (bt (bound-terms-of-term term))
		  (visit-bound-term bt)))))
	 
	 (visit-term (term)
	   (cond
	     ((variable-p term)
	      (assign-type (id-of-variable-term term) 'term))
	     ;;((iold-subst-term-p term)
	     ;;(visit-old-subst term))
	     ((isubst-term-p term)
	      (visit-subst term))
	     (t (dolist (p (parameters-of-operator (operator-of-term term)))
		  (visit-parameter p))
		(dolist (bt (bound-terms-of-term term))
		  (visit-bound-term bt)))))

	 (visit-bound-term (bound-term)
	   (dolist (v (bindings-of-bound-term-r bound-term))
	     (assign-type v *variable*))
	   (visit-term (term-of-bound-term bound-term)))

	 (visit-assumption (term &optional seen-subst-p)
	   (cond
	     ((iassumption-term-p term)
	      (assign-type (id-of-iassumption-term term) *variable*)
	      (visit-term (type-of-iassumption-term term)))
	     ((isubst-term-p term)
	      (visit-assumption (term-of-isubst-term term))
	      (visit-subs (subs-of-isubst-term term)))
	     ((iold-subst-term-p term)
	      (when seen-subst-p
		(raise-error (error-message '(rule-definition translate assumption subst) term)))
	      (visit-assumption (subbee-of-iold-subst-term term) t)
	      (dolist (bt (subbends-of-iold-subst-term term))
		(visit-bound-term bt)))
	     ((variable-p term)
	      (assign-type (id-of-variable-term term) 'assumption-list))
	     (t (raise-error (error-message '(rule-definition translate assumption) term)))))

	 (visit-sequent (term)
	   (unless (isequent-term-p term)
	     (raise-error (error-message '(rule-definition translate sequent) term)))
	   (map-isexpr (assumptions-of-isequent-term term)
		       (iassum-cons-op)
		       #'visit-assumption)
	   (visit-term (conclusion-of-isequent-term term))
	   (visit-term (extract-of-isequent-term term)))

	 ;; H, x:t, J, K, y:s, a:r, L
	 ;;     i    j     k   k+1
	 (check-assumption-indices (assumptions rule)
	   (let ((l (do ((i 0)
			 (assums (map-isexpr-to-list assumptions (iassum-cons-op))
				 (cdr assums)))
			((null assums) i)
		      (when (or (iassumption-term-p (car assums))
				(and (not (iassumption-term-p (car assums)))
				     (cdr assums)
				     (not (iassumption-term-p (cadr assums)))))
			(incf i))))
		 (m (let ((i 0))
		      (map-isexpr (args-of-irule-term rule)
				  (irule-arg-cons-op)
				  #'(lambda (term)
				      (when (assumption-index-term-p term)
					(incf i))))
		      i)))
	   
	     (unless (= l m)
	       ;;(setf a l b m c assumptions d rule)
	       (raise-error (error-message '(rule-definition translate assumption indices)
					   (princ-to-string l)
					   (princ-to-string m)))))))

      (visit-sequent (goal-of-irule-definition-term def))

      (let ((term (rule-of-irule-definition-term def)))
	(cond
	  ((irule-term-p term)
	   (map-isexpr (args-of-irule-term term) (irule-arg-cons-op) #'visit-rule-arg)
	   )
	  (t (raise-error (error-message '(rule-definition translate rule) term)))))
     
      (let ((term (let-of-irule-definition-term def)))
	(cond
	  ((ilet-term-p term)
	   (map-isexpr (lhs-of-ilet-term term) (irule-arg-cons-op) #'visit-rule-arg)
	   (visit-term (rhs-of-ilet-term term)))
	  ((or (ivoid-term-p term)
	       (inil-term-p term))
	   nil)
	  (t (raise-error (error-message '(rule-definition translate let) term)))))
    
      (if (variable-p (subgoals-of-irule-definition-term def))
	  (assign-type (id-of-variable-term (subgoals-of-irule-definition-term def)) 'goal-list)
	  (map-isexpr (subgoals-of-irule-definition-term def) (isubgoal-cons-op)
		      #'visit-sequent))
       
      (check-assumption-indices (assumptions-of-isequent-term 
				 (goal-of-irule-definition-term def))
				(rule-of-irule-definition-term def))

      (rule-definition-translate def env))))

(define-primitive |!term| ((variable . id)))

;; quote reserved ops of interp syntax.
;; translate others.
(defun rule-definition-translate (term env)
  (labels
      ((lookup-type (v)
	 (cdr (assoc (intern-system (string v)) env)))

       (visit-parameter (p) p)
       
       (visit-old-subst (term)
	 (isubst-term  (visit-term (subbee-of-iold-subst-term term))
		       (map-sexpr-to-isexpr (mapcar #'(lambda (bt)
							(iterm-sub-term (car (bindings-of-bound-term bt))
									(visit-term (term-of-bound-term bt))))
						    (cdr (bound-terms-of-term term)))
					    (isub-nil-term))))

       (visit-subs (term)
	 (terpri)
	 (if (isub-nil-term-p term)
	     term
	     (if (isub-cons-term-p term)
		 (isub-cons-term (visit-subs (icar term))
				 (visit-subs (icdr term)))
		 (visit-term term))))
			
       (visit-term (term)
	 (cond
	   ((variable-p term)
	    (let ((id (id-of-variable-term term)))
	      (case (lookup-type id)
		(|variable| (ivariable-term id))
		(term (iterm-term id))
		(assumption-index (iassumption-index-term (intern-system (string id))))
		(assumption-list (iassumption-list-term (intern-system (string id))))
		(goal-list (igoal-list-term (intern-system (string id))))
		(substitution-list (isubstitution-list-term (intern-system (string id))))
		(otherwise (iterm-term id)))))

	   ((assumption-index-term-p term)
	    (iassumption-index-term (intern-system (string (index-of-assumption-index-term term)))))
	    
	   ((ibound-id-term-p term)
	    (ibound-id-term (bindings-of-term-of-ibound-id-term term)
			    (visit-term (term-of-ibound-id-term term))))
	   
	   ((iold-subst-term-p term)
	    (visit-old-subst term))

	   ((isubst-term-p term)
	    (isubst-term (visit-term (term-of-isubst-term term))
			 (visit-subs (subs-of-isubst-term term))))

	   (t (instantiate-term (operator-of-term term)
				(mapcar #'(lambda (bt)
					    (instantiate-bound-term (visit-term
								     (term-of-bound-term bt))
								    (bindings-of-bound-term-r bt)))
					(bound-terms-of-term term))))))
	 

       (visit-sequent (term)
	 (isequent-term (visit-assumptions (assumptions-of-isequent-term term))
			(visit-term (conclusion-of-isequent-term term))
			(visit-term (extract-of-isequent-term term))))

       (visit-assumptions (term)
	 (if (iassum-cons-term-p term)
	     (iassum-cons-term (visit-assumptions (icar term))
			       (visit-assumptions (icdr term)))
	     (if (iassum-nil-term-p term)
		 term
		 (visit-assumption term))))
       
       (visit-assumption (term)
	 (if (iassumption-term-p term)
	     (iassumption-term (hidden-of-iassumption-term term)
			       (id-of-iassumption-term term)
			       (visit-term (type-of-iassumption-term term)))
	     (visit-term term)))


       (visit-rule (term)
	 (irule-term (id-of-irule-term term)
		     (visit-rule-args (args-of-irule-term term))))

       (visit-rule-args (term)
	 (if (irule-arg-nil-term-p term)
	     term
	     (if (irule-arg-cons-term-p term)
		 (irule-arg-cons-term (visit-rule-args (icar term))
				      (visit-rule-args (icdr term)))
		 (visit-term term))))

       (visit-let (term)
	 (if (inil-term-p term)
	     (ilet-nil-term)
	     (visit-term term)))

       (visit-subgoals (term)
	 (if (or (isubgoal-cons-term-p term)
		 (isubgoal-nil-term-p term))
	     (visit-subgoal-list term)
	     (if (isequent-term-p term)
		 (visit-sequent term)
		 ;; goal_list
		 (visit-term term))))

       (visit-subgoal-list (term)
	 (if (isubgoal-nil-term-p term)
	     (igoal-nil-term)
	     (if (isubgoal-cons-term-p term)
		 (igoal-cons-term (visit-subgoal-list (icar term))
				  (visit-subgoal-list (icdr term)))
		 (visit-sequent term)))))

    (irule-specification-term
     (visit-sequent (goal-of-irule-definition-term term))
     (visit-rule (rule-of-irule-definition-term term))
     (visit-let (let-of-irule-definition-term term))
     (visit-subgoals (subgoals-of-irule-definition-term term)))))

 

;;; RLE NAP call-lisp before matching goals sequent to get asummption indices??

(defun rule-definition-check (term)
  (let ((env nil)
	(indices nil))
    
    (labels
	(
	 ;; check for duplicate use of id.
	 ;; another method would be to collect all then spit out list of incompatibilities.
	 (assign-type (v type)
	   (let* ((var (intern-system (string v)))
		  (cur-type (cdr (assoc var env))))

	     (when (display-meta-variable-id-p v)
	       (raise-error (error-message (list 'rule-definition 'check 'display-meta var))))

	     (when (eql type 'assumption-index)
	       (push var indices))

	     (unless (or (null cur-type)
			 (eql type cur-type))
	       (raise-error (error-message (list 'rule-definition 'check 'type var type cur-type))))))

	 (visit-parameter (parameter)
	   (let ((value (value-of-parameter-m parameter)))
	     (when (if (level-expression-parameter-p parameter)
		       (level-variable-p value)
		       (abstraction-meta-variable-id-p value))
	       (assign-type value (type-id-of-parameter parameter)))))

	 (visit-subs (term)
	   (if (isubstitution-list-term-p term)
	       (assign-type (id-of-isubstitution-list-term term) 'substitution-list)
	       (map-isexpr (subs-of-isubst-term term)
			   (isub-cons-op)
			   #'(lambda (term)
			       (cond
				 ((iparameter-sub-term-p term)
				  (value-of-parameter-r (variable-of-iparameter-sub-term term)))
				 ((iterm-sub-term-p term)
				  (variable-id-p (id-of-iterm-sub-term term))
				  (visit-term (term-of-iterm-sub-term term)))
				 (t (raise-error (error-message '(rule-definition check assumption sub) term))))))))
       
	 (visit-term (term)
	   (cond
	     ((iterm-term-p term)
	      (assign-type (id-of-iterm-term term) 'term))
	     ((ivariable-term-p term)
	      (assign-type (id-of-ivariable-term term) *variable*))
	     ((isubst-term-p term)
	      (visit-term (term-of-isubst-term term))
	      (visit-subs (subs-of-isubst-term term)))
	     (t (dolist (p (parameters-of-operator (operator-of-term term)))
		  (visit-parameter p))
		(dolist (bt (bound-terms-of-term term))
		  (visit-bound-term bt)))))

	 (visit-bound-term (bound-term)
	   (dolist (v (bindings-of-bound-term-r bound-term))
	     (assign-type v *variable*))
	   (visit-term (term-of-bound-term bound-term)))

	 (visit-assumption-list (term)
	   (cond
	     ((isubst-term-p term)
	      (visit-assumption (term-of-isubst-term term))
	      (visit-subs (subs-of-isubst-term term)))
	     ((iassumption-list-term-p term)
	      (assign-type (id-of-iassumption-list-term term) 'assumption-list))
	     (t (raise-error (error-message '(rule-definition check assumption) term)))))

	 (visit-assumption (term)
	   (if (iassumption-term-p term)
	       (progn 
		 (assign-type (id-of-iassumption-term term) *variable*)
		 (visit-term (type-of-iassumption-term term)))
	       (visit-assumption-list term)))
	   
	 (visit-sequent (term)
	   (unless (isequent-term-p term)
	     (raise-error (error-message '(rule-definition check sequent) term)))
	   (map-isexpr (assumptions-of-isequent-term term)
		       (iassum-cons-op)
		       #'visit-assumption)
	   (visit-term (conclusion-of-isequent-term term))
	   (visit-term (extract-of-isequent-term term)))

	 (visit-rule (term)
	   (cond
	     ((irule-term-p term)
	      (map-isexpr (args-of-irule-term term) (irule-arg-cons-op) #'visit-rule-arg))
	     (t (raise-error (error-message '(rule-definition check rule) term)))))

	 (visit-rule-arg (term)
	   (cond
	     ((iassumption-index-term-p term)
	      (assign-type (id-of-iassumption-index-term term) 'assumption-index))
	     ((iterm-term-p term)
	      (assign-type (id-of-iterm-term term) 'term))
	     ((ivariable-term-p term)
	      (assign-type (id-of-ivariable-term term) *variable*))
	     ((ibound-id-term-p term)
	      (visit-bound-term (car (bound-terms-of-term term))))
	     ((iparameter-term-p term)
	      (let* ((p (parameter-of-iparameter-term term))
		     (value (value-of-parameter-m p)))
		(if (level-expression-parameter-p p)
		    (if (level-variable-p value)
			(assign-type value (type-id-of-parameter p))
			(raise-error (error-message '(rule-definition check rule-arg parameter) term)))
		    (if (abstraction-meta-variable-id-p value)
			(assign-type value (type-id-of-parameter p))
			(raise-error (error-message '(rule-definition check rule-arg parameter) term))))))
	     ((isubstitution-list-term-p term)
	      (assign-type (id-of-isubstitution-list-term term) 'substitution-list))
	     ((iassumption-list-term-p term)
	      (assign-type (id-of-iassumption-list-term term) 'assumption-list))
	     (t (visit-term term))))

       
	 (visit-let (term)
	   (map-isexpr term
		       (ilet-cons-op)
		       #'(lambda (term)
			   (unless (ilet-term-p term)
			     (raise-error (error-message '(rule-definition check let) term)))
			   (if (icall-lisp-term-p (rhs-of-ilet-term term))
			       (map-isexpr (lhs-of-ilet-term term)
					   (irule-arg-cons-op)
					   #'(lambda (term)
					       (if (igoal-list-term-p term)
						   (assign-type (id-of-igoal-list-term term) 'goal-list)
						   (visit-rule-arg term))))
			       (visit-term (lhs-of-ilet-term term))))))
    

	 (visit-subgoals (term)
	   (if (igoal-list-term-p term)
	       (assign-type (id-of-variable-term term) 'goal-list)
	       (map-isexpr term
			   (igoal-cons-op)
			   #'visit-sequent)))
       
	 ;; H, x:t, J, K, y:s, a:r, L
	 ;;     i    j     k   k+1
	 (check-assumption-indices (assumptions)
	   (let ((l (do ((i 0)
			 (assums (map-isexpr-to-list assumptions (iassum-cons-op))
				 (cdr assums)))
			((null assums) i)
		      (when (or (iassumption-term-p (car assums))
				(and (not (iassumption-term-p (car assums)))
				     (cdr assums)
				     (not (iassumption-term-p (cadr assums)))))
			(incf i))))
		 (m (length (remove-duplicates indices))))
		 	   
	     (unless (= l m)
	       (raise-error (error-message '(rule-definition check assumption indices)
					   (princ-to-string l)
					   (princ-to-string m)))))))

      (visit-sequent (goal-of-irule-definition-term term))

      (visit-rule (rule-of-irule-definition-term term))

      (visit-let (let-of-irule-definition-term term))
     
      (visit-subgoals (subgoals-of-irule-definition-term term))
       
      (check-assumption-indices (assumptions-of-isequent-term 
				 (goal-of-irule-definition-term term)))))

  t)



;;;;	Following is used to fixup 4.2 rules to conform to v5 !rule-definition syntax.
;;;;	
;;;;	in rule args:
;;;;
;;;;	assumption-index -> !assumption_index ???
;;;;
;;;;	level-expression -> !paramater
;;;;	token 
;;;;	natural_number
;;;;	

(defun fixup-v4-rule-definition (term)
  (let ((changed nil)
	(anyp nil)
	(fixups '((|level-expression| . |!parameter|)
		  (|token| . |!parameter|)
		  (|natural_number| . |!parameter|)
		  ;;(|assumption-index| . |!assumption_index|)
		  )))
    
    (labels ((visit-rule-args (iargs)
	       ;;(setf c iargs)
	       (let ((args (map-isexpr-to-list iargs
					       (irule-arg-cons-op))))
		 (let ((nargs (mapcar #'(lambda (arg)
					  (let ((fixup (assoc (id-of-term arg) fixups)))
					    ;;(setf a fixup b arg) (break)
					    (if fixup
						(progn
						  (setf changed t)
						  (instantiate-term (instantiate-operator (cdr fixup)
											  (parameters-of-term arg))
								    nil))
						arg)))
				      args)))
		   (if changed
		       (map-list-to-ilist nargs
					  (irule-arg-nil-term))
		       iargs)))))
      
      (let ((rule-args (visit-rule-args (args-of-irule-term (rule-of-irule-definition-term term)))))
	(when changed
	  (setf anyp t)
	  (format t "RuleDef fixup changed ~a.~%" (id-of-irule-term (rule-of-irule-definition-term term))))
	(let ((term2 (if changed
			 (irule-definition-term
			  (goal-of-irule-definition-term term)
			  (irule-term (id-of-irule-term (rule-of-irule-definition-term term))
				      rule-args)
			  (let-of-irule-definition-term term)
			  (subgoals-of-irule-definition-term term))
			 term)))
	  (setf changed nil)
	  (let ((let-args (when (ilet-term-p  (let-of-irule-definition-term term))
			    (visit-rule-args (lhs-of-ilet-term (let-of-irule-definition-term term2))))))
	    (let ((term3 (if changed
			     (irule-definition-term
			      (goal-of-irule-definition-term term)
			      (rule-of-irule-definition-term term2)
			      (ilet-term let-args (rhs-of-ilet-term (let-of-irule-definition-term term)))
			      (subgoals-of-irule-definition-term term))
			     term2)))
	      (when changed
		(setf anyp t)
		(format t "RuleDef fixup changed let ~a.~%" (id-of-irule-term
							     (rule-of-irule-definition-term term))))
	      (cons anyp term3))))))))

(defunml (|rule_def_fixup| (term))
    (term -> (bool |#| term))
  
  (if (irule-definition-term-p term)
      (fixup-v4-rule-definition term)
      (cons nil term)))



  
