%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%
;;;;	
;;;;	dag_make_root		: tok -> object_id
;;;;	dag_make_directory	: object_id -> tok -> object_id
;;;;	
;;;;	dag_insert		: object_id -> tok -> object_id -> unit
;;;;	dag_make_leaf		: object_id -> tok -> tok -> object_id
;;;;	dag_insert_leaf		: object_id -> tok -> tok -> term -> object_id
;;;;	
;;;;	dag_delete		: object_id -> tok -> bool
;;;;	  * does not deactivate.
;;;;	dag_remove_leaf		: object_id -> tok -> bool
;;;;	  * fails if directory. deactivates and removes.
;;;;	
;;;;	remove_property		: object_id -> tok -> unit
;;;;	get_property		: object_id -> tok -> term
;;;;	
;;;;	
;;;;	put_name		: object_id -> tok -> unit
;;;;	get_name		: object_id -> tok
;;;;	
%
% 
	Utils :
% 

let lib_listing oids = 
  map (\oid. (oid,
	     ((objc_kind (lib_object_contents oid)),
	      (lib_active_p oid))))
      oids
;;

let lib_list () = 
  letref acc = [] in 
    (map_library (\o. acc := (o . acc); false)) ; 
    acc;;


% 
	Evalet.
  Probably should require it be active and get reduced term from substance.
%

let oc oid = lib_object_contents oid;;

let lib_eval_term oid =
  let oc = oc oid in
  let term = term_objc_term oc in
  let desc = (objc_property oc `DESCRIPTION`) in
    orb_eval_expression desc term
;;



let blot_objc = blot o link_of_objc;;

let compile_code oid = 
 let objc = lib_object_contents oid in
  let nobjc = compile_code_objc_source objc in
    if (objc = nobjc) then ()
    else ((lib_unbind oid)();
	  (lib_bind_with_history oid nobjc (blot_objc objc))();
	  ())
;;
	  
	
   
%
;;;;
;;;;	Directive wrappers :
;;;;	
;;;;	
%

%
;;; false start on a objc history mechanism:

let history_link_term blot link = 
  make_term (`!history_link_objc`, []) [[],blot; [],link]
;;

let link_of_history_link_term t = subterm_of_term t 2;;
let blot_of_history_link_term t = subterm_of_term t 1;;
 
let objc_link_with_past oc = 
  let link = ((blot_of_history_link_term (objc_property oc `history_link`)) ? ivoid_term)
  and blot = (blot_objc oc) in
    objc_add_property ((objc_remove_property oc `history_link`) ? oc)
		      `history_link`
 		      (history_link_term blot link)
;;

let link_with_past oid = 
 ((lib_weak_bind oid (objc_link_with_past (oc oid)))(); ()) 
 ? (inform_message [oid] ``past link failed`` []); ()
;;


let objc_history_previous oc = 
  let link = (objc_property oc `history_link`) in
    recall_objc (link_of_history_link_term link)
;;
%



%
;;;;
;;;;	Translation and activation.
;;;;	  - 
;;;;	  
;;;;	Translation :
;;;;	  - requires rebinding of objc to oid as builds new objc.
;;;;	  - substance may be shared with previous translation or new.
;;;;	
;;;;	
%


let translate_objc objc =
  if (objc_translated_p objc & not (translation_required_p objc))
    then objc
    else
	(if `CODE` = (objc_kind objc) %& not (code_compiled_p objc)%
	    then (objc_translate (compile_code_objc_source objc))
            else (objc_translate objc))
;;

% TODO why isn't compilation implicit in the translation? %
let translate_objc_force objc =
  (if (`CODE` = (objc_kind objc))
      then (objc_translate_force (compile_code_objc_source objc))
      else (objc_translate_force objc))
;;

let translated_p oid = (objc_translated_p (lib_object_contents oid));;

let lib_rebind oid objc =
 let curr = (oc oid) in
 if not (objc = curr) then (
   (lib_unbind oid)()
 ; (lib_bind_with_history oid objc (blot_objc curr))()
 ; ())
;;  


let translate oid =
 let objc = lib_object_contents oid in
  let nobjc = translate_objc objc in
    if (objc = nobjc) then false
    else ((lib_unbind oid)();
	  (lib_bind_with_history oid nobjc (blot_objc objc))();
	  true)
;;


let translate_weak oid =
 let objc = lib_object_contents oid in
  if objc_translated_p objc then ()
    else (let nobjc = (objc_translate objc) in
	      (if (objc_similar_p objc nobjc)
		  % not bind_with_history as we assume old history carried forward is good enough %
		  then ((lib_unbind oid)();
		        (lib_bind oid nobjc)();
		        ())
		  else failwith `translate_weak`))
;;

let activate a =
if not (lib_active_p a)
  then
   let translated_p = translate a in
    ((lib_activate a)()) ;
    % linking should not cause retranslation as simply updates weak property %
   ()
;;


let deactivate a = if (lib_active_p a) then ((lib_deactivate a)() ; ());; 

let activate_force a =
  if (lib_active_p a) then deactivate a

  ; lib_rebind a (translate_objc_force (oc a))
  ; (lib_activate a)()
  ; ()
;;

let delete_strong a =
  deactivate a;			 
  (lib_unbind a)();
  ()
;;

letrec obliterate oid =
 letrec doitree itree =
   delete_strong (inf_tree_object_id itree);
   map doitree (inf_tree_children itree);
   () in

 (let objc = oc oid in
  let k = objc_kind objc in
  if k = `STM` then (map obliterate (stm_objc_src_proofs objc); ())
  else if k = `PRF` then doitree (prf_objc_src_inf_tree objc)
  else ())
 ? ();

 delete_strong oid;			 
 ()
;;
  					     

let lib_insert oid oc = 					      
    (lib_bind oid oc)()
;;

let lib_add oc =
  let oid = create_object_id () in					     
    (lib_bind oid oc)()
    ; oid					     
;;


let allow_collection oid = 
  if not (lib_collectable_p oid) then ((lib_allow oid)(); ())
;;


let disallow_collection oid = 
  if (lib_collectable_p oid) then ((lib_disallow oid)(); ())
;;


let delete_weak = allow_collection;;

 
let save oid oc =
  letref (curr : (tok + object_contents)) = (inl `none`) in
  letref active_p = false in
     if lib_bound_p oid
        then ( curr := ((inr (lib_object_contents oid)) ? inl `none`)
	     ; (active_p := lib_active_p oid)
	     ; (delete_strong oid)
	     )

   ; (if (isl curr)
	 then (lib_bind oid oc)()
	 else (lib_bind_with_history oid oc (blot_objc (outr curr)))())
   
   ; if active_p then activate oid

   ; ()
;;

let squash_substance oid =
 if lib_active_p oid then failwith `squash_substance-object-active`;

 save oid (objc_remove_substance (oc oid))
;;

				       
% 
;;;;	ostate mod results in new oc but does not require translation.
;;;;	save will cause ostate broadcast.
;;;;	thus if we broadcast_ostate when oc diff but no translation required
;;;;	we should get an ostate broadcast for any ostate mod.
;;;;	
%

let weak_save oid oc =
  (lib_weak_bind oid oc)()
  ; ()
;;

let maybe_save oid oc old_oc =
  if not (oc = old_oc)
     then (weak_save oid oc) ? (save oid oc);
     ()
;;


let oid_ap_oc oid f =
 let oc = (lib_object_contents oid) in
   maybe_save oid (f oc) oc
;;




%
;;;;	
;;;;	Property 
;;;;	
;;;;	with-save : deactivate, modify and activate if originally active.
;;;;	
;;;;	objc_add_property	: primitive only affects source. 
;;;;	objc_remove_property	: primitive
;;;;	
;;;;	get_property		: looks up source property.
;;;;	put_property		: replaces property, with-save.
;;;;	
;;;;	get_properties		: looks up source properties.
;;;;	put_properties		: merges properties, with-save.
;;;;	
;;;;	set_properties		: replaces properties, with-save.
;;;;	  * differs form put in that with set original properties are lost.
;;;;	    ie, dangerous.
;;;;	
;;;;	make_version_link oc -> oc -> unit. adds prop with objc concentrate.
%

let remove_property oid name =
  oid_ap_oc oid \oc. (objc_remove_property oc name)
;;

let get_property oid name =
   (objc_property (lib_object_contents oid) name)
;;

let description_of_objc objc = objc_property objc `DESCRIPTION`;;
let name_of_objc objc = objc_property objc `NAME`;;

let put_property oid name prop =
  oid_ap_oc oid \oc. (objc_add_property (objc_remove_property oc name) name prop)
;;




let get_properties oid = (objc_get_properties (lib_object_contents oid));;

let put_properties oid name_props =
  oid_ap_oc oid (\oc. accumulate (\oc np. objc_add_property (objc_remove_property oc (fst np)) (fst np) (snd np)) 
                                oc
		                name_props)
;;

% differs form put in that set replaces existing properties while put merges with existing.
  if in doubt use put.
%
let set_properties oid props = oid_ap_oc oid \oc. (objc_set_properties oc props);;

% PERF: make lisp call that only clones objc once. %
let objc_add_properties objc props =
  accumulate (\oc np. objc_add_property oc (fst np) (snd np)) objc props
;;
%
let objc_set_property objc name prop =
 objc_add_property (objc_remove_property objc name)
		    name prop
;;
%
%
;;;;	create		: token -> object_id 
;;;;	  * by default created as collectable.
;;;;	
;;;;	put_term	: object_id -> term -> unit
;;;;	get_term	: object_id -> term
;;;;	
;;;;	
%

let add_object objc =
 let oid = create_object_id () in
  save oid objc;
  oid
;;
				   
let create type =
  let oid = create_object_id () in
    save oid (objc_contents type);
    oid
;;

% create garbage oid %
let cgo () = let oid = create `COM` in allow_collection oid; oid;;



let create_with_properties type init_props =
  let oid = create_object_id () in
    let oc = (objc_contents type) in
      save oid (objc_set_properties oc init_props);
      oid
;;
	
let create_objc_with_term type init_term init_props =
  objc_modify_source (objc_set_properties (objc_contents type)
					  init_props)
		     init_term
;;

let create_with_term type init_term init_props =
  let oid = (create_object_id ()) in
      save oid (create_objc_with_term type init_term init_props);
      oid
;;

let create_with_some_term type init_term properties =
  let oid =
    (if (isome_term_p init_term)
       then create_with_term type
			     (term_of_isome_term init_term)
			     (term_to_property_list properties)
       else create_with_properties type
				   (term_to_property_list properties)) in
    disallow_collection oid;
    oid
;;   

let create_object type proplist term = 
 create_with_term type term proplist
;;

let create_object_aux type props = 
 create_object type (term_to_property_list props)
;;

letref put_term_hook = (inl ()) : (unit + (object_id -> term -> unit));;
letref ptap_debug = (ivoid_term, ivoid_term);;

let put_term oid term =
   ptap_debug := (term, ivoid_term);(if (isr put_term_hook) then ((outr put_term_hook) oid term); ())
   ; save oid (objc_modify_source (lib_object_contents oid) term)
;;

let get_term oid = objc_source (lib_object_contents oid) ;;

let get_substance_term oid = objc_substance_term (lib_object_contents oid) ;;

let get_term_and_properties oid toks =
  let oc = (lib_object_contents oid) in
    icons_term (objc_source oc) (property_list_to_term (map (\name. (name, objc_property oc name)) toks))
;;    

let put_term_and_properties oid term props = 				   
  ( if (isr put_term_hook) then ((outr put_term_hook) oid term); ())
  ; save oid (accumulate (\oc np. objc_add_property (objc_remove_property oc (fst np)) (fst np) (snd np)) 
			 (objc_modify_source (lib_object_contents oid) term)
			 props)
;;

let put_term_and_iproperties oid term iprops = 				   
   ptap_debug := (term,iprops);
   put_term_and_properties oid term (term_to_property_list iprops)
;;



let put_name_t oid name =
 put_property oid `NAME` (itoken_term name)
;;

let put_name oid name = 
 put_property oid `NAME` (itoken_term (string_to_tok name))
;;


let put_description oid desc = put_property oid `DESCRIPTION` desc;;
				       
let get_name oid  =
 let t = (get_property oid `NAME`) in
  (string_of_istring_term t)
  ? (tok_to_string (token_of_itoken_term t))				   
;;

let get_description oid = get_property oid `DESCRIPTION`;;
				       
let library_open tags =
  without_dependencies 				        
    (\(). (orb_match_local_environment tags) ? (open_environment_by_match tags))
;;

let library_open_as tags tag =
  without_dependencies 
    (\(). open_environment_by_match_as tags [sys_version(); `lib`; tag; (string_to_tok "0")])
;;

let library_quick_close tag = close_environment (orb_match_local_environment [tag]) true false;;

let library_close tag = close_environment (orb_match_local_environment [tag]) false false;;

let library_close_gc tag = close_environment (orb_match_local_environment [tag]) false true;;


% 

 Tree funcs: 

%

let idirectory_id = `!directory`;;

let iroot_directory_term name children =
  make_term (idirectory_id, [make_token_parameter name]) [([],children)]
;;

let idirectory_term children =
  make_term (idirectory_id, []) [([],children)]
;;

let children_of_idirectory_term t =
  let ((id, parms), [([], children)]) = destruct_term t in
    if id = idirectory_id then children else fail
;;

let idirectory_root_term_p t = 
  (let ((id, [name]), [([], children)]) = destruct_term t in
    if not (id = idirectory_id) then fail;
    if not (`TOKEN` = type_of_parameter name) then fail;
    true) ? false
;;

let idirectory_root_name t = 
  let ((id, [name]), [([], children)]) = destruct_term t in
    if not (id = idirectory_id) then fail;
    if not (`TOKEN` = type_of_parameter name) then fail
    else name
;;

let idirectory_term_p t = 
  (let ((id, parms), [([], children)]) = destruct_term t in
    if not (id = idirectory_id) then fail;
    true) ? false
;;

let replace_idirectory_term_children t new_children =
  let (op, [([], children)]) = destruct_term t in
    make_term op [([], new_children)]
;;
 


let dag_objc_directory_p oc =
  idirectory_term_p (objc_source oc)
  ? false
;;

let dag_objc_root_p oc =
  idirectory_root_term_p (objc_source oc)
  ? false
;;

let dag_objc_directory_r oid oc =
  if not (dag_objc_directory_p oc)
     then raise_error [oid]
		      ``child directory not``
		      [];
  ()
;;
	
let dag_directory_objc oid = 
 let oc = lib_object_contents oid in
   dag_objc_directory_r oid oc;
   oc
;;
   
% roots and directories should have an Object_Id_Tree property. Or possibly
	description with ObjectIdDAG purpose. purpose prop better as allows
	finer broadcast control.
%

% root should be created as not collectable
%
let dag_make_root name =
  let oid = create_with_term `TERM`
  		(iroot_directory_term name idag_nil_term)
		[(`DESCRIPTION`, object_id_dag_description_term)]
		in
	disallow_collection oid;
	activate oid;
	oid
;;



let create_dag_directory_objc () = 
  (objc_modify_source 
    (objc_set_properties (objc_contents `TERM`) [(`DESCRIPTION`, object_id_dag_description_term)])
    (idirectory_term (idag_nil_term)))
;;


let oc_ap_src f oc = 
 let old = (objc_source oc) in
 let new = (f old) in
  if lex_equal_terms new old
     then oc 
     else (objc_modify_source oc new)
;;
let oid_ap_src f oid  = oid_ap_oc oid (oc_ap_src f);; 


let ddg_children_to_term = map_to_ilist (\x. (idag_child (fst x) (snd x))) idag_cons_op;;

let ddg_term_to_children =
 map_isexpr_to_list idag_cons_op (\x. (name_of_idag_child_term x, oid_of_idag_child_term x));;
				
let ddg_children_ap f t =
 let new = (f (children_of_idirectory_term t)) in
   if (lex_equal_terms (children_of_idirectory_term t) new)
      then t
      else replace_idirectory_term_children t new
;;   
let ddg_child_list_ap f = ddg_children_ap (\t. ddg_children_to_term (f (ddg_term_to_children t)));;   

let ddg_oid_ap_children f = oid_ap_src (ddg_children_ap f);;
let ddg_oid_ap_child_list f = oid_ap_src (ddg_child_list_ap f);;
  
let ddg_oid_set_children children = ddg_oid_ap_children (\cur . children);;
let dag_oid_set_child_list children =  ddg_oid_set_children (ddg_children_to_term children);;
   
let dag_dir_children t =
  ddg_term_to_children (children_of_idirectory_term t)
;;  

let dag_dir_children_replace t c = 
 replace_idirectory_term_children t
   (ddg_children_to_term c)
;;


% child should be created as collectable
%

let ddg_merge_children doid new_children =
 ddg_oid_ap_children 
  (\t.
    let children = ddg_term_to_children t in
    let new = (filter (\c. not (member_p c children (\(n,oid) (nn,ooid). n = nn & equal_oids_p oid ooid)))
		      (ddg_term_to_children new_children)) in
       if not (null new)
          then ddg_children_to_term (append children new)
	  else t)
 doid
;;

let term_member element list = member_p element list alpha_equal_terms;;
  
letrec dup_names names =
 if (names = []) then []
 else
 let dups = dup_names (tl names) in
  if (member (hd names) dups)
     	then dups
     else if (member (hd names) (tl names))
  	then ((hd names) . dups)
     else dups
;;


let dag_add_children_src erroids idir nameoids = 
  let dups = dup_names (map fst nameoids) in

    (if (not (dups = []))
       then raise_error erroids [`duplicate`; `children`] [(itokens_term dups)]);

  let isexpr = children_of_idirectory_term idir in
  letref acc = isexpr in
    (map (\(name, oid) .
	  (search_isexpr idag_cons_op 
			 (\t. name = name_of_idag_child_term t)
			 (\t. raise_error erroids [`duplicate`; `child`] [t])
			 isexpr;
          acc := (idag_cons_term acc
				 (idag_child name oid))))
	nameoids)

  ; (replace_idirectory_term_children idir acc) 
;;

% this method seems better if we catch dups : %
let directory_append_children goid newchildren =
  ddg_oid_ap_child_list (\children . append children newchildren)
   goid
;;


let dag_add_children parent nameoids =
  let poc = dag_directory_objc parent in
    (objc_modify_source poc
			(dag_add_children_src [parent] (objc_source poc) nameoids))
;;    
    
let dag_add_child_choice_aux checkdup_p firstp parent name oid =
  let poc = dag_directory_objc parent in
  let idir = (objc_source poc) in

  let isexpr = children_of_idirectory_term idir in
      if checkdup_p then
        ( search_isexpr idag_cons_op 
		 	(\t. name = name_of_idag_child_term t)
			(\t. raise_error [parent] [`duplicate`; `child`] [t])
			isexpr
        ; ());
      (objc_modify_source poc
			  (replace_idirectory_term_children idir
			    (if firstp
			    	then (idag_cons_term (idag_child name oid) isexpr)	    
			        else (idag_cons_term isexpr (idag_child name oid)))))
;;

let dag_add_child_choice = dag_add_child_choice_aux true;;
let dag_add_child = dag_add_child_choice_aux true false;;

let dag_modify_aux f doid = 
  let doc = dag_directory_objc doid in
  let idir = (objc_source doc) in
  let ichildren = children_of_idirectory_term idir in
  let newichildren = f ichildren in
    if (lex_equal_terms newichildren ichildren) then ()
    else save doid (objc_modify_source doc
			(replace_idirectory_term_children idir newichildren))
;;  

let dag_modify_objc_aux f doid = 
  let doc = dag_directory_objc doid in
  let idir = (objc_source doc) in
  let ichildren = children_of_idirectory_term idir in
  let newichildren = f ichildren in
    if (lex_equal_terms newichildren ichildren) then doc
    else (objc_modify_source doc
			(replace_idirectory_term_children idir newichildren))
;;  

% null place and before means first
  null place and after means last
%   
let dag_add_child_at_aux before_or_after place name oid isexpr =

  let l = map_isexpr_to_list idag_cons_op (\x.x) isexpr in

    % fail if duplicate name. %
    (let dup = ((inr (find (\t. name = name_of_idag_child_term t) l)) ? inl ()) in
      if isr dup
       then raise_error [] [`duplicate`; `child`] [outr dup]);

  let (pre, suf) = if (place = null_token)
		      then (if before_or_after then (nil, l) else (l, nil))
		      else split_list (\t. place = name_of_idag_child_term t) l in
	 (map_to_ilist (\x.x) idag_cons_op
	     (if (before_or_after or (null suf))
		 then (append pre ((idag_child name oid) . suf))
		 else (append pre ((hd suf) . (idag_child name oid) . (tl suf))))
		 )
;;

	 
let dag_add_child_at before_or_after parent place name oid =
  dag_modify_objc_aux (dag_add_child_at_aux before_or_after place name oid) 
    parent
;;

let dag_add_child_at_aux_o checkdup_p before_or_after place name oid isexpr =

  let l = map_isexpr_to_list idag_cons_op (\x.x) isexpr in

    % fail if duplicate name. %
    if checkdup_p then
    (let dup = ((inr (find (\t. name = name_of_idag_child_term t) l)) ? inl ()) in
      if isr dup
       then raise_error [] [`duplicate`; `child`] [outr dup]);

  let (pre, suf) = split_list (\t. equal_oids_p place (oid_of_idag_child_term t)) l in
	 (map_to_ilist (\x.x) idag_cons_op
	     (if (before_or_after or (null suf))
		 then (append pre ((idag_child name oid) . suf))
		 else (append pre ((hd suf) . (idag_child name oid) . (tl suf))))
		 )
;;

let dag_add_child_at_o dupcheck_p before_or_after parent place name oid =
  dag_modify_objc_aux (dag_add_child_at_aux_o dupcheck_p before_or_after place name oid) 
    parent
;;

let dag_add_child_alpha parent name oid =
  let poc = dag_directory_objc parent in
  let idir = (objc_source poc) in

  let isexpr = children_of_idirectory_term idir in
  let l = map_isexpr_to_list idag_cons_op (\x.x) (children_of_idirectory_term idir) in

    % fail if duplicate name. %
    (let dup = ((inr (find (\t. name = name_of_idag_child_term t) l)) ? inl ()) in
      if isr dup
       then raise_error [parent] [`duplicate`; `child`] [outr dup]);

  let names = tok_to_string name in 				  
  let (pre, suf) = split_list (\t. not (string_lt (tok_to_string (name_of_idag_child_term t)) names)) l in
       (objc_modify_source poc
	  (replace_idirectory_term_children idir
		 (map_to_ilist (\x.x) idag_cons_op
		     (append pre ((idag_child name oid) . suf)) )))
;;

let dag_insert_alpha parent name oid = 
  save parent (dag_add_child_alpha parent name oid)
;;
 
let dag_change_name parent name oid newname =
  let poc = dag_directory_objc parent in
  let idir = (objc_source poc) in
   save parent
    (objc_modify_source poc
      (replace_idirectory_term_children idir
        (map_isexpr idag_cons_op
           (\t. %tty_print (tok_to_string (name_of_idag_child_term t));%
  	        if ( name = name_of_idag_child_term t
  	           & (equal_oids_p oid (oid_of_idag_child_term t)))
		then idag_child newname oid
		else t)
	   (children_of_idirectory_term idir)))) 
;;


let dag_add_child_aux before_or_after parent place name oid =
 if (isr place)
    then dag_add_child_at before_or_after parent (outr place) name oid
    else dag_add_child parent name oid
;;

let dag_make_directory_oid () =
  let oid = create_with_term `TERM`
		(idirectory_term (idag_nil_term))
		[(`DESCRIPTION`, object_id_dag_description_term)] in
    activate oid;
    oid
;;

let make_directory_object_wcontents name term  =
  let oid = create_with_term `TERM`
		(idirectory_term term)
		[ `DESCRIPTION`, object_id_dag_description_term
		; `NAME`, itoken_term name
		] in
    activate oid;
    oid
;;

let dag_make_directory_aux before_after parent place name =
  let oid = dag_make_directory_oid () in
    save parent (dag_add_child_aux before_after parent place name oid);
    oid
;;

let dag_make_directory parent name =
  dag_make_directory_aux true parent (inl ()) name
;;


let dag_make_directory_before parent place name = 
  dag_make_directory_aux true parent (inr place) name
;;

let dag_make_named_directory poid link name =
 let oid = dag_make_directory poid link in
   put_name oid name;
   oid
;;

let dag_make_named_directory_at boa parent place name =
 let oid = dag_make_directory_aux boa parent (inr place) name in
   put_name oid (tok_to_string name);
   oid
;;
   
let dag_make_named_directory_before = dag_make_named_directory_at true;;
let dag_make_named_directory_after = dag_make_named_directory_at false;;


let dag_make_directory_alpha parent name =
 let oid = dag_make_directory_oid () in
   put_name oid (tok_to_string name);
   dag_insert_alpha parent name oid;
   oid
;;

let dag_make_oid_alpha parent name kind =
 let oid = create kind in
   put_name oid (tok_to_string name);
   dag_insert_alpha parent name oid;
   oid
;;
 

%
;;;;	
;;;;	dag removal
;;;;	  - moves to ``local garbage queue`` 
;;;;	  - then gc's ``local garbage queue`` 
;;;;	  
%

letref upper_garbage_limit = 0
and lower_garbage_limit = 0;;

let find_garbage_limits () = 
 if not (upper_garbage_limit = 0) then (upper_garbage_limit, lower_garbage_limit)
 else
  (((let term = objc_substance_term (oc (descendent_s ``local garbage .config``)) in
     search_isexpr icons_op 
       (\t. (let (id, (p . r)), bts = destruct_term t in
	      if id = `upper_limit` then (upper_garbage_limit := destruct_natural_parameter p; ())
	      else if id = `lower_limit` then (lower_garbage_limit := destruct_natural_parameter p; ())
	      else ())
	     ? ()
	     ; false)
       (\t.())    
       term) 
   ? [])
   ; if (upper_garbage_limit < lower_garbage_limit) 
        then (let temp = upper_garbage_limit in
                upper_garbage_limit := lower_garbage_limit
	      ; lower_garbage_limit := temp
              ; ())
   ; if upper_garbage_limit = 0
        then 32,16 
        else (upper_garbage_limit, lower_garbage_limit))
;;
   
let garbage_queue () =
  let goid = (descendent_s ``local garbage``) in
     (child goid `queue`) 
     ? (let qoid = (dag_make_directory goid `queue`) in
        put_name qoid "queue";
	qoid)
;;

% saving garbage appears to be somewhat expensive 
  thus buffer, exposes us to failure losing garbage
  but garbage protects for human failure not system.
%   
letref garbage_buffer = [] : ((tok # object_id) list);;

let append_to_garbage_aux tokoids = 
  let goid = garbage_queue() in
  let goc = dag_directory_objc goid in
  let gdir = (objc_source goc) in
  let (upper, lower) = find_garbage_limits () in

  let children = dag_dir_children gdir in
   let lold = length children 
   and lnew = length tokoids in
   let lkeep = if (lold + lnew ) < upper then lold
	       else (max 0 (lower - lnew)) 
     in save goid
         (objc_modify_source goc
           (dag_dir_children_replace gdir 
	      (append tokoids (if lkeep = lold then children else (firstn lkeep children)))))
;;

let flush_garbage_buffer () = 
 upper_garbage_limit := 0;
 lower_garbage_limit := 0;

  append_to_garbage_aux garbage_buffer
;;

let append_to_garbage tokoids = 
  let (upper, lower) = find_garbage_limits () in

  let children = garbage_buffer in
   let lold = length children 
   and lnew = length tokoids in
   let lkeep = if (lold + lnew) < upper 
                  then lold
   	          else (max 0 (lower - lnew)) in
    garbage_buffer := (append tokoids (if lkeep = lold then children else (firstn lkeep children)))
   ; ()
;;

let dag_remove_root oid = 

  (append_to_garbage [root_name oid, oid] ? ());

  deactivate oid;
  allow_collection oid
;;


% failure means subtree unchanged
%
let ilistp icons_op t = 
  let (a, bterms) = destruct_term t in
    (equal_operators_p icons_op a) 
	& (bterms = nil
	   or ((let [([],car); ([],cdr)] = bterms in true) ? false))
;;

let inilp icons_op t = 
  (let (a, []) = destruct_term t in (equal_operators_p icons_op a)) ? false
;;


let filter_isexpr op predicate isexpr =
 letref num_nils = 0 in 		      
 letrec aux isexpr =

  % (predicate isexpr) is false %
  if not (ilistp op isexpr)
     then (if (predicate isexpr)
		then (make_term op [])
		else fail)
  else if (inilp op isexpr) then ( num_nils := num_nils + 1; fail )
  else let (a, [([],car); ([],cdr)]) = destruct_term isexpr in
	(let newcar = aux car in 
	  if (inilp op newcar) 
		then ( num_nils := num_nils + 1; (aux cdr ? cdr) )
		else (make_term op [([], newcar); ([], (aux cdr ? cdr))]))
	% car failed %
	? ((make_term op [([], car); ([], aux cdr)]) 
	   % both failed %
	   ? isexpr)

 in  let r = aux isexpr in
   if (num_nils > 5)
      then normalize_isexpr op r
      else r
;;

      
let dag_remove_child_aux gp predicate ichildren =
  filter_isexpr idag_cons_op
		(\t. if (predicate t)
			then ( if gp then append_to_garbage [( name_of_idag_child_term t
							     , oid_of_idag_child_term t)]
			     ; true)
			else false)
		ichildren
;;

let dag_remove_child_byoid oid =
  dag_remove_child_aux true (\dc. equal_oids_p oid (oid_of_idag_child_term dc))
;;

% predicate : term{!dag_child} -> bool %
let dag_remove_child parent predicate = 
  letref foundp = false in
  let poc = dag_directory_objc parent in
  let idir = (objc_source poc) in
  let newoc =
	 (objc_modify_source poc
		(replace_idirectory_term_children idir 
		 (filter_isexpr idag_cons_op
				(\t. if (predicate t) then foundp := true else false)
				(children_of_idirectory_term idir))))
	in
	if (not foundp) then raise_error [parent] [`child`; `none`] [];
	save parent newoc
;;  

	
let dag_search_directory pred dir  = 
  let idir = (objc_source (dag_directory_objc dir)) in
   (search_isexpr idag_cons_op
			(\t. pred (oid_of_idag_child_term t))
			(\t. oid_of_idag_child_term t)
			(children_of_idirectory_term idir))
;;  

% predicate : term{!dag_child} -> bool %
let dag_child_find parent name = 
  let idir = (objc_source (dag_directory_objc parent)) in
   (search_isexpr idag_cons_op
			(\t. name = (name_of_idag_child_term t))
			(\t. oid_of_idag_child_term t)
			(children_of_idirectory_term idir))
;;  

let get_dir oid =
 (children_of_idirectory_term (get_term oid))
;;

let dag_find_a_child parent name =
 let l = dag_child_find parent name in
   if l = [] 
      then raise_error [parent] [`find`; `child`; `none`] [itoken_term name]
   else if not ((tl l) = []) 
      then raise_error (parent . l) [`find`; `child`; `many`] [itoken_term name]
      else ();

   (hd l)
;;
  

%    
%%    
%%    dag_delete simply deletes path from dir.
%%    dag_remove_leaf/directory will deactivate and remove.
%%    
%


let dag_make_leaf parent name type =
  let oid = (create type) in
    save parent (dag_add_child parent name oid);
    oid
;;

let dag_make_named_leaf parent name type =
  let oid = (create type) in
    save parent (dag_add_child parent name oid);
    put_name oid (tok_to_string name);
    oid
;;

let dag_make_named_leaf_at boa parent place name type =
  let oid = (create type) in
    save parent (dag_add_child_at boa parent place name oid);
    put_name oid (tok_to_string name);
    oid
;;


let dag_insert_leaf parent name type term =
  let oid = (create_with_term type term []) in
    save parent (dag_add_child parent name oid);
    oid
;;

let dag_insert_named_leaf parent name type term =
  let oid = dag_insert_leaf parent name type term in
    put_name oid (tok_to_string name)

  ; oid
;;    
    
% fail if child exists and is active ?
  or deactive
%

let dag_set_leaf parent name type term =
  if (child_p parent name)
    then (let oid = (child parent name) in
           save oid (create_objc_with_term type term []);
	   oid)
    else (let oid = (create_with_term type term []) in
          save parent (dag_add_child parent name oid);
	  oid)
;;

let dag_insert parent name oid = save parent (dag_add_child parent name oid); ();;
let dag_insert_first poid name oid = save poid (dag_add_child_choice true poid name oid); ();;


let dag_insert_at boa parent place name oid =
  save parent (dag_add_child_at boa parent place name oid)
  ; ()
;;

% fail if place not present? %
let dag_insert_before = dag_insert_at true;;
let dag_insert_after = dag_insert_at false;;


let cpobj oid = 
  let noid = create_object_id () in
  save noid (oc oid);

    noid
;;

letrec copy_inf_tree itree =
  inf_tree (cpobj (inf_tree_object_id itree))
    (map copy_inf_tree (inf_tree_children itree))
;;

let set_stm_extract oid = 
 let objc = oc oid in
 let erpropp = ( bool_of_ibool_term (objc_property objc `extract_required`)
	       ? false) in
  if not erpropp
     then save oid (objc_add_property objc `extract_required` ibool_true_term)
;;

let require_termof = set_stm_extract;;

%
if stm then copies prfs and if prf copies inf.
%

let copy_object_strong name oid = 

 let doprf soid poid =
  let pobjc = oc poid in
  let noid = create_object_id () in
  let nprops = (let oprops = objc_get_properties pobjc in
                 [(`NAME`, itoken_term name); (`stm_oid`, ioid_term soid)]
                 @ (filter (\n,t. not (member n ``stm_oid NAME``)) oprops)) in

  let itree = copy_inf_tree (prf_objc_src_inf_tree pobjc) in
    save noid (prf_objc_src_modify_inf_tree
                  (objc_set_properties pobjc nprops)
                  itree);
    
    noid

 in

 let dostm soid = 
  let sobjc = oc soid in
  let noid = create_object_id () in

  save noid
    (stm_objc_src_modify_proofs sobjc
      (map (doprf noid) (stm_objc_src_proofs sobjc)))
  ; noid

  in

 let aux oid =
  let objc = oc oid in
  let kind = objc_kind objc in
    if (kind = `INF`) or (kind = `PRF`)
       then (raise_error [oid] [`make_save`; kind] []; fail)  % not an error just not prepared for it%
    else if (`STM` = kind)
       then dostm oid
    else cpobj oid

  in 

  aux oid
;;

let copy_object_at boa poid place name oid = 
 let noid = %cpobj% copy_object_strong name oid in
    dag_insert_at boa poid place name noid;
    put_property noid `NAME` (itoken_term name);
    (let theory = oid_of_ioid_term (get_property oid `theory`) in
      if not (equal_oids_p poid theory) then put_property oid `theory` (ioid_term poid)
    ) ? ();      
    noid
;;


let dag_move_object_id_before parent place name =
  let poc = dag_directory_objc parent in
  let idir = (objc_source poc) in

  let isexpr = children_of_idirectory_term idir in
  let l = map_isexpr_to_list idag_cons_op (\x.x) (children_of_idirectory_term idir) in

  let (n, ll) = divide_list (\t. name = name_of_idag_child_term t) l  in

   % fail if no name. %

  if (null n)
     then raise_error [parent] [`move`; `child`; `not`] [itoken_term name];

  let (pre, suf) = split_list (\t. place = name_of_idag_child_term t) ll in
     save parent
     (objc_modify_source poc
			 (replace_idirectory_term_children idir
			    (map_to_ilist (\x.x) idag_cons_op
			      (append pre
				     (append n suf)))))
  ; ()				     
;;


% insert list of name oid pairs.
  inserts in reverse order of the arg
%


let dag_inserts parent nameoids =
  save parent (dag_add_children parent nameoids);
 ()
;;	   

let dag_delete_wo_garbage parent name =

  dag_remove_child parent
	(\t. name = (name_of_idag_child_term t))
;;

let dag_delete parent name =

  letref oid = inl () in

  dag_remove_child parent
	(\t. if (name = (name_of_idag_child_term t))
	        then (oid := inr (oid_of_idag_child_term t)
		     ; true)
		else false
                )

  ; if (isr oid) 
       then (append_to_garbage [name, (outr oid)] ? ())
       else ()
;;

let dag_remove_directory parent name =

  letref goid = inl () in

  dag_remove_child parent
	(\t. (if (name = (name_of_idag_child_term t))
		then (let oid = (oid_of_idag_child_term t) in
			(if (not (equal_oids_p oid  (descendent_s ``local garbage queue``)))
			    then (goid := inr oid; ()) ? ());
			dag_objc_directory_r oid (lib_object_contents oid);
			deactivate oid;
			true)
		else false))

  ; if (isr goid) 
       then (append_to_garbage [name, (outr goid)] ? ())
       else ()
;;

let dag_remove_leaf parent name =

  letref goid = inl () in

  dag_remove_child parent
	(\t. (if (name = (name_of_idag_child_term t))
		then (let oid = (oid_of_idag_child_term t) in
			if (dag_objc_directory_p (lib_object_contents oid))
			   then ((raise_error [parent; oid] [`child`; `directory`]
						[itoken_term name]); false)
			   else (goid := inr oid; deactivate oid; true))
		else false))

  ; if (isr goid) 
       then (append_to_garbage [name, (outr goid)] ? ())
       else ()

;;

% does not object to dirs. %
let dag_remove_oid parent oid =

  letref gname = inl () in

  dag_remove_child parent
	(\t. (if (equal_oids_p oid (oid_of_idag_child_term t))
		then (let name = (name_of_idag_child_term t) in
		       gname := inr name; deactivate oid; true)
		else false))

  ; if (isr gname) 
       then (append_to_garbage [outr gname, oid] ? ())
       else ()
;;

%
;;;;	
;;;;	Need ability to delete tree,
;;;;	  - remove references.
;;;;	  - deactivate and allow collection of leaves. 
;;;;	      * leave unbind collect to unbind.
;;;;	
;;;;	
;;;;	Subtrees should only be deactivated if orphaned by reference removal.
;;;;	Either have gc which deactivates if orphaned or ability
;;;;	to test for orphans dynamically to allow remove tree to deactivate.
;;;;	
;;;;	  
%


%
;;;;	
;;;;	  delete_tree : rm top link and then recursively descend and deactivate all leafs.
;;;;	
;;;;	  
%
				 
letrec dag_deact_tree oid =
  (if (lib_active_p oid & directory_p oid)
      then ( map (\n,coid. dag_deact_tree coid) (directory_children oid)
	   ; allow_collection oid
	   ; deactivate oid
	   )
       else  (allow_collection oid
	     ; deactivate oid
	     ; ()))
;;


letrec dag_remove_tree_danger oid =
  let oc = (lib_object_contents oid) in
  let source =  (objc_source oc) in
  (if (lib_active_p oid & (dag_objc_directory_p oc))
       then let children = (children_of_idirectory_term source) in
	     % delete first to prevent loop if tree has up pointer creating a cycle, ie not really a dag.%
             ( allow_collection oid
	     ; delete_strong oid
	     ; (search_isexpr idag_cons_op 
		 (\t. true)
		 (\t. ((dag_remove_tree_danger (oid_of_idag_child_term t)); true) ? false)
		 children)
	     ; ())
       else  (allow_collection oid
	     ; delete_strong oid
	     ; ()))
;;

% was previous version, but that deemed too destructive %
let dag_remove_tree oid =
      allow_collection oid
    ; delete_strong oid
;;

let dag_remove_root_tree = dag_remove_tree;;

% does not differ much from dag_remove_directory ??? %
let dag_remove_directory_tree parent name =
  dag_remove_child parent
	(\t. (if (name = (name_of_idag_child_term t))
		then (let oid = (oid_of_idag_child_term t) in
			dag_objc_directory_r oid (lib_object_contents oid);
			dag_remove_tree oid;
			true)
		else false))
;;


let dag_delete_tree parent name =
  dag_remove_child parent
	(\t. (if (name = (name_of_idag_child_term t))
		then (let oid = (oid_of_idag_child_term t) in
			dag_objc_directory_r oid (lib_object_contents oid);
			dag_deact_tree oid;
			true)
		else false))
;;



let ioid_export_term cookie oid = 
  make_term (`!oid_export`, 
		[make_token_parameter cookie; 
		 make_object_id_parameter oid])
	 []
;;

let oid_export cookie oid = term_to_string (ioid_export_term cookie oid);;


let oid_import cookie s = 
  let t = (string_to_term s) in
  let ((opname, [ocookie; oid]), []) = (destruct_term t) in
     if cookie = (destruct_token_parameter ocookie)
	then (destruct_object_id_parameter oid)
	else ((raise_error [] [`object_id`; `import`; `cookie`] [t]); fail)
;;


% 	move dforms to subdirectory of thy. 
 	twould be neat to add dir to standard dir which had direct pointers to display subdirs. 
	ie (descendent (root `theories`) ``standard core_2 display``)
		== (descendent (root `theories`) ``standard display core_2``)
%		


% 	remote call to nuprl client	%

let post_activate_code_ap = null_ap (itext_term "post_activate_code ");;

let client_post_activate_code desc oid =
  orb_eval_args `ALL` desc (oid_ap post_activate_code_ap oid)
;;


letref lib_accept_port = config_accept () ? 0;;

let setup_accept port = lib_accept_port := port;;



%let nosa port = orb_start_accept port `nuprl`;;
 let mosa port = orb_start_accept port `mathbus`;;
%
let losa port = orb_start_accept port `nuprl-uncompressed`;;
let stop_accept () = orb_stop_accept lib_accept_port;;

%let start_accept () = orb_start_accept lib_accept_port `nuprl`;;
let stop_accept () = orb_stop_accept lib_accept_port;;
%
%

	Some more dag functions.

%	

%%% overwrite if name exists. %
let dag_ninsert_leaf dir name kind term = 
 (dag_delete dir name ? ());
 dag_insert_leaf dir name kind term
;;

 

let rdb_eval e = orb_eval_args `ONE` nuprl5_rdb_description_term e;;
let rdb_eval_to_term e = orb_eval_args_to_term `ONE` nuprl5_rdb_description_term e;;
			     
let rdb_eval_string = rdb_eval o begin_ap;;	

let rdb_eval_to_string = make_string_return rdb_eval_to_term;;
let rdb_eval_to_token = make_token_return rdb_eval_to_term;;
let rdb_eval_to_object_id = make_oid_return rdb_eval_to_term;;
let rdb_eval_to_object_ids = make_oids_return rdb_eval_to_term;;
let rdb_eval_to_bool = make_bool_return rdb_eval_to_term;;
let rdb_eval_to_terms = make_terms_return rdb_eval_to_term;;

let rdb_query_ap = (null_ap (itext_term "rdb_query "));;
let rdb_query schema sql = rdb_eval_to_term (term_ap (term_ap rdb_query_ap schema) sql);;

let rdb_exec_ap = (null_ap (itext_term "rdb_exec "));;
let rdb_exec sql = rdb_eval (term_ap rdb_query_ap sql);;

let lib_sql_code_query description sql schema = rdb_query schema sql;;
let lib_sql_code_exec description sql = rdb_exec sql;;

let rdb_get_table oid = table_of_sql_code (lib_object_contents oid) ;;

let remote_shell_eval kind s =
 ( make_bool_return
    (orb_eval_args_to_term `ANY` (itokens_term (orb_match_bus_environment [kind])))
       (string_ap (begin_ap "run_shell") s))
;;

let lib_abstraction_lookup oid = 
 let (conds, model, matrix, notprim) = abstraction_lookup oid in
   make_term (`!abstraction_data`, [make_bool_parameter notprim])
	     [([],itokens_term conds); ([],model); ([],matrix)]			     
;;


let migfix_object oid = 		
  let objc = oc oid in
  let term = objc_source objc in
   put_term oid (migfix_term (objc_kind objc) term)
;;


let find_env_name pattern = 
  let matchf = string_match_f false pattern in 
   filter (exists (\t. matchf (tok_to_string t))) 
     (map (\n,s. map_isexpr_to_list icons_op token_of_itoken_term n) (match_db_logs nil))
;;	      

let name_property oid = first_tok_of_term (get_property oid `NAME`);;



let make_directory_object name =
 let oid = dag_make_directory_oid () in
   put_name_t oid name;
   oid 
;;

	      
let remove_directory = dag_remove_directory;;
let remove_leaf = dag_remove_leaf;;

let insert_named_leaf = dag_insert_named_leaf;;
let make_named_directory = dag_make_named_directory;;    
let change_link_name = dag_change_name;;
let make_named_directory_at = dag_make_named_directory_at;;
let make_named_directory_after = dag_make_named_directory_after;;    
let make_named_directory_before = dag_make_named_directory_before;;    
let make_named_leaf_at = dag_make_named_leaf_at;;
let make_named_leaf = dag_make_named_leaf;;    
let make_named_leaf_after = dag_make_named_leaf_at false;;
let make_directory_alpha = dag_make_directory_alpha;;    
let make_oid_alpha = dag_make_oid_alpha;;
let set_directory_children oid t = ddg_oid_set_children t oid;;

let insert_at = dag_insert_at;;
let insert_alpha = dag_insert_alpha;;	      
let insert_object_id_after = insert_at false;;

let kind_of_oid oid = objc_kind (oc oid);;
let name_of_oid = name_property;;

% differs from save only in that activate_failure does not cause save failure. %
let save_force oid oc =
  letref (curr : (tok + object_contents)) = (inl `none`) in
  letref active_p = false in
     if lib_bound_p oid
        then ( curr := ((inr (lib_object_contents oid)) ? inl `none`)
	     ; (active_p := lib_active_p oid)
	     ; (delete_strong oid)
	     )

   ; (if (isl curr)
	 then (lib_bind oid oc)()
	 else (lib_bind_with_history oid oc (blot_objc (outr curr)))())
   
   ; if active_p then (activate oid ? ())

   ; ()
;;

let db_envs paddr =
 map (\t,addr. datetime_sortable t, addr) (list_environments paddr)
;;

%  - : ((* list -> **) -> (* list -> int) -> * list -> ** list) %
let regroup f posf l = 
 letrec aux l = 
  if null l then nil else
  let (prefix, suffix) = split (posf l) l ? (l,nil) in
   (f prefix) . (aux suffix) 

 in aux l
;;


let print_environments l = 
 map (\g. tty_print "";
          (map (\t,addr. tty_print (reduce $J " " ((datetime_sortable t). " :" . addr)))
               g))
  (quicksort (\g h. (fst (hd h)) < (fst (hd g)))
     (regroup id (\l. let t = hd (snd (hd l)) in (position (find (\m,ma. (not (t = (hd ma)))) l) l) -1)
      (quicksort (\(a,aa) (b,ba). string_lt (hd aa) (hd ba)) 
                 (map (\t, addr . t,  (map (\n. " " J (tok_to_string n)) (tl (tl addr))))
                      l))))
 ; ()			    
;;

let db_envs_print paddr =
 print_environments (list_environments paddr)
 ; ()
;;

let pid_of_dbe = snd o fst o snd ;;

let iobject_tree_cons_op =  (`!object_tree_cons`, []);;
let iobject_tree_term name kind status term children =
    make_term (`!object_tree`, [])
	      [[], itoken_term name; [], itoken_term kind; [], itoken_term status; [], term; [], children]
;;
			      
let collection_object_tree oid =
   let kind = kind_of_oid oid and
       name = name_of_oid oid and  %may want to check kind is ok%
       term = get_substance_term oid in
   let oids = static_oids_of_term term in

   iobject_tree_term name `GRP` `status?` ivoid_term
       (map_to_ilist (\x.x) iobject_tree_cons_op (map oid_to_iobject_tree_term_with_comments oids))
;;

% if stm then set reference_environment to be refenv containing obj at place if any
  otherwise set to be last refenv of refenv dir.
%	       
let lib_mkobj_aux dir place kind name props =
 let oid = create_with_properties kind
	       ( (`NAME`, itoken_term name)
	       . props) in
    dag_insert_at false dir place name oid;
    oid	       
;;	       

% inappropriately named, lib_mkobj_aux should replace  %
let lib_mkobjc_aux dir place kind name props =
 lib_mkobj_aux dir place kind name
      (term_to_property_list props)
;;	       
	       
let lib_mklink poid oid place name rmdup =
  let children = (directory_children poid) in
  let curpos = (search_list (\child. (equal_oids_p oid (snd child))) children ? -1) in
  let placepos = (search_list (\child. place = (fst child)) children ? -1) in
  
   dag_modify_aux 
     (\ichildren. 
       dag_add_child_at_aux false place name oid
         (if (rmdup & (not (curpos < 0))) 
	     then dag_remove_child_byoid oid ichildren
	     else ichildren))
     poid

 ; (curpos < placepos)

%  if (rmdup & (not (curpos < 0))) then dag_delete poid name;

  dag_insert_at false poid place name oid
 ; (curpos < placepos)
%;;

%view_show_oids
 (map (\n. let t = subtermn 3 (find (\t. n = first_tok (subtermn 2 t)) recovered_stms) in
           create_with_term `STM` t ((`NAME`,itoken_term n) . nil))
     stms_todo_new)	       
;;
%

let remove_properties prop_names oid =
 let objc = oc oid in 
 save oid
  (objc_set_properties objc 
    (filter (\n,p. not (member n prop_names)) 
            (objc_get_properties objc)))
;;


let lib_delete_orphans () =
  let orphans = lib_orphans () in
   lib_delete_list orphans;
   length orphans
;;