
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;
;;;;	RLE TODO : tt-link doc is nfg. tt-link is major kludge at this
;;;;	RLE TODO : point. If tt survives this will need to be cleaned up.
;;;;	
;;;;	new-tt-link(<closure{out-sexpr-f}>
;;;;		    <closure{in-sexpr-f}>)
;;;;	  : <link>
;;;;	 * out-sexpr-f <data> : <ascii>
;;;;	 * in-sexpr-f <ascii> : <data>
;;;;
;;;;	RLE TODO :  also the links were designed to have ability to support a fan-out.
;;;;	RLE TODO : this has not proven useful so far and is not being actively supported.
;;;;	RLE TODO :  It can be reverse engineered in
;;;;	RLE TODO : at later date if it is required after all.


;;;;	
;;;;	IO considerations:
;;;;	  - 8bit byte IO as char io underspecified in lisp standard and various
;;;;	    implementations differ (particularly aclpc).
;;;;	      * unfortunate, since write-string was more efficient than writing
;;;;		char-codes a byte at a time.
;;;;	  - abstract interface to data stream producers/consumers.
;;;;	  - abstract interface of io streams.
;;;;
;;;;	  
;;;;	


;;;; -docs- (mod io)
;;;;
;;;;
;;;;	Links:	send/recv terms
;;;;
;;;;
;;;;	new-stream-link((<socket{in-socket}> . <socket{out-socket}>) list
;;;;			<bool{server?}>
;;;;			<closure{out-f}>
;;;;			<closure{in-f}>)
;;;;	  : <link>
;;;;	 * out-f (<data> <closure{write-byte})	: NULL
;;;;	 * in-f (<closure{read-byte})> 		: <data>
;;;;	    - write-byte (BYTE8) : NULL
;;;;	    - read-byte () : BYTE8
;;;;
;;;;	new-mathbus-link((<socket{in-socket}> . <socket{out-socket}>) list
;;;;			<bool{server?}>
;;;;			<closure{out-f}>
;;;;			<closure{in-f}>)
;;;;	  : <link>
;;;;	 * out-f (<data> <closure{write-char})	: NULL
;;;;	 * in-f (<closure{read-char})> 		: <data>
;;;;	    - write-char (char) : NULL
;;;;	    - read-char () : char
;;;;
;;;;
;;;;	link-open (<link>)				: NULL	
;;;;	link-close(<link>)				: NULL
;;;;	
;;;;	link-send(<link> <data>)			: NULL
;;;;	link-recv(<link> <bool{blockp}>)		: <data>
;;;;
;;;;	link-errors(<link>)				: <bool>
;;;;	  * may emit messages.
;;;;	link-listen(<link>)				: <bool>
;;;;
;;;;	<link>			:<channel> list
;;;;				 BOOL{asynch-ok-p}
;;;;
;;;;	Channels: comprise link
;;;;
;;;;
;;;;	<channel>		:(model open-f close-f send-f recv-f listen-f)
;;;;	  * model currently 'stream 'mathbus or 'tt
;;;;
;;;;	<stream-channel>	:<channel>(in-socket out-socket out-f in-f
;;;;					   out-stream in-stream)
;;;;				 <compression-levels>
;;;;	  * used for sending terms between 2 lisp processes
;;;;
;;;;	new-stream-channel(<socket{in-socket}> <socket{out-socket}>
;;;;			   <closure{out-f}> <closure{in-f}>)		: <channel>
;;;;
;;;;	 * processes at each end of the channel play both server and client roles
;;;;	 ** in-socket is socket used for in-stream, created as server
;;;;	 *** out-socket is socket used for out-stream, created as client
;;;;
;;;;	stream-channel-open (<channel>)		: NULL
;;;;	 * opens 2 distinct bidirectional byte streams for input, output respectively.
;;;;	   2 streams (rather than 1) provide the ability to read and write
;;;;	   concurrently (multitasking).
;;;;
;;;;	stream-channel-close(<channel>)	      	: NULL
;;;;
;;;;	stream-channel-send(<channel> <data>)		: NULL
;;;;	stream-channel-recv(<channel> <bool{blockp}>)	: <data>
;;;;
;;;;	stream-channel-listen(<channel>)		: <bool>
;;;;
;;;;
;;;;	<mathbus-channel>	:<channel>(in-socket out-socket out-f in-f
;;;;					   out-stream in-stream)
;;;;	  * used for sending terms between 1 lisp process and 1 foreign process
;;;;	    ie. nuprl-light in ocaml.  Terms are converted to mathbus terms, which are
;;;;	    implementation independent.
;;;;	    
;;;;
;;;;	new-mathbus-channel(<socket{in-socket}> <socket{out-socket}>
;;;;			<closure{out-f}>
;;;;			<closure{in-f}>)	: <channel>
;;;;
;;;;
;;;;	mathbus-channel-open (<channel>)		: NULL
;;;;	 * opens 2 distinct bidirectional character streams for input, output respectively.
;;;;
;;;;	mathbus-channel-close(<channel>)	      	: NULL
;;;;
;;;;	mathbus-channel-send(<channel> <data>)		: NULL
;;;;	mathbus-channel-recv(<channel> <bool{blockp}>)	: <data>
;;;;
;;;;	mathbus-channel-listen(<channel>)		: <bool>
;;;;
;;;;

;;;;
;;;;    Sockets: comprise channels
;;;;
;;;;	
;;;;	<socket>	:(INTEGER{port} INTEGER{fd} INTEGER{stream-fd})
;;;;
;;;;	  *fd is the file-descriptor of the UNIX STREAM-SOCKET on which calls to
;;;;	   accept and connect are made (nil if not server)
;;;;	  **stream-fd is the file-descriptor returned by calls to accept and connect on
;;;;	   which term io occurs
;;;;
;;;;	new-socket(INTEGER{port}  STRING{host} <bool{server-p}>
;;;;		   &optional INTEGER{queue-size} <bool{block-p}>)	: <socket>
;;;;
;;;;	  *If server-p, new-socket calls ffi-make-socket to obtain a file descriptor
;;;;	   for a STREAM-SOCKET on port which can then be used by multiple channels
;;;;	  **queue-size sets the number of allowable channels, defaults to 5
;;;;	  ***If block-p, socket blocks until client connects.
;;;;	 
;;;;	destroy-socket(<socket> &optional <bool{stream-p}>)		:NULL
;;;;
;;;;	  *if stream-p, also closes stream-fd
;;;;
;;;;	accept-new-client(<socket>)		:INTEGER
;;;;						| NULL
;;;;	  *If socket is blocking, or if socket is non blocking and has a client
;;;;	   waiting, returns a fd for the new message socket.  Else, returns nil. 	
;;;;

;;;;
;;;;	Foreign Function Interface
;;;;	FFI: used to generate sockets
;;;;
;;;;	ffi-make-socket(INTEGER{port} INTEGER{queue-size})	:INTEGER
;;;;	ffi-open-server (INTEGER{fd})			:INTEGER
;;;;	ffi-open-client (INTEGER{port} STRING{host})	:INTEGER
;;;;	ffi-close-connection(INTEGER{fd})		:NULL
;;;;
;;;;
;;;; -doce- 

;;;;
;;;;	Connection Protocol:  an example
;;;;
;;;;	process1 <-> process2
;;;;
;;;;	process1 opens socket on a known port, and enters loop which periodically
;;;;	checks for connected client. Meanwhile, process2 opens a socket on a new port,
;;;;	connects to socket on known port and creates an output-stream.
;;;;	When process1's check succeeds, it opens an input stream
;;;;	for this connection, and adds a check for data on this stream to the loop.
;;;;	process2 sends iconnect-term containing this
;;;;	new port number, to which process1 is to connect.  process1 receives this term,
;;;;	opens an output stream, and sends response (via input stream).
;;;;	process2 receives response and  opens an input-stream.
;;;;
;;;;	For non-blocking sockets with 2 distinct streams: before process1 connects (to create
;;;;	out-stream), an intruding client may
;;;;	connect. if this happens, connect will fail and process1 will send
;;;;	message that connect failed, prompting process2 to destroy that connection.
;;;;
;;;;

;;;;	< : ascii recv read
;;;;	> : ascii send write
;;;;	\ : mathbus recv
;;;;	/ : mathbus send
;;;;	
;;;;	T : write to trace file.
;;;;	
;;;;	R : read from mathbus file
;;;;	W : write to mathbus file
;;;;	I : read from ascii file
;;;;	O : write to acii file
;;;;	
;;;;	b : bind
;;;;	u : unbind
;;;;	w : allow
;;;;	d : disallow
;;;;	+ : activate
;;;;	- : deactivate
;;;;	
;;;;	c : commit
;;;;	C : sneak-commit
;;;;	o : undo
;;;;	


(defvar *io-echo-stat-names*
  `(("u" . |unbind|)
    ("b" . |bind|)
    ("c" . |commit|)
    ("o" . |undo|)
    ("w" . |allow|)
    ("d" . |disallow|)
    ("+" . |activate|)
    ("-" . |deactivate|)
    ("T" . |trace|)
    ("R" . |read|)
    ("W" . |write|)
    ("I" . |read|)
    ("O" . |write|)
    ("<" . |send|)
    (">" . |receive|)
    ("/" . |send|)
    ("\\" . |receive|)
    ))

(defvar *io-echo-p* nil)
(defvar *io-echo-report-p* t)
(defvar *io-echo-stats* nil)
(defvar *io-echo-count* 0)

(defun io-echo (ch)
  ;;(break "io-echo")
  (let ((stat (assoc ch *io-echo-stats*)))
    (if stat
	(incf (cdr stat))
	(setf *io-echo-stats* (acons ch 1 *io-echo-stats*))))
    
  (when *io-echo-p*
    (format t "~a" ch))
  (incf *io-echo-count*)
  (when (> *io-echo-count* 4096)
    (format t "IO-echo line wrap[4096] ~%")
    (setf *io-echo-count* 0)))

(defmacro show-telemetry (s &rest args)
  `(when *io-echo-p*
    (format t ,s ,@args)))

(defun io-echo-on () (setf *io-echo-p* t))
(defun io-echo-off () (setf *io-echo-p* nil))
(defun io-echo-report-on () (setf *io-echo-report-p* t))
(defun io-echo-report-off () (setf *io-echo-report-p* nil))
(defun io-echo-report-toggle () (setf *io-echo-report-p* (not *io-echo-report-p*)))

(defmacro with-io-echo-stats ((p tag) &body body)
  `(let ((*io-echo-stats* nil)
	 (*io-echo-p* ,p))
    (format t "~% IO Stats [~a] ~6F : ~%" ,tag *bus-poll-sleep-quantum*)
    (multiple-value-prog1 (progn ,@body)
      (when *io-echo-report-p*
	(terpri)
	(dolist (stat (sort *io-echo-stats* #'> :key #'cdr))
	  (format t "~a, ~12:a :  ~6:D~%"
		  (car stat)
		  (or (cdr (assoc (car stat) *io-echo-stat-names* :test #'string=)) (car stat))
		  (cdr stat)))
	(terpri)))))


;;;;
;;;;	IO record keeping
;;;;

(defvar *disksave-version* nil)
(defvar *io-history* nil)
(defvar *io-history-count* 0)
(defvar *io-trace-file* nil)
(defvar *io-trace-file-name* "")

(defun io-trace-file-p () (and *io-trace-file* t))

(defun push-io-history (data inout)

  (when *io-trace-file*
    (write-io-trace-file inout data))

  (incf *io-history-count*)
  (push (cons inout data) *io-history*)
  
  (when (> *io-history-count* 30)
    (setf *io-history-count* 20)
    (setf (cdr (nthcdr 20 *io-history*)) nil)))

;;;; RLE ??? Possible that output will not be parseable.
;;;; RLE ??? ie output not neccessarily ascii, although as far as I know
;;;; RLE ??? output of standard characters is converted to ascii. aclplc ???
(defun write-io-trace-file (action data)

  (io-echo "t")
  (format *io-trace-file* "~a ~a ~a:~%~a~%~%"
	  action
	  (when (in-environment-p) (tags-of-environment (current-environment)))
	  (datetime-string (get-universal-time))
	  (if (term-p data)
	      (term-to-standard-character-string data)
	      data)))

(defun close-io-trace-file ()
  (when *io-trace-file*
    (format t "~%closed io journal file: ~a.~%" *io-trace-file-name*)
    (setf *io-trace-file-name* nil)
    (format *io-trace-file* "~%~%~%CLOSE **** Closed File [~a] ****~%~%~%"
	    (datetime-string (get-universal-time)))
    (close *io-trace-file*)
    (setf *io-trace-file* nil)))

(defvar *io-jnl-index* 0)

(defun open-io-trace-file (prefix &optional rename-p)

  (close-io-trace-file)
  
  (let ((fname (format-string "~aspool/~a-io~a.jnl"
			      (namestring (truename (user-homedir-pathname)))
			      prefix (incf *io-jnl-index*)))) ;;(format-string "/home/lolorigo/spool/~a-io~a.jnl" prefix (incf *io-jnl-index*))))
    (setf *io-trace-file-name* fname)
    (format t "~%~%io journal file: ~a.~%" fname)

    (setf *io-trace-file* (open fname
				  :direction :output
				  :if-exists (if rename-p :new-version :supersede)
				  :if-does-not-exist :create))
    (format *io-trace-file*
	    "~%~%~%OPEN **** Opened File [~a ~a] ****~%~%~%"
	    (or *disksave-version* "")
	    (datetime-string (get-universal-time)))))

(defun flush-io-history ()
  (when *io-trace-file*
    (finish-output *io-trace-file*)))


;;;;
;;;;	**new-stuff**	 to replace code surrounded by #| |#
;;;;

;;;;
;;;;	Foreign Functions
;;;;

#-:dlwin
(export '(new-socket destroy-socket))

;;;;	Load Foreign Files

;;;; LAL TODO: cmucl has predefined functions, may use instead of ffi
;;;; LAL TODO?: make 1 ffi-make-socket function that takes block-p as parameter
;;;; (instead of 2 separate functions)

#+allegro (require :foreign)

#+cmu
(progn
  (defunff (make-nonblocking-socket ffi-make-socket) int ((port int) (queue-size int)))
  (defunff (establish-connection-server ffi-make-socket-b) int ((port int)))
  (defunff (establish-connection ffi-open-client) int ((port int) (host string)))
  (defunff (server-accept ffi-open-server) int ((sock int)))
  (defunff (close-connection ffi-close-connection) int ((fd int)))
  )

#+:lucid
(progn
  (def-foreign-function (ffi-make-socket
			 (:language :c)
 			 (:name "_make_nonblocking_socket")
			 (:return-type :signed-32bit))  
    (port :signed-32bit)
    (queue_size :signed-32bit))
  (def-foreign-function (ffi-make-socket-b
			 (:language :c)
			 (:name "_establish_connection_server")
			 (:return-type :signed-32bit))  
    (port :signed-32bit))
  (def-foreign-function (ffi-open-client
			 (:language :c)
			 (:name "_establish_connection")
			 (:return-type :signed-32bit))  
    (port :signed-32bit)
    (host :simple-string))
  (def-foreign-function (ffi-open-server
			 (:language :c)
			 (:name "_server_accept")
			 (:return-type :signed-32bit))  
    (sock :signed-32bit))
  (def-foreign-function (ffi-close-connection
			 (:language :c)
			 (:name "_close_connection"))
    (fd1 :signed-32bit)))


#+:allegro
(progn
  (ff:defforeign 'ffi-make-socket
		 :entry-point (ff:convert-to-lang "make_nonblocking_socket")
		 :arguments '(integer integer)
		 :return-type :integer)
  (ff:defforeign 'ffi-make-socket-b
	 	 :entry-point (ff:convert-to-lang "establish_connection_server")
		 :arguments '(integer)
		 :return-type :integer)
  (ff:defforeign 'ffi-open-client
		 :entry-point (ff:convert-to-lang "establish_connection")
		 :arguments '(integer string)
		 :return-type :integer)
  (ff:defforeign 'ffi-open-server
		 :entry-point (ff:convert-to-lang "server_accept")
		 :arguments '(integer)
		 :return-type :integer)  
  (ff:defforeign 'ffi-close-connection
		 :entry-point (ff:convert-to-lang "close_connection")
		 :arguments '(integer)))


;;;;
;;;;	Sockets
;;;;

(defstruct socket
  (port nil)
  (fd nil)
  (stream-fd nil)
  (state nil)

  ;; these fields not needed, but may be useful in the future
   ;;(server-p nil)
   ;;(in-p nil)
   ;;(block-p nil)
  )

;;(defun server-p-of-socket (socket) (socket-server-p socket))
;;(defun in-p-of-socket (socket) (socket-in-p socket))
;;(defun block-p-of-socket (socket) (socket-block-p socket))

(defun port-of-socket (socket) (socket-port socket))
(defun fd-of-socket (socket) (socket-fd socket))
(defun stream-fd-of-socket (socket) (when socket (socket-stream-fd socket)))
(defun state-of-socket (socket) (when socket (socket-state socket)))

;;;;	
;;;;	Socket State
;;;;	  - secondary listening
;;;;	  - primary connected

;; client for server to connect to : (new-socket local-port nil t 1)
(defun new-socket-listen (port &optional (queue-size 5))
  (let ((fd (ffi-make-socket port queue-size)))
    (when (null fd)
      (raise-error (error-message '(socket-listen) port)))
  
    (make-socket :port port
		 :state 'listening
		 :fd fd)))

(defun new-socket-connect (port host)
  (let ((fd (ffi-open-client port host)))
    (when (null fd)
      (raise-error (error-message '(socket-connect) port)))

    (make-socket :port port
		 :state 'connected
		 :fd fd)))

(defun new-socket-accepted (port fd)
  (make-socket :port port
	       :fd fd
	       :state 'connected))


(defun socket-connected-p (s) (eql 'connected (state-of-socket s)))


(defun destroy-socket (sock &optional stream-p)
  (let ((fd (fd-of-socket sock)))
    (when fd
      (ffi-close-connection fd)
      (ffi-close-connection (port-of-socket sock))))

  (when stream-p
    (let ((stream-fd (stream-fd-of-socket sock)))
      (when stream-fd
	(ffi-close-connection stream-fd))))

  (setf (socket-state sock) 'destroyed))
	  
     
;;;;
;;;;	Channels
;;;;

(defvar *channel-models* (acons 'sml 'stream (acons 'syn 'tooltalk nil)))

(defstruct channel-dispatch
  (model nil)
  (open-f nil)
  (close-f nil)
  (send-f nil)
  (recv-f nil)
  (listen-f nil)
  (errors-f nil))

(defstruct channel
  (dispatch-vector nil)
  (kind nil)
  (state nil))


(defun state-of-channel (c) (channel-state c))
(defun kind-of-channel (c) (channel-kind c))

(defun open-f-of-channel (c) (channel-dispatch-open-f (channel-dispatch-vector c)))
(defun close-f-of-channel (c) (channel-dispatch-close-f (channel-dispatch-vector c)))
(defun send-f-of-channel (c) (channel-dispatch-send-f (channel-dispatch-vector c)))
(defun recv-f-of-channel (c) (channel-dispatch-recv-f (channel-dispatch-vector c)))
(defun errors-f-of-channel (c) (channel-dispatch-errors-f (channel-dispatch-vector c)))
(defun listen-f-of-channel (c) (channel-dispatch-listen-f (channel-dispatch-vector c)))

;; <channel>
(defun channel-open (channel)
  (funcall (open-f-of-channel channel) channel))

(defun channel-close (channel)
  (funcall (close-f-of-channel channel) channel))

(defun channel-send (channel data)
  (funcall (send-f-of-channel channel) channel data))

(defun channel-recv (channel blockp)
  (if blockp
      (funcall (recv-f-of-channel channel) channel)
      (when (channel-listen channel)
	(funcall (recv-f-of-channel channel) channel))))

(defun channel-errors (channel)
  (funcall (errors-f-of-channel channel) channel))

(defun channel-listen (channel)
  (funcall (listen-f-of-channel channel) channel))

(defun channel-dispatch (model)
  (make-channel-dispatch   
   :model model
   :open-f (case model
	     (stream #'stream-channel-open)
	     (uncompressed-stream #'uncompressed-stream-channel-open)
	     (standard-ascii-stream #'uncompressed-stream-channel-open)
	     (mathbus #'mathbus-channel-open)
	     (tt #'tt-channel-open)
	     (soft nil)
	     (otherwise (raise-error (error-message '(channel dispatch model not)
						    model))))
   :close-f (case model
	      (stream #'stream-channel-close)
	      (uncompressed-stream #'stream-channel-close)
	      (standard-ascii-stream #'stream-channel-close)
	      (mathbus #'mathbus-channel-close)
	      (tt #'tt-channel-close)
	      (soft #'accept-channel-close)
	      (otherwise (raise-error (error-message '(channel dispatch model not)
						     model))))
   :send-f (case model
	     (stream #'stream-channel-send)
	     (uncompressed-stream #'uncompressed-stream-channel-send)
	     (standard-ascii-stream #'standard-ascii-stream-channel-send)
	     (mathbus #'mathbus-channel-send)
	     (tt #'tt-channel-send)
	     (soft nil)
	     (otherwise (raise-error (error-message '(channel dispatch model not)
						    model))))
   :recv-f (case model
	     (stream #'stream-channel-recv)
	     (uncompressed-stream #'uncompressed-stream-channel-recv)
	     (standard-ascii-stream #'standard-ascii-stream-channel-recv)
	     (mathbus #'mathbus-channel-recv)
	     (tt #'tt-channel-recv)
	     (soft nil)
	     (otherwise (raise-error (error-message '(channel dispatch model not)
						    model))))
   :listen-f (case model
	       (stream #'stream-channel-listen)
	       (uncompressed-stream #'uncompressed-stream-channel-listen)
	       (standard-ascii-stream #'uncompressed-stream-channel-listen)
	       (mathbus #'mathbus-channel-listen)
	       (tt #'tt-channel-listen)
	       (soft  #'accept-channel-listen)
	       (otherwise (raise-error (error-message '(channel dispatch model not)
						      model))))
   :errors-f (case model
	       (stream #'stream-channel-errors)
	       (uncompressed-stream #'stream-channel-errors)
	       (standard-ascii-stream #'stream-channel-errors)
	       (mathbus #'mathbus-channel-errors)
	       (tt #'tt-channel-errors)
	       (soft #'accept-channel-errors)
	       (otherwise (raise-error (error-message '(channel dispatch model not) model))))
   ))


;;;;
;;;;	Stream channels
;;;;

;;;;	Allegro Streams Fixes
	       
#+(or :allegro-v4.3.1 allegro-v4.3 allegro-v5.0 allegro-v5.0.1 allegro-v6.2 allegro-v6.1 allegro-v6.0 ) ;;nil
(progn
  (defun my-stream-write-byte (stream byte)
    (stream-write-byte stream byte))
  (defun my-stream-read-byte (stream)
    (stream-read-byte stream))
  )

#| 
#+(or :allegro-v4.0 :allegro-v4.1 :allegro-v4.2 :allegro-v4.3.1 :dlwin)
(progn
  #+(or :allegro-v4.3.1 :dlwin)
  (defclass binary-socket-stream (stream:bidirectional-terminal-stream) ())
  #-(or :allegro-v4.3.1 :dlwin)
  (defclass binary-socket-stream (excl::bidirectional-terminal-stream) ())
 
  (defclass binary-socket-stream (stream:bidirectional-binary-socket-stream) ())
  ;;(defclass binary-bio-stream (stream:binary-bidirectional-file-stream) ())
  (defclass binary-terminal-stream (stream:bidirectional-terminal-stream)  ())
  
  (defmethod stream-read-byte ((s binary-socket-stream))
    ;;(or (read-byte s nil nil) :eof)
    (progn;;(do ((data-p (stream:stream-listen s) (stream:stream-listen s)))
      ;; (data-p nil)
      ;;  )
      (let ((c (read-byte s nil nil)))
	;;   (setf b c) (unless  c (break))
	(or c :eof))))
  
  (defmethod stream-write-byte ((s binary-socket-stream) integer)
    (write-byte integer s)
    (finish-output s))
  
  (defmethod stream-read-byte ((s binary-socket-stream))
    (let ((c (read-char s)))
      (if (eq :eof c)
	  :eof
	  (char-code c))))

  (defmethod stream-write-byte ((s binary-socket-stream) integer)
    (let ((c (code-char integer)))
      (write-char s c))))
|#

#+(or :allegro-v5.0 allegro-v5.0.1 allegro-v6.2 allegro-v6.1 allegro-v6.0  :allegro-v4.0 :allegro-v4.1 :allegro-v4.2 :allegro-v4.3 :allegro-v4.3.1 :dlwin)
(progn
  (defun char-stream-read-byte (stream)
    (let ((c (read-char stream)))
      (if (eq :eof c)
	  :eof
	  (char-code c))))

  (defun char-stream-write-byte (byte stream)
    (let ((c (code-char byte)))
      (write-char c stream))))

(defstruct (socket-channel (:include channel))

  (primary-socket nil)
  (secondary-socket nil)
  )
  
(defun primary-socket-of-channel (c) (socket-channel-primary-socket c))
(defun secondary-socket-of-channel (c) (socket-channel-secondary-socket c))

(defstruct (stream-channel-aux (:include socket-channel))

  (in-stream nil)
  (out-stream nil)
  
  (out-f nil)
  (in-f nil)
  )

(defun in-stream-of-stream-channel (c) (stream-channel-aux-in-stream c))
(defun out-stream-of-stream-channel (c) (stream-channel-aux-out-stream c))

(defun out-f-of-stream-channel (c) (stream-channel-aux-out-f c))
(defun in-f-of-stream-channel (c) (stream-channel-aux-in-f c))

(defstruct (stream-channel (:include stream-channel-aux (dispatch-vector (channel-dispatch 'stream))))
  (out-compression-levels nil)
  )

(defun compression-levels-of-stream-channel (c) (stream-channel-out-compression-levels c))

(defstruct (uncompressed-stream-channel
	     (:include stream-channel-aux (dispatch-vector
					   (channel-dispatch 'uncompressed-stream))))
  )

(defstruct (standard-ascii-stream-channel
	     (:include stream-channel-aux (dispatch-vector
					   (channel-dispatch 'standard-ascii-stream))))
  )

(defun maybe-update-out-stream-channel-compression-levels (c levels)
  ;; NFG : updating out-compression-levels not same as update out-stream levels.
  (unless (stream-channel-out-compression-levels c)
    (cprl-stream-out-update-levels (out-stream-of-stream-channel c) levels)
    (setf (stream-channel-out-compression-levels c) levels)))

#|(defun new-stream-channel (in-sock out-sock &optional out-f in-f)
  (make-stream-channel :in-socket in-sock
		       :out-socket out-sock
		       :out-f out-f
		       :in-f in-f))
|#

(defun new-cprl-stream-channel (sock levels)
  (make-stream-channel :kind 'ascii-compressed
		       :out-compression-levels levels
		       :primary-socket sock
		       :state 'primary))

(defun new-prl-stream-channel (primary-sock)
  (make-uncompressed-stream-channel
   :kind 'ascii-uncompressed
   :primary-socket primary-sock
   :state 'primary
   ))

(defun new-standard-ascii-stream-channel (primary-sock)
  (make-standard-ascii-stream-channel
   :kind 'standard-ascii
   :primary-socket primary-sock
   :state 'primary
   ))

;; maybe use to check to bother with two socket connections?
(defvar *uni-dir-p*
  #+(or allegro-v5.0 allegro-v5.0.1  allegro-v4.3.1 allegro-v4.3) t
  #-(or allegro-v5.0 allegro-v5.0.1  allegro-v4.3.1 allegro-v4.3) nil)

;; returns nil if no client waiting, else fd
(defun accept-new-client (sock)
  (let ((fd (the fixnum	(ffi-open-server (fd-of-socket sock)))))
    (if (>= fd 0)
	fd
	(if (> fd -10000)
	    (error "problem accepting new client")
	    nil))))

(defun make-io-stream (fd-in fd-out &optional (type '(unsigned-byte 8))
		       (instance #-cmu 'stream:bidirectional-binary-socket-stream))

  #+(or allegro-v6.2 allegro-v6.1 allegro-v6.0
	allegro-v5.0.1 allegro-v5.0
	allegro-v4.3.1 allegro-v4.3 allegro-v4.2 allegro-v4.1 allegro-v4.0)
  (make-instance instance
		 :fn-in fd-in
		 :fn-out fd-out
		 :element-type type
		 :eof nil
		 :error 5)

   #+:lucid
  (make-lisp-stream :input-handle fd-in
		    :output-handle fd-out
		    :element-type type
		    :auto-force nil
		    :positionable nil)
  #+allegro-v3.1
  (excl::make-buffered-terminal-stream fd-in fd-out t t)

  #+:dlwin
  (make-instance instance
		 :fn-in fd-in
		 :fn-out fd-out
		 :element-type type)

  #+cmu
  (let ((in (system:make-fd-stream fd-in :element-type type :input t :buffering :none))
        (out (system:make-fd-stream fd-out :element-type type :output t :buffering :none))
	)
    (make-two-way-stream in out)
    ))


(defun make-in-stream (fd &optional (type '(unsigned-byte 8)))
   #+:lucid
  (make-lisp-stream :input-handle fd
		    :element-type type
		    :auto-force nil
		    :positionable nil)
  #+allegro-v3.1
  (excl::make-buffered-terminal-stream fd fd t t)
  #+(or allegro-v6.2 allegro-v6.1 allegro-v6.0 allegro-v5.0 allegro-v5.0.1 allegro-v4.0 allegro-v4.1 allegro-v4.2 allegro-v4.3 allegro-v4.3.1)
  (make-instance 'stream:input-binary-socket-stream
		 :fn-in fd
		 :element-type type
		 :eof nil
		 :error 5)
  #+:dlwin
  (make-instance 'stream:input-binary-socket-stream
   :fn-in fd
   :element-type type)
  #+cmu
  (system:make-fd-stream fd :element-type type :input t :output t :buffering none))
  
(defun make-out-stream (fd &optional (type '(unsigned-byte 8)))
   #+:lucid
  (make-lisp-stream :output-handle fd
		    :element-type type
		    :auto-force nil
		    :positionable nil)
  #+allegro-v3.1
  (excl::make-buffered-terminal-stream fd fd t t)
  #+(or allegro-v6.2 allegro-v6.1 allegro-v6.0 allegro-v5.0 allegro-v5.0.1 allegro-v4.0 allegro-v4.1 allegro-v4.2 allegro-v4.3 allegro-v4.3.1)
  (make-instance 'stream:output-binary-socket-stream
		 :fn-out fd
		 :element-type type
		 :eof nil
		 :error 5)
  #+:dlwin
  (make-instance 'stream:output-binary-socket-stream
   :fn-out fd
   :element-type type)
  #+cmu
  (system:make-fd-stream fd :element-type type :input t :output t :buffering none))

(defun lisp-stream-read (s eof1 eof2)
  #-dlwin (read-byte s eof1 eof2)
  #+dlwin (my-stream-read-byte s))

(defun lisp-stream-write (byte s)
  #-dlwin (write-byte byte s)    
  #+dlwin (my-stream-write-byte s byte))

(defun byte-stream-write-buffer (s buffer offset count)
  #+lucid (write-array s buffer offset (+ count offset))
  #+allegro (stream:stream-write-sequence s buffer offset (+ count offset))
  #+cmu (progn (write-sequence buffer s :start offset :end (+ count offset)) (finish-output s)))

;;;; LAL: server may want to open stream on out-socket?
(defun stream-channel-open (channel)
  (socket-channel-open #'(lambda (fd)
			   (make-socket-streams
			    #'(lambda (in-stream)
				(new-cprl-in-stream (new-prl-in-stream in-stream #'lisp-stream-read)			 
						    #'(lambda (levels)
							(maybe-update-out-stream-channel-compression-levels 
							 channel levels))))
			    #'(lambda (out-stream)
				(new-cprl-out-stream (new-prl-out-stream out-stream
									 #'lisp-stream-write
									 #'byte-stream-write-buffer)
						     (compression-levels-of-stream-channel channel)
						     t))
			    fd))
		       channel))
		       
			   
#|
  ;;(break "AP")
  (let ((fd-in (stream-fd-of-socket (in-socket-of-stream-channel channel)))
	(fd-out (stream-fd-of-socket (out-socket-of-stream-channel channel))))

    (when (and fd-in (null (in-stream-of-stream-channel channel)))
      ;;  create in-stream
      (let ((stream (if *uni-dir-p* (make-in-stream fd-in) (make-io-stream fd-in fd-in))))
	(when (null stream)
	  (raise-error (error-message '(stream channel open cprl in stream  not))))
	(setf (stream-channel-in-stream channel)
	      (new-cprl-in-stream (new-prl-in-stream stream #'lisp-stream-read)			 
				  #'(lambda (levels)
				      (maybe-update-out-stream-channel-compression-levels 
					channel levels))))
	(unless fd-out
	  (setf (stream-channel-out-stream channel);;prob here
		(new-cprl-out-stream (new-prl-out-stream (if *uni-dir-p*
							     (make-out-stream fd-in)
							     stream)
							 #'lisp-stream-write
							 #'byte-stream-write-buffer)
				     (compression-levels-of-stream-channel channel)
				     t)))))

    (when (and fd-out (null (out-stream-of-stream-channel channel)))
      ;;  create out-stream	  
      (let ((stream (if *uni-dir-p* (make-out-stream fd-out) (make-io-stream fd-out fd-out))))
	(when (null stream)
	  ;;(setf a fd-out)
	  (raise-error (error-message '(stream channel open cprl out stream  not))))

	(setf (stream-channel-out-stream channel)
	      (new-cprl-out-stream (new-prl-out-stream stream
						       #'lisp-stream-write
						       #'byte-stream-write-buffer)
				   (compression-levels-of-stream-channel channel)
				   t))	    
	(unless fd-in
	  (setf (stream-channel-in-stream channel)
		(new-cprl-in-stream (new-prl-in-stream (if *uni-dir-p* 
							   (make-in-stream fd-out) 
							   stream) 
						       #'lisp-stream-read)
				    #'(lambda (levels)
					(maybe-update-out-stream-channel-compression-levels channel levels))))))))
|#

;; opened twice if duplex once for first socket and again for second.
;; state :
;;   primary -> open primary
;;   secondary -> open secondary
(defun make-socket-streams (in-f out-f fd)
  (if *uni-dir-p*
      (values (funcall in-f (make-in-stream fd)) (funcall out-f (make-out-stream fd)))
      (let ((io-stream (make-io-stream fd fd)))
	(values (funcall in-f io-stream) (funcall out-f io-stream)))))


(defun socket-channel-open (make-streams-f channel)

  (case (state-of-channel channel)
    (primary
     (let ((sock (primary-socket-of-channel channel)))
       (unless (socket-connected-p sock)
	 (raise-error '(socket-channel-open primary socket connected not)))
       (let ((fd (fd-of-socket sock)))

	 (mlet* (((in-stream out-stream) (funcall make-streams-f fd)))

		(when (null in-stream)
		  (raise-error (error-message '(socket-channel-open primary in-stream not))))
		(when (null out-stream)
		  (raise-error (error-message '(socket-channel-open primary out-stream not))))
		
		(setf (stream-channel-in-stream channel) in-stream
		      (stream-channel-out-stream channel) out-stream
		      (channel-state channel) 'primary-stream)))))
    
    (secondary

     ;; later :
     ;; set old out to be hipo out.
     ;; open secondary stream and set it to be out stream
     ;; set in-stream to be hipo in.

     ;; FTTB : secondary stream is pointless except that one less thing to add when add hipri streams.
     ;; open secondary stream and set it to be out stream
     
     (let ((sock (secondary-socket-of-channel channel)))
       (unless (socket-connected-p sock)
	 (raise-error '(socket-channel-open secondary socket connected not)))

       (when nil ;; fttb ignore
	 (let ((fd (fd-of-socket sock)))
	   (let ((out-stream (make-out-stream fd)))
	   
	     (when (null out-stream)
	       (raise-error (error-message '(socket-channel-open secondary out-stream not))))

	     (setf (stream-channel-out-stream channel) (funcall make-out-f  out-stream)
		   (channel-state channel) 'duplex-stream))))))
      
    (otherwise
     ;; should we be calling open when there's nothing to do?
     (break "uncompressed-stream-channel-open")
     nil)))

(defun uncompressed-stream-channel-open (channel)
  (socket-channel-open #'(lambda (fd)
			   (make-socket-streams 
			    #'(lambda (in-stream)
				(new-prl-in-stream in-stream
						   #'lisp-stream-read)) 
			    #'(lambda (out-stream)
				(new-prl-out-stream out-stream
						    #'lisp-stream-write
						    #'byte-stream-write-buffer))
			    fd))
		       
			channel))



(defun stream-channel-close (channel)
  (let* ((primary-socket (primary-socket-of-channel channel))
	 (secondary-socket (secondary-socket-of-channel channel)))

    (when primary-socket
      (destroy-socket primary-socket t))
    (when secondary-socket
      (destroy-socket secondary-socket t))
    (setf (channel-state channel) 'closed)))


(defun stream-channel-send (channel data &optional array-p)
  (let ((stream (out-stream-of-stream-channel channel)))    
    (push-io-history data 'send)    
    (if array-p
	(progn (break "stream-channel send") 
               (funcall (out-f-of-stream-channel channel) stream data)) ;; ???
	(cprl-stream-write stream data))

    (io-echo ">")
    (cprl-stream-finish-output stream)
    ))

(defun uncompressed-stream-channel-send (channel data)
  (let ((stream (out-stream-of-stream-channel channel)))    
    (push-io-history data 'send)    

    (write-term-to-ascii-stream data stream)

    (io-echo ">")
    (prl-stream-finish-output stream) ))

(defun standard-ascii-stream-channel-send (channel data)
  (let ((stream (out-stream-of-stream-channel channel)))    
    (push-io-history data 'send)    

    (write-term-to-standard-ascii-stream data stream)

    (io-echo ">")
    (prl-stream-finish-output stream) ))


;; desire compression-levels be accesible to allow
;; mod during io. ie compression is channel/stream level not link?
;; may need to be rehashed if ever multichannel links.
(defun stream-channel-recv (channel)
  (let ((data nil))
    (tagbody loop   
       (setf data (cprl-stream-read (in-stream-of-stream-channel channel)))
       (push-io-history data 'recv)
       ;;(break "recv 1")
       (io-echo "<")
       (when (null data);;data null means proc died
	 (break "null data") (raise-error (error-message '(channel recv not) nil)))
       (when (iping-term-p data)
	 (stream-channel-send channel (ipong-term (sequence-of-iping-term data)))
	 ;; might make sense to return nil to avoid waiting on read.
	 (return-from stream-channel-recv nil)
	 (go loop)))      
    data))

;; try hipo first
(defun uncompressed-stream-channel-recv (channel)
  ;;(break "recv")
  (let ((data nil))
    (tagbody loop   

       (setf data (read-term-from-ascii-stream (in-stream-of-stream-channel channel)))
       
       (push-io-history data 'recv)
       (io-echo "<")
       (when (null data);;data null means proc died
	 (break "null data") (raise-error (error-message '(channel recv not) nil)))
       ;;(setf -data data) (break "uscr")
       (when (iping-term-p data) 
	 (uncompressed-stream-channel-send channel (ipong-term (sequence-of-iping-term data)))
	 (return-from uncompressed-stream-channel-recv nil)
	 (go loop)))      
    data))

(defun standard-ascii-stream-channel-recv (channel)
  ;;(break "recv")
  (let ((data nil))
    (tagbody loop   

       (setf data (read-term-from-standard-ascii-stream (in-stream-of-stream-channel channel)))
       
       (push-io-history data 'recv)
       (io-echo "<")
       (when (null data);;data null means proc died
	 (break "null data") (raise-error (error-message '(channel recv not) nil)))
       ;;(setf -data data) (break "uscr")
       (when (iping-term-p data) 
	 (standard-ascii-stream-channel-send channel (ipong-term (sequence-of-iping-term data)))
	 (return-from standard-ascii-stream-channel-recv nil)
	 (go loop)))      
    data))


(defun stream-channel-listen (channel) 
  (cprl-stream-listen (in-stream-of-stream-channel channel)))

;; listen on in-stream and hipo-in
(defun uncompressed-stream-channel-listen (channel) 
  (prl-stream-listen (in-stream-of-stream-channel channel)))

(defun stream-channel-errors (&rest rest)
 (declare (ignore rest)) (values))
(defun accept-channel-errors (&rest rest)
 (declare (ignore rest)) (values))

;; LAL TODO: aclpc code

;;;;	 end of **new-stuff** that replaces #| |#



;;;; LAL TODO: tt code will need adjusting to support channel changes      

;;;
;;;	tt-channels
;;;


;;;	send : code length term-string.
;;;	 nuprl-tt  needs some indication as to what is being sent so as to mark
;;;	  as broadcast or request. We assume tt is delivering in sequential order
;;;	  thus no need to delay requests and no need for acks on broadcasts.
;;;
;;;	 Since we do not expect acks, and do not want multiple acks, the io layer
;;;	 will ack all broadcasts and then discard any acks not bound for pending requests.
;;;
;;;	recv : length term-string.




(defstruct (tt-channel (:include channel (dispatch-vector (channel-dispatch 'tt))))
  (types nil)
  (session "")
  (stream nil)
  (err-stream nil)
  (pid nil)
  (out-f nil)
  (in-f nil)
  (io-f nil)
  )


(defun types-of-tt-channel (ttc) (tt-channel-types ttc))
(defun session-of-tt-channel (ttc) (tt-channel-session ttc))
(defun stream-of-tt-channel (ttc) (tt-channel-stream ttc))
(defun err-stream-of-tt-channel (ttc) (tt-channel-err-stream ttc))
(defun pid-of-tt-channel (ttc) (tt-channel-pid ttc))
(defun out-f-of-tt-channel (ttc) (tt-channel-out-f ttc))
(defun in-f-of-tt-channel (ttc) (tt-channel-in-f ttc))
(defun io-f-of-tt-channel (ttc) (tt-channel-io-f ttc))

(defun new-tt-channel (types out-f in-f io-f &optional session)
  (make-tt-channel :types types
		   :out-f out-f :in-f in-f :io-f io-f
		   :session (if session session "")
		   ))


#+lucid
(defun tt-channel-open (channel)
  (tt-channel-close channel)

  (let ((ok nil))
    (unwind-protect
	 (progn 
	   (mlet* (((stream err-stream status pid)
		    (run-program (complete-system-path '("bin") "nuprl-tt")
				 :arguments (cons (session-of-tt-channel channel)
						  (types-of-tt-channel channel))
				 :input :stream
				 :output :stream
				 :error-output :stream 
				 :wait nil)
		    (declare (ignore status))))
		  (setf (tt-channel-stream channel)
			(new-prl-in-out-stream stream
					      #'(lambda (s eof1 eof2)
						  (char-code (read-char s eof1 eof2)))
					      #'(lambda (byte s)
						  (write-char (code-char byte) s))))
		  (setf (tt-channel-err-stream channel) err-stream)
		  (setf (tt-channel-pid channel) pid))
	   (setf ok t))
      (unless ok
	(tt-channel-close channel))))  )

#+lucid
(defun tt-channel-close (channel)
  
  (when (stream-of-tt-channel channel)
    (setf (tt-channel-stream channel)
	  (prl-stream-close (stream-of-tt-channel channel))))
  (when (err-stream-of-tt-channel channel)
    (setf (tt-channel-err-stream channel)
	  (close (err-stream-of-tt-channel channel))))

  ;;(run-program "kill"
  ;;:arguments (prin1-to-string (pid-of-tt-channel channel))
  ;;:wait t)
  )


;;;;	
;;;;	RLE ??? : would rather get compile-time error???
;;;;	RLE ??? : or do not want error unless actual attempt to use it.
;;;;	RLE TODO : all tt crap should be conditionalized away for aclpc.
;;;;


#-lucid 
(defun tt-channel-open (&rest args)
  (declare (ignore args))
  (error "tt-channel-open not defined for lisp version in use."))

#-lucid 
(defun tt-channel-close (&rest args)
  (declare (ignore args))
  (error "tt-channel-close not defined for lisp version in use."))


;;;	1 Expression
;;;	2 Broadcast	
;;;	3 Message


;;;	4 Result




;;;;	RLE TODO : In tt io, mixing of string and byte io bound to be a problem.
;;;;	RLE TODO : Determine type of stream run-program returns and coerce calls appropriately.

(defun tt-channel-send (channel data)

  ;; write
  (let ((stream (stream-of-tt-channel channel))
	(iotype (funcall (io-f-of-tt-channel channel) data)))
    

    ;;(setf a data b iotype) (break)
    ;;(setf a data) (when (= 5 iotype) (break))
   (push-io-history data 'send)
   (format t ">")

   ;;(setf a data) (break "tcs")
    
   (with-byte-accumulator (stream)

			  ;; accumulate bytes
			  (funcall (out-f-of-tt-channel channel) data)
			  
			  ;; 
			  (let ((s (format-string "~a ~a " (accumulated-length) iotype)))
			    (dotimes (i (length s))
				     (prl-stream-write (character-to-code (char s i))  stream)))

			  ;; with-byte-accumulator will dump accumlated bytes to stream on exit.
			  )

   (prl-stream-finish-output stream)))


(defun tt-channel-recv (channel)

  ;; rle could avoid creating s by using scan-term-ascii

  (let ((data nil))
    (tagbody
	 
     loop
	 
       ;; read
       (let* ((stream (stream-of-tt-channel channel))
	      (length (read stream nil nil)))
	
	 (when (null length)
	       (raise-error (error-message '(tooltalk recv EOF))))

	 (setf data (with-tag '(channel recv tooltalk ascii-to-data)
			      (funcall (in-f-of-tt-channel channel) stream)))

	 (push-io-history data 'recv)
	 (io-echo "<")
	   
	 (when (iping-term-p data)
	       (tt-channel-send channel (ipong-term (sequence-of-iping-term data)))
	       (go loop))))

    data))

(defun tt-channel-errors (channel)
  (let ((stream (err-stream-of-tt-channel channel)))
    (when (listen stream)
      (format t
	      "ErrStream:~% ~a~%End ErrStream.~%" 
	      (do ((acc (list (read-char stream)) (cons (read-char stream) acc)))
		  ((not (listen stream)) (coerce (nreverse acc) 'string))))
      t)))


(defun tt-channel-listen (channel)
  (prl-stream-listen (stream-of-tt-channel channel)))


;;;
;;;	Link
;;;

(defstruct link
  (channels nil)
  (asynch-ok-p t)
  )

(defun channels-of-link (l) (link-channels l))
 


(defun link-port-p (link port)
  (if port
      (some #'(lambda (channel)
		(and (socket-channel-p channel)
		     (or (let ((psock (primary-socket-of-channel channel)))
			   (and psock
				(= port (port-of-socket psock))))
			 (let ((ssock (secondary-socket-of-channel channel)))
			   (and ssock
				(= port (port-of-socket ssock)))))))
	    (channels-of-link link))
    (tt-channel-p (car (channels-of-link link)))))

;;;; LAL: now server opens in-socket and client out, but want flexibility, so maintain
;;;; server-p and in-p

(defun new-stream-link (primary-sockets &optional levels)
  (make-link :channels 
	     (mapcar #'(lambda (primary-socket) 
			 (new-cprl-stream-channel primary-socket levels))
		     primary-sockets)
	     ))

(defun new-uncompressed-stream-link (primary-sockets)
  (make-link :channels 
	     (mapcar #'(lambda (primary-socket) 
			 (new-prl-stream-channel primary-socket))
		     primary-sockets)
	     ))

(defun new-standard-ascii-stream-link (primary-sockets)
  (make-link :channels 
	     (mapcar #'(lambda (primary-socket)
			 (new-standard-ascii-stream-channel primary-socket))
		     primary-sockets)
	     ))

   
   
(defun new-tt-link (types out-f in-f io-f &optional session)
  (make-link :channels (list (new-tt-channel types out-f in-f io-f session))
	     :asynch-ok-p nil))
   

;; fttb, groups are static, later define group-add-channel, etc.
;;
;; May be better to open single stream and have c program do multiplexing!!
;; drawback is it complicates the lisp <-> c stream protocol.
;;

(defun link-open (link)
  (mapc #'channel-open (channels-of-link link)))

(defun link-close (link)
  (mapc #'channel-close (channels-of-link link))
  (setf (link-channels link) nil))

;; index : t -> all, nil -> any, i -> i.
(defun link-send (link data &optional index)
  (let ((channels (channels-of-link link)))
    (cond
      ((eql t index)
       (dolist (channel channels)
	 (channel-send channel data)))
      ((null index)
       (if (null (cdr channels))
	   (channel-send (car channels) data)
	   (let ((index (mod (get-universal-time) (length channels))))
	     (channel-send (nth index channels) data)
	     index)))
      (t (channel-send (nth index channels) data)))))
	    
(defun link-recv (link blockp &optional index)
  (declare (ignore index));; later.
  (let ((channels (channels-of-link link)))
    (cond
      ((not (null (cdr channels)))
       (break "Not prepared for multi-channel links yet"))
      (t (channel-recv (car channels) blockp)))))

(defun link-errors (link)
  (let ((error-p nil))
    (dolist (channel (channels-of-link link))
      (when (channel-errors channel) (setf error-p t)))
    error-p))

(defun link-listen (link)
  (do ((index 0 (1+ index))
       (channels (channels-of-link link) (cdr channels)))
      ((or (null channels) (channel-listen (car channels))) (not (null channels)))))


;;;;	soft link : used for journal, toploop, asynch and ???
;;;;	
;;;;	   - basically hooks to serialize lib functions over bus.
;;;;	
;;;;	
;;;;	
;;;;	channel : connects two processes.
;;;;	  - soft channel : one process is virtual,
;;;;	        ie filesystem (journal), user (top), bit bucket (asynch).
;;;;	  
;;;;	hooks : implements functions of virtual process. 
;;;;	 - send : called when send message received from other end.
;;;;	 - recv : called when recv message received from virtual end.
;;;;	     top : user entered command.
;;;;	     asynch : process queued request.
;;;;	 - listen 
;;;;	 - open
;;;;	 - close

;;;;	
;;;;	soft channel
;;;;	
;;;;	

(defstruct (soft-channel (:include channel (dispatch-vector (channel-dispatch 'soft)))))

(defun new-soft-link (c &key (open #'(lambda (&rest rest) (declare (ignore rest))))
			    (close #'(lambda (&rest rest) (declare (ignore rest))))
			    (recv #'(lambda (&rest rest) (declare (ignore rest))))
			    (send #'(lambda (&rest rest) (declare (ignore rest))))
			    (listen #'(lambda (&rest rest) (declare (ignore rest))))
			    (errors #'(lambda (&rest rest) (declare (ignore rest)))))

  (let* ((l (make-link :channels (list c) 
		       :asynch-ok-p t))
	 (dv (channel-dispatch-vector c)))
    (setf (channel-dispatch-open-f dv) open
	  (channel-dispatch-close-f dv) close
	  (channel-dispatch-recv-f dv) recv
	  (channel-dispatch-send-f dv) send
	  (channel-dispatch-listen-f dv) listen
	  (channel-dispatch-errors-f dv) errors)
    l))

(defun channel-of-soft-link (link)
  (car (channels-of-link link)))

(defun soft-link-p (link)
  (eql 'soft (channel-dispatch-model (channel-dispatch-vector (car (channels-of-link link))))))
    

;;;;	
;;;;	toploop channel.
;;;;	

(defstruct (toploop-channel (:include soft-channel))
  recv-queue
  )

(defun recv-queue-of-toploop-channel (ch) (toploop-channel-recv-queue ch))
(defun toploop-channel-recv-queue-push (ch term) (push term (toploop-channel-recv-queue ch)))
(defun toploop-channel-recv-queue-pop (ch) (pop (toploop-channel-recv-queue ch)))
(defun new-toploop-channel ()
  (make-toploop-channel
   :kind 'toploop
   ))
       


;;;;
;;;; accept channels
;;;;

;; LAL: accept should not really be soft, change later
(defstruct (accept-channel (:include soft-channel))
  (levels nil)
  (socket -1)
  (counter 0)
   ;; (increment 20)
  (mathbus-p nil)
  (compress-p nil)

  ;; desire ability to control how spawned links are used.
  (spawn-api-kind nil)
  )

(defun levels-of-accept-channel (channel) (accept-channel-levels channel))
(defun socket-of-accept-channel (channel) (accept-channel-socket channel))
(defun counter-of-accept-channel (channel) (accept-channel-counter channel))
(defun mathbus-p-of-accept-channel (channel) (accept-channel-mathbus-p channel))
(defun compress-p-of-accept-channel (channel) (accept-channel-compress-p channel))
(defun spawn-api-of-accept-channel (channel) (accept-channel-spawn-api-kind channel))

;; (tok{api} # (tok{sentry} # (term -> bool)) list) list
(defvar *api-sentries-alist* nil)

(defun add-api-sentries (api-kind sentries)
  (setf *api-sentries-alist*
	(cons (cons api-kind sentries)
	      (delete api-kind *api-sentries-alist* :key #'car))))

(defun list-api-sentries ()
  (mapcar #'(lambda (ac)
	      (cons (car ac) (length (cdr ac))))
	  *api-sentries-alist*))

(defun install-api-sentries (api blink)
  (dolist (s (cdr (assoc api *api-sentries-alist*)))
    (add-blink-input-sentry blink (car s) (cdr s))))


(defun new-accept-channel (sock &optional mathbus-p levels compress-p spawn-api)
  (make-accept-channel :levels levels
		       :socket sock
		       :kind 'accept
		       :state 'active
		       :counter 0
		       :mathbus-p mathbus-p
		       :compress-p compress-p
		       :spawn-api-kind spawn-api))
  
(defun accept-channel-close (channel)
   (when (accept-channel-p channel)
     (setf (channel-state channel) 'closed)
     (destroy-socket (socket-of-accept-channel channel))))

;;;; always returns nil, but if client waiting opens new stream channel link and
;;;; adds to bus, only checks for client every 20th time
(defun accept-channel-listen (channel)
  (when (accept-channel-p channel)
	(let ((counter (counter-of-accept-channel channel)))
	  (if (and (null (spawn-api-of-accept-channel channel)) ; if api want faster turn around
		   (< counter 10))
	      (setf (accept-channel-counter channel) (1+ counter))
	      (progn
		(let* ((socket (socket-of-accept-channel channel))
		       (fd (accept-new-client socket))
		       (spawn-api (spawn-api-of-accept-channel channel)))
		  (if fd
		      (progn
			;;(format t "~%got a client: ~a~%" fd) (setf -channel channel) (break "acl")
			(let* ((in-socket (new-socket-accepted (port-of-socket socket) fd))
			       (l (if (mathbus-p-of-accept-channel channel)
				      (new-term-mathbus-link (list in-socket))
				      (if (compress-p-of-accept-channel channel)
					  (new-cprl-stream-link (list in-socket)
								(levels-of-accept-channel channel))
					  (new-prl-stream-link (list in-socket) spawn-api)))))

			  ;;(setf -l l -s in-socket) (break "acl2")
			  (link-open l)
			  (let ((blink  (new-bus-link l t)))
			    (add-bus-link blink)
			    (install-api-sentries spawn-api blink))
			  )
			
			(accept-channel-listen channel));; try again
		      (setf (accept-channel-counter channel) 0))))))
	nil))
#|
;; secondary-link

;; completes link connection, returns nil if fails
(defun connect-callback (link host port)
  (let ((channel (car (channels-of-link link)))
	(out-socket (new-socket port host nil)))
    (if out-socket
	(if (mathbus-channel-p channel)
	    (progn
	      (setf (mathbus-channel-out-socket channel) out-socket)
	      (setf (mathbus-channel-out-stream channel)  nil)
	      t)
	  (progn
	    (setf (stream-channel-out-socket channel) out-socket)
	    (setf (stream-channel-out-stream channel) nil)
	    t))
      nil)))

(defun accept-callback (link)
  (let* ((channel (car (channels-of-link link)))
	 (mathbus-p (mathbus-channel-p channel))
	 (fd (or (accept-new-client (if mathbus-p
					(in-socket-of-mathbus-channel channel)
					(in-socket-of-stream-channel channel)))
		 (raise-error (error-message '(accept callback))))))
    	
    (if mathbus-p
	(progn (setf (mathbus-channel-in-stream channel) nil)
	       (setf (socket-stream-fd (in-socket-of-socket-channel channel)) fd))
      (progn (setf (stream-channel-in-stream channel) nil)
	     (setf (socket-stream-fd (in-socket-of-socket-channel channel)) fd)))
    fd))
	    
(defun out-stream-of-link (link) 
  (let ((channel (car (channels-of-link link))))
    (if (mathbus-channel-p channel)
	(mathbus-channel-out-stream channel)
      (stream-channel-out-stream channel))))

(defun set-dummy-out-stream (link &optional stream)
  (let ((channel (car (channels-of-link link))))
    (if (mathbus-channel-p channel)
	(setf (mathbus-channel-out-stream channel)
	      (or stream (in-stream-of-mathbus-channel channel)))
	(progn ;;(setf c channel) (break)
	  (setf (stream-channel-out-stream channel)
		(or stream 
                    (let ((s (prl-stream-stream (prl-stream-of-cprl-stream 
						(in-stream-of-stream-channel channel)))))
		      (new-cprl-out-stream
		        (new-prl-out-stream
		          (if *uni-dir-p*
			    (make-out-stream #+:lucid (lucid::stream-file-handle s)
			                     #+(or allegro cmu) (fd-of-stream s nil))
                            s)
		      #'lisp-stream-write
		      #'byte-stream-write-buffer)
		     nil
		     nil))))))))

|#

(defvar *link-term-hook* nil)

;;;;	S accept 
;;;;
;;;;	C orb-connect-stream 
;;;;	C  make-socket-pair
;;;;	C    new-socket local : opens socket waits for connect.
;;;;	C    new-socket remote : connects to S
;;;;
;;;;	S  accept-listen
;;;;	S  make-socket
;;;;	S  link-open
;;;;	S  bus-link
;;;;
;;;;	C  link-open : opens stream to remote for input/output
;;;;	C  bus-link
;;;;	C  config !connect
;;;;
;;;;	S  new-socket : ie connects to C
;;;;	S  connect-callback : twiddle socket fields
;;;;	S  link-open : setup stream 
;;;;
;;;;	C  accept-callback : twiddle socket fields
;;;;	C  link-open (again ?) : sets second stream for input/ ??first for output??
;;;;	
;;;;	
;;;;	
;;;;	need ability to delay some mod to link until some rsp sent, maybe an assoc list of 
;;;;	  req indices and closures to call.
;;;;	
;;;;	

(defun convert-channel (channel kind)
  (let ((cur-state (state-of-channel channel))
	(cur-kind (kind-of-channel channel)))

    ;; duplex-stream -> more work todo.
    (unless (member cur-state '(primary-stream))
      (message-emit (warn-message (append '(convert-channel state) (list cur-state cur-kind kind)))))

    ;; let failures rise, and caller should kill link ?
    (case kind
      (mathbus
       (let ((nch (new-mathbus-channel (primary-socket-of-channel channel)
				       #'(lambda (aterm stream)
					   (let ((term (if *link-term-hook* (funcall *link-term-hook* aterm) aterm)))
					     (write-node (term-to-mbterm term) stream)))
				       #'(lambda (stream) (mbterm-to-term (read-node stream))))))
	 (mathbus-channel-open  nch)
	 nch))

      (ascii-uncompressed
       (let ((nch (new-prl-stream-channel (primary-socket-of-channel channel))))
	 (uncompressed-stream-channel-open nch)
	 nch))

      ;; expect originator to send-levels or request send-levels. cprl-stream-send-levels
      (ascii-compressed
       (let ((nch (new-cprl-stream-channel (primary-socket-of-channel channel) (get-levels))))
	 (stream-channel-open nch)
	 nch))

      (otherwise 
       (raise-error (error-message (append '(convert-channel kind) (list cur-state cur-kind kind)))))
  )))


(defun convert-link-encoding (link kind)
  (make-link :channels (mapcar #'(lambda (ch)
				   (convert-channel ch kind))
			       (channels-of-link link))))



(defun update-process-activity-log (&optional note iutime)
  (labels ((host ()
	     (intern-system 
	      (let ((hosts (local-host)))
		(if (> (length hosts) (length ".cs.cornell.edu"))
		    (if (subseq hosts (- (length hosts) (length ".cs.cornell.edu")) (length hosts))
			(subseq hosts 0 (- (length hosts) (length ".cs.cornell.edu")))
			hosts)
		    hosts)))))
	       
    (let ((utime (or iutime (get-universal-time))))
      (handle-file-error
       (with-open-file (stream (complete-system-path (list "library") "active" "log")
			       :direction :output
			       :if-exists :append
			       :if-does-not-exist :create)
	 
	 (write  (list (host)
		       (unix-process-id)
		       *component-kind*
		       (datetime-string utime)
		       (process-id)
		       note
		       utime
		       (or *disksave-filename* "")
		       )
		 :stream stream :readably t)
	 (terpri stream)
	 )))))

(defun host-of-process-activity-log-entry (e) (nth 0 e))
(defun pid-of-process-activity-log-entry (e) (nth 1 e))
(defun kind-of-process-activity-log-entry (e) (nth 2 e))
(defun datetime-of-process-activity-log-entry (e) (nth 3 e))
(defun process-id-of-process-activity-log-entry (e) (nth 4 e))
(defun note-of-process-activity-log-entry (e) (nth 5 e))
(defun utime-of-process-activity-log-entry (e) (nth 6 e))
(defun disksave-of-process-activity-log-entry (e)
  (let ((d (nth 7 e)))
    (unless (string= d "") d)))

(defun read-process-activity-log ()
  (handle-file-error
   (with-open-file (stream (complete-system-path (list "library") "active" "log")
			   :direction :input
			   )
     (let ((acc nil))
       (do ((i (read stream nil nil) (read stream nil nil)))
	   ((null i))
	 (push i acc))

       (nreverse acc)))))

(defun filter-pal-host (host pal)
  (filter #'(lambda (e) (eql host (host-of-process-activity-log-entry e))) pal))

(defun filter-pal-disksave (dsave pal)
  (let ((searchf (string-pattern-search
		  #'disksave-of-process-activity-log-entry
		  dsave)))
    (filter #'(lambda (e) 
		(funcall searchf e))
	    pal)))

(defun print-pal (l)
  (dolist (e l)
    (format t "~12,a ~6,a ~5,a  ~20,a ~22,a ~12,a ~a~%"
	    (host-of-process-activity-log-entry e)
	    (pid-of-process-activity-log-entry e)
	    (kind-of-process-activity-log-entry e)
	    (datetime-of-process-activity-log-entry e)
	    (process-id-of-process-activity-log-entry e)
	    (note-of-process-activity-log-entry e)
	    (disksave-of-process-activity-log-entry e)
	    )))

(defun show-process-activity-log ()
  (filter #'(lambda (e)
	      (not (or (string= "exit" (note-of-process-activity-log-entry e))
		       (string= "stop" (note-of-process-activity-log-entry e)))))
   (remove-duplicates
    (sort (read-process-activity-log) #'< :key #'utime-of-process-activity-log-entry)
    :key  #'pid-of-process-activity-log-entry)
  ))

;; (print-pal (show-process-activity-log));;
;; (print-pal (filter-pal-host '|baldwin| (show-process-activity-log)));;
;; (print-pal (filter-pal-disksave "EDD-disksave-74" (show-process-activity-log)))
