
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      xnode-leader-exists-p get-evcb 
	      )))


(defstruct variable-selection
  (id nil)
  (term-index -1)
  (binding-index -1))


;;;  ----------------------------------------------------------------------
;;;  ---------------------------  term expansion  -------------------------
;;;  ----------------------------------------------------------------------

; original-id is the original-id
; selected-id is the selected-id if selection, otherwise it is the original id.
; the preferred is is the original id if id from G and is selected-id if id from F.
; conflict-id is a preferred-id with @i's removed from the end (trimmed).
;
;G-> original -> preferred-id --trim--> conflict-id
;F-> original -> --select--> selected-id -> preferred-id --trim--> conflict-id

; efficiency hacks    : if parm list has no variables use old rather than making new.

;;; 

;;;;  ------------------------------------------------------------------------
;;;;  -------------------- Abstraction Expansion -----------------------------
;;;;  todo:
;  -  fix-up notation
;  -  develop standard notation for relating functional description to code 
;       in invariants.
;  
;;;;  -doc-s-------------------------------------------------------------------
;;;;  -level-04-
;;;;
;;;;  ---------- Functional Description of Abstraction Expansion -------------
;;;;
;;;;
;;;;
;;;; Notation :
;;;;  small letters are variables
;;;;    x, y, z range over ids
;;;;    i j k l range over an alphabet
;;;;    a b c are words in that alphabet
;;;;    r s t denote terms
;;;;    n m o range over compounds ie (x i j)  
;;;;  capitals are sets (in most cases)
;;;;    O denotes a term operator
;;;;    R denotes a list of terms
;;;;    X Y Z are lists of id variables, and S is set of variable ids.
;;;;    Q Q' denote lists of type ids
;;;;    P P' V' denote lists of parameters
;;;;    N M W U V are sets of compounds
;;;;    A B C D are sets of words
;;;;    I J K L are sets of letter pairs
;;;;  - indicates member of set ie x-i is ith member of X.
;;;;  <i:n> indicates a list of x's of length n indexed with i, ie X-i is ith 
;;;;        member of <i:n>X
;;;;      i:n is not indicated when obvious from context.
;;;;  (...) denotes a list
;;;;  N#M denotes Union of Sets N M
;;;;  N#x denotes Union of Sets N {x}
;;;;  N-n denotes Difference of sets N {n}.
;;;;  x+X denotes concatenation of x onto front of list X
;;;;  X+x denotes concatenation of X onto back of list X
;;;;  1st(X) denotes first element of list X
;;;;  Rest(X) denotes all but first element of list x.
;;;;  juxtaposition denotes contatenation of a letter onto a word.
;;;;  << - a << b : a is a proper prefix of b
;;;;  - < Precondition for succeeding clause.
;;;;
;;;; (x a I A) : is a compound which will be referred to as a node.
;;;;  x is an id. a is the root binding address; ie the address of the
;;;;  term in which the binding occurs. I is a set of local binding addresses.
;;;;  A local binding address consists of a pair (i j) where i is index into the
;;;;  bound terms of the term at a and j is the index into the binding list of
;;;;  said bound term. A is a set (at times partial) of addresses of occurences
;;;;  of variable as a term.
;;;;
;;;;  The form of a term is  t : ( O (Q,P) <i:n>(X.s) )
;;;;   the x-ij in X-i will be called binding occurences.
;;;;   when s-i is a variable id x-ij, s-i will be called a term occurence 
;;;;    of x-ij.
;;;; 
;;;; Given
;;;;   Z		: second-order variables
;;;;   V'		: parameter variables
;;;;   <>(ys is js)	: selections
;;;;   T		: matrix (2nd order term)
;;;;
;;;;   and t = (O (Q',P') <i:n>(X.t))
;;;;  Let (t', N') be F(epsilon {} {} T)
;;;;  Let t'' be R(PMax t' N')
;;;;
;;;;
;;;; Expansion irrespective of capture and selected ids.
;;;;
;;;;  Invariant :  when F u<*> y,  y must either be in u<*> or in z<i:n>
;;;;  1. F U y = y when y in U
;;;;  2. F U { O (Q P) <>(Y.t) }
;;;;		= {O (Q P[p-k/v-k]) <>(Y.F U#Y t) }
;;;;  3. F U z-i.R = G U (X-i R) () t-i
;;;;
;;;;  4. G U (X R) V y = y when y in V or y not in X
;;;;  5. G U (X R) V {O (Q P) <>(Y.t) }
;;;;		= {O (Q P) <>(Y.G U (X R) V#Y t) }
;;;;  6. G U (X R) V x-i = F U r-i
;;;;
;;;;  -page------------------------------------------------------------------
;;;;
;;;;
;;;;  Utility Functions are
;;;;
;;;;  -  (i j) < (k l) : iff i < k or (i = k and j < l)
;;;;
;;;;  -  Term X.t 	-> t
;;;;  -  Ids  X.t	-> X
;;;;
;;;;  -	BreakI <>(X.t) 	-> { (X i) : (X.t)-i in <>(X.t) }
;;;;  -	BreakJ X i 	-> { (x i j) : x-j in X }
;;;;  -	Break <>(X.t) 	-> { (x i j) : ((x i j) in BreakJ X i) 
;;;;				       and ((X i) in BreakI <>(X,t)) }
;;;;	; creates list of ids and local addresses of binding occurences 
;;;;     ;  from list of bound terms.
;;;;
;;;;  -	Collect a W	-> { (x a I {} ) : ThereExists (x i j) in W
;;;;					   and I is { (i j) : (x i j) in W } }
;;;;	; Collects local address of binding occurences of each id.
;;;;
;;;;  -  Map a X N	-> { (y a) : y in X and (y a Ab At) in N }
;;;;	; Maps id of X to nodes in N with root binding address a.
;;;;
;;;;  -  Select y 	-> (x : (y k l) in <>(ys is js)
;;;;			    and (x k l) in Break <>(X.t))
;;;;			   or (y : y not in <>ys)
;;;;	; Finds id selected if id is selected otherwise finds self.
;;;;
;;;;  -  Sel S	-> { x : y in S and x = Select y }
;;;;	; Does select on a set.
;;;;
;;;;  -  MapSel a Y N    -> Map a Sel(Y) N
;;;;	; selects then maps a set.
;;;;
;;;;  -  Pick x S	-> (x a) in S : not (ThereExists (x b) in S : b > a)
;;;;	; Finds node n in S which has least address of nodes in S with id x.
;;;;
;;;;  -  PairMin I	-> (i j) in I : not (ThereExists (k l) in I
;;;;					     : (k l) < (i j))
;;;;  -  Min W		-> (x a I A) in W
;;;;			    : not (ThereExists (y b J B) in W 
;;;;				   : b < a or (a = b and PairMin J < PairMin I))
;;;;	; finds node in W whose address is Minimum.
;;;;
;;;;  -  NSub N y (x a)	-> { (y a I A) : (x a I A) in N and
;;;;			     (z b J B) : (z b J B) in N and (z b) not (x a) }
;;;;	; changes id of a node in N with id x and address a to y.
;;;;	<> Id and root address uniquely identify a node.
;;;;
;;;;  -  NUpdate y W a N	-> { (x b I A) : (x b I A) in N
;;;;					 and x b is not Pick(y W)
;;;;			     and (y c J B) : (y c j C) in N and y c is Pick(y W)
;;;;					     and B is a#C}
;;;;	; adds term occurence address a to node in N picked from W by id y.
;;;;
;;;;
;;;;  -page------------------------------------------------------------------
;;;;
;;;;
;;;;
;;;;  F is
;;;;   
;;;;  - Given a U O Q P
;;;;      Let H,K be
;;;;	   H i N <>(Y.s) <>(X.t)
;;;;		-> K i+1 
;;;;		     <>(Y.s) 
;;;;		     F (ai N U#MapSel(a Ids(1st(<>(X.t))) N) Term(1st(<>(X.t))))
;;;;		     Rest(<>(X.t))
;;;;	    K i <>(Y.s) (N X.t) <>(Z.r)
;;;;		-> H i N <>(Y.s)+X.t <>(Z.r)
;;;;	    K i <>(Y.s) (N X.t) ()
;;;;		-> ((O (Q P[p'-k/v'-k]) <>(Y.s)+X.t) N)
;;;;
;;;;  -	F a N U (O (Q P) <>(Y.s))
;;;;	 -> H 1 N#Collect(a Break(<>(Sel(Y).s))) () <>(Y.s)
;;;;
;;;;  - < ThereExists (z a) in U : z = Select y
;;;;	F a N U y -> (Select(y) NUpdate(Select(y) U a N))
;;;;
;;;;  - F a N U (z-i ({} {}) <>(().r) -> G a N U (X-i <>(Term ().r) {} t-i
;;;;	; z-i is operator and second order variable.
;;;;
;;;;
;;;; G is
;;;;
;;;;  - Given a U V O Q P X R
;;;;      Let H,K be
;;;;	   H i N <>(Y.s) <>(Z.r)
;;;;	      -> K i+1 
;;;;		   <>(Y.s) 
;;;;		   G ai N U (X R) V#Map(a Ids(1st(<>Z.r)) N) Term(1st(<>Z.r))
;;;;		   Rest(<>(Z.r))
;;;;	    K i <>(Y.s) (N X.t) <>(Z.r)
;;;;		-> H i N <>(Y.s)+X.t <>(Z.r)
;;;;	    K i <>(Y.s) (N X.t) ()
;;;;		-> ((O (Q P) (<>(Y.s) X.t))  N)
;;;;
;;;;  - G a N U (X R) V (O (Q P) <>(Y.s))
;;;;	 -> H 1 N#Collect(a Break(<>(Y.s))) () <>(Y.s)
;;;;
;;;;  -	< ThereExists (y a) in V
;;;;	G a N U (X R) V y = (y NUpdate(y V a N))
;;;;
;;;;  - < not ThereExists (y a) in V and y not in X
;;;;	G a N U (X R) V y = (y { (x b I A) : (x b I A) in N and x not y, and
;;;;				 (y epsilon {} A') 
;;;;				   : (ThereExists (y epsilon {} A) in N
;;;;				      and A' is a#A)
;;;;				     or (Not ThereExists (y epsilon {} A) in N
;;;;					 and A` is {}) }
;;;;
;;;;
;;;;  - < y is x-j in X
;;;;	G a N U (X R) V y = F a N U r-j
;;;;
;;;;  -page------------------------------------------------------------------
;;;;
;;;;
;;;;  { 1 ... PMax } are the priorities 
;;;;
;;;;  R is
;;;;
;;;;  - < p is 0
;;;;    R N p t -> D N { (x a I A) : (x a I A) in N and A is {} } t
;;;;
;;;;  - < p>0 
;;;;    R N p t -> PB N p {x : x in N and Priority(x) = p} t
;;;;
;;;;
;;;;  and PB is
;;;;				
;;;;  -  Occurs-in N (x a I A) (y b J B) 
;;;;		-> a << b and ThereExists a' in A : b << a'
;;;;  -  Conflicts N (x a I A)
;;;;		-> { y : (y b J B) in N and B not {}
;;;;			 and ((Priority (y b J B) > Priority (x a I A)
;;;;			       and (a = b
;;;;				    or Occurs-in (y b J B) (x a I A)
;;;;				    or Occurs-in (x a I A) (y b J B) ))
;;;;			      or 
;;;;			      (Priority (y b J B) = Priority (x a I A)
;;;;			       and ((a = b and PairMin J < PairMin I)
;;;;				    or (a not = b
;;;;					and Occurs-in (x a I A) (y b J B) )))) }
;;;;
;;;;  -  PB N p {} t -> R N p-1 t
;;;;
;;;;  -  PB N p M t -> H N p Min(M) M-Min(M) t
;;;;
;;;;  -  H N p (x a I {}) M t -> PB N p M t
;;;;     H N p (x a I A) M t -> K N p (x a I A) Conflicts(N (x a I A)) M t
;;;;
;;;;  -  K N p x {} M t -> PB N p M t
;;;;     K N p x S M t -> Sub N p M (y:y not in S) x
;;;;
;;;;  -  Sub N p M y (x a I A) -> PB NSub(N y (x a)) p M t[y/x @(a I A)]
;;;;				
;;;;	
;;;;  and D is
;;;;
;;;;  -  DConflicts (x a I A) N M 
;;;;		-> { y : (y b J B) in N 
;;;;			 and (( (y b J B) in M and a = b )
;;;;			      or Occurs-in (y b J B) (x a I A) ) }
;;;;
;;;;  -  D N {} t	-> t
;;;;     D N M t 	-> H N Min(M) M-Min(M) t
;;;;  
;;;;  -  H N x M t	-> K N x DConflicts(x N M) M t
;;;;
;;;;  -  K N x {} M t	-> D N M t
;;;;  -  K N x S M t	-> DSub N y:(y not in S) M x
;;;;  -  DSub N M y (x a I A) -> D NSub(N y (x a)) M t[y/x @(a I)]
;;;;
;;;;
;;;;  -page------------------------------------------------------------------
;;;;
;;;;
;;;;  at end
;;;;	<> Not ThereExists (x a I A) (y b J B) in N
;;;;		: x=y 
;;;;		  and a < b and ThereExists b' in B : b' < a
;;;;		  or a=b 



;;;;
;;;;  -page------------------------------------------------------------------
;;;;
;;;;  ----------------------  Implementation  --------------------------------
;;;;
;;;;  An implementation of the functional description as described would
;;;;  be inefficient. Thus the implementation differs from the description.
;;;;  
;;;;  - Sets U and V are implemented as stacks.
;;;;    These stacks are distributed according to id. E.g., rather than have
;;;;    a single large stack to implement V, there is a smaller stack for 
;;;;    each id used as a binding in G.
;;;;
;;;;  - For Each id, V is implemented as a single stack for all G invocations.
;;;;    Each node has an id used to check if node is from current G invocation.
;;;;
;;;;  - N is partitioned by priority and stacked by relative addresses within
;;;;	priority.
;;;;
;;;;  - The conflicts are detected during execution of F and G. Each node
;;;;    has a set of the detected higher priority conflicts.
;;;;    
;;;;  - Node Scope stack : during execution the node-scope-stack contains
;;;;    the nodes whose scopes the execution is currently in. When variable
;;;;    occurences are encountered the node-scope-stack is used to determine
;;;;    conflicts. Each node can have an associated leader on the stack which
;;;;    is used to know which conflicts have already been detected in order to
;;;;	avoid a node being placed in the conflict set of a node multiple times.
;;;;
;;;;  - Similar-Ids : Let x be an id. stripped(x) is x with any @i, where i is 
;;;;    and integer, trimmed from the right side of x. When renaming an id the
;;;;    new name derived from the old name by adding an @i to the right side
;;;;    of the stripped id. Thus, only ids which are the same after being
;;;;    stripped, may conflict with each other.  Such ids will be called
;;;;	similar.
;;;;
;;;;  - Like U and V the node-scope stack is distributed among the ids.
;;;;    As the node-scope-stack is used to detect conflicts all nodes which
;;;;    have the potential for conflict must be in the same stack. These nodes
;;;;    are those whose ids are similar.
;;;;
;;;;  - Conflicts among nodes whose root address are the same are detected
;;;;    when the nodes are collected. Thus their relative position with respect
;;;;	to each other on the node-scope-stack is irrelevant.
;;;;
;;;;  - The substitution of renamed variables is handled by building a result
;;;;    term with a unique term structure for each node so that by destructively
;;;;    modifiying the id of said term structure the renaming is effected.
;;;;
;;;;  - The term occurence addresses are not maintained as they are not needed.
;;;;	


;;;;  -doc-e-------------------------------------------------------------------
;;;;  -page------------------------------------------------------------------


;;;;  ---------------------------- Node --------------------------------------
;<> n-x in conflicts(o-x) -> not o-x in conflicts(n-x)

;; eXpansion NODE.
(defstruct (xnode (:include dlink)
		  (:print-function print-xnode))
  (leader nil)
  ;; to detect dummy.
  (exists-occurence nil)
  ;; prevent duplicate entries on node scope stack.
  (scope-entered nil)

  ;; list of root address and (i j) local address pairs.
  (addresses nil)		

  ;; unique occurence term for node.
  (term nil)

  ;; arity of associated variable.
  (arity 0)
  (priority nil)
  ;; a unique id of call of G in which root-addr occurred.
  (G-id nil)

  ;; selection if selectable and selected, otherwise original.
  (preferred-id nil)
  ;; the stripped preferred id.
  (conflict-id nil)

  ;; higher priority nodes with same conflict id.
  (conflicts nil))

(defstruct (leader (:include dlink))
  (xnode nil))

(defun xnode-leader-exists-p (xnode)
  (eq (xnode-leader xnode) xnode))

(defmacro xnode-local-binding-addresses (node)
  `(cdr (xnode-addresses ,node)))

(defmacro xnode-root-binding-address (node)
  `(car (xnode-addresses ,node)))

(defun print-xnode (node stream depth)
  (format stream 
	  "xnode: ~a~%preferred-id : ~a~%conflict-id : ~a~%priority :~a~%conflicts :"
	  depth
	  (xnode-preferred-id node)
	  (xnode-conflict-id node)
	  (xnode-priority node))
  (mapc #'(lambda (b)
	    (print-xnode b stream (1+ depth)))
	(xnode-conflicts node)))



;;;;  ------------------------------- EVCB -----------------------------------
;;;;
;;;;  The evcb is used to contain data related to individual ids such as U, V,
;;;;  and the node-scope-stack.
;;;;

(defstruct evcb		; Expansion Variable Control Block

  ;;; as original id 

  ;; selectedp and in matrix -> selectable.
  (selectedp nil)
  (selected-id nil)
  (stripped-id nil)
  (F-pick-stack nil)				; U.
  (G-pick-stack nil)				; V's.
  ;; node for id as free-var. Initialized at first occurence.	
  (free-xnode nil)

  ;;; as conflict-id
  ;;; built-in deBruijn stack binding id to node map.

  ;; used locally to find intra-term conflicts.
  (intra-term-conflict-stack nil)
  ;; node-scope-stack - implemented as doubly 
  ;; linked queue (dlq) to ease insertion and removal.
  (node-scope-dlq (make-dlist)))


(defun get-evcb (id)
  (or (evcb-of-variable-id id)
      (set-evcb-of-variable-id id 
			       (make-evcb :stripped-id (strip-variable-id id)
					  :selected-id id))))


;;;;  -page------------------------------------------------------------------


;;;;  ------------------------ Node Scope ------------------------------------
;;;;	<> ForEach n,o in node-scope-dlq : 
;;;;        if position(n) < position(o)
;;;;		(root-binding-address(o) =<< root-binding-address(n))
;;;;	<> position(leader(o-x)) < position(n-x)
;;;;       and position(n-x) < position(o-x)
;;;;		-> n-x in conflicts(o-x) 
;;;;		   or o-x in conflicts(n-x)

(defmacro add-free-node-scope (free-node)
  `(dadd ,free-node (node-scope-dlq-of-node ,free-node)))

(defmacro next-inner-node-scope (current-node)
  `(dlink-prev ,current-node))

(defmacro more-inner-node-scopes-p (current-node node-scope-dlq)
  `(eql ,current-node ,node-scope-dlq))

;<> ForEach n in node-scope-dlq : 
;		root-binding-address(n) =<< root-binding-address(xnode)
(defmacro enter-node-scope (xnode) 
  `(progn (setf (xnode-scope-entered ,xnode) t)
	  (dlist-insert ,xnode
		 (evcb-node-scope-dlq (get-evcb (xnode-conflict-id ,xnode))))))

(defmacro exit-node-scope (xnode)
  `(progn (setf (xnode-scope-entered ,xnode) nil)
	  (when (xnode-leader-exists-p ,xnode)
	    (dpop (xnode-leader ,xnode)))
	  (dpop ,xnode)))

(defmacro node-scope-dlq-of-node (xnode)
  `(evcb-node-scope-dlq (get-evcb (xnode-conflict-id ,xnode))))

; moves leader of node to top of node-scope-stack.
(defun pullup-node-scope-leader (node node-scope-dlq)
  ;; don't bother if already on top.
  (unless (eq node (if (leader-p (next-of-dlink node-scope-dlq))
		       (leader-xnode (next-of-dlink node-scope-dlq))
		       (next-of-dlink node-scope-dlq)))
    (if (not (xnode-leader-exists-p node))
	(dlist-insert (dpop (xnode-leader node))
	       node-scope-dlq)
	(dlist-insert (setf (xnode-leader node) 
		     (make-leader :xnode node))
	       node-scope-dlq))))



;;;;  -page------------------------------------------------------------------
;;;;
;;;;  --------------------- Code --------------------------------------------
;;;;
	     
(defun second-order-substitution-with-maintenance (term
						   parameter-subs
						   subs
						   &optional selections)
  ;;(setf -term term -psubs parameter-subs -subs subs)
  ;;(format t "~%sos ~a~%" (id-of-term term)) ;; (break "sos")
  (subst-count-incf)  
  (incf *no-subst2-m*)
  
  ;;(format t "second-order wout/sharing~%")
  (with-backtrace "SecondOrderSubstitutionWithMaintenance"
	     
  ;;;;  ------------------ Global Variables of Expansion ---------------------
    (let* ((MaxP (length '(free first selected unselected)))
	   (any-conflict nil)
	   (priority-buckets (make-array MaxP :initial-element nil)))
     
      (labels
      ;;;; --------------- utility functions ---------------------------------
	  ((construct-xnode (&rest args &key &allow-other-keys)
	     (let ((xnode (apply #'make-xnode args)))
	       (setf (xnode-leader xnode) xnode)
	       ;;<> For-all n,o in PriorityBucket i, 
	       ;;	if position(n) < position(o) 
	       ;;	then root-binding-address of n >= root-binding-address-of o
	       (unless (or (null (xnode-priority xnode))
			   (free-var-p xnode)) ; don't use them (at the moment).
		 (push xnode (aref priority-buckets (xnode-priority xnode))))
	       xnode))

	   ;; mappings from original-id to other ids.
	   (stripped-id (original-id)
	     (evcb-stripped-id (get-evcb original-id)))

	   (selected-id-p (original-id)
	     (evcb-selectedp (get-evcb original-id)))

	   (selected-id (original-id)
	     (evcb-selected-id (get-evcb original-id)))

	   (preferred-id (original-id context)
	     (if (eq context 'F)
		 (selected-id original-id)
		 original-id))

	   (conflict-id (original-id context)
	     (stripped-id (preferred-id original-id context)))

	   (determine-priority (context)
	     (case context
	       (sub 		nil)
	       (free		0)
	       (G 			1)
	       (F-selected		2)
	       (F			3)))
	   ;;(otherwise		
	   ;;(system-error (error-message '(second-order-substitute))))

	   (determine-priority-from-id (original-id context)
	     (if (eq 'F context)
		 (if (selected-id-p original-id)
		     (determine-priority 'F-selected)
		     (determine-priority 'F))
		 (determine-priority 'G)))

	   (free-var-p (xnode)
	     (eql 0 (xnode-priority xnode)))


       ;;;;  -page-----------------------------------------------------------
       ;;;;
       ;;;;  ----------------- identify conflicts ---------------------------
       ;;;;

	   (update-scope-conflicts-of-node (xnode)
	     (labels 
		 ((add-conflict (xnode conflict)
		    (setf any-conflict t)
		    (unless (eq (xnode-root-binding-address xnode)
				(xnode-root-binding-address conflict))
		      ;; push higher priority (lower priority number)
		      ;; onto other conflict list.
		      (if (< (xnode-priority conflict) (xnode-priority xnode))
			  ;;<> priority(xnode) > priority(conflict)
			  (push conflict (xnode-conflicts xnode))
			  ;;<> root-binding-address(conflict)
			  ;;   << root-binding-address(xnode)
			  (push xnode (xnode-conflicts conflict))))))

	       (let ((node-scope-dlq (node-scope-dlq-of-node xnode)))
		 (do ((conflict (next-inner-node-scope (xnode-leader xnode)) 
				(next-inner-node-scope conflict)))
		     ((more-inner-node-scopes-p conflict node-scope-dlq))
		   (when (xnode-p conflict)
		     (add-conflict xnode conflict)))
		 (pullup-node-scope-leader xnode node-scope-dlq))))


       ;;;;  ----------- maintain map of occurence ids to nodes -------------
       ;;;;  ----------- and maintain node scope stack ----------------------
       
	   ;;<> For pick-stack p, ForEach nodes x,y in p :
	   ;;	if position(x) < position(y) wrt p and x ^= y,
	   ;;	then node-root-address(y) << node-root-address(x)
       
	   (update-pick-map (original-id context xnode)
	     (cond ((eq context 'G)
		    (push xnode (evcb-G-pick-stack (get-evcb original-id))))
		   ((eq context 'F)
		    (push xnode (evcb-F-pick-stack (get-evcb original-id))))
		   (t (error "invalid context in pop-xnode during expand"))))

	   (restore-pick-map (original-id context)
	     (cond ((eq context 'G) (pop (evcb-G-pick-stack (get-evcb original-id))))
		   ((eq context 'F) (pop (evcb-F-pick-stack (get-evcb original-id))))
		   (t (error "invalid context in pop-xnode during expand"))))

	   (update-pick-map-and-node-scope-queue (ids context)
	     (mapc #'(lambda (id)
		       (let ((binding-id (value-of-parameter-value id)))
			 (when (variable-id-p binding-id)
			   (let ((xnode (id-to-node binding-id)))
			     (update-pick-map binding-id context xnode)
			     ;; must not let same node-scope be entered more than once.
			     (unless (xnode-scope-entered xnode)
			       (enter-node-scope xnode))))))
		   ids))

	   (restore-pick-map-and-node-scope-queue (ids context)
	     (mapc #'(lambda (id)
		       (let ((binding-id (value-of-parameter-value id)))
			 (when (variable-id-p binding-id)
			   (let ((xnode (id-to-node binding-id)))
			     ;; must not let same node-scope be exited more than once.
			     (when (xnode-scope-entered xnode)
			       (exit-node-scope xnode))
			     (restore-pick-map binding-id context)))))
		   ids))

	   (pick (original-id context &optional (arity 0) G-id)
	     (cond ((eq context 'G)
		    (let ((xnode
			   (do ((stack (evcb-G-pick-stack (get-evcb original-id))
				       (cdr stack)))
			       ((or (null stack)
				    (= arity (xnode-arity (car stack))))
				(car stack)))))
		      (when (and xnode
				 (eql G-id (xnode-G-id xnode)))
			xnode)))

		   ((eq context 'F)
		    (do ((stack (evcb-F-pick-stack (get-evcb original-id))
				(cdr stack)))
			((or (null stack)
			     (= arity (xnode-arity (car stack))))
			 (car stack))))
				 
		   (t (error "invalid context in pop-xnode during expand"))))

	   (get-free-xnode (id)
	     (let ((evcb (get-evcb id)))
	       (cond ((evcb-free-xnode evcb))
		     (t (let ((free-xnode (construct-xnode
					   :term (unless (dummy-variable-id-p id)
						   (modifiable-variable-term id))
					   :exists-occurence t
					   :priority (determine-priority 'free)
					   :preferred-id id
					   :conflict-id (conflict-id id 'free))))
			  (add-free-node-scope free-xnode)
			  (setf (evcb-free-xnode evcb) free-xnode))))))

	   (f-substitution-p (id arity)
	     (let ((xnode (pick id 'F arity)))
	       (and xnode (null (xnode-priority xnode)))))

       ;;;;  -page-----------------------------------------------------------

       ;;;;  ------- maintain map of binding ids to nodes -------------------
       ;;;;  ------- and detect intra-term conflicts ------------------------

	   ;;<> ForAll n : if root-address(n) << a then n in id-to-node-stack
	   ;; where a is as in functional description.
	   ;;<> ForAll n,o in id-to-node-stack : 
	   ;;	if root-address(n) << root-address(o)
	   ;;          then position(o) < position(n)

	   (id-to-node (original-id) (value-of-variable-id original-id))

	   (term-of-xnode (xnode)
	     (or (xnode-term xnode)
		 (raise-error (error-message '(second-order-substitute dummy)))))

       ;;; creates xnodes and updates id to node map.
       ;;; Also, finds conflicts among term xnodes.
	   (update-id-to-node-map (term root-addr context &optional G-id)
	     ;; requires two passes as priority of xnode is not certain until all
	     ;; xnodes of term have been examined and we must know relative 
	     ;; priorities to place conflicts correctly.

	     ;; new-pass to merge by selected ids.
	     (let ((select-conflicts nil)
		   (preferred-xnodes nil))
	       (when (eql 'F context)
		 (dolist (bound-term (bound-terms-of-term term))
		   (maplist #'(lambda (bindings)
				(when (and (selected-id-p (caar bindings))
					   (cdr bindings))
				  (let ((found nil)
					(v (cdar bindings))
					(o (caar bindings)))
				    ;;(format t "original ~a preferred ~a others ~a ~%"
				    ;;o v (cdr bindings))
				    (dolist (binding (cdr bindings))
				      (when (and (not (eql o (car binding)))
						 (selected-id-p (car binding))
						 (eql v (cdr binding)))
					(push (car binding) found)))
				    (when found
				      (setf select-conflicts
					    (nconc select-conflicts
						   (list (caar bindings))
						   found))))))
			    (mapcar #'(lambda (binding)
					(let ((binding (value-of-parameter-value binding)))
					  (when (variable-id-p binding)
					    (cons binding (preferred-id binding context)))))
				    (bindings-of-bound-term-n bound-term))))
		 (setf select-conflicts (delete-duplicates select-conflicts))
		 ;;(format t "select-conflicts ~a~%" select-conflicts)
		 )

	       ;; original-id in select-conflicts -> original-id xnode
	       ;; selected -> selected xnode

	       ;; first-pass : push xnodes of preferred-ids of term xnodes.
	       (with-variable-minor-invocation
		   (let ((bound-term-index 0)
			 (binding-id-index 0))
		     (mapc 
		      #'(lambda (bound-term)
			  (mapc
			   #'(lambda (id)
			       (let* ((original-id (value-of-parameter-value id)))
				 (when (variable-id-p original-id)
				   (let ((preferred-id (preferred-id original-id context)))
				     (if (set-variable-minor-use original-id)
					 (let ((xnode (id-to-node original-id)))
					   (push (cons bound-term-index binding-id-index) 
						 (xnode-local-binding-addresses xnode))
					   (setf (xnode-priority xnode)
						 (max (xnode-priority xnode)
						      (determine-priority-from-id original-id
										  context))))
					 (let* ((mergeable-preferred-p
						 (and (eql 'F context)
						      (selected-id-p original-id)
						      (not (member original-id select-conflicts))))
						(preferred-xnode
						 (when mergeable-preferred-p
						   (find-first #'(lambda (xnode)
								   (when (eql preferred-id
									      (xnode-preferred-id xnode))
								     xnode))
							       preferred-xnodes))))
					   (if preferred-xnode
					       (progn
						 (push (cons bound-term-index binding-id-index) 
						       (xnode-local-binding-addresses preferred-xnode))
						 (enter-binding original-id preferred-xnode))
					       (let ((xnode
						      (construct-xnode
						       :preferred-id preferred-id
						       :conflict-id (conflict-id original-id context)
						       :addresses (cons root-addr
									(list (cons bound-term-index
										    binding-id-index)))
						       :term (unless (dummy-variable-id-p preferred-id)
							       (modifiable-variable-term preferred-id))
						       :priority (determine-priority-from-id original-id
											     context)
						       :G-id G-id)))
						 (enter-binding  original-id xnode)
						 (when mergeable-preferred-p
						   (push xnode preferred-xnodes))
						 ;; lazy intra-term-conflict-stack purge
						 (setf (evcb-intra-term-conflict-stack 
							(get-evcb (conflict-id original-id
									       context)))
						       nil))) )))))
			       (incf binding-id-index))
			   (bindings-of-bound-term-n bound-term))
			  (incf bound-term-index)
			  (setf binding-id-index 0))
		      (bound-terms-of-term term))))

	       ;;(format t "preferred-xnodes ~a~%" (length preferred-xnodes))

	       ;; second-pass :
	       ;;   find-conflicts among term xnodes and store local addresses.
	       ;;<> ForEach n-x,o-x in intra-term-conflict-stack
	       ;;   root-address(n-x) = root-address(o-x)
	       ;;<> ForEach n-x,o-x in intra-term-conflict-stack
	       ;;	 position(n-x) < position(o-x) iff PairMin(n) < PairMin(o)
	       (with-variable-minor-invocation
		   (mapc
		    #'(lambda (bound-term)
			(mapc
			 #'(lambda (id)
			     (let* ((original-id (value-of-parameter-value id)))
			       (when (and (variable-id-p original-id)
					  (not (set-variable-minor-use original-id)))
				 (let* (;;(preferred-id (preferred-id original-id context))
					(xnode (id-to-node original-id))
					(conflict-evcb
					 (get-evcb (xnode-conflict-id xnode))))
				   ;; add conflicts among term xnodes.
				   (unless (member xnode (evcb-intra-term-conflict-stack conflict-evcb))
				     (mapc #'(lambda (earlier-xnode)
					       ;; prevent multiple selected conflits.
					       ;;(unless (and (selected-id-p original-id)
					       ;;	      (eql 2 (xnode-priority earlier-xnode))
					       ;;(eql preferred-id (xnode-preferred-id earlier-xnode))
					       ;;(not (exists-p #'(lambda (eaddr)
					       ;;(exists-p #'(lambda (addr)
					       ;;(= (car addr) (car eaddr)))
					       ;;(xnode-local-binding-addresses xnode)))
					       ;;(xnode-local-binding-addresses earlier-xnode)))
					       ;;))
					       (setf any-conflict t)
					       (if (< (xnode-priority xnode) 
						      (xnode-priority earlier-xnode))
						   (push xnode 
							 (xnode-conflicts earlier-xnode))
						   (push earlier-xnode 
							 (xnode-conflicts xnode))))
					   (evcb-intra-term-conflict-stack conflict-evcb))
				     (push xnode
					   (evcb-intra-term-conflict-stack conflict-evcb)))))))
			 (bindings-of-bound-term-n bound-term)))
		    (bound-terms-of-term term))) ))

	   ;; intra-term conflict : when two bindings of the binding lists of a
	   ;; subterm have differing original ids but the same conflict id, then 
	   ;; there is an intra-term conflict.

	   ;; there is a single xnode for each original id.

	   ;; all bindings with the same original id have the same preferred id.
	   ;; and then by extension the conflict-ids will also be the same.

	   ;; iterate through the bindings
	   ;;  - at first visit of an original-id place its xnode on
	   ;;    intra-term-conflict-stack of its conflict-id. Add conflicts
	   ;;    with others on stack below.
	   ;;  - if seen again, it is safe to ignore it.
	   ;;    any added to stack since first visit will have recorded conflict.
	   ;;    any to be added will find first and record conflict
	   ;; 

       ;;;;  -page-----------------------------------------------------------


	   (restore-id-to-node-map (term context)
	     (with-variable-minor-invocation
		 (mapc
		  #'(lambda (bound-term)
		      (mapc
		       #'(lambda (id)
			   (let ((original-id (value-of-parameter-value id)))
			     (when (and (variable-id-p original-id)
					(not (set-variable-minor-use
					      (preferred-id original-id context))))
			     (exit-binding original-id))))
		       (bindings-of-bound-term-n bound-term)))
		  (bound-terms-of-term term))))
			       

       ;;;;
       ;;;;  -page-----------------------------------------------------------
       ;;;;

       ;;;;  -------------------------     G     -----------------------------

	   (G (Xi-T R)
	     (let ((G-id (gensym)))
	       (labels

		   ((G-term (term)
		      (if (variable-p term)
			  (G-variable-occurence (id-of-variable-term term))
		  
			  (let ((root-address (cons nil nil)))
			    (setf (car root-address)
				  (prog2
				      (update-id-to-node-map term root-address 'G G-id)

				      (instantiate-term (operator-of-term term)
							(mapcar #'G-bound-term
								(bound-terms-of-term term)))

				    (restore-id-to-node-map term 'G))))))


		    (G-bound-term (bound-term)
		      (prog2
			  (update-pick-map-and-node-scope-queue
			   (bindings-of-bound-term-n bound-term) 'G)

			  (instantiate-bound-term
			   (G-term (term-of-bound-term bound-term))
			   ;; renaming replaces bindings.
			   (bindings-of-bound-term-n bound-term))
	     
			(restore-pick-map-and-node-scope-queue 
			 (bindings-of-bound-term-n bound-term) 'G)))
	 

		    (G-variable-occurence (id)
		      ;; when ThereExist (id b) in V
		      ;; let (id b) be pick(id V) and n be (id b I A) in N'
		      ;;   <> a-x in A
		      ;;   <> ForAll o-x in NSQ : root-address(o) << a
		      (let ((xnode (pick id 'G 0 G-id)))
			(if xnode    
			    ;;<> (priority(pick(x G')) = nil)
			    ;; iff not ThereExists (id b) in V
			    (if (null (xnode-priority xnode))
				(F (term-of-xnode xnode))

				;; bound occurence.
				(progn
				  (update-scope-conflicts-of-node xnode)
				  (setf (xnode-exists-occurence xnode) t)
				  (term-of-xnode xnode)))

			    ;; < not ThereExists (x a) in V
			    (progn
			      (setf xnode (get-free-xnode id))
			      (update-scope-conflicts-of-node xnode)
			      (term-of-xnode xnode))))) )
	   

	       (prog2
		   (mapc #'(lambda (xij rj)
			     (update-pick-map xij 'G
					      (construct-xnode 
					       :G-id G-id
					       :term (term-of-bound-term rj))))
			 (bindings-of-bound-term-r Xi-t)
			 R)

		   (G-term (term-of-bound-term Xi-t))

		 (mapc #'(lambda (xij)
			   (restore-pick-map xij 'G))
		       (bindings-of-bound-term-n Xi-t))) )))



       ;;;;  -page-----------------------------------------------------------


       ;;;;  -------------------------     F     -----------------------------

	(F (ri)
	   (labels 
	       ((F-instantiate-operator (operator-template)
		  ;;(implicit-substitute-in-operator operator-template)
		  (substitute-in-operator operator-template parameter-subs)
		  )
		
		(F-term (term)
		  (cond
		    ((and (variable-term-p term)
			  (F-substitution-p (id-of-variable-term term)
					    (length (bound-terms-of-term term))))
		     (F-substitute term))
		
		    ((variable-p term)
		     (F-variable-occurence (id-of-variable-term term)))

		    (t (abstraction-meta-variable-term-r term) ; flag possible sub into variable parameter.
		       (let ((root-address (cons nil nil)))
			 (setf (car root-address)
			       (prog2
				   (update-id-to-node-map term root-address 'F)
				   (instantiate-term (F-instantiate-operator
						      (operator-of-term term))
						     (mapcar #'F-bound-term
							     (bound-terms-of-term term)))

				 (restore-id-to-node-map term 'F)))))))

		  
		(F-bound-term (bound-term)
		  (prog2
		      (update-pick-map-and-node-scope-queue 
		       (bindings-of-bound-term-n bound-term) 'F)

		      (instantiate-bound-term (F-term (term-of-bound-term bound-term))
					    (mapcar #'(lambda (id)
							(let ((binding-id (value-of-parameter-value id)))
							  (when (variable-id-p binding-id)
							    (xnode-preferred-id
							     (pick binding-id 'F)))))
						    (bindings-of-bound-term-n bound-term)))

		    (restore-pick-map-and-node-scope-queue
		     (bindings-of-bound-term-n bound-term) 'F)))
					      

		;; there is a substitution
		(F-substitute (term)
		  ;;<> (priority(pick(id F')) = nil) iff not ThereExists (id b) in U
		  (G (term-of-xnode 
		      (pick (id-of-variable-term term)
			    'F
			    (length (bound-terms-of-term term))))
		     (bound-terms-of-term term)))

		;; first order var and no substitution
		(F-variable-occurence (id)
		  ;; when ThereExist (id b) in U
		  ;; let (id b) be pick(id U) and n be (id b I A) in N'
		  ;;   <> a-x in A
		  ;;   <> ForAll o-x in NSQ : root-address(o) << a
		  (let ((xnode (pick id 'F)))
		    ;;<> (priority(pick(id F')) = nil)
		    ;; iff not ThereExists (id b) in U
		    (if xnode
			;; bound occurence
			(progn 
			  (update-scope-conflicts-of-node xnode)
			  (setf (xnode-exists-occurence xnode) t)
			  (term-of-xnode xnode))
		    
			(progn
			  (setf xnode (get-free-xnode id))
			  (update-scope-conflicts-of-node xnode)
			  (term-of-xnode xnode)))))
		)

	     (F-term ri)))


       ;;;;  -page-----------------------------------------------------------

       ;;;;  -------------------------  Rename  -----------------------------

	(rename ()
		(labels
		    ((rename-variable (xnode)
		       ;; Let n=(x a I A) be xnode
		       ;;  <> ForEach o-x = (y b J B) in conflicts(n-x)
		       ;; idea is mimic conflicts conditions (below not quite right)
		       ;;    root-address(o-x) = root-address(n-x)
		       ;;      -> PairMin(o-x) < PairMin(n-x)
		       ;;    or priority(n) < priority(o)
		       ;;       and OccursIn(n,o) or OccursIn(o,n)
		       ;;     or priority(n) = priority(o) and OccursIn ...
		       (let ((ids-taken nil))
			 (mapc #'(lambda (higher-xnode)
				   (if (xnode-exists-occurence higher-xnode)
				       ;; resolve
				       (push (xnode-preferred-id higher-xnode) ids-taken)
				       ;; defer
				       (push xnode (xnode-conflicts higher-xnode))))
			       (xnode-conflicts xnode))
			 (when ids-taken
			   (let ((new-id (get-similar-allowed-id
					  (xnode-preferred-id xnode)
					  ids-taken
					  :trim-p t)))
			     (unless (eq new-id (xnode-preferred-id xnode))
			       (setf (xnode-preferred-id xnode) new-id)
			       (nreplace-variable-term-id (term-of-xnode xnode) new-id)
			       (mapc #'(lambda (index-pair)
					 (nreplace-binding 
					  new-id
					  (nth (car index-pair)
					       (bound-terms-of-term
						(car (xnode-root-binding-address xnode))))
					  (cdr index-pair)))
				     (xnode-local-binding-addresses xnode)))) ))))

		  (when any-conflict
		    (incf *no-subst2-renamed-m*)
		    (let ((dummies nil))
		      (do ((i 1 (1+ i)))
			  ((= i MaxP))
			(mapc #'(lambda (xnode)
				  (if (xnode-exists-occurence xnode)
				      (rename-variable xnode)
				      (push xnode dummies)))
			      (nreverse (aref priority-buckets i))))

		      (mapc #'(lambda (dummy)
				;; set flag so that conflicts between dummies are
				;; resolved rather than deferred in rename-variable.
				(setf (xnode-exists-occurence dummy) t)
				(rename-variable dummy))
			    (nreverse dummies)))))) )


      ;;;;  -page-----------------------------------------------------------

      ;;;;  ------------------   Initialization   --------------------------

      (with-variable-invocation
	  (prog2

	      ;; initialize
	      (progn
		;; selections.
		(mapc #'(lambda (s)	; selection
			  (let ((evcb (get-evcb (variable-selection-id s))))
			    (setf (evcb-selectedp evcb) t)
			    (setf (evcb-selected-id evcb)
				  (nth (variable-selection-binding-index s)
				       (bindings-of-bound-term-r
					(cdr (nth (variable-selection-term-index s) 
						  subs)))))))
		      selections)
	  
		;;  z-bindings
		(mapc
		 #'(lambda (sub)
		     (update-pick-map (car sub)
				      'F
				      (construct-xnode
				       :term (cdr sub)
				       :arity
				       (length 
					(bindings-of-bound-term-n (cdr sub))))))
		 subs)

		;; parameter instantiations
		;;(mapc #'(lambda (parameter-sub)
		;;(enter-binding (car parameter-sub)
		;; ;;(value-of-parameter-m (cdr parameter-sub))
		;;(cdr parameter-sub)
		;;))
		;;parameter-subs)
		)

	      ;; process
	      (F term)

	    ;; resolve names
	    (rename)))
      ))))
