
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; RLE TODO needs to be rewritten since dependency no longer contains address.aaaa

;;;;
;;;; -docs- (mod com)
;;;;	***** BEGIN VAPORWARE ALERT ****
;;;;
;;;;	DDAG: dependency dag.
;;;;
;;;;	Each object will have a stamp (thm's may have two) indicating last
;;;;	modification.  When a reference is made to an object, the object's stamp
;;;;	and the object's address make a dependency.
;;;;
;;;;	A ddag will be a set of <dependency, dependency sexpr> pairs. An
;;;;	object's address will occur on left of pair only once within a ddag. A
;;;;	pair may be added to a ddag via an open and removed via a close. Once a
;;;;	pair is open, its rhs dependency sexpr may be incrementally added to or
;;;;	wholly replaced.
;;;;
;;;;	In a ddag D, a dependency a references a dependency b iff there exists a
;;;;	pair <d, R> in D such that the address in dependency d equals address in
;;;;	dependency a and there exists a dependency c in R such that address in c
;;;;	equals address in b. Note that the stamps of the dependencies are not
;;;;	considered when determining references.
;;;;	
;;;;	In a ddag D, a dependency a occurring in R of a pair <d, R> in D is
;;;;	stale when there does not exist a pair <b, S> in D such that a equals b.
;;;;	To be equal both the address and the stamp of the dependencies need to
;;;;	be equal. Note that by definition if Exists <d, R> in D then d is not
;;;;	stale.
;;;;
;;;;	A consistent ddag will be one where :
;;;;	 * There exist no stale dependencies.
;;;;	 * There exist no cycles.
;;;;
;;;;	The dependency dag is meant to assist and not control the user.  The
;;;;	ddag is used to help the user indentify and correct inconsistencies, not
;;;;	to require a consistent state.  However, since it is not forseen that
;;;;	there will be much utility in allowing transient cycles, cycles will be
;;;;	prevented through failure.
;;;;
;;;;
;;;;	Some probable uses of the ddag:
;;;;	  * Produce partial ordering of dependencies such that there are no
;;;;	    foward references.
;;;;	  * produce ordered list of dependencies to be refreshed to make ddag
;;;;	    consistent.
;;;;	  * produce closure of dependencies which a dependency is referenced by.
;;;;	  * produce closure of dependenceis which a dependency references.
;;;;	
;;;;	
;;;;	DDdags may be combined to form compound ddags.
;;;;
;;;;  -page-
;;;;
;;;;	<ddag>		: <simple-ddag> | <ddag> list
;;;;	<simple-ddag>	: <primitive-ddag> | <ephmeral-ddag> 
;;;;
;;;;	<primitive-ddag>: primitive-ddag[]
;;;;	<ephemeral-ddag>: ephmeral-ddag[<closure{references}>]
;;;;	<compound-ddag>	: <ddag> list
;;;;
;;;;
;;;;	Invariants:
;;;;	 * In a primitive-ddag only one dependency may be opened per object-address.
;;;;	 * No primitive ddag function may be called on a dependency arg unless
;;;;	   the dependency is open.
;;;; 
;;;;	Additional ephemeral DDag Invariants.
;;;;	 * The dependencies returned by the references closure will not be stale.
;;;;	   Consequently, an ephemeral ddag can not contain stale dependencies.
;;;;	 * The dependencies returned by multiple calls of the references closure for
;;;;	   a dependency will be identical if there have been no intervening ddag 
;;;;	   opens or closes.
;;;;
;;;;  -page-
;;;;
;;;;	Creating DDags:
;;;;
;;;;	primitive-ddag(<id{name}>)				: <ddag>
;;;;	ephemeral-ddag(<id{name}> <closure{references}>)	: <ddag>
;;;;	 ** references(<dependency>)	: <dependency> sexpr
;;;;	compound-ddag(<ddag> list)				: <ddag>
;;;;
;;;;  -page-
;;;;
;;;;	DDag reference functions : 
;;;;
;;;;	** Stamps are meaningless except for refresh.
;;;;	** No duplicate addresses in results.
;;;;
;;;;	ddag-universe(<ddag>)					: <dependency> sexpr
;;;;
;;;;	references-of-dependency(<ddag> <dependency>)		: <dependency> sexpr
;;;;	reference-closure-of-dependency(<ddag> <dependency>)	: <dependency> sexpr
;;;;
;;;;	referenced-bys-of-dependency(<ddag> <dependency>)	: <dependency> sexpr
;;;;	reference-by-closure-of-dependency(<ddag> <dependency>)	: <dependency> sexpr
;;;;
;;;;	ddag-partial-order (<ddag> &optional <dependency> sexpr)
;;;;	 : [<dependency> list | <dependency>] list
;;;;	ddag-refresh(<ddag> &optional <dependency> sexpr)
;;;;	 : [<dependency> list | <dependency>] list
;;;;	 ** universe will be default <dependency> sexpr.
;;;;	 ** result is reference closure of subset of input sexpr referencing 
;;;; 	    stale dependencies.
;;;;
;;;;  -page-
;;;;
;;;;	Primitive DDag functions:
;;;;
;;;;	ddag-open(<primitive-ddag> <dependency>)			: NULL
;;;;	ddag-close(<primitive-ddag> <dependency>)			: NULL
;;;;
;;;;	ddag-add(<primitive-ddag> <dependency> <dependency> sexpr)	: NULL
;;;;	 ** fails if insertion would create cycle. 
;;;;	 ** when failure, ddag should not be modified.
;;;;
;;;;	ddag-cycle-p(<primitive-ddag> <dependency> <dependency> sexpr)	: <bool>
;;;;	 ** t if ddag-add would fail.
;;;;
;;;;	ddag-replace(<primitive-ddag> <dependency> <dependency> sexpr)	: NULL
;;;;	 ** fails if creates cycle. 
;;;;
;;;;	***** END VAPORWARE ALERT ****
;;;; -doce-

;;;; RLE ??? It is nonsensical to ask for a closure where there exists stale dependencies.
;;;;  Referencing A stale dependency should be a dead end/failure (depending on purpose)
;;;;  when computing a closure.

;;;; RLE ??? desire some method of attempting to damp ripples when making a lib consistent
;;;;	after a modification.




;;;;

;;;;	It is expected that ddag-add or ddag-cycle-p will be called frequently.
;;;;	Thus the implementation should be tuned for this purpose.
;;;;	Most likely, global operations such as closure and refresh will be called
;;;;	infrequently.
;;;;


;;;
;;; ddags
;;;



;;;;	4/99
;;;;	
;;;;	Sources of DDag data:
;;;;	  - translations of active objects (maybe do dynamic translations for inactive if eager?).
;;;;	  - opportunistic map on object term data.
;;;;	      - !dependency occurences are references.
;;;;	      - {<oid>:o} occurrences can be coerced to dependency(oid, nil nil)
;;;;	
;;;;	new variant for collect_orphans() :
;;;;	 only look at active objects, (except maybe look at com&term objects whether active or not)
;;;;	
;;;;	
;;;;	
;;;;	Flavors : ddag data may be paramterized by different types of dependencies,
;;;;	  eg, source, compile etc.
;;;;	
;;;;	<DDag>	: !ddag{<oid>}(<depedency>; <dependency_store>)
;;;;		| !ddag{<oid>}(<depedency>; <dependency_store>; <xref>)
;;;;	
;;;;	
;;;;	maybe ddag needs to be some subclass of definition, make it analogous to ostate.
;;;;	
;;;;	
;;;;	xref : could be reduced to dependencies,
;;;;	assume A references function defined in B.
;;;;	  - A will have dependency on B.
;;;;	  - B will have list of functions defined with types.
;;;;	  - Assume we can produce through source examination :
;;;;	      - A will have list of functions used possibly with oid of def if ambiguous.
;;;;	Then we can produce xref from dependencies.
;;;;	
;;;;	  producing the list of callers of a function :
;;;;	    - find list of dependent objects of object defining function.
;;;;	    - produce sub list of defined funcs referencing target func.
;;;;	  Point is that this is by nature a demand question and it can be supplied
;;;;	  on demand.
;;;; 
;;;;	  The object dependency graph is not a demand question as it is natural to navigate
;;;;	  the dependency graph fluidly.
;;;;	
;;;;	  Do still need ability to find defining object given an ML id.
;;;;	  thus need tok -> oid hash.
;;;;	
;;;;	
;;;;	Build oid keyed hash table with values:
;;;;	  
;;;;	  <dependency>, <dependency-store>
;;;;	
;;;;	
;;;;	build tok keyed hash tables (assoced by language) with values :
;;;;	  <oid>, <xref> 
;;;;	
;;;;	
;;;;	references_of_oid	: object_id -> (tok  # object_id list) list
;;;;	oid_references		: object_id -> (tok  # object_id list) list
;;;;	show_references_of_oid	: object_id -> term
;;;;	show_oid_references	: object_id -> term
;;;;
;;;;	oid_of_id		: tok{kind(eg ML)} -> tok{id} -> 
;;;;	ids_of_oid		: object_id -> tok list
;;;;
;;;;	oids_of_id		: tok{kind(eg ML)} -> tok{id} -> object_id list
;;;;	  * when we allow ambiguous ids.

;;;;	
;;;;	
;;;;	references_of_oid      
;;;;	  - build hash table which is lazily instantiated at request.
;;;;	  - clear hash when dependencies updated.
;;;;	      * initially clear whole table at each update.
;;;;	      * eventually incremental clear.
;;;;	
;;;;	show_dependencies oid 
;;;;	show_dependents oid

;;;;	
;;;;	ml-xref
;;;;	
;;;;	object_of_mlid : object defining id.
;;;;	called_of_mlid : mlxref
;;;;	callers_of_mlid : mlxref
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	

;; hash value could be list index by oid.
;; or cons mlid with oid stamp to make hash key or use mlid if null oid.

;;;;	should be part of environment MTT could be part of ddg-table.

(defun ddg-references-hash ()
  (let ((ddg (resource 'ddg)))
    (when ddg
      (let ((refs (ddg-table-references ddg)))

	(definition-table-map ddg
	    (current-transaction-stamp)
	    #'(lambda (oid def)
		(declare (ignore oid))

		(let ((store (store-of-ddg-def def))
		      (sdep (dependency-of-definition def)))
		  (dolist (deps (normal-list-of-dependency-store store))
		    (let ((tag (tag-of-dependencies deps)))
		      (dolist (dep (list-of-dependencies deps))
			(let ((key (stamp-of-oid (oid-of-dependency dep))))
			  (let ((rstore (gethash key refs)))
			    (unless rstore
			      (setf (gethash key refs) (setf rstore (normal-dependency-store nil))))

			    (let ((rdeps (tagged-dependencies-of-dependency-store tag rstore)))
			      (unless rdeps
				(dependency-store-add-dependencies rstore
								   (setf rdeps (new-dependencies tag nil))))
			      (setf -rdeps rdeps -dep dep)
			      (pushnew sdep (dependencies-list rdeps)
				       :key #'oid-of-dependency
				       :test #'equal-oids-p
				       ))))))))))
	)

      (setf (ddg-table-need-ref-p ddg) nil))))


(defun references-of-ddg-table (ddg)
  (when (ddg-table-need-ref-p ddg)
    (ddg-references-hash))

  (ddg-table-references ddg))


(defun ddg-equal-xid-source (src-a src-b)
  (cond
    ((null src-a) (null src-b))
    ((oid-p src-a) (and (oid-p src-b)
			(equal-oids-p src-a src-b)))
    (t (and (not (null src-b))
	    (not (oid-p src-b))
	    (compare-terms-p src-a src-b)))))
  

(defun ddg-lookup-xid (xref id src)
  (find-if #'(lambda (e) (ddg-equal-xid-source src (source-of-xref e)))
	   (gethash id xref)))

(defun ddg-set-xid (xref id src e)
  ;;(when (consp e)  (setf -oid oid -e e -xref xref) (break "dsi"))
  ;; src can be oid or term
  (let ((l (gethash id xref)))
    (setf (gethash id xref)
	  (cons e (delete-if #'(lambda (e)
				 (ddg-equal-xid-source src (source-of-xref e)))
			     l)))))

;;;;	
;;;;	pass 1 : init hash table with entries by caller and set called of caller.
;;;;	pass 2 : push callers onto called.
;;;;	pass 3 : index mlid's to list of oids defining.
;;;;		 
;;;;	note that we can get mlxref from oid directly.
;;;;	
;;;;	id | id list ; vast majority will be single id only mutually recursive ml functions
;;;;	produce list, thus simplify by assuming id and duplicating mutually recursive ml.
;;;;	
;;;;	
;;;;	id	: token
;;;;		| token # oid
;;;;		| token # term
;;;;	
;;;;	xref :
;;;;	  - oid
;;;;	  - type
;;;;	  - mutable	: bool
;;;;	  - called	: id+ list
;;;;	  - callers	: id+ list
;;;;	  
;;;;	* note : mutual definitions while listed independently share the called list.
;;;;	  eg if a and b are mutual definitions they will have identical call lists which
;;;;	  are the union of the called of a and b. This is an artifact of the ML compiler
;;;;	  and is not intentional nor desirable.
;;;;	
;;;;	

;;;;
;;;;	id+	: xref 
;;;;	id	: xref list
;;;;	oid	: xref list


(defun xref-ap-ddg-xrefs (f oid frefs)
  (cond
    ((xref-entry-p frefs)
     ;; may need to change dependencies to oids.
     (setf (xref-entry-source frefs) oid)
     (funcall f frefs))

      
    ;; backwards compatable mlxref?
    (t
     (let ((called (mapcar #'(lambda (called)
			       (xref-id
				(id-of-mlxref-called called)
				(let ((d (dependency-of-mlxref-called called)))
				  (cond
				    ((eql d 'this) oid)
				    (t (when (dependency-p d) (oid-of-dependency d)))))))
			   (called-of-mlxref frefs))))
       (mapc #'(lambda (caller)
		 (funcall f 
			  (new-xref-entry (refp-of-mlxref frefs)
					  (type-of-mlxref-caller caller)
					  (id-of-mlxref-caller caller)
					  oid
					  called)))
	     (callers-of-mlxref frefs))))))

;; mlxref hash value
(defun ddg-xref-hash ()
  (let ((ddg (resource 'ddg)))
    (when ddg
      (let ((xrefs (ddg-table-xref ddg)))

	(clrhash xrefs)

	(definition-table-map ddg
	    (current-transaction-stamp)
	  #'(lambda (oid def)

	      (let ((xref (xref-of-ddg-def def)))
		(when xref

		  (dolist (frefs xref)

		    (xref-ap-ddg-xrefs #'(lambda (xref)
					   (setf (xref-entry-term xref) nil)
					   (ddg-set-xid xrefs
							(xref-entry-id xref)
							oid
							xref))
				       oid
				       frefs)
		    )))))

	(format t "dxh 2")

	(definition-table-map ddg
	    (current-transaction-stamp)
	  
	  #'(lambda (oid def)

	      (setf -oid oid -def def)
	      
	      (let ((xref (xref-of-ddg-def def)))
		(when xref
		  (dolist (frefs xref)
		    (xref-ap-ddg-xrefs
		     #'(lambda (xref)
			 (setf -xref xref)
			 (let ((call (xref-id (id-of-xref xref) oid)))

			   (setf -call call)
			   (dolist (called (called-of-xref xref))

			     ;;(setf -called called -call call ) (break "h")

			     (let ((src (source-of-xref-id called))
				   (did (id-of-xref-id called)))

			       (when (and (term-p src) (ithis-term-p src))
				 (setf src oid))

			       (let ((dref (ddg-lookup-xid xrefs did src)))

				 (unless dref
				   (ddg-set-xid xrefs did src
						(setf dref (new-xref-pre-entry did src))))

				 ;;(when (eql did '|check_refenv|) (setf -did did -src src -dref dref) (break "ddg"))
				 (pushnew call
					  (xref-entry-callers dref)
					  :test
					  #'(lambda (calla callb)
					      (and (eql (car calla) (car callb))
						   (ddg-equal-xid-source (cdr calla)
									 (cdr callb)))) ))))))
		     oid
		     frefs))))))

	;; add oid lookups.
	(maphash #'(lambda (k v)
		     ;;(setf -k k -v v)
		     (when (symbolp k)
		       (dolist (xref v)
			 (let ((src (source-of-xref xref)))
			   (when (oid-p src)
			     ;;(when (equal-oids-p -xoooid src) (break "dfa"))
			     (push xref (gethash (stamp-of-oid src) xrefs)))))))
		 xrefs)
	)

      (setf (ddg-table-need-xref-p ddg) nil)
      )))

(defun xref-of-ddg-table (ddg)
  (when (ddg-table-need-xref-p ddg)
    (ddg-xref-hash))

  (ddg-table-xref ddg))


(defun xids-of-id (id)
  (mapcan #'(lambda (e)
	      (let ((src (source-of-xref e)))
		(when src
		  (list (cons id src)))))
	  (gethash id (xref-of-ddg-table (resource 'ddg)))))

(defun callers-of-xid (xid)
  (callers-of-xref (ddg-xref-of-xid (car xid) (cdr xid))))

(defun closure-of-xids (xids)
  ;;(setf -xids xids) (break "cox")
  (let ((acc (reverse xids)))

    (labels ((aux (xids)
	       (let ((new nil))
		 ;;(setf -xxids xids) (break "coxx")

		 (mapc #'(lambda (xid)
			   (mapc #'(lambda (xid)
				     ;;(setf -xid xid) (break "xid")
				     (unless (or (not (active-of-ostate (cdr xid)))
						 (member xid acc
							 :test #'(lambda (a b)
								   (and (eql (car a) (car b))
									(equal-oids-p (cdr a) (cdr b))))))
				       (push xid new)
				       (push xid acc)))
				 (unless (null xid)
				   (callers-of-xid xid))))
		       xids)
		 (when new (aux new)))))
      
      (aux xids)
      (nreverse acc))))

(defunml (|xids_of_id| (id))
    (tok -> ((tok |#| object_id) list))

  (let ((xids (xids-of-id id)))
    (if (exists-p #'(lambda (xid) (not (oid-p (cdr xid)))) xids)
	(raise-error (error-message '(|xids_of_id| source obid not) id))
	xids)))

(defunml (|xidts_of_id| (id))
    (tok -> (term list))

  (remove-duplicates 
   (mapcar #'(lambda (xid)
	       (when (not (eql (car xid) id))
		 (setf -xid xid)
		 (break "xtoi"))

	       (let ((src (cdr xid)))
		 (cond
		   ((oid-p src) (ioid-term src))
		   ((term-p src) src)
		   (t (setf -xid xid) (break "xtois")))))
	   (xids-of-id id))
   :test #'compare-terms-p))
	

(defunml (|closure_of_xids| (xids))
    (((tok |#| object_id) list) -> ((tok |#| object_id) list))

  (closure-of-xids xids))
)

  
(defun ddg-xrefs-of-id (id)
  (let ((ddg (resource 'ddg)))
    (unless ddg
      (raise-error (error-message '(ddg not))))
    
    (let ((xref (xref-of-ddg-table ddg)))
      ;;(break "dxoi")
      (sort (copy-list (gethash id xref))
	    #'(lambda (x y)
		(declare (ignore y))
		(let ((src (source-of-xref x)))
		  (and src
		       (when (oid-p src)
			 (active-of-ostate src))))) )) ))


;; oid has entry in xref table which would be list of
;; all id xrefs?
(defun ddg-xrefs-of-oid (oid)
  (let ((ddg (resource 'ddg)))
    (unless ddg
      (raise-error (error-message '(ddg not))))

    (let ((xref (xref-of-ddg-table ddg)))
      (gethash (stamp-of-oid oid) ;;(cons nil (stamp-of-oid oid))
	       xref))))

(defun ddg-xref-of-xid (id oid)
  (let ((ddg (resource 'ddg)))
    (unless ddg
      (raise-error (error-message '(ddg not))))

    (when (null id)
      (message-emit (oid-warn-message (list oid) '(ddg xid id null))))

    (let ((xref (xref-of-ddg-table ddg)))
      (or (find-if #'(lambda (x) (let ((s (source-of-xref x))) (and s (oid-p s) (equal-oids-p s oid))))
		   (gethash id xref))
	  (raise-error (oid-error-message (list oid) '(ddg-xref-of-xid none) id))))))


(defun ddg-xref-ids ()
  (let ((ddg (resource 'ddg)))
    (unless ddg
      (raise-error (error-message '(ddg not))))

    (let ((acc nil))
      (maphash #'(lambda (k v)
		   (when (symbolp k)
		     (push (string k) acc)))
	       (xref-of-ddg-table ddg))

      (sort acc #'string<))))

(defun ddg-xref-clash ()
  (let ((ddg (resource 'ddg)))
    (unless ddg
      (raise-error (error-message '(ddg not))))

    (let ((calleroids nil)
	  (acc nil))
      (maphash #'(lambda (k vv)
		   (when (symbolp k)
		     (format t ".")
		     ;; value is list of xrefs
		     (setf -vv vv)
		     (let ((v (filter #'(lambda (xe)
					  (let ((o (source-of-xref xe)))
					    (when (and o (oid-p o) (active-of-ostate o))
					      t)))
				      vv)))
		       (when (> (length v) 1)
			 (mapc #'(lambda (xe)
				   (unless (oid-p (source-of-xref xe))
				     (mapc #'(lambda (c)
					       (let ((o (source-of-xref-id c)))
						 (when (and o (oid-p o) (active-of-ostate o))
						   (push o calleroids))))
					   (callers-of-xref xe))))
			       v)
			 ;;(setf -k k -v v) (break "dxf")
			 (push (string k) acc)))))
	       (xref-of-ddg-table ddg))

      (format t "~%~a~%" (length calleroids))
      (cons (remove-duplicates calleroids :test #'equal-oids-p)
	    ;;(sort acc #'string<)
	    acc
	    ))))

(defunml (|ddg_xref_clash| (unit) :declare ((declare (ignore unit))))
    (unit -> (object_id list))

  (let ((r (ddg-xref-clash)))
    (mapc #'(lambda (x) (format t "~a~%" x)) (cdr r))
    (car r)))

  
  

(defun print-xref-ids-aux (stream)
  (let ((ids (ddg-xref-ids)))
    (dolist (s ids)
      (format stream "~a~%" s))))

(defun print-xref-ids ()
  (let ((ids (ddg-xref-ids)))
    (format t "~%~%XREF print of ~a ids start:~%~%" (length ids))
    (print-xref-ids-aux t)
    (format t "~%~%XREF ids print end.~%~%")))


(defun ddg-lookup-dependencies (oid)
  (let ((def (lookup-object-attr-def `ddg oid)))
    ;;(setf -def def -oid oid) (break "dld")
    (unless (or (null def) (object-attr-def-not-p def))
      (store-of-ddg-def def))))


(defun ddg-lookup-dependents (oid)
  (let ((ddg (resource 'ddg)))
    (if ddg
	(gethash (stamp-of-oid oid) (references-of-ddg-table ddg))
	(raise-error (oid-error-message (list oid) '(ddg lookup not))))))

	
(defunml (|object_dependencies| (oid))
    (object_id -> ((tok |#| (object_id list)) list))

  (let ((d (ddg-lookup-dependencies oid)))
    ;;(setf -d d) (break "od")
    (mapcar #'(lambda (dd)
		(cons (tag-of-dependencies dd)
		      (mapcar #'oid-of-dependency (list-of-dependencies dd))))
	    (dependencies-of-dependency-store d))
    ))

(defunml (|dependencies_of_object| (oid))
    (object_id -> term)

  (let ((d (ddg-lookup-dependencies oid)))
    (if d
	(dependency-store-to-term d)
	(ivoid-term))))

(defunml (|dependents_of_object| (oid))
    (object_id -> term)
  (let ((d (ddg-lookup-dependents oid)))
    (if d
	(dependency-store-to-term d)
	(ivoid-term))))





(define-primitive |!ddg_object| () (dependencies dependents))

(defunml (|ddg_object| (oid))
    (object_id -> term)

  (let ((dd (ddg-lookup-dependencies oid))
	(dt (ddg-lookup-dependents oid)))
    ;;(setf -dd dd -dt dt) (break "ddg")
    (iddg-object-term (if dd
			  (dependencies-sexpr-to-term
			   (dependencies-of-dependency-store dd))
			  (ivoid-term))
		      (if dt
			  (dependencies-sexpr-to-term
			   (dependencies-of-dependency-store dt))
			  (ivoid-term)))))
  


(defunml (|xrefs_of_object| (oid))
    (object_id -> term)
  
  ;;(setf -xoooid oid) (break "xoo")
  (xrefs-to-term (ddg-xrefs-of-oid oid)))

(defunml (|xrefs_of_id| (id))
    (token -> term)

  (xrefs-to-term (ddg-xrefs-of-id id)))

;;LAL returns oid of obj where id is defined and if no obj, returns oid of stm if id is a stm
(defunml (|xrefs_def_of_id| (id))
    (token -> object_id)
  (let ((xrefs (ddg-xrefs-of-id id)))
    (if xrefs
	(or (car (mapcan #'(lambda (x)
			     (let ((s (source-of-xref x)))
			       (when (and s (oid-p s))
				 (list s))))
			 xrefs))
	    (raise-error (error-message '(xrefs def id object not) id)))
	(raise-error (error-message '(xrefs def id not) id)))))
	

(defunml (|xref_of_xid| (id oid))
    (token -> (object_id -> term))

  (xref-to-term (ddg-xref-of-xid id oid)))

;; true prints to shell, false prints to spool file.
(defunml (|print_xref_ids| (b))
    (bool -> unit)

  (if b
      (print-xref-ids)
      (let ((fname "~/spool/xref-ids.txt"))
	(with-open-file (stream fname :direction :output)
	  (print-xref-ids-aux stream)))))



;;;;	
;;;;	validity : assume have ddg graph of oids where for each object we have the
;;;;	       list of objects it references.
;;;;	  + layering : 
;;;;	      - a layer is a grouping of independent objects.
;;;;	      - all dependents of an object occur in earlier layers.
;;;;	      - each object is placed in the least layer possible.
;;;;	  + cycle detection : if cycle then cannot layer.
;;;;	      - present algorithm which finds maximal set of layerable objects.
;;;;	      - an object is not layered iff it is part of a cycle.
;;;;	  + stale : an layering is stale if the time stamp of some referenced object
;;;;		is later than that of there referencing object.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	



;; ((oid{proxer} # oid{proxy} list) # (oid list){dependencies}) list -> (oid -> oid -> bool){po}
;;
;; 

(defun ddg-graph (dkinds proxy-kinds filter)
  (let ((ddg (resource 'ddg)))
    (when ddg
      (let ((acc nil))
	(without-dependencies
	 (definition-table-map ddg
	     (current-transaction-stamp)
	   #'(lambda (oid def)
	       (when (or (null filter) (funcall filter oid def))
		 (push (cons (cons oid
				   (mapcan #'(lambda (deps)
					       (when (member (tag-of-dependencies deps) proxy-kinds)
						 (mapcar #'oid-of-dependency (list-of-dependencies deps))))
					   (normal-list-of-dependency-store
					    (store-of-ddg-def def))))
			     (mapcan #'(lambda (deps)
					 (when (member (tag-of-dependencies deps) dkinds)
					   (mapcar #'oid-of-dependency (list-of-dependencies deps))))
				     (normal-list-of-dependency-store
				      (store-of-ddg-def def))))
		       acc)))))
	acc))))

(defun ddg-graph-aux (filter proxy-kinds)
  (let ((ddg (resource 'ddg)))
    (when ddg
      (let ((acc nil))
	(without-dependencies
	 (definition-table-map ddg
	     (current-transaction-stamp)
	   #'(lambda (oid def)
	       (let ((kinds (funcall filter oid def)))
		 (when kinds
		   (push (cons (cons oid
				     (mapcan #'(lambda (deps)
						 (when (member (tag-of-dependencies deps) proxy-kinds)
						   (mapcar #'oid-of-dependency (list-of-dependencies deps))))
					     (normal-list-of-dependency-store
					      (store-of-ddg-def def))))
			       (mapcan #'(lambda (deps)
					   (when (member (tag-of-dependencies deps) kinds)
					     (mapcar #'oid-of-dependency (list-of-dependencies deps))))
				       (normal-list-of-dependency-store
					(store-of-ddg-def def))))
			 acc))))))
	acc))))

(defun ddg-graph (kinds proxy-kinds filter)
  (ddg-graph-aux #'(lambda (oid def)
		     (when (funcall filter oid def)
		       (or kinds t)))
		 proxy-kinds))

(defunml (|ddg_graph| (filter proxykinds))
    ((object_id -> (tok list)) -> ((tok list) -> (((object_id |#| (object_id list)) |#| (object_id list)) list)))

  (ddg-graph-aux #'(lambda (oid def) (declare (ignore def)) (funmlcall filter oid))
	     proxykinds))

		      
(defun layer-ddg (type tags &optional filter)
  (let ((ddg (resource 'ddg)))
    (when ddg
      (new-layers
       #'(lambda (acc)
	   (definition-table-map ddg
	       (current-transaction-stamp)
	     #'(lambda (oid def)

		 (when (or (null filter) (funcall filter oid def))
		   (funcall acc
			    (or (let ((dag (layer-dag-of-ddg-def type def)))
				  (when dag
				    
				    ;; if deps changed then new def and thus this should accomplish nada.
				    (when nil (setf (layer-dag-refs dag)
						    (mapcar #'oid-of-dependency
							    (mapcan #'(lambda (deps)
									(when (member (tag-of-dependencies deps) tags)
									  (list-of-dependencies deps)))
								    (normal-list-of-dependency-store
								     (store-of-ddg-def def))))))
				    dag))
				(let ((dag (new-layer-dag
					    oid
					    (mapcar #'oid-of-dependency
						    (mapcan #'(lambda (deps)
								(when (member (tag-of-dependencies deps) tags)
								  (copy-list (list-of-dependencies deps))))
							    (normal-list-of-dependency-store
							     (store-of-ddg-def def)))))))
				  (setf (ddg-def-layers def) (acons type dag (layers-of-ddg-def def)))
				  dag)))))))))))


(defvar *code-layers* nil)
(defvar *code-layers-other* nil)

(defun code-layers-cycles ()
  (rehash-code-layers)
  (car *code-layers-other*))

(defun code-layers-undoable ()
  (rehash-code-layers)
  (cdr *code-layers-other*))

(defun code-layers-table ()
  (rehash-code-layers)
  *code-layers*)


(defun rehash-code-layers (&optional force)
  (when (or force (definition-table-flag-touched-p (resource 'ddg)))
    (mlet* (((table cycles undoable)
	     (layer-ddg 'compile
			'(compile)
			#'(lambda (oid def)
			    (declare (ignore def))
			    ;;(format t "~a " (kind-of-object-attr-def def))
			    (and (eql 'code (kind-of-ostate oid))
				 (active-of-ostate oid)
				 )))))

	   (setf *code-layers* table)
	   (setf *code-layers-other* (cons cycles undoable))

	   (definition-table-flag-set-touched (resource 'ddg) nil)
	   (list (hash-table-count table) (length cycles) (length undoable)))))

(defun find-ddg-loop (target)
  (let ((seen nil)
	(found nil)
	)
    (labels
	((visit (dag)
	   (if (eql dag target)
	       (progn
		 (setf found t)
		 (list dag))
	       (progn
		 (push dag seen)
		 (let ((r (visit-list (ref-dags-of-layer-dag dag) dag)))
		   (if found
		       (cons dag r)
		       r)))))

	 (visit-list (dags parent)
	   (if (null dags)
	       nil
	       (or (let ((dag (car dags)))
		     (unless (or (eql parent dag)
				 (member dag seen))
		       ;; don't visit self, ie cycle of one not a prob.
		       (visit dag)))
		   (visit-list (cdr dags) parent)))))

      (visit-list (ref-dags-of-layer-dag target) target))))


(defun find-all-code-cycles ()
  (let ((cycles (remove-duplicates (mapcar #'find-ddg-loop (code-layers-cycles))
				   :test #'(lambda (c1 c2) (equal-bags-p #'eql c1 c2)))))

    ;;(setf -cycles cycles) (break "facc")
    (mapcar #'(lambda (c) (mapcar #'layer-dag-oid c)) cycles)))

(defunml (|code_cycles| (unit) :declare ((declare (ignore unit))))
    (unit -> ((object_id list) list))

  (find-all-code-cycles))

(defun code-undoable-aux ()
  (mapcan #'(lambda (dag)
	      (when (exists-p #'null (ref-dags-of-layer-dag dag))
		(list dag)))
	  (code-layers-undoable)))

(defunml (|code_undoable| (unit) :declare ((declare (ignore unit))))
    (unit -> (object_id list))

  (mapcar #'oid-of-layer-dag (code-undoable-aux)))

(defunml (|code_undoables| (unit) :declare ((declare (ignore unit))))
    (unit -> ((object_id list) list))

  (let ((undoable (code-layers-undoable))
	(direct (code-undoable-aux)))

    (when undoable
      (list
       (let ((table (code-layers-table)))
	 (remove-duplicates
	  (mapcan #'(lambda (dag)
		      (mapcan #'(lambda (oid)
				  (when (null (hashoid-get table oid))
				    (list oid)))
			      (refs-of-layer-dag dag)))
		  direct)
	  :test #'equal-oids-p))
       (mapcar #'oid-of-layer-dag direct)
       (progn;;(setf -undoable undoable -direct direct) (break "ables")
	 (mapcan #'(lambda (dag)
		     (unless (member dag direct)
		       (list (oid-of-layer-dag dag))))
		 undoable)))))
  )

(defun code-layers-aux ()
  (let ((table (code-layers-table))
	(ll nil))

    (maphash #'(lambda (k dag)
		 (declare (ignore k))
		 (let ((i (index-of-layer-dag dag)))
		   (when i
		     (let ((l (assoc i ll)))
		       (if l
			   (setf (cdr l) (cons (oid-of-layer-dag dag) (cdr l)))
			   (setf ll (acons i (list (oid-of-layer-dag dag)) ll)))))))

	     table)

    (sort ll #'< :key #'car)))

(defunml (|code_layers| (unit) :declare ((declare (ignore unit))))
    (unit -> ((int |#| (object_id list)) list))
  (code-layers-aux))

;; t if aoid later than boid.
(defun ostate-modify-time-compare (aoid boid)
  (let ((a-subst (substance-of-ostate aoid))
	(b-subst (substance-of-ostate boid)))

    (or (null a-subst)
	(stamp-later-p a-subst b-subst))))

(defun code-stale-p (oid)

  (let ((table (code-layers-table))
	(subst (substance-of-ostate oid)))

    (or (null subst)
	(let ((dag (hashoid-get table oid)))
	  (exists-p #'(lambda (oid)
			(let ((a-subst (substance-of-ostate oid)))
			  (or (null a-subst)
			      (stamp-later-p a-subst subst)
			      )))
		    (refs-of-layer-dag dag))))))

(defunml (|code_stale_p| (oid))
    (object_id -> bool)

  (code-stale-p oid))
    
(defunml (|code_stale| (unit) :declare ((declare (ignore unit))))
    (unit -> (object_id list))

  (let ((table (code-layers-table))
	(stale nil))

    (dolist (ll (code-layers-aux))
      (let ((l (cdr ll)))
	(dolist (oid l)
	  (let ((subst (substance-of-ostate oid))
		(dag (hashoid-get table oid)))
	    (when (or (null subst)
		      (exists-p #'(lambda (oid)
				    (let ((a-subst (substance-of-ostate oid)))
				      (or (null a-subst)
					  (> (time-of-stamp a-subst) (time-of-stamp subst))
					  (and (= (time-of-stamp a-subst) (time-of-stamp subst))
					       ;; probably not robust.
					       (> (sequence-of-stamp a-subst) (time-of-stamp subst)))
					  (let ((dag (hashoid-get table oid)))
					    ;; technically if in the layers, it's got to have a dag,
					    ;; but that could be relaxed later.
					    (when dag
					      (layer-dag-flag-stale-p dag)))
					  )))
				(refs-of-layer-dag dag)))

	      (layer-dag-flag-set-stale dag t)
	      (push oid stale) )))))

    (nreverse stale)))

;; compare layers to a list and detect order conflicting with layers.
(defun code-out-of-order (dir avoid-f)

  (let ((ooo nil)
	(i 0)
	(table (code-layers-table))
	(listing (subtree-oids dir  #'(lambda (oid)
					(funmlcall avoid-f oid)))))
    
    ;;(setf -listing listing) (break "cooo")
    (let ((ltable (make-hash-table :test #'equal)))
      (dolist (oid listing)
	(incf i)
	(let ((dag (hashoid-get table oid)))
	  (when dag
	    (hashoid-set ltable oid (cons i dag)))))

      (maphash #'(lambda (stamp idag)
		   (declare (ignore stamp))

		   (let ((i   (car idag))
			 (dag (cdr idag)))
		     (let ((bad (mapcan #'(lambda (roid)
					    (let ((ridag (hashoid-get ltable roid)))
					      (when (and ridag (> (car ridag) i))
						(list roid))))
					(refs-of-layer-dag dag))))
		       (when bad
			 (pushnew (cons (oid-of-layer-dag dag) bad) ooo :key #'car :test #'equal-oids-p)
			 ))))
	       ltable))

    ooo))

;; find layered graph containing oid.
;; show include all descendents and ancestors.

(defun layer-list (layers l key test)
  (setf -l l -layers layers) (break "odag")
  ;; reconstitute layers.
  (mapcar #'(lambda (layer)
	      (cons (car layer)
		    (mapcan #'(lambda (oid)
				(when (member (funcall key oid) l :test test)
				  (list oid)))
			    (cdr layer))
		    ))
	  layers))

(defun code-layered-refs-aux (oid)
  (let* ((table (code-layers-table))
	 (layers (code-layers-aux))
	 (odag (hashoid-get table oid))
	 (l (list odag)))

    
    (when (and odag (index-of-layer-dag odag))

      ;; find ancestors
      (dolist (ll (nthcdr (1+ (index-of-layer-dag odag)) layers))
	(dolist (oid (cdr ll))
	  (let ((dag (hashoid-get table oid)))
	    (when (exists-p #'(lambda (rdag)
				(member rdag l))
			    (ref-dags-of-layer-dag dag))
	      (push dag l)))))

      ;; add descendents
      (labels ((visit (dag)
		 (unless (member dag l)
		   (push dag l)
		   (mapcar #'visit (ref-dags-of-layer-dag dag)))))

	;; can't just visit odag since odag member of l.
	(mapcar #'visit (ref-dags-of-layer-dag odag)))

      ;;(setf -odag odag -l l -layers layers) (break "odag")
      ;; reconstitute layers.
      (layer-list layers l #'(lambda (oid) (hashoid-get table oid)) #'eql)  
      #|(mapcar #'(lambda (layer)
		  (cons (car layer)
			(mapcan #'(lambda (oid)
				    (when (member (hashoid-get table oid) l)
				      (list oid)))
				(cdr layer))
			))
	      layers)|#
      )))

(defunml (|code_layered_refs| (oid))
    (object_id -> ((int |#| (object_id list)) list))

  (code-layered-refs-aux oid))

(defunml (|code_order_not| (avoid-f dir))
    ((object_id -> bool) -> (object_id -> ((object_id |#| (object_id list)) list)))

  (code-out-of-order dir avoid-f))


(defunml (|code_subtree_layers| (dir avoidp))
    (object_id -> ((object_id -> bool) -> ((int |#| (object_id list)) list)))

  (layer-list (code-layers-aux)
	      (subtree-oids dir
			    #'(lambda (oid) (funmlcall avoidp oid)))
	      #'identity #'equal-oids-p))


(defunml (|ref_code_object_id_p| (oid))
    (object_id -> bool)
  
 (and (let ((def (lookup-ostate-def oid)))
	(when (and def (not (ostate-def-not-p def))) 
	  (and (eql 'code (kind-of-iobject-state-term (state-of-ostate-def def)))
	       (member 'refine
		       (let ((desc (description-of-iobject-state-term (state-of-ostate-def def))))
			 (when (idescription-term-p desc)
			   (map-isexpr-to-list (purposes-of-idescription-term desc)
					       (icons-op)
					       #'token-of-itext-term)))))))
      t))

;;;;	
;;;;		
;;;;	closure : tok list(dependency types) -> object_id list ->
;;;;			(object_id list {consistent} # object_id list {inconsistent})
;;;;		
;;;;	closure : (dependency_store -> object_id -> object_id -> bool) {partial order inequality}
;;;;			-> object_id list
;;;;			-> (object_id list {consistent} # object_id list {inconsistent})
;;;;	
;;;;	
;;;;	  object list is converted to ddg partial order wrt types.
;;;;	  consistent objects are those whose mod times are consistent with the partial order.
;;;;	  
;;;;	Mod time is a total order, thus could define consistency to be that 
;;;;	the partial order and total order are consistent, ie that are there
;;;;	


;;;;	
;;;;	(oid{dependent} # oid list{dependencies}) list
;;;;	
;;;;	closure :  all dependents of seed list. 
;;;;
;;;;	dependency-closure : all dependencies of seed list.
;;;;	
;;;;	
;;;;	



;; twould be nice to fail in case where po is not a proper partial order relation.
;; twould be nice to fail in case where to is not a proper total order relation.
;; or maybe a bool arg to say whether to assume orders are proper.


;; if po only relates locally is this ok?
;; ie if a <p b & b <p c po might not report a <p c
;; looks ok since if a follows c and b preceds c in total order then
;; b must preceed a thus a already inconsistent.


;; if expect number of items to be huge may want some sort of
;; hash table implementation.
#|(defun partial-order-closure (check-cycles-p po eq seeds items)
  ;; improper po if we find a cycle.

  (let ((closure (if check-cycles-p
		     (delete-duplicates (copy-list seeds) :test eq)
		     (copy-list seeds)))
	(lastcl seeds)
	(rem (set-difference (if check-cycles-p
				 (remove-duplicates items :test eq)
				 items)
			     seeds :test eq))
	(haltp nil))

    (do ()
	(haltp)
	
      (let ((newcl (filter #'(lambda (a)
			       (exists-p #'(lambda (b)
					     ;;(setf -a a -b b) (break "ep")
					     (funcall po b a))
					 lastcl))
			   rem)))

	;;(setf -newcl newcl) (break "newcl")
	(if (null newcl)
	    (setf haltp t)
	    (progn
	      ;;(setf -rem rem)
	      (setf rem (set-difference rem newcl :test eq))
	      (setf lastcl newcl)
	      (setf closure (nconc newcl closure))
	      ))))

    (when (and check-cycles-p
	       (duplicate-p closure))
      (raise-error (error-message '(partial-order-closure cycle))))

    closure))
|#

;; find res to recompile:
;; use substance time-stamp 
;;    modify-time-compare
;;
;;  b declared c as proxy -> dependencies on c are dependencies on b.
;;  partial order : static dependencies - proxy
;;   ie a <p b : if a is a static dependency of b but not declared as b's proxy.
;;               or a <p c and c is proxy for b.

;; then given static dependencies and proxy declarations
;; ((oid{proxer} # oid{proxy} list) # (oid list){dependencies}) list -> (oid -> oid -> bool){po}
;; determine partial order.
;; if b declares c as proxy the call b the proxer.
;; remove proxy from proxers dependencies.
;; for any other occurence of proxy replace with proxer.

;; for refenvs want compile time total order not substance time.
;; 


;; reduce : a -> b, b -> c if b is kind to be reduce the result is a -> c.       
;; beware reduces only one level does not reduce refs between objects of reducekinds
;; also filters reduced from result graph.
		    


