%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
************************************************************************
************************************************************************
MATCHING AND SUBSTITUTION
************************************************************************
************************************************************************
]%

% 
tag indicated meta vars so that if lambda term is first-order substituted for 
var then computation will fully expand lambda term application.
Function is used to achieve so substitution in uses of nuprl rules.
%

let tag_all_so_var_terms vars_to_consider t 
  =
  let tag_so_var_term bvs t =
    ( let var,arity = var_and_arity_of_so_apply_of_var t in
        if member var vars_to_consider & not member var bvs 
        then 
          mk_tag_term (arity+1) t
        else 
          t 
    ) ? t 
  in
    sweep_up_map_with_bvars tag_so_var_term t 
;;

%
************************************************************************
Polarity Calculation
************************************************************************
Polarity calculations allow us to incorporate inequality reasoning into
matching. It is essential for level expression matching, and would 
come in useful when rewriting with respect to inequalities.

Would be nice, not to have to incorporate this semantic knowledge into
the matching procedure, but seems necessary.

Polarity of a term is denoted by an integer:

1. +1: +ve polarity
2. 0 :unknown polarity / both polarities.
3. -1: -ve polarity.

E.g. in the term.   (A=>B)=>C,  A and C have +ve polarities, B has -ve
polarity. 


Typical entry in parm_polarity alist:

  (opid,n),[p1;...;pn]

Says that term opid with n parameters, has parameter polarities p1... pn with
respect to term.

Typical entry in subterm_polarity alist:

  (opid,n),[p1;...;pn]

Says that term opid with n subterms, has subterm polarities p1... pn with
respect to term.

For efficiency, we only create cache entries for a term when the
polarity of at least one parameter/subterm is inverted.  (We assume
that such terms are not too common.)  This means that if entry for
term which inverts polarity is missing, the system will incorrectly
deduce polarity.  

Eventually, we should have system deduce the polarity info from
various kinds of monotonicity lemmas. 
%

letref parm_polarity_alist = [] : ((tok # int) # int list) list ;;
letref subterm_polarity_alist = [] : ((tok # int) # int list) list ;;

parm_polarity_alist := 
  [(`universe`,1),[1]
  ;(`prop`,1),[1]
  ]
;;




subterm_polarity_alist := 
  [(`function`,2),[-1;1]
  ;(`implies`,2),[-1;1]
  ;(`all`,2),[-1;1]
  ;(`rev_implies`,2),[1;-1]
  ;(`member`,2),[1;-1]
  ;(`equal`,3),[1;-1;-1]
% ;(`subtype`,2),[-1;1]
% ]
;;


%
Returns list with polarity for each parameter.
%

let get_parm_polarities (opid, parms:parm list) pol =
  let n = length parms in 
  if pol = 0 then
    replicate 0 n
  else
    map (\x.x*pol) (apply_alist parm_polarity_alist (opid,n))
    ?
    replicate pol n
;;


%
Returns list with polarity for each subterm.

Assumes polarity of subterms same unless otherwise stated.
Bad assumption??
%

let get_subterm_polarities t pol =
  let n = length (bterms_of_term t) in 
  if pol = 0 then
    replicate 0 n
  else
    map (\x.x*pol) (apply_alist subterm_polarity_alist (opid_of_term t,n))
    ?
    replicate pol n
;;




%
************************************************************************
Matching parameters
************************************************************************
%
% 
Check
1. opids match
2. types of parameters match.

return a parameter disagreement set of type (parm # parm) list such
that is p,p' is on the list, p and p' have the same parameter type, and
either 
1. p is a var nat or tok parameter variable
or 
2. p is a level expression parameter, and all level exp variables occurring 
   in p are meta.
   (we reject case when some variables in level expression are meta and
    some are not. This should never come up in practice.)
%

let mk_parm_dset meta_parm_vars (opid,parms) (opid',parms') polarity =
  if not opid = opid' then
    failwith `mk_parm_dset: opids disagree`
  if not length parms = length parms' then
    failwith `mk_parm_dset: different numbers of parameters`
  else 
    flatten 
    (map3
       (\p p' pol. 
          if not type_of_parm p = type_of_parm p' then
            failwith `mk_parm_dset: parameter types disagree`
          if (is_parm_variable p 
              & member (id_of_parm_variable p) meta_parm_vars
             )
             or
             (is_level_exp_parm p 
              & subset 
                  (vars_of_level_exp (dest_level_exp_parm p))
                  meta_parm_vars 
             )
             then
            [p, p',pol]
          if equal_parameters p p' then
            []
          else
            failwith `mk_parm_dset: parameter values disagree`
       )
       parms
       parms'
       (get_parm_polarities (opid,parms) polarity)
    )
;;

%
************************************************************************
Matching level expressions
************************************************************************
A level expression e is maintained in a normal form [v1,i1;...vn,in]
such that
  e =  mk_level_exp  [v1,i1;...vn,in]
%

let mle_mk_level_exp_binding (e:(tok # int) list) (u : tok ,j) =
  u
  ,
  mapfilter
    (\v,k.if k < j then fail else v,(k-j))
    e
;;

let mle_match (pat,inst) =
  map (mle_mk_level_exp_binding inst) pat
;;

%
if v is bound to both expression e1 and e2, mle_combine_bindings returns
the largest expression e compatible with both these bindings.

If e1 = [u1,i1;...;un,im]
   e2 = [v1,j1;...;vn,jn]
   e  = [w1,k1;...;wp,kp]

then wa occurs as ub and vc in e1 and e2 respectively, and ka = min(ib,jc) 
%

let mle_combine_bindings (e1:(tok # int) list) e2 =
  accumulate
    (\e (u,i).
       (u, min i (apply_alist e1 u)) . e
       ?
       e
    )
    []
    e2
;;

let mle_combine_substs s1 s2 =
  accumulate
  (\s (u,e_u).
     modify_or_add_alist_entry 
       (mle_combine_bindings e_u)
       e_u
       s
       u
  )
  s1
  s2
;;

let mle_find_best_dset_match le_dset =
  reduce
    mle_combine_substs
    []
    (map mle_match le_dset)
;;


%
************************************************************************
Matching parameter disagreement sets
************************************************************************
%

% 
Handles both standard parameter matching and level expression matching. 
Recognises polarities for level expression matching:
  +ve polarity : Desire Pat theta <= Inst
  -ve polarity : Desire Pat theta >= Inst
  no polarity  : Desire Pat theta = Inst

Here the <= relation is type inclusion. (A <= B if A is a subtype of B)
%

let match_parm_dset parm_dset =
  let level_exp_dset,dset' =
    divide_list (is_level_exp_parm o fst) parm_dset 
  in
  let std_sub = map (\p,p',pol.id_of_parm_variable p,p') dset' in
  let reduced_std_sub =
    (condense_alist_p $= equal_parameters std_sub ? failwith `match_parm_dset: inconsistent binding`)
  in
  let lower_le_dset = 
     mapfilter
       (\p,p',pol.if pol = (-1) then fail else
                    dest_level_exp (dest_level_exp_parm p)
                    , dest_level_exp (dest_level_exp_parm p')
       )
       level_exp_dset
  in
  let upper_le_dset =
     mapfilter
       (\p,p',pol.if pol = 1 then fail else
                    dest_level_exp (dest_level_exp_parm p)
                    , dest_level_exp (dest_level_exp_parm p')
       )
       level_exp_dset
  in
  let le_sub =
    level_exp_sub_to_parm_sub
    (map
       (id # mk_simple_level_exp)
       (lem_match_dset upper_le_dset lower_le_dset)
    )
  in
    le_sub @ reduced_std_sub
;;
    

%
************************************************************************
Matching Terms
************************************************************************
%

% 
Match pattern [px1;...;pxn].pt with instance [ix1;...;ixn].it ,

where pt = P[py1;...;pym] 

pys must be all bound in the pxs.
The pxs, pys, and ixs may all contain duplicates.

Returns a matching 

P,([z1;...;zm].it) where

zi is the variable in the ixs corresponding to yi in the pxs.

We check that there is a corresponding pyi for every free variable of
it bound in the ixs. 

Originally, the so_terms returned had completely new binding variables.
This is unnecessary. The original too, didn't allow for possible duplicated
binding variables.
%

let match_so_var_bterm (pxs,pt) (ixs,it) =
  let ix_px_prs = rev (zip ixs pxs) in
  let P, args = dest_so_var pt in
  let pys = map dest_var args in
  let fvs_of_it_bnd_in_ixs = intersection (free_vars it) ixs in
  let required_pys = map (translate_binding ix_px_prs) fvs_of_it_bnd_in_ixs in
  if not subset required_pys pys then 
    failwith `match_so_var_bterm: missing so var arg`
  else 
  let zs = (map (rev_apply_alist ix_px_prs) pys 
           ? failwith `match_so_var_bterm: an so var arg is unbound`)
  in
      P,(zs,it)
;;

  
  

% 
Perhaps (*) a full substitution whose first-order domain variables satisfy
is_fo_match_var and which yields instance when applied to pattern.  

(*: there must be an easily computed most general match; in particular, 
every 2nd-order var must have an instance involving only bound variables. )  
%


% 

A dset (disagreement set) is a list of triples, each of form

(xs,u),(ys,v),pol : bterm # bterm # int

A disagreement triple is matched by a substitution s if 

1. pol = 0:  s(xs,u) =alpha ys,v.
2. pol =+1:  s(xs,u) <=alpha ys,v.
3. pol =-1:  s(xs,u) >=alpha ys,v.

a <=alpha b iff a and b are alpha equal, except that when considering
level expressions, b is a larger type than a.
%

%
Checks for the following cases:

1. v is a meta var (perhaps second order) not bound in xs.  Returns `meta`
2. v is a fo var bound in xs. Alpha equiv check OK.         Returns `alpha-eq`
3. v is a fo var bound in xs. Alpha equiv check not OK.  Returns `not-alpha-eq`
4. otherwise.  Returns `otherwise`
%

let analyze_dpair (xs,v) (xs',v') meta_term_vars =

    if is_so_var_term v then
    (
      let var,n = var_and_arity_of_so_var_term v in

      if member var xs then       

        % var is bound in xs. Do alpha equiv check. %

        (if n = 0 & is_var_term v' & 
             dest_var v' = translate_binding (rev (zip xs xs')) var
         then
           `alpha-eq`
         else 
           `not-alpha-eq`
        )
      if member var meta_term_vars then
        `meta`
      else
        `otherwise`
    )
    else
      `otherwise`
;;

%
match_tops 
   alts  : term # term -> (term # term) list
   pol   : int
   mpvs  : tok list
   t     : term
   t'    : term
   = 
   parm_dset
   , bvar_sub
   , pols
   , bterms 
   , bterms' 
     : (parm # parm # int) list  
       # (var # var) list 
       # int list
       # bterm list 
       # bterm list

Matches operators. If operators match, we return 
1. the parameter disagreement set
2. bound variable substitution,
3. polarity of subterms,
4. boundterms of t and t'.

If initial match fails, applies alt(ernative)s function to pat and inst,
to get alternative pat inst pairs to try matching.

Possible alts:
  a) unfold any top soft abstractions 
  b) open up subtypes

%

let match_tops alts pol meta_parm_vs pat inst =
  let match (t,t') =
    let op,  bts  = dest_term t in
    let op', bts' = dest_term t' in
    let bvs = map fst bts in
    let bvs' = map fst bts' in
    let bv_sub = remove_if 
                  (\v,v'.v = null_var) 
                  (zip (flatten bvs) (flatten bvs'))
    in
    if map length bvs = map length bvs' then
      mk_parm_dset meta_parm_vs op op' pol
      ,bv_sub
      ,get_subterm_polarities t pol
      ,bts
      ,bts' 
    else
      failwith `match_tops: arity mismatch`
  in
  ( match (pat,inst)
    ?\x
    let p_i_prs = alts (pat,inst)
    in
    if null p_i_prs then
      failwith x
    else
      first_value match (alts (pat,inst))
  )
;;

%
simplify_dset reduces down a dset so that the 1st bterm in each triple
is of form xs,P[ys], where P[] is a so variable with 0 or more subterms
and P is a meta.

simplify_dset also generates a disagreement set for parameters, and 
a set of bound variable correspondences.

%




let simplify_dset alts meta_parm_vars meta_term_vars dset 
  =
  letrec simpl dset simp_dset parm_dset bvar_sub
    =
    if null dset then 
      simp_dset,parm_dset,bvar_sub 
    else
    let ((xs,v),(xs',v'),pol) . dset' = dset in
    let so_v = (so_ize_so_ap_of_var_under_bvars meta_term_vars xs v ? v) in

    let case = analyze_dpair (xs,so_v) (xs',v') meta_term_vars in
    if case = `meta` then
      simpl 
          dset' 
          ( ((xs,so_v), (xs',v'),pol) . simp_dset) 
          parm_dset
          bvar_sub
    if case = `alpha-eq` then
      simpl dset' simp_dset parm_dset bvar_sub
    if case = `not-alpha-eq` then
       failwith `simplify_dset: alpha eq test failed`
    else
    ( let parm_dset',bvar_sub',subt_pols,bts,bts' =
         match_tops alts pol meta_parm_vars v v' 
      in
        simpl (map3 (\(vs,u) (vs',u') pol'. (xs@vs,u), (xs'@vs',u'), pol') 
                    bts 
                    bts'
                    subt_pols
               @ dset')
              simp_dset
              (parm_dset' @ parm_dset)
              (bvar_sub' @ bvar_sub)
    )  
  in
    simpl dset [] [] []
;;

% 
match a term disagreement set 

  This proceeds in three stages:
  1. split dset into active and passive components.
  2. Generate bindings from active dset
  3. Verify bindings in passive dset.
%

let split_dset dset = 
  letrec aux a_dset a_vs p_dset in_dset = 
    if null in_dset then 
      a_dset,p_dset 
    else
    let in_dpair . in_dset' = in_dset 
    in let (xs,v),btm':bterm,i:int = in_dpair 
    in let so_var,so_var_args = dest_so_var v 
    in if 
         all (\t. is_var_term t & member (dv t) xs) so_var_args
         & not member so_var a_vs
    then 
       aux (in_dpair.a_dset) (so_var.a_vs) p_dset in_dset'
    else
       aux a_dset a_vs (in_dpair.p_dset) in_dset'
  in
    aux [] [] [] dset
;;

let match_active_dset dset = 
  map
    (\btm,btm',i:int. match_so_var_bterm btm btm')
    dset
;;

% returns true if OK %

let check_passive_dset sub dset = 
  all
    (\btm,btm',i:int.
       aequal_bterms (so_subst_in_bterm sub btm) btm'
    )
    dset
;;

let match_dset dset = 
  let active_dset,passive_dset = split_dset dset
  in let sub = match_active_dset active_dset
  in 
  if check_passive_dset sub passive_dset then
    sub
  else 
    failwith `match_dset: cannot find consistent match`
;;
let parm_to_str p = ((tok_to_string (type_of_parameter p)) J ":" J (string_of_parameter p));;
let print_parm_dset nmemonic dset =
 map (\p, q, i.
     tty_print (nmemonic  J (parm_to_str p) J " " J (parm_to_str q) J " " J (int_to_string i)))
     dset
;;

let half_match_with_retry
  alts polarity meta_parm_vars meta_term_vars pattern instance 
  =
  let dset,parm_dset,bvar_sub = 
         simplify_dset 
               alts               
               meta_parm_vars
               meta_term_vars
               [([],pattern), ([],instance), polarity] 

  in let so_sub = match_dset dset 
  in let so_sub' = map (id # bterm_to_so_lambda) so_sub
  in let bvar_sub' = map mk_bvar_binding bvar_sub
  in parm_dset,(bvar_sub' @ so_sub')
;;

let full_match_with_retry
  alts pol meta_parm_vars meta_term_vars pattern instance 
  =
  let parm_dset,term_sub =
    half_match_with_retry
    alts pol meta_parm_vars meta_term_vars pattern instance
  in
    parm_sub_to_term_sub (match_parm_dset parm_dset) @ term_sub
;;

let half_match = half_match_with_retry (\x.[]) ;;
let full_match = full_match_with_retry (\x.[]) ;;
let match = full_match 0 [] ;;



% Basic first order matching %

let fo_match vars pat inst =
  first_order_match pat inst vars 
;;


%
************************************************************************
Matching Summary
************************************************************************
The main functions defined above are:

full_match
  polarity  : int
  meta_parm_vars : tok list
  meta_term_vars : var list
  pat : term
  inst : term
  = 
  (var # term) list

match
  meta_term_vars : var list
  pat : term
  inst : term
  = 
  (var # term) list

fo_match
  meta_term_vars : var list
  pat : term
  inst : term
  = 
  (var # term) list

Each returns a substitution suitable for full_subst, subst, and fo_subst
respectively. In addition, we have:

semi_match 
  polarity  : int
  meta_parm_vars : tok list
  meta_term_vars : var list
  pat : term
  inst : term
  = 
  parm_dset,sub : (parm # parm # int) list # (var # term) list


match_parm_dset : (parm # parm # int) list -> (tok # parm) list
parm_sub_to_term_sub (tok # parm) list -> (var # term) list

The latter are used by the match_in_context routines, where
matching of level expression parameters is separated from the rest
of the matching procedure.
%

