%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*********************************************************************
*********************************************************************
ATOMIC-TACTICS.ML
*********************************************************************
*********************************************************************

Tactics in this file encode the primitive rules. One should rarely need
to access rules directly. (If one does repeatedly, add an entry to this file.)


General principles for variable naming and visibility of vars in hyp lists.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
N.B. Nuprl variables (ml objects of type `variable') must be explicitly 
injected into `term', the ml type of Nuprl terms.

As of May 5th `91 we have effectively 3 disjoint kinds of variables.

1. null variables.

    (Dummy variables) These are used as placeholders in binding positions of 
    Nuprl terms. They can not be injected into the `term' type and 
    consequently can never bind term occurrences of themselves. Null 
    variables can not be declared in hypothesis lists.

2. visible variables.

   These can be declared in hypotheses, used in binding positions of terms and
   injected into the term type. Their names are always visible in hypothesis
   lists.

3. invisible variables.

   These should only be used in declarations, although it is not an error
   to use them in other locations. Invisible variables are not
   visible in hypothesis lists, but will appear in extract terms. Tactics 
   which might introduce invisible variables into terms should rename them 
   to be visible. This restriction on the use of invisible variables allows 
   tactics to do quick declared variable occurrence checks by simply checking 
   the visibility of variables. NB. one can have visible variables declared 
   but not used in any term.
   

Currently there is one unique null variable. A variable is invisible iff its 
name starts with a percent character. By adopting the above scheme we avoid 
the need to choose/rename variables on term extraction.


Recommended practices:

1. If a declared variable is used anywhere, it must be visible. If it isn't
   used anywhere it may be visible or invisible.

2. If a declaration is considered a type then the declared variable should be
   visible. If the declaration is a proposition, the variable should be 
   invisible.

3. Automatically chosen new variable names, should be based on if not 
   identical to existing variable names.( e.g. on elim of concl function.) 
   An exception is if the existing var name is the null name or if there is no 
   appropriate existing variable. 

4. New variable names should be chosen so as not to force substitution
   renaming. (e.g. in '\x.yx' if y is renamed to x the bound x gets renamed
   by the substitution.) We want tactics, not substitution to choose new
   names. Often for simplicity, tactics can rename more often than would be
   strictly necessary.

5. The user should always have the option of explicitly picking new variable
   names for variables which might be visible, rather than having to go with
   the automatically generated names.

6. Invisible variable names should only be used in declarations. 

7. Tactics should never fail because an explicitly supplied variable name clashes
   with an existing one. Rather, a similar name should be used.


Dealing with abstractions.
~~~~~~~~~~~~~~~~~~~~~~~~~~
Rules for the basic propositional abstractions should be built in.
(and, or, not, implies, all, exists). (iff?). For others we invoke one
of a list of extension tactics.

Extension tactics will be added in when files which define them are loaded.
Ideally, the extension tactics will never need access to the variable
renaming mechanisms.

Levels of rule wrapping up.
~~~~~~~~~~~~~~~~~~~~~~~~~~~

Can we wrap up rules on a most basic level, such that we never have to delve
below this level? All renaming would take place below this level. If a user
wanted system supplied names, he/she say uses the null variable in place
of a real argument. If the user supplied names are not acceptable, variants
on them are used.


Design decisions in coding of primitive decomposition tactics.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

1. Choice of variable names
----------------------------
For visible variables: If user supplies name use it. If not, try to use 
existing variable in term. If none, (or is null) use some default name.
(Specified within each tactic.) If any name clashes with existing variable
name in hyp list, modify its integer suffix to be greater than any existing
suffix of the same root. Do not modify variable name if its use might
cause renaming of bound variables by the substitution function.

For invisible variables: Choose symbol with root the percent character and
with integer suffix greater than any existing suffix. An alternative would
be to keep a continually incremented integer so that suffices would always 
be distinct.

N.B. In cambridge ML, tokens are implemented as symbol names. Creating
new tokens involves interning new symbols, a rather time consuming operation.
(several millisec.) Even accessing token character strings is slow. If 
are trying to speed up rules, one should consider how tokens are being
used.

2. Free variable checks
------------------------
Tactics decide whether a variable is possibly used one or more times by
looking at its visibility.

3. Labelling of wf subgoals
-------------------------
Seems nice idea to label wf subgoals as such. Done occasionally at present.
Should be consistent...

Design of rules
~~~~~~~~~~~~~~~
1. Currently, on hyp elims, rules shove all new hyps to end of hyp list. Might
   be nicer to insert these new hyps directly after elimmed hyp.

2. Sometimes when hyp is elimed, we generate an extra equality goal and
   leave off doing possible substitions in other hypotheses. We could make
   default to do these substitutions, in which case we need never generate
   this extra goal.

]%
%[
**********************************************************************
Thinning hypotheses.
**********************************************************************
]%

let Thin i p =
  let i' = get_pos_hyp_num i p in
    if i=0 then
      failwith `Thin: cannot thin conclusion`
    else
      Refine `thin` [mk_int_arg i'] p ;;



let ThinBeyond i p = 
  let n = num_hyps p in
  let i' = get_pos_hyp_num i p in
  if n > i' then 
    OnHyps (rev (upto (i'+1) n)) Thin p
  else
    AddHiddenLabel `main` p
;;

%[
**********************************************************************
Alpha renaming of hypothesis declarations
**********************************************************************
]%

% rename variable declared in hypothesis list %

let RenameVar new_v i p =
  let i' = get_pos_hyp_num i p in
  let old_v = var_of_hyp i' p in
    Refine `rename` [mk_var_arg old_v;mk_var_arg new_v] p ;;


%[
**********************************************************************
WF Subgoal support.
**********************************************************************
For supporting tactics which generate wf subgoal of form:

>> T = T in Ui

Given 
 a) the term T
 b) the proof p supplying a binding env for vars in T, and poss explicit prop
    or universe arg.
 
generate:
 a) the level expression arg for the primitive rule.
 b) a tactic to pretty up the wf subgoal. Explicitly to
    1. fold member term.
    2. fold prop abstraction if appropriate. (or whatever else is used...)
    3. add `wf` label.

If user wants some other label, it can be simply supplied afterward.

For backward compatibility, set wf_pretty_up to false before expanding any
theorems.
]%

letref wf_pretty_up = true ;;

let mk_le_arg_and_wf_tac p T =
  let UTerm = get_universe p T
  in let opid,le = ti_dest_lp_term UTerm
  in let le' = simplify_level_exp le
  in 
    (mk_level_exp_arg le')
    ,
    (if not wf_pretty_up then
       AddHiddenLabel `wf`
     if opid = `universe` then
       FoldTop `member` 0 THEN AddHiddenLabel `wf`
     else
       FoldTop `member` 0 THEN FoldAtAddr opid [1] 0 THEN AddHiddenLabel `wf`
    )
;;

let WFPrettyUp p = 
  if wf_pretty_up then
  (FoldTop `member` 0 THEN AddHiddenLabel `wf`) p
  else
    AddHiddenLabel `wf` p
;;


%[
**********************************************************************
Conclusion Type Decomposition
**********************************************************************
These tactics eliminate the top level type constructor in the conclusion
type.

The naming of these tactics as `Intro' tactics is a historical anomaly.
]%

%
~~~~~~~~~~~~~~~~~~~~~~
Function decomposition
~~~~~~~~~~~~~~~~~~~~~~
%

%
The relevant rule is:


H1...Hn  >> x:A->B

       BY lambdaFormation i:int z:var

        z:A >> B[z/x] 
         >> A = A in Ui
%
%
FunCD handles variable name selection 
%

let FunCD z_in p =
  let x,A,B = dest_function (concl p) in
  let z_root = if not is_null_var z_in then z_in
               if not is_null_var x then x
               else basic_invisible_var 

  in let z = maybe_new_proof_var z_root p
  in let le_arg,WFTac = mk_le_arg_and_wf_tac p A
  in

  %mlbreak `lambdaFormation`;%
  ( Refine `lambdaFormation` [le_arg ; mk_var_arg z] 
    THENL
    [Id
    ;WFTac
    ]
  ) p
;;  

%
| The relevant rule is:
| 
| H1...Hn  >> {f | x:A->B}
| 
| BY rfunction_lambdaFormation i:int R:term g:var y:var z:var
|
| >> A in Ui
| >> R in A -> A -> Ui
| >> squash{WFounded{i}(A, R)}
| >> y:A, g:{f | x:{z:A| z R y} -> B} >> B[y,g/x,f]
|
| RFunCD handles variable name selection 
%
let RFunCD R g y z p =
  let f, x, A, B = dest_rfunction (concl p) in
  let Bvars = free_vars B in
  let g_root =
      if not is_null_var g then
	  g
      else if member f Bvars then
	  f
      else
	  basic_invisible_var
  in
  let y_root = if not is_null_var y then y else x in
  let z_root = if not is_null_var z then z else x in
  let (u.v.rest) = mk_new_var_args [[`u']; [`v']; [g_root]; [y_root]; [z_root]] p in
  let le_arg, WFTac = mk_le_arg_and_wf_tac p A
  in
  ( Refine `rfunction_lambdaFormation` (le_arg . u . v . mk_term_arg R . rest)
    THENL
    [WFTac;
     WFTac;
     AddHiddenLabel `aux`;
     Id
    ]
  ) p
;;  

%
| The relevant rule is:
| 
| H1...Hn  >> isect x:A.B
| 
| BY isect_memberFormation i:int z:var
| 
| z:A >> B[z/x] 
|     >> A = A in Ui
| ISectCD handles variable name selection 
%

let ISectCD z_in p =
  let x,A,B = dest_isect (concl p) in
  let z_root = if not is_null_var z_in then z_in
               if not is_null_var x then x
               else basic_invisible_var 

  in let z = maybe_new_proof_var z_root p
  in let le_arg,WFTac = mk_le_arg_and_wf_tac p A
  in
  ( Refine `isect_memberFormation` [le_arg ; mk_var_arg z] 
    THENL [Id; WFTac]
  ) p
;;  

%
~~~~~~~~~~~~~~~~~~~
Product decomposition
~~~~~~~~~~~~~~~~~~~

H1...Hn >> x:A#B

     By dependent_pairFormation i:int a:term y:var
        >> a = a in A
        >> B[a/x]
    y:A >> B[y/x] = B[y/x] in Ui


or

     By independent_pairFormation 
       
       >> A
       >> B
%
%
Here we just wrap up the dependent rule. The independent rule is invoked
by PrimCD directly.
%

let DepProdCD a y_in p =
  let x,A,B = dest_product (concl p) in
  let y_root = if not is_null_var y_in then y_in
               if not is_null_var x then x
               else basic_invisible_var

  in let y = get_distinct_var y_root p
  in let p' = extend_sequent p [y,A] 
  in let B' = (y = x) => B | subst [x,mvt y] B
  in let le_arg,WFTac = mk_le_arg_and_wf_tac p' B'
  in
     (Refine `dependent_pairFormation` 
        [le_arg
        ;mk_term_arg a
        ;mk_var_arg y
        ] 
      THENL
      [WFPrettyUp
      ;Id
      ;WFTac
      ]
     ) p
;;  



%
~~~~~~~~~~~~~~~~~~~
Set decomposition
~~~~~~~~~~~~~~~~~~~

H1...Hn >> {x:A|B}

     By dependent_set_memberFormation i:level a:term y:var
        >> a = a in A
        >> B[a/x]
    y:A >> B[y/x] = B[y/x] in Ui


or

     By independent_set_memberFormation
       
       >> A
       >> B
%
%
Here we just wrap up the dependent rule. The independent rule is invoked
by PrimCD directly.
%

let DepSetCD a y_in p =
  let x,A,B = dest_set (concl p) in
  let y_root = if not is_null_var y_in then y_in
               if not is_null_var x then x
               else basic_invisible_var

  in let y = get_distinct_var y_root p
  in let p' = extend_sequent p [y,A] 
  in let B' = (y = x) => B | subst [x,mvt y] B
  in let le_arg,WFTac = mk_le_arg_and_wf_tac p' B'
  in
     (Refine `dependent_set_memberFormation` 
        [le_arg
        ;mk_term_arg a
        ;mk_var_arg y
        ] 
      THENL
      [WFPrettyUp
      ;Id
      ;WFTac
      ]
     ) p
;;  


%
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Union Concl type decomposition
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%

let UnionCD n p = 
  let ld,rd = dest_union (concl p) in
  if n = 1 then
           (  let le_arg,WFTac = mk_le_arg_and_wf_tac p rd
              in
                Refine `inlFormation` [le_arg] 
                THENL [Id;WFTac]
           ) p
  if n = 2 then
           (  let le_arg,WFTac = mk_le_arg_and_wf_tac p ld
              in
                Refine `inrFormation` [le_arg] 
                THENL [Id;WFTac]
           ) p
  else
    failwith `UnionCD`
;;

%
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
General Concl type decomposition
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

The variable arguments of PrimCD are optional. If not supplied, 
the system try to supply suitable names.

The term argument is only needed for a dependent product elimination.

`wf` labels are applied to well formedness goals.
%

let get_optional_var_arg id p = 
  get_var_arg id p ? null_var
;;

let PrimCD p =

        let T = concl p  in
        let opid = opid_of_term T in
        
% two subterm cases %

        if opid = `function` then

           FunCD (get_optional_var_arg `v1` p) p

	if opid = `rfunction` then

	   RFunCD (get_term_arg `t1` p)
	          (get_optional_var_arg `v1` p)
	          (get_optional_var_arg `v2` p)
		  (get_optional_var_arg `v3` p)
		  p

	if opid = `isect` then

	  ISectCD (get_optional_var_arg `v1` p) p

        if opid = `product` then
        ( if fails_p (get_term_arg `t1`) p then
            (
                Refine `independent_pairFormation` [] p
              ?
                failwith `PrimCD: need witness term`
            )
          else
            DepProdCD (get_term_arg `t1` p) (get_optional_var_arg `v1` p) p
        )
        if opid = `quotient` then
           Refine `quotient_memberFormation` 
              [infer_level_exp_arg p (subterm_of_term T 1)] p

        if opid = `set` then
        ( if fails_p (get_term_arg `t1`) p then
            (
                Refine `independent_set_memberFormation` [] p
              ?
                failwith `PrimCD: need witness term`
            )
          else
            DepSetCD (get_term_arg `t1` p) (get_optional_var_arg `v1` p) p
        )

        if is_rec_term T then
           failwith `rec`

        if opid = `union` then
          UnionCD (get_int_arg `n` p) p

% one subterm cases %
      
        if opid = `list` then
           Refine `nilFormation` [infer_level_exp_arg p T] p

% zero subterm cases %

        if opid = `atom` then
           Refine `tokenFormation` [mk_tok_arg `apple`] p
      
        if opid = `U` then
           Refine `universeFormation` [] p
      
        if opid = `int` then
           Refine `natural_numberFormation`
	     [mk_term_arg (get_term_arg `t1` p ?
			     mk_natural_number_term 0)] p
      
        else failwith 
               `PrimCD: no simple conclusion decomposition rule exists.`

;;


%[
**********************************************************************
Hypothesis Type Decomposition.
**********************************************************************
These tactics decompose the top level type constructor in a hypothesis
type. All these tactics recognise negative hyp nums as indexing from end
of hyp list. i.e. The last hyp is hyp -1. last but one is -2 etc.

Almost all thin the original hypothesis. The exceptions are the function 
decomp tactics. They only thin if the declaration is for an invisible variable.

]%
%
~~~~~~~~~~~~~~~~~~~~~~~~~~~
1. Product type elimination.
~~~~~~~~~~~~~~~~~~~~~~~~~~~

the primitive rule is:


H, z:x:A # B, J >> C

  BY productElimination i u v 

  H, z:x:A # B, u:A, v:B[u/x], J[<u,v>/z] >> C[<u,v>/z]



Notes on operation:
~~~~~~~~~~~~~~~~~~~
User can supply u and v. 
If z is visible, we need to make sure that u and v are
visible too.
%

let ProdHD u v i p =
  
  let i' = get_pos_hyp_num i p in
  let z,T = dest_hyp i' p in
  let x,A,B = dest_product T in
  
  ( Refine `productElimination`  
      (mk_int_arg i'
      . mk_new_var_args [[u;x;z];[v;z]] p
      )
    THEN
    Thin i'
  ) p
;;


%
  Possible scenarios of use.
  1. Exists x:A. B
      a. desire u = x, or u based on x.
      b. user supplies u, system provides v.

  2. A and B
      User doesn't see any new variables.

  3. z:(A # B)  
      a. system generated u,v pair.
      b. user supplies u, v pair.

  4. z:(x:A # B[x])
      a. system generates u,v pair basing u on x.
      b. user supplies u, v pair.

%


%
~~~~~~~~~~~~~~~~~~~~~~~~~
Function type elimination.
~~~~~~~~~~~~~~~~~~~~~~~~~
The relevant rule is:
        
H, f:(x:A -> B), J >> C

  BY dependent_functionElimination i a y v

  H, f:(x:A -> B), J >> a = a in A
  H, f:(x:A -> B), J, y:B[a/x], v:y = f a in B[a/x] >> C



Operation of DepFunHD tactic
~~~~~~~~~~~~~~~~~~~~~~~~~~~

1. pick distinct new variables for y and v. Allow user to supply y.

2. do rule. (v should always be hidden.)

3. If f is not visible, thin the equality hyp in the second subgoal.

4. If f is not visible, allow option of thinning its declaration in both 
   subgoals.

5. Label the well formedness goal as such.

6. if a is a so_lambda term. Unfold it in wf goal.
%

let DepFunHDAux thin a y_in i p =

  let i' = get_pos_hyp_num i p in
  let f = var_of_hyp i' p in
  let f_is_visible = is_visible_var f in 
  let y_default = if f_is_visible then 
                    mkv `y` 
                  else 
                    basic_invisible_var 
  in
  let WFTac = is_term `so_lambda` a 
              =>
              Unfold `so_lambda` 0 THEN WFPrettyUp          
              |
              WFPrettyUp
  in
  let term = h i p in
  let name =
      if is_function_term term then
	  `dependent_functionElimination`
      else if is_rfunction_term term then
	  `rfunctionElimination`
      else if is_isect_term term then
	  `isectElimination`
      else
	  failwith `DepFunHDAux: hyp is not a function`
  in
  
  ( Refine name
      (mk_int_arg i'
       . mk_term_arg a
       . mk_new_var_args [[y_in;y_default];[]] p
      )
    THENL
    if f_is_visible then 
      [WFTac
      ;Id
      ]
    if thin then
      [Thin i' THEN WFTac
      ;OnHyps [-1;i'] Thin
      ]
    else
      [WFTac
      ;Thin (-1)
      ]
  ) p
;;

let DepFunHD = DepFunHDAux true ;;

%
operation of IndepFunHD tactic
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
The basic rule is:

H, f:(x:A -> B), J >> C

  BY independent_functionElimination i y 

  H, f:(x:A -> B), J >> A
  H, f:(x:A -> B), J, y:B >> C

In addition we:
1. pick invisible variable for y if user doesn't supply one.

2. Allow option of thinning x:A->B if f is invisible.
3. Label the first subgoal as `antecedent` .
%

let IndepFunHDAux thin y_in i p =

  let i' = get_pos_hyp_num i p in
  let f = var_of_hyp i' p in
  
  ( Refine `independent_functionElimination`  
      (mk_int_arg i'. mk_new_var_args [[y_in]] p)
    THENL
    if thin & is_invisible_var f then
      [Thin i' THEN AddHiddenLabel `antecedent`;Thin i']
    else
      [AddHiddenLabel `antecedent`;Id]
  ) p
;;

let IndepFunHD = IndepFunHDAux true ;;


%
  Possible scenarios of use:
  1. All x:A. B
    Always need dependent E. term a is user supplied. y,v are system generated,
    last hyp is thinned.

  2. A => B
    Always independent case. y system generated.

  3. f:A -> B
    Probably use dependent case. User provides term a. and probably var y.
    (Option of y system generated?)

  4. f:(x:A -> B[x])
    Definitely use dependent case. User provides term a. and probably var y.
    (Option of y system generated?)
%

%
~~~~~~~~~~~~~~~~~~~~~
3. Integer induction
~~~~~~~~~~~~~~~~~~~~~
        
H, x:Int, J >> C

  BY intElimination i z y v

  H, x:Int, J, y:Int, v:y < 0, z:C[y+1/x] >> C[y/x] 
  H, x:Int, J, >> C[0/x] 
  H, x:Int, J, y:Int, v:0 < y, z:C[y-1/x] >> C[y/x] 



Operation of IntInduction tactic
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

1. pick distinct new invisible variables for v and z. Allow user to supply y.
   (fancier versions of IntInduction will take care moving dependent hyps
    over to concl)
   
2. do rule. 
%

let IntInduction y_in i p =

  let i' = get_pos_hyp_num i p in
  
  (Refine `intElimination`  
         (mk_int_arg i'. mk_new_var_args [[];[y_in;mkv `i`];[]] p)
   THENL
   [AddHiddenLabel `downcase`
   ;AddHiddenLabel `basecase`
   ;AddHiddenLabel `upcase`
   ]
  ) p
;;


%
~~~~~~~~~~~~~~~~~
4. List induction
~~~~~~~~~~~~~~~~~
        
H, x:A list, J >> C

  BY listElimination i w u v

  H, x:A list, J >> C[nil/x] 
  H, x:A list, J, u:A, v:A list, w:C[v/x] >> C[u.v/x] 



Operation of IntInduction tactic
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

1. pick new invisible variable w. Allow user to supply u and v.

2. do rule. 
%

let ListInduction u_in v_in i p =

  let i' = get_pos_hyp_num i p in
  
  Refine `listElimination`  
         (mk_int_arg i'. mk_new_var_args [[];[u_in;mkv `u`];[v_in;mkv `v`]] p)
         p
;;

% as list induction, but thin induction hyp %

let ListD u_in v_in i p =

  let i' = get_pos_hyp_num i p in
  
  (Refine `listElimination`  
         (mk_int_arg i'. mk_new_var_args [[];[u_in;mkv `u`];[v_in;mkv `v`]] p)
   THENL [Id ;Thin (-1)]
  ) p
;;

% Some old theories require simplelistp to be false or the hypnums get changed.
%					  
letref simplelistp = true;;

letref or_simplelist_ref_state = or_ref_state `simplelist`;;

let simplelist_p () = 
 ref_state_get or_simplelist_ref_state (current_ref_environment_index `simplelist`)
;;

let simplelist_do_updates oid edges oids =
  or_simplelist_ref_state := ref_state_do_updates or_simplelist_ref_state oid oids edges
; ()
;;

let simplelist_add_data oid data = 
  or_simplelist_ref_state
    := ref_state_set_data or_simplelist_ref_state [oid,  data]
;;

let simplelist_add oid data =
 reset_ref_environment_data oid;
 add_ref_environment_data oid `simplelist` simplelist_add_data  data
;;

update_ref_state_view
 (\(). or_simplelist_ref_state)
 (ref_state_view_bool_entry)
;;     

update_ref_state_merge `simplelist`
 (\index edges. 
   or_simplelist_ref_state :=
       or_ref_state_merge `simplelist` or_simplelist_ref_state index edges
   ; true)
;;

let undeclare_or_simplelist index =
 (or_simplelist_ref_state := ref_state_remove or_simplelist_ref_state index; ())
 ? ()
;;
%-------%

let simpleListD u_in v_in i p =

  let i' = get_pos_hyp_num i p in
  
  (Refine `simpleListElimination`  
         (mk_int_arg i'. mk_new_var_args [[u_in;mkv `u`];[v_in;mkv `v`]] p)
  ) p
;;


%
~~~~~~~~~~~~~~~~~~~~
5. Union Elimination
~~~~~~~~~~~~~~~~~~~~
        
H, z:A | B, J >> C

  BY unionElimination i x y v

  H, z:A | B, x:A, J[inl x/z] >> C[inl x/z]
  H, z:A | B, y:B, J[inr y/z] >> C[inr y/z]



Operation of UnionHD tactic
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

1. Allow user to optionally supply x and y.
2. do rule. 
%

let UnionHD x_in y_in i p =

  % get info from sequent %

  let i' = get_pos_hyp_num i p in
  let z = var_of_hyp i p in

  let x_default,y_default =
    if is_visible_var z then  
      mkv `x` , mkv `y`
    else
      basic_invisible_var, basic_invisible_var
  in

  ( Refine `unionElimination`  
           (mk_int_arg i'
           . mk_new_var_args [[x_in;x_default];[y_in;y_default]] p
           )
    THEN
    Thin i'
  ) p
;;

%
~~~~~~~~~~~~~~~~~~~~~~~~~~~
6. Set type elimination.
~~~~~~~~~~~~~~~~~~~~~~~~~~~

the primitive rule is:

H, z:{x:A | B}, J >> C

  BY setElimination i u v 

  H, z:{x:A | B}, u:A, [v:B[u/x]], J[u/z] >> C[u/z]



Notes on operation:
~~~~~~~~~~~~~~~~~~~
The rule we really want is:

H, z:{x:A | B}, J >> C

  BY SetHD... 

  H, u':A, [v:B[z/x]], J >> C

where u' = 
    1) v1 tactic arg if supplied
 or 2) if z visible then z
 or 3) if x visible then something based on x
 or 4) invisible var.

Case 3) comes up e.g. if the set type is hidden in a squash-exists term.
 
To achieve this, we ensure that the u introduced at the primitive rule
level is invisible. (guarantees no renaming of bound vars.) and then we
rename u appropriately.

It seems over general to have an option for specifying v. After all,
B will always be a proposition.
%

let BasicSetHD i p =
  
  let i' = get_pos_hyp_num i p in
  let z,T = dest_hyp i' p in
  let x,A,B = dest_set T in
  
  let u' = get_var_arg `v1` p ? z in
                
  ( Refine `setElimination`  
      (mk_int_arg i'
      . mk_new_var_args [[];[]] p
      )
    THEN
    Thin i'
    THEN 
    \p'.(if is_visible_var u' then 
           RenameVar u' i' 
         if is_visible_var x then
           RenameVar (maybe_new_proof_var x p') i'
         else 
        Id) p'
  ) p
;;

let BasicAbSetHD i = Repeat (UnfoldTopAb i) THEN BasicSetHD i
;;


%
~~~~~~~~~~~~~~~~~~~~~~~~~~~
7. Quotient Type elimination.
~~~~~~~~~~~~~~~~~~~~~~~~~~~
 The rule is:

H,#j: u: x,y:A//E(x,y), J >> s(u) = t(u) in T(u)
  
  BY quotientElimination j $i v w z

  ...   ... v:A, w:A >> E(v,w) = E(v,w) in U$i
  ...   ... >> T(u) = T(u) in U$i
  ...   ... v:A, w:A, z:E(v,w) >> s(v) = s(w) in T(v)

Rule should be really changed to allow one to supply separate universe
level expressions for T and E.
%
let QuotientHD vars i p =

  let conclT,(),() = dest_equal (concl p) in  
  let i' = get_pos_hyp_num i p in
  let u,T = dest_hyp i' p in
  let x,y,A,Exy = dest_quotient T in
  let [v;w;z] = new_var_set 
                 [[first vars ; u]
                 ;[second vars ; u]
                 ;[]
                 ]
                 (declared_vars p)
  in
  let p' = extend_sequent p [v,A;w,A] in
  let Evw = subst [x,mk_var_term v;y,mk_var_term w] Exy in
                
  ( Refine `quotientElimination` 
      (mk_int_arg i'
      . infer_level_exp_arg p' (mk_and_term conclT Exy)
      . map mk_var_arg [v;w;z]
      )
    THENL
    [WFPrettyUp
    ;WFPrettyUp
    ;Id
    ]
  ) p
;;


%
~~~~~~~~~~~~~~~~~~~~~~~~~
General Primitive term HD
~~~~~~~~~~~~~~~~~~~~~~~~~

PrimHD takes optional variable and term arguments, which are used as
necessary.
%

let PrimHD i p =

  let i' = get_pos_hyp_num i p in
  let T = h i' p in
  let opid = opid_of_term T in

  % two subterm cases %

  if opid = `union` then

    UnionHD (get_optional_var_arg `v1` p) (get_optional_var_arg `v2` p) i' p
    
  if opid = `product` then

    ProdHD (get_optional_var_arg `v1` p) (get_optional_var_arg `v2` p) i' p

  if opid = `function` then
  (if fails_p (get_term_arg `t1`) p then
     IndepFunHD (get_optional_var_arg `v1` p) i' p
   else
     DepFunHD (get_term_arg `t1` p) (get_optional_var_arg `v1` p) i p
  )

  if opid = `rfunction` or opid = `isect` then

    DepFunHD (get_term_arg `t1` p) (get_optional_var_arg `v1` p) i p

  if opid = `set` then

    BasicSetHD i' p

  if opid = `quotient` then

    QuotientHD [get_optional_var_arg `v1` p
               ;get_optional_var_arg `v2` p
               ] i' p

  if opid = `void` then

    Refine `voidElimination` [mk_int_arg i'] p

  if opid = `int` then
  
    IntInduction (get_optional_var_arg `v1` p) i' p

  if opid = `list` then
    ((if (simplelist_p ())  then simpleListD else ListD)
       (get_optional_var_arg `v1` p) (get_optional_var_arg `v2` p) i' p)

  if opid = `equal` then

    Refine `equalityElimination` [mk_int_arg i'] p

  if opid = `sqequal` then

    Refine `sqequalElimination` [mk_int_arg i'] p

  else failwith `PrimHD: no case applies`
;;



let PrimInduction i p =

  let i' = get_pos_hyp_num i p in
  let T = h i' p in
  let opid = opid_of_term T in

  if opid = `int` then
  
    IntInduction (get_optional_var_arg `v1` p) i' p

  % one subterm cases %

  if opid = `list` then
  
    ListInduction (get_optional_var_arg `v1` p) (get_optional_var_arg `v2` p) i' p
  
  else
    failwith `PrimInduction: no case applies`
;;

%[
**********************************************************************
Clause Type Elimination.
**********************************************************************
]%


let PrimD i p =
  if i = 0 then 
    PrimCD p
  else 
    PrimHD i p
;;

% 
id's in D_additions are there to make it easy to check contents
of D_additions
%%
letref D_additions 
  = [] :(tok # (int -> tactic)) list ;;
%% 
new items added to END of list, so items added first will be 
tried first
%%
let update_D_additions id T = 
  D_additions := update_alist D_additions id T
  ;
  ()
;;
%%
let insert_D_additions id T = 
  D_additions := update_insert_alist D_additions id T
  ;
  ()
;;
%
%
  1. Try D_additions
  2. if ith clause is abstraction, unfold 1 step and go back to 1.
  3. Run PrimD
%

let D i p =
  letrec T p =
    first_value
      (\id,T.T i p)
      (lookup_D_additions ())
    ?
    ( If (\p'.is_ab_term (clause_type i p')) 
         (UnfoldTopAb i THEN T)
         (PrimD i) 
    ) p
  in T p
;;

% For backward compatibility %
  
let DTerm t i = With t (D i) ;;
let DVars vs i = New vs (D i) ;;

% Don't use Sel here, since that will be caught by DisjunctCD D_addition
defined in basic-tactics.ml.
%
let DNth n i = 
  if i = 0 then 
     Repeat (UnfoldTopAb 0) THEN UnionCD n 
  else 
    failwith `DNth`
;;




%[
**********************************************************************
Tactics invoking decision procedure rules.
**********************************************************************
We need to make sure that relevant primitive term constructors are exposed

See arith.ml for details on arith routines.


]%

let PrimEq = Refine `equality` [] ;;

let Eq p=
  if is_member_or_equal_term (concl p) then
  ( TryOnAllClauses (UnfoldTop `member`)
    THEN PrimEq
  ) p
  else 
    failwith `Eq: Conclusion not suitable for equality reasoning`
;;


%[
**********************************************************************
Trivial atomic tactics.
**********************************************************************
]%

let NthHyp i p =
  let i' = get_pos_hyp_num i p in 
  Refine `hypothesis` [mk_int_arg i'] p
;;
let Hypothesis = OnSomeHyp NthHyp ;;

let NthDecl i p =
  let i' = get_pos_hyp_num i p in 
  let (),t,() = dest_member_or_equal (concl p) in
  if var_of_hyp i' p = dest_var t then
  ( Try (OnConcl (UnfoldTop `member`)) 
    THEN Refine `hypothesisEquality` [mk_int_arg i'] 
  ) p
  else failwith `NthDecl`
;;


let Declaration p =
 (let (),t,() = dest_member_or_equal (concl p) in
  let decl_num = get_decl_num (dest_var t) p 
  in
    NthDecl decl_num p
 ) 
 ? failwith `Declaration` 
;;



let SoftNthHyp i p =
 let a = concl p in
 let b = h i p in
 let case = soft_equal a b in

 if case = `true` then
   NthHyp i p
 if case = `top-soft` then
 ( Repeat (UnfoldSoftAb 0) 
   THEN Repeat (UnfoldSoftAb i)
   THEN NthHyp i 
 ) p
 if case = `soft` then
 ( UnfoldAllSoftAbs 0 
   THEN UnfoldAllSoftAbs i
   THEN NthHyp i 
 ) p
 else
   failwith `SoftNthHyp`
;;

let SoftNthDecl i p =
 ( let S,a,b = dest_member_or_equal (concl p) in
   let v,T = dest_hyp i p in
   if not (v = dest_var a) & (v = dest_var b) then
     fail
   else
   let case = soft_equal S T in

   if case = `true` then
     NthDecl i p
   if case = `top-soft` then
   ( Repeat (UnfoldSoftAbAtAddr [1] 0) 
     THEN Repeat (UnfoldSoftAb i)
     THEN NthDecl i 
   ) p
   if case = `soft` then
   ( Repeat (UnfoldSoftAbsBelowAddr [1] 0)
     THEN UnfoldAllSoftAbs i
     THEN NthDecl i 
   ) p
   else
     fail
 )
 ?
 failwith `SoftNthDecl`
;;




%
If there are hyps H1 ... Hn The cut rule takes an argument i and inserts the
new hyp after the ith hyp. (if i = 0 the new hyp goes before H1.)

The Assert*AtHyp i t asserts t BEFORE the current ith hyp, so t becomes the new
ith hyp. If i LE 0 then i indexes the hyp position from the list end. For 
convenience we allow i=0 to indicate after the last hyp.
%

let AssertDeclAtHyp i v t p =
  let i' = if i > 0 then i - 1 else (num_hyps p + i) 
  in
  ( Refine `cut` [mk_int_arg i'; mk_term_arg t; mk_var_arg v] 
    THENL
    [AddHiddenLabel `assertion`
    ;Id
    ]
  )p
;;

let AssertDecl v t = AssertDeclAtHyp 0 v t
;;

let AssertAtHyp i prop p =
  let v = new_invisible_var p
  in
    AssertDeclAtHyp i v prop p
;;

let Assert prop = AssertAtHyp 0 prop 
;;

%
... |- C[t]

By BasicSubst t = t' in T   z.C[z]

`equality` ... |- t = t' in T
`main`     ... |- C[t']
`wf`       ... z':T |- C[z'] in U{?}

? is inferred type of C[z'] or value of `universe` optional parameter.
%

let BasicSubst t_eq_t'_in_T  z Cz p =
  let z' = maybe_new_proof_var z p
  in let Cz' = z = z' => Cz | subst [z,mvt z'] Cz
  in let p' = 
    copy_arg_annotation p
      (extend_sequent p [z', eq_type t_eq_t'_in_T])

  in let le_arg,WFTac = mk_le_arg_and_wf_tac p' Cz'
  in
 (Refine `substitution` 
         [le_arg
         ;mk_term_arg t_eq_t'_in_T
         ;mk_bterm_arg [z'] Cz']
  THENL
  [AddHiddenLabel `equality`
  ;Id
  ;WFTac
  ]
 ) p
;;



let Lemma name = Refine `lemma` [mk_tok_arg name ]
;;

let Lemma_o obid = Refine `lemma_by_obid` [mk_obid_arg obid ]
;;

%
AssertLemma takes as second arg a substitution list of type 
(var # term) list which is used to instantiate the level variables in 
the lemma.

In the event that the bindings are for null variables, we
we infer their names from the theorem.

%


let AssertLemmaWithSub obid sub p = 
 let lem_tm = statement_lookup obid in 
 let sub' =
 ( if exists (\v,t.v = null_var) sub then
     (zip
        (map tok_to_var (level_vars lem_tm))
        (map snd sub)
     ? failwith `AssertLemma: wrong number of level expressions`
     )
   else
     sub
 ) in
 let inst_lem_tm = full_subst sub' lem_tm 
 in
 ( Assert inst_lem_tm
   THENL
   [OnAllHyps Thin
    THEN Refine `instantiate` 
          [mk_assumption_list_arg []
          ;mk_term_arg lem_tm
          ;mk_parm_sub_arg (term_sub_to_parm_sub sub') 
          ]
    THEN Lemma_o obid
   ;Id
   ]
  ) p
;;

let AssertLemmaWithSub_t name =  AssertLemmaWithSub (lemma_lookup name);;

let AssertLemma name parm_terms =
  AssertLemmaWithSub_t name (map (\t.null_var,t) parm_terms)
;;
let AssertLemma_o obid parm_terms =
  AssertLemmaWithSub obid (map (\t.null_var,t) parm_terms)
;;


%
AssertTermWithSub is similar to AssertLemmaWithSub, but instead of
a lemma name, we provide a term and a tactic that proves it.
This is used when we want to create a proof of a lemma `inline`.
%


let AssertTermWithSub lem_tm tac sub p = 
 let sub' =
 ( if exists (\v,t.v = null_var) sub then
     (zip
        (map tok_to_var (level_vars lem_tm))
        (map snd sub)
     ? failwith `AssertTermWithSub: wrong number of level expressions`
     )
   else
     sub
 ) in
 let inst_lem_tm = full_subst sub' lem_tm 
 in
 ( Assert inst_lem_tm
   THENL
   [OnAllHyps Thin
    THEN Refine `instantiate` 
          [mk_assumption_list_arg []
          ;mk_term_arg lem_tm
          ;mk_parm_sub_arg (term_sub_to_parm_sub sub') 
          ]
    THEN tac
   ;Id
   ]
  ) p
;;

let TrueCD p =
  (OnConcl (UnfoldTop `true`)
   THEN UnfoldTopAb 0
   THEN Refine `natural_numberEquality` [] 
  ) p
;;

let VoidHD i p = 
  let i' = get_pos_hyp_num i p in
    Refine `voidElimination` [mk_int_arg i']  p
;;

let FalseHD i p =
  let i' = get_pos_hyp_num i p in
  (UnfoldTop `false` i'
   THEN VoidHD i'
  ) p
;;

let LogicAxiom p =
 (TrueCD
  ORELSE
  OnSomeHyp FalseHD
 ) p
;;

%
letref Trivial_additions = [`LogicAxiom`, LogicAxiom
			    ;`OnSomeHyp:VoidHD`, OnSomeHyp VoidHD
			   ];;

let update_Trivial_additions id T = 
 Trivial_additions :=  update_alist Trivial_additions id T; ();;
%
let Trivial p =
  ( OnSomeHyp SoftNthDecl
    ORELSE Eq
    ORELSE First (map snd (lookup_Trivial_additions()))
    ORELSE OnSomeHyp SoftNthHyp
  ) p
;;

%[
**********************************************************************
Function Extensionality
**********************************************************************
See equality-tactics.ml
]%

%[
**********************************************************************
Explicit Intro
**********************************************************************
]%

let UseEqWitness t =
   Refine `introduction` [mk_term_arg t] 
;;

let UseWitness t = UseEqWitness t THEN FoldTop `member` 0
;;

% 
... a = a in T ... >> ...
BY GeneralizeEqWitness i
  ... T ... >> ...

%

let GeneralizeEqWitness i p = 
  let i' = get_pos_hyp_num i p 
  in let T,a,() = dest_equal (h i p)
  in 
  ( AssertAtHyp (i'+1) T
    THEN IfLabL
    [`main`,Thin i'
    ;`assertion`,UseEqWitness a THEN NthHyp i'
    ]
  ) p
;;

let GeneralizeWitness i = 
  UnfoldTop `member` i THEN GeneralizeEqWitness i
;;

%[
**********************************************************************
The Satanic rule!
**********************************************************************
]%

let Fiat p = Refine `because` [] p
;;

%[
**********************************************************************
Special purpose Decomp tactics
**********************************************************************
]%

% Handle not in hyps properly. %

let NotD i p = 
 (if i=0 then fail
  if is_term `not` (clause_type i p) then
  (  UnfoldTop `not` i
     THEN UnfoldTop `implies` i
     THEN (IndepFunHD null_var i 
           THEN IfLab `aux` (AddHiddenLabel `main`) (FalseHD (-1)))
  
  ) p
  else fail
 ) ? failwith `NotD`
;;

let CAndD i p =
  if not i = 0 then failwith `CandD`
  else
  let [A;B] = dest_simple_term_with_opid `cand` (concl p)
  in
  ( Assert A
    THEN IfLab `main` 
    ( UnfoldTop `cand` 0
      THEN Refine `independent_pairFormation` []
      THENL [NthHyp (-1);Id]
    )
    (AddHiddenLabel `main`)
  ) p
;;

let CAndCD p = 
 (UnfoldTop `and` 0 THEN FoldTop `cand` 0 THEN CAndD 0) p
;;


%[
**********************************************************************
Trivial additions
**********************************************************************
]%

let Contradiction p =
  let hyp_types = map type_of_declaration (hyps p) in
  let hyp_type_and_num_prs =
    zip hyp_types (upto 1 (length hyp_types)) in
  let neg_hyps = 
    filter
      (\t,i.is_not_term t)
      hyp_type_and_num_prs 
  in
  let pos_hyp_num,neg_hyp_num =
  ( first_value
      (\t,i.
         let negated_term = dest_not t in
         let pos_hyp_num =
           first_value
             (\t,i.if (alpha_equal_terms t negated_term) then i	
	   else fail)
             hyp_type_and_num_prs
         in
           pos_hyp_num,i
       )
      neg_hyps
  ?
    failwith `Contradiction`
  )
  in
  if neg_hyp_num > pos_hyp_num then
    (NotD neg_hyp_num THEN NthHyp pos_hyp_num) p
  else
    (NotD neg_hyp_num THEN NthHyp (pos_hyp_num - 1)) p
;;

%
update_Trivial_additions `Contradiction` Contradiction
;;%


%[
**********************************************************************
Set and Squash related tactics.
**********************************************************************
]%

let BasicSquashHD i p =
 (let i' = get_pos_hyp_num i p in
  % H, (z:)Sq{P}, H' >> C %

  OnHyp i' (UnfoldTop `squash`)
  THEN
  BasicSetHD i'

  % H, (z:)True, [P], H' >> C %

  THEN 
  Thin i'
  
  % H, [P], H' >> C  %
 ) p
;;

let SetEqTypeCD p =
  let T,(),() = dest_equal (concl p) in
  let x,A,P_of_x = dest_set T in
  if is_null_var x then
   
  Refine `independent_set_memberEquality` [] p

  else
  let x' = get_distinct_var x p in

  % >> a = b in {x:A| P[x]} %

  ( Refine `dependent_set_memberEquality`
           [infer_level_exp_arg 
              (extend_sequent p [x',A]) 
              (x = x' => P_of_x | subst [x,mvt x'] P_of_x)
           ;mk_var_arg x'
           ]
    %
    >> a = b in A
    >> P[a]
    x':A >> P[x'] = P[x'] in U{i}
    %

    THENL
    [Id;AddHiddenLabel `set predicate`;AddHiddenLabel `wf`]
  ) p
;;

let AbSetEqTypeCD p =
  ( Repeat (UnfoldAtAddr [1] 0)
    THEN SetEqTypeCD
  ) p
;;

let AbSetMemEqTypeCD p =
  EqToMemberEq (\i.AbSetEqTypeCD) 0 p
;;

% should this give >> Sq(P) rather than >> P ?? %

let SquashEqTypeCD p =
  
  % H >> axiom = axiom in Sq{P} %

 (OnConcl (UnfoldAtAddr [1])
  
  % H >> axiom = axiom in {True|P} %
  THEN

  SetEqTypeCD

  THENL
  [% axiom = axiom in True - should deal with this more abstractly %

   OnConcl (UnfoldAtAddr [1])
   THEN OnConcl (UnfoldAtAddr [1])
   THEN Refine `axiomEquality` []  
   THEN Refine `natural_numberEquality` [] 
  ;
   % H >> P %
   Id
  ]
 ) p
;;

let SquashEqTypeHD i p =

  %  ... #i: axiom = axiom in Sq(P), ... >> C %

  let i' = get_pos_hyp_num i p in
  ( AssertAtHyp i' (eq_type (h i' p))
    THEN IfLabL
    [`assertion`,
     UseEqWitness axiom_term THEN NthHyp i'
    ;`main`,
     Thin (i'+1)
     %  ... #i: Sq(P), ... >> C %
    ]
  ) p
  ? failwith `SquashEqTypeHD`
;;



let SquashCD p =
( UnfoldTop `squash` 0 
  THEN Refine `independent_set_memberFormation` []
  THENL
  [TrueCD;Id]
) p
;;

let UnhideSinceSquashedConcl p =
  let c = concl p in
  if not is_squash_term c then failwith `UnhideSinceSquashedConcl`
  else
  % >> sq{P} %
  ( UseEqWitness axiom_term 

    % hidden hyps get unhidden at this stage...%
    % >> axiom = axiom in sq{P} %

    THEN Assert (concl p)
    THENL 
    [
     % >> sq{P} %
     AddHiddenLabel `main`
    ;
     % sq{P} >> axiom = axiom in sq{P} %
     BasicSquashHD (-1)
     % P >> axiom = axiom in sq{P} %
     THEN SquashEqTypeCD 
     %P >> P %
     THEN Hypothesis
    ]
  ) p
;;

%
Handles all basic cases. Used in proving sq_stable lemmas.
Use in general the Unhide or UnhideSinceSqStableConcl in general-tactics.ml.

Add here handling of arith relations and nequal??

%


let UnhideSinceCompTrivialConcl p =
  let c = concl p in
  let opid = opid_of_term c in
  if opid = `not` then
  ( D 0
    THENM
   (Unfold `false` 0 
    THEN Assert c
    THENL
    [Thin (-1);Contradiction]
   )
  ) p   
  if opid = `false` or opid = `member` then
  ( Unfold opid 0 THEN Fold opid 0 ) p
  if opid = `squash` then
    UnhideSinceSquashedConcl p
  else
    failwith `UnhideSinceCompTrivialConcl: no progress`
;;

% 
Doesn't actually change concl, but ensures that system recognizes that concl is
computationally trivial.

These functions supercede the UnhideSinceCompTrivialConcl
%

let TrivializeNotConcl p =
  let c = concl p in
  if is_term `not` c then
  ( D 0
    THENM
    ( Unfold `false` 0 
      THEN Assert c
      THEN IfLabL [`assertion`,Thin (-1);`main`,Contradiction]
    )
  ) p
  else 
    failwith `TrivializeNotConcl`
;;

   
let TrivializeHardConcl p =
  let c = concl p in
  let opid = opid_of_term c in
  if member opid ``void equal sqequal less_than`` then
    AddHiddenLabel `main` p
  if opid = `false` then
  ( UnfoldTopAb 0 THEN Fold `false` 0) p
  if opid = `not` then
    TrivializeNotConcl p
  if member opid ``le nequal`` then
  ( Unfold opid 0 
    THEN TrivializeNotConcl
    THEN Fold opid 0
  ) p
  if opid = `squash` then
    UnhideSinceSquashedConcl p
  else
    failwith `TrivializeHardConcl: concl not obviously computationally trivial`
;;
  
let TrivializeConcl p =
  ( UnfoldTopSoftAbs 0 
    THEN TrivializeHardConcl 
    THENM
     (Assert (concl p)
      THEN IfLab `main` (SoftNthHyp (-1)) (AddHiddenLabel `main`)
     )
  ) p
;;


%[
**********************************************************************
Quotient Type Tactics
**********************************************************************
]%


let analyze_ab_quot_type T = 
  if is_term `quotient` T then `quotient` else
  let T' = unfold_ab T
  in
  if is_term `quotient` T' then opid_of_term T else
    failwith `analyze_ab_quot_type: not quotient type or ab quotient type`
;;

%
Rule:
H >> s = t in x,y:A//E(x,y)

  BY quotient_memberEquality $i

  H >> x,y:A//E(x,y) = x,y:A//E(x,y) in U$i
  H >> s = s in A
  H >> t = t in A
  H >> E(s,t)

%
let QuotEqTypeCD p =
 (Refine `quotient_memberEquality`
         [infer_level_exp_arg p (eq_type (concl p))]
  THENL
  [WFPrettyUp
  ;WFPrettyUp
  ;WFPrettyUp
  ;Id
  ]
 ) p
;;

let AbQuotEqTypeCD p = 
  let T,(),() = dest_equal (concl p) in
  let quotid = analyze_ab_quot_type T in
  if quotid = `quotient` then
    QuotEqTypeCD p
  else
  ( UnfoldAtAddr [1] 0
    THEN QuotEqTypeCD
    THENL [KeepingAnnotation (FoldAtAddr quotid [2] 0);Id;Id;Id]
  ) p
;;
 
  
%
Rule:
H >> s = t in x,y:A//E(x,y)

  BY quotient_memberWeakEquality $i

  H >> x,y:A//E(x,y) = x,y:A//E(x,y) in U$i
  H >> s = t in A
%

let WeakQuotEqTypeCD p =
 (Refine `quotient_memberWeakEquality`
         [infer_level_exp_arg p (eq_type (concl p))]
  THENL
  [WFPrettyUp
  ;Id
  ]
 ) p
;;
  
let WeakAbQuotEqTypeCD p = 
  let T,(),() = dest_equal (concl p) in
  let quotid = analyze_ab_quot_type T in
  if quotid = `quotient` then
    WeakQuotEqTypeCD p
  else
  ( UnfoldAtAddr [1] 0
    THEN WeakQuotEqTypeCD
    THENL [KeepingAnnotation (FoldAtAddr quotid [2] 0);Id]
  ) p
;;
 

%
Rule:
H >> x,y:A//E(x,y) = u,v:B//F(u,v) in Ui

  BY quotientEquality r s v

  H >> x,y:A//E(x,y) in Ui
  H >> u,v:B//F(u,v) in Ui
  H >> A = B in Ui
  H, v:A = B in Ui, r:A, s:A >> E(r,s) <=> F(r,s)
%

let QuotEqCD p = 
  let x,y,(),() = dest_quotient (fst (equands (concl p)))
  in
  (Refine
     `quotientEquality`
     (mk_new_var_args [[x];[y];[]] p)
   THENL
   [AddHiddenLabel `wf`
   ;AddHiddenLabel `wf`
   ;AddHiddenLabelAndNumber `subterm` 1
   ;AddHiddenLabelAndNumber `subterm` 2
   ]
  ) p
;;

%
Rule:

H, a = b in x,y:S//E, J |- T ext t

BY quotient_equalityElimination (i:int) (j:level exp) (v : var arg)

  H, a = b in x,y:S//E,v:[ E[a,b/x,y]], J |- T ext t
  H, a = b in x,y:S//E, J |- E[a,b/x,y] = E[a,b/x,y] in Uj ext Ax
%

let BasicQuotEqTypeHD i p = 
  let i' = get_pos_hyp_num i p
  in let Q,a,b = dest_equal (h i' p)
  in let x,y,S,Exy = dest_quotient Q
  in let [v] = get_distinct_var_set [[]] p
  in let Eab = subst [x,a;y,b] Exy
  in 

 (( Refine `quotient_equalityElimination`
           [mk_int_arg i
           ;infer_level_exp_arg p Eab
           ;mk_var_arg v
           ]
  ) THENL
    [Try (Thin i') 
    ;Try (Thin i') THEN WFPrettyUp
    ]
  ) p
;;


let AbQuotEqTypeHD i p = 
  let i' = get_pos_hyp_num i p in
  let T,(),() = dest_equal (h i' p) in
  let quotid = analyze_ab_quot_type T in
  let PreTac = quotid = `quotient` => Id | UnfoldAtAddr [1] i' in

  ( PreTac
    THEN BasicQuotEqTypeHD i' 
    THENM Try TrivializeConcl
  ) p
;;



let AbQuotDAux i p =
  if i = 0 then failwith `AbQuotD` else
  let i' = get_pos_hyp_num i p in
  let quotid = analyze_ab_quot_type (h i' p) in
  let vs = [get_optional_var_arg `v1` p; get_optional_var_arg `v2` p] in
  let ThinTac = KeepingAnnotation (Try (Thin i')) in
  let AbThinTac p = KeepingAnnotation (Thin i' ORELSE FoldTop quotid i') p in
  if quotid = `quotient` then
   (QuotientHD vs i' THEN ThinTac) p
  else
   ( UnfoldTopAb i'
     THEN QuotientHD vs i'
     THEN AbThinTac
   ) p
;;

% 
Expand options for concl to member and squash terms.
%

let AbQuotD i p =
  let c = concl p in 
  if is_term `equal` c then
     AbQuotDAux i p 
  if is_term `member` c then
    (UnfoldTop `member` 0 
     THENM AbQuotDAux i) p
  if is_term `squash` c then
    (UseEqWitness axiom_term
     THENM AbQuotDAux i
     THENM SquashEqTypeCD
    ) p
  else
    failwith `AbQuotD: concl must be equal, member or squash`
;;


%[
**********************************************************************
Subtype Tactics
**********************************************************************
Tactics for subtype relations. See rules_2 for definition of rules
called here.

EqCD tactics here are not integrated directly into Main EqCD tactic.
Rather they are used to prove wf lemmas. The motivation for this
is 
1. get_type can use the wf lemma for type inference.
2. it's one small step towards getting Nuprl type-theory details out the 
   tactics.
]%

%[
Support for `basic subtype' relation.
NB: widening rule not valid for this relation.
]%

let SubtypeEqCD p = 
  (Refine `subtypeEquality` []
   THENL 
   [AddHiddenLabelAndNumber `subterm` 1
   ;AddHiddenLabelAndNumber `subterm` 2]
  ) p
;;

% SubtypeCD redefined later in inclusion-tactics.ml
  however the D_additions reference variable needs this version.
  Thus renmaed from SubTypeCD so as to make is accessible to D_additions.
%  
let SubtypeCD_atomic p = 
  let [S;T] = dest_simple_term_with_opid `subtype` (concl p) in
  let x = get_optional_var_arg `v1` p in

  ( Refine `subtypeIntro`
           (infer_level_exp_arg p S
            . mk_new_var_args [[x;mkv `x`]] p)
    THENL [Id;AddHiddenLabel `wf`]
  ) p
;;

let EqTypeCNarrow S p = 
 (Refine `subtypeElim` [mk_term_arg S] 
  THENL 
  [Id
  ;AddHiddenLabel `inclusion`]
 ) p
;;



%[
Support for `class subtype' relation.
]%

let CSubtypeEqCD p = 
  (Refine `csubtypeEquality` []
   THENL 
   [AddHiddenLabelAndNumber `subterm` 1
   ;AddHiddenLabelAndNumber `subterm` 2]
  ) p
;;

let CSubtypeCD p = 
  let [S;T] = dest_simple_term_with_opid `csubtype` (concl p) in
  let x,y = get_optional_var_arg `v1` p, get_optional_var_arg `v2` p in

  ( Refine `csubtypeIntro`
           (infer_level_exp_arg p S
            . mk_new_var_args [[x;mkv `x`];[y;mkv `y`];[]] p)
    THENL [Id;Id;AddHiddenLabel `wf`;AddHiddenLabel `wf`]
  ) p
;;

let EqTypeCClassWiden T p = 
 (Refine `csubtypeElim1` [mk_term_arg T] 
  THENL 
  [Id
  ;AddHiddenLabel `inclusion`
  ;AddHiddenLabel `wf`]
 ) p
;;

let EqTypeCClassNarrow S p = 
 (Refine `csubtypeElim2` [mk_term_arg S] 
  THENL 
  [Id
  ;AddHiddenLabel `inclusion`]
 ) p
;;



