%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%***************************************************************************
****************************************************************************

GENERAL CONVERSIONALS

****************************************************************************
***************************************************************************%

%
Following must be supplied, in order to load this file:

Type definitions:

rel(ation), 
just(ification), 
env(ironment)
( the type conv(ersion) is derived from these. See below)

Function definitions:

get_sub_envs 
  (e : env) 
  (t : term)
  = 
  (sub_es : env list)

get_identity_rel_and_just 
  (t : term) 
   = 
  (r,j : rel # just)
  
get_transitivity_rel_and_just
  (r1,j1 : rel # just) 
  (r2,j2 : rel # just) 
  (e : term)
  (tb : term)
   =
  (r,j : rel # just)

get_functionality_rel_and_just
  ([r1,j1;...;rn,jn] : (rel # just) list)
  (t : term)
  = 
  (r,j : rel # just)


Notes:
1. Do any of the get_ functions need access to the environment?
2. When justifications are tactics, we don't need most of the term info.
   However this does come in handy for the direct computation rewrite
   justifications.
3. The types of the above functions might change when we consider
   performance optimisation.

%




%***************************************************************************

General atomic conversions

***************************************************************************%

let (FailC : convn) e t = failwith `FailC`
;;

let IdC (e:env) (t:term) = t, get_identity_rel_and_just t
;;

%***************************************************************************

General conversionals

***************************************************************************%
%
Note that these conversionals are set up to treat the identity conversion
as completely valid. None of them do progress check.
%

let IfC (predicate:env->term->bool) (c:convn) e t =
  if predicate e t then c e t else failwith `IfC`
;;

let IfIsC t' (c:convn) e t =
  if alpha_equal_terms t t' then c e t else failwith `IfIsC`
;;


ml_curried_infix `ANDTHENC` ;;
ml_curried_infix `ORTHENC` ;;
ml_curried_infix `ORELSEC`;;


                           
let $ORELSEC (c1:convn) (c2:convn) e t =
  c1 e t ? c2 e t
;;

let $ANDTHENC (c1:convn) (c2:convn) e ta =
  let tb,r1_j1_pr = c1 e ta in
  let tc,r2_j2_pr = c2 e tb 
  in
    tc, get_transitivity_rel_and_just r1_j1_pr r2_j2_pr e tb 
;;


let TryC c =
  c ORELSEC IdC
;;


let $ORTHENC c1 c2 = 
  (c1 ANDTHENC TryC c2) ORELSEC c2
;;

%
The t1 = t2 check isn't strictly necessary, but it depends
on no conversion ever returning the same term, with a non
identity relation. Most of the time, this is the case
but occasionally it has caused problems e.g. if TagC
mistakenly tags term which doesn't compute.
%

let ProgressC (c:convn) e t1 = 
  let t2,r,j = c e t1 in
  if rel_equal r identity_reln 
     or alpha_equal_terms t1 t2
  then
    failwith `ProgressC: no progress`
  else 
    t2,r,j
;;


%
Repeatedly apply c until no further progress made. Never fails.
%

letrec RepeatC c e t =
    TryC (ProgressC c ANDTHENC RepeatC c) e t
;;

%
Like RepeatC but fails if c fails on first try.
%

let Repeat1C c =
  c ANDTHENC RepeatC c 
;;

letrec RepeatForC n c e t =
 (if n = 0 then 
    IdC
  else
    c ANDTHENC RepeatForC (n-1) c
 ) e t
;;




letrec FirstC (cs:convn list) e t =
  if null cs then failwith `FirstC`
  else 
    ((hd cs) ORELSEC (FirstC (tl cs))
    ) e t
;;

letrec SomeC (cs:convn list) e t =
  if null cs then failwith `SomeC`
  else 
    ((hd cs) ORTHENC (SomeC (tl cs))
    ) e t
;;

letrec AllC (cs:convn list) e t =
  if null cs then IdC e t
  else 
    ((hd cs) ANDTHENC (AllC (tl cs))
    ) e t
;;



%
SubIfC p c e t applies c to none/some/all of t's immediate subterms in 
left to right order. c is applied to the ith subterm if (p e t i) is true.
Care is taken to to evaluate p e t once, not once for each subterm.
If possible, propagate up the failure token from the last failure of c.

Relies on map3 applying f to the items of a list in left to right order.
%

let SubIfC (p:env->term->int->bool) (c:convn) e t =

  let is_selected_subterm = p e t in
  let TrappingCondConv i e' t' = 
    if is_selected_subterm i then (inl (c e' t') ?\x inr (x)) 
    else inr (`SubIfC: no active subterms`) 
  in
  let operator,bterms  = rw_quick_dest_term t in
  let subterms = map snd bterms in
  let subenvs = get_sub_envs e t in
  let trapped_results = 
        map3
          TrappingCondConv
          (upto 1 (length subterms))
          subenvs
          subterms
  in
    if not some trapped_results isl then
    ( let failure_tokens = map outr trapped_results in
      let real_failure_tokens = 
            filter 
            (\x.not x = `SubIfC: no active subterms`)
            failure_tokens
      in
        failwith (last real_failure_tokens ? `SubIfC: no active subterms`)
    )
    else
      let fixed_results = 
           map2
             (\trapped_result t.
                if isl trapped_result then 
                  outl trapped_result
                else          
                  IdC null_env t)
             trapped_results
             subterms
      in
      let new_subterms,rel_just_prs =
        unzip fixed_results in
      let rel_and_just = 
        get_functionality_rel_and_just rel_just_prs t in
      let new_bterms =
        map2
          (\bterm new_tm. fst bterm , new_tm)
          bterms 
          new_subterms
      in
      let new_term = rw_quick_mk_term operator new_bterms 
      in
        new_term,rel_and_just
;;


%
Apply c to all subterms of t.
%

let SubC (c:convn) =
  SubIfC (\e t i.true) c
;;

%
Apply c to only nth immediate subterm of t. Fails if c fails on subterm.
%

let NthSubC n (c:convn) e t =
  if n > length (bterms_of_term t) then
    failwith `NthSubC: invalid subterm index`
  else
    SubIfC (\e t i.i = n) c e t
;;

%
Apply c to addressed subterm. Fail if c fails.
%

let AddrC address (c:convn) e t =
  letrec Aux addr e' t' =
    if null addr then c e' t'
    else
      NthSubC (hd addr) (Aux (tl addr)) e' t' in
  Aux address e t
  ?? [`NthSubC: invalid subterm index`] failwith `AddrC: invalid address`
;;


%
Try applying successively to all subterms in sweep from top of term down to Nth
level from top. Fails if no application successful. 
%

letrec SweepDnNC n (c:convn) e t =
  ( c ORTHENC 
    if n = 1 then failwith `SweepDnNC`
    else SubC (SweepDnNC (n-1) c)
  ) e t
;;

% 
Same, with depth check disabled.
%
let SweepDnC = SweepDnNC 0
;;

%
Like SweepDnC, except if c successful on subterm t, don't try it 
on any subterms of t.
%

letrec HigherC (c:convn) e t =
  (c ORELSEC SubC (HigherC c)
  ) e t
;;

%
Try applying successively to all subterms in sweep from Nth level down up to
top of term. Fails if no application successful.
%

letrec SweepUpNC n (c:convn) e t =
  ( if n = 1 then c
    else 
      SubC (SweepUpNC (n-1) c) ORTHENC c
  ) e t
;;

% 
Same, with depth check disabled.
%
let SweepUpC = SweepUpNC 0
;;


%
Like SweepUpC, except if c successful on subterm t, don't try it 
on any superterms of t. (a superterm of t is a term which has t as a subterm.)
%

letrec LowerC (c:convn) e t =
  (SubC (LowerC c) ORELSEC c
  ) e t
;;

% 
Used in conjunction with NthsC. Conversion must succeed at some
point, and after success, attempt is made to run it on subterms of point.
%

let HereDnC c = c ANDTHENC TryC (SubC (SweepDnC c)) ;;


%
NthsC ns c t applies a modified c, c' to subterms of t in a preorder order, 
avoiding stepping into any converted subterms. c' works much like c,
failing when c fails, but not succeeding every time c would have succeeded.
c' succeeds the ith time c would have succeeded iff i is in ns. When c' 
succeeds, it returns the same result as as c would have given.
NthsC fails if c' hasn't succeeded each of the times listed in ns.

Is this the most natural way to do things?  The failure on subterms of 
converted terms ensures that all the conversions done are on independent
subterms. Also, it does seem important to allow descent into subterms which
c would have succeeded on. 

A purely functional version is possible, but probably wouldn't be as simple
or as fast.
%

let NthsC ns (c:convn) e t =
( let uncurried_c',id_if_lucky = nths_time_lucky ns (uncurry c) in
  let c' = curry uncurried_c' 
  in
    id_if_lucky (HigherC c' e t)
) ?? [`nths_time_lucky: too bad. count not in ns`
     ;`nths_time_lucky: too bad. count >= max ns`
     ]
       failwith `NthsC: argument conversion failed because not selected`
  ?? [`nths_time_lucky: still waiting for last success`]
       failwith `NthsC: every selected conversion did not succeed`
;;

let NthC n = NthsC [n]
;;

% 
Some compound tree walking conversionals. 

Depth, ReDepth, and ReTop come up in Paulson's higher order implementation 
of rewriting paper. (ReTop here = his TopDepth). Dave Basin played with Top.
Top and Depth fail if no application of c succeeds.
ReTop and ReDepth never fail. (Desirable since these will most likely be 
used as term normalisers. It will be of no interest whether they made 
progress or not.)

%

let TopC c = HigherC (Repeat1C c) ;;

let DepthC c = SweepUpC (Repeat1C c) ;;

% 
NB in both ReDepthC and ReTopC we try to avoid infinite recursion by
ensuring that the AuxC functions only succeed if they make progress. 
%

let ReDepthC c e t=
  letrec AuxC e' t' =
  ( SubC AuxC ORTHENC (c ANDTHENC TryC AuxC)
  ) e' t'
  in
    TryC AuxC e t
;;

let ReTopC c e t = 
  letrec AuxC e' t' =
  ( Repeat1C c 
    ORTHENC SubC AuxC 
    ORTHENC (c ANDTHENC TryC AuxC)
  ) e' t'
  in
    TryC AuxC e t
;;


%
***************************************************************************
Debugging conversionals:
***************************************************************************
BreakC c e t runs c on e t to get t',rel,just . It caches just with a number
and substitutes for just a tactic to add a visible label with that number.
%

letref just_cache = [] : (int # just) list ;;

letref just_count = 0 ;;

let reset_just_count () = 
  just_cache := [] ; just_count := 0 ; ()
;;

let add_entry_to_just_cache just =
  just_count := just_count + 1 
  ;
  just_cache := (just_count,just) . just_cache
  ; just_count
;;

let BreakC (c:convn) e t =
  let t',rel,just = c e t in
  let tag = add_entry_to_just_cache just in
    t',rel,form_tactic_just (AddHiddenLabelAndNumber `break` tag)
;;

update_stop_labels `break` ;;

let RunJustN i =
    open_tactic_just (apply_alist just_cache i)
;;

let RunJust p =
    RunJustN (number_of_proof p) p
;;


