%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*********************************************************************
*********************************************************************
GENERAL-TACTICS.ML
*********************************************************************
**********************************************************************
]%

%[
**********************************************************************
Generalization.
**********************************************************************
Generalization is The replacement of a subterm of a clause by a new
free variable, declared to be equal to the subterm. It is useful
for setting up induction, and doing case splits.
]%

let GenConclThen Tac t_eq_x_in_T p =
  let T,t,x = dest_equal t_eq_x_in_T in
  let new_concl = replace_subterm t x (concl p) in
  let seq_term =
    % All x:T. x=t in T => C[x/t] % 
    mk_all_term (dv x) T (mk_implies_term t_eq_x_in_T new_concl)   in
  let t_in_T = mk_member_term T t in
( AssertL [t_in_T; seq_term]
  THENL
  [% >> t in T %

   AddHiddenLabel `wf`

  ;% t in T >> all x:T. x=t in T => C[x/t] %

   D 0 THENM (Tac (-1) THEN Try (D 0 THENM Thin (num_hyps p + 1)))

  ;% t in T, all x:T. x=t in T => C[x/t] >> C %

   SeqOnSameConcl [DTerm t (-1); D (-1)] THEN Trivial
  ]
) p
;;

let OldGenConcl = GenConclThen (\i.Id) ;;

% OldGenConcl is profligate with wf goals. NewGenConcl is better %

let NewGenConcl t_eq_x_in_T p =
  let T,t,x = dest_equal t_eq_x_in_T in
  let new_concl = replace_subterm t x (concl p) in
  let seq_term =
    % All x:T. x=t in T => C[x/t] % 
    mk_all_term (dv x) T (mk_implies_term t_eq_x_in_T new_concl)   in
  let t_in_T = mk_member_term T t in
  let U = get_universe p T in
  let T_in_U = mk_member_term U T in

( AssertL [t_in_T; T_in_U; seq_term]
  THENL
  [% >> t in T % AddHiddenLabel `wf`
  ;% t in T,  >> T in U % Thin (-1) THEN AddHiddenLabel `wf`
  ;% t in T, T in U >> all x:T. x=t in T => C[x/t] %

   %(At U (D 0) THENA NthHyp (-1))%
   (At U (D 0) THENA Try (NthHyp (-1)))

   % t in T, T in U, x:T >> x=t in T => C[x/t] %
   THENw
   %(At U (D 0) THENA (MemCD THEN Trivial))%
   (At U (D 0) THENA (MemCD THEN (Try Trivial)))

   % t in T, T in U, x:T, x=t in T => C[x/t] %
   THENw
   OnHyps [-3;-3] Thin

  ;% t in T, T in U,  all x:T. x=t in T => C[x/t] >> C %

   SeqOnSameConcl [DTerm t (-1); D (-1)] THEN Trivial
  ]
) p
;;

letref GenConcl = NewGenConcl ;;

let Let x_eq_t_in_T = 
  let T,x,t = dest_equal x_eq_t_in_T
  in
    GenConcl (mk_equal_term T t x)
    THENM SwapEquands (-1)
;;


%[
****************************************************************************
Substitution functions.
****************************************************************************
Tactics for rewriting using substitution and equality rules. The 
rewrites possible are more limited than those possible with the rewrite
package, but they generate fewer well formedness goals.
]%

%
Substitute for whole clause
~~~~~~~~~~~~~~~~~~~~~~~~~~~
%

%
Hs  >> T

   By SubstConclClause T'

`equality` Hs >> T = T' in U{*}
`main`     Hs >> T'


* is lub of levels of T and T', unless overridden by optional `universe`
parameter.
%

let SubstConclClause T' p =
  let T = concl p in
  let Univ = get_term_arg `universe` p 
             ?
             U_lub (get_type p T) (get_type p T') 
  in
  let v = mkv `xxx`
  in
  At Univ 
     (BasicSubst (mk_equal_term Univ T T') v (mk_var_term v)
        THENL
        [Id
        ;Id
        ;OnHyp (-1) SoftNthDecl
        ]
     )
     p
;;

%
Hs x:T Js >> C

   By SubstHypClause T' i

`equality` Hs >> T = T' in U{*}
`main`     Hs x:T' Js >> C


* is lub of levels of T and T', unless overridden by optional `universe`
parameter.
%

let SubstHypClause T' i p =
  let i' = get_pos_hyp_num i p 
  in let T = h i' p 
  in let Univ = get_term_arg `universe` p 
             ?
             U_lub (get_type p T) (get_type p T') 
  in
  ( Refine `hyp_replacement`
           [mk_int_arg i'
           ;mk_term_arg T'
           ;mk_level_exp_arg (snd (dest_lp_term Univ))
           ]
    THENL
    [Id
    ;AddHiddenLabel `equality`
    ]
  ) p
;;

%
Combine SubstHypClause and SubstConclClause Generalizes SubstConcl above...
%

let SubstClause T' i p =
  if i = 0 then 
    SubstConclClause T' p
  else
    SubstHypClause T' i p
;;


%
Substitute for part of clause
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%


%
>> C[t1] 

  BY SubstInConcl 't1 = t2 in T'

`equality` >> t1 = t2 in T
`main`     >> C[t2]
`wf`   z:T >> C[z] in U{*}

* is inferred type of C[z'] or value of `universe` optional parameter.
%

let SubstInConcl t1_eq_t2_in_T p =
  let C_of_t1 = concl p in
  let T,t1,t2 = dest_equal t1_eq_t2_in_T in
  let z = maybe_new_proof_var (mkv `z`) p in
  let C_of_z = replace_subterm t1 (mk_var_term z) C_of_t1 
  in
    BasicSubst t1_eq_t2_in_T z C_of_z p
;;



%
 ... i:H[t1] ... >> C 

  BY SubstInHypProp 't1 = t2 in T' i

`equality` ... i:H[t1] ... >> t1 = t2 in T  
`wf`       ... i:H[t1] ...  z:T >> H[z] in U{*} 
`main`     ... i:H[t2] ... >> C 

* is inferred type of C[z] or value of `universe` optional parameter.

Will fail if variable declared in hyp i is used in later hyps or in concl.
%


let SubstInHypProp t1_eq_t2_in_T i p =
  % ... i:H[t1] ... >> C %
  let i' = get_pos_hyp_num i p in
  let H_of_t1 = h i' p in
  let T,t1,t2 = dest_equal t1_eq_t2_in_T in
  let z = maybe_new_proof_var (mkv `z`) p in
  let H_of_z = replace_subterm t1 (mk_var_term z) H_of_t1 in
  let H_of_t2 = fo_subst [z,t2] H_of_z in
  ( AssertAtHyp i' H_of_t2
    THENL
    [
    % ... i:H[t1] ... >> H[t2] %
      BasicSubst (mk_equal_term T t2 t1) z H_of_z
      THEN IfLabL
      [`main`,NthHyp i'
      ;`equality`,SwapEquands 0 THEN AddHiddenLabel `equality`
      ;`wf`,Try (FoldTop `member` 0) THEN AddHiddenLabel `wf`
      ]
    ;
     % ... i:H[t2], H[t1] ... >> C %
     Thin (i'+1)
     % ... i:H[t2] ... >> C %
    ]
  ) p
;;


%
 ... #i: x:H[t1] ... >> C 

  BY SubstInHypDecl 't1 = t2 in T' i

`main`     ... #i: x:H[t2] ... >> C 
`wf`       ... #i: x:H[t1] ... >> All z:T. H[z] in U{*} 
`equality` ... #i: x:H[t1] ... >> t1 = t2 in T  

* is inferred type of C[z] or value of `universe` optional parameter.

Works if variable declared in hyp i is used in later hyps or in concl.
%

let SubstInHypDecl t1_eq_t2_in_T i p =
  % ... i:H[t1] ... >> C %
  let i' = get_pos_hyp_num i p in
  let H_of_t1 = h i' p in
  let T,t1,t2 = dest_equal t1_eq_t2_in_T in
  let z = maybe_new_proof_var (mkv `z`) p in
  let H_of_z = replace_subterm t1 (mk_var_term z) H_of_t1 in
  let H_of_t2 = fo_subst [z,t2] H_of_z in
  ( SubstHypClause H_of_t2 i'
    THEN IfLabL
    [`equality`,
    % ... i:H[t1] ... >> H[t1] = H[t2] in U{*}%
     \p.
     let U = eq_type (concl p) 
     in let level = snd (dest_lp_term U) 
     in let U' = mk_U_term (mk_inc_level_exp level 1)
     in  
    (Assert (mk_all_term z T (mk_member_term U H_of_z))
     THEN IfLabL
     [`assertion`, 
      % ... i:H[t1] ... >> All z:T. H[z] in U{*}%
      AddHiddenLabel `wf`
     ;`main`,
      % ... i:H[t1] ... All z:T. H[z] in U{*} >> H[t1] = H[t2] in U{*} %
      Assert t1_eq_t2_in_T
      THEN IfLabL
      [`assertion`,
       % ... i:H[t1] ... All z:T. H[z] in U{*} >> t1 = t2 in T %
       Thin (-1) THEN AddHiddenLabel `equality`
      ;`main`,
       % ... All z:T. H[z] in U{*}, t1 = t2 in T >> H[t1] = H[t2] in U{*} %
       At U'
         (BasicSubst t1_eq_t2_in_T z (mk_equal_term U H_of_z H_of_t2))
       THEN IfLabL
       [`main`,
        % ... All z:T. H[z] in U{*}, t1 = t2 in T >> H[t2] = H[t2] in U{*} %
        DTerm t2 (-2) THEN Trivial
       ;`equality`,
        % ... All z:T. H[z] in U{*}, t1 = t2 in T >> t1 = t2 in T %
        NthHyp (-1)
       ;`wf`,
        % ... All z:T. H[z] in U{*}, t1 = t2 in T, z: T 
          >> (H[z] = H[t2] in U{*}) in U{*+1} %
        MemCD THENL
        [%>> U{*} in {U*+1} %
         MemCD
        ;%>> H[z] in {U*} %
         DTerm (mk_var_term z) (-3) THEN Trivial
        ;%>> H[t2] in {U*} %
         DTerm t2 (-3) THEN Trivial
        ]
       ]
      ]
     ]
    ) p
    ]
  ) p
;;


%
Do substitution of t1 for t2 in clause i. Generates 3 subgoals 
The main one with the substitution completed, a well formedness subgoal
labelled with `wf` and and equality subgoal labelled with 
`equality`
%

let Subst t1_eq_t2_in_T i p =
  ( % Special reasoning for sqequal terms %
    if is_sqequal_term t1_eq_t2_in_T then
      SqSubst t1_eq_t2_in_T i
    else if i = 0 then
      SubstInConcl t1_eq_t2_in_T
    else if is_invisible_var (var_of_hyp i p) then
      SubstInHypProp t1_eq_t2_in_T i 
    else
      SubstInHypDecl t1_eq_t2_in_T i 
  ) p
;;

% 
Uses equality in hyp j to rewrite clause i.  Do substitution of t2
for t2 in clause i. Generates 2 subgoals. The main one with the
substitution completed with no label, and a well formedness subgoal labelled 
with `wf`.  

HypSubst uses equality left to right.
RevHypSubst uses equality right to left.
%

let HypSubstEqualAux ta_eq_tb_in_T fwd j i p =
  % ... j:ta = tb in T ... >> C %
 (let T,ta,tb = dest_equal ta_eq_tb_in_T in
  let t1,t2 = if fwd then ta,tb else tb,ta in
  let EqTac = if fwd then NthHyp j else Eq 
  in
    Subst (mk_equal_term T t1 t2) i
    THEN IfLabL [`equality`,EqTac;`wf`,Id;`main`,Id]
 ) p
;;

let HypSubstSqEqualAux ta_sim_tb fwd j i p =
  % ... j:ta ~ tb ... >> C %
 (let ta,tb = dest_sqequal ta_sim_tb in
  let t1,t2 = if fwd then ta,tb else tb,ta in
  let EqTac = if fwd then NthHyp j else SqEq 
  in
    Subst (mk_sqequal_term t1 t2) i
    THEN IfLabL [`equality`,EqTac;`main`,Id]
 ) p
;;

let HypSubstAux fwd j i p =
    let eq = h j p in
        (if is_equal_term eq then
             HypSubstEqualAux eq
         else if is_sqequal_term eq then
             HypSubstSqEqualAux eq
         else
             failwith `HypSubst: not equality`) fwd j i p
;;

let HypSubst = HypSubstAux true ;;
let RevHypSubst = HypSubstAux false ;;




%
  ...#i:T = B in Ui ...#j: x = y in T  >> C
OR ...#i:T = B in Ui ... >> x = y in T     (j = 0)
%

let HypSubstEqType i j p = 
  let j' = get_pos_hyp_num j p in
  let T,x,y = dest_equal (clause_type j' p) in
  let U,A,B = dest_equal (h i p) in
  if not (alpha_equal_terms T A) then
    failwith `HypSubstEqType: lhs type different from eq type`
  else
  let new_eq = mk_equal_term B x y in
  if j' = 0 then 
  ( Assert new_eq THEN IfLabL
    [`main`,Eq
    ;`assertion`,AddHiddenLabel `main`
    ]
  ) p
  else
  ( AssertAtHyp j' new_eq THEN IfLabL
    [`main`,Thin (j'+1)
    ;`assertion`,Eq
    ]
  ) p
;;

let HypSubstMEType i = EqToMemberEq (HypSubstEqType i) ;;


%[
****************************************************************************
Alpha Conversion of clauses
****************************************************************************
]%


let RenameBVars bv_sub i p =
  if not i = 0 & is_visible_var (var_of_hyp i p) then
    SubstHypClause 
      (alpha_conv_term_mi bv_sub (h i p))
      i
      p
  else
    SubstAlphaEqProp
      (alpha_conv_term_mi bv_sub (clause_type i p))
      i
      p
;;

% variant, allowing individual renaming %

let RenameBVars' bv_sub i p =
  if not i = 0 & is_visible_var (var_of_hyp i p) then
    SubstHypClause 
      (alpha_conv_term bv_sub (h i p))
      i
      p
  else
    SubstAlphaEqProp
      (alpha_conv_term bv_sub (clause_type i p))
      i
      p
;;



%[
**********************************************************************
Backchaining tactics
**********************************************************************
Further on in this file we see an application of these tactics:
inferring recursively defined properties of terms such as

squash stability:  Squash(P)=>P 
stability:         not(not(P))=>P 
decidability:      P or not(P).

finite:                exists bijection between T and {1...n} for some n.
countably denumerable: exists bijection between T and N
enumerable:            T is either finite or countably denumerable

These properties are useful for doing various types of "classical" reasoning,
including using classical exists and or, and unhiding hidden hyps resulting
from set and squash hyp decomposition.

]%

letrec FirstThen Ts T p =
   if null Ts then failwith `FirstThen`
   if isl (hd Ts) then
     FirstThen (tl Ts) T p
   else
  (  (outr (hd Ts) THEN T)
     ORELSE FirstThen (tl Ts) T 
  ) p
;;

%[ 
CondFmlaBackchain 
  is_active_term : term -> bool
  complete  : bool
  names tok list
  hyps : int list
  = 
  T : tactic

is_active_term: selects subterms to continue backchaining on.
complete: if true, then fail if no applicable rule to active term.
          if false, leave leaves of backchain tree as subgoals.
hyps: list of hyp nums to backchain through. If 0 is included, then
      include new hyps in backchaining.
names: list of names of lemmas and hyps to backchain through.
       names tried in order given.

       Special names are as follows:

i(integer) hyp i. (only works with +ve integers at moment)
hyps: hyps 1...n
rev_hyps: hyps n...1
new_hyps: new hyps introduced by backchaining, least recent first.
rev_new_hyps: new hyps introduced by backchaining, most recent first.

We do some lemma preprocessing.

Not considered yet:
  Backchaining through nots.
  ( need refined loop checking, since False terms might occur > 1 time.)
  Optional non decomposition of concl term.
* Easy user extension:
  addition of tactics for `new` rules.
  e.g. unfolding of soft abstractions in properties inference stuff.
       or Arith, Hypothesis.
   add special control token for tactics!

*  Allow option of continuing backchaining after extension.



Special #s.  0 = all hyps.
             n+1 = all hyps 1...n
            -(n+1) = all hyps n ... 1  
how about new_hyps?
all_hyps 3 4 5 new_hyps

rev_new
new_hyps
hyps
explicit_hyps

use tokens rather than ints for hyp nums in this case...

Need to clean this up to make it shorter, more modular...
]%

let get_prop_hyp_nums hyp_nums p =
  if null hyp_nums then []
  else
  let vs = declared_vars p in
  filter
    (\i. is_invisible_var (nth i vs))
    hyp_nums
;;


let is_natural_number_tok tok =
  (tok_to_int tok ; true) ? false
;;

let CondBackchainWith AuxGoalTac Tac is_active_term lemma_and_hyp_names =
  let prelim_taclist =
      map
      (\name. if member name ``hyps rev_hyps new_hyps rev_new_hyps``
                 or is_natural_number_tok name
              then
                 inl name
              else
                 inr (BackThruLemma name)
      )
      lemma_and_hyp_names
  in
  let use_new_hyps = 
         member `new_hyps` lemma_and_hyp_names
         or member `rev_new_hyps` lemma_and_hyp_names
  in
  \explicit_hyp_nums p.
  let n = num_hyps p in
  let lemma_and_initial_hyp_taclist =
    flatten
    ( map
      (\t.
        if isr t then [t]
        else
        let ctl = outl t in
        if member ctl ``new_hyps rev_new_hyps`` then
          [t]
        else
        let hyp_nums = 
          (if ctl = `hyps` then (upto 1 n)
           if ctl = `rev_hyps` then rev (upto 1 n)
           if ctl = `explicit_hyps` then explicit_hyp_nums
           if is_natural_number_tok ctl then [tok_to_int ctl]
           else
             failwith `CondFmlaBackchain: bad case`
          ) 
        in
        let real_hyp_nums = 
             get_prop_hyp_nums hyp_nums p
        in
          map (\i. inr (QuickBackThruHyp p i)) real_hyp_nums
      )
      prelim_taclist
    )    
  in

  letrec Aux tac_list history p =
  ( % only backchain on antecedents (main goals) %
    if is_aux_proof p then
      AuxGoalTac
    else
    AddHiddenLabel `main`
    THEN GenUnivFmlaCD []
    THENM
    (\p'.
     (let c = concl p' in
      if member_p c history alpha_equal_terms then
        Fail
      if is_active_term c then
      ( let new_prop_hyps =
          (if use_new_hyps then
             get_prop_hyp_nums (upto (num_hyps p + 1) (num_hyps p')) p'
           else
             []
          )
        in
        let new_tac_list =
        ( if null new_prop_hyps then tac_list
          else
          flatten
            (map
             (\t.
               if isr t then [t]
               if outl t = `new_hyps` then
                 map (\i.inr (QuickBackThruHyp p' i)) new_prop_hyps @ [t]
               else
                 [t] @ map (\i.inr (QuickBackThruHyp p' i)) new_prop_hyps 
             )
             tac_list
          )    
        )
        in
          FirstThen new_tac_list (Aux new_tac_list (c.history))
          ORELSE (SwapEquands 0 
                  THEN FirstThen new_tac_list (Aux new_tac_list (c.history)))
          ORELSE Tac (Aux new_tac_list (c.history))
      )
      else
        Id 
     ) p'
    )
  ) p
  in
  ( AddHiddenLabel `main` THEN Aux lemma_and_initial_hyp_taclist []) p
;;

let BackchainWith Tac names = CondBackchainWith Id Tac (\t.true) names [] ;;
let Backchain = BackchainWith (\T.Id) ;;
let CompleteBackchain = BackchainWith (\T.Fail) ;;

let HypBackchain = Backchain ``rev_new_hyps rev_hyps`` ;;
let CompleteHypBackchain = CompleteBackchain ``rev_new_hyps rev_hyps`` ;;
 

%[ 
here we provide a quick way of backchaining without going through 
all trouble of running tactic.
]%

letrec first_fun_then fs f t =
  if null fs then 
    false
  else
    all f ((hd fs) t) ? first_fun_then (tl fs) f t
;;
  

let lemma_backchain lemma_names is_active_term =
  let step_back_funs = 
    map
    (\name.
       let x_A_prs,Bs,C = dest_simple_lemma name in
       let Bs' = filter is_active_term Bs in
       let xs = map fst x_A_prs 
       in
       \t. let sub = match xs C t in
           map (subst sub) Bs'
  
    )
    lemma_names
  in
  letrec bchain t =
    first_fun_then step_back_funs bchain t
  in
    bchain
;;



%[
*********************************************************************
Reasoning about excluded middle predicate.
**********************************************************************
]%

% 
infers appropriate level to use for lemma, if no At arg supplied.
Assumes xmiddle abstraction in library. %
  
let AddXM i p = 
 let i' = get_pos_hyp_num i p
 in
 let le = 
 (  snd (dest_lp_term (get_term_arg `universe` p))
   ?
   let le_vars = 
     level_vars 
     (mk_iterated_and
        (concl p . map type_of_declaration (hyps p))
     )
   in let le_vars' = null le_vars => [unit_le_id] | le_vars
   in simplify_level_exp
         (mk_level_exp (map (\v.v,1) le_vars'))
 )
 in
 ( Assert (mk_squash_term (mk_lp_term `xmiddle` le))
   THEN IfLab `assertion` 
     (UseWitness axiom_term THEN Fiat)
     (MoveToHyp i' (-1) THEN BasicSquashHD 1
      THEN Try TrivializeConcl)
 ) p
;;
  

% PERF
let get_xm_hyp_num p = 
  let vThs = dest_full_hyps p
  in 
    (search_list (\v,t,h.is_term `xmiddle` t & not h) vThs)
;;
%

let get_xm_hyp_num p = 
  search_list (\h. is_term `xmiddle` (type_of_declaration h)
                 & not (is_hidden_declaration h))
	       (hyps p)
;;

let exists_xm_hyp p = (get_xm_hyp_num p; true) ? false
;;



%[
**********************************************************************
Proving Squash Stability, Stability and Decidability.
**********************************************************************

SqStable(P) == Squash(P) => P
Stable(P) == not(not(P)) => P
Decidable(P) == P or not(P)

Squash-stability is important for unhiding hyps; a hyp can be unhidden if
1. the concl is sq stable.
2. the hidden hyp is sq stable.

Stability is important for encoding of classical logic using double negation.

Note that with the assumption that the xmiddle abstraction is a lemma,
(see above for definition), Stability and SqStability are the same.

(This is expressed by lemma: sq_stable_iff_stable)

Decidability is important when reasoning by cases constructively.
]%



%
Auxiliary function. Eliminates need to have stability and decidability
lemmas for soft definitions which expand to other propositions.
%

let PrStabDecRetry T = 
  UnfoldSoftAbAtAddr [1] 0
  THEN 
  \p.let arg = get_addressed_subterm [1] (concl p)
     in let id = opid_of_term arg
     in
     if is_primitive_opid id & not member id ``equal lt`` then
       Fail p
     else
       T p
;;

%[
--------------------------------------------------------------------------
Decidability Reasoning
--------------------------------------------------------------------------
]%
let ProveDecidableFromXMHyp p = 
 if is_term `decidable` (concl p) then
 (let i  = get_xm_hyp_num p
           ? failwith `ProveDecidableFromXMHyp: no XM hyp`
  in
  ( Unfold `xmiddle` i
    THEN BackThruHyp i
  ) p
 )
 else
   failwith `ProveDecidableFromXMHyp: not decidable concl`
;;

% 
No backtracking should be necessary.

sq_stable recognition is useful if after running ProveSqStable1, one
wants to push back to subgoals which can't be proven decidable.
%
%
letref decidable__lemmas = (inl () : (unit + (tok list)));;

let get_decidable__lemmas () = 
 if isl decidable__lemmas then
    outr (decidable__lemmas := inr (names_of_statements_with_prefix `decidable__`))
 else outr decidable__lemmas
;;

let decidable__lemmas_reset () = decidable__lemmas := (inl ());;
%
let get_decidable__lemmas = lookup_Decidable__lemmas;;

% 9/2002 shadowed by next def ???
let filter_numbered_hyps p f hs = 
 letrec aux i hs =
    if null hs then []
    else if (p h)
	    then (f i h) . (aux (1 + i) (tl hs))
    else aux (1 + i) (tl hs)
 in aux 1 hs
;;
%
let filter_numbered_hyps p hs =
 letrec aux i hs = 
    if null hs then []
    else if (p (hd hs)) then i . (aux (1 + i) (tl hs))
    else aux (1 + i) (tl hs) 
 in aux 1 hs
;;


let ProveDecidable1 p =
 % let get_dec_fmla_num (i,v,h) = 
      if is_term `decidable` (snd (snd (dest_simple_formula h))) then
        int_to_tok i
      else
        fail
  in %

  let hyp_nums = map int_to_tok
	             (filter_numbered_hyps
                       (\h. is_term `decidable`
	  		            (simple_formula_con (type_of_declaration h)))
                       (hyps p))
      % mapfilter get_dec_fmla_num (number (dest_hyps p)) %
  in
  let fmla_names = hyp_nums @ (get_decidable__lemmas ())
  in let Tac T =  PrStabDecRetry T
                  ORELSE AddHiddenLabel `decidable?`
  in
  if is_term `decidable` (concl p) then
    BackchainWith Tac fmla_names p
  if is_term `sq_stable` (concl p) then
  ( BackThruLemma `sq_stable_from_decidable`
    THENM BackchainWith Tac fmla_names 
  ) p
  else
    failwith `ProveDecidable1: concl must be decidable term`
;;

let ProveDecidable p =
 (ProveDecidableFromXMHyp
  ORELSE
  ( ProveDecidable1 THEN
  IfLab `decidable?`
    (FailWith `ProveDecidable: could not complete proof. Try ProveDecidable1`)
    Id
  )
 ) p 
;;

%[
--------------------------------------------------------------------------
SqStability Reasoning
--------------------------------------------------------------------------
]%


let ProveSqStableFromXMHyp p = 
 if is_term `sq_stable` (concl p) then
 (let i  = get_xm_hyp_num p
           ? failwith `ProveSqStableFromXMHyp: no XM hyp`
  in
  ( Unfold `xmiddle` i
    THEN BackThruLemma `sq_stable__from_stable`
    THENM BackThruLemma `stable__from_decidable`
    THENM BackThruHyp i
  ) p
 )
 else
   failwith `ProveSqStableFromXMHyp: not sq_stable concl`
;;


%
Here we try unfolding soft abstractions under property predicate if no match 
initially found.

Have to remove reference to old lemma in library

Some stability lemmas have decidability antecedents, so we try to
solve these too.

Note that we insist after converting stability concl to dec concl
that ProveDecidable complete.  This seems desirable since this is
a significant weakening step.
%

%
letref sq_stable__lemmas = (inl () : (unit + (tok list)));;

let get_sq_stable__lemmas () = 
 if isl sq_stable__lemmas then
    outr (sq_stable__lemmas := inr (filter (\x. not (x = `sq_stable__from_stable`))
			   		  (names_of_statements_with_prefix `sq_stable__`)))
 else outr sq_stable__lemmas
;;

let sq_stable__lemmas_reset () = sq_stable__lemmas := (inl ());;
%
let get_sq_stable__lemmas = lookup_sq_stable__lemmas;;

let ProveSqStable1 p =
  if not is_term `sq_stable` (concl p) then
    failwith `ProveSqStable: concl must be sq_stable term`
  else
  ( let thm_names = get_sq_stable__lemmas ()
    in
    %let get_sq_st_fmla_num (i,v,h) = 
      if is_term `sq_stable` (snd (snd (dest_simple_formula h))) then
        int_to_tok i
      else
        fail
    in %
    let hyp_nums = map int_to_tok 
                       (filter_numbered_hyps
                         (\h. is_term `sq_stable`
				      (simple_formula_con (type_of_declaration h)))
			 (hyps p))
       % mapfilter get_sq_st_fmla_num (number (dest_hyps p)) %
    in
    let Tac T = PrStabDecRetry T
                ORELSE (BackThruLemma `sq_stable_from_decidable` 
                        THENM ProveDecidable)
                ORELSE IfOnConcl (is_term `decidable`) ProveDecidable1 Id
          
                ORELSE AddHiddenLabel `squash stable?` 
    in
      BackchainWith Tac (hyp_nums @ thm_names) p
  )
;;

let ProveSqStable p =
 (ProveSqStableFromXMHyp
  ORELSE
  ( ProveSqStable1 THEN
  IfLabL
   [`squash stable?`,
    FailWith `ProveSqStable: could not complete proof. Try ProveSqStable1`
   ;`decidable?`,
    FailWith `ProveSqStable: could not complete proof. Try ProveSqStable1`
   ]
  )
 ) p 
;;


%[
--------------------------------------------------------------------------
Stability Reasoning
--------------------------------------------------------------------------
]%

let ProveStableFromXMHyp p = 
 if is_term `stable` (concl p) then
 (let i  = get_xm_hyp_num p
           ? failwith `ProveStableFromXMHyp: no XM hyp`
  in
  ( Unfold `xmiddle` i
    THEN BackThruLemma `stable__from_decidable`
    THENM BackThruHyp i
  ) p
 )
 else
   failwith `ProveStableFromXMHyp: not stable concl`
;;

let ProveStable1 p =
  if not is_term `stable` (concl p) then
    failwith `ProveStable: concl must be stable term`
  else
  ( BackThruLemma `sq_stable_iff_stable`
    THENM ProveSqStable1
  ) p
;;

let ProveStable p =
 (ProveStableFromXMHyp
  ORELSE
  ( ProveStable1 THEN
  IfLab `squash stable?`
    (FailWith `ProveStable: could not complete proof. Try ProveStable1`)
    Id
  )
 ) p 
;;



%[
**********************************************************************
Unhiding tactics
**********************************************************************
]%


let XMUnhide i p = 
  let i' = get_pos_hyp_num i p
  in let decl = nth_decl i' p in

  if not exists_xm_hyp p then
    failwith `XMUnhide: no XM hyp`
  if not is_hidden_declaration decl then
    failwith `XMUnhide: hyp not hidden`
  else
  let xmi = get_xm_hyp_num p
  in let P = type_of_declaration decl
  in 
  ( AssertAtHyp i' (mk_simple_term `decidable` [P])
    THEN IfLabL
    [`assertion`,
     CopyToEnd xmi
     THEN Unfold `xmiddle` (-1)
     THEN BackThruHyp (-1)
     THEN IfLab `main` (Thin (-1)) 
                       (Thin (-1) THEN AddHiddenLabel `wf`)
    ;`main`,
     D i' THENL
     [% P % Thin (i'+1)
     ;%not P %
      Assert (mk_simple_term `false` [])
      THEN IfLabL
      [`assertion`,
       TrivializeConcl THEN Contradiction
      ;`main`,Trivial
      ]
     ]
    ]
  ) p
;;



% SqStable{H} = Sq{H} => H %

let UnhideSqStableHyp1 i p =
  let decl = nth_decl i p in

  if not is_hidden_declaration decl then
    failwith `UnhideSqStableHyp: hyp not hidden`
  else

  % ... #i:[H]... >> C %
 (Assert (mk_simple_term `sq_stable` [type_of_declaration decl])

  THENL
  [
  % ... #i:[H]... >> SqStable{H} %
   AddHiddenLabel `sq stable`
  ;
  % ... #i:[H]...SqStable{H} >> C %
   D (-1)
   THENL
   [
    % ... #i:[H]... >> Sq{H} %
    UnhideSinceSquashedConcl
    % ... #i:H... >> Sq{H} %
    THEN SquashCD 
    % ... #i:H... >> H %
    THEN NthHyp i
   ;
    % ... #i:[H]...H >> C %
    Thin i 
    THEN MoveToHyp i (-1)
    % ... #i:H... >> C %
   ]
  ]
 ) p
;;

let UnhideSqStableHyp i p =
 ( UnhideSqStableHyp1 i 
  THEN IfLab `sq stable` ProveSqStable Id
 ) p
;;

let CSquash2 p = 
 (Assert (mk_simple_term `sq_stable` [concl p])
  THENL
  [
   % ... [H] ... >> SqStable{C} %
   AddHiddenLabel `squash stable?`
  ;
   % ... [H] ..., SqStable{C} >> C %
   D (-1)
   THENL
   [AddHiddenLabel `main`
   ;
    % ... [H] ...,C >> C %
    NthHyp (-1)
   ]
  ]
 ) p
;;

let CSquash1 p = (CSquash2 THENA ProveSqStable1) p ;;
let CSquash p = (CSquash2 THENA ProveSqStable) p ;;


let NoteConclSqStableAux ProverTac p =
 (
  CSquash2 
  THEN IfLabL
  [`main`,
    % ... [H] ... >> Sq{C} %
    UnhideSinceSquashedConcl
    % ... H ... >> Sq{C} %
    THEN SquashCD 
    % ... H ... >> C %
  ;`squash stable?`,
   ProverTac
  ]
 ) p
;;

let NoteConclSqStable = NoteConclSqStableAux ProveSqStable ;;
let NoteConclSqStable1 = NoteConclSqStableAux ProveSqStable1 ;;
let NoteConclSqStable2 = NoteConclSqStableAux Id ;;


let UnhideAllHypsSinceSqStableConcl p = 
  if all ($not o is_hidden_declaration) (hyps p) then
    failwith `UnhideAllHypsSinceSqStableConcl: no hidden hyps`
  else
    NoteConclSqStable p 
;;

let HUnhide p =
  let hs = hyps p in
  let hidden_hyps =
    map_omitting_failures
      (\decl,i. if is_hidden_declaration decl then i else fail)
      (zip hs (upto 1 (length hs)))
  in
  if null hidden_hyps then
    failwith `Unhide: no hyps to unhide`
  else
    OnHyps hidden_hyps UnhideSqStableHyp p
;;

let CUnhide p = 
 (TrivializeConcl
  ORELSE UnhideAllHypsSinceSqStableConcl
 ) p
;;

%
We try checking concl first, because most sq stable hyps should be unhidden
at Squash or Set HD time.
%

let Unhide p = (CUnhide ORELSE HUnhide) p
;;

%[
**********************************************************************
Properties of abstract set terms
**********************************************************************
Here we offer a desirable way of doing set hyp decomposition which always
unhides the extra information in the set type.
We refer to what we call `properties' lemmas. 

e.g. if we have an abstraction definition:

DEF(z) == {x:A|B(x,z)}

and we have a lemma DEF_properties:

all z:T all y:DEF(z). B(y,z)

Then decomposing a hyp with DEF(z) in it invokes the lemma to get the 
properties.

Assumes lemma of form:

All xs:As. Bs. All y:DEF(xs) . B(y,xs)

If B starts with All or => it should be wrapped in a guard term.


... #i:y:DEF(z) ... >> C

  BY AddProperties i

`main`  ... #i:y:DEF(z),B(y,z) ... >> C


OR:
... #i:a in DEF(z) ...>> C

  BY AddProperties i

`main`  ... #i:a in DEF(z),B(a,z) ...>> C

OR:

... #i:a = b in DEF(z) ...>> C

  BY AddProperties i

`main`  ... #i:a = b in DEF(z), B(a,z) ...>> C

Ideally, system should try automatically to prove and add properties lemmas.

AddProperties handles cases when DEF in hyp is wrapped in one or more
soft abstractions
]%


let AddPropertiesAux mk_lemma_name i p =
  let i' = get_pos_hyp_num i p in
  let z,H = dest_hyp i' p in
  let a,ab_set_instance = 
    (let T,a,() = dest_member_or_equal H in a,T) 
    ? mk_var_term z,H
  in
  let hard_ab_set_inst =  unfold_soft_abs ab_set_instance in
  let opid = opid_of_term hard_ab_set_inst in
  let lemma_name = mk_lemma_name hard_ab_set_inst in
  let bindings,(),() = dest_simple_lemma lemma_name in  
  let x_A_prs,[y,ab_set_pattern] = split_lastn 1 bindings in
  if not opid_of_term ab_set_pattern = opid then
    failwith `AddProperties: lemma unsuitable`
  if not (is_statement lemma_name) then
    failwith `AddProperties: lemma incomplete`
  else
  let xs = map fst x_A_prs in
  let sub = match_in_context 
              x_A_prs ab_set_pattern ab_set_instance [] (env_of_sequent p) 
  in
  ( InstLemmaWithSub (lemma_lookup lemma_name) (sub @ [y,a])
    THEN_OnEach 
      (\ps.replicate (Try Trivial) (length ps - 1) @ [MoveToHyp (i'+1) (-1)])
  ) p
;;

let AddProperties i p =
  AddPropertiesAux (\k.opid_of_term k ^ `_properties`) i p
;;


let AddAllPropertiesA i p = 
  let i' = get_pos_hyp_num i p
  in
  ( AddPropertiesAux (\k.opid_of_term k ^ `_all_properties`) i' 
    THENM RepeatAndHD (i'+1)
  ) p
;;


% maybe want to restrict to one hyp as well ?? %
let AUnfoldsTop_aux attrs i p =
  if exists (\a. has_ab_attr_t a (clause_type i p)) attrs then
    UnfoldTopAb i p
  else
    Fail p
;;

let AUnfoldsTop attrs  =
 let tattrs = map (string_to_tok o string_upcase) attrs in
   AUnfoldsTop_aux tattrs
;;

let AUnfoldTop attr =
 let attrs = [string_to_tok (string_upcase attr)] in
  (\i p. AUnfoldsTop_aux attrs i p) ;;

let ARepD attrs = 
 let tattrs = map (string_to_tok o string_upcase) attrs in
 (\p.
  RepeatM 
   (SeqOnM 
     [UnivCD
     ;TryOnAllHyps RepeatAndHD
     ;TryOnAllClauses (AUnfoldsTop_aux tattrs)
     ] ) p)
;;

let AGenRepD attrs = 
 let tattrs = map (string_to_tok o string_upcase) attrs in
 (\p.
  RepeatM 
   (SeqOnM 
     [GenUnivCD
     ;TryOnAllHyps RepeatAndHD
     ;TryOnAllClauses (AUnfoldsTop_aux tattrs)
     ] ) p)
;;


let FlattenCompounds p = 
  let HTac i p = 
    if has_ab_attr_t `COMPOUND` (h i p) then
    ( UnfoldTopAb i THEN RepeatAndHD i) p
    else
      Fail p
  in
    RepeatM (TryOnAllHyps HTac) p
;;





%[
**********************************************************************
More Set-related tactics
**********************************************************************
]%


let SquashHD i p =
  (BasicSquashHD i
   THEN (TrivializeConcl ORELSE Try (XMUnhide (i+1)))
  ) p
;;


%
This could be optimized by checking relative sizes of concl and hyp...
or deleted altogether!
%

let SetHD i p =
  let i' = get_pos_hyp_num i p in
  (BasicSetHD i'
   THEN 
   ( TrivializeConcl
     ORELSE UnhideSqStableHyp (i'+1) 
     ORELSE Try UnhideAllHypsSinceSqStableConcl
   )
  ) p
;;


let AbSetHD i p =
 (let i' = get_pos_hyp_num i p in
  ( AddProperties i'
    THENM
    ( Repeat (UnfoldTopAb i')
      THEN BasicSetHD i'
      THEN Thin (i'+1)
    )
  )
% Is innappropriate to do the squash stable analysis here that SetHD does. %
  ORELSE
  ( Repeat (UnfoldTopAb i')
    THEN BasicSetHD i'
    THEN (TrivializeConcl ORELSE Try (XMUnhide (i'+1)))
  )
 ) p
;;

let SquashD i p =
  if not is_term `squash` (clause_type i p) then
    failwith `SquashD`
  if i = 0 then
    SquashCD p
  else
    SquashHD i p
;;

let AbSetD i p =
  if not is_term `set` 
     (repeatf 
       (unfold_ab o prune_subterms)   % unfold_ab can be slow w/o pruning %
       (clause_type i p)) 
  then
    failwith `AbSetD`
  if i = 0 then
  ( Repeat (UnfoldTopAb 0) THEN PrimCD) p
  else
    AbSetHD i p
;;

% important to do SquashD first since AbSetD also works in the same cases. %



%
Recap:
BasicSetHD : invokes primitive rule. Proposition always hidden.
             Used by the PrimHD tactic.

SetHD      : Does BasicSetHD and then tries hard to unhide proposition by
             using squash stability backchaining.

AbSetHD    : Unfolds any abstractions.
             Uses prim rule to get type hyp. Looks for property lemma
             for proposition. If none, we get hidden hyp, which we only 
             try trivial things on for unhiding.
             Used in extensions to D tactic.

%


let AbSetEqTypeHD i p = 
 (let i' = get_pos_hyp_num i p in
  ( AddProperties i'
    THENM
    ( Repeat (UnfoldAtAddr [1] i')
      THEN BasicSetEqTypeHD [1] i'
      THENM Thin i'
    )
  )
  ORELSE
  ( Repeat (UnfoldAtAddr [1] i')
    THEN BasicSetEqTypeHD [1;2] i'
    THENM SquashHD (i'+1)
  )
 ) p
;;

% repeat using properties lemma %

let AddAllPropertiesB i p = 
  let i' = get_pos_hyp_num i p in
  let z,H = dest_hyp i' p in
  let PreTac = 
    if is_terms ``member equal`` H then 
    ( let T,a,() = dest_member_or_equal H in
        AssertAtHyp (i'+1) (mk_member_term T a) THENA Eq
    )
    else
        AssertAtHyp (i'+1) (mk_member_term H (mk_var_term z)) THENA NthDecl i'
  in
  ( PreTac THEN UnfoldTop `member` (i'+1) THEN
    RepeatM (AbSetEqTypeHD (i'+1)) THENM Thin (i'+1)
  ) p
;;


let AddAllProperties i p = 
  (AddAllPropertiesA i ORELSE AddAllPropertiesB i) p
;;

%
********************************************************************************
Hooks for adding functionality for decomposing equality types.
********************************************************************************
%
letref EqTypeCD_additions = [] : (tok # tactic) list;;

let update_EqTypeCD_additions id T = 
    EqTypeCD_additions :=  update_alist EqTypeCD_additions id T; ();;

letref EqTypeHD_additions = [] : (tok # (int -> tactic)) list;;

let update_EqTypeHD_additions id T =
    EqTypeHD_additions := update_alist EqTypeHD_additions id T; ();;

%[
****************************************************************************
User-level tactics for decomposing member and equality types.
****************************************************************************
]%


let WeakEqTypeCD p =
  (SquashEqTypeCD
   ORELSE WeakAbQuotEqTypeCD
   ORELSE AbRecTypeEqTypeCD 
   ORELSE AbSetEqTypeCD
   ORELSE First (map snd EqTypeCD_additions)
   ORELSE FailWith `WeakEqTypeCD: can\'t decompose type`
  ) p
;;

let MemTypeCD p =
  EqToMemberEq (\i.WeakEqTypeCD) 0 p
;;

let EqTypeCD p =
  (SquashEqTypeCD
   ORELSE AbQuotEqTypeCD
   ORELSE AbRecTypeEqTypeCD 
   ORELSE AbSetEqTypeCD
   ORELSE First (map snd EqTypeCD_additions)
   ORELSE FailWith `EqTypeCD: can\'t decompose type`
  ) p
;;

let EqTypeHD i = 
  AbSetEqTypeHD i
  ORELSE AbQuotEqTypeHD i
  ORELSE First (map (\(),tac. tac i) EqTypeHD_additions)
  ORELSE FailWith `EqTypeHD: can\'t decompose type`
;;

let MemTypeHD i = EqToMemberEq EqTypeHD i
;;

let EqTypeD i = 
  if i = 0 then EqTypeCD else EqTypeHD i
;;

let MemTypeD i = EqToMemberEq EqTypeD i ;;


let WidenEqType p = 
  let T,a,b = dest_equal (concl p) in
 (Assert (mk_member_term T a) THEN IfLabL 
  [`main`,
    EqTypeCD
    THEN IfLabL 
    [`set predicate`, AddProperties (-1) THENM Trivial
    ;`main`,Thin (-1)
    ;`wf`,KeepingAnnotation (Thin (-2))
    ]
  ;`assertion`,AddHiddenLabel `wf`
  ]
 ) p
;;


%[
**********************************************************************
Miscellaneous
**********************************************************************
]%

let StdizeApartVarsInConcl p = 
  ( Assert (stdize_apart_vars_of_term (declared_vars p) (concl p))
    THENL [Id;Hypothesis]
  ) p
;;


%[
**********************************************************************
Reasoning by cases
**********************************************************************
We provide some beefing up of the basic induction principles.
]%


% ... #i: C1 or (C2 or ( ... (Cn-1 or Cn) ...)) ... >> T 

 get n subgoals:

 ... #i: C1 ... >> T 
  .     .    .    .
  .     .    .    .
  .     .    .    .
 ... #i: Cn ... >> T 
%

letrec CaseSplitNWay n i p =
  if n < 2 then Id p
  else
  ( D i 
    THENL [Id;CaseSplitNWay (n-1) i]
  ) p
;;

% get length Cs `main` subgoals.
  and 1 `assertion` subgoal.
%

let Cases Cs =
  let disj = mk_iterated_or Cs in
  Assert disj 
  THENM CaseSplitNWay (length Cs) (-1)
;;


let Decide P =
  Assert (mk_simple_term `decidable` [P]) 
  THEN
  IfLabL
  [`assertion`,ProveDecidable
  ;`main`,OnHyp (-1) D
  ]
;;

let Decide1 P =
  Assert (mk_simple_ab_term `decidable` [P]) 
  THEN
  IfLabL
  [`assertion`,ProveDecidable1
  ;`main`,OnHyp (-1) D
  ]
;;


%[
**********************************************************************
DoubleNeg Elimination
**********************************************************************
]%

%
Here we try unfolding soft abstractions under property predicate if no match 
initially found
%

let AssertStable i p = 
  let A = clause_type i p 
  in 
  ( Assert (mk_simple_term `stable` [A])
    THENA Try ProveStable
  ) p
;;

let is_dnot_term t = is_not_term (dest_not t) ? false ;;

% assume hi is dnot term %

%
let DNotHDByFalseConcl i p =
   if is_term `false` (concl p) then
%


% Stable{H} = not(not(H)) => H %

let DNotHD i p =
  let decl = nth_decl i p in

  if not is_hidden_declaration decl then
    failwith `UnhideSqStableHyp: hyp not hidden`
  else

  % ... #i:[H]... >> C %
 (Assert (mk_simple_term `sq_stable` [type_of_declaration decl])

  THENL
  [
  % ... #i:[H]... >> SqStable{H} %
   AddHiddenLabel `sq stable`
  ;
  % ... #i:[H]...SqStable{H} >> C %
   D (-1)
   THENL
   [
    % ... #i:[H]... >> Sq{H} %
    UnhideSinceSquashedConcl
    % ... #i:H... >> Sq{H} %
    THEN SquashCD 
    % ... #i:H... >> H %
    THEN NthHyp i
   ;
    % ... #i:[H]...H >> C %
    Thin i 
    THEN MoveToHyp i (-1)
    % ... #i:H... >> C %
   ]
  ]
 ) p
;;

let NegateConcl2 p = 
  let c = concl p in

 (Assert (mk_simple_ab_term `stable` [concl p])
  THEN IfLabL
  [`assertion`,AddHiddenLabel `stable?`
  ;`main`,
   D (-1) THEN IfLabL
   [`antecedent`,D 0
   ;`main`,NthHyp (-1)
   ]
  ]
 ) p
;;

% Generalization of NegateConcl:

|- d1 \/ ... di ...  \/ dn

BY NegateConclDisjunct2 i

`main`  not(di) |- d1 \/ ... ... \/ dn
`decidable?         |- Dec(di)

TO BE WRITTEN:

let NegateConclDisjunct i p = 
%


let NegateConcl1 p =
  (NegateConcl2 THEN IfLabL [`stable?`,ProveStable1]) p
;;
let NegateConcl p =
  (NegateConcl2 THEN IfLabL [`stable?`,ProveStable]) p
;;

let NegateHyp i = 
  Assert false_term THEN IfLabL
  [`main`,Trivial
  ;`assertion`,MoveToConcl i THEN FoldTop `not` 0]
;;





let Negate i = 
  if i = 0 then NegateConcl else NegateHyp i ;;

%[
**********************************************************************
Function to predicate conversion
**********************************************************************
Assume lemma: <opid of f>_sat_pred

All xs. P(xs,(f(xs))

f(xs) must be last subterm of P


... |- C[f(zs)]

BY FunElim f(zs) = y in T  

... y:T, P(zs,y) |- C[y]

]%

let FunElim f_eq_y_in_T p = 
  let T,f_zs,y = dest_equal f_eq_y_in_T
  in let lemma_name = opid_of_term f_zs ^ `_sat_pred`
  in let xAs,(),P_xs_fxs = dest_simple_lemma lemma_name 
  in let fxs = last (subterms P_xs_fxs)
  in let sub = match_in_context xAs fxs f_zs [] (env_of_sequent p) 
  in 
    SeqOnM
    [ GenConcl f_eq_y_in_T

      % y:T, f(zs) = y in T |- C[y] %

    ; InstLemmaWithSub (lemma_lookup lemma_name) sub

      % y:T, f(zs) = y in T, P(zs,f(zs)) |- C[y] %

    ; HypSubst (-2) (-1)
    ; Thin (-2)
    ] p
;;

