%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%
;;;;	scroll list
;;;;	
;;;;	 - static list 
;;;;	!static_list (<cons_op>; <list>)
%

    
let zero_parm = make_natural_parameter 0;;

let modify_static_push item op point static =
 (0, (make_icons_term op item static))
;;

let modify_static_pop point op static =
 (0, (itail static))
;;

letref ssdebug = (ivoid_term, 0);;

let modify_static_ap f op point static =
  let npoint, nlist = (f point (map_isexpr_to_term_list op static)) in
   (npoint, (map_to_ilist (\x.x) op nlist))
;;

let modify_static_set list =
  modify_static_ap (\p l. (0,list))
;;

let modify_static_replace item =
 modify_static_ap 
   (\point l. 
     let (prefix, suffix) = split point l in 
       (point, (append prefix (item . (tl suffix)))))
;;

let modify_static_remove =
 modify_static_ap 
   (\point l. 
     let (prefix, suffix) = split point l in 
       (point, (append prefix (tl suffix))))
;;



%let peek_static point op static =
  nth point (map_isexpr_to_term_list op static)
;;
%

let get_static point op static =
  (map_isexpr_to_term_list op (term_of_wrapped_term static))
;;

let peek_static point op static =
  (term_of_wrapped_term static)
;;

let modify_scroll_wop m scroll =
 let (id, [oid; offset; size; point]), [[],op; [],filter] = destruct_wrapped_term scroll in
   if not (id = `!scroll_list`) then failwith `peek_scroll_filter`;

   let n = (destruct_natural_parameter point) in
   let (r, e) = (m (operator_of_term op) n) (term_of_wrapped_term filter) in
    (make_term (id, [oid; offset; size; make_natural_parameter r]) 
	        [[],op; [],e])
;;

let modify_scroll m scroll =
 let (id, [oid; offset; size; point]), [op; [],filter] = destruct_wrapped_term scroll in
   if not (id = `!scroll_list`) then failwith `peek_scroll_filter`;

   let n = (destruct_natural_parameter point) in
   let (r, e) = (m n) filter in
    (make_term (id, [oid; zero_parm; size; make_natural_parameter r]) 
	        [op; [],e])
;;

let scroll_list_push_static item scroll scrolled = modify_scroll_wop (modify_static_push item) scroll;;
let scroll_list_replace_static item scroll scrolled = modify_scroll_wop (modify_static_replace item) scroll;;
let scroll_list_set_static list scroll scrolled = modify_scroll_wop (modify_static_set list) scroll;;
let scroll_list_remove_static scroll scrolled = modify_scroll_wop modify_static_remove scroll;;

let scroll_list_static_ap f = modify_scroll_wop (modify_static_ap f);;

let peek_scroll_wop f scroll  =
 let (id, [oid; offset; size; point]), [[],op; [],filter] = destruct_wrapped_term scroll in
   if not (id = `!scroll_list`) then failwith `peek_scroll_filter`
   else (f (destruct_natural_parameter point) (operator_of_term op) filter)
;;

let scroll_list_get_static scroll scrolled  = peek_scroll_wop (get_static) scroll;;
let scroll_list_peek_static scroll scrolled  = peek_scroll_wop (peek_static) scroll;;


%

;;;;	Dyneval
;;;;	
;;;;	
;;;;	


  Scroll select
%

let nav_oid_term oid pos =
  make_term (`!navigator_oid`, [make_object_id_parameter oid; make_natural_parameter pos]) []
;;

let pos_of_nav_oid_term t =
  (let ((id, [oid; pos]), bts) = destruct_term t in
    destruct_natural_parameter pos)
  ? 0
;;

let idag_root = make_term (`!dag_root`,[]) [];;


let modify_oid_stack m stack =
  let (ops, [cons; [],l]) = destruct_wrapped_term stack in
  if not ((fst ops) = `!oid_stack`) then failwith `modify_oid_stack`;

  let (r, s) = m (operator_of_term (snd cons)) l in
    (r, make_term ops [cons; [],s])
;;

let oid_stack_push oid pos = 
  modify_oid_stack 
   (\op l. 
	0,make_term op [[],(nav_oid_term oid pos); [],l])
;;

let oid_stack_pop =
  modify_oid_stack
   (\op stack.
      let l = (map_isexpr_to_list op (\x.x) stack) in
	( pos_of_nav_oid_term (hd l)
	, if (l = [] or (tl l) = []) 
	     then idag_root
	     else (map_to_ilist (\x.x) op (tl l)))
	)
;;


let modify_filter_arg m filter =
 let (opf, [[],arg; e]) = destruct_wrapped_term filter in
  let (r,a) = m arg in
   r,make_term opf [[],a; e]
;;    
   
let modify_dyneval_expression m dyneval =
 let (opd, (cond . (([],expr) . rest))) = destruct_wrapped_term dyneval in
  if not ((fst opd) = `!dyneval`) then failwith `modify_dyneval_expression`;
  let (r,e) = m expr in
   r,make_term opd (cond . (([],e) . rest))
;;
    

let peek_scroll_filter scroll  =
 let (id, parms), [[],op; [],filter] = destruct_wrapped_term scroll in
   if not (id = `!scroll_list`) then failwith `peek_scroll_filter`
   else filter
;;


let peek_dyneval_expression dyneval  =
 let (opd, (cond . (([],expr) . rest))) = destruct_wrapped_term dyneval in
   expr
;;

let peek_filter_arg filter =
 let (opf, [[],arg; e]) = destruct_wrapped_term filter in
   arg
;;    

let oid_stack_peek stack = 
  let (ops, [cons; [],l]) = destruct_wrapped_term stack in
  let op = (operator_of_term (snd cons)) in
    (map_isexpr_to_list op (\x.x) l)
;;

let scroll_dyneval_filter_oid_stack_peek = 
  oid_stack_peek 
      o (peek_filter_arg
	o (peek_dyneval_expression 
	    o peek_scroll_filter))
;;

let scroll_dyneval_filter_arg_peek = 
 peek_filter_arg o (peek_dyneval_expression o peek_scroll_filter)
;;

let scroll_root_p scroll =
  let l = scroll_dyneval_filter_oid_stack_peek scroll
  in 
    if l = [] then true
    else (identical_terms_p idag_root (hd l))
;;	     



%%% returns <oid{parent}> %
let scroll_directory_oid scroll =
 let l = scroll_dyneval_filter_oid_stack_peek scroll
 in
     if l = [] then failwith `scroll_directory`
     else first_oid_of_term (hd l)
;;


let total_of_scrolled t = destruct_natural_parameter (parameter_of_term t 1);;

let list_of_scrolled op scrolled = 
 let (spid, parms), [[],l] = destruct_wrapped_term scrolled in
   map_isexpr_to_list op (\x.x) l 
;;

letref debugss = 0,0;;
let scroll_selected scroll scrolled =
 let (id, [oid; offset; size; ppoint]), [[],op; [],af] = destruct_wrapped_term scroll in
 let op, bts = (destruct_wrapped_term op) in
   if not (id = `!scroll_list`) then failwith `scroll_select_point`
   else
    let n = (1 + (destruct_natural_parameter ppoint))
 	      - (destruct_natural_parameter offset) in
     debugss := n, (length (list_of_scrolled op scrolled));
     term_of_wrapped_term  (nth n (list_of_scrolled op scrolled))
;;

letref debugsfp = nil : term list;;
let scroll_find_pos p scroll scrolled = 
 let (id, [oid; offset; size; ppoint]), [[],op; [],af] = destruct_wrapped_term scroll in
 let op, bts = (destruct_wrapped_term op) in
 let l = (list_of_scrolled op scrolled) in
  debugsfp := (scroll . scrolled . l);
 ((destruct_natural_parameter offset) + (term_position (find (\i. p (term_of_wrapped_term i)) l) l) - 1)
;;

let scroll_selected_pos scroll scrolled =
 let (id, [oid; offset; size; ppoint]), [[],op; [],af] = destruct_wrapped_term scroll in
 let op, bts = (destruct_wrapped_term op) in
   if not (id = `!scroll_list`) then failwith `scroll_selected_pos`
   else destruct_natural_parameter ppoint
;;


let scroll_selected_oid scroll scrolled =
 let selected = (scroll_selected scroll scrolled ? failwith `scroll_selected_oid term`) in
  ( first_oid_of_term selected
  ? failwith `scroll_selected_oid`)

;; 


%%% returns <oid{parent}>, <tok{path}>, <oid{child}> %
let scroll_selected_addr scroll scrollpos =
 let point = scroll_selected scroll scrollpos
   in 
	( scroll_directory_oid scroll
	, first_tok_of_term point
	, first_oid_of_term point
	)
;;

let scroll_selected_path scroll scrollpos =
  let point = scroll_selected scroll scrollpos in
    (first_tok_of_term point, first_oid_of_term point)
;;


let modify_scroll_dyneval_filter_stack f  =

   modify_scroll
	(\pos input. modify_dyneval_expression
	                 (\filter. modify_filter_arg (f pos)
				      filter)
			 input)
;;


let scroll_select_pop scroll =
  modify_scroll_dyneval_filter_stack 
	(\pos stack. oid_stack_pop stack)
	scroll
;;


let scroll_select_remove scroll scrollpos =
 if (scroll_root_p scroll)
    then (let (path, oid) = scroll_selected_path scroll scrollpos in remove_root oid)
    else (let (poid, path, oid) = scroll_selected_addr scroll scrollpos in
	     if (directory_p oid)
		then remove_directory_tree poid path
		else remove_leaf poid path)
;;

let scroll_select_remove_leaf scroll scrollpos =
 if (scroll_root_p scroll)
    then (let (path, oid) = scroll_selected_path scroll scrollpos in remove_root oid)
    else (let (poid, path, oid) = scroll_selected_addr scroll scrollpos in
	     if (directory_p oid)
		then remove_directory_tree poid path
		else remove_leaf poid path)
;;
    

let scroll_select_delete scroll scrollpos =
 if (scroll_root_p scroll)
    then (let (path, oid) = scroll_selected_path scroll scrollpos in
	    remove_root oid)
    else (let (poid, name, a) = scroll_selected_addr scroll scrollpos in
	    delete_object_id poid name)
;;

let scroll_select_deactivate scroll scrolled =
  deactivate (scroll_selected_oid scroll scrolled);;

let scroll_select_activate scroll scrolled = 
  activate (scroll_selected_oid scroll scrolled);;

let scroll_select_activate_dir scroll scrolled = 
  map (\(a,b). (activate b) ? ()) (directory_children (scroll_selected_oid scroll scrolled));
  ()
;;


let scroll_status scroll scrolled = 
 let (id, [oid; offset; size; point]), [[],op; [],af] = destruct_wrapped_term scroll in
   ( destruct_natural_parameter size
   , destruct_natural_parameter offset
   , destruct_natural_parameter point
   , total_of_scrolled scrolled
   )
;;


let local_dir name = 
  (let lroot = (root `local`) in
      (descendent lroot [name]) ? (make_directory lroot name))
    ? (make_directory (make_root `local`) name)
;;


let system_obid names = 
  let sroot = (root `system`) in
      (descendent sroot names) 
;;

letref local_open_count = 0;;

let find_oid_in_dir poid oid =
 (find (\de. oid = (snd de)) (directory_children poid)) 
;;

let oid_in_dir_p poid oid =
 ((find (\de. oid = (snd de)) (directory_children poid); true) ? false)
;;


let local_open_add oid =
 
 let lodir = (local_dir `open`) in

 if (oid_in_dir_p lodir oid)
    then ()
  else (
 
   local_open_count := local_open_count + 1;
   insert_object_id lodir
		    (string_to_tok
			     (concatenate_strings 
				[ (tok_to_string (name_property oid) ? "o")
				; "_"
				; int_to_string (utime())
				; "_"
				; int_to_string local_open_count]))
		    oid
    ; ())
;;


let local_open_delete oid =
 ((let lodir = (descendent (root `local`) ``open``) in
   let n = fst (find_oid_in_dir lodir oid) in
     delete_object_id lodir n)
  ; ())
 ? ()
;;


let dyn_navigator_clone term =
 let oid = create_object `TERM`
		[ `NAME`, itoken_term `Navigator`
		; `DYNAMICREFRESH`, ivoid_term
		; `EDITREFRESH`, itoken_term `Navigator`
		; `DESCRIPTION`, object_id_dag_description_term
		; `GEOMETRY`, map_to_ilist inatural_term icons_op [20; 20; 55; 20]
		]
             term 

  %
        insert? (local_dir `open`) 
		       (concat `Navigator` (int_to_tok (utime())))
  %

  in
     activate oid
   ; view_open oid
   ; ivoid_term
;;

letref open_helper = inl () : unit + (object_id -> bool);;
  
let scroll_select_view scroll scrollpos =
  let oid = scroll_selected_oid scroll scrollpos in
   if (if isr open_helper then (((outr open_helper) oid) ? false) else false) then ()
   else (view_open oid; ())
;;


let scroll_select_point scroll scrolled =
 let oid = scroll_selected_oid scroll scrolled in

   if not (directory_p oid)
      then (scroll_select_view scroll scrolled; scroll)
      else 
	modify_scroll_dyneval_filter_stack 
		(\pos stack. oid_stack_push oid pos stack)
		scroll
;;

let scroll_set_position offset position scroll scrolled =
 %tty_print (concatenate_strings ["ssp "; int_to_string offset;  int_to_string position]);%
 let (id, [oid; poffset; size; ppoint]), bts = destruct_wrapped_term scroll in
   make_term (id, [oid; make_natural_parameter offset; size; make_natural_parameter position])
     bts
;;

let scroll_set_size f scroll scrolled =
 let (id, [oid; poffset; size; ppoint]), bts = destruct_wrapped_term scroll in
   make_term (id, [oid; poffset;
		   make_natural_parameter (f (destruct_natural_parameter size)); ppoint])
     bts
;;


%
;;;;	
;;;;	
;;;;	<scroll>	: !scroll_list{<oid>:o>, <offset:n>, <size:n>, <point:n>}
;;;;				(<cons_op>; <filter>)
;;;;	
;;;;	
;;;;   
;;;;	modify_oid_stack	: <op{oid_cons}> -> term{oid ilist} -> (* # term{oid ilist})
;;;;					-> term{oid stack} -> * # term{oid stack}
;;;;	
;;;;	modify_filter_arg	: (term{arg} -> int # term{arg}) -> term{filter} -> int{point} # term{filter}
;;;;	
;;;;	modify_scroll	: (int{point} -> term{filter} -> (int{point} # term{filter})) 
;;;;				-> term{scroll} -> term{scroll}
;;;;	modify_scroll_dyneval_filter_stack 
;;;;	
;;;;	
%

let scroll_select_path scroll point oidstack  % int#oid list % =
  modify_scroll_dyneval_filter_stack 
		(\pos stack. 
		   modify_oid_stack 
		   (\op l. (point, 
	  		    oidstack))
		    stack)
		scroll
;;

let evalet_property oid =
  (bool_of_ibool_term (object_state_property `evalet` oid))
  ? false ;;

let scroll_select_eval scroll scrolled = 
 let oid = scroll_selected_oid scroll scrolled in
  
   if (evalet_property oid) then
      lib_eval (oid_ap (begin_ap "lib_eval_term ") oid)
   else failwith `eval_not`
;;

let evalet_obid point =
 let oid = first_oid_of_term point in
  
   if (evalet_property oid) then
      lib_eval (oid_ap (begin_ap "lib_eval_term ") oid)
   else failwith `eval_not`
;;
%
let import_theory_update_obid point =
 let oid = first_oid_of_term point in
  
   if (evalet_property oid) then
      lib_eval (oid_ap (begin_ap "lib_eval_term ") oid)
   else failwith `eval_not`
;;

%

let dyn_scroll_select_view dyn = 
  (let (id, p), [c; [],e; s; [],l] = destruct_wrapped_term dyn in
    if not (id = `!dyneval`)
 	then failwith `dyn_scroll_select_view`
	else scroll_select_view e l)
  ; ivoid_term
;;

letref dyndebug = ivoid_term;;

let dyn_scroll_apply command dyn =
  let (id, p), [c; [],e; s; [],l] = destruct_wrapped_term (dyndebug := dyn) in
    if not (id = `!dyneval`) then failwith `dyn_scroll_apply`
    else command e l
;;


let dyn_scroll_modify command dyn =
  let (id, p), [c; [],e; s; [],l] = destruct_wrapped_term dyn in
    if not (id = `!dyneval`) then failwith `dyn_scroll_apply`
    else make_term (id, p) [c; [],command e l; s; [],l]
;;


let dyn_scroll_void_apply command dyn =
  dyn_scroll_apply command dyn
  ; ivoid_term
;;

let dyn_scroll_apply_unit command dyn =
  dyn_scroll_apply command dyn
  ; ()
;;

let dyn_scroll_select_pop	= dyn_scroll_modify (\e r. scroll_select_pop e);;

let dyn_scroll_select_deact_unit	= dyn_scroll_apply_unit scroll_select_deactivate;;
let dyn_scroll_select_act_unit		= dyn_scroll_apply_unit scroll_select_activate;;
let dyn_scroll_select_act_dir_unit		= dyn_scroll_apply_unit scroll_select_activate_dir;;

let dyn_scroll_select_deact	= dyn_scroll_void_apply scroll_select_deactivate;;
let dyn_scroll_select_act	= dyn_scroll_void_apply scroll_select_activate;;
let dyn_scroll_select_delete	= dyn_scroll_void_apply scroll_select_delete;;
let dyn_scroll_select_remove	= dyn_scroll_void_apply scroll_select_remove;;
let dyn_scroll_select_eval	= dyn_scroll_void_apply scroll_select_eval;;

let dyn_scroll_select_point	= dyn_scroll_modify scroll_select_point;;

let dyn_scroll_selected_oid	= dyn_scroll_apply scroll_selected_oid;; 
let dyn_scroll_selected_pos	= dyn_scroll_apply scroll_selected_pos;; 
let dyn_scroll_selected_path	= dyn_scroll_apply scroll_selected_path;;
let dyn_scroll_directory_oid	= dyn_scroll_apply (\e r. scroll_directory_oid e);; 
let dyn_scroll_root_p		= dyn_scroll_apply (\e r. scroll_root_p e);; 

let null_token_term = (itoken_term (string_to_tok ""));;

let dyn_scroll_get_name t	= (itoken_term (name_property (dyn_scroll_selected_oid t))) ? null_token_term;;
let dyn_scroll_get_link_name t	= (itoken_term (fst (dyn_scroll_selected_path t))) ? null_token_term;;
let dyn_scroll_get_oid t	= ioid_term (dyn_scroll_selected_oid t) ? ivoid_term;;

let dyn_scroll_kill_view t	= (view_quit (view_of_oid (dyn_scroll_selected_oid t)) ? ()) ; ivoid_term;;



let dyn_scroll_peek_static	= dyn_scroll_apply scroll_list_peek_static;;
let dyn_scroll_get_static	= dyn_scroll_apply scroll_list_get_static;;
let dyn_scroll_set_static l	= dyn_scroll_modify (scroll_list_set_static l);;
let dyn_scroll_remove_static	= dyn_scroll_modify scroll_list_remove_static;;
 
let dyn_scroll_static_ap f	= 
  dyn_scroll_modify (\scroll x. (scroll_list_static_ap f scroll))
;;

let dyn_scroll_static_list_ap f	= 
  dyn_scroll_static_ap (\p l. 0,(f l))
;;

let wm_omerge t = lib_eval (token_ap (begin_ap "\\tok. overwriting_merge [tok]") t);;
let wm_smerge t = lib_eval (token_ap (begin_ap "\\tok. shadowing_merge [tok]") t);;
let wm_lmerge t = lib_eval (token_ap (begin_ap "\\tok. shadowing_merge_lite [tok]") t);;


let prefixplus a b = a + b;;
let prefixsubtract a b = a - b;;

letref offset_ratio = (10,3);;

let scroll_minmax_offset offset pos size =
 let delta = ((snd offset_ratio) * size) / (fst offset_ratio) in
 let minoff =  min pos (max 0 (min offset (pos - delta))) in
  (max minoff (pos - (size - (1+delta))))
;;
	  
let new_scroll_position dis dir scroll scrolled =
 let op = if dir then prefixplus else prefixsubtract in
 let (win, off, pos, tot) = scroll_status scroll scrolled in
 let increment = (if dis = `line` then 1 
             else if dis = `halfpage` then win/2 
             else if dis = `page` then win
             else 0) in

 let newoff, newpos =
    if dis = `line` then (off, (op pos increment))
    else if (dis = `halfpage` 
             or dis = `page` )
            then (op off increment , (op pos increment))
    else if dis = `end` 
            then (if dir
		    then (tot - win), (tot - 1)
                    else (0,0))
    else off,pos
 
  in

  %
  tty_print (concatenate_strings ["dss "; int_to_string newoff; " "; int_to_string newpos;
				  " "; int_to_string win;
				  " "; int_to_string tot;
				  tok_to_string dis
	    ]);
  %
  let nnoff = (min (max 0 newoff) (max (tot - win) 0))
  and nnpos = (min (max 0 newpos) (max (tot - 1) 0)) in
  scroll_set_position
    (scroll_minmax_offset nnoff nnpos win)
    nnpos
    scroll
    scrolled
   
;;

let new_scroll_position_jump_aux adj_off npos scroll scrolled =
 let (win, off, pos, tot) = scroll_status scroll scrolled in
 let nnpos  = (min (max 0 npos) (max (tot - 1) 0)) in
 % only change offset if necessary to make nnpos visible. %
 let offset = if adj_off then (min (max 0 off) (max (1 + (tot - win)) 0))
              else (min (max 0 off) (max 0 (tot - 1))) in 
  scroll_set_position
    (if adj_off then scroll_minmax_offset offset nnpos win
        else offset)
    nnpos
    scroll
    scrolled
;;

let new_scroll_position_jump = new_scroll_position_jump_aux false;;


let dyn_scroll_scroll distance direction = dyn_scroll_modify (new_scroll_position distance direction);;


let dyn_scroll_set_position off pos = dyn_scroll_modify (scroll_set_position off pos);;

let dyn_scroll_set_size f = dyn_scroll_modify (scroll_set_size f);;

let dyn_scroll_find_link oid name = 
 dyn_scroll_apply
   ( scroll_find_pos (\t. ( (equal_oids_p oid (first_oid_of_term t))
                          & name = (first_tok_of_term t))
		          ? false))
;; 

let dyn_scroll_find_oid_pos oid = 
 dyn_scroll_apply
   ( scroll_find_pos (\t. (equal_oids_p oid (first_oid_of_term t))
		          ? false)) 
;; 

let dyn_scroll_link_aux adj_off name oid t =                            
 dyn_scroll_modify 
   (new_scroll_position_jump_aux adj_off (dyn_scroll_find_link oid name t))
   t
;;  

let dyn_scroll_link = dyn_scroll_link_aux false;;


let raise_openlist term =
  view_open (system_obid ``view filters open``)
  ; ivoid_term;;
  
let openlist_open () =
   let lodir = (local_dir `open`) in
      map view_open (map snd (directory_children lodir))
      ; ()
  ? ()
;;

let openlist_kill () = 
   let lodir = (local_dir `open`) in
	     map (\c . view_quit (view_of_oid (snd c))) (directory_children lodir)
	     ; ()
   ? ()				  
;;



 							      
letref evaluable_wrap_cache = inl () : unit + term;;

let evaluable_wrap_term () =
  outr
   (if (isr evaluable_wrap_cache)
       then evaluable_wrap_cache
       else evaluable_wrap_cache :=
          (inr ( (term_lookup (descendent_s [`system`; `support`; `templates`; `evaluable buttons`]))
	       ? (term_lookup (descendent_s [`system`; `support`; `show`; `templates`; `evaluable buttons`]))
	       ? failwith `evaluable_wrap_term`
               )))
;;

let evaluable_wrap t = replace_term 2 (evaluable_wrap_term()) t;;


let lit_wrap t = make_term (`!text_term_literal`,nil) [nil,t];;

% todo : want to change link name if current obj and link name same. %
% todo : MvLink. %
let build_rnobj_but nav t =
  lit_wrap 
  (evaluable_wrap
   (make_term (`!name_change`,
	       [  (make_string_parameter
		   (tok_to_string
		     (name_property (dyn_scroll_selected_oid nav))))
		 ?  string_slot
		 ])
             [([],t)]))
;;


let dyn_cpobj dir name oid term =
  let place = token_of_itoken_term (dyn_scroll_get_link_name term) in
   copy_object_after dir place name oid
;;

let dyn_mkobj_props kkind dir =
  let lang,kind = if kkind = `ML` then (`ML`,`CODE`)
             else if kkind = `LISP` then (`LISP`,`CODE`)
	     else if kkind = `CODE` then (`ML`, `CODE`)
             else (null_token,kkind) in
  let props1 = if      (member kind ``DISP PREC``)    then [`DESCRIPTION`, nuprl5_edit_description_term] 
               else if (member kind ``CODE STM ABS``) then [`DESCRIPTION`, nuprl5_refiner_description_term]
               else [] in
  
    kind, (if (`CODE` = kind) then props1 @ [`LANGUAGE`, (itoken_term lang)]
            else if (`STM` = kind)  then (props1 @ [`theory`,   (ioid_term dir)])
            else props1)
;;

	       
let dyn_mkobj kind place dir name =
  let kkind, props =  (dyn_mkobj_props (tok_upcase kind) dir) in
    lib_mkobj dir place kkind name props
;;
		  
% returns true if rm'ing duplicate which is before place we are inserting at. % 
let dyn_mklink poid place name oid rmdup =
 lib_eval_to_bool
   (bool_ap (token_ap (token_ap
    		        (oid_ap (oid_ap (begin_ap "lib_mklink ") poid) oid)
	            place) name)
	    rmdup)
;;

%
letrec unquote q t = 
 let op =  ((let ((id, [qu]), bts) = destruct_term t in 
	       if (q = (destruct_quote_parameter qu)) then (id, nil) else fail)
	    ? operator_of_term t) in
  
   make_term op (map (\b,t. b, (fu t)) (bound_terms_of_term t))
;;
%

let dyn_scroll_filter_oid_stack t = 
  let ll = (dyn_scroll_apply (\e r. scroll_dyneval_filter_oid_stack_peek e) t) in
   map oid_of_ioid_term (firstn ((length ll) - 1) ll)
;;

let dyn_scroll_filter_arg_term t = 
  (dyn_scroll_apply (\e r. scroll_dyneval_filter_arg_peek e) t)
;;

 		  

%  oid and index are point in search to be returned on next search call.
   they are not related to position in dir 
%
letref nj_debug = ivoid_term;;
letref nj_debug_nav = ivoid_term;;
letref nj_debug_path = ``fu``;;

%
;;;;	
;;;;	<oid_stack>	: !dag_root() 
;;;;			| !oid_cons(!navigator_oid{<oid>:o, <pos:n>}; <oid-stack>)
;;;;	
;;;;	  * pos is position of oid in parents(ie next on stack) directory.
;;;;	

;;;;	tok path is list of toks from leaf to root.
;;;;	
;;;;	result should be oid_stack plus position of root in top dir.
%

let inavigator_oid_term oid pos =
  make_term (`!navigator_oid`, [make_object_id_parameter oid; make_natural_parameter pos]) nil
;;

let ioid_stack_bot = (ioid_cons_term idag_root (make_term ioid_cons_op nil));;

let tok_path_to_oid_stack_aux toks =
  letrec aux toks = 
     if (null (tl toks)) 
         then (let pos = (position (hd toks) (map fst (roots()))) - 1 in
	       let oid = (root (hd toks)) in
		 ( ioid_cons_term (inavigator_oid_term oid pos)
				  ioid_stack_bot
		 , oid))
	 else (let (tail, poid) = aux (tl toks) in
	        let (pos, oid) = child_and_position poid (hd toks) in
		 ( ioid_cons_term (inavigator_oid_term oid pos) tail)
		 , oid)
   in
   aux toks
;;
let tok_path_to_oid_stack_path toks =
   if (null toks) then ioid_stack_bot
   else fst (tok_path_to_oid_stack_aux toks)
;;

let tok_path_to_oid_stack toks =
   if (null toks) then (0, ioid_stack_bot)
   else if (null (tl toks)) then ((position (hd toks) (map fst (roots()))) - 1, ioid_stack_bot) 
   else let stack, poid = tok_path_to_oid_stack_aux (tl toks) in 
          (fst (child_and_position poid (hd toks)), stack)
;;

let oid_stack_to_tok_path oids = 
 letrec aux oids = 
  if null (tl oids) 
     then [root_name (hd oids)]
     else ((child_name (hd (tl oids)) (hd oids)) . (aux (tl oids)))
 in

  if (null oids) then [] 
  else aux oids
;;      

let nav_set_path point path nav =
    nj_debug := path;
  dyn_scroll_modify (\e l. scroll_select_path e point path) nav
;;

let navigator_jump nav path = 
  %nj_debug_nav := nav;  nj_debug_path := path;%
  let point, ioidstack = tok_path_to_oid_stack path in
       nav_set_path point ioidstack nav
;;



let jump_set_dir_forward term = 
  let (id, (oid . wp . gp . tp . dp . rp)), bts = destruct_term term in
   make_term (id, (oid . wp . gp . tp . (make_bool_parameter true) . rp)) bts
;;

let jump_set_dir_backward term = 
  let (id, (oid . wp . gp . tp . dp . rp)), bts = destruct_term term in
   make_term (id, (oid . wp . gp . tp . (make_bool_parameter false) . rp)) bts
;;


%
;;;;	
;;;;	Navigator button functions.
;;;;	
;;;;	There are three main components to the navigator window:
;;;;	  - the navigator,
;;;;	  - the buttons,
;;;;	  - command templates.
;;;;	
;;;;	FTTB call the navigator + buttons + templates an xnavigator.
;;;;	
;;;;	
;;;;	A navigator button applys a function to the xnavigator.
;;;;	Such a function may return
;;;;	  - an updated xnavigator.
;;;;	  - an xnavigator with an additional template.
;;;;	  - unit, ie acts via side-effects.
;;;;	 or fail.
;;;;	
;;;;	Evaluation of a command template : 
;;;;	  - command is an abstraction expanding into ml expression or
;;;;	  - !apply{ML:t}(f; a)
;;;;	      * f will be called with a as arg.
;;;;		a should be an xnavigator, a will not be source reduced.
;;;;		f should return an xnavigator or fail.
;;;;	
;;;;	
;;;;	
;;;;	button_term : term {xnavigator} -> term {xnavigator}
;;;;	
;;;;	button_unit : term {navigator} -> unit
;;;;	  * note not an xnavigator.
;;;;	
;;;;	
;;;;	xnavigator ops :
;;;;
;;;;	nav_find	: term -> term{nav}
;;;;	  * access embedded nav.
;;;;	
;;;;	nav_replace	: term -> term{nav} -> term
;;;;	  * re-embed nav in term.
;;;;	
;;;;	nav_ap		: (term{nav} -> term{nav}) -> term -> term
;;;;	
;;;;	  * if two navs embedded in term then uses an arbitrary one.
;;;;	
;;;;	
;;;;	navigator access :
;;;;	
;;;;	nav_stack	: term -> object_id list
;;;;
;;;;	nav_oid		: term {Navigator} -> oid
;;;;	nav_dir		: term {Navigator} -> oid
;;;;	nav_name	: term {Navigator} -> tok	
;;;;	nav_link	: term {Navigator} -> tok	
;;;;	  * place!
;;;;	
;;;;	navigator update :
;;;;	
;;;;	nav_set_stack	: term -> oid list -> term
;;;;	nav_push	: term -> oid -> term
;;;;	nav_pop		: term -> term
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	buttons : map nav stack and look for .navigator-buttons in each
;;;;	directory, collect and display as buttons.
;;;;	
;;;;	allow for grouping of buttons. 
;;;;	
;;;;	tree-p : display buttons whenever navigator in tree.
;;;;	  true by default.
;;;;	
;;;;	``system view navigator buttons .buttons``
;;;;	
;;;;	
;;;;	button motion : 
;;;;	  - <<<< top
;;;;	  - <<< page up
;;;;	  - << half page up
;;;;	  - < line up
;;;;	  - >>>> bottom
;;;;	  - >>> page down
;;;;	  - >> half page down
;;;;	  - > line down
;;;;	  - .. out
;;;;	  - -- in
;;;;	
;;;;	nav_scroll : tok{distance} -> bool{direction} -> int{position} -> int{size} -> int{position}
;;;;	  - tok : line | halfpage | page | end 
;;;;	
;;;;	scroll_status : dyneval -> int{window size} # int{offset} # int{position} # int{total size}
;;;;	nav_scroll_set : navigator -> int{position}
;;;;	
%

let nav_stack		= dyn_scroll_filter_oid_stack;;
let nav_stack_term	= ((subtermn 2) o dyn_scroll_filter_arg_term);;
let nav_path nav	= oid_stack_to_tok_path (nav_stack nav);;
let nav_link nav	= (fst (dyn_scroll_selected_path nav));;
let nav_oid		= dyn_scroll_selected_oid;;
let nav_pos		= dyn_scroll_selected_pos;;
let nav_dir		= dyn_scroll_directory_oid;;
let nav_rootp		= dyn_scroll_root_p;;
let nav_name nav	= name_property (nav_oid nav);;


letref ndebug = ivoid_term;;
let nav_remove_object n =
 ndebug := n;
 if directory_p (nav_oid n)
    then (raise_error [nav_oid n] ``remove_object directory`` nil)
    else remove_leaf (nav_dir n) (nav_link n)
;;  

let nav_buttons_lookup defaults nav =
  letref default = inr (term_lookup (outr defaults)) ? inl () in
    %if isr defaults then (default := inr (term_lookup (outr defaults)); ());%
    let aux oid =
      ((default := inr (term_lookup (child oid `.buttons-navigator-default`)); ()) ?  ());
      let boid = child oid `.buttons-navigator` in
	 term_lookup boid
       in
    let bs = remove_prior_duplicates_p lex_equal_terms
              (map_omitting_failures aux (rev (dyn_scroll_filter_oid_stack nav))) in 
     rev (if isr default 
             then (outr default . bs)
             else bs)
;;       

let build_rmtree_but nav t =
 let name = nav_name nav in
 let oid = nav_oid nav in
 let poid = nav_dir nav in

 if not (directory_p oid)
   then raise_error [oid; poid] ``RemoveTree directory not`` [itoken_term name];

  lit_wrap 
  (evaluable_wrap
    (make_term (`!apbut`, [make_string_parameter "rmtree_but"])
     [[],( make_term (`!remove_tree`, 
		  [ make_token_parameter name
		  ; tok_slot])
		 [([],ioid_term oid); ([],ioid_term poid); ([],t)])]))
;;

let rmtree_but data = 
 let ack = (destruct_token_parameter (parameter_of_term data 2)) in
 if (`y` = ack or `Y` = ack) 
    then delete_tree
	   (oid_of_ioid_term (subterm_of_term data 2)) 
	   (destruct_token_parameter (parameter_of_term data 1))
    else ()
;;
 


%
;;;;	
;;;;	toploop : evaluator
;;;;	
;;;;	
;;;;	
%

let navigator_view_p v =
 ( `Navigator` = token_of_itoken_term (property_of_ostate `EDITREFRESH` (oid_of_view v)))
 ? false
;;

let evaluator_history_view_p v =
 ( `EvaluatorHistory` = token_of_itoken_term (property_of_ostate `EDITREFRESH` (oid_of_view v)))
 ? false
;;

let evaluator_view_p v =
 ( `Evaluator` = token_of_itoken_term (property_of_ostate `EDITREFRESH` (oid_of_view v)))
 ? false
;;


let evaluator_history_clone_aux makef =
 let oid = create_object `TERM`
		[ `NAME`, itoken_term `Evaluator History`
		; `GEOMETRY`, map_to_ilist inatural_term icons_op [20; 20; 80; 40]
		; `EDITREFRESH`, itoken_term `EvaluatorHistory`
		]
	     ivoid_term
  in

 let v = view_open oid in
 let t = open_evaluator_history_walk oid in

   % count on point being at top. %
   view_point_cut v;
   view_point_paste_term (makef oid t) v;

 v
;;

let evaluator_history_clone f = (evaluator_history_clone_aux f); ();;


let evaluator_history_peek t =
  evaluator_history_walk_peek 
    (first_oid_of_term (term_of_wrapped_term t))
;;

let evaluator_history_forward t =
 let (id, (oid . r)), bts = destruct_term t in
  make_term (id, (oid . (make_bool_parameter true) . (tl r))) 
   (([],evaluator_history_walk true (destruct_object_id_parameter oid)) . (tl bts))
;;

let evaluator_history_backward t =
 let (id, (oid . r)), bts = destruct_term t in
  make_term (id, (oid . (make_bool_parameter false) . (tl r))) 
   (([],evaluator_history_walk false (destruct_object_id_parameter oid)) . (tl bts))
;;


let find_evaluator x = x;;

let expr_of_evaluator x = subterm_of_term x 1;;
let history_of_evaluator x = subterm_of_term x 2;;
let outlen_of_evaluator x = destruct_natural_parameter (parameter_of_term x 3);;
let inlen_of_evaluator x = destruct_natural_parameter (parameter_of_term x 2);;
let history_point_of_evaluator x = destruct_natural_parameter (parameter_of_term x 1);;

let input_of_eval_history x = subterm_of_term x 1;;
let eval_history_op = (`!eval_history`, nil);;
let eval_history_term input output = 
  make_term eval_history_op 
    [ (nil, input)
    ; (nil, output)
    ]
;;

let eval_history_clean t = 
 (let (op, (in . out . r)) = destruct_term t in

   make_term op (in . (nil, ivoid_term) . r))
 ? t
;;

let remote_eval_term_p x = member (id_of_term x) ``!remote_ml_eval !remote_lisp_eval``;;

let eval_term_replace_input t n =
 letrec auxt t = 
  if ((remote_eval_term_p t) ? false)
     then replace_term 1 t (auxt (subterm_of_term t 1))
     else n

  in auxt t
;;

letref medebug = (ivoid_term,ivoid_term);;


let make_evaluator e r =
 medebug := (e,r);
 let l = map_isexpr_to_list double_break_cons_op (\x.x) (history_of_evaluator e) in
 let ex = expr_of_evaluator e in
   
   let keepers = (firstn ((inlen_of_evaluator e) - 1) l)  ? l in
   let outkeepers, inkeepers = split ((outlen_of_evaluator e) - 1) keepers ? (keepers, nil) in

   let eht = (eval_history_term ex r) in

   % new method, at some point may want to simplify evaluator to remove builtin history list%
   eval_history_push eht;
       
   let h = map_to_ilist (\x.x) double_break_cons_op 
	    ( (eval_history_term ex r)
	    . append outkeepers (map eval_history_clean inkeepers))
    in

   make_term (id_of_term e, (make_natural_parameter 0 . (tl (parameters_of_term e))))
    [ nil,(eval_term_replace_input ex (itext_term ""))
    ; nil, h
    ]
;;


let set_evaluator e ex hp =
   make_term (id_of_term e, (make_natural_parameter hp . (tl (parameters_of_term e))))
    [ nil, ex
    ; nil, history_of_evaluator e
    ]
;;

let eval_history_move newindex e = 
  let n = (history_point_of_evaluator e) in
  let il = (inlen_of_evaluator e) in 
  (let nn = newindex n il in
   let h = nth nn (map_isexpr_to_list double_break_cons_op (\x.x) (history_of_evaluator e))

    in

   set_evaluator e (input_of_eval_history h) nn
   ) ? e
;;

let eval_history_reset e = 
  set_evaluator e
    (eval_term_replace_input (expr_of_evaluator e)
			     (itext_term ""))
    0
;;

let evaluator_previous = 
  eval_history_move 
    (\n il. if n = il then fail; n+1)
;;

let evaluator_next = 
  eval_history_move 
   (\n il. if n = 0 then fail; n-1)
;;

let evaluator_history_top term = 
  view_show (subterm_of_term (history_of_evaluator term) 1)
;;
    
let evaluator_eval term = 
 let r = local_eval (expr_of_evaluator term) in
   make_evaluator term r
;;

let evaluator_save term = 
  make_evaluator term (ivoid_term)
;;


let evaluator_remove e =
  let n = (history_point_of_evaluator e) in
  let il = (inlen_of_evaluator e) in 
   let pre,post = split (n-1) (map_isexpr_to_list double_break_cons_op (\x.x) (history_of_evaluator e))
    in

    let newh = (append pre (tl post)) in
    let newn = (if n = 0 then 0 else (n-1)) in
    let newinput = (if newn = 0 
		       then (eval_term_replace_input (expr_of_evaluator e)
			       (itext_term ""))
		       else (input_of_eval_history (nth newn newh)))
      in

   make_term (id_of_term e, ((make_natural_parameter newn) . (tl (parameters_of_term e))))
    [ nil, newinput
    ; nil, map_to_ilist (\x.x) double_break_cons_op  newh
    ]
;;

%
 find evaluator with similar input
 replace input
 raise evaluator
%


%let subterm i t = subterm_of_term t i;;
defined earlier.
%

let description_of_evaluator_expression = subtermn 2;;


let expression_of_evaluator t =
 letrec auxbts bts =
   if (null bts) then fail
   else (auxt (snd (hd bts)) ? (auxbts (tl bts)))
 and auxt t = 
  if ((remote_eval_term_p t) ? false)
     then t
     else auxbts (bound_terms_of_term t)

  in auxt t
;;


% similar input -> same description %
let evaluator_similar_expression expr v = 
 (lex_equal_terms (description_of_evaluator_expression expr)
		  (description_of_evaluator_expression (expression_of_evaluator (view_term_q v)))
  ) ? false
;;


let raise_navigators () =
 map view_raise (filter navigator_view_p (views()));
 ()
;;

let raise_evaluators () =
 map view_raise (filter evaluator_view_p (views()));
 ()
;;


let evaluator_hijack evaluator_ap expr = 
 let evaluators = filter (\v. evaluator_view_p v & evaluator_similar_expression expr v) (views()) in
 let v =  if (null evaluators) 
	     then fail
	     else (hd evaluators)
 in
  
   let nterm = evaluator_ap (\e. set_evaluator e expr 0) (view_term_q v) in

   edit_remove_label point_tag v;
   edit_tag_address point_tag nil true v;
   view_point_cut v;
   view_point_paste_term nterm v;

   view_oid (oid_of_view v);

 ()
;;

 
 
let maybe_putedesc ooid =
  let kind = object_state_kind ooid in
  if (`DISP` = kind) then putedesc ooid;
  if (`PREC` = kind) then putedesc ooid
;;

let dump_patchdir i = export_tree (make_tmp_filename ["patches"] ("patchdir" J (int_to_string i)) "")
				  ``system patches patchdir``;; 
let dump_patchlist_aux oid i = 
  export_list (make_tmp_filename ["patches"] ("patchlist" J (int_to_string i)) "")
    (map_isexpr_to_list ioid_cons_op oid_of_ioid_term
      (oed_filter (get_term oid)))
;;
let dump_patchlist i = dump_patchlist_aux (descendent_s ``system patches patchlist``) i;;


let load_patchdir i = read_patch (make_tmp_filename ["patches"] ("patchdir" J (int_to_string i)) "");;
let load_patchlist i = read_patch (make_tmp_filename ["patches"] ("patchlist" J (int_to_string i)) "");;

let view_show_oids oids = view_show (map_to_ilist (\oid. ioid_term oid) ioid_cons_op oids);;
