
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	Window Title :
;;;;	
;;;;	
;;;;	<title>		: <view-info> @ <env> @ <host>
;;;;	  - of could be element sign? but better to leave ascii
;;;;	
;;;;	<view-info>	: <status> <type>: <name>
;;;;	
;;;;	<status>	: Constant length sequence of T and F
;;;;	  - modified : since last save or load
;;;;	  - active   :
;;;;	
;;;;	<type>		: Constant width type of object (ie PREC DISP ABS DIR TERM ... ) or NONE.
;;;;	
;;;;	<icon-title>	: <name> @ <env> : <status> @ <host>
;;;;	
;;;;	environment-name ()			: STRING
;;;;	
;;;;	view-set-window-title (<view>)		: unit
;;;;	
;;;;
;;;;	xwin-set-window-title(<window> STRING{window} STRING{icon})	: (values)
;;;;	  @ tacks on @ <host> and updates X.
;;;;	
;;;;	xwin-set-x-title(<x-window> STRING{window} STRING{icon})	: (values)
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	* need to detect status change in order to keep title current.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	



;--
;-- prl()
;--
;--   Main routine of PRL system:
;--     a read-eval-print loop in which chars read
;--     from the window interface are either processed
;--     locally or directed to the appropriate system
;--     module.
;--

(defvar *prl-running* nil)

(defun do-x-input (ch)
  (when ch
    (cond 
      ((or (null *views*) (null (current-view)))
       (display-message '#.(istring (format-string "No active view (ignored).~%"))))

      (t (receive-edit-char (normalized-char->edit-char ch))))))

(defun get-x-input (&optional (block-p t))

  (let* ((ch (get-character block-p)))
    ;;(format t "get char returned ~a ~a~%" block-p ch)

    ;; or could have bus-wait call prl loop which exits after nil char.
    ;;(orb-request-one)

    (do-x-input ch)

    (and ch t)))


(defun get-input ()
  (get-x-input))


(defun get-input-aux ()
  (get-input))



;;; --------------------------------------------------------------------------

(defun edit-mouse-blip (code state)
  (let ((ch (case code
	      (1  'mouse-left)
	      (2  'mouse-middle)
	      (3  'mouse-right)
	      (otherwise nil)))
	(modifiers (modifiers-of-xevent-state state nil)))
    
    (when ch
      (if modifiers
	  (cons modifiers ch)
	  ch))))

(defun menu-size-event (&rest r)
  (declare (ignore r))
  (break "todo"))


(defun recompute-window-contents-and-redraw (window)
  ;; prl wants to set the window's position, but it really doesn't care,
  ;; as long as the args yield the right width and height.
  (multiple-value-bind (x y) (window-coordinates window)
    (menu-size-event window nil		  ;second arg is obsolete
		     y x
		     (1- (+ y (height-of-oed-window window)))
		     (1- (+ x (width-of-oed-window window))))))

;;; a-list with keys of type (modifier list) # character
(defvar *key-bindings* nil)
(defvar *initial-key-bindings* nil)

;;; Case and char-bits are ignored.
(defun key-binding (modifiers-and-character)
  (let ((ch (cdr modifiers-and-character))
	(mods (car modifiers-and-character)))
    ;;(setf a ch b mods) (break "kb")
    (find-first #'(lambda (binding)
		    (let ((y (car binding)))
		      (when (and (set-equal mods (car y))
				 (if (characterp ch)
				     (and (characterp (cdr y))
					  (char= ch (cdr y)))
				     (eql ch (cdr y))))
			binding)))
		*key-bindings*)))



;;; A modifier is a member of '(:control :meta :alt :shift).
(defun define-key-binding (modifier-or-modifiers character ichar)
  (let ((modifiers (if (listp modifier-or-modifiers)
		       modifier-or-modifiers
		       (list modifier-or-modifiers))))
    (when (and (null modifiers)
	       (characterp character)
	       (standard-char-p character)
	       (graphic-char-p character))
      (error "Can't redefine standard printing characters (~A in this case)." character))
    (let ((old-binding (key-binding (cons modifiers character))))
      (if old-binding
	  (setf (cdr old-binding) ichar)
	  (push (cons (cons modifiers character)
		      ichar )
		*key-bindings*))))
  (values))

(defun modifiers-of-xevent-state (state ch)
  ;; note that order is important for equal check on edit side.
  ;; order should be alphabetical on keyword generated.
  (append (unless (zerop (logand #.(xlib:make-state-mask :mod-2)
				 state))
	    '(:alt))
	  (unless (zerop (logand #.(xlib:make-state-mask :control)
				 state))
	    '(:control))
	  (unless (zerop (logand #.(xlib:make-state-mask :mod-1)
				 state))
	    '(:meta))
	  (unless (or (zerop (logand #.(xlib:make-state-mask :shift)
				      state))
		      (not (shift-modifiable-char-p ch)))
	    '(:shift))))

(defun shift-modifiable-char-p (ch)
  (or (not (characterp ch))
      (not (graphic-char-p ch))
      (eql #\space ch)))

(defvar *show-keypress* nil)

(defun keycode->normalized-char (keycode state)
  (declare (special *display*))
  (let* ((unmodified-state ;;(logandc2 state
	  ;;#.(logior (xlib:make-state-mask :control)
	  ;;(xlib:make-state-mask :mod-1)
	  ;;(xlib:make-state-mask :mod-2)))
			   (logand state
				   (xlib:make-state-mask :shift)))
	 (ch (or (let ((ch (xlib:keycode->character *display* keycode unmodified-state)))
		   (and (characterp ch) ch))
		 (let ((keysym (xlib:keycode->keysym *display* keycode unmodified-state)))
		   (keysym->keych keysym))))
	 (modifiers (modifiers-of-xevent-state state ch)))

    (when *show-keypress*
      (format t "keycode: ~a keysym: ~X char: ~a(~a) :modifiers ~a~%"
	      keycode (xlib:keycode->keysym *display* keycode unmodified-state)
	      ch (when (characterp ch) (char-code ch))
	      modifiers))

    ;; kludge for linux which for some reason is not using the xmodmap in xlib:keycode->character
    ;; picking up the xmodmap
    ;;(when (char= ch #\backspace) (setf ch #\Rubout))
    (if ch
	(if modifiers
	    (cons modifiers ch)
	    ch)
	nil)))
	
	
(defun normalized-char->edit-char (ch)
  (cond
    ((and (consp ch) (characterp (cdr ch)))
     (cons (car ch) (char->ichar (char-upcase (cdr ch)))))
    ((characterp ch) (char->ichar ch))
    (t ch)))


(defun ichar->font-index (ch)
  (if (numberp ch)
      ch
      ispace))

(defun font-index->char-or-string (i)
  (ichar->char-or-string i))

(defun font-index->char-for-latexize (i)
  (ichar->char-for-latexize i))



(defun do-snapshot ()
  (let ((w (oed-current-focus)))
    (when w (snapshot-window w))))

(defun initialize-application-variables ()
  (declare (special *display* SCRWIDTH SCRHEIGHT))
  (let ((screen (xlib:display-default-screen *display*)))
    (setf SCRWIDTH (X-width->width (xlib:screen-width screen))
	  SCRHEIGHT (X-height->height (xlib:screen-height screen)))))

(defun reset-application ()
  (reset-prl))

;;; ------------------------------------------------------------------------

(defvar *character-X-width*)
(defvar *line-X-height*)
(defvar *baseline-X-height*)
(defvar *border-X-width*)
(defvar *margin-X-width*)
(defvar *interline-X-space*)
(defvar *vertical-X-offset*)
(defconstant *window-minimum-width* 1)
(defconstant *window-minimum-height* 1)

(defvar *X-window-output-suppressed?* nil)

(defvar *display* nil)
(defvar *gcontext*)
(defvar *reverse-gcontext*)
(defvar *foreground*)
(defvar *background*)
(defvar *cursor-font*)
(defvar *cursor*)
(defvar *wait-cursor*)
(defvar *text-cursor-font-index*)
(defvar *blob-cursor-font-index*)
(defvar *null-cursor-font-index*)

(defun width->X-width (w)
  (* w *character-X-width*))

(defun height->X-height (h)
  (* h *line-X-height*))
(defun X-width->width (w)
  (truncate w *character-X-width*))
(defun X-height->height (h)
  (truncate h *line-X-height*))
 
(defvar *init-file-loaded?* nil)

;(deftype font-index () 'fixnum)


;;; The following breaks Lucid's production compiler.
;(proclaim '(inline width->X-width height->X-height X-width->width X-height->height))

(defvar *options* 
  (list :font-name "nuprl-13"
	:cursor-font-name "cursor" :cursor-font-index 68
	:wait-cursor-font-index 150
	:text-cursor-font-index 199
	:blob-cursor-font-index 203
	:null-cursor-font-index 200
	:frame-left 30 :frame-right 98
	:frame-top 30 :frame-bottom 98
	:no-warp? t
	:backing-store? t
	:host-name-in-title-bars? t
	:title-bars? nil
	:foreground-color "black"
	:background-color "white"
	:resource "Nuprl"))

(defun typecheck-options (options)
  (unless (null options)
    (let ((option (first options)) (v (second options)))
      (unless (case option
		((:host :font-name :cursor-font-name
			:foreground-color :background-color
			:resource)
		 (stringp v))
		((:cursor-font-index :display :wait-cursor-font-index
				     :text-cursor-font-index
				     :blob-cursor-font-index
				     :null-cursor-font-index)
		 (and (numberp v) (<= 0 v)))
		((:frame-left :frame-right :frame-top :frame-bottom)
		 (and (numberp v) (<= 0 v 100)))
		((:no-warp? :title-bars? :host-name-in-title-bars? :backing-store?)
		 (or (null v) (eql v t)))
		(otherwise
		 (error "TypeCheckOption : Option ~a is unknown.~%" option)))
	(error "The value ~A specified for option ~A is of the wrong type."
	       v option)))
    (typecheck-options (cddr options))))


(defun check-frame-options (options)
  (let ((l (getf options :frame-left))
	(r (getf options :frame-right))
	(top (getf options :frame-top))
	(bot (getf options :frame-bottom)))
    (unless (and (<= 0 l (+ l 20) r 100)
		 (<= 0 top (+ top 20) bot 100))
      (error "Frame parameters must be at least 20 apart, etc."))))

(defvar *effected-options* nil)

(defun check-options ()
  (typecheck-options *options*)
  (check-frame-options *options*)
  (flet ((important-options (options)
	   (mapcar #'(lambda (option) (getf options option))
		   '(:font-name :cursor-font-name
		     :cursor-font-index :wait-cursor-font-index
		     :text-cursor-font-index :blob-cursor-font-index :null-cursor-font-index
		     :frame-left :frame-right :frame-top :frame-bottom
		     :title-bars? :backing-store
		     :foreground-color :background-color :resource))))
    (unless (equal (important-options *options*)
		   (important-options *effected-options*))
      (oed-x-reset))))



(defun split-list (l)
  (if (null l) 
      (values () ())
      (multiple-value-bind (x y)
	  (split-list (cddr l))
	(values (cons (car l) x) (cons (cadr l) y)))))

(defun change-options (&rest plist)
  (multiple-value-bind (l1 l2)
      (split-list plist)
    (mapc #'(lambda (x y) (setf (getf *options* x) y))
	  l1 l2)))

;;; For use in user init files.
(defun get-option (option)
  (getf *options* option))


;; map new config style to old option style.
(defun change-config-options ()
  (let ((options (mapcan #'(lambda (config)
			     (case (car config)
			       (background (list :background-color (cadr config)))
			       (foreground (list :foreground-color (cadr config)))
			       (font  (list :font-name (cadr config)))
			       ))
			 (remove-duplicates (config-data-all) :key #'car :from-end t))))
    (apply #'change-options options)
    ))



;;; For internal use only.
(defun option (option)
  (getf *effected-options* option))

(defvar *output-buffer*
	(make-array 200 :element-type 'fixnum :initial-element
		    (ichar->font-index ispace)
		    :adjustable t))

(defun adjust-output-buffer (new-width)
  (when (> new-width (array-dimension *output-buffer* 0))
    (adjust-array
      *output-buffer*
      (+ (array-dimension *output-buffer* 0) 50)
      :element-type 'fixnum
;      :element-type 'font-index
      :initial-element (ichar->font-index ispace))))

(defun contract-geometry-into-frame (x y width height)
  (let* ((screen (xlib:display-default-screen *display*))
	 (screen-X-width (xlib:screen-width screen))
	 (screen-X-height (xlib:screen-height screen))
	 (horizontal-factor (/ (- (option :frame-right) (option :frame-left))
			       100.0))
	 (vertical-factor (/ (- (option :frame-bottom) (option :frame-top))
			     100.0))
	 (frame-x (X-width->width
		    (truncate (* screen-X-width (option :frame-left)) 100)))
	 (frame-y (X-height->height
		    (truncate (* screen-X-height (option :frame-top)) 100))))
    (values (+ frame-x (truncate (* x horizontal-factor)))
	    (+ frame-y (truncate (* y vertical-factor)))
	    (truncate (* width horizontal-factor))
	    (truncate (* height vertical-factor)))))

(defun X-size->size (width height)	  
  (values (X-width->width (- width (* 2 *margin-X-width*)))
	  (X-height->height (- height *vertical-X-offset* *margin-X-width*))))

(defun geometry->X-geometry (x y width height)
  (values (- (width->X-width x) *margin-X-width* *border-X-width*)
	  (- (height->X-height y) *vertical-X-offset* *border-X-width*)
	  (+ (width->X-width width) (* 2 *margin-X-width*))
	  (+ (height->X-height height) *vertical-X-offset* *margin-X-width*)))


;;; coodinates within a window (as opposed to within the screen).
(defun X-coordinates->coordinates (x y)
  (values (X-width->width (- x *margin-X-width*))
	  (X-height->height (- y *vertical-X-offset*))))

(defun coordinates->X-coordinates (x y)
  (values (+ (width->X-width x) *margin-X-width*)
	  (+ (height->X-height y) *vertical-X-offset*)))

(defun xwindow-metrics (w)
  (mlet* (((foo parent) (xlib:query-tree w) (declare (ignore foo)))
	  ((x y) (x-coordinates->coordinates (xlib:drawable-x parent)
					     (xlib:drawable-y parent)))
	  ((w h) (x-coordinates->coordinates (xlib:drawable-width w)
					     (xlib:drawable-height w))))
	 (list (1+ x) (1+ y) (1- w) (1- h))))

;;; May be grossly wrong.  Use only for window placement heuristics.
(defun window-coordinates (window)
  (let ((w (top-x-of-oed-window window)))
    (xlib:with-state (w)
	(values (X-width->width
		 (+ (xlib:drawable-x w)
		    *margin-X-width* *border-X-width*))
		(X-height->height
		 (+ (xlib:drawable-y w)
		    *vertical-X-offset* *border-X-width*))))))

(defun check-and-open-font (display font-name default-font-name)
  (cond ((null (xlib:list-font-names display font-name))
	 (format t "~%Unknown font: ~A.  Using default: ~A.~%" 
		 font-name default-font-name)
	 (xlib:open-font display default-font-name))
	(t
	 (xlib:open-font display font-name))))

(defun create-X-window (X-x X-y X-width X-height)
  (let* ((screen (xlib:display-default-screen *display*))
	 (win (xlib:create-window
	       :parent (xlib:screen-root screen)
	       :x X-x :y X-y
	       :width X-width :height X-height
	       :background *background*
	       :border *foreground*
	       :border-width *border-X-width*
	       :colormap (xlib:screen-default-colormap screen)
	       :cursor *cursor*
	       :bit-gravity :forget
	       :backing-store (if (option :backing-store?)
				  :when-mapped
				  :not-useful)				  
	       :event-mask '(:exposure :button-press :key-press 
			     :substructure-notify :structure-notify :enter-window
			     :property-change))))
    (xlib:set-wm-class win (get-option :resource) "Nuprl")
    (setf (xlib:wm-name win) "")
    (setf (xlib:wm-normal-hints win)
	  (xlib:make-wm-size-hints :user-specified-position-p t
				   :user-specified-size-p t
				   :program-specified-position-p t
				   :x X-x :y X-y
				   :width X-width :height X-height
				   :min-width (width->X-width *window-minimum-width*)
				   :min-height (height->X-height
						*window-minimum-height*)))
    (setf (xlib:wm-hints win)
	  (xlib:make-wm-hints :input :on
			      :initial-state :normal))


    win))

(defun maybe-map (win)
  (format t "maybe-map~%")
  (when (or nil (eql :unmapped (xlib:window-map-state win)))
    ;;(xlib:unmap-window win)
    (format t "Unmapped~%")
    (xlib:map-window win)))
    
(defun oed-warp (w)
  ;;(setf -w w) (break "ow")
  (unless (or (eql w (xwin-event-window))
	      ;;(option :no-warp?)
	      )

    (let ((xwin (window-x-window w)))
      ;;(xlib:unmap-window window)
      ;;(xlib:map-window window)
    
      (xlib:warp-pointer xwin 1 ;; (- (xlib:drawable-width xwin) 10)
			 1))))

(defun edit-warp (state)
  (oed-warp (window-of-view state))
  state)

(defun xwin-lower (win)
  (multiple-value-bind (children parent root)
      (xlib:query-tree win)
    (declare (ignore children root))
    (setf (xlib:window-override-redirect parent) :on)
    (setf (xlib:window-priority parent) :bottom-if)
    (setf (xlib:window-override-redirect parent) :off)))

(defun xwin-raise (win)
  (multiple-value-bind (children parent root)
      (xlib:query-tree win)
    (declare (ignore children root))
    (maybe-map parent)
    (setf (xlib:window-override-redirect parent) :on)
    (setf (xlib:window-priority parent) :top-if)
    ;;(xlib:window-raise parent)
    ;;(xlib:circulate-window-up parent)
    (setf (xlib:window-override-redirect parent) :off)
  
    ;;(multiple-value-bind (children pparent root)
	;;(xlib:query-tree parent)
     ;; (setf (xlib:window-override-redirect pparent) :on)
     ;; (setf (xlib:window-priority pparent) :top-if)
     ;; (xlib:circulate-window-up pparent)
      ;;(setf (xlib:window-override-redirect pparent) :off))
    
    ))

  
(defun edit-raise (state)
  (let ((w (window-of-view state)))
    (xwin-raise (window-x-window w))
    )
  state)

(defun edit-lower (state)
  (let ((w (window-of-view state)))
    (xwin-lower (window-x-window w))
    )
  state)


(defun edit-focus (state)
  (oed-focus-on (window-of-view state))
  state)

(defun edit-swap-focus (state)
  (declare (ignore state))  ;; or should we focus on first
  
  (focus-swap)
  (oed-focus-on (car *oed-focus-stack*))
  (view-of-window (oed-current-focus)))

(defun edit-rotate-focus (state)
  (declare (ignore state))  ;; or should we focus on first

  (focus-rotate)
  (oed-focus-on (car *oed-focus-stack*))
  (view-of-window (oed-current-focus)))

(defun edit-reverse-rotate-focus (state)
  (declare (ignore state))  ;; or should we focus on first

  (focus-reverse-rotate)
  (oed-focus-on (car *oed-focus-stack*))
  (view-of-window (oed-current-focus)))


(defvar *current-mouse-cursor-waitp* nil)



(defun xwin-change-cursor (win waitp)
  (when win
    ;;(format t "changing a win cursor ~a~%" waitp)
    (if waitp
	(setf (xlib:window-cursor (window-x-window win)) *wait-cursor*)
	(setf (xlib:window-cursor (window-x-window win)) *cursor*))))

(defun edit-change-cursors (waitp)
  (unless (eql *current-mouse-cursor-waitp* waitp)
    ;;(format t "changing-cursors ~a~%" waitp)
    (dolist (oedwin *oed-focus-stack*)
      (xwin-change-cursor oedwin waitp))
    (setf *current-mouse-cursor-waitp* waitp)
    (xlib:display-finish-output *display*)
    ))

(defun edit-non-wait-cursors () (edit-change-cursors nil))
  
;;; Window properties: :width, :height, :cursor-x, :cursor-y, :backing-store
;;; and :actual-top-level-X-window.  The last property is needed because
;;; window managers don't like giving us the information we need.  In
;;; particular, prl's window placement decisions require knowing where
;;; currently mapped windows are placed.  To hack around the problem, after a
;;; window is mapped we find the new parent.  :actual-top-level-X-window is
;;; the new parent if there is one, otherwise it is just the window itself.
                                                                             


(eval-when (compile)
  (proclaim '(inline
	      draw-of-buffer-state
	      display-of-buffer-state
	      line-of-buffer-state
	      text-cursor-of-buffer-state
	      null-cursor-of-buffer-state
	      blob-cursor-of-buffer-state)))

(defstruct buffer-state
  draw
  display
  line
  draw-cursor				; 
  (drawn-line-cursor-p nil)
  display-cursor			;
  display-echo
  )

(defun draw-of-buffer-state (bstate)
  (buffer-state-draw bstate))
(defun display-of-buffer-state (bstate)
  (buffer-state-display bstate))
(defun line-of-buffer-state (bstate)
  (buffer-state-line bstate))

(defun draw-cursor-of-buffer-state (bstate)
  (buffer-state-draw-cursor bstate))
(defun display-cursor-of-buffer-state (bstate)
  (buffer-state-display-cursor bstate))
(defun display-echo-of-buffer-state (bstate)
  (buffer-state-display-echo bstate))


(defun set-draw-cursor (w c)
  ;;(setf -w w -c c) (break "sdw")
  (setf (buffer-state-draw-cursor (buffer-state-of-oed-window w)) c))

(defun set-display-cursor (w c)
  (setf (buffer-state-display-cursor (buffer-state-of-oed-window w)) c))

(defun set-display-echo (w e)
  (setf (buffer-state-display-echo (buffer-state-of-oed-window w)) e))


;; PERF could have element-type of byte-8 or at most byte-16
(defun new-buffer-state (w h)
  ;; 1+ for continuation char.
  (make-buffer-state :draw (make-array (list h (1+ w))
				       :element-type 'integer
				       :initial-element ispace)
		     :display (make-array (list h (1+ w))
				       :element-type 'integer
				       :initial-element ispace)
		     :line (make-array (list (1+ w))
				       :element-type 'integer
				       :initial-element ispace)))
		     


(defun last-non-white-of-draw-line (bstate r w)
  (format t "last : ~a ~%" r)
  (let ((draw (draw-of-buffer-state bstate)))
    (do ((i (1- w) (1- i)))
	((or (zerop i)
	     (not (whitespace-code-p (abs (aref draw r i)))))
	 i))))

(defstruct (oed-window (:include base-win)
		       (:print-function
			 (lambda (w stream depth)
			   (declare (ignore depth))
			   (format stream "OED-window: ~a ~a" (width-of-oed-window w) (height-of-oed-window w)
				   ))))
  x-win

  top-x
  backing-store
  cursor-x
  cursor-y
  cursor
  offset

  buffer-state
  )


(defun width-of-oed-window (w) (base-win-w w))
(defun height-of-oed-window (w) (base-win-h w))
(defun top-x-of-oed-window (w) (oed-window-top-x w))
(defun backing-store-of-oed-window (w) (oed-window-backing-store w))
;;(defun cursor-x-of-oed-window (w) (oed-window-cursor-x w))
;;(defun cursor-y-of-oed-window (w) (oed-window-cursor-y w))
;;(defun cursor-of-oed-window (w) (oed-window-cursor w))
(defun offset-of-oed-window (w) (oed-window-offset w))
(defun buffer-state-of-oed-window (w) (oed-window-buffer-state w))

(defun window-X-window (window)
  (oed-window-x-win  window))

(defun make-window (X-window w h)
  (make-oed-window :x-win x-window
		   :w w
		   :h h
		   :buffer-state (new-buffer-state w h)
		   ))

(defun oed-window-resize (win w h)
  (setf (base-win-w win) w
	(base-win-h win) h
	(oed-window-buffer-state win) (new-buffer-state w h)))
  


(defun window-equal (w1 w2)
  (xlib:window-equal (window-X-window w1)
		     (window-X-window w2)))

(defvar *auth-kind* nil)
(defvar *auth-value* nil)

(defun initialize-window-system ()
  (let (display)
    (unwind-protect
	(progn
	  (setf display (if *auth-kind*
			    (xlib:open-display *host* :display *display-index* :authorization-name *auth-kind*
					       :authorization-data *auth-value*)
			  (xlib:open-display *host* :display *display-index*))
		)
	  (setf -display display)
	  (let* ((font (check-and-open-font display
					    (option :font-name)
					    ;"9x15"
					    "nuprl-13"
					    ))
		 (screen (xlib:display-default-screen display))
		 (colormap (xlib:screen-default-colormap screen))
		 (root (xlib:screen-root screen))
		 (foreground ;;(and nil)
		  (xlib:alloc-color colormap
				    (xlib:lookup-color colormap 
						       (option :foreground-color))))
		 (background ;;(and nil)
		  (xlib:alloc-color colormap 
				    (xlib:lookup-color colormap 
						       (option :background-color))))
		 )
	    (when (eql foreground background)
	      (format
	       t
	       "Foreground and backgound color options map to same hardware color")
	      (setf foreground (xlib:screen-black-pixel screen)
		    background (xlib:screen-white-pixel screen)))
	    (let ((colors (xlib:query-colors (xlib:screen-default-colormap screen)
					     (list foreground background))))

	      (setf *character-X-width* (xlib:max-char-width font)
		    *line-X-height* (+ (xlib:font-ascent font)
				       (xlib:font-descent font))
		    *baseline-X-height* (xlib:font-ascent font)
		    *border-X-width* (truncate *character-X-width* 3)
		    *margin-X-width* (truncate *character-X-width* 3)
		    *vertical-X-offset* (if (option :title-bars?)
					    (+ *line-X-height* *margin-X-width*)
					    *margin-X-width*)
		    *foreground* foreground
		    *background* background
		    *gcontext* (xlib:create-gcontext
				:drawable root
				:background background
				:foreground foreground
				:font font)
		 
		    *reverse-gcontext* (xlib:create-gcontext
					:drawable root
					:background foreground
					:foreground background
					:font font)
		    *cursor-font* (check-and-open-font display
						       (option :cursor-font-name)
						       "cursor")
		    *cursor* (xlib:create-glyph-cursor
			      :source-font *cursor-font*
			      :source-char (option :cursor-font-index)
			      :mask-font *cursor-font*
			      :mask-char (1+ (option :cursor-font-index))
			      :foreground (first colors)
			      :background (second colors))
		    *wait-cursor* (xlib:create-glyph-cursor
				   :source-font *cursor-font*
				   :source-char (option :wait-cursor-font-index)
				   :mask-font *cursor-font*
				   :mask-char (1+ (option :wait-cursor-font-index))
				   :foreground (first colors)
				   :background (second colors))
		    *text-cursor-font-index* (option :text-cursor-font-index)
		    *blob-cursor-font-index* (option :blob-cursor-font-index)
		    *null-cursor-font-index* (option :null-cursor-font-index))))
	  (setf *X-window-output-suppressed?* nil)
	  (setf *display* display)
	  (initialize-application-variables))
      (unless *display* (xlib:close-display display)))))



(defun clear-backing-store (window)
  (let ((width (width-of-oed-window window))
	(store (backing-store-of-oed-window window)))
    (dotimes (i (height-of-oed-window window))
      (dotimes (j width)
	(setf (aref store i j) (ichar->font-index ispace))))))

(defun adjust-backing-store-size (window w h)
  (let* ((a (backing-store-of-oed-window window))
	 (old-w (array-dimension a 1))
	 (old-h (array-dimension a 0)))	 
    (when (or (< old-w w) (< old-h h))
      (adjust-array a (list (max old-h h) (max old-w w))
		    :element-type 'fixnum
;		    :element-type 'font-index
		    :initial-element (ichar->font-index ispace)))))

(defun initialize-backing-store (window)
  (setf (oed-window-backing-store window)
	(make-array (list (height-of-oed-window window)
			  (width-of-oed-window window))
 		    :element-type 'fixnum
; 		    :element-type 'font-index
		    :initial-element (ichar->font-index ispace)
		    :adjustable t)))

(defvar *onscreen-p* nil)

(defvar *scrwidth* nil)
(defvar	*scrheight* nil)

(defun onscreen (x y w h)
  (let* ((screen (xlib:display-default-screen *display*))
	 (scrwidth (or *scrwidth* (xlib:screen-width screen)))
	 (scrheight (or *scrheight* (xlib:screen-height screen))))
    (let* ((xdiff (- (+ x w 20) scrwidth))
	   (ydiff (- (+ y h 20) scrheight))
	   (newx (if (< 0 xdiff) (max (- x xdiff) 0) x))
	   (newy (if (< 0 ydiff) (max (- y ydiff) 0) y)))

      ;;(setf a (list xdiff ydiff newx newy scrwidth scrheight))
    
      (if *onscreen-p*
	  (values newx newy w h)
	  (values x y w h)))))
	   

;;; If reconfiguration-ok? is t, then the position and shape given to the
;;; window may be different than what is specified.
(defun create-window (x y width height &key (reconfiguration-ok? nil))
  (multiple-value-bind (x y width height)
      (cond
	(reconfiguration-ok? (contract-geometry-into-frame x y width height))
	(t (values x y width height)))

    (mlet* (((a b c d)	(geometry->X-geometry x y width height))
	    ((X-x X-y X-width X-height) (onscreen a b c d)))
      
	   (let* ((window (make-window (create-X-window X-x X-y X-width X-height) width height))
		  (X-window (window-X-window window)))

	     (initialize-backing-store window)

	     (setf (oed-window-top-x window) X-window
		   (oed-window-cursor-x window) 0
		   (oed-window-cursor-y window) 0
		   (oed-window-cursor window) t)

	     (xlib:with-state (X-window)
	       (setf (xlib:drawable-x X-window) X-x
		     (xlib:drawable-y X-window) X-y
		     (xlib:drawable-width X-window) X-width
		     (xlib:drawable-height X-window) X-height))
	     (xlib:map-window X-window)
	     (xlib:display-finish-output *display*)

	     window))))

(defun set-X-window-title (X-window string)
  (setf (xlib:wm-name X-window) string
	(xlib:wm-icon-name X-window) string)) 

(defun xwin-set-x-title (X-window winstring iconstring)
  (setf (xlib:wm-name X-window) winstring
	(xlib:wm-icon-name X-window) iconstring)) 


(defun xwin-set-window-title (win wintitle icontitle)
  (let ((hoststr (local-host)))
    (let ((ws wintitle)
	  (is icontitle))
      (when hoststr
	(setf ws (concatenate 'string ws " @" hoststr)
	      is (concatenate 'string is " @" hoststr)))
      (xwin-set-x-title (window-x-window win) ws is)
      )))
	      

;;; Not implemented yet.  One way to do this is to give a window an
;;; identification stamp at creation time.
(defun replace-lost-window (window)
  (declare (ignore window))
  (oed-x-reset)
  (error "Windows can only be destroyed using ^D.  ~
          You must now abort and reinvoke Nuprl (state will be preserved)."))


(defun draw-title-bar (window)
  ;;(break "dtb")
  (when (option :title-bars?)
    (unless *X-window-output-suppressed?*
      (let* ((X-window (window-X-window window))
	     (X-w (xlib:drawable-width X-window)))
	(xlib:clear-area X-window :x 0 :y 0 :width X-w :height *line-X-height*)
	(xlib:draw-rectangle X-window *gcontext* 0 0 X-w *line-X-height* t)
	(xlib:draw-glyphs X-window *reverse-gcontext* *margin-X-width* *baseline-X-height*
			  (xlib:wm-name X-window))))))

(defun redo-backing-store (window)
  (adjust-backing-store-size
    window (width-of-oed-window window) (height-of-oed-window window))
  (clear-backing-store window)
  (let ((*X-window-output-suppressed?* t))
    (recompute-window-contents-and-redraw window)
    ))


#|
(defun maybe-process-char-immediately (X-window ch)
  (cond ((equal (normalized-char->ted-char ch) '(SNAPSHOT))
	 (let ((v (view-of-xwindow X-window)))
	   (let ((w (window-of-view v)))
	     (when w (snapshot-window w))))
	 nil)
	(t
	 ch)))

(defun with-fast-redrawing (window redrawing-function &rest args)
  (let ((result (let ((*X-window-output-suppressed?* t))
		  (apply redrawing-function args))))
    (refresh-window window)
    result))

(defun refresh-window (window)
  (draw-title-bar window)
  (let ((state (view-of-window (window-x-window window))))
    (edit-asynch-write state)))

;;; Size may have changed.
(defun redraw-window (window)
  (redo-backing-store window)
  (refresh-window window))


(defun set-cursor (window x y)
  (setf (oed-window-cursor-x window) x
	(oed-window-cursor-y window) y))


(defun get-window-title (win)
   (xlib:wm-name win))

|#


(defun modify-window-title-bar (window start string &optional (clear-to-end? nil) xwinp)
  (let ((xwin (if xwinp window (window-x-window window))))
    ;; the supplied strings must not contain "@"
    (let* ((start (or start 0))		; prl needs this.
	   (host-name (or (local-host) ""))
	   (suffix (if nil ;;host-name
		       (format nil " @ ~A" host-name)
		     ""))
	   (long-title (string (xlib:wm-name xwin)))
	   (title (if (search suffix long-title)
		      (subseq long-title 0 (- (length long-title)
					      (length suffix)))
		    long-title))
	   (title-length (length title))
	   (str-length (length string))
	   (new-title
	    (cond ((>= start title-length)
		   (concatenate 'string title string))
		  ((or (>= (+ start str-length) title-length)
		       clear-to-end?)
		   (concatenate 'string (subseq title 0 start) string))
		  (t
		   (concatenate 'string
				(subseq title 0 start)
				string
				(subseq title (+ start str-length))))))
	   (new-long-title
	    (if (option :host-name-in-title-bars?)
		(concatenate 'string new-title suffix)
	      new-title)))	       
      (set-X-window-title
       xwin
       new-long-title))
    (draw-title-bar window)))






(defun xwindow-geometry (w)
  (mlet* (((foo parent) (xlib:query-tree w) (declare (ignore foo)))
	  ((x y) (x-coordinates->coordinates (xlib:drawable-x parent)
					     (xlib:drawable-y parent)))
	  ((w h) (x-coordinates->coordinates (xlib:drawable-width w)
					     (xlib:drawable-height w))))
	 (list (1+ x) (1+ y) w h)))

(defun get-ewin-geometry (w)
  (let ((xw (oed-window-x-win w)))
    (unless xw
      (raise-error (error-message '(window geometry get not))))
    (xwindow-geometry xw)))



(defvar *host* nil)
(defvar *display-index* 0)

;; find cookie via "xauth list <display>" eg, xauth nlist nuprl4:1
;;1b9b26887acda581be0c7956e8c3cfeb
(defunml (|set_xhost_wcookie| (mit-cookie s n))
    (string -> (string -> (int -> unit)))
  (setf *host* s
	*display-index* n)
  (if (not (string= "" mit-cookie))
      (setf *auth-kind* "MIT-MAGIC-COOKIE-1"
	    *auth-value* mit-cookie)
    (setf *auth-kind* nil
	    *auth-value* nil))
  nil)

(defunml (|xhost| (unit) :declare ((declare (ignore unit))))
    (void -> string)
  (format-string "~a:~a" *host* *display-index*))

(defun x-host ()
  (let ((display (get-environment-variable "DISPLAY")))
    (or (when display
	  (with-string-scanner (display)
	      (let ((host (scan-string (standard-character-sbits
					(list icolon)))))
		(let ((display (if (scan-at-byte-p icolon)
				   (progn (scan-ichar icolon)
					  (scan-decimal-num))
				   0)))
		  (cons (if (or (string-equal host "unix") (string-equal host ""))
			    (local-host)
			    host)
			display)))))
	(cons (prompt-and-read :string "Name of display: ") 0))))

(defun run-application (&optional (host nil) xdisplay)
  (edit-buffer-clear)
  (unless *initial-key-bindings*
    (setf *initial-key-bindings* *key-bindings*))

  (if host
      (setf *host* host)
      (let ((xh (x-host)))
	(setf *host* (car xh) *display-index* (cdr xh))))

  (when xdisplay (setf *display-index* xdisplay))

  (unless *host* (error "You must supply a host name."))
  (unless *display-index* (setf *display-index* 0))

  (change-config-options)

  ;;(break "ra")
  (check-options)
  ;;(break "ra")
  (setf *effected-options* (map 'list #'identity *options*))
  
  (unless *display* (initialize-window-system))

  
  ;;(application) ; call directly.
  )

(defun X-parent (window)
  (multiple-value-bind (children parent root)
      (xlib:query-tree (window-X-window window))
    (declare (ignore children root))
    parent))


(defvar *suppress-warp* nil)

(defun suppress-warp ()
  (setf *suppress-warp* nil))

(defmacro without-warp-suppressed (&body body)
  `(progn
    (setf *suppress-warp* nil)
    ,@body))


(defun view-of-xwindow (xwin)
  (find-first #'(lambda (v)
		  (unless (lightweight-view-p v)
		    (let ((oedwin (window-of-view v t)))
		      (when (and oedwin (eql xwin (window-x-window oedwin)))
			v))))
	      *views*) )

(defun oedwindow-of-xwindow (xwin)
  (find-first #'(lambda (oedwin)
		  (when (and oedwin (eql xwin (window-x-window oedwin)))
		    oedwin))
	      *oed-focus-stack*))


(defun xwin-exposure (window)
  (let ((state (view-of-xwindow window)))
    (when state
      (view-refresh state))))

(defun xwin-focus (window)
  (let ((win (oedwindow-of-xwindow window))
	(state (view-of-xwindow window)))
    ;;(setf -state state) (break "xf")
    (with-dummy-transaction
      (when state
	(oed-focus-on win)
	(refresh-views)))))

(defvar *last-x-event-time* nil)

(defun x-event-happened ()
  (setf *last-x-event-time* (get-universal-time)))


(defun xwin-button-press (window x y code state)
  (x-event-happened)
  (multiple-value-bind (x y) (X-coordinates->coordinates x y)
    (let ((view (view-of-xwindow window)))
      (if (null view)
	  (progn (message-emit (warn-message '(x button view window not)))
		 nil)
	  (progn
	    (window-of-view view)
	    (set-edit-state-mouse-position view (1+ y) (1+ x))
	    (edit-mouse-blip code state))
	  ))))

(defun xwin-key-press (window code state)
  (declare (ignore window))
  (x-event-happened)
  (keycode->normalized-char code state))


(defun xwin-configure-notify (window width height)
  (let ((state (view-of-xwindow window)))
    (if (null state)
	(progn (message-emit (warn-message '(x config view window not)))
	       nil)
	(let* ((window (window-of-view state)))
	  (when (and window
		     (or (not (= width (width-of-oed-window window)))
			 (not (= height (height-of-oed-window window)))))
	    (oed-window-resize window width height)
	    ;; no need to refresh since an exposure event is due.
	    ;;(redo-backing-store window)
	    )
	  
	  ;; todo make sure gets redisplayed, ie layout required gets turned on
	  (view-flag-set-layout-required state t)
	  (view-flag-set-present-required state t)
	  ))))

(defvar *destroyed* nil)

(defun destroy-window (window)
  (xwin-destroy-window (window-x-window window)))

(defun xwin-destroy-window (window)
  (setf *destroyed*  window)
  (window-destroyed  window)
  ;;(setf -ww window) (break "xdw")
  (xlib:destroy-window window)
  (xwin-require-flush)
  )

(defun window-destroyed (win)

  (let ((w (find-first #'(lambda (ww) (when (eql win (oed-window-x-win ww)) ww))
		       *oed-focus-stack*)))	       
    (when w

      (setf (base-win-handle w) nil
	    (oed-window-x-win w) nil)

      (view-window-destroyed w)

      (setf *oed-focus-stack* (delete w *oed-focus-stack*))
      (when (eql (oed-current-focus) w)
	(setf *oed-focus* nil))
      )))  
  

(defvar *xwin-event-window* nil)
(defvar *xwin-event-x* nil)
(defvar *xwin-event-y* nil)

(defun xwin-event-window ()
  (oedwindow-of-xwindow *xwin-event-window*))

;; coerce focus !
(defun set-event-window (win)
  (setf *xwin-event-window* win))

(defvar *xwin-cut-win* nil)
(defvar *xwin-cut-time* nil)
(defvar *xwin-cut-wrapper* nil)

(defun set-xwin-cut (v s)
  (let ((w (oed-window-x-win (window-of-edit-state v))))
    ;;(setf -v v -s s -w w) (break "sxc")
    ;; set property on xwin of v
    (let ((ss (if *xwin-cut-wrapper*
		  (format-string "~a~a~a" (car *xwin-cut-wrapper*) s (cdr *xwin-cut-wrapper*))
		  s)))
      (format t "~%set_x_cut ~a" ss)

      (xlib:change-property w :cut
			    (string-to-byte-array ss *no-escape-sbits*)
			    :string 8)
      (setf *xwin-cut-win* w)
      )))

(defunml (|set_x_cut| (v s))
    (view -> (string -> unit))

  (set-xwin-cut v s)
  nil)

(defunml (|set_x_cut_wrapper| (pre suf))
    (string -> (string -> unit))

  (setf *xwin-cut-wrapper* (cons pre suf))
  nil)


(defun xwin-property-notify (w atom time)
  ;;(format t "~%xwin-property-notify ~a ~a" atom time)
  (when (and (eql atom :cut)
	     (eql w *xwin-cut-win*))
    (setf (xlib:selection-owner *display* :primary time) w
	  *xwin-cut-time* time)
    ;;(break "xpn")
    nil
    ;; supposedly should check time stamp of primary selection
    ;; but don't know how.
    ))

(defun xwin-selection-request (w r sel type prop time)
  ;;(setf -w w -r r -sel sel -type type -prop prop -time time)
  ;;(format t "~%xwin-selection-request ~a ~a ~a ~a" sel type prop time)
  ;;(break "xsr")
  (let ((b (when (eql sel :primary)
	     (cond
	       ((eql type :timestamp)
		(xlib:change-property r prop
				      (let ((a (make-array 1 :element-type 'xlib:card32)))
					(setf (aref a 0)
					      (xlib::encode-type xlib:card32 *xwin-cut-time*))
					a)
				      :timestamp 32)
		t)
	       ((or (eql type :text)
		    (eql type :string))
					;(xlib::set-string-property r prop "fubar")
		(xlib:change-property r prop
				      (xlib:get-property *xwin-cut-win* :cut)
				      type 8)
		t
		)
	       (t (format t "XwinSelectionRequest: Unidentified request type ~a" type)
		  ;;(break "xsr2")
		  nil)))))
    
    (if b       
	(xlib:send-event r :selection-notify nil
			 :selection sel
			 :property prop
			 :window r
			 :target type
			 :time time)
	(xlib:send-event r :selection-notify nil
			 :selection sel
			 :property prop
			 :window r
			 :target type
			 :time time))
    nil))

    
;; then setf selection-owner

;; convert selection when receive selection-request?
;;(xlib:convert-selection :primary :string w)


;;; gross hack:  force prl to select new window by generating
;;; a bogus mouse-jump.
(defun get-character (&optional block-p)

  (edit-non-wait-cursors)

  (xlib:event-case (*display* :discard-p t :force-output-p t :timeout (if block-p 1 0))
    (exposure  
      (window count)
      (when (zerop count)
	(with-dummy-transaction
	    (xwin-exposure window)))
      nil)

    ;;((focus-in)
    ;;(window) ; mode kind
    ;;(set-event-window window)
    ;;(xwin-focus window)
    ;;nil)

    (button-press
      (window x y code state)
      (set-event-window window)
      (xwin-button-press window x y code state))

    (key-press
      (state window code)
      (set-event-window window)
      (xwin-key-press window code state))

    ((enter-notify)
     (window)
     (set-event-window window)
     (xwin-focus window)
     nil)

    (configure-notify
      (window width height)
      (mlet* (((width height) (X-size->size width height)))
	(when (or (zerop width)
		  (zerop height))
	  (error "Width or height too small.  Abort and reset."))
	(xwin-configure-notify window width height))
      nil)

    (property-notify
     (window atom state time)
     ;;(format t "~%PropertyNotify ~a ~a" atom time)
     (xwin-property-notify window atom time)
     )

    (selection-clear
     (window)
     ;;(format t "~%SelectionClear ")
     )
    (selection-notify
     (window)
     ;;(format t "~%SelectionNotify ")
     )
    (selection-request
     (window requestor selection target property time)
     ;;(format t "~%SelectionRequest ")
     (xwin-selection-request window requestor selection target property time)
     )

    (destroy-notify
      (window)
      ;;(setf -window window) (break "dn")
      (window-destroyed window)
      nil)

    (reparent-notify
      (window parent)
      (let ((v (view-of-xwindow window)))
	;;(setf -v v -w window -p parent) (break "rn")
	(if (null v)
	    (progn
	      (message-emit (warn-message '(x reparent view window not)))
	      nil)
	    (let ((w (window-of-view v)))
	      (when w (setf (oed-window-top-x w) parent)))))
      nil)

    (otherwise nil)))


#|
(defun draw-or-erase-text-cursor (window drawp)
  (unless (or *X-window-output-suppressed?*
	      (not (window-in-use? window)))
    (let ((cx (cursor-x-of-oed-window window))
	  (cy (cursor-y-of-oed-window window)))

      (multiple-value-bind (x y)
	  (coordinates->x-coordinates cx cy)
	(when (cursor-of-oed-window window)
	  (xlib:draw-image-glyph
	   (window-x-window window) (if drawp *reverse-gcontext* *gcontext*)
	   x
	   (+ *baseline-x-height* y)
	   (aref (backing-store-of-oed-window window) cy cx)
	   :width *character-x-width*)))))
  nil)
|#


;;; mill out selected window.
;;; mill in window arg to cursor funcs
;;; maybe should bite bullet and mill out window property and replace by direct fields in structure? yes.



(defvar *current-input-line* nil)
(defvar *current-prompt* nil)




(defun snapshot-window (window)
  (let ((filen  (or *snapshot-file*
		    (merge-pathnames "snapshot.text" (user-homedir-pathname)))))
    (handle-process-err #'(lambda (msg) (display-msg msg t))
			(handle-file-error
			 (with-open-file (s filen
					    :direction :output
					    :if-exists :append
					    :if-does-not-exist :create)
			   (format t "Snapshot[~a].~%" filen)
			   (format s "~3%~A~2%" (xlib:wm-name (window-X-window window)))
			   (let ((a  (backing-store-of-oed-window window)))
			     (dotimes (y (height-of-oed-window window)) 
			       (dotimes (x (width-of-oed-window  window))
				 (let* ((z (if (output-for-latexize-p)
					       (font-index->char-for-latexize (aref a y x))
					       (font-index->char-or-string (aref a y x)))))
				   (if (stringp z)
				       (princ z s)
				       (write-char z s))))
			       (format s "~%"))))))))

(defvar *cmd-p* nil)
(defvar *cmd-state* nil)
(defvar *cmd-win* nil)

(defun reset-cmd ()
  (setf *cmd-state* nil
	*cmd-win* nil
	*cmd-p* nil))

(defun oed-x-reset ()
  (when *display* (xlib:close-display *display*))
  (setf *display* nil
	*oed-focus* nil
	*oed-focus-stack* nil)
  (reset-application)
  (reset-cmd)
  nil)

(defunml (|oed_x_reset| (unit)  :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> unit)

  (oed-x-reset)
  (quit-prl-loop)
  nil
   )

(defun windows-p ()
  (and *display* t))


;;;---------------------------------------------------------------------------

;;; Remembers arguments to previous call.
(defun fdl-editor (&optional host xdisplay)
  ;;(in-package "FDL0")
  (setf *make-view-f* #'new-edit-state)
  (add-transaction-end-hook 'oed #'oed-transaction-end-hook)(add-transaction-end-hook 'oed #'oed-transaction-end-hook)
  (add-transaction-end-hook 'dyneval #'dyneval-transaction-end-hook)
  ;;(add-transaction-end-hook 'dform #'dform-transaction-end-hook)  ;; appears that incremental updates work.
  (add-transaction-end-hook 'precedence #'precedence-transaction-end-hook)
  (add-transaction-end-hook 'ddg #'ddg-transaction-end-hook)
  (add-dyneval-hook 'update-edit #'edit-dyneval-touched)

  ;;(change-options :font-name "nuprl-13") ;;
  (run-application (or host *host*) (or xdisplay *display-index*)))


;; initilializes
(defunml (|fdl_editor| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)

  (fdl-editor)

  (set-x-sleeper (address-of-environment (match-environment-in-list '(|edd|) *component*)))
  nil
  )

(defun getchr ()
  (get-character))


(defun changesizew (w y1 y2 x1 x2)
  (declare (ignore w y1 y2 x1 x2))
  ;; A stub.
  nil)

(defun enter-prl-state$ ()
  nil)

(defun leave-prl-state$ ()
  nil)

;;; There's no easy way to do this.  It doesn't matter, since
;;; it's only used as a convenience.
(defmacro with-terminal-io-on-frame (&body body)
  `(progn . ,body))


(defun window-width (w)
  (width-of-oed-window w))

(defun window-height (w)
  (height-of-oed-window w))


(defun xwin-clear (window)
  (xlib:clear-area window))


;; overwrite by using draw-glyphs instead of draw-image-glyphs.


(defun xflush ()
  (xlib:display-finish-output *display*))




;;;;
;;;; Window buffers.
;;;;

;;; I/O will be buffered.
;;;
;;; Each window will have two buffers.
;;;  - display : whats currently on the screen.
;;;  - draw :  
;;;
;;;  write-buffer :
;;;   - writes differences between input and displayed to screen
;;;   - copies input buffer to display buffer.
;;;
;;;  Buffer:
;;;   - two dimension array of integers.
;;;     - assume integer 4 bytes.
;;;     - two bytes are font index and two are flags.
;;;      - flags are
;;;        -reverse video
;;;        -overwritten?
;;;   - text-cursor : x, y
;;;   - null-cursor : x, y  : term with no text.
;;;    

(defun mytest (x y)
  (let ((m (make-array (list x y) :element-type 'integer :initial-element 1 ))
	(n  (make-array (list x y) :element-type  'integer :initial-element 2))
	(o  (make-array y :element-type  'integer :initial-element 0)))
    (time (do ((i 0 (1+ i))
	       (acc nil))
	      ((= i x) acc)
	    (do ((j 0 (1+ j)))
		((= j y))
	      (let ((ne (aref n i j)))
		(unless (= (aref m i j) ne)
		  (setf (aref o j) ne))))
	    (push o acc)))
    (time (do ((i 0 (1+ i)))
	      ((= i x))
	    (do ((j 0 (1+ j)))
		((= j y))
	      (setf (aref m i j) 0))))
    nil))

;;; xwin-write-text-cursor (win rol col)
;;;   -implicitly clears cursors.  
;;; xwin-clear-text-cursor (win)
;;; xwin-write-null-cursor (win rol col)
;;;   -implicitly clears cursors.  
;;; xwin-clear-null-cursor (win)
;;; xwin-clear-reverse-video (win)
;;;
;;; xwin-clear (win)
;;;   -implicitly clears cursors.  
;;; xwin-write-lines (win rol col indent lines rv-p)
;;;   -implicitly clears cursors and reverse video.
;;;   

;;; lines - list of lines
;;; indent - applys to subsequent lines of lines.

(defstruct xwin-cursor
  row
  col)

(defstruct (spot-cursor (:include xwin-cursor))
  font-index)

;; line-cursor where lines is null should be coerced to null cursor.
(defstruct (line-cursor (:include xwin-cursor))
  (indent 0)
  (lines nil))


(defun new-spot-cursor (r c fi)
  (make-spot-cursor :row r :col c :font-index fi))

(defun new-null-cursor (r c) (new-spot-cursor r c *null-cursor-font-index*))
(defun new-text-cursor (r c) (new-spot-cursor r c *text-cursor-font-index*))
(defun new-blob-cursor (r c) (new-spot-cursor r c *blob-cursor-font-index*))

(defun new-line-cursor (r c i l)
  (if (null l)
      (new-null-cursor r c)
      (make-line-cursor :row r :col c :indent i :lines l)))

(defun row-of-xcursor (c) (xwin-cursor-row c))
(defun col-of-xcursor (c) (xwin-cursor-col c))

(defun font-index-of-spot-cursor (c) (spot-cursor-font-index c))
(defun indent-of-line-cursor (c) (line-cursor-indent c))
(defun lines-of-line-cursor (c) (line-cursor-lines c))


;;;;	
;;;;	buffer-state : 
;;;;
;;;;	Two buffers of font-indices, negative means reverse-video
;;;;	  - display	: reflects current window state. (except echo/flash)
;;;;	    - buffer, spot-cursor.
;;;;	  - draw	: accumulates mods.
;;;;	    - buffer, spot-cursor
;;;;
;;;;	  - point cursor is encoded in display/draw as reverse video.
;;;;	
;;;;	echo is sneaked into display but not draw, so draw implicitly
;;;;	undoes it.
;;;;	
;;;;	spot-cursor : INT{row} INT{col} INT{font-index}
;;;;	lines-cursor : INT{row} INT{col} INT{indent} <lines>
;;;;
;;;;	Line :
;;;;	  - line	: used to buffer output to xlib.
;;;;	
;;;;	Cursors : 
;;;;
;;;;	  Spot cursors: these are written directly to screen.
;;;;	  - text : thin vertical line.
;;;;	  - null : null cursor, slightly wider thin vert line. Believe used
;;;;		when point has no visible text.
;;;;	  - blob : screen cursor, may shadow point.
;;;;
;;;;	  Echo is written to display buffer/screen.
;;;;	  - echo : direct to display buffer.
;;;;		ie not part of state, but seems some flag should be set when
;;;;		display modified.
;;;;	
;;;;	  Lines are written to draw-buffer.
;;;;	  - lines : reverse video of some lines.
;;;;	
;;;;	Unwriting of spot cursor is implicit in buffer-state/xwin.
;;;;	Unwriting of echo cursor is implicit in xwin-display.
;;;;	  thus view-display-required == t clears echo.
;;;;	Unwriting of lines implicit in xwin-write-lines or xwin-buffer-lines
;;;;	  thus view-present-required == t clears lines.
;;;;	  also view-cursor-present-required == t clears lines.
;;;;	
;;;;	
;;;;	

;;;;
;;;;	Cursor flags : 
;;;;	 cursor-modified		: mod not yet reflected in ?? view ??. THis may be same as cursor-layout required.
;;;;	 cursor-layout-required		: mod not yet reflected in cursor lines.
;;;;	 cursor-present-required	: mod not yet reflected in draw buffer.
;;;;
;;;;	view flags used:
;;;;	 display-required : ie to erase echo
;;;;
;;;;	 cursor-echo
;;;;	 cursor-text
;;;;	 cursor-blob
;;;;	 cursor-null
;;;;	 cursor-lines
;;;;	
;;;;	
;;;;	display-cursor implicit in display of window.
;;;;	Thus present-required implies cursor-present-required.
;;;;	However if only mod is cursor move then cursor-present-required 
;;;;	
;;;;	
;;;;	view - flags/window
;;;;	window - x/buffer
;;;;	buffer - display/draw
;;;;	
;;;;	Present :
;;;;	  draw : modify buffers.
;;;;	  display : dump diff to screen then draw becomes display.
;;;;	
;;;;	
;;;;	
;;;;	view-draw
;;;;	view-cursor
;;;;	view-present
;;;;
;;;;	view-refresh
;;;;	view-relayout
;;;;	


;;;;	
;;;;	
;;;;	xwin-buffer-clear (<buffer>)			: <values>
;;;;	  * sets all to space.
;;;;
;;;;	xwin-buffer-copy (<buffer{dest}> <buffer{src}>)	: <values>
;;;;	  * copys src to destination.
;;;;	
;;;;	xwin-draw-lines (<buffer> INT{row} INT{col} INT{indent}
;;;;			   <lines> <bool{reverse-video}>)
;;;;	    : <values>
;;;;	  * write lines into buffer. Undoes reverse video of any
;;;;	    elements not overwritten.
;;;;	
;;;;	xwin-write-lines (<buffer-state> INT{row} INT{col} INT{indent}
;;;;			   <lines> <bool{reverse-video}>)
;;;;	    : <values>
;;;;	  * write lines into draw buffer. Undoes reverse video of any
;;;;	    elements not overwritten.
;;;;	
;;;;	xwin-refresh (<window> <buffer-state>)		: <values>
;;;;	  * copies draw to display
;;;;	  * writes display to window
;;;;	
;;;;	xwin-display (<window> <buffer-state>)		: <values>
;;;;	  * undraws cursor : seems like that could be elsewhere (but text/blob/null)
;;;;	  * simultaneous compares/copies draw to display and writes to window.
;;;;	    only draws those that differ between draw and display.
;;;;	  * draws cursor.
;;;;	
;;;;	<cursor>	: INT{x} . INT{y}
;;;;	
;;;;	
;;;;	xwin-undraw-cursor (<window> <buffer-state>)			: <values>
;;;;	  * undraws any recorded cursor and resets state.
;;;;	
;;;;	xwin-draw-null-cursor  (<window> INT{x} INT{y})			: <values>
;;;;	xwin-draw-text-cursor  (<window> INT{x} INT{y})			: <values>
;;;;	xwin-draw-blob-cursor  (<window> INT{x} INT{y})			: <values>
;;;;	  * first undraw any recorded cursor.
;;;;	  * record position for undraw.
;;;;	  * draw directly to window.
;;;;	
;;;;	
;;;;	Observation : cursor info is missing from refresh.
;;;;	 ie no need to involve view if oed-window knows cursor.
;;;;	
;;;;	
;;;;	line-cursor gets drawn to draw buffer and then display is done by usual
;;;;	buffer display. Spot cursors display'ed directly and thus must be 
;;;;	done at display time although they may be undisplayed at draw time.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	draw-cursor :
;;;;	  * draws line cursor to draw-buffer. no-ops on spot cursor.
;;;;	
;;;;	display-cursor : 
;;;;	  * no-ops on line cursor but displays spot cursor 
;;;;	
;;;;	undisplay-cursor
;;;;	  * no-ops on line cursor but un-displays spot cursor 
;;;;	
;;;;	undraw-cursor :
;;;;	  * undraws line cursor to draw-buffer. no-ops on spot cursor.
;;;;	


(defun xwin-buffer-clear (buffer)
  (let ((dimensions (array-dimensions buffer)))
    (do ((i 0 (1+ i)))
	((= i (car dimensions)))
      (do ((j 0 (1+ j)))
	  ((= j (cadr dimensions)))
	(setf (aref buffer i j) ispace)))))

(defun xwin-buffer-copy (dest src)
  (let ((dimensions (array-dimensions dest)))
    (do ((i 0 (1+ i)))
	((= i (car dimensions)))
      (do ((j 0 (1+ j)))
	  ((= j (cadr dimensions)))
	(setf (aref dest i j) (aref src i j))))))


;; xwin-draw-lines (buffer 1 1 0 nil nil) will clear reverse-video.
;; dumps lines to buffer


(defun xwin-draw-lines (buffer inrow incol indent new-lines reverse-video-p)
  (let ((dimensions (array-dimensions buffer))
	(row (1- inrow))
	(col (1- incol))
	(mode 'off)
	(started nil)
	(done (null new-lines))
	(lines new-lines)
	(line nil))


    ;; start -> t, mode -> on,  when row/col reached.
    ;; mode -> off when null line.
    ;; mode -> on  when started, inner loop count is at left indent and car lines is a contains a line.
    ;; done -> t when null lines.

    ;; mode off will clear reverse video.

    (do ((i 0 (1+ i)))
	((= i (car dimensions)))
      (do ((j 0 (1+ j)))
	  ((= j (cadr dimensions)))
	(when (and (not done) (eql mode 'off))
	  (if started
	      (when (and lines (= j (if (continued-line-p (car lines))
					0
					(or indent col))))
		(setf line (text-of-line (car lines)))
		(setf lines (cdr lines))
		(when line (setf mode 'on)))
	      (when (and (= row i) (= col j))
		(setf started t)
		(setf line (text-of-line (car lines)))
		(when line (setf mode 'on))
		(setf lines (cdr lines)))))

	(if (eql mode 'on)
	    (progn
	      ;;(when (not (integerp (car line)))
	      ;;(setf a line b lines c i d j e buffer f done g mode) (break))
	      (setf (aref buffer i j) (if reverse-video-p
					  (- (car line))
					  (car line))))
	    (let ((c (aref buffer i j)))
	      (when (minusp c)
		;;(when (not (integerp c))
		;;(setf a line b lines c i d j e buffer) (break))
		(setf (aref buffer i j) (- c)))))

	(when line
	  (setf line (cdr line))
	  (when (null line)
	    (setf mode 'off)
	    (when (null lines)
	      ;;(setf a line b lines c i d j e buffer f done g mode) (break "done")
	      (setf done t))))
	))))


;; resets reverse-video to implicitly undo line-cursor. 
(defun xwin-refresh-draw (buffer)
  (xwin-draw-lines buffer 1 1 0 nil nil))

(defun xwin-draw (buffer inrow incol indent new-lines reverse-video-p)
  (xwin-draw-lines buffer inrow incol indent new-lines reverse-video-p))


(defun xwin-draw-line-aux (window gcontext row col a length)
  (let ((aa (if (some #'(lambda (i) (> i 256)) a)
		(map 'array #'(lambda (i) (ldb (byte 8 0) i)) a)
		a)))
    (multiple-value-bind (X-x X-y)
	(coordinates->X-coordinates col row)
      (xlib:draw-image-glyphs window gcontext
			      X-x (+ *baseline-X-height* X-y)
			      aa
			      :start 0 :end length
			      :width (* length *character-X-width*)))))


(defun xwin-draw-line (window line row-col length gcontext)
  (xwin-draw-line-aux window gcontext (car row-col) (cdr row-col) line length)
  (xwin-require-flush))
	 

(defun xwin-screen (window)
  (let* ((bstate (buffer-state-of-oed-window window))
	 (xwindow (oed-window-x-win window))
	 (display (display-of-buffer-state bstate))
	 (line (line-of-buffer-state bstate))
	 (dimensions (array-dimensions display))
	 (started nil)
	 (gcontext nil)
	 (count nil))

    (do ((i 0 (1+ i)))
	((= i (car dimensions)))
      (do ((j 0 (1+ j)))
	  ((= j (cadr dimensions)))
	(let ((a (aref display i j)))
	  (let ((context (if (minusp a) *reverse-gcontext* *gcontext*)))
	    (when (or (not (eql context gcontext)) (zerop j))
	      (unless (null gcontext) (xwin-draw-line xwindow line started count gcontext))
	      (setf gcontext context)
	      (setf started (cons i j))
	      (setf count 0))
	    (setf (aref line count) (if (minusp a) (- a) a))
	    (incf count)))))))



;; copies into display buffer and refreshes screen.
(defun xwin-display (window)

  ;; line cursor implicit in draw-buffer.
  (let* ((bstate (buffer-state-of-oed-window window))
	 (xwin (oed-window-x-win window))
	 (display (display-of-buffer-state bstate))
	 (draw (draw-of-buffer-state bstate))
	 (dimensions (array-dimensions display))
	 (line (line-of-buffer-state bstate))
	 (started nil)
	 (gcontext nil)
	 (count 0))

    ;;(setf -b bstate) (break "xd")

    (let ((display-cursor (display-cursor-of-buffer-state bstate)))
      (when (and display-cursor (spot-cursor-p display-cursor))
	(xwin-undisplay-spot-cursor xwin bstate display-cursor)
	;;(break "xdb")
	(set-display-cursor window nil)))  


    (do ((i 0 (1+ i)))
	((= i (car dimensions)))
      (do ((j 0 (1+ j)))
	  ((= j (cadr dimensions)))
	(let ((a (aref display i j))
	      (b (aref draw i j)))
	  (if (not (= a b))
	      (let ((context (if (minusp b) *reverse-gcontext* *gcontext*)))
		(if started
		    (when (or (not (eql context gcontext)) (zerop j))
		      (xwin-draw-line xwin line started count gcontext)
		      (setf gcontext context)
		      (setf started (cons i j))
		      (setf count 0))
		    (progn
		      (setf gcontext context)
		      (setf started (cons i j))))
		(setf (aref line count) (if (minusp b) (- b) b))
		(setf (aref display i j) b)
		(incf count))
	      (when started
		(xwin-draw-line xwin line started count gcontext)
		(setf started nil)
		(setf count 0)))))))

  (xwin-display-cursor window))





(defun xwin-draw-line-cursor (buffer cursor reverse-video-p)
  (xwin-draw-lines buffer
		   (row-of-xcursor cursor)
		   (col-of-xcursor cursor)
		   (indent-of-line-cursor cursor)
		   (lines-of-line-cursor cursor)
		   reverse-video-p))




(defun xwin-display-spot-cursor (xwin bstate cursor)
  
  (let ((r (row-of-xcursor cursor))
	(c (col-of-xcursor cursor))
	(findex (font-index-of-spot-cursor cursor)))
	
    ;;(setf -a r -b c -c findex) (break "xdsc")
 
    (let ((line (line-of-buffer-state bstate))
	  (row (1- r))
	  (col (1- c)))

      (setf (aref line 0) findex)
      (multiple-value-bind (X-x X-y)
	  (coordinates->X-coordinates col row)
	;;(setf -d line) (break)
	(xlib:draw-glyphs xwin *gcontext*
			  X-x (+ *baseline-X-height* X-y)
			  line
			  :start	0
			  :end	1
			  :width	(* 1 *character-X-width*)))))
  
  (xwin-require-flush)
  )


(defun xwin-undisplay-spot-cursor (xwin bstate cursor)
	
  (let ((line (line-of-buffer-state bstate))
	(r (row-of-xcursor cursor))
	(c (col-of-xcursor cursor))
	(fi (font-index-of-spot-cursor cursor)))

    
    ;;(setf -a r -b c -c fi) (break "xusc")
    ;;(setf -spot-cursor cursor)
    ;;(setf (spot-cursor-col -spot-cursor) 1)

    (let ((row (1- r))
	  (col (1- c)))

      (multiple-value-bind (X-x X-y)
	  (coordinates->X-coordinates col row)

	;; redraw cursor in reverse video to clear foreground of cursor image.
	;; only really necessary for cursor which draw outside the lines, ie null cursor.
	;; however, easier to do for all than distinquish.
	(when fi
	  (setf (aref line 0) fi)
	  (xlib:draw-glyphs xwin *reverse-gcontext*
			    X-x (+ *baseline-X-height* X-y)
			    line
			    :start 0 :end 1
			    :width (* 1 *character-X-width*))

	  (unless (zerop col)
	    ;;(format t "Still have to figure how to clear left of col 0")
	    ;; twould seem that be done by the reverse-video draw of cursor.

	    ;; this used to check reverse video, hopefully it didn't need to.
	    (let ((display (display-of-buffer-state bstate)))
	      (when (and (< row (array-dimension display 0))
			 (< col (array-dimension display 1)))
		(setf (aref line 0) (abs (aref (display-of-buffer-state bstate) row (1- col))))
		(xwin-draw-line-aux xwin *gcontext* row (1- col) line 1)
		#|(multiple-value-bind (X-x X-y)
		    (coordinates->X-coordinates (1- col) row)

		  (xlib:draw-image-glyphs xwin
					  *gcontext*
					  X-x (+ *baseline-X-height* X-y)
					  line
					  :start 0 :end 1
					  :width (* 1 *character-X-width*)))|#
		))))

	;; redraw char at cursor position.
	(let ((display (display-of-buffer-state bstate)))
	  (when (and (< row (array-dimension display 0))
		     (< col (array-dimension display 1)))
	    (let ((c (aref (display-of-buffer-state bstate) row col)))
	      (setf (aref line 0)
		    (if (minusp c) (- c) c))

	      (xwin-draw-line-aux xwin (if (minusp c) *reverse-gcontext* *gcontext*)
				  row col ;;(+ *baseline-X-height* X-y) X-x
				  line 1)
	      #|(xlib:draw-image-glyphs xwin (if (minusp c) *reverse-gcontext* *gcontext*)
				      X-x (+ *baseline-X-height* X-y)
				      line
				      :start 0 :end 1
				      :width (* 1 *character-X-width*))|#
	      )))))

    (xwin-require-flush)
    ))



;;;
;;; text/null cursors are not reflected in buffer arrays.
;;;  but are treated like an overlay.
;;;

;; returns t if draw buffer modified.
(defun xwin-draw-cursor (window)

  (let ((bstate (buffer-state-of-oed-window window))
	)

    (let ((draw (draw-cursor-of-buffer-state bstate)))

      ;; if old and new lines then undraw not needed as draw implicitly undraws old.
      ;; but how about draw spot cursor and line display cursor.

      (cond
	((and draw (line-cursor-p draw))
	 (xwin-draw-line-cursor (draw-of-buffer-state bstate) draw t)
	 (setf (buffer-state-drawn-line-cursor-p bstate) t)
	 (set-draw-cursor window nil)
	 t
	 )

	;; need to be done in case last cursor was line-cursor
	((buffer-state-drawn-line-cursor-p bstate)
	 (xwin-refresh-draw (buffer-state-draw bstate))
	 t)

	;; let display handle spot-cursors.
	(t nil)))) )


;; displays draw cursor and moves cursor to display.
;; dump display buffer lines and then display cursor to refresh display.
(defun xwin-display-cursor (window)
  (let ((bstate (buffer-state-of-oed-window window))
	(xwindow (oed-window-x-win window))
	)

    ;;(break "xdc")
    (let ((display (display-cursor-of-buffer-state bstate))
	  (draw (draw-cursor-of-buffer-state bstate)))

      (when (and display (spot-cursor-p display))
	(xwin-undisplay-spot-cursor xwindow bstate display)
	(set-display-cursor window nil))
	   
      (when (and draw (spot-cursor-p draw))
	(xwin-display-spot-cursor xwindow bstate draw)
	(set-display-cursor window draw))))

  (xwin-display-echo window))


;; el problemo : does not draw to screen only to display buffer.
;; best : compare with display and only update screen with diff.
;; ok   : update display buffer and screen independently.
;; ng   : draw then display
;; ??   : write directly to screen without updating display. requires screen refresh, but big deal.
(defun xwin-display-echo (window &optional flashp)
  (let* ((bstate (buffer-state-of-oed-window window))
	 (ecursor (display-echo-of-buffer-state bstate)))

    ;;(setf -a window) (break "xde")

    (when ecursor
      (let* ((xwindow (oed-window-x-win window))
	     (buffer (line-of-buffer-state bstate))
	     (row (max 0 (1- (row-of-xcursor ecursor))))
	     (col (max 0 (1- (col-of-xcursor ecursor))))
	     (display (display-of-buffer-state bstate))
	     (count nil))

	(do ((lines (lines-of-line-cursor ecursor) (cdr lines))
	     (i row (1+ i)))
	    ((null lines))
	  (setf count 0)
	  (do ((line (text-of-line (car lines)) (cdr line))
	       (j (if (= i row) col 0) (1+ j)))
	      ((null line))
	    (setf (aref buffer count) (car line))
	    (setf (aref display i j) (- (car line)))
	    (incf count))

	  (multiple-value-bind (X-x X-y)
	      (coordinates->X-coordinates (if (= i row) col 0) i)
	    (xlib:draw-image-glyphs xwindow *reverse-gcontext*
				    X-x (+ *baseline-X-height* X-y)
				    buffer
				    :start 0 :end count
				    :width (* count *character-X-width*)))

	  ;; the echo spot cursor
	  (when (and (not flashp) (null (cdr lines)))
	    (setf (aref buffer 0) ispace)
	    (multiple-value-bind (X-x X-y)
		(coordinates->X-coordinates (+ (if (= i row) col 0) (- count 2)) i)
	      (xlib:draw-image-glyphs xwindow *gcontext*
				      X-x (+ *baseline-X-height* X-y)
				      buffer
				      :start 0 :end 1
				      :width (* 1 *character-X-width*))))))))
  (xwin-require-flush))

;; dumps display to screen.
(defun xwin-refresh (window)
  (let* ((bstate (buffer-state-of-oed-window window))
	 (display (display-of-buffer-state bstate))
	 (draw (draw-of-buffer-state bstate)))

    ;; ?
    (xwin-buffer-copy display draw)

    ;;(break)
    (xwin-screen window)

    (xwin-display-cursor window)))


#|
;; doesn't do display or cursors.
;; defunct? subsumed by view-present.
(defun view-draw (v)
  ;; it might be possible to not change to be here with same ttree.
  ;; ie, cursor is overlay so moving point may not change ttree.
  ;; layout-of-view should then cause some draw-required flag?

  (let* ((win (window-of-view v))
	 (dt (dtree-of-view v))
	 (tt (layout-of-dtree-c dt (width-of-window win))))


    ;;(setf a win b dt c tt d v)  (break)

    ;; must be checked after layout-of-dtree-c
    (when (or (dtree-flag-present-required-p dt)
	      (view-flag-present-required-p v))

      ;; update draw-buffer
      (let ((lines (layout-visit tt
				 (width-of-oed-window win)
				 (height-of-oed-window win)
				 (offset-of-edit-state v)
				 dt)))
	;;(setf d lines)

	(let ((bstate (buffer-state-of-oed-window win)))
	  (xwin-buffer-clear (buffer-state-draw bstate))
	  (xwin-draw (buffer-state-draw bstate)
		     1 1 0
		     lines
		     nil)
	  (setf (buffer-state-drawn-line-cursor-p bstate) nil)))
      
      (view-flag-set-display-required v t)
      (dtree-flag-set-present-required dt nil))))


;; draw should be done already if needed.
(defun view-cursor (v)
  
  (let ((c (cursor-of-view-c v))
	(w (window-of-view v)))
    
    ;; setting view-cursor to null should set view-display-required
    ;; so that no need to handle null cursor here.
    (when c
      (cond
	;;possible to require display but not present (echo-cursor).
	((and (view-flag-cursor-display-required-p v)
	      (not (view-flag-cursor-present-required-p v)))
	 (set-display-cursor w c)
	 (xwin-display-cursor (window-of-view v)))
    
	((view-flag-cursor-present-required-p v)
	 (set-draw-cursor w c)
	 (if (xwin-draw-cursor (window-of-view v))
	     (view-flag-set-display-required v t)
	     (view-flag-set-cursor-display-required v t))
	 (view-flag-set-cursor-present-required v nil)))
      )))
|#
#|
;; shouldn';t be necessary, but effectively used via dummy-line-cursor.
(defun xwin-undisplay-line-cursor (window)
  (xwin-draw-lines (buffer-state-display (buffer-state-of-oed-window window))
		   1 1 0 nil nil)
  (xwin-screen window))


;; hopefully not used.
(defun xwin-undisplay-cursor (window)
  (let ((bstate (buffer-state-of-oed-window window))
	(xwin (oed-window-x-win window)))

    (let ((cursor (display-cursor-of-buffer-state bstate)))
      (if (and cursor (spot-cursor-p display))
	  (xwin-undisplay-spot-cursor xwin bstate cursor)
	  ;; could be dummy as contents not needed.
	  (xwin-undisplay-line-cursor xwin bstate) ))))

|#
#|
;; draw into display buffer, then draw-buffer undoes.


(defun xwin-undraw-echo (window)
  (xwin-display window))
|#



		  
			


(defvar *pending-flash* nil)
(defvar *flash-active-p* t)

(defun impend-flash (str)
  (when *flash-active-p*
    (setf *pending-flash* str)))

(defun maybe-pending-flash (state)
  (when *pending-flash*
    (flash *pending-flash* state)
    (setf *pending-flash* nil)))


(defun convert-cr-to-nl (string)
  (do ((input (istring string) (cdr input))
       (acc nil))
      ((null input) (implode-to-string (nreverse acc)))
    (cond
      ((and (null (cdr input))
	    (eql ireturn (car input))))
      ((eql ireturn (car input))
       (push inewline acc))
      (t (push (car input) acc)))))

(defvar *displayed-newline-last-p* nil)

(defun display-msg (str &optional flash-p)
  (when flash-p (impend-flash str))
  (princ (convert-cr-to-nl str))
  (terpri)
  (setf *displayed-newline-last-p* t))

(defun display-string (str)
  (setf *displayed-newline-last-p* nil)
  (princ str)
  (when (eql inewline (car (last (istring str))))
    (setf *displayed-newline-last-p* t)))

(defun display-message (istr)
  (setf *displayed-newline-last-p* nil)
  (princ (implode-to-string istr))
  (when (eql inewline (car (last istr)))
    (setf *displayed-newline-last-p* t)))


(defun display-fresh-line ()
  (unless *displayed-newline-last-p*
    (terpri)
    (setf *displayed-newline-last-p* t)))

    
(defvar *cmd-warp* nil)
(defvar *prl-warp* nil)
(defvar *prl-input* 'emacs)


(defun warp-to-prl-window (window)
  (unless (or (eql window *prl-warp*)
	      (option :no-warp?)
	      (eql *prl-input* 'emacs))
    (setf *prl-warp* window)
    (unless *suppress-warp*
      ;;(xlib:unmap-window window)
      ;;(xlib:map-window window)
      (xlib:warp-pointer window (- (xlib:drawable-width window) 10) 1)))
  (setf *suppress-warp* t))



;;;;	need assoc of win <-> view
;;;;	
;;;;	edd does not track focus.
;;;;	
;;;;
;;;;

;;;;	
(defun new-ewin (x y w h)
  (create-window x y w h)
  )


(defun edit-reset-views ()
  (dolist (v *views*)
    (setf (view-state-window v) nil)))

(defun view-of-window (w)
  (find-first #'(lambda (v) (when (eql w (window-of-view v t)) v))
	      *views*))

(defun view-window-destroyed (w)
  (let ((v (view-of-window w)))
    (when v
      (setf (view-state-window v) nil)
      )))

;; let cleanup happen as in asynch case.
(defun edit-close-win (v)
  (destroy-window (window-of-view v))
  ;;temp
  (xwin-flush)
  )

;; 
(defun edit-open-win (v)
  (let ((w (window-of-view v)))

    (if (oed-window-x-win w)
	(setf (base-win-handle w) t);; dummy
	(raise-error (error-message '(edit open win xwin not))))

    (setf *oed-focus-stack* (nconc *oed-focus-stack* (list w))))
  
  (edit-init-state v)
  (values)
  )





;;;;	
;;;;	focus<i>
;;;;	  0 : focus at start of command.
;;;;	  1-8 changes of focus during input.
;;;;	  LHS : focus change during input.
;;;;	  RHS : jump to associated view of focus.
;;;;	
;;;;	  - move mouse to diff window then type a 
;;;;	  (focus1)a is input.
;;;;	  - on rhs is command to switch to that view.
;;;;	
;;;;	(mouse-left)==...  ==> (focus1)(mouse-left)==(focus1)...
;;;;	a == (focus1)a
;;;;	(c-a)(focus1)(c-b)(focus2)
;;;;	
;;;;	
;;;;	if  focus1 is first input 
;;;;	and matching macro does not explicitly mention, then autofocus.
;;;;	sequential focii besides (focus0)(focus1) not possible.
;;;;	
;;;;	
;;;;	correspondence with view stack ???
;;;;	  - little.
;;;;	  - changed via oed and x actions while views changed via ml.
;;;;	could be a dangerous dichotomy. 
;;;;	views should not be considered ordered.

;; what does it mean to have the focus.
;; input affects you by default, edit actions may 
;; you cursor is visible.



(defvar *oed-focus-stack* nil)

(defvar *oed-focus* nil)

(defun oed-current-focus () *oed-focus*)

(defun oed-focus-on (w)

  ;;(unless (view-of-window w) (break "ofo"))
	  
  (unless (eq w *oed-focus*)
    ;;(setf -w w) (break "fov")

    (when *oed-focus*
      (let ((v (view-of-window *oed-focus*)))
	(when v
	  (view-flag-set-cursor-visible v nil)
	  (view-flag-set-cursor-display-required v t)
	  (view-flag-set-cursor-present-required v t))))

    (when w
      (setf *oed-focus* w)
      (set-new-focus-p)

      (let ((v (view-of-window w)))
	(when v
	  (view-flag-set-cursor-visible v t)
	  (view-flag-set-cursor-display-required v t)
	  (view-flag-set-cursor-present-required v t)))

      (when (null w) (break "ofs"))
      ;;(break "ofo")
      (setf *oed-focus-stack* (cons w (delete w *oed-focus-stack*))))))


(defun focus-on-stack ()
  (when *oed-focus-stack*
    (oed-focus-on (car *oed-focus-stack*))))

(defun focus-on-view (v)
  (oed-focus-on (when v (window-of-view v))))

(defun focus-raise (w)
  (unless w (break "fr"))
  (setf *oed-focus-stack* (cons w (delete w *oed-focus-stack*))))

(defun focus-lower (w)
  (unless w (break "fl"))
  (setf *oed-focus-stack* (nconc (delete w *oed-focus-stack*) (list w))))

(defun focus-rotate ()
  (setf *oed-focus-stack*
	(nconc (cdr *oed-focus-stack*)
	       (list (car *oed-focus-stack*)))))

(defun focus-reverse-rotate ()
  (setf *oed-focus-stack*
	(nconc (last *oed-focus-stack*)
	       (butlast *oed-focus-stack*))))

(defun focus-swap ()
  (setf *oed-focus-stack*
	(cons (cadr *oed-focus-stack*)
	      (cons (car *oed-focus-stack*)
		    (cddr *oed-focus-stack*)))))

;;;;	
;;;;	some flags like layout required set on path to root 
;;;;	but only examined at root ?? must be sure to turn off
;;;;	as much as matters to avoid curtailing path update 
;;;;	later due to stale bit.
;;;;	

(defvar *xwin-flush-required* nil)

(defun xwin-flush ()
  (when (and *xwin-flush-required* *display*)
    (xlib:display-finish-output *display*)
    (setf *xwin-flush-required* nil)))

(defun xwin-require-flush ()
  (setf *xwin-flush-required* t))


(defun oed-transaction-end-hook (th)

  ;;(break "oteh")
  (let ((any-p nil))
    (dolist (tr th)
      (let ((troid (oid-of-touch-record tr)))
	(mapc #'(lambda (v)
		  (let ((void (oid-of-view v)))
		    (when (and void (equal-oids-p troid (oid-of-view v)))
		      (setf any-p t)

		      ;; set title update request.
		      ;;(with-dummy-transaction
			  (view-touched v)
			;;)
		      )))
	      *views*)))

    ;;
    (when any-p
      ;; refresh-views? or queue something which causes refresh?
      ;; fttb do nothing. If transaction due to edit act then will refresh.
      ;; otherwise, in top loop and will refresh at switch to edit mode.
      ;; when edit and asynch updates possible may need to revisit.
      )
  ))
      


;; layout of cursor may affect layout of window by shifting offset.
;; thus cursor must be layed-out prior to view-draw. but must not be draw.
(defun view-present (v)

  ;;(format t "view_present~%~%")
  ;;(setf -v v) (break "vp")
  ;; refresh data
  (let* ((w (window-of-view v))	
	 (dt (dtree-of-view-c v))
	 (tt (layout-of-dtree-c dt (width-of-window w)))
	 (c (cursor-of-view-c v)) ; does cursor layout.
	 )

    (when (view-flag-title-present-required-p v)
      (draw-title-bar w)
      (view-flag-set-title-present-required v nil)
      )

    ;;(setf -a w -b dt -c tt -d v) (break "vp1")

    (view-flag-set-layout-required v nil)

    ;; draw
    (when (or (dtree-flag-present-required-p dt)
	      (view-flag-present-required-p v))
      ;; must be checked after layout-of-dtree-c

      ;; update draw-buffer
      (let ((lines (layout-visit tt
				 (width-of-oed-window w)
				 (height-of-oed-window w)
				 (offset-of-edit-state v)
				 dt)))

	;; draw 
	(let ((bstate (buffer-state-of-oed-window w)))
	  (xwin-buffer-clear (buffer-state-draw bstate))
	  (xwin-draw (buffer-state-draw bstate)
		     1 1 0
		     lines
		     nil)
	  (setf (buffer-state-drawn-line-cursor-p bstate) nil)
	  ))
      
      (view-flag-set-display-required v t)
      (view-flag-set-cursor-present-required v t)
      (dtree-flag-set-present-required dt nil))

    ;; draw cursor.
    (when c

      ;; setting view-cursor to null should set view-display-required
      ;; so that no need to handle null cursor here.
      (cond

	((view-flag-cursor-present-required-p v)
	 (if (view-flag-cursor-visible-p v)
	     (set-draw-cursor w c)
	     (set-draw-cursor w nil))
	 ;;(break "wha")
	 (if (xwin-draw-cursor w)
	     (view-flag-set-display-required v t)
	     (view-flag-set-cursor-display-required v t))
	 (view-flag-set-cursor-present-required v nil))))

    ;; echo
    (let ((echo (when (view-flag-echo-display-required-p v)
		  ;; lines
		  (echo-of-view-c v))))

      ;;(setf -a echo) (break "pr")
      
      ;; needs to be done prior to view or cursor display.
      (set-display-echo w echo)
      
      ;; display 
      (cond
	((view-flag-display-required-p v)
	 (xwin-display w)
	 (view-flag-set-display-required v nil)
	 (view-flag-set-cursor-display-required v nil)
	 (view-flag-set-echo-display-required v nil))

	((view-flag-cursor-display-required-p v)
	 (xwin-display-cursor w)
	 (view-flag-set-cursor-display-required v nil)
	 (view-flag-set-echo-display-required v nil))

	((view-flag-echo-display-required-p v)
	 (xwin-display-echo w)
	 (view-flag-set-echo-display-required v nil))
	))

    (view-flag-set-present-required v nil)
    (view-update-history v)

    ;;temp
    (xwin-flush)
    ))


(defun view-refresh (v)
  ;;(setf -v v) (break "vr")
  (view-present v)
  (xwin-refresh (window-of-view v))
  ;;temp
  (xwin-flush)
  )


(defun view-relayout (v)
  (let ((dtree (dtree-of-view-c v)))
    (view-flag-set-layout-required v t)
    (view-flag-set-cursor-layout-required v t)
    (dtree-flag-set-layout-required dtree t)
    (view-present v)
    ))

      

(defunml (|view_require_layout| (v))
    (view -> unit)

  (let ((dtree (dtree-of-view-c v)))
    (view-flag-set-layout-required dtree t)))

(defunml (|view_raise| (v))
    (view -> unit)
  (edit-raise v)
  nil)


;; may want some force version. ie currently doesn't do anything if nothing changed.
(defunml (|view_refresh| (view))
    (view -> unit)

  ;;(setf -view view) (break "vr)
  (view-present view)
  )




;;;
;;; edit loop
;;;

(defun initialize ()

  (setq *prl-running* nil)

  t )


(defun quit-prl-loop ()
  (set-orb-sleeper nil)

  (setf *prl-running* nil))

(defun prl-loop ()

  (setf *prl-running* t)

  (do ()
      ((not *prl-running*))
    (handle-process-err #'(lambda (err)
			    (setf -a err) (break "prl-loop")
			    (format t "~a~%" err))
      (get-input-aux)
      (when (null *oed-focus-stack*)
	(quit-prl-loop))))

  (format t "PRL Terminated.")
  (terpri))


(defvar *x-timeout* nil)
(defvar *x-timeout-hook* #'(lambda (e) (format t "~%timeout expired ~a~%" e)))

(defun timeout-quit (e)
  (declare (ignore e))

  (funmlcall (ml-text "queue_quit") nil))

(defunml (|set_quit_timeout| (i))
  (int -> unit)

  (x-event-happened)
  (if (zerop i)
      (setf *x-timeout* nil)
    (setf *x-timeout* i
	  *x-timeout-hook* #'timeout-quit))
  nil)


(defun prl-loop-alittle ()

  ;;(format t "z")
  (setf *prl-running* t)

  (do ()
      ((not *prl-running*))
    (handle-process-err #'(lambda (err)
			    ;;(setf -a err) (break "prl-loop")
			    (format t "~a~%" err))
			(let ((gia (get-input-aux)))
			  ;;(format t "gia ~a~%" gia)
			  (unless gia
			    (setf *prl-running* nil))
			  (when (null *oed-focus-stack*)
			    (quit-prl-loop)))))

  (when *x-timeout*
    (let ((elapsed (- (get-universal-time) *last-x-event-time*)))
      (when (> elapsed *x-timeout*)
	(funcall *x-timeout-hook* elapsed))))
    
  ;;(format t "PRL Terminated.")
  ;;(terpri)
  )


(defun runprl (tag)

  (unless *display*
    (fdl-editor))

  (insys
   (let ((env (match-environment-in-list (if (consp tag)
					     tag
					     (list tag))
					 *component*)))
     
     (unless env
       (raise-error (error-message '(run prl environment not) tag)))

     (with-environment-actual env
       (with-appropriate-transaction (nil t)
	 (ml-text "win_init()")))

     (prl (address-of-environment env) )) ))


(defun reset-current-view ()
  (unless (current-view)
    (set-current-view (find-first #'(lambda (v) (when (view-window-open-p v) v))
				  *views*)))
     
  (when (null (current-view))
    (raise-error (error-message '(run prl view none))))

  (focus-on-view (current-view))
  
  (refresh-views))

(defun set-x-sleeper (env-addr)
  (set-orb-sleeper #'(lambda (&rest r)
		       (declare (ignore r))
		       (with-environment (env-addr)
					 (prl-loop-alittle)))))



(defun prl (env-addr)

  (with-environment (env-addr)
    (when *prl-running*
      (error "FDL editor is already running."))

    (with-dummy-transaction
     (dolist (v *views*)
	     (when (and (view-flag-open-p v) (null (view-window-open-p v)))
	       (view-open-window v)))

     (reset-current-view)))

  ;;(break "prl")
  (set-x-sleeper env-addr)
  
  ;;(unwind-protect
  ;;(prl-loop)
  ;;(setq *prl-running* nil)
  )


(defun reset-prl ()
  (edit-reset-views)
  (initialize)
  (edit-init)
  t)



;; following would be called from within a transaction which we need to avoid.
;; if running from top loop may x. type of thing.

#|
(defun application ()
  (prl))


(defunml (|edit_loop| (unit) :declare ((declare (ignore unit))))
    (unit -> unit)

  (prl)
 
  (values))


(defunml (|quit_edit_loop| (unit)  :declare ((declare (ignore unit))) :error-wrap-p nil)
    (unit -> unit)

  (setf *prl-running* nil))
|#
    


