
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; -docs- (mod ptr)
;;;;
;;;;	Overview ::
;;;;
;;;;	Editor is split into two parts on a client/server model.
;;;;	 - edit : server
;;;;	      * ML programmable by UI coder
;;;;	      *  runs in CommonLisp.
;;;;     - presentation : client.
;;;;	   ??? Assume JAVA ???
;;;;	      * macro programmable by UI coder 
;;;;	      * portable, expected to run on various platforms using
;;;;		various lanquages/windows.
;;;;		Eg, java, or basic/windows, lisp/clx, elisp/emacs, c/X.
;;;;
;;;;	 - assume simple unsigned byte link.
;;;;	    * no term knowledge required
;;;;	    * s-expr syntax probably sufficient.
;;;;
;;;;	  - shared state.
;;;;	      * ptrees
;;;;	      * registers
;;;;	
;;;;	  - predefined commands.
;;;;	    both edit and presentation support a set of predefined commands.
;;;;	
;;;;	  - edit eval : presentation can send edit commands to be evaled.
;;;;	    no args? evaled command finds args in state. Ie looks at registers
;;;;	    and ptrees??? Maybe window list as single arg.
;;;;	
;;;;	
;;;;	Why split :
;;;;	  - desire ML programmability.
;;;;	     term hacking.
;;;;	  - desire JAVA portabilty.
;;;;	     ascii input.
;;;;	
;;;;	  - some pieces could be either side.
;;;;	      * layout/dtree : lose ml access to dtree.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;  -page-
;;;;
;;;;	Characters : unicode
;;;;
;;;;	Character string representation options :
;;;;	  - array of unsigned bytes, two bytes to a character.
;;;;	      * not attractive, due to marshalling needed when converting strings to
;;;;		data such as tokens, ascii strings.
;;;;	  - string, but two string chars to a unicode character.
;;;;	      * When viewing internal data, normal ascii strings will be funny looking.
;;;;	      * All builtin symbols and strings used as data must be in this form.
;;;;	      * Error prone, as must distinquish data symbols and strings from program data.
;;;;	  - string, but non-ascii chars converted to three char sequence
;;;;	    where first char is some type of escape and other two are hex rep for unicode char.
;;;;	      * Marshalling points better defined. Ie, unmarshall for display.  
;;;;	
;;;;	Looks like string with escapes for non-ascii chars is the winner, ie all
;;;;	strings are ascii strings, except that at layout, some ascii sequences
;;;;	are interpreted as non ascii characters.
;;;;	
;;;;	
;;;;	** 3/99
;;;;	** appears that actual embedding is \uuuu all ascii
;;;;
;;;;	A non-ascii unicode character will be called an ascii embedded character.
;;;;	It will be three characters long. First character will be an escape character,
;;;;	used to delimit the embedding and the following two characters will be 
;;;;	the two halves of the unicode character.
;;;;
;;;;	** Note that if every lisp had a least two-byte char codes, this would all be moot.
;;;;	** In fact it might be worth an alternate implementation for those lisps with two-byte
;;;;	** chars. (Which are probably likely to become more prevalent).
;;;;
;;;;  -page-
;;;;
;;;;	Unicode character blocks desired:
;;;;
;;;;	Character Blocks:	Character Range		# of characters	Subtotal
;;;;
;;;;	Essential :		
;;;;	  - ascii 		0020-007f		96
;;;;	  - greek		0390-03cf		64
;;;;	  - supers & subs	2070-208f		32
;;;;	  - Latin1		00a0-00bf		32
;;;;	  - math		2200-22ff		256		480
;;;;utf-8 2200 = 0010 0010 0000 0000 -> 1110 0010 10 001000 10 000000 = &#xe28880
;;;;	Useful :
;;;;	  - arrows		2190-21ff		112
;;;;	  - misc tech		2300-232f		48		640
;;;;
;;;;	Questionably useful :
;;;;	  - geom shapes		25a0-25ef		80
;;;;	  - form & chart	2500-257f		128     
;;;;	  - letterlike		2100-213f		64		912
;;;;	
;;;;	Notes:
;;;;
;;;;	  ** Geometrical shapes contains small black square.
;;;;	  ** matches 8E glyph in nuprl-13 font. There may
;;;;	  ** be a better choice in one of the other character blocks
;;;;	  ** that has not yet been identified.
;;;;
;;;;	  ** Form and chart is include to encode arrow stems for concatenation
;;;;	  ** with arrow heads. I'm not satisfied that this is the best solution
;;;;	  ** to the arrow mapping.
;;;;	
;;;;	  ** letterlike contains some characters for symbols suchs as Reals or Rationals.
;;;;	  ** However, it may be better to have a double strike font style for Capital Alpha 
;;;;	  ** chars instead, particularly as letterlike is missing U and B that we need.
;;;;
;;;;  -page-
;;;;
;;;;	FTTB, the following unicode character ranges are supported by Nuprl.
;;;;	Characters may be used which fall outside of these ranges but they will
;;;;	not be displayed properly.  There will also be many characters within these ranges
;;;;	which may not display properly.
;;;;
;;;;	  - ascii 		0020-007f		96
;;;;	  - Latin1		00a0-00bf		32
;;;;	  - greek		0390-03cf		64
;;;;	  - supers & subs	2070-208f		32
;;;;	  - arrows		2190-21ff		112
;;;;	  - math		2200-22ff		256
;;;;
;;;;	  - misc tech		2300-232f		48
;;;;
;;;;	  - form & chart	2500-257f		128     
;;;;	  - geom shapes		25a0-25ef		80
;;;;
;;;;  -page-
;;;;
;;;;	Proposed font styles:
;;;;	  - Normal
;;;;	  - Bold, Thin.
;;;;	  - Underline 
;;;;	  - Italic : for alphanumeric characters.
;;;;	  - Double : for capital aphabetic characters.
;;;;	      * italic, underline bold normal and thin combos too.
;;;;	      * requires dform mapping when converting from v4.2 to v5.
;;;;		
;;;;	Proposed sizes:
;;;;	 - Small Medium Large
;;;;	 - Smaller than small could all map to dot
;;;;	 - subscript alpha should be done with reduce dform format wrapper.
;;;;	      * requires dform mapping when converting from v4.2 to v5.
;;;;	
;;;;  -page-
;;;;	
;;;;	Latex : it would be nice if fonts could be used with latex and font
;;;;	  style mappings make it into latex source output. Then we have
;;;;	  assurance that what is visible on screen could be the same on paper.
;;;;	
;;;;  -page-
;;;;
;;;;	Presentation and layout have to have same font resources.
;;;;	  - font resource will be an array of fonts to be indexed
;;;;	    by glyph.
;;;;	  
;;;;	Shared by presentation and edit:
;;;;
;;;;	inform-fonts (<string{font name}> list)		: NULL
;;;;	  * BYTE16{font} is the position of the named font in the list.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Ptree : shared edit-presentation data structure.
;;;;	  - Edit : creates ptree by layout of dtree.     
;;;;	  - Presentation : presents ptree.
;;;;
;;;;	Following syntax is an abstract syntax. Later a concrete syntax
;;;;	for tranmission of ptrees will be presented.
;;;;
;;;;	<ptree>		: ptree[<label> list
;;;;				<pformat> array
;;;;				]
;;;;
;;;;	<pformat>	: <pbreak>
;;;;			| <pindent>
;;;;			| <ptext>
;;;;			| <ptree>
;;;;			* More will be added.
;;;;	
;;;;	
;;;;	<pindent>	: indent[INTEGER{amt}]		* push
;;;;			| indent[]			* pop
;;;;	
;;;;	
;;;;	<ptext>		: text [<label> list
;;;;				'!dead | !live
;;;;				<glyphs>]
;;;;
;;;;	<glyphs>	: EPSILON
;;;;			| <glyph-array> <glyphs>
;;;;			| <label> <glyphs>
;;;;
;;;;	<glyph-array>	: <glyphID> array
;;;;
;;;;	<glyphID>	: font and font index.	
;;;;
;;;;	Indents are handled by including relative indent amounts. The amount is
;;;;	the number of pixels to indent.  It is allowable to pop indents passed
;;;;	from the parent. However, this should not affect the parent.  Ie later
;;;;	siblings should inherit the same indents.  Presentation must maintain a
;;;;	stack of indents, and treat parent pops properly.
;;;;	
;;;;	Indents are included in ptree rather than just specifing indent at break
;;;;	so that ptrees are reusable within different indent environments.
;;;;	
;;;;	There is an obvious, easily detectable structure corollary between dtree
;;;;	and ptree so that exported/imported ptree can be matched against its dtree.
;;;;
;;;;
;;;;  -page-
;;;;	
;;;;	Edit-Presentation Protocol (EPP) : Protocol for transmiting requests,
;;;;	  replies, and data between edit and presentation.
;;;;
;;;;	Abstract Functionality :
;;;;
;;;;	Edit -> Presentation :
;;;;
;;;;	Model : presentation will implement windows. Edit can open, close,
;;;;	 raise, lower, and update windows.  Windows will be indentified by a
;;;;	 handle. Edit can specify initial size and position.  Note that this
;;;;	 does not have to correspond to a real window wrt presentation. For
;;;;	 instance emacs may implement windows as buffers.
;;;;
;;;;     Edit will maintain ptree and communicate updates to presentation.
;;;;     Presentation will make calls to have edit modify term and update ptree.
;;;;	 Presentation can move labels and do some text editing locally.
;;;;
;;;;	TODO : interrupt?
;;;;
;;;;  -page-
;;;;
;;;;	SHORT : 16 bit value.
;;;;
;;;;	handle		: SHORT
;;;;	 * id of window
;;;;
;;;;	size : size of paintable area. Borders and other reserved space such as
;;;;		for wrap indicator must be accounted for in presentation.
;;;;
;;;;  -page-
;;;;
;;;;	Base Win : window data common to both edit and presentation.
;;;;
;;;;	<base-win>	: <handle> <ptree>
;;;;			  <x> <y>{position}
;;;;			  <w> <h> {size}
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Edit -> Presentation
;;;;
;;;;	open (x y w h handle) 
;;;;	 * (x,y) : position, (w,h) : size.
;;;;
;;;;	close (handle) 
;;;;	
;;;; 	raise (handle)
;;;;	lower (handle)
;;;;
;;;;	position(handle) : x y
;;;;
;;;;	add-label (handle label address)
;;;;	delete-label (handle label address)
;;;;	move-label (handle label from-address to-address)
;;;;
;;;;	update (handle replacements)
;;;;	 
;;;;	 updates to the ptree are done transmitting a list of replacements. The
;;;;	 replacements are applied to the previous tree to produce a new tree.
;;;;	 Replacement should be done constructively so that the original tree is
;;;;	 not modified. A replacment is a pair of a ptree address and a ptree.
;;;;	 The ptree of a replacement may contain addresses itself.  These
;;;;	 addresses should be replaced with the ptrees addressed in the orginal
;;;;	 tree.  Thus the replacement ptrees are essentially fragments to be
;;;;	 inserted into the original tree.  When replacements complete, new tree
;;;;	 becomes current tree.  If bad address or other error, then abort and
;;;;	 have edit refresh ptree.
;;;;
;;;;	reset()
;;;;	 ** dumps whole state through sequence of opens and updates.
;;;;	 ** can be used for error recovery or initialization.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Presentation -> Edit 
;;;;
;;;;	refresh (handle) : ptree
;;;;
;;;;	size (handle w h) 
;;;;	position (handle x y) 
;;;;
;;;;	destroy(handle) 
;;;;
;;;;	TODO : input stuff. more edit calls.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Synchronization : FTTB, I've punted the problem of edit and presentation
;;;;	 simultaneously modifying ptree. Could do some kind of locking or just
;;;;	 lose modification from one or the other.
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	PPE Concrete Syntax : byte level syntax suitable for transmission.
;;;;	
;;;;	Notation Notes: 
;;;;	  - FirstCapsWithoutHyphens	: Command Code.
;;;;	  - all-lowercase-with-hyphens	: Data Syntax
;;;;	  - ALLCAPS			: Keyword
;;;;	
;;;;	Operation types :
;;;;	  - Request : reply, significant action.
;;;;	  - Query : reply, insignifcant action.
;;;;	  - Command : no reply, significant action.
;;;;	  - Notify : no reply, insignifcant action.
;;;;
;;;;	Keywords : a flag for reader to indicate kind of operation.
;;;;	  - To edit from presentation.
;;;;	      * EREQ : request to edit from
;;;;	      * ENOTE : notify edit.
;;;;	      * EQUY
;;;;	      * ECMD
;;;;	  - To presentation from edit.
;;;;	      * PREQ 
;;;;	      * PNOTE
;;;;	      * PQUY
;;;;	      * PCMD
;;;;	
;;;;  -page-
;;;;
;;;;	Syntax:
;;;;
;;;;	Requests, Queries, and Commands :
;;;;
;;;;	<transmission>		: <items>
;;;;	
;;;;	<items>			: <item>
;;;;				| <item> <items>
;;;;	
;;;;	<item>			: <operation>
;;;;				| <reply>
;;;;
;;;;	<operation>		: <SHORT{sequence}> <op>
;;;;
;;;;	<reply>			: <SHORT{sequence}> <data>	
;;;;	  ** type of data depends on op being replied to.
;;;;	  ** so to scan data must know op to determine data type expected.
;;;;	
;;;;	<op>			: <OpenPCMD>   <SHORT{handle}> <SHORT{x}> <SHORT{y}>
;;;;							       <SHORT{w}> <SHORT{h}>
;;;;				| <ClosePCMD>  <SHORT{handle}>
;;;;				| <RaisePCMD>  <SHORT{handle}>
;;;;				| <LowerPCMD>  <SHORT{handle}>
;;;;				| <ModifyPCMD> <SHORT{handle}> <mod-list>
;;;;				| <PositionPQUY[<SHORT{x}> <SHORT{y}>] <SHORT{handle}>
;;;;				
;;;;				| <RefreshEREQ[<ptree>]> <SHORT{handle}>
;;;;				| <SizeENOTE>  <SHORT{handle}> <SHORT{w}> <SHORT{h}>
;;;;
;;;;	<mod-list> 		: <SHORT{count}> <replacements>
;;;;
;;;;	<replacements>		: EPSILON
;;;;				| <ptree-address> <ptree'> <replacements>
;;;;
;;;;
;;;;	
;;;;  -page-
;;;;
;;;;	Data:
;;;;
;;;;	<ptree-address>		: <ptree-address-id> <SHORT{count}> <ptree-indices>
;;;;	  ** ptree-address with 0 count indicates root.
;;;;
;;;;	<ptree-indices>		: EPSILON
;;;;				| <SHORT{index}>  <ptree-indices>
;;;;
;;;;
;;;;	<ptree>		: <ptree-id> <labels> <label-props> <pformats>
;;;;			| <ptree-pform-id> SHORT{pform-index} <labels> <label-props> <ptrees>
;;;;			| <ptree-text-id> <labels> <ptext>
;;;;			| <ptree-text-pform-id> SHORT{pform-index} <labels> <ptext>
;;;;
;;;;
;;;;	<ptrees>	: EPSILON
;;;;			| <ptree> <ptrees>
;;;;
;;;;	
;;;;
;;;;	<pformats>	: SHORT{count} <pformats-list>
;;;;
;;;;	<pformats-list>	: EPSILON
;;;;			| <pformat> <pformats-list>
;;;;
;;;;	<pformat>	: <ptree>
;;;;			| <ptext>
;;;;			| <break-id>
;;;;			| <indent-push>
;;;;			| <indent-pop>
;;;;	
;;;;  -page-
;;;;
;;;;	<indent-push>	: <indent-push-id> <SHORT{amt}>
;;;;	<indent-pop>	: <indent-pop-id>
;;;;	
;;;;	<ptext> 	: <ptext-id> <labels> <liveness> <SHORT{count}> <glyphs>
;;;;
;;;;	<liveness>	: <live-id>
;;;;			| <dead-id>
;;;;
;;;;	<glyphs>	: EPSILON
;;;;			| <glyphs-id> <glyphs-array> <glyphs>
;;;;			| <label> <glyphs>
;;;;
;;;;	<ascii>		: EPSILON
;;;;			| <ascii-id> <ascii-array> <ascii>
;;;;			| <label> <ascii>
;;;;
;;;;	<ascii-array>	: <SHORT{count}> (<BYTE> sequence)
;;;;
;;;;	<glyphs-array>	: <SHORT{count}> (<SHORT{font}> <font-indices> sequence)
;;;;
;;;;	<font-indices>	: <BYTE{count}> (<SHORT{font-index}> sequence)
;;;;
;;;;	<labels>	: <SHORT{count}> <label-list>
;;;;
;;;;	<label-list>	: EPSILON
;;;;			| <label><label-list>
;;;;	
;;;;	<label>		: <label-ascii-id> <ascii-array>
;;;;			| <label-index-id> <SHORT{index}>
;;;;	
;;;;	<label-props>	:  <SHORT{count}> <label-prop-list>
;;;;	
;;;;	<label-prop-list>	: EPSILON
;;;;			| <label-prop><label-prop-lists>
;;;;	
;;;;	<label-prop>	: <label> <label>
;;;;	
;;;;	
;;;;
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	<ptree'>	: <ptree-id> <labels> <pformats'>
;;;;			| <ptree-address>
;;;;	  ** basically just ptree with addresses.
;;;;
;;;;	<pformats'>	: SHORT{count} <pformats-list'>
;;;;
;;;;	<pformats-list'>: EPSILON
;;;;			| <pformat'> <pformats-list'>
;;;;
;;;;	<pformat'>	: <ptree'>
;;;;			| <ptext>
;;;;			| <break-id>
;;;;			| <indent-push>
;;;;			| <indent-pop>
;;;;	
;;;;  -page-
;;;;
;;;;	IDs :
;;;;	
;;;;	40's and 50's : Ptree-ids  
;;;;
;;;;	<ptree-id>	: x40
;;;;	<ptext-id>	: x41
;;;;	<pformats-id>	: x42
;;;;	<glyphs-id>	: x43
;;;;	<ascii-id>	: x44
;;;;	<label-ascii-id>	: x45
;;;;	<label-index-id>	: x46
;;;;	<live-id>	: x48
;;;;	<dead-id>	: x49
;;;;	<ptree-address-id>	: x4A
;;;;	
;;;;	<ptree-text-pform-id>	: x4D
;;;;	<ptree-text-id>		: x4E
;;;;	<ptree-pform-id>	: x4F
;;;;	
;;;;
;;;;	<break-id>	: x50
;;;;	<indent-push-id>: x51
;;;;	<indent-pop-id>	: x52
;;;;	
;;;;  -page-
;;;;	
;;;;	80's and 90's : PREQ's and PCMD's
;;;;	A0's and B0's : PQUY's and PNOTE's
;;;;
;;;;	<OpenPCMD>	: x80
;;;;	<ClosePCMD>	: x21
;;;;	<RaisePCMD>	: x22
;;;;	<LowerPCMD>	: x23
;;;;	<ModifyPCMD>	: x24
;;;;	<PositionPQUY>	: xA0
;;;;	
;;;;	C0's and D0's : EREQ's and ECMD's
;;;;	E0's and F0's : EQUY's and ENOTE's
;;;;
;;;;	<RefreshEREQ>   : xC0
;;;;	<SizeENOTE>	: xE0
;;;;
;;;;	
;;;;  -page-
;;;;
;;;;	Label indices : some labels can have associated index which will be
;;;;	 be transmitted in place of the ascii name.
;;;;	
;;;;	At some point assignment and synchonization of label indices can be 
;;;;	automated.
;;;;
;;;;	FTTB: these builtins have been assigned static numbers to demonstrate
;;;;	transmission capability with ptree.
;;;;	
;;;;	!PARAMETER	: 1
;;;;	!SLOT		: 2
;;;;	!ERROR		: 3
;;;;	!DISPLAY	: 4
;;;;	!ABSTRACTION	: 5
;;;;	!NON-MODIFIABLE	: 6
;;;;	!MODIFIABLE	: 7
;;;;	
;;;; -doce- 



;;;;	TODO  : there seem to be a few to many counts in the <glyphs>/<glyphs array>/<font-indices>
;;;;	 maybe some variant syntax that uses a different id byte.
;;;;	


;;;;	Ptree code is common to both edd(edit) and prs(presentation).
;;;;	
;;;;	As such it contains such things as fonts and glyphs.
;;;;	


(defvar *label-indices-array* (make-array 256 :adjustable t :initial-element nil))


(defstruct label-info
  (flags nil)
  (shadows nil)
  (index nil))

(define-flags (label-info)
    ((scope-layout nil t)
     (scope-presentation nil t)
     (extent tree node text nil)))

(defun new-label-info (label
		       shadows
		       scope-layout-p presentation-layout-p
		       extent
		       &optional index)

  (let ((info (init-label-info-flags (make-label-info :shadows shadows :index index)
				     (list (cons 'scope-layout scope-layout-p)
					   (cons 'scope-presentation presentation-layout-p)
					   (cons 'extent extent)))))

    (when index
      (setf (aref *label-indices-array* index) label))
    info))

(defun find-label-by-index (i)
  (or 
   (aref *label-indices-array* i)
   (break "flbi")))


(defun shadows-of-label-info (l) (label-info-shadows l))
(defun index-of-label-info (l) (label-info-index l))



(defun update-label-table (table
			   labels
			   scope-layout-p scope-presentation-p
			   extent
			   &optional indices)

  (if (consp labels)
      (do ((rlabels labels (cdr rlabels))
	   (rindices indices (cdr rindices)))
	  ((null rlabels))
	(setf (gethash (car labels) table)
	      (new-label-info (car rlabels)
			      labels
			      scope-layout-p scope-presentation-p
			      extent
			      (when rindices (car rindices)))))
      (setf (gethash labels table)
	    (new-label-info labels
			    nil
			    scope-layout-p scope-presentation-p
			    extent
			    indices))))

;;;;	RLE TODO : *label-table* should be an edd-resource such that it is not shared.
;;;;	RLE TODO : This is true of all mapping info?? font-tables color-tables??
;;;;	RLE TODO : font-tables and color tables should be shared but mapping info should not.

(defvar *label-table*
  (let ((table (make-hash-table)))
    
    (update-label-table table '!parameter t t 'node 1)
    (update-label-table table '!slot t t 'node 2)
    (update-label-table table '!error t t 'node 3)
    (update-label-table table '!display t t 'node 4)
    (update-label-table table '!abstraction t t 'node 5)
    (update-label-table table '(!non-modifiable !modifiable) t t 'tree (list 6 7))
    (update-label-table table '!cons t t 'node 8)
    (update-label-table table '(!dead !live) t t 'text)

    table))



(defun inform-label (labels
			scope-layout-p presentation-layout-p
			extent)

  (update-label-table *label-table*
		      labels
		      scope-layout-p presentation-layout-p
		      extent))


(defun info-of-label (label)
  (gethash label *label-table*))


(defun inform-labels (labels)
  (dolist (l labels)
    (inform-label (car l)
		  (and (eql 'true (cadr l)) t)
		  (and (eql 'true (caddr l)) t)
		  (cadddr l))))


;;;;	RLE TODO :  could check keys for appropriate scope when informed of mappings.
;;;;	RLE TODO : 
;;;;	RLE TODO : 

;;;;	
;;;;	Shadows Example : You want a subterm to be unmodifiable except for some holes inside of it.
;;;;	  Let modifiable unmodifiable be a shadow class and mark subterm as unmodifiable and
;;;;	  holes as modifiable. Then during recursive descent operators between subterm and hole
;;;;	  are not modifiable, but those in the holes are modifiable.
;;;;	Protect might have better connotation than shadow.

(defun map-labels (map context local format liveness)
  (find-first #'(lambda (entry)
		  ;; when all map lhs labels occur unshadowed in args.
		  (when (forall-p #'(lambda (key)
				      ;; key is a lhs label
				      (or (eql liveness key)
					  (let* ((info (info-of-label key))
						 (shadows (when info (shadows-of-label-info info))))
					    (if (null shadows)
						(or (member key format)
						    (member key local)
						    (member key context))
					        ;; is lhs-label the first member of its class found in env?
						(eql key (or (find-first #'(lambda (label)
									     (when (member label shadows)
									       label))
									 format)
							     (find-first #'(lambda (label)
									     (when (member label shadows)
									       label))
									 local)
							     (find-first #'(lambda (label)
									     (when (member label shadows)
									       label))
									 context)))))))
				  (car entry))
		    (cdr entry)))
	      map))


(defun equal-label-lists-p (a b scopep-f &optional partialp)
  (let ((shadowed nil))
    (and (forall-p #'(lambda (label)
		       (let* ((info (info-of-label label))
			      (shadows (shadows-of-label-info info)))
			 (or (not (funcall scopep-f info))
			     (cond
			       ((and shadows
				     (member label shadowed :test #'member))
				t)
			       ((and shadows)
				(when (eql label (find-first #'(lambda (blabel)
								 (when (member blabel shadows)
								   blabel))
							     b))
				  (push shadows shadowed)
				  t))
			       (t (and (member label b) t))))))
		   a)
  
	 (or partialp
	     (forall-p #'(lambda (label)
			   (let* ((info (info-of-label label))
				  (shadows (shadows-of-label-info info)))
			     (or (not (funcall scopep-f info))
				 (cond
				   ((and shadows
					 (member label shadowed :test #'member))
				    t)
				   ((and shadows)
				    ;; if label had been member of a then it would be in the shadowed now.
				    nil)

				   (t (and (member label a) t))))))
		       b)))))			 
		  





(defstruct glyphs
  (font-modifiers nil)
  ;;(color nil)
  (length nil)
  (count nil)
  (runs nil)
  ;; height
  ;; width
  )

(defun font-modifiers-of-glyphs (g) (glyphs-font-modifiers g))
;;(defun color-of-glyphs (g) (glyphs-color g))
(defun length-of-glyphs (g) (glyphs-length g))
(defun count-of-glyphs (g) (glyphs-count g))
(defun runs-of-glyphs (g) (glyphs-runs g))

(defstruct glyph-run
  (font 0)
  (array nil))

(defun array-of-glyph-run (fr) (glyph-run-array fr))
(defun font-of-glyph-run (fr) (glyph-run-font fr))

(defun new-glyph-index-array (l)
  (make-array l :element-type '(unsigned-byte 16)))

(defun new-glyph-run (font l)
  (make-glyph-run :font font
		  :array (new-glyph-index-array l)))

;; RLE ??? these work fine in lucid, but I wonder if weaker lisps might require
;; RLE ??? coercion to 16 bit bytes before setf.

(defun set-glyph-index (run i index) (setf (aref run i) index))
(defun get-glyph-index (run i) (aref run i))


(defun new-glyphs (num-runs &optional font-modifiers)
  (make-glyphs :font-modifiers font-modifiers
	       :count num-runs
	       :runs (make-array num-runs)))


(defstruct ptree
  (labels nil)
  (formats nil)

  ;; properties whose name is a label with presentation scope and whose
  ;; value is a natural, string or token will be included. The value
  ;; should be coerced to a string?
  (properties nil))

(defun labels-of-ptree (p) (ptree-labels p))
(defun formats-of-ptree (p) (ptree-formats p))
(defun properties-of-ptree (p) (ptree-properties p))


(defstruct (ptree-pform  (:include ptree))
  (id ))

(defstruct (ptree-text (:include ptree)))
(defstruct (ptree-text-pform (:include ptree-pform)))


(defun pform-of-ptree-pform (p) (ptree-pform-id p))
(defun ptrees-of-ptree-pform (p) (ptree-formats p))
(defun text-of-ptree-text (p) (aref (ptree-formats p) 0))

(defun new-ptree-pform (id labels props ptrees)
  (unless (every #'ptree-p ptrees)
    (raise-error (error-message '(ptree read pform child ptree not))))
  
  (make-ptree-pform :id id :labels labels :formats ptrees :properties props))

(defun new-ptree-text (labels text)
  (let ((a (make-array 1)))
    (setf (aref a 0) text)
    (make-ptree-text :labels labels :formats a)))

(defun new-ptree-text-pform (id labels text)
  (let ((a (make-array 1)))
    (setf (aref a 0) text)
    (make-ptree-text-pform :id id :labels labels :formats a)))


;; flags have been replace with labels.

(defun new-term-ptree (formats labels props)
  (make-ptree :formats formats
	      :labels labels
	      :properties props))


(defun new-ptree (labels props formats)
  (make-ptree :formats formats
	      :labels labels
	      :properties props))


(defstruct ptext
  (labels nil)
  (liveness '!dead)
  (glyphs nil))

(defun labels-of-ptext (p) (ptext-labels p))
(defun liveness-of-ptext (p) (ptext-liveness p))
(defun glyphs-of-ptext (p) (ptext-glyphs p))

(defun new-ptext (labels liveness glyphs)
  (make-ptext :labels labels
	      :liveness liveness
	      :glyphs glyphs))

(defun live-ptext-format-p (p)
  (and (ptext-p p)
       (liveness-of-ptext p)))
       

(defvar *null-ptext*
  (let ((a (make-array 1)))
    (setf (aref a 0) (new-ptext nil '!live (new-glyphs 0)))
    a))


;; kludge alert ??
(defun new-parameter-ptree (formats tags)
  
  (if (and formats
	   (onep (length formats))
	   (live-ptext-format-p (aref formats 0)))
      (make-ptree-text :labels (cons '!parameter tags) :formats formats)

      (if (and formats
	       (zerop (length formats)))
	  (progn
	    (format t ".")
	    (make-ptree-text :labels (cons '!parameter tags)
			   :formats *null-ptext*))
	  (progn
	    (message-emit (warn-message '(parameter ptree unexpected) tags))
	    (make-ptree :formats formats
			:labels (cons '!parameter tags))))))



(defun new-pbreak () 'break)
(defun pbreak-p (pf) (eql 'break pf))


(defun new-pindent (amt)
  (if amt
      (cons 'indent amt)
      '(indent)))

(defun pindent-p (pf) (eql 'indent (car pf)))

(defun amt-of-pindent (pf) (cdr pf))

(defun push-pindent-p (pf)
  (and (eql 'indent (car pf))
       (integerp (cdr pf))))

(defun pop-pindent-p (pf)
  (and (eql 'indent (car pf))
       (null (cdr pf))))




;;;; window stuff common to prs and edd
;;;;
;;;;


;;;;	Want hanldes to be treated uniformly, unfortunately due to need to allow
;;;;	for growth of handle arrays makes for additional complexity.
;;;;	
;;;;	
;;;;	Toyed with the idea of having pres-win and edd-win share structure if in
;;;;	same process but decided that might be too restrictive. Ie want to allow
;;;;	pres and edd to have diff size windows. End result would be not much
;;;;	sharing but more complexity.  They do share some code but they are
;;;;	distinct occurences.
;;;;

(defstruct win-handles
  (array (make-array 256 :adjustable t :initial-element nil)))

(defun array-of-win-handles (wh) (win-handles-array wh))

(defun reset-win-handles (wh)
  (let ((a (array-of-win-handles wh)))
    (dotimes (i (array-dimension a 0))
      (setf (aref a i) nil))))
	   

;; INVARIANT: presence of handle indicates window open.

;; INVARIANT: geo (x y w h) always initialized.

(defstruct base-win
  (handle nil)
  (ptree nil)
  x y w h)


(defun handle-of-window (nw) (base-win-handle nw))
(defun ptree-of-window (nw) (base-win-ptree nw))
(defun width-of-window (nw) (base-win-w nw))
(defun height-of-window (nw) (base-win-h nw))

(defun window-open-p (w) (and (base-win-handle w) t))

(defun new-win-handles ()
  (make-win-handles))

(defun lookup-window (handles handle)
  (aref (array-of-win-handles handles) handle)
  )


(defun win-grow-handles (handles)
  ;;	RLE TODO :  put out warning at multiples of 1024
  ;;	RLE TODO :  maybe continuable error as likely loop or leak.
  (let ((array (array-of-win-handles handles)))
    (setf (win-handles-array handles)
	  (adjust-array array (+ 128 (array-dimension array 0)))))
  (values))

(defun win-add-handle (handles handle win)
  (let* ((array (array-of-win-handles handles))
	 (dim (array-dimension array 0)))
    (unless (< handle dim)
      (win-grow-handles handles))
    (let ((array (array-of-win-handles handles)))
      (unless (< handle (array-dimension array 0))
	(system-error (error-message '(edit add-handle) handle)))

      (setf (aref array handle) win
	    (base-win-handle win) handle)))

    (values))


(defun win-delete-handle (handles handle)
  (let* ((array (array-of-win-handles handles))
	 (win (aref array handle)))
    (setf (aref array handle) nil)
    (when (and win (base-win-p win))
      (setf (base-win-handle win) nil))))
	

(defun win-assign-handle (handles win)
  (let* ((array (array-of-win-handles handles))
	 (dim (array-dimension array 0))
	 (foundp nil))
    (do ((i 0 (1+ i)))
	((or foundp
	     (= i dim)))
      (when (null (aref array i))
	(setf (aref array i) win
	      foundp t
	      (base-win-handle win) i)))

    (unless foundp
      (win-grow-handles handles)
      (let ((array (array-of-win-handles handles)))
	(setf (aref array dim) win
	      (base-win-handle win) dim))))

  (values))






;;;;	<open-id>	: x21
;;;;	<close-id>	: x22
;;;;	<raise-id>	: x23
;;;;	<lower-id>	: x24
;;;;	<draw-id>	: x25

;;;;	prs reqs 30's

;;;;	<ptree-id>	: x40
;;;;	<ptext-id>	: x41
;;;;	<pformats-id>	: x42
;;;;	<glyphs-id>	: x43
;;;;	<label-id>	: x44
;;;;
;;;;	<break-id>	: x50
;;;;	<indent-push-id>: x51
;;;;	<indent-pop-id>	: x52
;;;;	
;;;;	<live-id>	: x80
;;;;	<dead-id>	: x81
;;;;
;;;;	

(defparameter  *ptx-noop* #x00)

(defparameter  *ptx-nil* #x10)
(defparameter  *ptx-paren-left* #x12)
(defparameter  *ptx-paren-right* #x13)

(defparameter  *ptx-open-id* #x21)
(defparameter  *ptx-close-id* #x22)
(defparameter  *ptx-raise-id* #x23)
(defparameter  *ptx-lower-id* #x24)
(defparameter  *ptx-draw-id* #x25)

(defparameter  *ptx-ptree-id* #x40)
(defparameter  *ptx-ptext-id* #x41)
(defparameter  *ptx-pformats-id* #x42)
(defparameter  *ptx-glyphs-id* #x43)
(defparameter  *ptx-ascii-id* #x44)
(defparameter  *ptx-label-ascii* #x45)
(defparameter  *ptx-label-index* #x46)

(defparameter  *ptx-live-id*			#x48)
(defparameter  *ptx-dead-id*			#x49)

(defparameter  *ptx-ptree-pform-id*		#x4F)
(defparameter  *ptx-ptree-text-id*		#x4E)
(defparameter  *ptx-ptree-text-pform-id*	#x4D)


(defparameter  *ptx-break-id* #x50)
(defparameter  *ptx-indent-push-id* #x51)
(defparameter  *ptx-indent-pop-id* #x52)

(defparameter  *ptx-open-pcmd* #x80)
;;;;	

(defparameter *glyph-first-byte* (byte 8 8))
(defparameter *glyph-second-byte* (byte 8 0))


#|
;;;;	RLE TODO : whats this.
;; cltl2 lisps don't like base-character type.
;; this array looks like it would be used to convert
;; bytes to chars for char io on char streams.
;; however, not all lisps will work correctly?
;; FTTB, not used so removed.
(defparameter *base-char-array*
  (make-array 256
	      :element-type 'base-character
	      :initial-contents (let ((inits nil))
				  (dotimes (i 256)
				    (push (code-char i) inits))
				  (nreverse inits))))
|#

;;;;	RLE TODO : Lift ptr write functions out of write-ptree.
;;;;	RLE TODO : 
;;;;	RLE TODO : 

(defvar *ppe-stream*)
(defvar *ppe-stream-count*)

(defmacro with-ppe-stream ((stream) &body body)
  `(let ((*ppe-stream* ,stream)
	 (*ppe-stream-count* 0))
    (prog1 (progn ,@body)
      (when (oddp *ppe-stream-count*)
	(write-pbyte *ptx-noop*)))))

(defun ppe-write-pbyte (pbyte)
  (incf *ppe-stream-count*)
  (prl-stream-write pbyte *ppe-stream*))

;; assumes standard-char-p 
(defun ppe-write-character (char)
  (incf *ppe-stream-count*)
  (prl-stream-write (character-to-code char) *ppe-stream*))
	 
(defun ppe-write-short-int (i)
  (ppe-write-pbyte (ldb *glyph-first-byte* i))
  (ppe-write-pbyte (ldb *glyph-second-byte* i)))


(defun write-ptree-to-stream (ptree stream)
  (labels
      ((write-pbyte (byte)
	 (ppe-write-pbyte byte))

       ;;
       ;; RLE TODO : might be best to have ptree rep glyphs this way?
       ;;
       (write-glyphs (glyphs)
	 (let ((l (count-of-glyphs glyphs))
	       (runs (runs-of-glyphs glyphs)))
	   
	   (ppe-write-short-int l)

	   (dotimes (i l)
	     (let* ((run (aref runs i))
		    (r-array (array-of-glyph-run run))
		    (dim (array-dimension r-array 0)))
	       (ppe-write-short-int (font-of-glyph-run run))
	       (ppe-write-pbyte dim)
	       (dotimes (j dim)
		 (ppe-write-short-int (get-glyph-index r-array j)))))))


       (write-ascii-label (label)
	 (let* ((s (string label))
		(l (length s)))
	   (ppe-write-pbyte *ptx-label-ascii*)
	   (ppe-write-short-int l)
	   (dotimes (i l)
	     (ppe-write-character (char s i)))))


       (write-label (label)
	 (let* ((info (info-of-label label))
		(index (when info (index-of-label-info info))))
	   (if index
	       (progn
		 (ppe-write-pbyte *ptx-label-index*)
		 (ppe-write-short-int index))
	       (write-ascii-label label))))

       (count-glyph-sexpr (glyphs-sexpr count)
	 (cond
	   ((null glyphs-sexpr))
	   ((consp glyphs-sexpr)
	    (+ count 
	       (count-glyph-sexpr (car glyphs-sexpr) 0)
	       (count-glyph-sexpr (cdr glyphs-sexpr) 0)))
	   ((glyphs-p glyphs-sexpr)
	    (1+ count))
	   ((symbolp glyphs-sexpr)
	    (1+ count))))


       (write-glyphs-sexpr (glyphs-sexpr)
	 (setf b glyphs-sexpr)
	 (cond
	   ((null glyphs-sexpr))
	    
	   ((consp glyphs-sexpr)
	    (write-glyphs-sexpr (car glyphs-sexpr))
	    (write-glyphs-sexpr (cdr glyphs-sexpr)))

	   ((glyphs-p glyphs-sexpr)
	    (ppe-write-pbyte *ptx-glyphs-id*)
	    (write-glyphs glyphs-sexpr))

	   ((symbolp glyphs-sexpr);; labels-p ???
	    (write-label glyphs-sexpr))))

       (write-labels (labels)
	 (ppe-write-short-int (length labels))
	 (dolist (label labels)
	   (write-label label)))

       (write-label-props (props)
	 (ppe-write-short-int (length props))
	 (dolist (prop props)
	   (write-label (car prop))
	   (write-label (cdr prop))))

       (write-ptext (ptext)
	 (ppe-write-pbyte *ptx-ptext-id*)
	 (write-labels (labels-of-ptext ptext))
	 (ppe-write-pbyte (if (eql '!live (liveness-of-ptext ptext))
			      *ptx-live-id*
			      *ptx-dead-id*))
	 (when (zerop (count-glyph-sexpr (glyphs-of-ptext ptext) 0)) (break))
	 (ppe-write-short-int (count-glyph-sexpr (glyphs-of-ptext ptext) 0))
	 (write-glyphs-sexpr (glyphs-of-ptext ptext)))

       (write-indent (indent)
	 (if (push-pindent-p indent)
	     (progn
	       (ppe-write-pbyte *ptx-indent-push-id*)
	       (ppe-write-short-int (amt-of-pindent indent)))
	     (ppe-write-pbyte *ptx-indent-pop-id*)))

       (write-break ()
	 (ppe-write-pbyte *ptx-break-id*))

       (write-pformat (pformat)
	 (cond
	   ((null pformat))
	   ((ptree-p pformat) (write-ptree pformat))
	   ((ptext-p pformat) (write-ptext pformat))
	   ((pbreak-p pformat) (write-break))
	   ((pindent-p pformat) (write-indent pformat))
	   (t (setf a pformat) (break "unknown pformat")
	      ;; rle todo : need better long term solution, like ignore but warn.
	      )))

       (write-pformats (pformats)
	 (let ((l (array-dimension pformats 0)))
	   (ppe-write-short-int l)
	   (dotimes (i l)
	     (write-pformat (aref pformats i)))))
       

       (write-ptree (ptree)
	 ;;(setf a ptree) (break "wp")
	 (cond
	   ((ptree-text-p ptree)
	    (ppe-write-pbyte *ptx-ptree-text-id*)
	    (write-labels (labels-of-ptree ptree))
	    (write-pformat (text-of-ptree-text ptree))
	    )

	   ((ptree-pform-p ptree)
	    (ppe-write-pbyte *ptx-ptree-pform-id*)
	    (ppe-write-short-int (pform-of-ptree-pform ptree))
	    (write-labels (labels-of-ptree ptree))
	    (write-label-props (properties-of-ptree ptree))
	    (write-pformats (formats-of-ptree ptree))
	   )
	   
	   (t
	    (ppe-write-pbyte *ptx-ptree-id*)
	    (write-labels (labels-of-ptree ptree))
	    (write-label-props (properties-of-ptree ptree))
	    (write-pformats (formats-of-ptree ptree)))
	   ))
       )
	 
  (with-ppe-stream (stream)
    (write-ptree ptree))))



(defun read-ptree-from-stream (stream)
  ;;(setf -s stream)
  (let ((count 0))

    (labels
	((read-pchar ()
	   (let ((byte (read-pbyte)))
	     (if (standard-character-code-p byte)
		 (byte-to-standard-character byte)
		 (raise-error (error-message '(ptree read character standard not) byte)))))

	 (read-pbyte ()
	   (let ((byte   (prl-stream-read stream))
		 )
	     ;;(format t "~a ~x~%" count byte)
	     ;;(incf count)
	     byte))

	 (read-short-int ()
	   (let ((first (read-pbyte))
		 (second (read-pbyte)))
	     ;; RLE PERF: there is probably a more efficient method
	     (+ (* 256 first) second)))

	 (read-ascii-label ()
	   (let* ((l (read-short-int))
		  (s (make-string l)))
	     (dotimes (i l)
	       (setf (char s i) (read-pchar)))
	     (intern s)))

	 (read-index-label ()
	   (find-label-by-index (read-short-int)))

	 (read-label ()
	   (let ((id (read-pbyte)))
	     (cond
	       ((= *ptx-label-ascii* id) (read-ascii-label))
	       ((= *ptx-label-index* id) (read-index-label))
	       (t (raise-error (error-message '(label) id))))))

	 (read-label-list (l)
	   (let ((labels nil))
	     (dotimes (i l)
	       (push (read-label) labels))
	     (nreverse labels)))

	 (read-label-prop-list (l)
	   (let ((label-props nil))
	     (dotimes (i l)
	       (push (cons (read-label) (read-label)) label-props))
	     (nreverse label-props)))
	   
	 (read-labels ()
	   (read-label-list (read-short-int)))

	 (read-label-props ()
	   (read-label-prop-list (read-short-int)))

	 (read-glyphs ()
	   (let* ((l (read-short-int))
		  (glyphs (new-glyphs l))
		  (g-array (runs-of-glyphs glyphs)))
	   
	     (dotimes (i l)
	       (let* ((font (read-short-int))
		      (run-l (read-pbyte))
		      (run (new-glyph-run font run-l))
		      (run-array (array-of-glyph-run run)))
		      
		 (setf a run-l b font c i)

		 (dotimes (j run-l)
		   (set-glyph-index run-array j (read-short-int)))
		 (setf (aref g-array i) run)))

	     glyphs))

	 (read-glyph-sexpr ()
	   (let ((acc nil))
	     (dotimes (i (read-short-int))
	       (let ((id (read-pbyte)))
		 (push (cond
			 ((= *ptx-glyphs-id* id) (read-glyphs))
			 ((= *ptx-label-ascii* id) (read-ascii-label))
			 ((= *ptx-label-index* id) (read-index-label))
			 (t (raise-error (error-message '(glyph-sexpr member) id))))
		       acc)))
	     (if (null (cdr acc))
		 (car acc)
		 (nreverse acc))))

	 (read-ptext ()
	   (new-ptext (read-labels)
		      (if (= *ptx-live-id* (read-pbyte)) '!live '!dead)
		      (read-glyph-sexpr)))

	 (read-pformat ()
	   (let ((id (read-pbyte)))
	     (cond
	       ((= *ptx-ptext-id* id) (read-ptext))
	       ((= *ptx-ptree-id* id) (read-ptree))
	       ((= *ptx-ptree-pform-id* id) (read-ptree-pform))
	       ((= *ptx-ptree-text-id* id) (read-ptree-text))
	       ((= *ptx-ptree-text-pform-id* id) (read-ptree-text-pform))
	       ((= *ptx-break-id* id) (new-pbreak))
	       ((= *ptx-indent-push-id* id) (new-pindent (read-short-int)))
	       ((= *ptx-indent-pop-id* id) (new-pindent nil))
	       (t (raise-error (error-message '(pformat) id))))))

	 (read-pformats ()
	   (let* ((l (read-short-int))
		  (f-array (make-array l)))
	     (dotimes (i l)
	       ;;(setf a i b l) (break "rp")
	       (setf (aref f-array i) (read-pformat)))
	     f-array))
       
	 (read-ptree ()
	   (new-ptree (read-labels)
		      (read-label-props)
		      (read-pformats)))

	 (read-ptree-pform ()
	   (new-ptree-pform (read-short-int)
			    (read-labels)
			    (read-label-props)
			    (read-pformats)))
       
	 (read-ptree-text ()
	   (new-ptree-text (read-labels)
			   (read-pformat)))

	 (read-ptree-text-pform ()
	   (new-ptree-text-pform (read-short-int)
				 (read-labels)
				 (read-pformat)))
	 )

      (with-tag '(ptree read stream)
	(let ((pformat (read-pformat)))
	  (unless (ptree-p pformat)
	    (raise-error (error-message '(ptree read))))
	  pformat)))))



(defun read-ptree-from-file (filename)
  (with-prl-open-file (stream filename in)
    (read-ptree-from-stream stream)))
					

(defun read-resources (filename f)
  (with-open-file (s filename :direction :input)
    (do* ((kind (read s nil nil) (read s nil nil))
	  (value (when kind (read s nil nil)) (when kind (read s nil nil))))
	 ((null kind))
      (funcall f kind value))))
	 
