
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; -docs- (mod trm data)
;;;;
;;;;	Ascii term syntax:
;;;;
;;;;	<term>			: <string>{<parameter-list>}(<bound-term-list>)
;;;;				| <string>(<bound-term-list>)
;;;;				| <string>{<parameter-list>}
;;;;				| <string>{}
;;;;				| <variable-id>[<term-list>]
;;;;				| <variable-id>
;;;;
;;;;	<parameter-list>	: <parameter> | <parameter>, <parameter-list>
;;;;	<parameter>		: <parameter-value>:<string>
;;;;	<parameter-value>	: <string>
;;;;
;;;;	<bound-term-list>	: <bound-term> 
;;;;				| <bound-term>; <bound-term-list>
;;;;
;;;;	<bound-term>		: <binding-list>.<term> | .<term> | <term>
;;;;	<binding-list>		: <variable-id> | <variable-id>, <binding-list>
;;;;	<variable-id>		: <parameter-value>
;;;;
;;;;
;;;;	.<term> indicates a binding list containing single null string.
;;;;	<term> indicates a null binding list.
;;;;	
;;;;	Characters \{}()[]:;,. and Space, Tab, Newline and Return must be
;;;;	escaped to be included in a string. They are otherwise interpreted as
;;;;	syntax delimiters.
;;;;
;;;;	An escape character followed by a hexadecimal digit, ie one of
;;;;	0123456789abcdefABCDEF, indicates an embedded unicode character. The
;;;;	four characters following the escape are interpreted as ascii
;;;;	hexadecimal characters. They can be converted to a two byte unicode
;;;;	character.  Thus the hexadecimal digits should not be escaped except to
;;;;	signal an embedded unicode character.
;;;;	
;;;;  -page-
;;;;
;;;;	There is further syntax encoded into the parameter values:
;;;;	
;;;;	  Any string not starting with %  represents itself.
;;;;	  %%xxx represents %xxx
;;;;	  %axxx represents the abstraction meta-var with identifying string xxx.
;;;;	  %dxxx represents the display meta-var with identifying string xxx.
;;;;	  %Sxxx represents the normal op-parm value slot with string xxx.
;;;;	  %Axxx represents the abstraction meta-var slot with string xxx.
;;;;	  %Dxxx represents the display meta-var slot with string xxx.
;;;;	
;;;;	  Any other string starting with % is reserved for future use.
;;;;	
;;;;	Need to embed obid's into mnemonic strings for external use.
;;;;	  consider obid-rep : object_id -> string
;;;;		   obid-unrep : string -> object-id
;;;;	  %oxxx represents (obid-unrep xxx)
;;;;	
;;;;	  do not want this encoding to be applied except on demand
;;;;	
;;;;	 maybe %Nyyxxx means on demand yy encoding of xxx
;;;;	 then %NOrxxx means real obid
;;;;	 then %NOaxxx, %NOSxxx,%NOAxxx, %NOdxxx, %NODxxx mean appropriate obid meta/slot values. 
;;;;	
;;;;	
;;;;	Characters \{}()[]:;,. and Space, Tab, Newline and Return must be escaped to
;;;;	be included in a string. They are otherwise interpreted as syntax delimiters.
;;;;	

(defun demand-encoded-string-p (s)
  (and (> (length s) 4)
       (char= #\% (char s 0))
       (char= #\N (char s 1))))
  
(defun repd-oid-string-p (s)
  (and (demand-encoded-string-p s)
       (char= #\O (char s 2))))


;;;;
;;;;	term-to-standard-character-string (<term>)	: <string>
;;;;	standard-character-string-to-term (<string>)	: <term>
;;;;
;;;;	
;;;;	parameter-to-string (<parameter>)		: <string>
;;;;	string-to-parameter (<string> <parm-type>)		: <parameter>
;;;;	
;;;;	parameter-value-to-string (<parm-value> <parm-type>)	: <string>
;;;;	string-to-parameter-value (<string> <parm-type>)	: <parm-value>
;;;;
;;;; -doce-



;; In tt ssl he has binding list as list of variables it should be list of parm strings.

;;;;	Unicode chars : internal representation will be ascii escape followed by two byte
;;;;	unicode ascii chars.
;;;;	
;;;;	ascii rep will be \xxxx where x are ascii reps of hex chars.
;;;;	
;;;;	this conversion will take place in term-to-ascii and ascii-to-term.
;;;;	escape string needs to reconize embedded unicode and do blowup.
;;;;
;;;;	ascii chars are following  (all other chars even if < x100 will be unicode chars)
;;;;

;;;;	Whitespace:
;;;;	
;;;;	  space   20
;;;;	  tab     09
;;;;	  newline 0c
;;;;	  linefeed 0a
;;;;	  vt       0b
;;;;	  ff       0c
;;;;	  printable ISO latin  21-7e and a0-ff



;;;;	6) Representing op-parm values as strings. A proposal.
;;;;	
;;;;	  Plainly, how one represents op-parm values of the various kinds other
;;;;	  than values proper is not so important internally, and since there is
;;;;	  little commitment to the methods used in the past, we propose a simple
;;;;	  method here.
;;;;	  NotaBene: This layer of encoding various kinds of op-parm values is
;;;;	            not to be confused with representing general-form terms
;;;;	            as character strings.
;;;;	
;;;;	  One requirement is that we be able to continue extending
;;;;	  the forms of op-parms in the future.
;;;;	
;;;;	  Let us take % as our special character for what follows. And let xxx stand
;;;;	  for any string.
;;;;	
;;;;	  Any string not starting with %  represents itself.
;;;;	  %%xxx represents %xxx
;;;;	  %axxx represents the abstraction meta-var with identifying string xxx.
;;;;	  %dxxx represents the display meta-var with identifying string xxx.
;;;;	  %Sxxx represents the normal op-parm value slot with string xxx.
;;;;	  %Axxx represents the abstraction meta-var slot with string xxx.
;;;;	  %Dxxx represents the display meta-var slot with string xxx.
;;;;	
;;;;	  Any other string starting with % is reserved for future use.
;;;;	  (Perhaps the user should be notified if such an unexplained op-parm value
;;;;	   is encountered.  Also don't assume that any new forms of these "headers"
;;;;	   will be only two characters.)


;;;
;;;	Output
;;;


(defconstant *compression-symbol-type* 0)
(defconstant *compression-opid-type* 0)
(defconstant *compression-binding-type* 1)
(defconstant *compression-parameter-type* 2)
(defconstant *compression-operator-type* 3)
(defconstant *compression-term-type* 4)

(defconstant *compression-num-types* 5) ;; 0-4 inclusive

;; should be c5?
(defconstant *hex-word-byte* #xC5)

;;;;	<hexword>	: #b11LLU100


;; need decimal default to be able to read old terms.
;; also use decimal as default for ridiculously big bignums.
(defmacro accumulate-int (n)
  (let ((nn (gensym)))
    `(let ((,nn ,n))
      (if (< ,nn 1000)
	  (if (< ,nn 100)
	      (if (< ,nn 10)
		  (vector-push-extend (+ ,nn #.(char-code #\0)) *byte-buffer*)
		  (let ((digit1 (mod ,nn 10)))
		    (vector-push-extend (+ (floor (- ,nn digit1) 10) #.(char-code #\0)) *byte-buffer*)
		    (vector-push-extend (+ digit1 #.(char-code #\0)) *byte-buffer*)
		    ))
	      (let* ((digit1 (mod ,nn 10))
		     (digit2 (mod (floor (- ,nn digit1) 10) 10)))
		(vector-push-extend (+ (floor (- ,nn digit2 digit1) 100) #.(char-code #\0)) *byte-buffer*)
		(vector-push-extend (+ digit2 #.(char-code #\0)) *byte-buffer*)
		(vector-push-extend (+ digit1 #.(char-code #\0)) *byte-buffer*)))
	      (let ((l (ceiling (integer-length ,nn) 8)))
		(if (< l 256)
		    (let ((fp (fill-pointer *byte-buffer*)))
		      (byte-buffer-space (+ l 2))
		      (setf (aref *byte-buffer* fp) *hex-word-byte*)
		      (setf (aref *byte-buffer* (1+ fp)) (ldb (byte 8 0) l))
		      (accumulate-int-hex-bytes (+ 2 fp) ,nn))
		    (accumulate-standard-string (princ-to-string ,nn))))))))
	      



;; walk-term-ascii (<term> <closure{acc-f}) 	: <byte-array>
;; acc-f (<byte>) : NULL

(defun walk-term-ascii (term &optional acc-f)
  (labels ((accumulate-string (s)
	     (string-to-byte-accumulator s *ascii-escape-sbits* acc-f))

	   (visit-term (term)
	     (cond
	       ((variable-term-p term)
		(visit-parameter-value (id-of-variable-term term) *variable-type*)
		(unless (variable-p term)
		  (walk-list-delimited (bound-terms-of-term term)
				       #'(lambda (bound-term)
					   (visit-term (term-of-bound-term bound-term)))
				       acc-f
				       isemicolon ilsquare irsquare)))
		 
	       (t
		(let ((parameters (parameters-of-term term))		   
		      (bound-terms (bound-terms-of-term term)))
		  (accumulate-string (string (id-of-term term)))
		  (when parameters
		    (walk-list-delimited parameters
					 #'visit-parameter
					 acc-f
					 icomma ilcurly ircurly))
		  (when (or bound-terms (null parameters))
		    (walk-list-delimited bound-terms
					 #'visit-bound-term
					 acc-f
					 isemicolon ilparen irparen))))))
	   (visit-bound-term (bound-term)
	     (let ((bindings (bindings-of-bound-term-n bound-term)))
	       (if bindings
		   (progn (walk-list-delimited bindings
					       #'(lambda (v)
						   (visit-parameter-value v *variable-type*))
					       acc-f
					       icomma nil nil)
			  (funcall acc-f idot)
			  (visit-term (term-of-bound-term bound-term)))
		   (visit-term (term-of-bound-term bound-term)))))

	   (visit-parameter (p)
	     (let ((type (type-of-parameter p)))
	       (visit-parameter-value (value-of-parameter-n p) type)
	       (funcall acc-f icolon)
	       (accumulate-string (type-to-short-string type))))

	   ;; RLE PERF TODO PROJECT : creating sexpr is unnecessary.
	   ;; should have a parameter-value-to-string (v acc-f), acc-f (CHAR or STRING)
	   (visit-parameter-value (val type)
	     (let ((sexpr (parameter-value-to-sexpr val type)))
	       (walk-parameter-sexpr-ascii sexpr acc-f))))		 
    
    (visit-term term)))

;; expect (funcall accumulate-f ipercent) has happened.
(defun parameter-sexpr-ascii-aux (sexpr accumulate-f)
  (labels ((accumulate-string (s)
	     (string-to-byte-accumulator s *ascii-escape-sbits* accumulate-f)))
    ;;(setf -s sexpr -a accumulate-f) (break "psaa")
    (case (car sexpr)
      (a (cond
	   ((stringp (cdr sexpr))
	    (funcall accumulate-f (character-to-code #\a))
	    (accumulate-string (cdr sexpr)))
	   ((and (consp (cdr sexpr)) (eql 's (cadr sexpr)))
	    (funcall accumulate-f (character-to-code #\A))
	    (accumulate-string "slot"))
	   (t (system-error (error-message '(term-to-ascii parameter-value abstraction-meta)
					  (princ-to-string sexpr))))))
      (d (cond
	   ((stringp (cdr sexpr))
	    (funcall accumulate-f (character-to-code #\d))
	    (accumulate-string (cdr sexpr)))
	   ((and (consp (cdr sexpr)) (eql 's (cadr sexpr)))
	    (funcall accumulate-f (character-to-code #\D))
	    (accumulate-string "slot"))
	   (t (system-error (error-message '(term-to-ascii parameter-value display-meta)
					  (princ-to-string sexpr))))))
      (s
       (funcall accumulate-f (character-to-code #\S))
       (accumulate-string "slot"))
		      
      (otherwise (system-error (error-message '(term-to-ascii parameter-value cons)
					     (princ-to-string sexpr)))))))

;; accumulate-f(BYTE) : NULL
(defun walk-parameter-sexpr-ascii (sexpr accumulate-f)
  ;;(setf -a sexpr) (format t "~a" -a) (break "wpsa")
  (labels ((accumulate-string (s)
	     (string-to-byte-accumulator s *ascii-escape-sbits* accumulate-f)))

    (cond
      ((stringp sexpr)
       (when (and (> (length sexpr) 0) (eql #\% (char sexpr 0)))
	 (funcall accumulate-f ipercent))
       (accumulate-string sexpr))

      ((consp sexpr)
       (funcall accumulate-f ipercent)
       (parameter-sexpr-ascii-aux sexpr accumulate-f))

      (t (system-error (error-message '(term-to-ascii parameter-value)
				     (princ-to-string sexpr)))))))


;; not necessarily : assume within dynamic scope of with-fbb
;; want to accumulate in a byte buffer then copy to another to be saved in cache.
;;  then let caller copy to fbb.
(defconstant *bb-slot* (string-to-byte-array "%Sslot" *ascii-escape-sbits*))

;; twould be nice if parameterized by buffer so as to allow direct
;; write if we do not want caching. difficult since fbb doesn't have fill-pointer so
;; can't just expose array. But could copy out of byte-buffer to fbb instead of creating
;; new byte array to be cached. Ie move caching out.

(defun parameter-to-fbb-cache (p)

  (or (parameter-fbb-cache p)

      (setf (parameter-fbb-cache p)
	    (with-byte-accumulator (#+fbbstring 'string #-fbbstring 'byte-array)
	      (accumulate-parameter p)
	      ;;(setf -p p)  (break "ptfc")
	      ))))


(defun accumulate-parameter (p)

  (let ((value (value-of-parameter p))
	(type (type-of-parameter p)))
    
    ;;(setf v value b type) (break)
    (cond
      ((extended-parameter-value-p value)
       (cond

	 ((slot-parameter-value-p value)
	  (let ((s (descriptor-of-slot-parameter-value value)))
	    (if (meta-variable-id-p s)
		(progn
		  (accumulate-byte-m ipercent)
		  (accumulate-byte-m (meta-type-id-code (type-of-meta-variable-id s) t)))
		(accumulate-array *bb-slot*))))
		 
	 ((error-parameter-value-p value)
	  (walk-parameter-sexpr-ascii (sexpr-of-error-parameter-value value)
				      #'accumulate-byte))

	 (t (system-error (error-message '(accumulate-parameter extended))))))

      ((real-parameter-value-p value type)
       (if (or (natural-type-p type) (time-type-p type))
	   (accumulate-int value)
	   (let ((s (real-parameter-value-to-string value type)))
	     (when (and (> (length s) 0) (eql #\% (char s 0)))
	       (accumulate-byte-m ipercent))
	     ;; PERF could define an accumulate string macro.
	     (string-to-byte-accumulator s *ascii-escape-sbits* #'accumulate-byte))))

      ((meta-parameter-value-p value)
       (accumulate-byte-m ipercent)
       (accumulate-byte-m (meta-type-id-code (type-of-meta-variable-id value) nil))
       (string-to-byte-accumulator (variable-id-to-string value) *ascii-escape-sbits* #'accumulate-byte))
	      
      (t ;;(setf v value b type) (break "foo")
       (system-error (error-message '(parameter-value-to-sexpr value?)))
	 ))

    (accumulate-byte-m icolon)
    ;;(break  "yy")

    (accumulate-standard-string (type-to-short-string type))))



;; when building should copy from input buffer rather than byte at a time read.
;; byte array will be escaped.
;; need to accumulate string 
(defun scan-compressed-parameter-value (code)
  (if (= code *hex-word-byte*)
      (prog2
	  (scan-bump)	; need to catch escape so can't scan-bump.
	  (scan-int-bytes)
	(scan-next))
      (raise-error (error-message '(scan parameter-value compressed) code))))


(defun parameter-from-fbb-cache ()

  (let ((p nil))
    (let ((fbb-cache (with-byte-accumulator (#-fbbstring 'byte-array #+fbbstring 'string)
		       (if (= *hex-word-byte* (scan-cur-byte))
			   (progn
			     (accumulate-byte-m *hex-word-byte*)
			     (scan-bump)
			     (let ((l (scan-cur-byte)))
			       (accumulate-byte-m l)
			       (scan-bytes-n l)
			       (scan-bump)
			       ))
			   (scan-bytes *ascii-escape-sbits*))
		       (scan-byte icolon)
		       (accumulate-byte-m icolon)
		       (scan-bytes *ascii-escape-sbits*)
		       
		       (setf p (byte-array-to-parameter *byte-buffer*)))))

      (setf (parameter-fbb-cache p) fbb-cache))
    p))
    


(defvar *stamp-string-hash* (make-hash-table :size 16000 :test #'equal))
(defvar *stamp-string-hash-hits-count* 0)

(defun oid-string-to-oid-parm (s)
  (let ((h (gethash s *stamp-string-hash*)))
    (when h (incf *stamp-string-hash-hits-count*))
    (or h
	(let ((ss (with-string-scanner (s) (scan-ascii-string))))
	  (let ((p (oid-parameter (new-oid-ss ss (term-to-stamp (standard-character-string-to-term ss))))))
	    (setf (gethash s *stamp-string-hash*) p)
	    p)))))


(defun byte-array-to-parameter (bb)
  (declare (type (vector (unsigned-byte 8)) bb))

  (let ((first (aref bb 0)))

    (cond
      
      ((= first *hex-word-byte*)
       (let ((l (aref bb 1)))
	 (instantiate-parameter
	  (hexadecimal-bytes-to-int bb 2 l)
	  (let ((ll (length bb)))
	    (if (and (= ll (+ 4 l))
		     (= #.(char-code #\n) (aref bb (+ 3 l))))
		*natural-type*
		(type-id-to-type (intern-system (bytes-to-string bb (+ 3 l)))))))))

      ((let ((fp (fill-pointer bb)))
	 (and (not (scan-escape-p))
	      (not (= ipercent first))
	      (eql #.(char-code #\o) (aref bb (1- fp)))
	      (eql icolon (aref bb (- fp 2)))))

	(oid-string-to-oid-parm (bytes-to-string bb 0 (- (fill-pointer bb) 2))))
      
      ;; check for string with no escape sequences either unicode or syntactic or meta syntatic(ie %).
      ((and (not (scan-escape-p))
	    (not (= ipercent first))
	    (let ((ic nil))
	      (do ((i 0 (1+ i)))
		  ((let ((b (aref bb i)))
		     (or (when (= icolon b) (setf ic i) t)
			 ;; must be escaped if not standard char thus sufficient to check for escape char.
			 (= iescape b)))))
	      ;;(setf -ic ic -bb bb) (break "bb")
	      (when ic
		(if (= (length bb) (1+ ic))
		    (case (aref bb (1+ ic))
		      (#.(char-code #\s)	(instantiate-parameter (bytes-to-string bb 0 ic) *string-type*))
		      (#.(char-code #\t)	(instantiate-parameter (intern-system (bytes-to-string bb 0 ic)) *token-type*))
		      (#.(char-code #\n)	(instantiate-parameter (decimal-bytes-to-int bb (1- ic))
								       *natural-type*))
		      (#.(char-code #\o)	(oid-string-to-oid-parm (bytes-to-string bb 0 ic)))
		      (otherwise
		       (let ((type (type-id-to-type (intern-system (bytes-to-string bb (1+ ic))))))
			 (instantiate-parameter (maybe-string-to-parameter-value (bytes-to-string bb 0 ic) type)
						type))))
		    (let ((type (type-id-to-type (intern-system (bytes-to-string bb (1+ ic))))))
		      (instantiate-parameter (maybe-string-to-parameter-value (bytes-to-string bb 0 ic) type)
					     type)))))))
	 
	
      (t (with-byte-scanner (bb)
	   ;;(setf -bb bb) ;;(break "gu")

	   (let ((second nil))
	     (let ((s (if (= ipercent first)
			  (progn
			    (scan-bump)
			    (setf second (scan-cur-byte))
			    (if (= second ipercent)
				(scan-ascii-string)
				(progn (scan-bump) (scan-ascii-string))))
			  (scan-ascii-string))))

	       (scan-byte icolon)
	       (let ((type (type-id-to-type (intern-system (scan-ascii-string)))))
		 (unless (scan-eof-p)
		   (raise-error (error-message '(scan parameter byte-array eof not))))
		 
		 ;;(when (eq type *oid-type*) (setf -a first -b second -c s -d type) (break "fu"))
		 (cond

		   ((null second)
		    (instantiate-parameter (maybe-string-to-parameter-value s type) type))

		   (t (instantiate-parameter
		       (case second
			 (#.(char-code #\%)	(maybe-string-to-parameter-value s type))
			 (#.(char-code #\d)	(get-display-meta-variable-id s))
			 (#.(char-code #\a)	(get-abstraction-meta-variable-id s))
			 (#.(char-code #\S)	(slot-parameter-value "slot"))
			 (#.(char-code #\D)	(slot-parameter-value (get-display-meta-variable-id "slot")))
			 (#.(char-code #\A)	(slot-parameter-value (get-abstraction-meta-variable-id "slot")))
			 (otherwise (raise-error (error-message '(scan parameter byte-array extended)
								second))))
		      type)))))))))))



(defun term-sig-to-standard-character-string (sig)
  (with-output-to-string (s)

    (labels ((accumulate (a)
	       (write-string a s))

	     (accumulate-escape (s)
	       (accumulate (escape-string s *ascii-escape-sbits*))))

      (accumulate-escape (string (id-of-term-sig sig)))
      (let ((parameters (parameters-of-term-sig sig))
	    (arities (arities-of-term-sig sig)))
	(when parameters
	  (walk-list-delimited parameters
			       #'(lambda (p) (accumulate-escape
					      (type-id-to-short-string p)))
			       #'accumulate
			       "," "{" "}"))
	(when (or arities (null parameters))
	  (walk-list-delimited arities
			       #'(lambda (a) (accumulate (princ-to-string a)))
			       #'accumulate
			       ";" "(" ")"))))))



;; RLE TODO : when writing to tt
;; RLE TODO : twould be better to take stream arg and pass to byte accumulator
;; RLE TODO : then use accumulated-length to write length. Saves copying accumulated array.
(defun term-to-ascii-array (term)
  (with-byte-accumulator ('byte-array)
    (walk-term-ascii term
		     #'accumulate-byte)))

(defun term-to-standard-character-string (term)
  (with-byte-accumulator ('standard-string)
    (walk-term-ascii term
		     #'accumulate-byte)))


(defun parameter-to-string (parameter)
  (with-byte-accumulator ('standard-string)
			 (let ((type (type-of-parameter parameter)))
			   (walk-parameter-sexpr-ascii
			    (parameter-value-to-sexpr (value-of-parameter-n parameter)
						      type)
			    #'accumulate-byte)
			   (accumulate-byte icolon)
			   (string-to-byte-accumulator (type-to-short-string type)
						       *ascii-escape-sbits*
						       #'accumulate-byte))))

(defun parameter-value-to-string (value type)
  (with-byte-accumulator ('standard-string)
			 (walk-parameter-sexpr-ascii
			  (parameter-value-to-sexpr value type)
			  #'accumulate-byte)))
			   

;;;
;;;	Input
;;;

(defun scan-ascii-string ()
  (scan-string *ascii-escape-sbits*))


(defun scan-at-any-open-paren ()
  (let ((byte (scan-cur-byte)))
    (or (= byte ilparen)
	(= byte ilcurly)
	(= byte ilsquare))))


(defun scan-term-ascii ()
  (labels
      ((scan-term (&optional s)
	 (let ((id (if s s (scan-ascii-string))))

	   (scan-whitespace)
	   (cond
	     ((or (scan-eof-p)
		  (not (scan-at-any-open-paren)))
	      (variable-term (get-variable-id id)))

	     ((scan-at-byte-p ilsquare)
	      (variable-term (get-variable-id id)
			     (scan-delimited-list #'(lambda () (instantiate-bound-term (scan-term)))
						  ilsquare irsquare
						  #'(lambda () (scan-byte isemicolon)))))
	     (t
	      (instantiate-term
	       (instantiate-operator (intern-system id)
				     (prog1
					 (when (scan-at-byte-p ilcurly)
					   (scan-delimited-list
					    #'(lambda ()
						(let ((sexpr (scan-parameter-value-sexpr (scan-ascii-string)))
						      (type (progn (scan-byte icolon)
								   (type-id-to-type
								    (intern-system (scan-ascii-string))))))
						  (instantiate-parameter (sexpr-to-parameter-value sexpr type)
									 type)))
					    ilcurly ircurly
					    #'(lambda ()
						(scan-byte icomma))))
				       (scan-whitespace)))
	       (when (scan-at-byte-p ilparen)
		 (scan-delimited-list #'scan-bound-term
				      ilparen irparen
				      #'(lambda() (scan-byte isemicolon)))))))))

       (string-to-binding (s)
	  (sexpr-to-parameter-value (scan-parameter-value-sexpr s) *variable-type*))

       (scan-bound-term ()
	 (if (scan-at-byte-p idot)
	     (progn
	       (scan-next)
	       (instantiate-bound-term (scan-term)  (list (get-dummy-variable-id))))
	     (let ((s (scan-ascii-string)))
	       (scan-whitespace)
	       ;;(setf a s)
	       (cond
		 ((scan-at-any-open-paren)
		  (instantiate-bound-term (scan-term s)))
		 ((or (scan-at-byte-p irparen)
		      (scan-at-byte-p irsquare)
		      (scan-at-byte-p isemicolon))
		  (instantiate-bound-term (variable-term (string-to-binding s))))
		 (t (let ((bindings (cons (string-to-binding s)
					  (if (scan-at-byte-p idot)
					      (progn
						(scan-next)
						nil)
					      (scan-delimited-list #'(lambda ()
								       (string-to-binding (scan-ascii-string)))
								   icomma idot
								   #'(lambda ()
								       (scan-byte icomma)))))))
		     (scan-whitespace)
		     ;; #\. scanned by scan-delimited-list.
		     ;;(scan-byte idot)
		     (instantiate-bound-term (scan-term) bindings)))))))
       )

    (scan-term)))

;; no opids
;; no bindings
;; no abbreviated variables

(defconstant *opid-kludge* '|!!OPIDKludge|)

(defun scan-term-ascii-standard ()
  (labels
      ((scan-term ()
	 (scan-whitespace)
	 (let ((parms (scan-delimited-list
		       #'(lambda ()
			   (let ((sexpr (scan-parameter-value-sexpr (scan-ascii-string)))
				 (type (progn (scan-byte icolon)
					      (type-id-to-type
					       (intern-system (scan-ascii-string))))))
			     (instantiate-parameter (sexpr-to-parameter-value sexpr type)
						    type)))
		       ilcurly ircurly
		       #'(lambda ()
			   (scan-byte icomma))))
	       (subterms (progn (scan-whitespace)
				(scan-delimited-list #'scan-term
						     ilparen irparen
						     #'(lambda() (scan-byte isemicolon))))) )
	   (instantiate-term 
	    (let ((p (car parms)))
	      (cond
		((null p)
		 (instantiate-operator *opid-kludge* parms))
		((and (opid-type-p (type-of-parameter p))
		      (real-parameter-p p))
		 (instantiate-operator (value-of-parameter-r p)
				       (cdr parms)))
		(t (instantiate-operator *opid-kludge* parms))))
	    (mapcar #'instantiate-bound-term subterms)))) )

    (scan-term)))

(defun walk-term-ascii-standard (term &optional acc-f)
  (labels ((accumulate-string (s)
	     (string-to-byte-accumulator s *ascii-escape-sbits* acc-f))

	   (visit-term (term)
	     (let ((id (id-of-term term))
		   (parameters (parameters-of-term term))		   
		   (bound-terms (bound-terms-of-term term)))

	       (unless (eql id *opid-kludge*)
		 (push (opid-parameter id) parameters))
	       
	       (walk-list-delimited parameters
				    #'visit-parameter
				    acc-f
				    icomma ilcurly ircurly)
	       (walk-list-delimited bound-terms
				    #'visit-bound-term
				    acc-f
				    isemicolon ilparen irparen)))

	   (visit-bound-term (bound-term)
	     (let ((bindings (bindings-of-bound-term-n bound-term)))
	       (if bindings
		   (raise-error (error-message '(walk-term-ascii-standard bindings) term))
		   (visit-term (term-of-bound-term bound-term)))))

	   (visit-parameter (p)
	     (let ((type (type-of-parameter p)))
	       (visit-parameter-value (value-of-parameter-n p) type)
	       (funcall acc-f icolon)
	       (accumulate-string (type-to-short-string type))))

	   ;; RLE PERF TODO PROJECT : creating sexpr is unnecessary.
	   ;; should have a parameter-value-to-string (v acc-f), acc-f (CHAR or STRING)
	   (visit-parameter-value (val type)
	     (let ((sexpr (parameter-value-to-sexpr val type)))
	       (walk-parameter-sexpr-ascii sexpr acc-f))))		 
    
    (visit-term term)))


(defun scan-term-sig-ascii ()
  (list* (intern-system (scan-ascii-string))
	 (progn (scan-whitespace)
		(when (scan-at-byte-p ilcurly)
		      (scan-delimited-list
		       #'(lambda ()
			   (unalias-typeid (intern-system (scan-ascii-string))))
		       ilcurly ircurly
		       #'(lambda ()
			   (scan-byte icomma)))))
	 (progn (scan-whitespace)
		(when (scan-at-byte-p ilparen)
		      (scan-delimited-list #'scan-decimal-num
					   ilparen irparen
					   #'(lambda() (scan-byte isemicolon)))))))


(defun standard-character-string-to-term (s)
  ;;(format t "SCST")
  ;;(setf a s)
  (with-string-scanner (s)
    (scan-term-ascii)))

(defun standard-character-string-to-term-sig (s)
  ;;(setf a s)
  (with-string-scanner (s)
    (scan-term-sig-ascii)))


(defun scan-parameter-demand-value-sexpr (s)
  ;; know that %N were first two characters. Also checked third
  ;; character was kind we expected.
  (let ((c (char s 3)))
    (cons (char s 2)
	  (let ((restindex 4))
	    (cond
	      ((char= #\r c) (subseq s restindex))

	      ((char= #\S c) '(s))
	      ((char= #\d c) (cons 'd (subseq s restindex)))
	      ((char= #\a c) (cons 'a (subseq s restindex)))
	      ((char= #\D c) (cons 'd (cons 's (subseq s restindex))))
	      ((char= #\A c) (cons 'a (cons 's (subseq s restindex))))

	      (t  (message-emit (warn-message '(ascii read parameter demand escape unknown) s))
		  s))))))

(defun scan-parameter-value-sexpr (s)
  (let ((len (length s)))
    (if (zerop len)
	s
	(let ((first (char s 0)))
	   
	  (if (char= #\% first)
	      (let ((second (char s 1)))
		(cond
		  ((char= #\% second) (subseq s 1))
		  ((char= #\S second) '(s))
		  ((char= #\d second) (cons 'd (subseq s 2)))
		  ((char= #\a second) (cons 'a (subseq s 2)))
		  ((char= #\D second) (cons 'd (cons 's (subseq s 2))))
		  ((char= #\A second) (cons 'a (cons 's (subseq s 2))))
		  ;; pass N's thru as themselves. Unencoded on demand.
		  ((char= #\N second) s)
		  (t  (break "arpeu") (message-emit (warn-message '(ascii read parameter escape unknown) s))
		     s)))
	      s)))))



(defun string-to-parameter (string)
  (with-string-scanner (string)
		       (let ((sexpr (scan-parameter-value-sexpr (scan-ascii-string)))
			     (type (progn (scan-byte icolon)
					  (type-id-to-type
					   (intern-system (scan-ascii-string))))))
			 (instantiate-parameter (sexpr-to-parameter-value sexpr type)
						type))))

(defun string-to-parameter-value (string type)
  (with-string-scanner (string)
		       (let ((sexpr (scan-parameter-value-sexpr (scan-ascii-string))))
			 (sexpr-to-parameter-value sexpr type))))
   

;;;
;;;	Compression
;;;

;;;;	RLE TODO : one way streams like journals and object db can do variant
;;;;	RLE TODO : which allows dynamic reuse of compression indices.

;;;;	RLE TODO : still may be worthwhile savings with factoring ops on iteration.
;;;;	RLE TODO : want general purpose invisible solution. could use something
;;;;	RLE TODO : like quoting to make invisible.

;;;;	
;;;;	Compression:
;;;;	  - compress terms in memory.
;;;;	  - compress ascii string representation of terms.
;;;;	      * reduces size of tranmissions.
;;;;	      * avoids parsing/unparsing of duplicated data.
;;;;	      * allows sharing at read.
;;;;
;;;;	Compression is performed by finding equivalent term structures and
;;;;	forcing sharing. Sharing is forced by modifying pointers in term
;;;;	structures. Modification of pointers should free duplicate structures
;;;;	allowing them to be collected.
;;;;	
;;;;	Compression will force sharing for the following term structures:
;;;;	  - symbol : opids, bindings
;;;;	  - parameter
;;;;	  - operator
;;;;	  - term 
;;;;
;;;;	Compression session : A compression session is an interval during which
;;;;	 a compression table and array is in effect. Compression sessions may be
;;;;	 nested. The compression level is depth of session nesting. Any open
;;;;	 session may be added to. A session is open unless all indices are used up
;;;;	 or it has been explicitly closed. Structures are compressed
;;;;	 when sharing is identifed. Heuristics may be used to identify sharing.
;;;;	 The default heuristic identifies sharing within any static sequence of
;;;;	 terms. Structures equivalent to structures within previous sequences of
;;;;	 terms are not identified. 
;;;;
;;;;	Normally, a compression session would be point to point with a server
;;;;	and client. Only the server may add structures to the compression
;;;;	tables.  Both the sever and client can compress and uncompress data. The
;;;;	members of the session may switch server and client roles by mutual
;;;;	agreement at any time.
;;;;
;;;;	The Level0 session is indefinite and is always closed. Any number of
;;;;	members are allowed in the session, and all will behave as clients.  The
;;;;	level0 table and array is initialized from a file created during a
;;;;	training session.  The training session differs from a normal session in
;;;;	that reference counts are kept for the duration of the session and then
;;;;	used to choose the structures to be compressed.
;;;;
;;;;	There is also a dynamic LevelN session whose extent is limited to a
;;;;	static sequence of terms. In the levelN session, only the server
;;;;	compresses data and only the client uncompresses.
;;;;
;;;;
;;;;	Normally compression will operate with only the Level0 and LevelN
;;;;	sessions.
;;;;
;;;;	Sessions may not be nested more than 4 deep and each level has limit of 4k
;;;;	indices.
;;;;
;;;;
;;;;
;;;;	Random Access : 
;;;;
;;;;	Low : Level index is less.
;;;;	High : Level index is more.
;;;;
;;;;	Static : batch, segregated by type, always closed.
;;;;	
;;;;	Dynamic : incremental, sequential, sometimes open. 
;;;;		  Once closed could be converted to static.
;;;;
;;;;	Persistent : Adding to table causes append to diskfile.
;;;;	  - file is list of terms similar to log file.
;;;;	
;;;;	Current implementation requires a reader to have read all prior terms from session
;;;;	server in order for term to be sensible. It is desirable that a reader 
;;;;	be able to read the session table directly to avoid having to read
;;;;	all prior terms. This is essential in a random access database to support
;;;;	session levels. 
;;;;	
;;;;	Thus :
;;;;	  Persistent write then is qualitatively different form session write.
;;;;	    - difference is hidden in level implementation.
;;;;	        * write structure to level log.
;;;;		* write index to term file.
;;;;	
;;;;	
;;;;	Shared
;;;;	  level0 : per database
;;;;	    - opids, parameters, ops, and term carriers only.
;;;;	  level1 : per log.
;;;;	    - oids/dependencies : !stamp/!depenency/!oid and oid parameters.
;;;;	    - twould be nice to allow more than 4k entries. 
;;;;	      1M should be sufficient. (when 3 byte indices).
;;;;	    - init at log open, keep those in start of new journal.
;;;;
;;;;	Link : links have static compression levels. 
;;;;	  - Multiple logs may share a link.
;;;;	  - must dynamically bind/unbind log levels for each link transmission.
;;;;	  - or different schema for links.
;;;;	
;;;;
;;;;	Journal : sequential access
;;;;	  leveln : per write.
;;;;
;;;;	Data : random access.
;;;;	  level2 : per oid.		- optional
;;;;	  leveln : per term.
;;;;	
;;;;	Minimal implementation level0+levelN.
;;;;	  - then add new level0 with carrier heuristic.
;;;;	  - add static level references to ascii syntax.
;;;;	     ie encode name of table into term, use 03 or 04 bit to indicate level id follows/ends
;;;;	     eg #x03 This is level id #x03
;;;;	     but also need db garbage collect to recognize/see.
;;;;	  - then add Level1 with oid heuristic.
;;;;	  - then add data Level2 with prf heuristic.
;;;;	
;;;;	Domain Knowledge :
;;;;	  - do not hash file names in persist-data terms.
;;;;	  - do not hash oids in level0. 
;;;;	  - maybe special case for oids, ie another syntactic class.
;;;;		but want increased quantity
;;;;
;;;;	Desire centralized control of access to dynamic level files to enable
;;;;	sharing and caching of tables and limit open files.
;;;;	  - level1 (log) always open.
;;;;	  - level2 (oid) breifly open. lru of 8(parameterize) level2 files.
;;;;
;;;;	DB-awareness of level log files.
;;;;	  - level0 is not a level log file.
;;;;	  - need -log-reopen-write for level2 (oid) logs.
;;;;	
;;;;
;;;;	Level2 (oid) heuristic?
;;;;	  - add till it fills then close ??
;;;;	  - generational, init next generation from most frequently used/largest of last.
;;;;	    ie keep reference counts and size measure(ascii byte count).
;;;;	      - want to limit rollover to avoid continual dup of most popular.
;;;;		need balance between adding new and duping old.
;;;;		maybe don't rollover when filled but when some percentage is no longer
;;;;		frequently referenced.
;;;;		maybe track more than written to allow inclusion in next generation.
;;;;	  - identify specific filetypes to include/avoid, ie avoid dependency-store etc.
;;;;	
;;;;	Can be arbitrarily complex. Halfsize rollover seems acceptable.
;;;;	
;;;;	Should be optional, may be groups of oids. Ie all oids in prf should use prf level2.
;;;;	Other kinds of objects probably do not need level2. grouping needs to be static.
;;;;	
;;;;	
;;;;	Visibility : for the most part compression should be invisible.
;;;;	  - encode in streams?
;;;;	
;;;;	
;;;;
;;;;	Compression and IO:
;;;;
;;;;
;;;;	Compressor : component doing compression. Server.
;;;;	Compressee : component doing uncompression. Client.
;;;;	
;;;;
;;;;	In theory, allowing expanding field length would be straightforward. In
;;;;	practice, it seems unlikely to be necessary and simplifies storage
;;;;	allocation strategies to place upper limit.
;;;;	
;;;;	
;;;;	Syntax is simplified ascii syntax with added compression syntax.
;;;;	
;;;;	Ascii pre-compression term syntax: 
;;;;
;;;;	<term>			: <operator>(<bound-term-list>)
;;;;				| <operator>
;;;;
;;;;	<operator>		: <string>{<parameter-list>}
;;;;				| <string>{}
;;;;				| <string>
;;;;
;;;;	<parameter-list>	: <parameter> | <parameter>, <parameter-list>
;;;;
;;;;	<parameter>		: <parameter-value>:<string>
;;;;
;;;;	<parameter-value>	: <string>
;;;;
;;;;	<bound-term-list>	: <bound-term> 
;;;;				| <bound-term>; <bound-term-list>
;;;;
;;;;	<bound-term>		: <binding-list>.<term> | .<term> | <term>
;;;;
;;;;	<binding-list>		: <variable-id> | <variable-id>, <binding-list>
;;;;
;;;;	<variable-id>		: <parameter-value>
;;;;
;;;;	<string>		: 7bit ascii.
;;;;
;;;;
;;;;	If no bound terms then operator will not be a simple string.
;;;;
;;;;	Escaping is done as in non-compressed syntax.
;;;;
;;;;	Add one-byte compression flags with syntactic descriptors.
;;;;	Add two-byte compression indices for strings, parameters, operators, terms and bound terms.
;;;;	When reading subsequent associated indices, flags can be used to disambiguate syntax.
;;;;
;;;;	80 bit on in a byte indicates compression data.
;;;;
;;;;	20 and 10 bit indicate level.
;;;;
;;;;	40 bit on indicates a one byte add field
;;;;
;;;;	In a one byte add the  01 and 02 bits used to indicate syntax type.
;;;;
;;;;	40 bit off indicates a two byte index field with remaining 14 bit
;;;;	 being the index into the level.
;;;;
;;;;	
;;;;	TODO : add control chars between terms, like start term end term. start level etc.
;;;;	  then can do away with bogus . problem and avoid encodeing level messages in term
;;;;	  syntax.
;;;;
;;;;	
;;;;	Every file/stream has level info encoded. Nothing is assumed.
;;;;	Levels can be referred to via <data-persist> terms.
;;;;	  - level0 too. In some sense larger than any particular process
;;;;	    but produced from some regression test in a process.
;;;;	    But if all garbage removed from process dir, no big deal to keep dir.
;;;;	
;;;;	when using dynamic read from file if index exceeds known then reopen file and
;;;;	read again. Ie, possible for data to have been appended since read?
;;;;    Need to be able to compute file pointer from current level data. Ie keep track
;;;;	of num bytes read from file.
;;;;   
;;;;   
;;;;	We extend the syntax to allow compression control characters to 
;;;;	be sent between terms. 
;;;;	
;;;;	TERM	: precedes and follows a Term.  (character-to-code #\t)
;;;;	LEVEL	: precedes and follows a Level. (character-to-code #\l)
;;;;	
;;;;	
;;;;	<level>		: <data-persists>
;;;;			| !level{index:n, size:n}
;;;;			   <level-symbols> <level-parameters> <level-operators> <level-terms>
;;;;	
;;;;	When asked to read a term, first all level data is read until a term is encountered.
;;;;	
;;;;	When writing a term, level data added to a static level is written differently than
;;;;	a dynamic level. The Static level structure is written to a static file, and the index
;;;;	is sent. Dynamic level data is sent in line following an add byte. 
;;;;	?? how is this parameterized. ??
;;;;	
;;;;	3 Byte: 14 bit indices will be inadequate. 
;;;;	  - all indices are 3 bytes, 22 bit address space.
;;;;	    4M of indices
;;;;	  - use a bit to indicate 2 or 3 bit. 13 bit and 21 bit address spaces.
;;;;	    8K 2 byte indices.
;;;;	    2M 3 byte indices.
;;;;	    
;;;;	Advantage to the mixed size is that the 8k indices will be more frequently used
;;;;	resulting in an average index size of close to 2bytes. 2M of indices should be sufficient
;;;;	for some time.
;;;;
;;;;	But do the indices use and noticeable amount of space or is the size of the ascii sequences
;;;;	dominated by the uncompressed data?
;;;;	
;;;;
;;;;	Ascii-compression term syntax: 
;;;;
;;;;	<term'>			: <term> | <add-term><term> | <index>
;;;;
;;;;	<term>			: <operator'>(<bound-term-list>)
;;;;				| <operator'>
;;;;
;;;;	<operator'>		: <operator>
;;;;				| <add-op><operator>
;;;;				| <index>
;;;;
;;;;	<operator>		: <string'>{<parameter-list>}
;;;;				| <string'>{}
;;;;				| <string'>
;;;;				
;;;;
;;;;	<parameter-list>	: <parameter'> | <parameter'>, <parameter-list>
;;;;
;;;;	<parameter'>		: <parameter> | <add-parm><parameter> | <index>
;;;;
;;;;	<parameter>		: <parameter-value>:<string>
;;;;
;;;;	<parameter-value>	: <string>
;;;;				| <hexword><byte><byte><byte><byte>
;;;;
;;;;	<bound-term-list>	: <bound-term'> 
;;;;				| <bound-term'>; <bound-term-list>
;;;;
;;;;	<bound-term'>		: <bound-term> | <add-bt><bound-term> | <index>
;;;;
;;;;	<bound-term>		: <binding-list>.<term'> | .<term'> | <term'>
;;;;
;;;;	<binding-list>		: <variable-id> | <variable-id>, <binding-list>
;;;;
;;;;	<variable-id>		: <parameter-value>
;;;;
;;;;	<string'>		: <string> | <add-string><string> | <index>
;;;;
;;;;	<string>		: 7bit ascii.
;;;;
;;;;	<index>			: <byte><byte>
;;;;
;;;;	
;;;;	<add-string>	: #b11LLU000
;;;;	<add-parm>	: #b11LLU001
;;;;	<add-op>	: #b11LLU010
;;;;	<add-term>	: #b11LLU011
;;;;	? <hexshort>	: #b11LLU101
;;;;	? <hexbyte>	: #b11LLU110
;;;;	? <hexuni>	: #b11LLU111
;;;;
;;;;	Where LL is level and UU is unused. 80 40 20 10 8 4 2 1
;;;;	
;;;;	Note parameter value and parameter type strings may not be indices.
;;;;	No whitespace allowed except when escaped.
;;;;	
;;;;	For the time being all indices are limited to two bytes. At some point
;;;;	we may need larger indices or some expanding opcode method of identifying
;;;;	larger indices.
;;;;
;;;;	When an add byte is seen a index count should be incremented and the
;;;;	initialized with the next structure parsed.  The add byte may be used to
;;;;	disambiguate otherwise ambiguous syntax, both when parsing structure
;;;;	following add byte and when parsing structure containing associated
;;;;	index.
;;;;
;;;;	hexwords are used for encoding integer parameter values for
;;;;	values of x : 9999 < x < #x1,0000,0000. 
;;;;	  - faster to read then parsing from string
;;;;	  - requires less space in stream.
;;;;	
;;;;	Could do similar for unicode : ie define a control code and then send as two hex bytes rather
;;;;	than \uuuu ascii as now. However, we do not expect much unicode so may not be worth the control index.
;;;;	
;;;;	
;;;;	Compression Algorithm : equivalence is lexico-graphic equivalence.
;;;;	 
;;;;	Each syntactic type (symbol, parameter, operator, and term) has a pair
;;;;	of hash tables:
;;;;	  - eq : uses builtin eq hash.
;;;;	      * used to curtail recursive descent and find compression indices.
;;;;	  - hash : uses home-grown hash function.
;;;;	      * used to identify members of equivalence class.
;;;;	
;;;;	Each equivalence class has a compression hash table entry structure(chte).
;;;;
;;;;	chte	: chte [flags(seen,add)
;;;;			structure
;;;;			hash
;;;;			index
;;;;			]
;;;;	
;;;;	      * implemented as eq-hash on hash numbers with a list of chte's
;;;;
;;;;	Compression :
;;;;	  - bottom up hash cons.
;;;;	      * if in eq table then return chte found.
;;;;	      * if not in eq table
;;;;		   Then recurse,
;;;;		     for each substructure, if structure of returned chte is not eq to substructure
;;;;		     then replace substructure with chte structure.
;;;;		   Next compute hash of structure (use hash field of substructures chtes).
;;;;		   Next find equivalent structure in hash table, equivalence is eq check on
;;;;		     substructures as recursion replaces substructures with eq value.
;;;;		     If equivalent structure found
;;;;			then return chte
;;;;			else add new chte to hash table and eq table and return chte.
;;;;
;;;;	Encoding in ascii string : If multiple terms then each step is done on
;;;;	every term before proceeding to next step.
;;;;	  - top down add identification.
;;;;	     If index assigned or add-p then return.
;;;;	     If seen-p then set add-p true.
;;;;	     Else set seen-p true and recurse.
;;;;	  - top down ascii conversion :
;;;;	     If index assigned accumulate index
;;;;	     If add-p then assign-index, accumulate add-byte, and accumulate structure.
;;;;	     Else accumulate structure.
;;;;	  - reset flags (avoids adding index for later term).
;;;;
;;;;	If multiple compression levels are in effect:
;;;;	  - during hash-pass lookup lower to higher, add to highest level.
;;;;	  - during ascii-encoding-add-pass lookup higher to lower, if chte found in closed level
;;;;	    and add-p turns on, then shadow in highest open level-clear flags of lower version.
;;;;	  - during ascii conversion : turn off flags!
;;;;
;;;;
;;;;	During duplex compression session, client must update compression tables as
;;;;	well as compression array.  Similarly, sever must update compression arrays
;;;;	as well as compression tables. Client compression table update differs from
;;;;	server table update in that index is known, flags are irrelevant, table is closed,
;;;;	and level is known. Server compression array update differs from client update
;;;;	in that index is known, and actual is known apriori. . Note that sever/client
;;;;	roles can be fluid. Only restriction is two can not be adding indices 
;;;;	simultaneously, ie there must not be more than one server at any time.
;;;;	If there is more than a single client in session, then some protocol must
;;;;	be developed so that clients will not miss array updates when ignoring
;;;;	session messages addressed to other clients. One solution is not to allow
;;;;	updates in point to point messages, but only in broadcast messages.
;;;;
;;;;	Domain knowledge heuristics :
;;;;
;;;;	  Compression could be parameterized by some knowledge of operator
;;;;	usage. For example, it may make sense to avoid attempting compression of terms with cons
;;;;	operators. Compress the operators and elements of the sexpr but not the sexpr itself.
;;;;	
;;;;	Training could produce reports of operators with good/bad compression behaviour and
;;;;	input that to higher levels via the level0 table.
;;;;	
;;;;	Data : compression level data should be managed via database. Ie, the garbage collector
;;;;	needs to do the right thing with compression data in the database.
;;;;	
;;;;	!data_persists : avoid at lower levels ??
;;;;	
;;;;	Structures are segregated by type.
;;;;	
;;;;
;;;;	<level>		: <level> <symbols> <parameters> <operators> <terms>
;;;;
;;;;	<level_desc>	: !level {<index>:n, <size>:n}
;;;;
;;;;	<symbols>	: !symbols{<s_parameters>}
;;;;
;;;;	<s_parameters>	: <s_parameter>, <s_parameters>
;;;;			| EPSILON
;;;;
;;;;	<s_parameter>	: <opid>:token
;;;;			| <binding>:variable
;;;;	
;;;;	<parameters>	: !parameters{<parameters>}
;;;;	
;;;;	<parameters>	: <parameter>, <parameters>
;;;;			: EPSILON
;;;;	
;;;;	<operators>	: !operators(<ops>)
;;;;
;;;;	<ops>		: <op>(); <ops>
;;;;			| EPSILON
;;;;	
;;;;	<terms>		: !terms(<terms'>)
;;;;	
;;;;	<terms'>	: <term>; <terms'>
;;;;			| EPSILON
;;;;
;;;;	<binding> may be variable, meta-variable, or slot.
;;;;

;;;;	Level0 training:
;;;;	
;;;;	A training session watches a session and builds a single level
;;;;	compression table with reference counts. At the end of the session, the
;;;;	table chte's are sorted by the reference counts from high to low. A cut
;;;;	is made such that the the set of chtes contains no more entries than
;;;;	allowed in a compression array. Then the chtes are segregated by
;;;;	syntactic type and the term set is order from lo to high wrt op counts.
;;;;	Then the structures are saved to a file in the syntax presented above.
;;;;	
;;;;	To initialize a level0 client, the file is read and a table is updated
;;;;	as for a session client.  and an array is updated as for a session
;;;;	server. The level is then closed.
;;;;
;;;;	
;;;;	Sessions could be used to compress files as well. A level0 type of file
;;;;	can be created to be shared amongst a group of files. Need header info
;;;;	in files, and streamlined leve0 init for uncompression only.
;;;;

;;;;	
;;;;	<compression-table>	: <hash-table{eq}> array <hash-table{hash}> array
;;;;	
;;;;	<compression-array>	: array
;;;;	
;;;;	<compression-level>	: <compression-table> <compression-array> 
;;;;	<compression-levels>	: <compression-level> array
;;;;	
;;;;
;;;;	compression-read-static-level (<compression-levels> <read-term>)
;;;;	  : <compression-levels>
;;;;	  * read-term-f ( <compression-levels> ) : <term>
;;;;	  * assumes called in scope of compression-levels
;;;;
;;;;	compression-write-static-level (<filespec>, <compression-level>)	: NULL
;;;;	

;;;;	
;;;;	compression-table-increment-index (<level> <chte> <code>)	: BOOL
;;;;	  * updates chte index, then increments table-index.
;;;;	  * calls compression-level-update-array
;;;;		
;;;;	compression-level-update-array (<level> <code> <*{actual}>)	: <int{index}>
;;;;	
;;;;	compression-level-update-table (<level> <int{level-index}> <*{actual}> ...
;;;;	 * updates chte index
;;;;	 * calls add-structure-to-compression-levels
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Compression Redux.
;;;;	
;;;;	  - expanding codes to allowing for larger indices and more control info.
;;;;	
;;;;	  - allow for multiple term files and easy indexing to arbitrary terms.
;;;;	      * each file has .index file which has offsets?
;;;;	      * each file has .gc file which indicates which indices have been gc'd.?
;;;;
;;;;	  - gc needs to thin files? while remaining readable? 
;;;;	      * maybe allow version nums on files, how to know when latest is closed during gc.??
;;;;	        each file has file plus ln.
;;;;		write new
;;;;		then rm ln
;;;;		then ln to newest. 
;;;;		then not old, move en masse at end of gc.
;;;;		reader follow ln if available else finds latest. if no link then it is new.
;;;;		maybe some coarse method like after 24 hrs feel free to rm old versions.
;;;;	
;;;;	  - when file is read cache contents as string buffers for some period.
;;;;	
;;;;	  - need to support old compression indefinitely.
;;;;	
;;;;	  - index info, ie what file pointer, oids, and words contained in terms in files.
;;;;	
;;;;	  - offline compression by mapping over database and add new compression files
;;;;	    for by global, dir basis.
;;;;	
;;;;	ONE FILE : actually log, data file, and map/index file.
;;;;	  * though file monolithic still add object data as multiple terms to allow
;;;;	    incremental finer grained updates.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	have some discernable immutable directory characteristics, such as what compression
;;;;	version used, to keep in a table to use in creating filenames.
;;;;	
;;;;	
;;;;	
;;;;	multidimensional groupings of object by time and kind or other immutable characteristics.
;;;;	file : object list 
;;;;	object : source dependencies etc.
;;;;




(defconstant *fixnum-length*
  (1- (integer-length most-positive-fixnum))
  ;;(declare (type fixnum *fixnum-length*))
  )

;; twould be a big win if used fixnums instead of 32 bits (ie bignums).
#|
(defun hash-rot (x b)
  (logand #xffffffff
	  (let ((b1 (logand #x1f b)))
	    ;; or top bits with shifted value, ie move the b1 high bits to low.
	    (logior (ash x b1)
		    ;; grab b1 bits at top
		    (ldb (byte b1 (- 32 b1)) x)))))
|#

;;(proclaim '(function hash-rot (fixnum fixnum) fixnum))
;;(proclaim '(function hash-member (fixnum fixnum fixnum) fixnum))


;; hi is on the right.
;; |yxxxxx|
;; |xxxxxy|  b num bits of y. length x is *fixnum-length* - b
;; need masks for hi y and hi and lo x
(defvar *hi-source-byte-specifiers*
  (let ((a (make-array 16)))
    (dotimes (i 16)
      (setf (aref a i) (byte i (- *fixnum-length* i))))
    a))

(defvar *lo-source-byte-specifiers*
  (let ((a (make-array 16)))
    (dotimes (i 16)
      (setf (aref a i) (byte (- *fixnum-length* i) 0)))
    a))

(defvar *hi-target-byte-specifiers*
  (let ((a (make-array 16)))
    (dotimes (i 16)
      (setf (aref a i)  (byte (- *fixnum-length* i) i)))
    a))

(defmacro hash-rot (x b)

  (let ((xx (gensym)))
    `(let ((,xx ,x))
      (declare (type fixnum ,xx))
      (let ((b1 (min ,b 15)))
	(declare (type fixnum b1))

	;; or top bits with shifted value, ie move the b1 high bits to low.
    
	(dpb
	 (ldb (aref *lo-source-byte-specifiers* b1) ,xx) ; orginal lo bits
	 (aref *hi-target-byte-specifiers* b1) ; hi bits targets
	 (ldb (aref *hi-source-byte-specifiers* b1) ,xx) ;original hi bits now lo bits.
	 )))))

(defun hash-list (l)
  (proclaim `(ftype (function (list) fixnum) hash-list))

  (do* ((rot-amt 1 (mod (+ rot-amt 3) 15))
	(restp l (cdr restp))
	(value 0))
       ((null restp) (hash-rot value 5))
    (setf value 
	  (logxor value (hash-rot (funcall #'hash-list (car restp)) rot-amt)))))

(defmacro hash-member (hash-acc hash i)
  `(logxor ,hash-acc (hash-rot ,hash (mod (1+ (* 3 ,i)) 15))))
  

;;;;	Summary of compressing term to ascii stream :
;;;;	
;;;;	  - compress : walk term and add structures to hash table
;;;;	      * find sharing. 
;;;;	  - index : walk term and add indices to chte.
;;;;	      * this pass can be skipped if there is not a dynamic table.
;;;;	  - stream : walk term and send codes.
;;;;	     
;;;;	Stream pass seems to predominate cost. However compress and index
;;;;	are not insignificant.   
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	Possible problem areas :
;;;;	  - Multiple hash lookups when multiple levels.
;;;;	  - per char io. Ability to do byte-array io should be added.
;;;;	    even if lisp primitive is per byte, can drive lisp primitive from array 
;;;;	    call rather than drive a byte at a tiem.
;;;;	  




;; LAL TODO:
;; increased size from 2 exp10 as possible fix to compression-array out-of-range error
;; needs further attention
(defparameter *compression-level-size* (expt 2 12))
(defparameter *compression-level-max* (expt 2 2))

(defun compression-code-p (code)
  ;;(= #x80 (logand code #x80))
  (>= code #x80))

(defun compression-add-byte-p (code)
  ;;(= #xC0 (logand code #xC0))
  (>= code #xC0))


(defparameter *lo-byte* (byte 8 0))
(defparameter *hi-byte* (byte 8 8))

(defun compression-level-of-byte (byte)
  (ash (logand byte #x30) -4))

(defun compression-index-of-bytes (first-byte second-byte)
  (+ (ash (logand first-byte #x0F) 8)
     second-byte))

(defun add-compression-level-to-byte (byte level)
  (logior byte (ash level 4)))

(defun structure-type-of-add-byte (byte)
  (logand byte #x07))

(defun compression-index-to-bytes (level index acc-f)
  (funcall acc-f (+ (logior #x80 (ash level 4)) (ash index -8)))
  (funcall acc-f (ldb *second-byte-address* index)))

(defmacro fbb-compression-accumulate-index-bytes (index)
  (let ((ii (gensym)))
    `(let ((,ii ,index))
      (fbb-add-byte (ldb *first-byte-address* ,ii))
      (fbb-add-byte (ldb *second-byte-address* ,ii)))))

(defun make-add-byte (type level)
  (logior #xC0 type (ash level 4)))

(defvar *add-bytes*
  (let ((a (make-array (list *compression-level-max*
			     *compression-num-types*)
		       :element-type '(unsigned-byte 8))))
    (dotimes (i *compression-level-max*)
      (dotimes (j *compression-num-types*)
	(setf (aref a i j) (make-add-byte j i))))
    a))


(defmacro add-byte (type level)
  `(aref *add-bytes* ,level ,type))



;;;;	
;;;;	Term compression hash :
;;;;
;;;;	Only updates last table.
;;;;	
;;;;	Each syntactic term structure (symbol (opid and binding), parameter, operator, and term)
;;;;	has two hash tables :
;;;;	  - eq : allows early termination if found.
;;;;	  - structure : finds sharing.
;;;;
;;;;	Destructively modifies terms structures to share when equivalent structures found.
;;;; 	  - opid	: always eq
;;;;	  - binding	: modifies binding list
;;;;	  - parameter	: modifies parameter list of op.
;;;;	  - operator	: modifies sexpr of term
;;;;	  - term	: modifies cons of bound-term
;;;;

;;;;	
;;;;	Need non add version. don't call add-indices!
;;;;	
;;;;	
;;;;	

(defmacro compression-type-to-index (type)
  (let ((tt (gensym)))
  `(let ((,tt ,type))
    (if (zerop ,tt)
	0
	(1- ,tt)))))


(defvar *compression-training-p* nil)


(defstruct (chte (:copier wimpy-copy-chte)) ;; compression hash table entry
  (flags nil)
  actual
  (hash 0 :type fixnum)
  index
  )

(defun copy-chte (chte)
  (let ((schte (init-chte-flags
		(wimpy-copy-chte chte))))
    (setf (chte-index schte) nil)
    schte))

(defstruct (count-chte (:include chte)) ;; compression hash table entry
  count)


(define-flags (chte)
    ((add nil t)
     (seen nil t)
     (ephemeral nil t)
     (resident nil t)  ; resident in closed table but without in index. clean resets index.
     ;; when assigned index it should be assigned in open level.
     ))

(defun actual-of-chte (c) (chte-actual c))
(defmacro hash-of-chte (c) `(chte-hash ,c))
(defmacro index-of-chte (c) `(chte-index ,c))
(defun count-of-chte (c) (count-chte-count c))

(defun new-chte (a h l)
  ;;(break "nc")
  (if *compression-training-p*
      (make-count-chte :actual a :hash h :count 0)
      (init-chte-flags
       (make-chte :actual a :hash h :index (- 0 1 l)))))



;;;
;;;	table
;;;

(defvar *compression-table-default-size* 1024)

(defstruct compression-table
  (open t)
  (level nil)
  (index nil)				; inclusive, ie start at initial value.
  (max nil)				; exclusive, ie bad if equal.
  (eq nil) ;; array of eq tables
  (hash nil) ;; array of hash tables, 
  )

(defconstant *num-kinds-of-compression-table* 4)

;; could track if rehash happens and learn best values for each table.
(defvar *opcount-table-size-divisor* 4.0)

;; start tables out small but grow quick. could actually init size by looking at num ops of term.
;; maybe opcount/4?
;; acturall if we have a high threshold before trying compression then just boost the initial size is good.
(defun new-compression-table (level size opcount)
  (let ((size (if opcount
		  (floor opcount *opcount-table-size-divisor*)
		  (or size *compression-table-default-size*))))
    ;;(format t "size ~a ~%" size)
    (make-compression-table :level level
			    :index 0
			    :max *compression-level-size*
			    :eq (make-array *num-kinds-of-compression-table*
					    :initial-contents
					    (list (make-hash-table :size size
								   :rehash-size size)
						  (make-hash-table :size size
								   :rehash-size size)
						  (make-hash-table :size size
								   :rehash-size size)
						  (make-hash-table :size size
								   :rehash-size size)))
			    :hash (make-array *num-kinds-of-compression-table*
					      :initial-contents
					      (list (make-hash-table :size size
								     :rehash-size size)
						    (make-hash-table :size size
								     :rehash-size size)
						    (make-hash-table :size size
								     :rehash-size size)
						    (make-hash-table :size size
								     :rehash-size size))))))

(defun new-ephemeral-compression-table (level table)
  (make-compression-table :level level
			    :index 0
			    :max *compression-level-size*
			    :eq (compression-table-eq table)
			    :hash (compression-table-hash table)))





(defmacro chte-index-set-p (chte)
  `(> (index-of-chte ,chte) 0))

(defun table-clean-close (table)
  (labels
      ((clean-f ()
	 #'(lambda (k v)
	     (declare (ignore k))
	     ;;(setf -k k -v v)
	       
	     (when (and v (not (chte-index-set-p v)))
	       (chte-flag-set-resident v t))) ))

    (let ((eqhash (eq-of-compression-table table)))
      (dotimes (i (length eqhash))
	(let ((h (aref eqhash i)))
	  (maphash (clean-f) h))))
    ))


(defun table-clean-ephemerals (table)
  ;;(setf -table table) (break "tce")
  (let ((cleaned 0))
    (labels
	((clean-f (h)
	   #'(lambda (k v)
	       ;;(setf -k k -v v)
	       ;;(setf -k k -h h -v v) (break "hhh")
    
	       (cond

		 ((null v) (remhash k h))

		 ((consp v)
		  (let ((l v))
		    (do ()
			((null (cdr l)))
		      (cond 
			((chte-flag-ephemeral-p (cadr l))
			 (incf cleaned)
			 (setf (cdr l) (cddr l)))
			(t (setf l (cdr l)))))

		    (when (chte-flag-ephemeral-p (car v))
		      (incf cleaned)
		      ;;(setf -k k -h h -v v -l l) (break "hhh")
		      (if (cdr l)
			  (setf (gethash k h) (cdr l))
			  (remhash k h)))))
		
		 ((chte-flag-ephemeral-p v)
		  (incf cleaned)
		  (remhash k h)))))
		 )
    
      (let ((eqhash (eq-of-compression-table table)))
	(dotimes (i (length eqhash))
	  (let ((h (aref eqhash i)))
	    (maphash #'(lambda (k chte)
			 ;;(setf -k k -v v)
			 (cond
			   ((null chte) (remhash k h))

			   ((chte-flag-resident-p chte)
			    (setf (chte-index chte) -1)
			    (chte-flag-set-seen chte nil)
			    (chte-flag-set-add chte nil)
			    )
			   
			   ((chte-flag-ephemeral-p chte)
			    (incf cleaned)
			    (remhash k h))))

		     h))))

      (let ((hash (hash-of-compression-table table)))
	(dotimes (i (length hash))
	  (let ((h (aref hash i)))
	    (maphash (clean-f h) h))))

      ;;(when (> cleaned 20000) (break))
      (when *io-echo-p*
	(format-string "cleaned ~a~%" cleaned))
      )))


(defun compression-table-open-p (ct) (compression-table-open ct))
(defun level-of-compression-table (ct) (compression-table-level ct))
(defun index-of-compression-table (ct) (compression-table-index ct))
(defun max-of-compression-table (ct) (compression-table-max ct))

(defun eq-of-compression-table (ct) (compression-table-eq ct))
(defun hash-of-compression-table (ct) (compression-table-hash ct))

(defmacro eq-type-of-compression-table (ct type)
  `(aref (compression-table-eq ,ct) (compression-type-to-index ,type)))

(defmacro eq-type-of-compression-eq (ct type)
  `(aref ,ct (compression-type-to-index ,type)))

(defmacro hash-type-of-compression-table (ct type)
  `(aref (compression-table-hash ,ct) (compression-type-to-index ,type)))

(defmacro hash-type-of-compression-hash (ct type)
  `(aref ,ct (compression-type-to-index ,type)))

(defun compression-table-stats (ct)
  (labels
      ((clash-stats (s table)
	 ;; key count is number of keys with clashes
	 ;; max is max number of clashes
	 ;; total is sum of count of clashes for all keys.
	 (let ((max 0)
	       (count 0)
	       (total 0)
	       (total-key-count 0))
	   (maphash #'(lambda (k v)
			(declare (ignore k))
			(incf total-key-count)
			(let ((l (1- (length v))))
			  (setf max (max max l))
			  ;;(when (= l max) (setf cc v))
			  (when (> l 0)
			    (incf count)
			    (incf total l))))
		    table)
	   (format t "~a hash clash stats : max ~a, total ~a, key count ~a.~%"
		   s max total count))))

    (format t "Compression table index: ~a~%" (compression-table-index ct))
    (terpri)
    
    (format t "Binding hash count   ~a~%" (hash-table-count
					   (hash-type-of-compression-table ct *compression-binding-type*)))
    (format t "Parameter hash count ~a~%" (hash-table-count
					   (hash-type-of-compression-table ct *compression-parameter-type*)))
    (format t "Operator hash count  ~a~%" (hash-table-count
					   (hash-type-of-compression-table ct *compression-operator-type*)))
    (format t "Term hash count      ~a~%" (hash-table-count
					   (hash-type-of-compression-table ct *compression-term-type*)))
    (terpri)

    (format t "Binding eq count   ~a~%" (hash-table-count
					 (eq-type-of-compression-table ct *compression-symbol-type*)))
    (format t "Parameter eq count ~a~%" (hash-table-count
					 (eq-type-of-compression-table ct *compression-parameter-type*)))
    (format t "Operator eq count  ~a~%" (hash-table-count
					 (eq-type-of-compression-table ct *compression-operator-type*)))
    (format t "Term eq count      ~a~%" (hash-table-count
					 (eq-type-of-compression-table ct *compression-term-type*)))
    (terpri)

    (clash-stats "Binding" (hash-type-of-compression-table ct *compression-binding-type*))
    (clash-stats "Parameter" (hash-type-of-compression-table ct *compression-parameter-type*))
    (clash-stats "Operator" (hash-type-of-compression-table ct *compression-operator-type*))
    (clash-stats "Term" (hash-type-of-compression-table ct *compression-term-type*))
    (terpri)

    nil))

;;;
;;;	array
;;;

;; could store chte in actual until set,
;; and then set actual in chte (if present) and cate.
(defstruct cate ;; compression hash table entry
  byte
  actual
  )

(defun actual-of-cate (c) (cate-actual c))
(defun byte-of-cate (c) (cate-byte c))

(defun structure-type-of-cate (c)
  (structure-type-of-add-byte (cate-byte c)))


(defstruct (symbol-cate (:include cate)))
(defstruct (parameter-cate (:include cate)))
(defstruct (operator-cate (:include cate)))

(defstruct (term-cate (:include cate))
  (null-bindings-bound-term nil))

(defun null-bindings-bound-term-of-term-cate (c)
  (or (term-cate-null-bindings-bound-term c)
      (setf (term-cate-null-bindings-bound-term c)
	    (instantiate-bound-term (actual-of-cate c)))))

(defun new-cate (byte)
  (let ((type (structure-type-of-add-byte byte)))
    (cond
      ((eql type *compression-opid-type*) (make-symbol-cate :byte byte ))
      ((eql type *compression-binding-type*) (make-symbol-cate :byte byte ))
      ((eql type *compression-parameter-type*) (make-parameter-cate :byte byte))
      ((eql type *compression-operator-type*) (make-operator-cate :byte byte))
      ((eql type *compression-term-type*) (make-term-cate :byte byte))
      (t (raise-error (error-message '(scan compress unknown cate) byte))))))


(defun cate-set-actual (cate a)
  (setf (cate-actual cate) a)
  cate)


;; maybe growable array for levelN to avoid allocating entire array
;; for small terms.
;; may not need structure but keep fttb. Easier to remove later than add back later.
(defstruct (compression-array (:print-function (lambda (a stream &rest r)
						 (declare (ignore a r))
						 (format stream "CompressionArray"))))
	     array )

;; lazily allocate array in case not used. First use will be assign to index 0.

;;(defun initial-size-of-compression-array (a) (compression-array-size a))
;;(defun index-of-compression-array (a) (compression-array-index a))
(defun array-of-compression-array (a) (compression-array-array a))

(defun new-compression-array (&optional size)
  ;;(when (or t (< size 2048)) (break "ca"))
  (make-compression-array :array (when size
				   (make-array size
					       :adjustable t
					       :fill-pointer 0)
				   )))

(defun compression-array-assign (a cate)

  (let ((vector (array-of-compression-array a)))
    ;; lazily allocate array 
    (when (null vector)
      ;;(setf -a a -cate cate)(break "caa")
      (setf vector
	    (setf (compression-array-array a)
		  (make-array *compression-table-default-size*
			      :adjustable t
			      :fill-pointer 0))))

    (vector-push-extend cate vector *compression-table-default-size*)
    nil))


(defun compression-array-fill-pointer (a)
  (fill-pointer (array-of-compression-array a)))


(defun compression-array-stats (ca)
  (let ((sym 0)
	(parm 0)
	(op 0)
	(term 0))
    (dotimes (i (compression-array-fill-pointer ca))
      (let ((cate (aref (array-of-compression-array ca) i)))
	(let ((type (structure-type-of-cate cate)))
	  (cond
	    ((eql type *compression-opid-type*) (incf sym))
	    ((eql type *compression-binding-type*) (incf sym))
	    ((eql type *compression-parameter-type*) (incf parm))
	    ((eql type *compression-operator-type*) (incf op))
	    ((eql type *compression-term-type*) (incf term))
	    (t (raise-error (error-message '(stats compress unknown cate) )))
	    ))))

  (format t "Shared Symbol count    ~a~%" sym)
  (format t "Shared Parameter count ~a~%" parm)
  (format t "Shared Operator count  ~a~%" op)
  (format t "Shared Term count      ~a~%" term)

  (list sym parm op term (+  sym parm op term))))


;;;
;;;	level
;;;

(defstruct compression-level
  direction
  table
  array

  persist-term
  short-term
  
  ;; persistent tables
  heuristics	;; some method of deciding when to add to table.
  file		;; may be stream or filename. may want to close level2 after io completes.
  )

(defmacro table-of-compression-level (l) `(compression-level-table ,l))
(defmacro array-of-compression-level (l) `(compression-level-array ,l))
(defun persist-term-of-compression-level (l) (compression-level-persist-term l))
(defun short-term-of-compression-level (l) (compression-level-short-term l))

(defun set-persist-term-of-compression-level (l p)
  (setf (compression-level-persist-term l) p))

(defun set-short-term-of-compression-level (l term)
  (setf (compression-level-short-term l) term))


(defun compression-level-in-p (level)
  (or (eql (compression-level-direction level) 'in)
      (eql (compression-level-direction level) 'in-out)))

(defmacro compression-level-out-p (level)
  (let ((ll (gensym)))
    `(let ((,ll (compression-level-direction ,level)))
      (or (eql ,ll 'out)
	  (eql ,ll 'in-out)))))




(defun new-compression-level (level direction &optional z za opcount)
  (let ((size (or z *compression-level-size*)))
    ;;(unless (eql 'in direction) (break (format-string "ncl ~a" direction)))
    (when (< size 2048) (break "cl"))
    (make-compression-level :direction direction
			    :table (when (or (eql direction 'out) (eql direction 'in-out))
				     (new-compression-table level size opcount))
			    :array (when (or (eql direction 'in) (eql direction 'in-out))
				     (new-compression-array za)))))



;; returns index.
(defun compression-level-update-array (level code actual)
  (let ((cate (new-cate code)))
    (cate-set-actual cate actual)
    (compression-array-assign (array-of-compression-level level) cate)))



(defstruct compression-levels
  count
  levels
  )

(defmacro count-of-compression-levels (cts) `(compression-levels-count ,cts))
(defmacro levels-of-compression-levels (cts) `(compression-levels-levels ,cts))


(defmacro level-of-compression-levels (levels level-index)
  `(aref (levels-of-compression-levels ,levels) ,level-index))

(defmacro table-of-compression-levels (levels level)
  `(table-of-compression-level (aref (levels-of-compression-levels ,levels) ,level)))

(defmacro array-of-compression-levels (levels level)
  `(array-of-compression-level (aref (levels-of-compression-levels ,levels) ,level)))

(defun persist-term-of-compression-levels (levels &optional index)
  (or (persist-term-of-compression-level
       (level-of-compression-levels levels (or index
					       (1- (count-of-compression-levels levels)))))
      (raise-error (error-message '(compression levels persist not)))))

(defun short-term-of-compression-levels (levels &optional index)
  (or (short-term-of-compression-level
       (level-of-compression-levels levels (or index
					       (1- (count-of-compression-levels levels)))))
      (persist-term-of-compression-levels levels index)))


;;;;	It is safe to assume that only one compression level will be open at
;;;;	any time, we might also assume that closed levels are shared sequentially
;;;;	but not simultaneoulsy. Then we can assume that we may avoid adding
;;;;	compression tables for an ephemeral level by using the latest closed
;;;;	levels tables. Then when done removing the ephemeral entries.
;;;;	
;;;;	If an open level is closed rather than discarded. then the table
;;;;	should be copied, and the original should be restored. Copy 
;;;;	could happen at open and then avoid having to restore original.
;;;;	
;;;;	
;;;;	could can assume latest level has only hash table, then ephemeral
;;;;	level inherits latest.
;;;;	
;;;;	eq table avoids term traversal, however could opportunistically
;;;;	 look for cached hash num of term and check hash hash-table first and
;;;;	 maybe avoid eq check, or cache chte on term and avoid either hash lookup.
;;;;	 tradeoff is that chte is more ephemeral than hash-num. hash-num is constant
;;;;	 for any term.


(defun new-out-compression-levels (cur-levels &optional opcount)
  (new-compression-levels cur-levels 'out nil nil opcount))


(defun construct-compression-levels (cur-levels level-index level)
  (make-compression-levels
   :count (1+ level-index)
   :levels (let ((a (make-array (1+ level-index))))
	     (dotimes (i level-index)
	       (setf (aref a i) (level-of-compression-levels cur-levels i)))
	     (setf (aref a level-index)  level)
	     a)))

(defun new-compression-levels (cur-levels dir &optional size sizea opcount)
  (let ((count (or (and cur-levels
			(count-of-compression-levels cur-levels))
		   0)))
    
    (when (>= count *compression-level-max*)
      (raise-error (error-message '(compression level max)
				  count *compression-level-max*)))
    
    (if (null cur-levels)
	(construct-compression-levels nil 0 (new-compression-level 0 dir size sizea opcount))
	(construct-compression-levels cur-levels
				      count
				      (new-compression-level count dir size sizea opcount)))))


(defvar *ephemeral-compression-tables-count* nil)
(defmacro ephemeral-compression-p ()
  `(when *ephemeral-compression-tables-count*
    (incf *ephemeral-compression-tables-count*)
    t))

(defun new-ephemeral-compression-level (levels level)
  (make-compression-level :direction 'out
			  :table (new-ephemeral-compression-table
				  level
				  (table-of-compression-level (level-of-compression-levels levels (1- level))))
			  :array nil))

(defvar *compression-level*)
(defvar *compression-table*)
(defvar *compression-hash-table*)
(defvar *compression-eq-table*)
(defvar *compression-level-index*)

(defmacro with-levels-out ((levels) &body body)
  
  (let ((lll (gensym)))
    `(let* ((,lll ,levels)
	    (*compression-level-index* (1- (count-of-compression-levels ,lll)))
	    (*compression-level* (level-of-compression-levels ,lll *compression-level-index*))
	    (*compression-table* (table-of-compression-level *compression-level*))
	    (*compression-eq-table* (compression-table-eq *compression-table*))
	    (*compression-hash-table* (compression-table-hash *compression-table*)))
      
      (unless (compression-level-out-p *compression-level*)
	(system-error '(compression-tables-find-add)))	

      ,@body)))

(defmacro compression-level ()		`*compression-level*)
(defmacro compression-table ()		`*compression-table*)
(defmacro compression-hash-table ()	`*compression-hash-table*)
(defmacro compression-eq-table ()	`*compression-eq-table*)
(defmacro compression-level-index ()	`*compression-level-index*)



;; direction : in(arrays), out(tables), or in-out.
;; level0 : in
;; leveli : in-out
;; leveln : out(server) or in(client)

;;;; compression-table-set-add-flags (on) and compression-table-encode (off) do all chte flag bit twiddling.
;;;; It is expected that add-indices-to-compression-tables (on) will be followed by
;;;; walk-term-ascii-compress(off), and end result will be all bits off.

;; want flag bits off when done, so reset flags no longer relevant.
;; returns t if index or add.
(defun compression-table-set-add-flags (structure type)

  (let* ((chte (gethash structure (eq-type-of-compression-eq (compression-eq-table) type))))

    (when chte
      ;;(let ((a (actual-of-chte chte)))
      ;;(when (and (parameter-p a)
      ;;		   (eql 37 (parameter-value a)))
      ;; (setf -chte chte) (break "ctsaf")))
      (return-from compression-table-set-add-flags
	(if (count-chte-p chte)
	    (progn (incf (count-chte-count chte))
		   (> (count-chte-count chte) 1))
	    (if (or (chte-index-set-p chte) (chte-flag-add-p chte))
		t
		(if (not (chte-flag-seen-p chte))
		    (progn
		      (chte-flag-set-seen chte t)
		      nil)
		    ;; set add
		    (progn
		      ;;(setf -chte chte -table table) (break "ctsaf")
		      (chte-flag-set-seen chte nil)
		      (if (compression-table-open-p (compression-table))
			  (progn
			    (chte-flag-set-add chte t)
			    t)
			  ;; shadow chte in last table if open.
			  ;; should be moot.
			  (progn
			    (break "moo t")
			    )))))))))
    nil)


(defmacro make-index-bytes (level index)
  (let ((ii (gensym)))
    `(let ((,ii ,index))
      (declare (type fixnum ,ii))

      (dpb (+ (logior #x80 (ash ,level 4))
	      (ldb *first-byte-address* ,ii))
       *first-byte-address*
       ,ii))))


;; nil (do not add) or level (add)
(defvar *funky-chtes* nil)

(defun compression-table-increment-index (chte code &optional level-index)
  (let ((index (index-of-compression-table (compression-table))))

    (if (>= index (max-of-compression-table (compression-table)))
	(setf (compression-table-open (compression-table)) nil)

	(progn
	  ;;(setf -chte chte) (break "ctii")
	  (if (chte-index-set-p chte)
	      ;; question is why did training session add duplicates??
	      (progn
		;;(setf -chte chte) (break "ctii")
		(when (null *funky-chtes*)
		  (format t "See funky-chtes in trm-asc.~%"))
		(push (list index (chte-index chte) (ldb  (byte 14 0) (or level-index (chte-index chte))))
		      *funky-chtes*))
	      (setf (chte-index chte)
		    ;; make bytes.
		    (make-index-bytes (or level-index (- 0 (chte-index chte) 1)) index)))

	  (incf (compression-table-index (compression-table)))

	  ;; if direction in-out add to array as well.
	  (when (compression-level-in-p (compression-level))
	    ;; could check if update array returns index??
	    (compression-level-update-array (compression-level) code (actual-of-chte chte)))
	  t))))


(defun compression-table-encode (structure type)
  ;;(let ((level-count (count-of-compression-levels levels))))
  ;;(dotimes (i level-count))
  ;;(let ((level (level-of-compression-levels levels (1- level-count))))) ;;(- (1- level-count) i)

  ;;(setf -s structure -b type) ;; (break "cte")
  ;;(when (and (term-p -s)
  ;;(eql (id-of-term -s) '|!inf_tree_cons|))
  ;;(break "cteb"))

  (let* ((chte (gethash structure (eq-type-of-compression-eq (compression-eq-table) type))))

    ;;(when (and (term-p -s) (eql (id-of-term -s) '|!inf_tree_cons|))
    ;;  (setf a chte) (break "cte") )

    (when chte
      ;;(setf a chte) (break "cte")
  
      (return-from compression-table-encode

	(let ((index (index-of-chte chte)))
	  (declare (type fixnum index))
	  ;;(setf -aaa  (level-of-compression-table table) -bbb type)

	  (if (chte-index-set-p chte)
	      ;; assume fbb
	      (progn
		;;(format t "index ~x ~o ~a~%" index index index)
		;;(when (= index #x80e3) (setf -chte chte -table table) (break "e3"))
		(fbb-compression-accumulate-index-bytes index)
		'index
		;;(funcall index-f (level-of-compression-table table) index)
		)
	      (if (chte-flag-add-p chte)
		  (let ((code (add-byte type (compression-level-index))))
		    ;;(format t "~x ~o ~a ~%" code code -iii) (incf -iii)
		    ;;(setf -chte chte -level level -levels levels -code code) (break "cte")
		    (chte-flag-set-add chte nil)
		    (when (compression-table-increment-index chte code (compression-level-index))
		      (fbb-add-byte code)
		      ;;(funcall add-f code)
		      'add))
		  (when (chte-flag-seen-p chte)
		    (chte-flag-set-seen chte nil)))))))))



;; lookup structure in hash table.
(defun compression-table-encode-check (structure type)
  (let ((chte (gethash structure (eq-type-of-compression-eq (compression-eq-table) type))))

    ;;(setf a chte) (break "cte")
	
    (when chte

      (return-from compression-table-encode-check

	(let ((index (index-of-chte chte)))
	  (declare (type fixnum index))
	  
	  ;;(setf -chte chte -index index) ;;(break "ctec")

	  (if (chte-index-set-p chte)
	      ;; assume fbb
	      (progn
		;;(fbb-compression-accumulate-index-bytes index)
		(cons 'index index)
		;;(funcall index-f (level-of-compression-table table) index)
		)
	      (if (chte-flag-add-p chte)
		  (let ((code (add-byte type (compression-level-index))))
		    (chte-flag-set-add chte nil)
		    (when (compression-table-increment-index chte code (compression-level-index))
		      ;;(fbb-add-byte code)
		      ;;(funcall add-f code)
		      (cons 'add code)))
		  (when (chte-flag-seen-p chte)
		    (chte-flag-set-seen chte nil)))))))))


(defun compression-level-sanity-check (levels level-index)
  (let* ((level (level-of-compression-levels levels level-index))
	 (array (array-of-compression-array (array-of-compression-level level)))
	 (error-p nil)
         (insane 0))
    
    (dotimes (i (fill-pointer array))
      (let ((cate (aref array i)))
	(let ((action (compression-table-encode-check (actual-of-cate cate)
						      (structure-type-of-add-byte (byte-of-cate cate)))))
	  (if (eql 'index (car action))
	      (unless (eql (ldb (byte 14 0) (cdr action)) i)
		;;(setf -cet (compression-eq-table))
		;;(setf -a action -i i -cate cate) (break "clsc")
		(setf error-p t)
                (incf insane)
		)
	      (if (eql 'add (car action))
		  (progn
		    (setf error-p t)
		    (format t "Compression add insane ~a ~a.~%" i (cdr action)))
		  (when action (break)))))))

    (when error-p
      (format t "Compression index insane ~a ~a.~%" insane level-index)
      (raise-error (error-message '(compression sanity not))))))


(defun compression-tables-level-add (levels level-index chte type hash-p)
  (let* ((table (table-of-compression-levels levels level-index))
	 (hash-table (when hash-p (hash-type-of-compression-table table type)))
	 (eq-table (eq-type-of-compression-table table type))
	 (schte (copy-chte chte)))
	      
    (break)
    (when hash-p
      (setf (gethash (hash-of-chte schte) hash-table)
	    (cons schte (gethash (hash-of-chte schte) hash-table))))
    (setf (gethash (actual-of-chte chte) eq-table) schte)
    schte))



;; return chte.
(defun compression-tables-find-add (structure type hash-f 
					   &optional (hash-p t) (eq-f #'eql))

  ;; look in eq table.
  (let* ((chte (gethash structure (eq-type-of-compression-eq (compression-eq-table) type))))
	
    (when chte
      ;;(setf -chte chte) (break "hello")
      (return-from compression-tables-find-add chte)))

  ;; look in hash table.
  (let ((hash (funcall hash-f structure)))
    (when hash-p

      ;;(dotimes (i level-count))
      (let* ((chte (find-first-optimized (chte (gethash hash (hash-type-of-compression-hash (compression-hash-table) type)))
					 (when (funcall eq-f structure (actual-of-chte chte))
					   chte))))

	(when chte
	  (return-from compression-tables-find-add chte))))
    

    ;; not found, add to eq and hash tables.
    (let* ((hash-table (when hash-p (hash-type-of-compression-hash (compression-hash-table) type)))
	   (eq-table (eq-type-of-compression-eq (compression-eq-table) type))
	   (chte (new-chte structure hash (compression-level-index))))

      (when (ephemeral-compression-p) (chte-flag-set-ephemeral chte t))
      
      (when hash-p
	(let ((lhashed (gethash hash hash-table)));; should be able to combine with lookup above.
	  (setf (gethash hash hash-table) (if lhashed
					      (if (consp lhashed)
						  (cons chte lhashed)
						  (list chte lhashed))
					      chte))))
      ;;(unless hash-p (break "huh"))
      (setf (gethash structure eq-table) chte)
      chte)))



    

;; compare potential with actual :
;;;;	
;;;;	Hashing terms replac
;;;;	
;;;;	  add-to hash table.  returns chte may be older chte
;;;;	  destructively modifies structure to use structure of chte.
;;;;	  then when encoding hash on structure will find chte.
;;;;	
;;;;	Structures with no depth do usual lex-eq.
;;;;	Structures with depth do lex-eq with only eq check on sub-structures.

;; compute hash numbers, use to find eq entry. Update arg or update table.
;; might make sense to not add opids or bindings which are <= 2 in length.

;; if not in tables add to last. A Table must be pair of hash and eq. 
;; rle not certain why passing thought hash locals should lose add-level index
;;  so kludged to make it work.
(defun add-structure-to-compression-levels (structure type)
  (labels
      ((hash-operator (o)
	 (let ((ohash (hash-of-chte (visit-symbol (id-of-operator o)))))
	   (declare (type fixnum ohash))
	   (do ((parameters (parameters-of-operator o) (cdr parameters))
		(i 0 (1+ i)))
	       ((null parameters))
	     (let* ((p (car parameters))
		    (chte (visit-parameter p)))
	       (unless (or (null chte) (eql p (actual-of-chte chte)))
		 (setf (car parameters) (actual-of-chte chte)))
	       (setf ohash
		     (logxor ohash
			     (hash-rot (if chte (hash-of-chte chte) (hash-parameter p)) 5)))))
	   ohash))

       (hash-bound-term (bt)
	 (let ((bhash 0))
	   (declare (type fixnum bhash))
	   (do ((bindings (bindings-of-bound-term bt) (cdr bindings))
		(i 1 (1+ i)))
	       ((null bindings))
	     (let* ((b (car bindings))
		    (chte (visit-binding b)))
	       (unless (eql (actual-of-chte chte) b)
		 (setf (car bindings) (actual-of-chte chte)))
	       (setf bhash (hash-member bhash (hash-of-chte chte) i))))

	   (let* ((term (term-of-bound-term bt))
		  (chte (visit-term term)))

	     ;;(when (eql '|!inf_tree_cons| (id-of-term term))
	     ;;(setf -term term -chte chte) (break "again"))

	     (unless (eq term (actual-of-chte chte))
	       ;; RLE TODO MTT WARNING DANGER : when multi-tasking must synch with variable invocations.
	       (setf (cdr bt) (actual-of-chte chte)))

	     (logxor (hash-of-chte chte) (hash-rot bhash 5)))))

       (hash-term (term)
	 (let* ((op (operator-of-term term))
		(ochte (visit-operator op)))
	   
	   (unless (eq op (actual-of-chte ochte))
	     (setf (car (term-values term)) (actual-of-chte ochte)))

	   ;; bound-terms
	   (let ((thash 0))
	     (declare (type fixnum thash))
	     (do ((bound-terms (bound-terms-of-term term) (cdr bound-terms))
		  (i 0 (1+ i)))
		 ((null bound-terms))
	       (setf thash
		     (hash-member thash
				  (hash-bound-term (car bound-terms))
				  i)))

	     (logxor (hash-of-chte ochte)
		     (hash-rot thash 5)))))

       (visit-symbol (s)
	 (compression-tables-find-add s *compression-symbol-type*
				      #'sxhash nil)
	 )

       (visit-binding (b)
	 (if (symbolp b)
	     (visit-symbol b)
	     (compression-tables-find-add b *compression-symbol-type*
					  #'(lambda (b)
					      (hash-parameter-value b *variable-type*))
					  (not (symbolp b)))))
       
       (visit-parameter (p)
	 ;;(when (token-parameter-p p) (setf -p p) (break (format-string "vp~a" p)))
	 (unless (and nil
		      (or (natural-parameter-p p)
			  (time-parameter-p p))
		      (real-parameter-p p))
		
	   (compression-tables-find-add p *compression-parameter-type*
					#'hash-parameter
					t
					#'equal-parameters-p
					)))
					  

       (visit-operator (o)
	 (compression-tables-find-add o *compression-operator-type*
				      #'hash-operator
				      t
				      #'equal))

       (visit-term (term)
		   
	 (compression-tables-find-add
	  term *compression-term-type*
	  ;; following  lambda wrapper circumvents compiler error
	  ;; in lucid 4.1 (DBCS, 12 October 1992) production compiler.
	  #'(lambda (x) (hash-term x))
	  t
	  #'(lambda (term-a term-b)
	      (and (eq (operator-of-term term-a) (operator-of-term term-b))
		   (apply-predicate-to-list-pair
		    (bound-terms-of-term term-a)
		    (bound-terms-of-term term-b)
		    #'(lambda (bt-a bt-b)
			(and (equal (bindings-of-bound-term bt-a)
				    (bindings-of-bound-term bt-b))
			     (eq (term-values (term-of-bound-term bt-a))
				 (term-values (term-of-bound-term bt-b)))))))))))

    (cond
      ((eql type *compression-term-type*) (visit-term structure))
      ((eql type *compression-operator-type*) (visit-operator structure))
      ((eql type *compression-parameter-type*) (visit-parameter structure))
      ((eql type *compression-binding-type*) (visit-binding structure))
      ((eql type *compression-opid-type*) (visit-symbol structure))
      (t (raise-error (error-message '(compress add-structure unknown type) type))))))



;; if chte seen twice then add bit is on.
;; if chte has index then assumed add previously performed.


;;	this can be avoided when all tables persistent tables.
;;	this pass may be able to flag pure terms ie terms with no compressed substructure
;;	to avoid unparse from having to check for compression on substructures.

;; might make sense to reverse tables? may be more likely to find hits in later tables.

(defun add-indices-to-compression-levels (term)
  (labels
      ((visit-constant (c)
	 (compression-table-set-add-flags c *compression-symbol-type*))

       (visit-parameter (p)
	 (unless (and nil
		      (or (natural-parameter-p p)
			  (time-parameter-p p))
		      (real-parameter-p p))

	   (compression-table-set-add-flags p *compression-parameter-type*)))

       (visit-operator (o)
	 (unless (compression-table-set-add-flags o *compression-operator-type*)
	   (visit-constant (id-of-operator o))
	   (mapc #'visit-parameter (parameters-of-operator o))))

       (visit-term (term)
	 (unless (compression-table-set-add-flags term *compression-term-type*)
	   (visit-operator (operator-of-term term))
	   (mapc #'visit-bound-term (bound-terms-of-term term))))
       
       (visit-bound-term (bt)
	 (mapc #'visit-constant (bindings-of-bound-term bt))
	 (visit-term (term-of-bound-term bt)))
       )
		    
    (visit-term term)))


;;;;	RLE TODO PERF IO: need to to byte io model rather than string accumulation
;;;;	RLE TODO PERF IO: to do tt need to catch and put in big byte array but do by
;;;;	RLE TODO PERF IO: bytes.


(defun walk-term-ascii-compress (term prl-stream &optional (compress-p t))

  (with-fbb prl-stream

    ;; PERF could make walk list delimited a macro.
    (let ((accumulate-f #'(lambda (b) (fbb-add-byte b)))
	  ) 
      (labels ((accumulate-string (s)
		 (string-to-byte-accumulator s *ascii-escape-sbits*
					     accumulate-f))
	       
	       ;; returns nil if ascii term needs to be sent
	       ;; t if index was sent instead.
	       (compressed (structure type)
		 (when compress-p

		   ;;(setf a structure b type)
		   ;;(when (ctae a b d) (break "wtacc"))
		   ;;(defun ctae (a b c) (when (= b 2) (format t "~a ~a ~%" c a)) nil)
		   (eql 'index (compression-table-encode structure type))
		   #|
					       #'(lambda (level index)
						   ;;(setf d index)
						   (compression-index-to-bytes level index
									       accumulate-f)
						   (setf index-sent-p t))
					       #'(lambda (code)
						   ;;(setf d nil)
       						   (fbb-add-byte code))
			   |#

		   ))


		 (visit-operator (op alwaysp)
				 (unless (compressed op *compression-operator-type*)

				   (let ((opid (id-of-operator op)))
				     (unless (compressed opid *compression-opid-type*)
				       (accumulate-string (string opid))))

				   (let ((parameters (parameters-of-operator op)))
				     (when (or parameters alwaysp)
				       (walk-list-delimited parameters
							    #'(lambda (p)
								(visit-parameter p))
							    accumulate-f
							    icomma ilcurly ircurly)))))
	   
		 (visit-term (term)
			     (unless (compressed term *compression-term-type*)
	       
				 (let ((bound-terms (bound-terms-of-term term)))
		    
				   (visit-operator (operator-of-term term) (null bound-terms))
				   (when bound-terms
				     (walk-list-delimited bound-terms
							  #'visit-bound-term
							  accumulate-f
							  isemicolon ilparen irparen)))))

		 (visit-parameter-value (val type)
					(let ((sexpr (parameter-value-to-sexpr val type)))
					  (walk-parameter-sexpr-ascii sexpr accumulate-f)))

		 (visit-binding (b)
				(unless (compressed b *compression-binding-type*)
				  (visit-parameter-value b *variable-type*)))

		 (visit-bound-term (bt)
				   (let ((bindings (bindings-of-bound-term bt)))
				     (if bindings
					 (progn (walk-list-delimited bindings
								     #'visit-binding
								     accumulate-f
								     icomma)
						(fbb-add-byte idot)
						(visit-term (term-of-bound-term bt)))
					 (visit-term (term-of-bound-term bt)))))

		 (visit-parameter (p)
				  (unless (or (and nil
						   (or (natural-parameter-p p)
						       (time-parameter-p p))
						   (real-parameter-p p))
					      (compressed p *compression-parameter-type*))
				    (fbb-add-array (parameter-to-fbb-cache p))))
		 )		 
    
	(visit-term term)))))


(defvar *compression-op-count-threshold* 128)
(defvar *compression-op-count-threshold-override* 1024)



(defun term-out-with-compression-level (pstream cur-levels leveln-p term)
  ;;(break "towcl")
  
  (let ((count (or (and cur-levels
			(count-of-compression-levels cur-levels))
		   0)))

    (when nil
      (let ((opc (term-op-count (cdr *big-term*))))
	(format t "~%Big term? ~a~%" opc)
	(when (> opc (car *big-term*))
	  (format t "~%Big term ~a~%" opc)
	  (setf *big-term* (cons opc *big-term*)))))
    
    (when (>= count *compression-level-max*)
      (raise-error (error-message '(compression level max with)
				  count *compression-level-max*)))

    (let ((overp nil))
      (let ((add-level-p (or (and leveln-p
				  (term-op-count-exceeds-p term *compression-op-count-threshold*))
			     (setf overp
				   (term-op-count-exceeds-p term *compression-op-count-threshold-override*)))))
	      
	(let ((*ephemeral-compression-tables-count* 0)
	      (levels (if add-level-p
			  (if (null cur-levels)
			      (construct-compression-levels nil 0 (new-compression-level 0 'out))
			      (construct-compression-levels cur-levels
							    count
							    (new-ephemeral-compression-level cur-levels count)))
			  cur-levels)))

	  ;; not strictly necessary but should find sharing.
	  ;; todo meter regression test to see if this helps or hurts.
	  ;; could add-structure-to-compression-levels but 
	  ;; problem would be that we accumulate garbage in level0 hash tables
	  ;; lots of overhead adding to tables, seems like a loser.
	  ;; it might find some sharing in current table.
	  ;; if did not add in find-add might be better, not so easy since
	  ;; need tree of chte's to produce hash to lookup in table.

	  ;;(format t "opcount ~a ~%" (term-op-count term))

	  ;; if overp factor term  -> but how to know to inflate on receiveing end.
	  ;;  combine inflation with scan?? to reduce cost or inflation.
	  ;;  combine factor with write to reduce cost of factor
	  ;; maybe use compression syntax to encode factorization rather than term
	  

	  (unwind-protect
	       ;;(with-io-echo-stats (nil (format-string "Term-write-~a" (term-op-count term))) )
	       (if levels
		   (with-levels-out (levels)
		     (let ((wterm
			    (if add-level-p
				;;(with-io-echo-stats (nil "add-structure-to-compression-levels")
				(actual-of-chte
				 (add-structure-to-compression-levels term *compression-term-type*))
				;; )
				term)))
	
		       ;;(setf a levels b term c wterm) (break "ff")
		       (when add-level-p
			 ;;(with-io-echo-stats (nil "add-indices-to-compression-levels")
			 (add-indices-to-compression-levels wterm))
		       ;;)
		 
		       (let ((opc (term-op-count wterm)))
			 (if (and nil (> opc 18000))
			     (progn
			       (break "wtacp")
			       ;;#+allegro(prof:with-profiling (:type :time :count t)
			       ;;(walk-term-ascii-compress wterm pstream t))
			       ;;#-allegro(walk-term-ascii-compress wterm pstream t)
			       ;;(setf -a levels -b wterm -c pstream) (break "wtac")
			       )
			     (walk-term-ascii-compress wterm pstream (and levels t))
			     )

			 )))
			 
		   (walk-term-ascii-compress term pstream nil))
		      

	    ;;(setf -a add-level-p -b cur-levels -c levels) (break)
		
	    (when (and add-level-p cur-levels)
	      ;; only reason to clean is to allow contents to be gc'ed thus if little added ignore.
	      ;; could be nickeled and dimed to death though. Any added must be removed or have
	      ;; bits reset otherwise next used is nfg.
	      (when (> *ephemeral-compression-tables-count* 0)
		;;(format t "ephemeral-compression-hash-entries ~a~%" *ephemeral-compression-tables-count*)
		(table-clean-ephemerals
		 (table-of-compression-level (level-of-compression-levels levels count)))))))))))

(defun add-term-to-training-tables (term)
  (format t "~%training-term sized ~a.~%" (term-op-count term))
  (let ((*compression-training-p* t))
    (with-levels-out (*compression-training-tables*)
      (add-structure-to-compression-levels term *compression-term-type*)
      (add-indices-to-compression-levels term))))


(defun compression-level-static-update (structure type)
  (compression-table-increment-index
   (add-structure-to-compression-levels	structure
					type
					)
   (add-byte type (compression-level-index))
   ;; added with with-levels change, not sure if apporpriate
   (compression-level-index)))
				       
#|

(defun compression-level-update-table (levels level-index structure type index)

  ;;(setf a levels b level-index c structure d type e index) (break "clut")
  (let* ((chte (add-structure-to-compression-levels levels
						    structure
						    type
						    level-index)))
    (setf -index index -level-index level-index) (break "clut")
    (setf (chte-index chte) (make-index-bytes level-index index))
    (when (> (chte-index chte) 2048)
      (break "clut"))
	
    (values)))

|#
	

(defun compression-levels-array-assign (levels code structure-f)
  (let* ((level-index (compression-level-of-byte code))
	 (level (level-of-compression-levels levels level-index))
	 (cate (new-cate code)))
    
    (unless (compression-level-in-p level)
      (system-error '(compression-levels-array-assign in not)))
    
    (let ((index (compression-array-assign (array-of-compression-level level) cate))
	  (structure (funcall structure-f)))
      (cate-set-actual cate structure)
      (when (compression-level-out-p level)
	(compression-level-update-table levels level-index
					structure (structure-type-of-cate cate)
					index)))

    cate))


(defun compression-arrays-lookup (levels level index)
  (let ((a (array-of-compression-array (array-of-compression-levels levels level))))
    (if a
	(aref a index)
	(raise-error (error-message '(compress lookup array not) level index)))))


;; needs to take multiple arrays with exclusive index ranges.
(defun scan-compressed-term-ascii (levels)
  (labels
      (;; returns cate
       (scan-compressed (code)
	 (if (compression-add-byte-p code)
	     (progn
	       (scan-next nil)		; need to catch escape so can't scan-bump.
	       (compression-levels-array-assign levels code
						#'(lambda ()
						    (scan-item (structure-type-of-add-byte code))))
	       )			; returns cate
	     (prog2
		 (scan-bump)		; must not scan-next-char here.
		 (progn
		   ;;(setf -levels levels)
		   ;;(format t "~a ~o ~o ~a~%" (compression-index-of-bytes code (scan-cur-byte))
		   ;;code (scan-cur-byte)	(compression-level-of-byte code))
		   (compression-arrays-lookup levels
					    (compression-level-of-byte code)
					    (compression-index-of-bytes code 
									(scan-cur-byte)))
		 )
	       (scan-next nil))))
       
       (scan-item (type)
	 ;;(setf a add-byte) (break "si")
	 (cond
	   ((eql type *compression-opid-type*)
	    (intern-system (scan-ascii-string)))
	   ((eql type *compression-binding-type*)
	    (sexpr-to-parameter-value (scan-parameter-value-sexpr (scan-ascii-string))
				      *variable-type*))
	   ((eql type *compression-parameter-type*)
	    (scan-parameter))
	   ((eql type *compression-operator-type*)
	    (scan-operator))
	   ((eql type *compression-term-type*)
	    (scan-term))
	   (t (raise-error (error-message '(scan compress add unknown type) type)))))

       (scan-parameter ()
	 (let ((code (scan-cur-byte)))
	   (if (and (compression-code-p code)
		    (not (= code *hex-word-byte*)))
	       (let ((cate (scan-compressed code)))
		 ;;(setf a cate b code)
		 (if (parameter-cate-p cate)
		     (actual-of-cate cate)
		     (raise-error (error-message '(scan compress parameter not)
						 (byte-of-cate cate)))))
	       (parameter-from-fbb-cache))))

       (scan-parameters ()
	 (when (scan-at-byte-p ilcurly)
	   (scan-delimited-list
	    #'scan-parameter
	    ilcurly ircurly
	    #'(lambda ()
		(scan-byte icomma)))))
       
       (scan-operator()
	 (let ((code (scan-cur-byte)))
	   ;;(setf -code code)
	   ;;(when (= 128 code)  (break "sct"))
	   (if (compression-code-p code)
	       (let* ((cate (scan-compressed code)))
		 (if (symbol-cate-p cate)
		     (instantiate-operator (actual-of-cate cate)
					   (scan-parameters))
		     (if (parameter-cate-p cate)
			 (actual-of-cate cate)
			 (raise-error (error-message '(scan compress op not) 
						     (byte-of-cate cate))))))
		 
	       (instantiate-operator (intern-system (scan-ascii-string))
				     (scan-parameters)))))

       (scan-term ()
	 
	 (when (scan-eof-p)
	   (raise-error (error-message '(scan compress term eof))))
	 
	 (let ((code (scan-cur-byte)))
	   (if (compression-code-p code)
	       (let* ((cate (scan-compressed code))
		      (type (structure-type-of-cate cate)))
		 (cond 
		   ((eql type *compression-opid-type*)
		    (instantiate-term
		     (instantiate-operator (actual-of-cate cate)
					   (scan-parameters))
		     (scan-bound-terms)))
		   ((eql type *compression-operator-type*)
		    (instantiate-term (actual-of-cate cate)
				      (scan-bound-terms)))
		   ((eql type *compression-term-type*)
		    (actual-of-cate cate))
		   (t (raise-error (error-message '(scan compress term not) (byte-of-cate cate))))))

	       (instantiate-term (scan-operator)
				 (scan-bound-terms))))
	 )
       (scan-bound-terms ()
	 (when (scan-at-byte-p ilparen)
	   (scan-delimited-list #'scan-bound-term
				ilparen irparen
				#'(lambda() (scan-byte isemicolon)))))

       (string-to-binding (s)
	 (sexpr-to-parameter-value (scan-parameter-value-sexpr s) *variable-type*))

       (scan-binding ()
	 (let ((code (scan-cur-byte)))
	   (if (compression-code-p code)
	       (let ((cate (scan-compressed code)))
		 (if (symbol-cate-p cate)
		     (actual-of-cate cate)
		     (raise-error (error-message '(scan compress binding not) (byte-of-cate cate)))))
	       (string-to-binding (scan-ascii-string)))))

       (scan-bound-term ()
	 (let ((code (scan-cur-byte)))
	   ;;(setf a code)
	   (if (compression-code-p code)
	       (let* ((cate (scan-compressed code))
		      (type (structure-type-of-cate cate)))
		 (cond
		   ;;    <opid>{
		   ;;    <opid>\(
		   ((eql type *compression-opid-type*)
		    (cond
		      ((scan-at-byte-p ilcurly)
		       (instantiate-bound-term
			(instantiate-term
			 (instantiate-operator (actual-of-cate cate)
					       (scan-parameters))
			 (scan-bound-terms))))
		       
		      ((scan-at-byte-p ilparen)
		       (instantiate-bound-term
			(instantiate-term
			 (instantiate-operator (actual-of-cate cate))
			 (scan-bound-terms))))

		      (t (setf -cate cate -code code) (break "sbt")
		       (raise-error
			(error-message '(scan compress bound-term cate not)
				       (scan-cur-char))))))
					
		   ((eql type *compression-binding-type*)
		    ;;    <binding>,
		    ;;    <binding>.
		    (cond
		      ((scan-at-byte-p icomma)
		       (let ((bindings (cons (actual-of-cate cate)
					     (scan-delimited-list #'(lambda () (scan-binding))
								  icomma idot
								  #'(lambda () (scan-byte icomma))))))
			 (instantiate-bound-term (scan-term) bindings)))

		      ((scan-at-byte-p idot)
		       (scan-next)
		       (instantiate-bound-term (scan-term)
					       (list (actual-of-cate cate))))))
		   
		   ((eql type *compression-operator-type*)
		    (instantiate-bound-term
		     (instantiate-term (actual-of-cate cate) ; op
				       (scan-bound-terms))))
		   
		   ((eql type *compression-term-type*)
		    (null-bindings-bound-term-of-term-cate cate)) ; term

		   (t (raise-error (error-message '(scan compress bound-term not)
						  (byte-of-cate cate))))))
	       
	       ;;    <binding>,
	       ;;    <binding>.
	       ;;    <opid>{
	       ;;    <opid>\(
	       (if (scan-at-byte-p idot)
		   (progn
		     (scan-next)
		     (instantiate-bound-term (scan-term)  (list (get-dummy-variable-id))))
		   (let ((s (scan-ascii-string)))
		     (cond
		       ((scan-at-byte-p icomma)
			(let ((bindings (cons (string-to-binding s)
					      (scan-delimited-list #'(lambda () (scan-binding))
								   icomma idot
								   #'(lambda () (scan-byte icomma))))))
			  (instantiate-bound-term (scan-term) bindings)))

		       ((scan-at-byte-p idot)
			(scan-next)
			(instantiate-bound-term (scan-term)
						(list (string-to-binding s))))
					
		       ((scan-at-byte-p ilcurly)
			(instantiate-bound-term
			 (instantiate-term
			  (instantiate-operator (intern-system s)
						(scan-parameters))
			  (scan-bound-terms))))
		       
		       ((scan-at-byte-p ilparen)
			(instantiate-bound-term
			 (instantiate-term
			  (instantiate-operator (intern-system s))
			  (scan-bound-terms))))

		       (t;;(setf a s) (break)
			(raise-error (error-message '(scan compress bound-term not) (scan-cur-char))))))))))
       )

    (scan-term)))

#|

;; RLE TODO : twould be better to accumulate a byte array rather than a string here??
;; RLE TODO : yes, but then need inverse, whose only purpose would be for testing.
;; RLE TODO : thus conditonalize so that only lisps which map ascii and char-codes one-to-one
;; RLE TODO : can use. In long run may make sense to do byte array. Testing is important
;; RLE TODO : as compression is complex.
#+lucid
(defun session-term-to-string (levels term)
  (with-output-to-string (s)
    (when (compression-training-tables-p)
      (add-term-to-training-tables term))
    (when levels
      (add-structure-to-compression-levels term *compression-term-type*)
      (add-indices-to-compression-levels term))
    (walk-term-ascii-compress levels
			      term
			      #'(lambda (byte)
				  (write-char s (code-char byte))))))


#+lucid
(defun session-string-to-term (levels s)
  (with-string-scanner (s)
    (scan-compressed-term-ascii levels)))

(defun test-non-compress (term)
  (let ((s (time (term-to-standard-character-string term))))
    (format t "Length ~a~%" (length s))
    (compare-terms-p term
		     (time (standard-character-string-to-term s)))))



#+lucid
(defun test-compress-long (terms  &optional levels (comp-stats-p nil))

  (format t "~%  *****~%  *****  ~a ~a. ~%  *****~%" "Compression" (reduce #'+ (mapcar #'term-op-count terms)))

  (let ((dcwp-length 0)
	(ndcwp-length 0)
	(dcop-length 0)
	(noc-length 0)
	
	(dcwp-intime 0)
	(ndcwp-intime 0)
	(dcop-intime 0)
	(noc-intime 0)

	(dcwp-outtime 0)
	(ndcwp-outtime 0)
	(dcop-outtime 0)
	(noc-outtime 0)

	(dcwp-inec 0)
	(ndcwp-inec 0)
	(dcop-inec 0)
	(noc-inec 0)

	(dcwp-outec 0)
	(ndcwp-outec 0)
	(dcop-outec 0)
	(noc-outec 0))
	
    (gc)
    (when levels

      ;; dry run to reduce paging.
      (let ((*compression-levels* levels))
	(let ((s (with-new-compression-level ('out *compression-level-size*)
		   (session-term-to-string (car  terms)))))
	  (with-new-compression-level ('in) (session-string-to-term s))))
		     
      (my-time ("Dynamic Compression with levels, total")
	       (let ((cs nil)
		     (table nil)
		     (new-terms nil)
		     (array nil))

		 (let ((*compression-levels* levels))
		   
		   (with-new-compression-level ('out)
		     (setf cs  (my-time ("Compress Out dcwp"
					 #'(lambda (time ec)
					     (incf dcwp-outtime time)
					     (incf dcwp-outec ec)))
					(mapcar #'session-term-to-string terms)))
		     (let ((levels (compression-levels)))
		       (setf table (table-of-compression-levels levels
								(1- (count-of-compression-levels levels))))))
		   (with-new-compression-level ('in)
		     (setf new-terms (my-time ("Compress In"
					       #'(lambda (time ec)
						   (incf dcwp-intime time)
						   (incf dcwp-inec ec)))
					      (mapcar #'session-string-to-term cs)))
		     (let ((levels (compression-levels)))
		       (setf array (array-of-compression-levels levels
								(1- (count-of-compression-levels levels)))))))
	      
		 (terpri)
	
		 (when comp-stats-p (compression-stats terms table array))

		 (setf dcwp-length (reduce #'+ (mapcar #'length cs)))

		 (terpri)
		 (format t "Lengths ~a~%" (mapcar #'length cs))
		 (format t "Compares ~a~%" (mapcar #'compare-terms-p new-terms terms))
		 ))

      (gc)
      (let ((*compression-levels* levels))
	(session-string-to-term
	 (session-term-to-string (car  terms))))
	
      (my-time ("No Dynamic Compression, levels only.")
	       (let ((*compression-levels* levels))
		 
		 (let ((cs nil)
		       (table nil)
		       (new-terms nil)
		       (array nil))
	      
		   (setf cs (my-time ("Compress Out ndcwp"
				      #'(lambda (time ec)
					  (incf ndcwp-outtime time)
					  (incf ndcwp-outec ec)))
				     (mapcar #'session-term-to-string terms)))
		   (let ((levels (compression-levels)))
		     (setf table (table-of-compression-levels levels
							      (1- (count-of-compression-levels levels)))))
	      
		   (setf new-terms (my-time ("Compress In"
					     #'(lambda (time ec)
						 (incf ndcwp-intime time)
						 (incf ndcwp-inec ec)))
					    (mapcar #'session-string-to-term cs)))
		   (let ((levels (compression-levels)))
		     (setf array (array-of-compression-levels levels
							      (1- (count-of-compression-levels levels)))))
	      
		   (terpri)
	
		   (when comp-stats-p (compression-stats terms table array))

		   (setf ndcwp-length (reduce #'+ (mapcar #'length cs)))
		   
		   (terpri)
		   (format t "Lengths ~a~%" (mapcar #'length cs))
		   (format t "Compares ~a~%" (mapcar #'compare-terms-p new-terms terms))
		   ))))
    
    (gc)
    (let ((*compression-levels* nil))
      (let ((s (with-new-compression-level ('out)
		 (session-term-to-string (car  terms)))))
	(with-new-compression-level ('in) (session-string-to-term s))))

    (my-time ("Dynamic Compression only.")
	       (let ((*compression-levels* nil))
		 
		 (let ((cs nil)
		       (table nil)
		       (new-terms nil)
		       (array nil))
	
		   (with-new-compression-level ('out)
		     (setf cs  (my-time ("Compress Out dcop"
					 #'(lambda (time ec)
					     (incf dcop-outtime time)
					     (incf dcop-outec ec)))
					(mapcar #'session-term-to-string terms)))
		     (let ((levels (compression-levels)))
		       (setf table (table-of-compression-levels levels
								(1- (count-of-compression-levels levels))))))
	      
		   (with-new-compression-level ('in)
		     (setf new-terms (my-time ("Compress In"
					       #'(lambda (time ec)
						   (incf dcop-intime time)
						   (incf dcop-inec ec)))
					      (mapcar #'session-string-to-term cs)))
		     (let ((levels (compression-levels)))
		       (setf array (array-of-compression-levels levels
								(1- (count-of-compression-levels levels))))))
	      
		   (terpri)
	
		   (when comp-stats-p (compression-stats terms table array))

		   (setf dcop-length (reduce #'+ (mapcar #'length cs)))
		   
		   (terpri)
		   (format t "Lengths ~a~%" (mapcar #'length cs))
		   (format t "Compares ~a~%" (mapcar #'compare-terms-p new-terms terms))
		   )))

    (gc)
    (standard-character-string-to-term (term-to-standard-character-string (car terms)))
		    
    (my-time ("No Compression.")
	     (let ((*compression-levels* nil))
		 
	       (let ((cs nil)
		     (new-terms nil))
	      
		 (setf cs  (my-time ("Compress Out noc"
				     #'(lambda (time ec)
					 (incf noc-outtime time)
					 (incf noc-outec ec)))
				    (mapcar #'term-to-standard-character-string terms)))
		   	      
		 (setf new-terms (my-time ("Compress In"
					   #'(lambda (time ec)
					       (incf noc-intime time)
					       (incf noc-inec ec)))
					  (mapcar #'standard-character-string-to-term cs)))
	      
		 (terpri)

		 (setf noc-length (reduce #'+ (mapcar #'length cs)))
		   
		 (terpri)
		 (format t "Lengths ~a~%" (mapcar #'length cs))
		 (format t "Compares ~a~%" (mapcar #'compare-terms-p new-terms terms))
		 )))

  (terpri)

     (format t "~%  *****~%  *****  ~a ~%  *****~%" "Summary")
     (when levels
       (format t "~%  ** Level0-N : Length ~10:D. Out: time ~a, consing ~a. In: time ~a, consing ~a."
	       dcwp-length
	       (num-secs-to-string dcwp-outtime) (num-bytes-to-string dcwp-outec)
	       (num-secs-to-string dcwp-intime) (num-bytes-to-string dcwp-inec))
       (format t "~%  ** Level0   : Length ~10:D. Out: time ~a, consing ~a. In: time ~a, consing ~a."
	       ndcwp-length
	       (num-secs-to-string ndcwp-outtime) (num-bytes-to-string ndcwp-outec)
	       (num-secs-to-string ndcwp-intime) (num-bytes-to-string ndcwp-inec)))
     (format t "~%  ** LevelN   : Length ~10:D. Out: time ~a, consing ~a. In: time ~a, consing ~a."
	     dcop-length
	     (num-secs-to-string dcop-outtime) (num-bytes-to-string dcop-outec)
	     (num-secs-to-string dcop-intime) (num-bytes-to-string dcop-inec))
     (format t "~%  ** None     : Length ~10:D. Out: time ~a, consing ~a. In: time ~a, consing ~a."
	     noc-length
	     (num-secs-to-string noc-outtime) (num-bytes-to-string noc-outec)
	     (num-secs-to-string noc-intime) (num-bytes-to-string noc-inec))))

|#

(defun compression-stats (terms ct ca)

  ;;(setf dd ct ee ca)

  (term-stats terms)
  (terpri)
    
  (compression-table-stats ct)

  (compression-array-stats ca)

  nil)


;; op(a; (op (b; c)) -> !rft(op(); a; b; c)


;; quote !rfts .
;; rft could include unique,static, and global obid to insure no clash without overhead of quoting.
(defun right-factor-term (term)
  ;; return bt list.
  (labels ((accumulate (op bterm)
	     ;; not here unless bterm has null bindings.
	     (let* ((term (term-of-bound-term bterm))
		    (bts (bound-terms-of-term term))
		    (snd (cadr bts)))
	       (if (and snd (null (cddr bts))
			(null (bindings-of-bound-term snd))
			(equal-operators-p op (operator-of-term term)))
		   (cons (car bts) (accumulate op snd))
		   (list (maybe-instantiate-bound-term bterm
						       nil
						       (visit term)
						       )))))

	   (visit (term)
	     (let* ((bts (bound-terms-of-term term))
		   (snd (cadr bts)))
	       (if (and snd (null (cddr bts))
			(null (bindings-of-bound-term snd)))
		   (let ((sterm (term-of-bound-term snd))
			 (op (operator-of-term term)))
		     (if (equal-operators-p op (operator-of-term sterm))
			 (instantiate-term (instantiate-operator '!rft nil)
					   (cons (instantiate-bound-term (instantiate-term op nil))
						 (cons (car bts)
						       (accumulate op snd))))
			 (maybe-instantiate-term term
						 (operator-of-term term)
						 (mapcar #'(lambda (bt) (maybe-instantiate-bound-term bt
										     (bindings-of-bound-term bt)
										     (visit (term-of-bound-term bt))))
							 (bound-terms-of-term term)))))

		   (maybe-instantiate-term term
					   (operator-of-term term)
					   (mapcar #'(lambda (bt)
						       (maybe-instantiate-bound-term bt
										     (bindings-of-bound-term bt)
										     (visit (term-of-bound-term bt))))
						   (bound-terms-of-term term))))))
	   )

    (visit term) ))

(defun inflate-right-factor-term (term)
  (labels ((visit (term)
	     (if (eql (id-of-term term) '!rft)
		 (accumulate (operator-of-term (icar term))
			     (cdr (bound-terms-of-term term)))
		 (let ((bts (bound-terms-of-term term)))
		   (if bts
		       (maybe-instantiate-term term
					       (operator-of-term term)
					       (mapcar #'(lambda (bt)
							   (maybe-instantiate-bound-term
							    bt
							    (bindings-of-bound-term bt)
							    (visit (term-of-bound-term bt))))
						       bts))
		       term))))

	   (accumulate (op bts)
	     (if (cddr bts)
		 (instantiate-term op
				   (cons (car bts)
					 (list (instantiate-bound-term (accumulate op (cdr bts))))))
		 (instantiate-term op bts))))

    (visit term)))

