
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


#-dontinline
(eval-when (compile)
  (proclaim '(inline
	      free-vars-p free-vars-of-term
	      tags-of-imessage-term stamp-of-imessage-term pmsg-of-imessage-term
	      icar icdr ilist-nil-p icons-term-of-op
	      itext-token-term  itext-string-term itext-term
	      real-itext-term-p
	      )))


;;;;
;;;; -docs- (mod trm data)
;;;;	
;;;;	Terms as Data : 
;;;;
;;;;	  Any data of global significance will be marshalled to terms
;;;;	  to provide a unified method of transmitting and storing
;;;;	  data.
;;;;	
;;;;	Data which can be marshalled/unmarshalled includes but is not
;;;;	limited to the following:
;;;;
;;;;	Text
;;;;
;;;;	Object Address
;;;;	Stamp
;;;;	Dependency
;;;;	Message
;;;;	Definition
;;;;
;;;;	Abstraction Definition
;;;;	Rule Definition
;;;;	Statement Definition
;;;;	Proof Definition
;;;;	Inference Tree
;;;;	
;;;;	Broadcast
;;;;	ML Source Expressions
;;;;	
;;;;	Requests and Responses.
;;;;
;;;; -doct- (mod trm)
;;;;	
;;;;	ILists: An ilist is a term representing an sexpr.
;;;;
;;;;	<isexpr-term>		: <cons-op>()
;;;;				| <cons-op>(<isexpr-term>; <isexpr-term>)
;;;;				| <term>
;;;;
;;;;	<ilist-term>		: <cons-op>()
;;;;				| <ilist-term-aux>
;;;;	<ilist-term-aux>	: <cons-op>(<term>; <ilist-term-aux>)
;;;;				| <term>
;;;;
;;;;
;;;;	ilist-cons-term-p (<term> <op>)		: <bool>
;;;;	ilist-nil-term-p (<term> <op>)		: <bool>
;;;;
;;;;	icar(<term>)				: <term>
;;;;	icdr(<term>)				: <term>
;;;;
;;;;	icons-term-of-op (<op> <term> <term>)	: <term>
;;;;
;;;;	map-sexpr-to-ilist (<*> sexpr <term{nil}> &optional <closure{f}>)	: <ilist-term>
;;;;	 ** f (<*>)		: <term>
;;;;	map-sexpr-to-isexpr (<*> sexpr <term{nil}> &optional <closure{f}>)	: <isexpr-term>
;;;;	 ** f (<*>)		: <term>
;;;;	 ** Maintains cons structure of sexpr.
;;;;	
;;;;	map-isexpr (<isexpr-term> <op> <closure{f}>)			: NULL
;;;;	map-isexpr-to-list (<isexpr-term> <op> &optional <closure{f}>)	: <*> list
;;;;	map-isexpr-to-sexpr (<isexpr-term> <op> &optional <closure{f}>)	: <*> list
;;;;	 ** f (<term>)		: <*>
;;;;	 ** Maintains cons structure of isexpr.
;;;;	
;;;;	** Default closure is #'(lambda (x) x)
;;;;	
;;;;
;;;;	Text/Terms.
;;;;
;;;;	Extend <text> :
;;;;
;;;;	<text>  : nil | 'SPACE | 'BREAK | <string> | <symbol> | <term>
;;;;		| <break-control(<type>, <text>)>
;;;;		| (<text> . <text>)
;;;;
;;;; -doct- (mod trm data)
;;;;
;;;;	Terms are added to text by extending the <text> syntax to include terms.
;;;;	In order to allow for the primitive terms representing text to be passed 
;;;;	as data, there is a !text_term_literal(0) operator defined. When intepreting
;;;;	term as text the subterm of the literal is passed directly as data.
;;;;	Thus when injecting terms into text which may contain text operators
;;;;	which are meant to be interpreted as terms, they should be wrapped in
;;;;	the !text_term_literal. Note that it does no harm to wrap non text operators.
;;;;
;;;;
;;;;	<text-term>	: !text_cons()
;;;;			| !space()	
;;;;			| !break()
;;;;			| !text{<text>:s}
;;;;			| !text{<text>:t}
;;;;			| !break_control{<type>:t}(<text-term>)
;;;;			| !text_cons(<text-term>; <text-term>)
;;;;			| <term>
;;;;			| !text_term_literal<term>
;;;;
;;;; -doct- (mod trm)
;;;;
;;;;	text-to-term (<text>)		: <term>
;;;;	term-to-text (<term>)		: <text>
;;;;
;;;; -doce- 
;;;;
;;;;
;;;;	RLE NAP
;;;;	where pmsg is pmsg or term version of pmsg.
;;;;	what's the point to having sexpr pmsg usable by ML??
;;;;	possibly could have <text> as ml abstract type and
;;;;    avoid overhead of producing term to do recursive eval?
;;;;	not a priority
;;;;
;;;;	RLE MAYBE
;;;;	 have message as abstract type and have fail do msgs instead
;;;;	 of tokens. Allow access and manipulation.
;;;;

;;;;
;;;; Search template should be template ala dform model.
;;;;	might be able to share code with dform match.
;;;;
;;;;	want search for stamp stuff too. Like objects changed during a particular time period.
;;;;
;;;;	term collection :
;;;;
;;;;	some functionality to make it easier to force global sharing of terms.
;;;;
;;;;	some difficulties : makes marks on subterms ephemeral as
;;;;			    term collection may replace subterm.
;;;;
;;;;	marks should be presented as ephemeral, as a means of avoiding
;;;;	computation by storing values. Or union marks when replacing subterm
;;;;	Marks are safe within a dynamic extent.
;;;;	could go so far as to identify a class of marks which preclude
;;;;	collection of any term in which they occur, eg maybe lib-display-cache?. And a class of
;;;;	marks which gets migrated to the shared term, eg free-vars, maybe core dtree.
;;;;	Need stats, which indicate how much sharing before and after collection.
;;;;
;;;;  

;;;; term_string_search 
;;;; term_term_search 


;;;;	collect-terms-permanent (term)
;;;;	 ** assumes term could never be collected.
;;;;	collect-terms-map (<id{tag}> <map{closure}>) 
;;;;	 ** map(<closure{collector}>) 	: NULL
;;;;	 ** collector (<term>)
;;;;

;;;;  ------------------------------------------------------------------------
;;;;  ----------------------- terms ------------------------------------------
;;;;  -doc-s-------------------------------------------------------------------
;;;;  -level-80-
;;;;
;;;;  -level-80-
;;;; ;;; with macros : marks terms with free-vars of term during execution of
;;;; ;;;               body of macro.
;;;;
;;;;  value-of body	: with-free-vars (term body)
;;;;
;;;;  value-of body	: with-free-vars-l (terms body)
;;;;
;;;;  value-of body	: with-free-var-tree 
;;;;				(term 
;;;;                             terms:term...term)	; optional
;;;;				body
;;;;
;;;;  -level-08-
;;;;  bool		: free-vars-p (term)
;;;;	; tests if term marked with free-vars.
;;;;
;;;;  variable-ids	: free-vars-of-term (term)
;;;;	; returns value of free-vars mark; returns nil if term not marked. 
;;;;	; should not be called unless it is marked.
;;;;
;;;;  -level-80-
;;;;  Object : address - list of integers which are indices into bound-term
;;;;			 or binding lists.
;;;;
;;;;  term		: get-term-in-term (term address)
;;;;
;;;;  variable-id	: get-binding-in-term (term address)
;;;;
;;;;  bound-term	: get-bound-term-in-term (term address)
;;;;
;;;;
;;;;  -level-80-
;;;;  -page------------------------------------------------------------------
;;;;
;;;; ;;; map functions 
;;;;
;;;;  void		: map-on-bound-terms	
;;;;				function
;;;;				(term ... term)		; one required
;;;;				lists:*'s ... *'s	; optional
;;;;
;;;;  *s		: mapl-on-bound-terms
;;;;				function
;;;;				(term ... term)		; one required
;;;;				lists:*'s ... *'s	; optional
;;;;
;;;;  bool		: mapt-on-bound-terms
;;;;				function
;;;;				(term ... term)		; one required
;;;;				lists:*'s ... *'s	; optional
;;;;
;;;;  bool		: mapn-on-bound-terms
;;;;				function
;;;;				(term ... term)		; one required
;;;;				lists:*'s ... *'s	; optional
;;;;
;;;;  term		: mapcr-on-bound-terms
;;;;				function
;;;;				(term ... term)		; one required
;;;;				lists:*'s ... *'s	; optional
;;;;
;;;;  -doce-------------------------------------------------------------------



;;;  --------------------------------------------------------------
;;;  -----------------  term free-vars ----------------------------
;;;  --------------------------------------------------------------

;;  push and pop free vars allows the user to store the free vars
;;  of certain terms with the term. This allows a convenient method
;;  of avoiding recomputation of the free vars when the user is aware
;;  of a dynamically scoped context which is likely to use free-vars repeatedly.

;; tests if term is marked with free-vars.
(defun free-vars-p (term)
  (markp term 'free-vars))

;; should only be used when caller is certain that term is marked with free-vars.
(defun free-vars-of-term (term)
  (mark-value term 'free-vars))

(defmacro with-free-vars (term &body body)
  `(with-mark-f ('free-vars #'free-vars ,term)
	   ,@body))

(defmacro with-free-vars-l (terms &body body)
  `(with-map-mark-f ('free-vars #'free-vars ,terms)
     ,@body))


;;;  ----------------------------------------------------------------------
;;;  -----------------------  mapping functions  --------------------------
;;;  ----------------------------------------------------------------------
;;; 
;;;  map   - mapc
;;;  mapl  - mapcar
;;;  mapt  - every
;;;  mapn  - some
;;;
;;;  ----------------------------------------------------------------------


;;   ----------------------------------------------------------------------
;;   ------------------- map on bound terms of term -----------------------
;;   ----------------------------------------------------------------------


;;; RLE TODO mapcr etc. could be improved by avoiding consing up new list in case where
;;;	identical result. If improved should be more widely used. Might also look into
;;;	some type of catch throw mechanism.

(defmacro mapcr-on-bound-terms (f (term &rest more-terms) &rest lists)
  `(let* ((term-var ,term)
	  (exists-diff nil)
	  (bts (mapcar #'(lambda (bound-term &rest args)
			   (let ((bt (apply ,f bound-term args)))
			     (unless (eq bt bound-term)
			       (setf exists-diff t))
			     bt))
		       (bound-terms-of-term term-var)
		       ,.(mapcar #'(lambda (other) 
				     (bound-terms-of-term other))
				 more-terms)
		       ,.lists)))
     (if exists-diff
	 (instantiate-term (operator-of-term term-var) bts)
	 term-var)))


(defmacro mapcr-on-subterms (f (term &rest more-terms) &rest lists)
  `(let* ((term-var ,term)
	  (exists-diff nil)
	  (bts (mapcar 
		 #'(lambda (bound-term &rest args)
		     (let ((tt (apply ,f (term-of-bound-term bound-term) args)))
		       (if (eq tt (term-of-bound-term bound-term))
			   bound-term
			   (progn
			     (setf exists-diff t)
			     (instantiate-bound-term tt (bindings-of-bound-term-n bound-term))))))
		 (bound-terms-of-term term-var)
		 ,.(mapcar #'(lambda (other) 
			       (mapcar #'term-of-bound-term-f 
				       (bound-terms-of-term other)))
			   more-terms)
		 ,.lists)))
     (if exists-diff
	 (instantiate-term (operator-of-term term-var) bts)
	 term-var)))


;; assume hits infrequently.
(defmacro mapcr-on-subterms-optimized (term body)
  (let ((term-var (gensym))
	(hits (gensym))
	(bt (gensym))
	(bts (gensym))
	(btt (gensym))
	(nbtt (gensym))
	(i (gensym)))
    
    `(let ((,term-var ,term)
	   (,hits nil))
      (let ((,bts (bound-terms-of-term ,term-var)))
	 
	(do ((,i 0)
	     (,bt ,bts (cdr ,bt)))
	    ((null ,bt))
	  (let ((,btt (term-of-bound-term (car ,bt))))
	    (let ((,nbtt (,body ,btt)))
	      (unless (eq ,nbtt ,btt)
		(setf ,hits (cons (cons ,i ,nbtt) ,hits))))
	    (incf ,i)))
	;;(setf -hits ,hits) (break "hits")
      (if ,hits
	  (instantiate-term (operator-of-term ,term-var)
			    (do ((hits (nreverse ,hits))
				 (acc nil)
				 (j 0 (1+ j))
				 (bts ,bts (cdr bts)))
				((or (null hits) (null bts))
				 (if bts
				     (append (nreverse acc) bts)
				     (nreverse acc)))
			      (push (if (eql (caar hits) j)
					(prog1
					    (instantiate-bound-term (cdar hits) (bindings-of-bound-term (car bts)))
					  (setf hits (cdr hits)))
					(car bts))
				    acc)))
	  ,term-var)))))
	   


;(defmacro mapt-on-bound-terms (f term &rest more-terms)
;  `(every ,f ,term ,@more-terms))

;(defmacro mapn-on-bound-terms (f term &rest more-terms)
;  `(some ,f ,term ,@more-terms))



;;;  ----------------------------------------------------------------------
;;;  --------------------- term construction ------------------------------
;;;  ----------------------------------------------------------------------


;; remove non-variable bindings, non-variable bindings should be quite rare.
(defun real-bindings (bindings)
  (if (forall-p-optimized (b bindings) (variable-id-p b))
      bindings
      (mapcan #'(lambda (binding)
		  (let ((var (value-of-parameter-value binding)))
		    (when (variable-id-p binding)
		      (list var))))
	      bindings)))


;; marks all nodes of a term with their free vars.
(defun set-free-var-tree (term)
  (labels
    ((visit (term)
       (if (markp term 'free-vars)
	   (progn
	     (mapc #'(lambda (bound-term)
		       (visit (term-of-bound-term bound-term)))
		   (bound-terms-of-term term))
	     (mark-incf term 'free-vars))
	   ;; variables should always have free vars pushed.
	   (let ((free-vars
		   (if (variable-p term)
		       (list (id-of-variable-term term))
		       (unionl-vars
			 (mapcar #'(lambda (bound-term)
				     (diff-vars (visit (term-of-bound-term bound-term))
						(real-bindings (bindings-of-bound-term-n bound-term))))
				 (bound-terms-of-term term))))))
		 (mark-incf term 'free-vars free-vars)
		 free-vars) )))

    (when term
      (visit term))))

;; RLE NAP if function survives then pair (void(); void()) should return nil and not (nil (nil) (nil))
;;; and callers should recognize this
(defun make-free-var-tree (term)
  (labels
      ((visit (term)
	 (if (variable-p term)
	     (cons (list (id-of-variable-term term)) nil)
	     (let* ((free nil)
		    (children (mapcar #'(lambda (bound-term)
					  (let ((subterm-free (visit (term-of-bound-term bound-term))))
					    (push (diff-vars (car subterm-free)
							     (real-bindings (bindings-of-bound-term bound-term)))
						  free)
					    subterm-free))
				      (bound-terms-of-term term))))
	       (cons (unionl-vars free) children)))))
    (visit term)))




;
;;; second-order-occurences.
;;;
;;; ?? when is a free first-order occurence a second-order occurence ie v =? v[] ??
;;; always.

(defmacro id-of-occurence (occurence)
  `(car ,occurence))
(defmacro arity-of-occurence (occurence)
  `(cdr ,occurence))
(defmacro make-occurence (id arity)
  `(cons ,id ,arity))

(defmacro with-second-order-variable-occurence-tree ((term) &body body)
  `(unwind-protect 
       (progn (second-order-variable-occurence-tree ,term)
	      ,@body)
     (clear-second-order-variable-occurences ,term)))

;; may only be used within with-second-order-variable-occurence-tree
(defun second-order-occurences-of-term (term)
  (mark-value term 'second-order-occurences))
    
(defun second-order-occurences-of-term-less-bindings (term bindings)
  (if bindings
      (mapcan #'(lambda (occurence)
		  (unless (and (= 0 (arity-of-occurence occurence))
			       (member (id-of-occurence occurence) bindings))
		    (list occurence)))
	      (second-order-occurences-of-term term))
      (second-order-occurences-of-term term)))

(defmacro second-order-occurences-of-bound-term (bound-term)
  (let ((x (gensym)))
    `(let ((,x ,bound-term))
       (second-order-occurences-of-term-less-bindings (term-of-bound-term ,x) 
						      (bindings-of-bound-term-n ,x)))))

(defun second-order-variable-occurence-tree (term)
  (labels
    ((visit (term)
       (mapc #'(lambda (bound-term)
		 (visit (term-of-bound-term bound-term)))
	     (bound-terms-of-term term))
	 
       (with-variable-invocation
	 (let ((occurences nil))
	   (mapc #'(lambda (bound-term)
		     (mapc #'(lambda (occurence)
			       (unless (or (and (= 0 (arity-of-occurence occurence))
						(member (id-of-occurence occurence)
							(bindings-of-bound-term-n bound-term)))
					   (markv-p (id-of-occurence occurence)
						    (arity-of-occurence occurence)))
				 (markv (id-of-occurence occurence) 
					(arity-of-occurence occurence))
				 (push occurence occurences)))
			   (second-order-occurences-of-term (term-of-bound-term bound-term))))
		 (bound-terms-of-term term))
					       
	   (when (variable-term-p term)
	     (unless (markv-p (id-of-variable-term term)
			      (length (bound-terms-of-term term)))
	       (push (make-occurence (id-of-variable-term term)
				     (length (bound-terms-of-term term)))
		     occurences)))

	   (when occurences
	     (mark term 'second-order-occurences occurences))
	   occurences))))

  (visit term)))


(defun clear-second-order-variable-occurences (term)
  (when (markp term 'second-order-occurences)
    (unmark term 'second-order-occurences)
    (mapc #'(lambda (bound-term) 
	      (clear-second-order-variable-occurences 
		(term-of-bound-term bound-term)))
	  (bound-terms-of-term term))))

(defun clear-free-var-tree (term)
  (mark-decf term 'free-vars)
  (mapc #'(lambda (bound-term) 
	    (clear-free-var-tree (term-of-bound-term bound-term)))
	(bound-terms-of-term term)))

(defmacro with-free-var-tree ((term &rest more-terms) &body body)
  `(let ((l (list ,@more-terms)))
     (unwind-protect 
	 (progn (set-free-var-tree ,term)
		(mapc #'set-free-var-tree
		      l)
		,@body)
       (progn
	 (clear-free-var-tree ,term)
	 (mapc #'clear-free-var-tree 
	       l)))))


;; does not include zero arity vars.
(defun second-order-occurences (term)
  (let ((result nil))
    (labels
      ((visit (term)
	 (when (variable-term-p term)
	   (let ((arity (length (bound-terms-of-term term))))
	     (unless (or (zerop arity)
			 (markv-p (id-of-variable-term term) arity))
	       (markv (id-of-variable-term term) arity)
	       (push (cons (id-of-variable-term term) arity) result))))

	 (mapc #'(lambda (bound-term)
		   (visit (term-of-bound-term bound-term)))
	       (bound-terms-of-term term))))
      
      (with-variable-invocation
	(visit term)))
    
    result))



;; includes  0 arity vars.
(defun second-order-free-vars (term)
  (let ((result nil))
    (labels
      ((visit (term)
	 (when (variable-term-p term)
	   (let ((arity (length (bound-terms-of-term term))))
	     (unless (or (and (zerop arity)
			      (bound-var-p (id-of-variable-term term)))
			 (markv-p (id-of-variable-term term) arity))
	       (markv (id-of-variable-term term) arity)
	       (push (cons (id-of-variable-term term) arity) result))))

	 (mapc #'(lambda (bound-term)
		   (enter-bindings (bindings-of-bound-term-n bound-term))
		   (visit (term-of-bound-term bound-term))
		   (exit-bindings (bindings-of-bound-term-n bound-term)))
	       (bound-terms-of-term term))))
      
      (with-variable-invocation
	(visit term)))
    
    result))


(defun count-ops (term)
  (1+ (apply #'+ (mapcar #'(lambda (bt)
			    (count-ops (term-of-bound-term bt)))
			(bound-terms-of-term term)))))






;;;;
;;;;	Ilist
;;;;

(define-primitive |!cons| nil (car cdr))

(defun icar (term) (term-of-bound-term (car (bound-terms-of-term term))))
(defun icdr (term) (term-of-bound-term (cadr (bound-terms-of-term term))))

(defun icons-term-of-op (op a b)
  (instantiate-term op (list (instantiate-bound-term a) (instantiate-bound-term b))))

(defun ilist-cons-p (term op)
  (and (equal-operators-p op (operator-of-term term))
       (let ((bound-terms (bound-terms-of-term term)))
	 (and bound-terms
	      (let ((one (car bound-terms))
		    (rest (cdr bound-terms)))
		(and one
		     (null (bindings-of-bound-term-n one))
		     rest
		     (let ((two (car rest)))
		       (and two
			    (null (bindings-of-bound-term-n two))
			    (null (cdr rest))))))))))

(defun ilist-nil-p (term op)
  (and (equal-operators-p op (operator-of-term term))
       (null (bound-terms-of-term term))))


(defun count-sexpr (s &optional (report-threshold 10000) (depth-limit 100) (width-limit 500) (acc-limit 100000))

  (let ((maxw 0)
	(maxd 0)
	(acc 0)
	(reportp nil))

    (labels ((report ()
	       (format t
		       "~%;;; S-Expression Metrics~%;;; Total ~a~%;;; Width ~a~%;;; Depth ~a~%"
		       (num-to-string acc)
		       (num-to-string maxw)
		       (num-to-string maxd))
	       (when (> maxw width-limit)
		 (format t "count-sexpr width limit exceeded~%"))
	       (when (> maxd depth-limit)
		 (format t "count-sexpr depth limit exceeded~%"))
	       (when (> acc acc-limit)
		 (format t "count-sexpr acc limit exceeded~%")))
	     
	     (mycount (s depth)
	       (setf maxd (max depth maxd))
	       (when (and (not reportp) (> depth depth-limit))
		 (setf reportp t)
		 ;;(report)
		 ;;(break "count-sexpr depth limit exceeded")
		 )
	       (let ((c s)
		     (width 0))


		 (do ()
		     ((not (consp c))
		      (setf maxw (max maxw width))
		      ;;(setf -c c -s s -depth depth -acc acc) (break "fu")
		      )
      
		   (incf acc)
		   (incf width)

		   (when (> width width-limit)
		     (unless reportp
		       (format t "count-sexpr width limit exceeded")
		       ;;(break "wle")
		       )
		     (setf reportp t)
		     ;;(report)
		     )

		   (when (> acc acc-limit)
		     (setf reportp t)
		     ;;(report)
		     (when *process-break* 
		       (break "count-sexpr acc limit exceeded"))
		     )

		   (mycount (car c) (1+ depth))
		   (setf c (cdr c))))))
      (mycount s 0)
      (let ((total acc))
	(when (or reportp (> total report-threshold))
	  (report))
	  ))
    acc))



;; ilist is nil-term or single item or list* of cons terms.
(defun map-sexpr-to-ilist (l nil-term &optional f)
  (let ((list (progn ;;(format t "~%msl ~a " (count-sexpr l))
		     (count-sexpr l)
		     (reverse-flatten-sexpr l))))
    (cond
      ((null list) nil-term)
      ((null (cdr list))
       (if f (funcall f (car list)) (car list)))
      (t (let ((op (operator-of-term nil-term)))
	   (do ((l (cddr list) (cdr l))
		(acc (icons-term-of-op op
				       (if f (funcall f (cadr list)) (cadr list))
				       (if f (funcall f (car list)) (car list)))
		     (icons-term-of-op op (if f (funcall f (car l)) (car l)) acc)))
	       ((null l) acc)))))))


(defun map-isexpr (isexpr op f)
  (labels ((visit (isexpr)
	     (cond
	       ((ilist-nil-p isexpr op))
	       ((not (ilist-cons-p isexpr op))
		(funcall f isexpr))
	       (t (visit (icar isexpr)) (visit (icdr isexpr))))))
    (visit isexpr)))


(defun isexpr-exists-p (isexpr op predicate)
  (let ((p nil))
    (map-isexpr isexpr op #'(lambda (term)
			      (when (funcall predicate term)
				(setf p t))))
    p))
	
(defun isexpr-forall-p (isexpr op predicate)
  (let ((p t))
    (map-isexpr isexpr op #'(lambda (term)
			      (unless (funcall predicate term)
				(setf p nil))))
    p))
	

;; kludge alert
(defun term-of-maybe-itag-term (term) term)

;; returns normal list.
(defun map-isexpr-to-list (isexpr op &optional f)
  (if (ilist-nil-p (term-of-maybe-itag-term isexpr) op)
      nil
      (let ((acc nil))
	(labels ((visit (iisexpr)
		   (let ((isexpr (term-of-maybe-itag-term iisexpr)))
		     (cond
		       ((ilist-nil-p isexpr op))
		       ((not (ilist-cons-p isexpr op))
			(push (if f (funcall f iisexpr) iisexpr) acc))
		       (t (visit (icar isexpr)) (visit (icdr isexpr)))))))
	  (visit isexpr))
	(nreverse acc))))


;; list* ok too.
;; rle todo doc
(defun map-list-to-ilist (list nil-term &optional f)
  (let ((op (operator-of-term nil-term)))
    (labels ((visit (list)
	       (cond
		 ((null list) nil-term)
		 ((consp list) (icons-term-of-op op
						 (if f (funcall f (car list)) (car list))
						 (visit (cdr list))))
		 (t (if f (funcall f list) list)))))

      (visit list))))
  
(defun map-ilist-to-list (ilist op &optional f)
  (let ((acc nil))
    (labels ((visit (ilist)
	       (if (equal-operators-p op (operator-of-term ilist))
		   (let ((bts (bound-terms-of-term ilist)))			  
                     (if bts
			 (cons (if f
				   (funcall f (term-of-bound-term (car bts)))
				   (term-of-bound-term (car bts)))
			       (visit (term-of-bound-term (cadr bts))))
			 acc))

		   (push (if f (funcall f ilist) ilist) acc))))
      (visit ilist))))
  
;; maintains cons structure
(defun map-sexpr-to-isexpr (sexpr nil-term &optional f)
  (let ((op (operator-of-term nil-term)))
    (labels ((visit (sexpr)
	       (cond
		 ((null sexpr) nil-term)
		 ((consp sexpr)
		  (icons-term-of-op op
				    (visit (car sexpr))
				    (visit (cdr sexpr))))
		 (t (if f (funcall f sexpr) sexpr)))))

      (visit sexpr))))


(defun map-isexpr-to-sexpr (isexpr op &optional f)
  (labels ((visit (isexpr)
	     (cond
	       ((ilist-nil-p isexpr op)
		nil)
	       ((ilist-cons-p isexpr op)
		(cons (visit (icar isexpr)) (visit (icdr isexpr))))
	       (t (if f (funcall f isexpr) isexpr)))))

    (visit isexpr)))


(defun map-list-to-isexpr (l nil-term  &optional f)
  (let ((list (reverse l)))
    (cond
      ((null list) nil-term)
      ((null (cdr list))
       (if f (funcall f (car list)) (car list)))
      (t (let ((op (operator-of-term nil-term)))
	   (do ((l (cddr list) (cdr l))
		(acc (icons-term-of-op op
				       (if f (funcall f (cadr list)) (cadr list))
				       (if f (funcall f (car list)) (car list)))
		     (icons-term-of-op op (if f (funcall f (car l)) (car l)) acc)))
	       ((null l) acc)))))))


;;;;
;;;;	Text-Messages-Stamps/Terms.
;;;;





(define-primitive |!text_cons| () (car cdr))

;; rle todo  these should be acceptable on dform lhs! !text_cons too?
(define-primitive |!blank|)
(define-primitive |!newline|)	; degenerate break, vestigial.

(define-primitive |!break|) 
(define-primitive |!break_control| ((token . type)) (text))
;; types expected : linear, break, soft, multilinear, nil. See dtree stuff.

(defvar *itext* '|!text|)

(defvar *itext-parameter-typeids* (list *string-typeid* *token-typeid*))

(defun itext-term-p (term)
  (and (eql *itext* (id-of-term term))
       (let ((parameters (parameters-of-term term)))
	 (and (car parameters)
	      (null (cdr parameters))
	      (and (member (type-id-of-parameter (car parameters))
			   *itext-parameter-typeids*))))))

(defun itext-term (text)
  (cond
    ((symbolp text)
     (instantiate-term
      (instantiate-operator *itext* (list (instantiate-parameter text *token-type*)))))


    ((stringp text)
     (instantiate-term
      (instantiate-operator *itext* (list (instantiate-parameter text *string-type*)))))

    (t (system-error (error-message '(itext))))))


(defun itext-token-term (s)
  (instantiate-term
   (instantiate-operator *itext* (list (instantiate-parameter s *token-type*)))))

(defun itext-string-term (s)
  (instantiate-term
   (instantiate-operator *itext* (list (instantiate-parameter s *string-type*)))))


;;;
;;; generalized itext term : any term whose first parameter is a string or a token.
;;;

(defun text-of-itext-term (term) (value-of-parameter-r (car (parameters-of-term term))))


(defun string-of-text-parameter (p)
  (let ((v (value-of-parameter p)))
    (when (real-parameter-value-p v (type-of-parameter p))
      (string v))))


(defun string-of-itext-term (term)
  (let ((p (car (parameters-of-term term))))
    (if (member (type-id-of-parameter p) *itext-parameter-typeids*)
	(string-of-text-parameter p)
	(raise-error (error-message '(itext-term text not) term)))))


(defun string-of-itext-term-r (term)
  (let ((p (car (parameters-of-term term))))
    (if (eql (type-id-of-parameter p) *string-typeid*)
	(value-of-parameter-r p)
	(if (eql (type-id-of-parameter p) *token-typeid*)
	    (string (value-of-parameter-r p))
	    (raise-error (error-message '(itext-term text not) term))))))

(defun token-of-itext-term (term)
  (let ((p (car (parameters-of-term term))))
    (if (eql (type-id-of-parameter p) *string-typeid*)
	(intern-system (value-of-parameter-r p))
	(value-of-parameter-r p))))
	
;; assumes generalized itext term, but does not require it so useful to check
;; fro real text in first parameter no matter opid or other parameters.
(defun real-itext-term-p (term)
  (real-parameter-p (car (parameters-of-term term))))

(defun real-itext-term-pp (term)
  (and (itext-term-p term)
       (real-parameter-p (car (parameters-of-term term)))))

(define-primitive |!text_term_literal| () (term))

									      
;;; RLE TODO : possible to combine interpretation with term to text
;;; RLE TODO : and not produce this sexpr. Ie scanner operates directly on term.
;;; RLE TODO : might make a good introductory project for terms!

(defun iml-quote-term-p (term)
  (eql (id-of-term term) '|!ml_quote|))


;; temp as should be expanded away by abs.
(defun iml-text-cons-term-p (term)
  (and (eql (id-of-term term) '|!ml_text_cons|)
       (null (parameters-of-term term))
       (null (cddr (bound-terms-of-term term)))
       (forall-p #'(lambda (bt) (null (bindings-of-bound-term bt)))
		 (bound-terms-of-term term))))

(defun iml-text-nil-term-p (term)
  (and (eql (id-of-term term) '|!ml_text_cons|)
       (null (parameters-of-term term))
       (null (bound-terms-of-term term))))


(define-primitive |!force_xref| ((string . string) (token . id) (oid . obid)))

(defun term-to-text (term)

  (labels ((docons (term)
	     (list* (visit (icar term))
		    ;; #\space rle ???
		    (visit (icdr term))))

	   (visit (term)
	     (cond

	       ((or (inil-term-p term)
		    (itext-nil-term-p term)
		    (iml-text-nil-term-p term)
		    )
		nil)

	       ((or (icons-term-p term)
		    (itext-cons-term-p term)
		    (iml-text-cons-term-p term)
		    )
		(docons term))

	       ((iforce-xref-term-p term)
		(force-xref (string-of-iforce-xref-term term)
			    (id-of-iforce-xref-term term)
			    (obid-of-iforce-xref-term term)))
    
	       ((itext-term-p term) (text-of-itext-term term))

	       ((iblank-term-p term) #\space)

	       (;;(or (inewline-term-p term) (ibreak-term-p term)) #\newline)
		(or (inewline-term-p term) (ibreak-term-p term)) #\space)

	       ((ibreak-control-term-p term)
		(break-control (type-of-ibreak-control-term term)
			       (visit (text-of-ibreak-control-term term))))

	       ((itext-term-literal-term-p term)
		(term-of-itext-term-literal-term term))

	       ((iml-quote-term-p term)
		(term-of-itext-term-literal-term term))

	       ((opquoted-term-p term 'scan)
		(erase-quotes term 'scan))

	       ;; literal is unnecessary if we just pass unrecoginzed terms through.
	       ;;(break "unknown term")
	       (t term))))

    (visit term)))

(defun first-text-string (term)
  (cond

   ((itext-nil-term-p term) nil)
    
   ((itext-cons-term-p term);;concat instead of list??lal
    (or (first-text-string (icar term))
	;; #\space rle ???
	(first-text-string (icdr term))))

   ((itext-term-p term) (text-of-itext-term term))

   ((iblank-term-p term) nil)

   ((or (inewline-term-p term) (ibreak-term-p term)) nil)

   ((ibreak-control-term-p term)
    
    (first-text-string ;;(text-of-ibreak-control-term term)
     nil))

   ((itext-term-literal-term-p term)
    (first-text-string (term-of-itext-term-literal-term term)))

   ((iml-quote-term-p term)
    nil)

   ;; LAL
   ((iml-text-nil-term-p term) nil)
   ((iml-text-cons-term-p term)
    (or (first-text-string (icar term))
	(first-text-string (icdr term))))
      
   (t;;(break "unknown term")
    nil));;??
  )

(defun debug-text (text)
  (labels ((visit (e)
	     (unless (null e)
	       (if (eql #\space e)
		   (format t " ")
		   (if (term-p e)
		       (format t "[TERM]")
		       (if (stringp e)
			   (format t e)
			   (if (consp e)
			       (progn
				 (visit (car e))
				 (visit (cdr e)))
			       (progn (setf -e e) (break "debug-text")))))))))
     (visit text))
  text)




(defun term-to-text-string (term)

  (cond

   ((itext-nil-term-p term) nil)
    
   ((itext-cons-term-p term) ;;concat instead of list??lal
    (concatenate 'string (term-to-text-string (icar term))
	   ;; #\space rle ???
	   (term-to-text-string (icdr term))))

   ((itext-term-p term) (text-of-itext-term term))

   ((iblank-term-p term) #\space)

   (;;(or (inewline-term-p term) (ibreak-term-p term)) #\newline)
    (or (inewline-term-p term) (ibreak-term-p term)) #\space)

   ((ibreak-control-term-p term)
    (break-control (type-of-ibreak-control-term term)
		   (term-to-text (text-of-ibreak-control-term term))))

   ((itext-term-literal-term-p term)
    (term-of-itext-term-literal-term term))

   ((iml-quote-term-p term)
    (term-of-itext-term-literal-term term))

    ;; LAL
   ((iml-text-nil-term-p term) nil)
   ((iml-text-cons-term-p term)
    (concatenate 'string (term-to-text-string (icar term))
	   ;; #\space rle ???
	   (term-to-text-string (icdr term))))
      
   (t term)));;??

;; need inverse of  string-to-standard-character-string

(defun standard-character-string-to-string (sstring)
  #+16bit-chars sstring
  #-16bit-chars (with-string-scanner (sstring)
		  (with-output-to-string (news)
		    (do ()
			((scan-eof-p))
		      (if (scan-escape-p)
			  (let ((digit1 (scan-cur-byte))
				(digit2 (unless (scan-eof-p) (scan-next nil) (scan-cur-byte)))
				(digit3 (unless (scan-eof-p) (scan-next nil) (scan-cur-byte)))
				(digit4 (unless (scan-eof-p) (scan-next nil) (scan-cur-byte))))
			    (if (and (hex-code-p digit1)
				     digit2 (hex-code-p digit2)
				     digit3 (hex-code-p digit3)
				     digit4 (hex-code-p digit4))
				(write-char (code-char (+ (hex-code-to-int digit4)
							  (ash (hex-code-to-int digit3) 4)
							  (ash (hex-code-to-int digit2) 8)
							  (ash (hex-code-to-int digit1) 12)))
					    news)
				(progn
				  (write-char #\\ news)
				  (when (write-char (code-char digit1) news))
				  (when digit2 (write-char (code-char digit2) news))
				  (when digit3 (write-char (code-char digit3) news))
				  (when digit4 (write-char (code-char digit4) news)))))
			  (write-char (scan-cur-char) news))
		      (scan-next nil)))))



(defun text-to-string (text)
  (labels
      ((visit (text s)
	 (cond
	   ((null text))
	   ((consp text)
	    (visit (car text) s)
	    (visit (cdr text) s))
	   ((stringp text)
	    (unless (zerop (length text))
	      (write-string (standard-character-string-to-string text) s)))
           ((or (eql #\newline text)
		(eql #\space text))
	    (write-string " " s))
	   ((symbolp text)
	    (let ((ts (string text)))
	      (unless (zerop (length ts))
		(write-string (string (standard-character-string-to-string ts)) s))))
	   ((break-control-p text)
	    nil)
	   (t (break "text-to-string")))))

  (with-output-to-string (s)
    (visit text s))))
    


(define-primitive |!cut_break|)

(defun text-to-term (text)

  (cond

    ((null text) (itext-nil-term))

    ((consp text)
     (cond
       ((null (car text))
	(text-to-term (cdr text)))
       ((null (cdr text))
	(text-to-term (car text)))
       (t (itext-cons-term (text-to-term (car text)) (text-to-term (cdr text))))))

    ((eql #\space text) (iblank-term))

    ((eql #\newline text) (ibreak-term))

    ((eql #\return text) (icut-break-term))

    ((integerp text)
     (if (>= text 0)
	 (inatural-term text)
	 (itext-string-term (princ-to-string text))))

    ((stringp text) (itext-string-term text))

    ((symbolp text)
     (if (eql text 'cut-break)
	 (icut-break-term)
	 (itext-token-term text)))

    ;;; RLE ??? Not sure if this should be literal or not
    ;;; assume no so anyone injecting terms into text must literalize them if necessary.
    ((term-p text)
     ;; force break
     (if (term-op-count-exceeds-p text 16)
	 (itext-cons-term  (ibreak-term) text)
	 text))
  
    ((break-control-p text)
     (ibreak-control-term (type-of-break-control text) (text-of-break-control text)))

    (t (let ((ss (princ-to-string text)))
	 (let ((s (if (> (length ss) 256)
		      (subseq ss 0 256)
		      ss)))
	   (message-emit (warn-message '(term text) (list s)))
	   (itext-term s))))))



;;;;
;;;;	term collection :
;;;;

;;; places to look for terms:
;;;
;;;	Definition (caches export term)
;;;	Object-address (caches term version)
;;;	Dependency (caches term version)
;;;	


;; maybe should use a termsig table.
;; maybe a new kind of table based on term-sig-tree
;; might want to try hashing based on term-sig tree.
;; easier to compute and probably not that many clashes.
;; every !text{s}() will be in one big list. So would
;; need adhoc special cases?
;; on second thought maybe normal hash table is best.
(defvar *collect-permanents* (make-hash-table))
(defvar *collect-maps* nil)

(defun collect-terms-permanent (term)
  (let* ((hash (hash-term term))
	 (terms (gethash hash *collect-permanents*)))
    (if terms
	(unless (member term terms :test #'compare-terms-p)
	  (setf (gethash hash *collect-permanents*) (cons term terms)))
	(setf (gethash hash *collect-permanents*) (list term)))))


(defun collect-terms-map (tag closure)
  (setf *collect-maps*
	(acons tag closure *collect-maps*)))


(defun term-reader (file-spec call-back)
  (with-handle-error (()
		      (raise-error (flush-message '(error term-reader file) file-spec)))

    (with-prl-open-file (stream file-spec in t)
      (do ((term (prl-stream-read stream) (prl-stream-read stream)))
	  ((null term) (values))
	(funcall call-back (sexpr-to-term term))))))


;; (filename  f)
;; f(<closure{g}>) : nil
;; g()  : nil | term
;; once f returns file is closed, and calling g will result in failure.
;; once g has returned nil calling g will result in failure.
(defun term-reader-lazy (file-spec f)
  (with-prl-open-file (stream file-spec in t)
    (funcall f
	     ;; g
	     #'(lambda ()
		   (let ((sexpr (prl-stream-read stream)))
		     (when sexpr
		       (sexpr-to-term sexpr)))))))
			 
;; term reader could be implement with term-reader-lazy:
;; (defun term-reader-alt (file-spec call-back)
;;  (term-reader-lazy file-spec
;;		    #'(lambda (reader)
;;			(do ((term (funcall reader) (funcall reader)))
;;			    ((null term))
;;			  (funcall call-back term)))))
			     
				     

(defun term-writer (file-spec call-back)
  (with-prl-open-file (stream file-spec out t)
    (do ((term (funcall call-back) (funcall call-back)))
	((null term) (values))
      (prl-stream-write (term-to-sexpr term) stream))))


(defun term-writer-lazy (file-spec f)
  (with-prl-open-file (stream file-spec out t nil)
    (funcall f
	     #'(lambda (term)
		 (if (null term)
		     (prl-stream-close stream)
		     (prl-stream-write (term-to-sexpr term) stream))))))


(defun term-write (stream term)
  (write (term-to-sexpr term)
	 :stream stream :pretty nil :level nil :length nil)
  (terpri stream)
  (terpri stream))



;;;;	
;;;;	term-walk-d (<term> <closure{pred}> <closure{permute}>)	: <term>
;;;;	  * pred (<term>)				: BOOL
;;;;	  * permute (<term> <closure{continue}>)	: <term>
;;;;	  ** continue (<term>)				: <term> 
;;;;	
  
(defun term-walk-d (term p f)
  (labels ((visit-term (term)

	     ;;(setf c4 term pp p) (break "twd")
	     (if (funcall p term)

		 (funcall f term
			  #'(lambda (term)
			      (visit-term term)))

		 (maybe-instantiate-term term
					 (operator-of-term term)
					 (mapcar #'(lambda (bt)
						     (maybe-instantiate-bound-term
						      bt
						      (bindings-of-bound-term bt)
						      (visit-term (term-of-bound-term bt))))
						 (bound-terms-of-term term)))
		 )))

    (visit-term term)))

;; this appears to be equivalent to term-walk below? no since this constructs
;; new term.
(defun term-walk-ops (term f)
  (term-walk-d term
	       #'(lambda (term) (declare (ignore term)) t)
	       #'(lambda (term contf)
		   (let ((contt (maybe-instantiate-term term
							(operator-of-term term)
							(mapcar #'(lambda (bt)
								    (maybe-instantiate-bound-term
								     bt
								     (bindings-of-bound-term bt)
								     (funcall contf (term-of-bound-term bt))))
								(bound-terms-of-term term)))))
		     (funcall f contt)))))




;; while structures match pass both. when differ pass a but pass nil for b.
;; test-pred only on term not term-b or vice-versa.
(defun terms-walk-d (term term-b p f)
  (labels ((visit-term (term term-b)

	     ;;(setf c4 term pp p) (break "twd")
	     (if (funcall p term term-b)

		 (funcall f term term-b
			  #'(lambda (term term-b)
			      (visit-term term term-b)))

		 (if (or (null term-b)
			 (not (= (length (bound-terms-of-term term))
				 (length (bound-terms-of-term term-b)))))
		     (maybe-instantiate-term term
					     (operator-of-term term)
					     (mapcar #'(lambda (bt)
							 (maybe-instantiate-bound-term
							  bt
							  (bindings-of-bound-term bt)
							  (visit-term (term-of-bound-term bt) nil)))
						     (bound-terms-of-term term)))
		     (maybe-instantiate-term term
					     (operator-of-term term)
					     (mapcar #'(lambda (bt btb)
							 (maybe-instantiate-bound-term
							  bt
							  (bindings-of-bound-term bt)
							  (visit-term (term-of-bound-term bt)
								      (term-of-bound-term btb))))
						     (bound-terms-of-term term)
						     (bound-terms-of-term term-b))))
		 )))

    (visit-term term term-b)))

;; applies f (term continuation) to all subterms for which p is true.
;; curtails walk when p is true.
(defun term-walk (term p &optional f)
  (labels ((visit-term (term)
	     (if (funcall p term)
		 (when f 
		   (funcall f term #'visit-term))
		 (mapc #'(lambda (bt)
			   (visit-term (term-of-bound-term bt)))
		       (bound-terms-of-term term)))))
    (visit-term term)))

(defun term-walk-p (term p)
  (let ((found nil))
    (term-walk term #'(lambda (tt) (when (funcall p tt) (setf found t))))

    found))


;;;;
;;;;
;;;;
;;;;

(defvar *heritage-term-embedding-opid* '|!!uGh|)

(defun leaves-of-term (term)
  (let ((id (id-of-term term)))
    (if (eql id *heritage-term-embedding-opid*)
	(parameters-of-term term)
	(or (mark-value term 'leaves)
	    (let ((l;; TODO later make more efficient by adding leaves to term structure (but not cons struct).
		   (cons (token-parameter id)
			 (parameters-of-term term))))
	      (mark term 'leaves l)
	      l)))))



(defun term-replace (term p f)
  (labels ((visit-term (term)
	     (if (funcall p term)
		 (funcall f term)
		 (maybe-instantiate-term term
					 (operator-of-term term)
					 (mapcar #'(lambda (bt)
						     (maybe-instantiate-bound-term
						      bt
						      (bindings-of-bound-term bt)
						      (visit-term (term-of-bound-term bt))))
						 (bound-terms-of-term term))) )))

    (visit-term term)))


(defun term-find (p term)
  (labels ((visit-term (term)
	     (if (funcall p term)
		 term
		 (find-first-optimized (bt (bound-terms-of-term term))
				       (visit-term (term-of-bound-term bt))))))

    (visit-term term)))



(defun term-replace-wcont (term p f)
  (labels ((visit-term (term)
	     (if (funcall p term)
		 (funcall f term #'visit-term)
		 (maybe-instantiate-term term
					 (operator-of-term term)
					 (mapcar #'(lambda (bt)
						     (maybe-instantiate-bound-term
						      bt
						      (bindings-of-bound-term bt)
						      (visit-term (term-of-bound-term bt))))
						 (bound-terms-of-term term))) )))

    (visit-term term)))

#|
(defunml (|term_to_text_string| (term))
     (term -> string)
    (term-to-text-string term))
|#



(defvar *max-print-depth* 10)

(defun max-print-depth ()
  (or *max-print-depth*
      (when *print-level*
	(* 2 *print-level*))))

(defun print-term (term &optional (depth 0) (stream t))

  (labels ((print-bound-term (bound-term tabs)
	     (when (bindings-of-bound-term-n bound-term)
	       (format stream "~a" (parameter-value-to-pretty-string (car (bindings-of-bound-term-n bound-term))
								     *variable-type*))
	       (dolist (b (cdr (bindings-of-bound-term-n bound-term)))
		 (format stream ",~a" (parameter-value-to-pretty-string b *variable-type*)))
	       (format stream "."))
	     (stringize (term-of-bound-term bound-term) tabs))
	     
	   (stringize (term tabs)
	     (setf tabs (concatenate 'string tabs " "))
	     (let ((tabs+ (concatenate 'string tabs " ")))
	       (cond
		 ((variable-p term)
		  (format stream "~a" (parameter-value-to-pretty-string (id-of-variable-term term)
									*variable-type*)))
		 ((and (itext-term-p term)
		       (let ((p (car (parameters-of-term term))))
			 (real-parameter-value-p (value-of-parameter-n p) (type-of-parameter p))))
		  (format stream "~a" (text-of-itext-term term)))
		 ((or (ibreak-term-p term) (icut-break-term-p term))
		  (format stream "~%"))
		 ((itext-cons-term-p term)
		  (stringize (icar term) tabs)
		  (stringize (icdr term) tabs))
		 (t (if (< depth (max-print-depth))
		       (progn
			 (incf depth)
			 (format stream "~a" (operator-to-pretty-string (operator-of-term term)))
			 (when (bound-terms-of-term term)
			   (format stream "~a(" tabs)
			   (print-bound-term (car (bound-terms-of-term term)) tabs+)
			   (dolist (bt (cdr (bound-terms-of-term term)))
			     (format stream tabs+)
			     (print-bound-term bt tabs+))
			   (format stream ")"))
			 (decf depth))
		       (format stream "...")))))))

    (stringize term (format-string "~%~a" (blank-string depth)))
    (values)))


(setf *term-to-pretty-string-f*
      #'(lambda (term) (with-output-to-string (stream) (print-term term 0 stream))))


