
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;;	What belongs in com and what belongs in sub modules ?
;;;;	Com supports the working map exported by the lib. 
;;;;	So anything needed to implement WM resides in com. extensions 
;;;;	go in sub-modules.
;;;;	
;;;;	cert question
;;;;	 if tactic def refers to lemma via certification
;;;;	 then modification of lemma means recertifying -> recompile all code calling
;;;;	 -> rerun all steps using any tactic recompiled, all proofs whose infs modified.
;;;;	 partial answer is that tactic def should be passed lemmas as args thereby
;;;;	 lessening tactic code dependency on lemmas.
;;;;	 
;;;; -docs- (mod com)
;;;;
;;;;	Oid : 
;;;;	
;;;;	<oid> : <stamp>
;;;;
;;;;	oid-parameter(<oid>)	: <parameter>
;;;;	oid-parameter-p(<oid>)	: <bool>
;;;;
;;;;	oid-to-term (<oid>)	: <term>
;;;;	term-to-oid (<term>)	: <oid>
;;;;	
;;;;
;;;; -doct- (mod com)
;;;;	Object Id/Message : often messages are about a particular object
;;;;		or objects. Oid-messages include a list of object-ids.
;;;;
;;;;
;;;;	<oid-message>	: oid-message[<message> <oid> list]
;;;;
;;;;	oid-message (<oid> list <tag> list &optional <pmsg>)
;;;;	  : <oid-message>
;;;;	oid-error-message (<oid> list <tag> list &optional <pmsg>)
;;;;	  : <oid-message>
;;;;	oid-warn-message (<oid> list <tag> list &optional <pmsg>)
;;;;	  : <oid-message>
;;;;
;;;;  -page-
;;;;
;;;;	Messages-Stamps/Terms.
;;;;
;;;;	<stamp-term>	: !stamp{<sequence>:n, <time>:time, <transaction>:n,
;;;;				 <process-id>:t}
;;;;
;;;;	<ioids-term>	: !oid{<oid>:o list}()
;;;;	
;;;;	<message-term>	: !message{<tags>}(<ioids-term>;
;;;;					   <stamp-term>;
;;;;					   <pmsg-term>)
;;;;	    
;;;;	<tags>		: <id>:t | <id>:t, <tags>
;;;;
;;;;	<pmsg-term>	: <text-term>
;;;;			| <message-term>
;;;;			| !break_control{<type>:t}(<pmsg-term>)
;;;;			| !text_cons(<pmsg-term>; <pmsg-term>)
;;;;
;;;; -doct- (mod com)
;;;;
;;;;	pmsg-to-term (<pmsg>)		: <term>
;;;;	message-to-term (<message>)	: <term>
;;;;
;;;;  -page-
;;;;
;;;;	Description: describes an environment, object or event.
;;;;	
;;;;	Descriptions are used to :
;;;;	  - test applicability.
;;;;	  - provide information.
;;;;
;;;;	The general framework for description data should satisfy most uses.
;;;;	However, at the moment, the description syntax should be viewed as
;;;;	ad hoc.
;;;;
;;;;	Current implementation :
;;;;
;;;;	A term syntax is presented, the terms are usually converterd
;;;;	to an internal representation.
;;;;
;;;;	<description>	: !description{<system>:t}
;;;;				( <version>
;;;;				; <purposes> )
;;;;
;;;;	<version>	: !natural{<v>:n} !cons list
;;;;
;;;;	<purposes>	: !token{<purpose>:t} !cons list
;;;;
;;;;
;;;;	Match function used to test applicability :
;;;;
;;;;	match-descriptions-p (<description> <description>)	: BOOL
;;;;	match-descriptions-p (<description> <term{description>)	: BOOL
;;;;
;;;;
;;;;	Possible Extension is to divided attributes into three classes :
;;;;	  - require	: required attributes must match.
;;;;	  - use		: ??? some middle ground.
;;;;	  - info	: informational attributes.
;;;;
;;;;	In this scheme the current purpose and version become required
;;;;	attributes.
;;;;
;;;;	<description>	: !description{<system>:t}
;;;;				( <require> !cons list
;;;;				; <use> !cons list
;;;;				; <inform> !cons list)
;;;;
;;;;
;;;;	Object descriptions :
;;;;
;;;;	Object descriptions are included in broadcasts associated with object so
;;;;	that receivers can judge applicability.
;;;;
;;;;	Environment descriptions :
;;;;
;;;;	Environments can export their descriptions to prevent tranmission of
;;;;	inapplicable broadcasts.
;;;;
;;;;	Some possible attributes :
;;;;	
;;;;	<require>	: !system{<name>}
;;;;			| !version([!natural | !variable({<v>:v)] !cons list) 
;;;;			| !purpose{<type>:t}		EDIT REFINE LIBRARY ANY
;;;;			| !language{<lang>:t}
;;;;	
;;;;	<use>		: ???
;;;;
;;;;	<inform>	: ???
;;;;
;;;;	As evidenced by the use of !varible in the version attribute, meta
;;;;	values to the match function are desirable. Also rather than a straight
;;;;	forward !cons list of attributes expression operators such as !and, !or
;;;;	and !not could be allowed.
;;;;
;;;;
;;;;	Launch :
;;;;
;;;;	Another use of description might be as guide to launching a new process
;;;;	or environment. In this use the aforemention attribute classes can be
;;;;	viewed differently :
;;;;
;;;;	  - REQUIRE : interpreter can not be changed dynamically to
;;;;		      satisfy, thus must be present a-priori.
;;;;		      eg. system and version.
;;;;	  - USE : can be dynamically configured,
;;;;		eg. load/activate or someother type of plug-in.
;;;;	  - INFORM : ???
;;;;
;;;;
;;;;  -page-
;;;;
;;;;	Dependencies: references made during an event. An event may span
;;;;	  multiple environments and may involve various types of references.
;;;;	  The dependencies are grouped such that the event is the largest
;;;;	  container, then the environment then the type.
;;;;
;;;;	  An environment may call another, the called will return set of
;;;;	  enviroment dependencies which the caller must assimilate.
;;;;
;;;;	  If an environment calls back a caller then the environment
;;;;	  called back need not report dependencies but can accumulate the
;;;;	  dependencies itself.
;;;;
;;;;	  EG, a calls b which calls c which returns dependencies Dc to b,
;;;;	  then b calls back a, a returns no dependencies, and then b returns
;;;;	  to a with dependencies Dc, and Db. A then assimilates these to 
;;;;	  the dependencies it accumulated prior to the call to b, during the
;;;;	  callback from b and after the return from b resulting in
;;;;	  dependencies Da, Db, Dc. A could have returned dependencies Da' to 
;;;;	  b and then b would have returned Da', Db, Dc which a would have
;;;;	  assimilated to Da, Db, Dc.
;;;;
;;;;
;;;;	We require ability to express dependency dag fragments in objects:
;;;;
;;;;	<dependency>		: dependency[<stamp{data}>, <stamp{oc}, <oid>]
;;;;	  * null oid ok.
;;;;
;;;; 	<dependencies>		: (<tok{tag}> . <dependency> sexpr)
;;;;	  * allow sexpr here to avoid copying during accumulation.
;;;;
;;;; 	<env-dependencies>	: env-dependencies[<stamp{env}> <dependencies> list)
;;;;
;;;; 	<evt-dependencies>	: evt-dependencies
;;;;						[ <stamp{event}>
;;;;						  <description{event}>
;;;;						  <env-dependencies> list ]
;;;;
;;;; 	<dependency-store>	: dependency-store[evt-dependencies list]
;;;;	  * (dependency-store[nil]) is used when there are no dependencies.
;;;;	  * nil is used when the dependency field is uninitialized.
;;;;	
;;;; 	<event>			: event[<stamp{event}> <description> <ddag-fragment>]
;;;;
;;;;	<ddag-link>		: ddag-link[<stamp{referencer}> <evt-dependencies>]
;;;;
;;;;	<ddag-fragment>		: <ddag-link> list
;;;;	
;;;;	  * event dependencies in an object should not be duplicated in the event.
;;;;	    In fact, if an event is contained in a dependency store than its ddag-fragment
;;;;	    should be nil.
;;;;	  * an event should generate a broadcast to be logged to make assoc between stamp
;;;;	    and event persistent if event is not contained in an object.
;;;;
;;;; -doct- (mod com data)
;;;;
;;;;	<dependency-term>	: !dependency{<oid>:o}(<stamp{oc}>; <stamp{data}>)
;;;;
;;;;	<dependency-term-list>	: <dependency-term> !dependency_cons ilist
;;;;
;;;;	<dependencies-term>	: !dependencies{<tag>:t}(<dependency-term-list>)
;;;;
;;;;	<dependencies-list>	: <dependencies-term> !dependencies_cons ilist
;;;;
;;;;	<env-depend-term>	: !environment_dependencies( <istamp{env}>
;;;;							   ;  <dependencies-list>)
;;;;	  * stamp is term as environment stores as term.
;;;;
;;;;	<env-depend-list>	: <env-depend-term> !environment_dependencies_cons ilist
;;;;
;;;;	<evt-depend-term>	: !event_dependencies( <stamp{env}>
;;;;						     ; <description>
;;;;						     ; <env-depend-list>)
;;;;
;;;;	<dep-store-term>	: !dependency_store(<evt-depend-term> !event_dependencies_cons ilist)
;;;;
;;;;	  ** !void() signifies dependencies not determined.
;;;;	  ** !dependency_store() signifies no dependencies.
;;;;
;;;; -doct- (mod com)
;;;;
;;;;	equal-dependencies-p(<dependency> <dependency>)		: <bool>
;;;;	  * only data stamps compared.
;;;;
;;;;	equal-dependency-stores-p(<dependency-store> <dependency-store>)
;;;;	  : BOOL
;;;;
;;;;	dependency-store-to-term(<dependency-store>)		: <term>
;;;;	term-to-dependency-store(<term>)			: <dependency-store>
;;;;
;;;;
;;;;	dependency-to-string(<dependency>)			: <string>
;;;;	  * for display only, not parseable.
;;;;
;;;;
;;;;	Dependencies are emitted at time of reference. Accumulators
;;;;	may be established to catch and collect emissions. 
;;;;	
;;;;	Event:
;;;;	
;;;;	with-dependency-event (&body)	: MACRO
;;;;	  * <desc>	: <description> | <tok{purpose}>
;;;;	    if tok supplied then environment description used with tok prepended to purposes.
;;;;	  * collects <env-dependencies>.
;;;;	
;;;;	dependency-note-environment (<env-dependencies>)	: NULL
;;;;	
;;;;	event-dependencies-collected(<desc> <stamp>)		: <event-dependency>
;;;;	  
;;;;
;;;;
;;;;	Evironment:
;;;;
;;;;	with-dependency-environment (&body)	: MACRO
;;;;	  * collects dependencies.
;;;;
;;;;	dependency-note-dependencies(<dependencies> list)	: NULL
;;;;
;;;;	environment-dependencies-collected (<term{stamp}>)	: <env-dependencies>
;;;;	  * stamp of environment.
;;;;
;;;;	Dependencies :
;;;;
;;;;	with-dependencies(&body)		: MACRO
;;;;	 * upon leaving scope normally or abnormally,
;;;;	   adds lingering references to outer scope,
;;;;	   Ie, references made on failing paths are collected.
;;;;	   Outer scope may be dependency environment, ie 
;;;;	   dependency-note-dependencies is called implicity when
;;;;	   leaving scope if pending dependencies.
;;;;
;;;;	dependency-note-reference(<tag{type}> <dependency>)	: NULL
;;;;	
;;;;	dependencies()				: <dependencies> list
;;;;	  * explicit flush of nearest dependencies accumulator.
;;;;	
;;;;	
;;;;	5/2001  dependency substance detect
;;;;	
;;;;	compiling code
;;;;	  - updates global ml state.
;;;;	  - uses global ml state.
;;;;	  - relation between objects compiled
;;;;	    and objects defining referenced ids.
;;;;	      * need to capture version of object
;;;;		  - 
;;;;		
;;;;	  - caches id cross reference.
;;;;	
;;;;	
;;;;	event : a record of an action.
;;;;	  - stamp for equality check but not inequality.
;;;;	  - events form a graph by pointing to each other.
;;;;	  - may be subtyped to record other dynamic data 
;;;;	    or to summarize static data.
;;;;	
;;;;	examples:
;;;;	  - reduce : should event contain reduced term or should object.
;;;;	     - compile
;;;;	     - refine 
;;;;	  - translate/activate
;;;;	
;;;;	events can be overwitten (or written again) so as to squelch a refresh.
;;;;	eg consider modify source of object but reduce reduces to identical term.
;;;;	since source modified reduce event is stale as no longer points to proper source.
;;;;	when reduced if resulting event is identical (besides source pointer) then 
;;;;	overwrite using old reduce event stamp.
;;;;	
;;;;	
;;;;	
;;;;	Assume at compile time, the object and version defining an id in the
;;;;	global state is discernable.
;;;;	
;;;;	want a method to "certify" `id a in object p calls id b in object q`
;;;;	 - at compile-time of p, b was defined by q.
;;;;	 - current version of q = compile-time version of p.
;;;;	 - current version of p = compile-time version of q
;;;;	 --> compile-time of q precedes that of p
;;;;	 
;;;;	consequently need to be able to distinquish versions.
;;;;	at compile time need to notate versions of compiled object and referenced objects.
;;;;
;;;;	When a modified object is compiled, referencing objects are now in question.
;;;;	Stuart's suggestion is to lose all certifications depending on modified object,
;;;;	and then rebuild certifications. A certification is an object relating other
;;;;	objects. In this case the relation could be the ml id dependency between objects.
;;;;	Ie when compiling an object produce a certification relating the object to all
;;;;	referenced objects.
;;;;	
;;;;	To approximate this, we can record the ml id cross reference and the
;;;;	compile-time versions of the related objects. Rather than place this
;;;;	in a separate object it will be stored as part of the translation.
;;;;	We will refer to this as a certificate. 
;;;;	
;;;;	Version : the certificate of the object defines the version. The certificate
;;;;	 will have a stamp. Stamps are not compared but instead used only for identity.
;;;;	 Compiling an object defines a new certificate thereby replacing the old.
;;;;	 
;;;;	Consider, compiling object q from above: 
;;;;	  - q gets new certificate which certifies that object q defines id b.
;;;;	  * object p can see that the version of q has changed
;;;;
;;;;	Consequently, each ml id must know the object and the certificate which
;;;;	defines it, so that it can be reported when referenced.
;;;;	
;;;;	
;;;;	<event-base>	: <stamp{event-id}> # object_id # term{description}
;;;;			  # ((<object_id> # <stamp{tmnt}>) | <tmnt>) list
;;;;			| <stamp> # <tmnt> list

;;;;	
;;;;	<tmnt-dog>	: <tmnt-base> # <dog>
;;;;
;;;;	<tmnt-compile>	: <tmnt-dog> # <xref>
;;;;	
;;;;	<tmnt-refine>	: <tmnt-dog>
;;;;	
;;;;	<tmnt-static>	: static obid instances in an object.
;;;;	
;;;;	<tmnt-abs>	: obids of abstraction instances in an object.
;;;;	
;;;;	
;;;;	From dependency info and xref recorded we should be able to derive tmnt
;;;;	at translation if xref returns oid#tmnt using objc stamp from dependency might work to.
;;;;	Later might want to more directly gather tmnt data.
;;;;	
;;;;	tmnt-base defines an order for testaments. Still allows cycles for 
;;;;	 t0 : b, t1 : a -> b{t0}, t2 : b -> a{t1}, but t0 is somehow know to be bogus.
;;;;	  t0 is bogus since there exists a newer testament for obid b ie t2
;;;;	
;;;;	resides ???
;;;;	  - likely that refresh will occur such that new testament is produced without
;;;;	    modifying the object itself in any way. (especially if dog and xref are in tmnt.
;;;;	      * xref changes if ref'd id moves from one object to another.
;;;;	
;;;;	testament log : 
;;;;	  - multiple terms per testament and subtypes 
;;;;	      * allow for skipping subtypes at read.
;;;;	  - queue write requests and only write when idle.
;;;;	    do same for other logs.
;;;;	      * inline persist data refs if in queue or unconditionally.
;;;;	  - narrow optimizations such as object-id compression and possibly token (xref?).
;;;;	    obid could be separate file to allow skipping of terms.
;;;;	  - if testament log includes stactic refs then might be able to use for
;;;;	    for object_id closures without reading object.
;;;;
;;;;	currently xref is in src. can leave or mv to tmnt.
;;;;	maybe tmnt should be in src, no want distinct so recompile does not modify object.
;;;;	
;;;;	event is relation between objects|object_id/events/stamped data
;;;;	
;;;;	
;;;;	<event-base>	: <stamp{event-id> # term{description}
;;;;                      # (<object_id> | <stamp{event}>) 
;;;;			  # ((<object_id> # <stamp{event}>) | <event>) list
;;;;	   !!! the event-id can be syntactically an object-id, but will not be in lib-table.
;;;;	      but rather an event table. then syntactically
;;;;	
;;;;	src-reduce	: event-id # SRC-REDUCE 
;;;;			  # oid,event{source} # oid,event{substance} list{ abs}
;;;;	   !! events may shadow events ie a src reduce will shadow an earlier src-reduce for same oid.		  
;;;;	      in fact, event id should be recorded in event but computed at event(new) time not saved in obj.
;;;;	      
;;;;	
;;;;	
;;;;	<dog>	: directed object_id graph.
;;;;		  (<token{kind}> # (<object_id> list)) list
;;;;	
;;;;	commit-event  : COMMIT, <last-commit-event>, (object_id ?# event{substance}?list)
;;;;	  - relation between events defining ordering of events.
;;;;	  - with some care can probably use event stamps and commit-event 
;;;;	    order to order all events. ie, the commit event, either explicitly(better) or implicitly,
;;;;	    references other events that took place "in" the commit event. 
;;;;	  ? says what about referenced objects.
;;;;	  ? substance event probably not worthwhile. Need to be able to detect if
;;;;	    binding of substance to oid changed not just substance change.
;;;;	
;;;;	!event_ref{o}() implies 
;;;;	
;;;;	
;;;;	substance-event might say this substance{stamp} bound to this obid
;;;;	 by this event?
;;;;	
;;;;	squelch : how to take advantage of noticing that an event did not
;;;;	change relation. Maybe event equivalence classes!.
;;;;	
;;;;	gc : events can be collected when they are not referenced and they reference nothing???
;;;;	     an event reference to an object_id does not preclude gc of object_id??.
;;;;	     events can be referenced by lib-log, ie verry similar to object_id.
;;;;	
;;;;	
;;;;	<dependency>		: dependency[<stamp{data}>, <stamp{oc}>, <oid>]
;;;;	  - desire oc stamp to detect if oc changed since dep made.
;;;;	  - similar with data stamp
;;;;	
;;;;	
;;;;	consider a depends on b.
;;;;	 b is unsubstantively modified
;;;;	 b is substantively modified
;;;;	 b is rebound to objc o
;;;;	  need to be able to detect these
;;;;	
;;;;	if objc stamp differs than we know different objc
;;;;	
;;;;	touching objc in any way modifies data stamp.
;;;;	touching objc in substantive way modifies oc stamp
;;;;	rebinding objc change oc stamp.
;;;;	
;;;;	so can compare oc stamp of oid of dependency with oc-stamp of dependency
;;;;	 to detect substantive modification or 
;;;;	
;;;;	desire dep to be an obid.
;;;;
;;;;	
;;;;	
;;;;	<dstore> : <stamp{translation?}> . ((<token> . <obid> list) list)
;;;;	<objc> : (<oid> . <stamp{bind}> . <stamp{translation}>){dep} list . <data> . <substance>
;;;;	  - at bind add dep unless dep is present with same oid and translation stamp.
;;;;	    at unbind add dep
;;;;	    unbind then bind within same transaction  with same oid and substance should not alter bind stamp.
;;;;	    otherwise bind stamp updated.
;;;;	  at unbind destructively note unbind and transaction then at bind check if not there then update.??
;;;;	    indicates which substance was present at last bind of objc to oid.
;;;;	    indicates 
;;;;	  - bind stamp set when substance stamp differs from last bound to oid ??
;;;;	 
;;;;	then stale if bind stamp or substance stamp is later than store stamp
;;;;	 
;;;;	At the time a reference to an oid was made was made was the same substance bound to that oid.
;;;;	  - is substance older than reference?
;;;;	  - is substance binding older than reference.
;;;;	
;;;;	
;;;;	?import sets bind stamp? 
;;;;	
;;;;	two stamps of same process can be ordered?
;;;;	  - simultaneous transactions, dependent sees other as at start of its transaction.
;;;;	    so can-not order
;;;;	      * objc stamp in dependency
;;;;	      * detect simultaneous transactions and ?tag? dependent if dependedupon involved
;;;;	      * certificate/intermediary
;;;;	 * a stamp for each commit sequence, then certificate contains latest such stamp 
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	
;;;;  -page- hard
;;;;
;;;;	Threads/Transactions : A thread can be used to process only a single
;;;;	  transaction.  A transaction may employ more than one thread. A tid is
;;;;	  a tag which identifies the transaction and all threads assisting. Much
;;;;	  of this is moot at the moment since we are not using multi-tasking.
;;;;	
;;;;	When a remote request is made the tid is included in the request so that
;;;;	the the remote evaluator may call back and the callback can be evaluated
;;;;	in the originating thread/transaction. It is desirous to use same
;;;;	transaction so as to inherit locks.
;;;;	
;;;;	A remote process may send a request with a tid so that the local process
;;;;	can callback the remote process. In order to avoid relying on remote
;;;;	process correctly implementing uniquness of ids we may map incoming tids
;;;;	to local tids being careful to reverse map on callback.
;;;;
;;;;	If transaction begins in scope of tid call then transaction inherits tid.
;;;;	If request sent in scope of null tid then tid derived from transaction-id.
;;;;
;;;;	INVARIANT: all tids generated by any process must be unique.
;;;;	
;;;;	
;;;;	<tid>		: (().!transaction_id{<tseq>:n; <pid>:t} . nil)
;;;;	  * ie an singleton bound-term list.
;;;;
;;;;	
;;;;	A transaction contains 
;;;;	  - collect-queue : a queue of transaction table entrys needing to be gc'ed.
;;;;	    a transaction may cause data in a table to become inaccessible. Such 
;;;;	    inacessable data should be removed. The collect-queue is a list of table
;;;;	    entries which may contain inaccessable data. See Transaction Tables.
;;;;	  - completions : pending completions of transaction.
;;;;	  - stamp : begin stamp of transaction.
;;;;	  - tid
;;;;	
;;;;
;;;;  -page- hard
;;;;
;;;;	ML :
;;;;	
;;;;	dependency	: Primitive abstract type.
;;;;	
;;;;	
;;;;	
;;;; -doct- (mod com ml)
;;;;
;;;; -doct- (mod com)
;;;;  -page- hard
;;;; RLE milled to here.
;;;;
;;;;	Definitions and Definition Tables.
;;;;
;;;; 	A definition table holds a collection of definitions.
;;;;	Definitions are grouped by term-basis.
;;;;
;;;;
;;;;	<definition>		: definition[<term{key}>
;;;;						<dependency>
;;;;						<term{export}>
;;;;						<*{data}>]
;;;;
;;;;
;;;;	dependency-of-definition(<definition>)		: <dependency>
;;;;	oid-of-definition(<definition>)			: <oid-address>	
;;;;
;;;;  -page- 
;;;;
;;;;	define-definition-table (<process-id>
;;;;				 <tag> list
;;;;				 <bool{oid-lookup-p}>
;;;;				 <bool{term-lookup-p}>
;;;;				&key <closure{key-match-f}>
;;;;				     <closure{key-f}>
;;;;				     <closure{order-f}>
;;;;				     <closure{key-clash-f}>
;;;;				     <closure{inter-clash-tables-f}>
;;;;				     <closure{import-f}>
;;;;				     <closure{export-f}>
;;;;				     <closure{delete-f}>
;;;;				     )
;;;;	 : <definition-table>
;;;;
;;;;	
;;;;	** tags will be used in error messages when if insertion clashes detected.
;;;;	** key-match-f(<term{key}> <term{instance}>)	: <bool>
;;;;	**  * First def whose key matches instance will be returned at lookup.
;;;;	** key-f(<definition{insert}>)		: <term{key}>
;;;;	**  * default is nil.
;;;;	**  * It is expected that the key will be computable from the data. Rather
;;;;	**  * than require caller to compute, definition-insert will compute if
;;;;	**  * key is not present in insert. If key-f not specified then it is an
;;;;	**  * error if insert does not have a key.
;;;;	** order-f(<definition{insert}> <definition>)	: <bool>
;;;;	**  * if t then insert will preceed any term-sig equivalent definition.
;;;;	**  * default is #'(lambda (x y) t)
;;;;	** key-clash-f (<term> <term>)			: <bool>
;;;;	**  * default is #'(lambda (x y) nil)
;;;;	**  * It will be an error to insert any def whose key clashes with any 
;;;;	**  * other term-sig equivalent key.
;;;;	** oid-lookup-p default is t.
;;;;	**  * If oid lookup is enabled, then only one definition may be in table
;;;;	**  *  for an oid.
;;;;	** export-f (<definition>)			: <term{export}>
;;;;	**  * default is nil. export-f or export term must be specified in order 
;;;;	**  * to export def.
;;;;	** import-f (<term>)				: <definition>
;;;;	**  * default is nil. Must be specified to import defs.
;;;;	** delete-f (<term>)				: <definition>
;;;;	**  * default is nil.
;;;;	** inter-clash-tables-f ()			: <definition-table> list
;;;;
;;;;
;;;;	insert-f, delete-f, commit-f, and undo-f only available if oid table
;;;;	present.  Any table receiving broadcast will have oid table.
;;;;	  - undo-f can be used to undo side-effects of the delete-f and insert-f
;;;;	    hooks. It is called after undo done. Note that undo of delete calls
;;;;	    insert-f and vice-versa.
;;;;	  - commit-f can be used in place of insert-f and delete-f hooks when
;;;;	    side effects are amenable to delay. It is called after commit done.
;;;;	
;;;;	   * delete-f (<definition>) 
;;;;	      - definition is def deleted.
;;;;
;;;;	   * insert-f (<definition>) 
;;;;	   * create-f (<definition>) 
;;;;	   * move-f (<definition>) 
;;;;	   * activate-f (<definition>) 
;;;;	   * deactivate-f (<definition>) 
;;;;
;;;;	   * undo-f (<definition> <bool{insert?}>)
;;;;	   * commit-f (<definition> <bool{insert-p}>)
;;;;
;;;;	The import and delete hooks should be written to support atomicity of
;;;;	import and delete of lists of definitions. ???
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Definition Table functions : 
;;;;
;;;;	
;;;;	term-to-definition (<definition-table> <term>)	: <definition>
;;;;	
;;;;	definition-delete (<definition-table> <oid> <stamp> INT{sequence})
;;;;	  : NULL
;;;;	definition-insert (<definition-table>  <definition> <stamp> INT{sequence})
;;;;	  : NULL
;;;;
;;;;	definition-commit (<definition-table> <oid> <stamp> INT)	: NULL
;;;;	definition-undo   (<definition-table> <oid> <stamp> INT)	: NULL
;;;;
;;;;	
;;;;	
;;;;	
;;;;	definition-create (<definition-table> <oa>)			: <definition>
;;;;	definition-store (<definition-table> <definition>)
;;;;	  : <definition{original}>
;;;;
;;;;	definition-lookup (<definition-table> <term>)			: <definition>
;;;;	definition-oid-lookup (<definition-table> <oid>)		: <definition>
;;;;	 ** if oid not in table then fails.
;;;;
;;;;	definition-lookup-group (<definition-table> <term>)
;;;;	  : (<term-sig> . <definition> list)
;;;;
;;;;	definition-table-map (<definition-table> <closure{f}>)		: NULL
;;;;	 ** f <definition> 	: NULL
;;;;
;;;;
;;;; -doct- (mod com data)
;;;;
;;;	<definition-term>	: !definition(<dependency-term>; <export-term>)
;;;;
;;;;	<export-term>		: <abs-export-term>
;;;;				| <stm-export-term>
;;;;				| <prf-export-term>
;;;;				| <disp-export-term>
;;;;				| <prec-export-term>
;;;;
;;;; -doct- (mod com)
;;;;
;;;;	definition-import (<definition-table> <term>)		: <definition>
;;;;	definition-import-list (<definition-table> <term>)	: <definition> list
;;;;	 ** failure leaves table unchanged.
;;;;	definition-export (<definition-table> <definition>)	: <definition-term>
;;;;	definition-export-table (<definition-table>)	
;;;;	  : <definition-term> list
;;;;
;;;;	Export terms are cached. Import initializes cache to term imported.
;;;;	At export if term is cached, export-f function will not be called.
;;;;	As a consequence it is difficult to use a table such that its' import
;;;;	and export terms use different syntax. This is not anticipated to be
;;;;	a serious limitation. 
;;;;
;;;;	definition-table-clear-export-cache (<definition-table>)	: NULL
;;;;
;;;;	*** There is some overhead in building these tables. One should not go
;;;;	*** overboard when creating them. Also, due to the way the key-terms are stored,
;;;;	*** a table containing many terms with the same term sig may have slow lookups.
;;;;
;;;; -doct- (mod com ml)
;;;;
;;;;	
;;;;	Touch history : 
;;;;
;;;;	A record of recent changes to system tables.  Each entry will be a
;;;;	triple of the object-address, the table type (ie one of statement,
;;;;	proof, rule, abstraction, or display), and an indication of the type of
;;;;	change(ie add or delete).
;;;;	
;;;;	<touch-history>		: (<oid> . (<tok{table}> . <tok{type}> ) list
;;;;
;;;;	type : `insert` | `delete`
;;;;	table : `ABSTRACTION` | `STATEMENT` | `RULE` etc.
;;;;
;;;;	touch_history	 	: unit -> (object_id # tok # tok) list
;;;;	clear_touch_history	: unit -> unit
;;;;	
;;;;	Insert or delete of data into a table updates touch history.
;;;;
;;;;	TODO : use in ML needs to be updated.	
;;;;	
;;;; -doce-

;;;;	<data>		:
;;;;    Insert or delete of data into a table updates touch history.
;;;;
;;;;


;; needs to be after orb-defs and also loaded in 4.5

(defun close-environment (clenv &optional quickp (gcp t))
  (stop-db-buffering)
  (let* ((addr  (address-of-environment clenv))
	 (iaddr (ienvironment-address-term addr)))

    ;;(break "ce")
    (when (and (not quickp)
	       (member 'library (environment-resources clenv) :key  #'car))

      (when gcp
	(with-environment-actual clenv
	  (with-transaction (t)
	    (ap (ml-text "\\u. (dag_remove_directory (descendent_s ``local garbage``) `queue` ? ())") nil)
	    )))
      
      (with-environment-actual clenv
	(with-transaction (t)
	  (without-dependencies
	   (when gcp
	     
	     (format t ";;;     begin deactivate orphans~%")
	     (mlet* (((orphan-count count) (deactivate-orphans-aux (resource 'library))))
		    (format t "***    Orphans ~8:D object ids.~%" orphan-count)
		    (format t "***    Total ~8:D object ids.~%" count))

	     (format t ";;;     begin unbind~%")
	     (mlet* (((unbind-count count) (unbind-collect-aux (resource 'library))))
		    (format t "***    Unbound ~8:D object ids.~%" unbind-count)
		    (format t "***    Visited ~8:D object ids.~%" count))

	     (when (completions-p)
	       (format t ";;;     begin commit~%")
	       (commit-completions (completion-peek-first)))
	     )))))

    (when (member 'library (environment-resources clenv) :key #'car)
      (format t ";;;     begin close table-logs~%")
      (handle-process-err #'(lambda (err-str)
			      (format t "err-str ~a" err-str)
			      nil)
			  (close-table-logs clenv (not quickp)))
      )

    ;; if journal close journal.
    ;; todo : this is somewhat of a kludge. stopping the broadcasts should cause journal to close itself??
    ;; or closing journal should stop broadcasts?
    ;;(setf a (car (journals-of-environment clenv))) (break "ec")
    (format t ";;;     begin close journal~%")
    (let ((stamps (mapcar #'(lambda (j)
			      (if quickp
				  (journal-close j)
				  (journal-checkpoint-aux clenv j)))
			  (journals-of-environment clenv))))

      ;; MTT if producer, halt production.

      ;; first halt incoming.
      ;; here we are initiating revoke in consumer.
      ;;(setf -env clenv) (break "cl")
      (dolist (bstate (broadcast-states-of-environment clenv))
	(let ((paddr (producer-address-of-broadcast-state bstate)))
	  ;; exports revoke in consumer.
	  (unless (or (equal paddr addr)
		      (not (member paddr (orb-bus-environments) :test #'equal)))

	    (unsubscribe-client-initiate (table-types-of-broadcast-state bstate)
					 paddr
					 nil))))

      ;; now halt outgoing. for each link, look at export environments
      ;; here we are initiating revoke in producer.
      (dolist (blink *bus-links*)
	(let ((env (find-environment-in-list addr
					     (exported-environments-of-bus-link blink))))

	  (when env
	    (dolist (bstate (broadcast-states-of-environment env))
	      (let ((paddr (producer-address-of-broadcast-state bstate)))
		;; we should process export locally first? We shouldn't be here if
		;; broadcast are being produced, so order should not matter.
		;; send configure does send then export.
		(if (equal paddr addr)
		    (unsubscribe-server-initiate (consumer-address-of-broadcast-state bstate)
						 (consumer-address-of-broadcast-state bstate))
		    (system-error (raise-error '(environment close producer))))
	  
		;; revoke addrs and table types.
		(config-unexport-address blink paddr))))))

      (format t ";;;     delete environment~%")
      (delete-environment addr)

      (mapcar #'stamp-to-string stamps)
      (format t ";;;     Done.~%"))))

(defun close-environment-client (clenv &optional quickp)
  (let* ((addr  (address-of-environment clenv))
	 (iaddr (ienvironment-address-term addr)))

    ;;(setf -env clenv) (break "cl")

    ;; first halt incoming.
    ;; here we are initiating revoke in consumer.
    (unless quickp
      (dolist (bstate (broadcast-states-of-environment clenv))
	(let ((paddr (producer-address-of-broadcast-state bstate)))
	  ;; exports revoke in consumer.
	  (unless (or (equal paddr addr)
		      (not (member paddr (orb-bus-environments) :test #'equal)))

	    (unsubscribe-client-initiate (table-types-of-broadcast-state bstate)
					 paddr
					 nil)))))

    (format t ";;;     delete environment~%")
    (delete-environment addr)

    (format t ";;;     Done.~%")))


;; cleans-up local environment.
;; assumes connected components will not respond and thus is just
;;  cleaning up local state.
#|
;; not certain what the point to this was
(defun cleanup-environment (clenv)
  (stop-db-buffering)

  (let* ((addr  (address-of-environment clenv))
	 (iaddr (ienvironment-address-term addr)))

    ;;(break "ce")
    (when (member 'library (environment-resources clenv) :key #'car)
      (format t ";;;     begin close table-logs~%")
      (handle-process-err #'(lambda (err-str)
			      (format t "err-str ~a" err-str)
			      nil)
			  (close-table-logs clenv nil)))

    ;; if journal close journal.
    ;; todo : this is somewhat of a kludge.
    ;;   stopping the broadcasts should cause journal to close itself??
    ;; or closing journal should stop broadcasts?
    ;;(setf a (car (journals-of-environment clenv))) (break "ec")
    (format t ";;;     begin close journal~%")
    (let ((stamps (mapcar #'(lambda (j)
			      (journal-close j) )
			  (journals-of-environment clenv))))

      ;; MTT if producer, halt production.

      ;; first halt incoming.
      ;; here we are initiating revoke in consumer.
      (dolist (bstate (broadcast-states-of-environment clenv))
	(let ((paddr (producer-address-of-broadcast-state bstate)))

	  ;; exports revoke in consumer.
	  (unless (or (equal paddr addr)
		      (not (member paddr (orb-bus-environments) :test #'equal)))

	    (let ((blink (bus-link-of-environment-address paddr)))
	      (configure-export (irevoke-term
				 (istart-term
				  (itable-types-term (table-types-of-broadcast-state bstate)
						     (ienvironment-address-term paddr))
				  (stamp-term-of-environment clenv)
				  (ienvironment-address-term addr)
				  (ivoid-term)))
				blink
				)))))

      ;; now halt outgoing. for each link, look at export environments
      ;; here we are initiating revoke in producer.
      (dolist (blink *bus-links*)
	(let ((env (find-environment-in-list addr
					     (exported-environments-of-bus-link blink))))

	  (when env
	    (dolist (bstate (broadcast-states-of-environment env))
	      (let ((paddr (producer-address-of-broadcast-state bstate)))
		(if (not (equal paddr addr))
		    (system-error (raise-error '(environment close producer)))
			 
		    ;; we should process export locally first? We shouldn't be here if
		    ;; broadcast are being produced, so order should not matter.
		    ;; send configure does send then export.
		    (configure-export blink
				      (irevoke-term
				       (istart-term
					(itable-types-term (table-types-of-broadcast-state bstate)
							   (ienvironment-address-term paddr))
					(stamp-term-of-environment clenv)
					(consumer-address-of-broadcast-state bstate)
					(ivoid-term)))))))))
	  
	;; revoke addrs and table types.
	(configure-export (irevoke-term iaddr)
			  blink)
	)

      (format t ";;;     delete environment~%")
      (delete-environment addr)

      (mapcar #'stamp-to-string stamps)
      (format t ";;;     Done.~%")))
  
  ;;(start-db-buffering)
  )
|#


(defun new-object-id ()
  (let ((oid (new-oid (new-transaction-stamp))))
    (push-io-history (ioid-term oid) 'new)
    oid))

(define-primitive |!cut_cons| () (car cdr))
(defun icut-oid-list (oids) (map-list-to-ilist oids (icut-nil-term) #'ioid-term))
(defun oids-of-icut-list (oids) (map-ilist-to-list oids (icut-cons-op) #'oid-of-ioid-term))



(define-primitive |!rdb_table| () (tuples))
(define-primitive |!terms| () (subterm))

(defun iterms (terms)
  (instantiate-term (iterms-op) (mapcar #'instantiate-bound-term terms)))

(defun subterms-of-iterms-term (iterms)
  (mapcar #'term-of-bound-term-f (bound-terms-of-term iterms)))

(defvar *null-oid*
  (let ((stamp '((0 . NULL) . (0 . 0))))
    (make-oid :stamp stamp
	      :string (term-to-standard-character-string (stamp-to-term stamp)))))

(defun null-oid-p (oid)
  (equal-oids-p oid *null-oid*))

(defun subterms-of-term (term)
  (mapcar #'term-of-bound-term-f (bound-terms-of-term term)))

(defun sort-oids-by-time (oids)
  (sort oids
	#'>
	:key #'(lambda (o) (time-of-stamp (stamp-of-oid o)))))

;; was returning list of variable-terms:
;; I dont' know the point to this function but it can not work in general
;; since if binding is dummy, variable-term will fail.
;; there is ability to allow dummy variable-terms but can of worms.
(defun bindings-of-term (term)
  (mapcar #'(lambda (bt)
	      (bindings-of-bound-term bt))
	  (bound-terms-of-term term)))
	
(defun equal-oids-p (oida oidb)
  (or (eq oida oidb)
      (equal-stamps-p (stamp-of-oid oida)
		      (stamp-of-oid oidb))))



;;;;	
;;;;	Data
;;;;
;;;;	Data may exists as proxy disk pointer(possibly inlined) or as subclass.
;;;;	If proxy then can be inflated using data-import method.
;;;;	Import does not alter proxy. Thus caller probably needs to set
;;;;	pointer to proxy to be imported data.
;;;;	
;;;;	Once imported term representation can be recovered via export.
;;;;	Other users of imported data may want to ensure that data is persistent
;;;;	and export to disk if not present. 
;;;;	
;;;;	It appears it would be beneficial to more tightly control persistence
;;;;	and terms to avoid re-exportation and to implicitly know if data is 
;;;;	persistent without a trip to disk.
;;;;	
;;;;	
;;;;	Possible Invariant.
;;;;	  data-term == export (import data)
;;;;	   - this should certainly be true at the instant of import. 
;;;;	   - but can imported data be modified such that later export produces diff data-term.
;;;;	      * no since if so then if persistent we will lose diff since we do assume export
;;;;		matches persistent data.
;;;;	  
;;;;	stamp and type comprise disk address.
;;;;	if not inline then must be persistent ? or may not yet have been made persistent
;;;;	thus flag can say if we came from disk.

;; data addressed by a stamp.
(defclass data ()
  ((stamp :reader stamp-of-data
	  :initform (new-transaction-stamp)
	  :initarg stamp)

   (flags :reader flags-field-of-data
	  :initform nil
	  :writer set-data-flags-field
	  )

   ;; redundant if data is provided.
   ;; otherwise tells how to provide.
   ;; if present accurate, however in provided forms in may be nil.
   (type :reader super-type-of-data
	 :writer set-data-type
	 :initform nil
	 :initarg type)
   
   ;; true if data has been read from db and unmarshalled, ie is present.
   ;;(provided  :reader data-provided-p
   ;;:writer set-data-provided
   ;;:initarg provided
   ;;:initform nil)
   ))


(defvar *debug-type-of-data* t)
(defun type-of-data (data)
  (or (super-type-of-data data)
      (if (data-flag-provided-p data)
	  (type-of data)
	  (progn
	    ;; expect unprovided to have accurate type field.
	    ;;(setf -data data)
	    (when *debug-type-of-data*
	      (break "type-of-data"))
	    nil))))

(defun set-data-provided (v data) (data-flag-set-provided data v))
(defun data-provided-p (data) (data-flag-provided-p data))
  


;; contains flags for all subclass as well.
(define-flags (data nil t)
    ((provided t nil)

     ;; persistent if read from disk. inline counts as persistent.
     ;; need to carry forward at import, but not clone.
     ;; if persistent and not inline then conclude exists on disk.
     ;; if inline and need persist pointer then re-inline.
     ;; or was inlined and decide diskpersist better then write.
     (persistent nil t)
     (inline nil t)
     ))



;; RLE TODO twould be neat to define so:
;;
;;	(define-flags (data t)
;;	    ((data ((provided t nil)))
;;
;;	     (objc ((require-ostate nil) ))))
;;;;	
;;;;	
;;;;	
;;;;	
;;;;	

(defclass inline-data  (data)
  ((inline :reader inline-of-data
	   :writer set-data-inline
	   :initarg inline)
   ))

(defun data (stamp type)
  ;;(break "data")
  ;;(format t "data ~a ~a~%" type stamp)
  (let ((d (make-instance 'data 'stamp stamp 'type type)))
    (data-flag-set-provided d nil)
    (data-flag-set-persistent d t)
    d))

(defun inline-data (stamp type term)
  ;;(format t "inline-data ~a ~a~%" type stamp)
  (let ((d (make-instance 'inline-data 'stamp stamp 'type type 'inline term)))
    (data-flag-set-provided d nil)
    (data-flag-set-persistent d t)
    (data-flag-set-inline d t)
    ;;(push d -dd )
    d))


(defun provide-data (data intype &optional force)

  (if (and (not force) (data-provided-p data))
      data

      ;; data should have type implicit and thus the arg is superfluous.
      ;; However I suspect that the type-of data may be incorrect is some cases
      ;; thus the check. Certainly the intype is incorrect for csource.
      ;; if we run a while with no breaks then we could remove arg and
      ;; mill callers.
      (let ((type (if (and (eql 'source intype)
			   (eql 'code-source (type-of-data data)))
		      'code-source
		      (progn
			(unless (eql intype (type-of-data data))
			  ;;(setf -intype intype -type (type-of-data data) -data data)
			  (when *debug-type-of-data* (break "pd type ???")))
			intype))))
		  
	(if data
	    (let ((d (make-instance type 'stamp (stamp-of-data data) 'type type)))
	      ;;(break "pd")
	      ;;(format t "make ~a ~a ~a ~%" (data-flag-inline-p data) type (stamp-of-data data))

	      ;;(data-flag-set-provided d t)  ; true by default.
	      (data-flag-set-persistent d (data-flag-persistent-p data))
	      (data-flag-set-inline d (data-flag-inline-p data))

	      (data-import d
			   (if (typep data 'inline-data)
			       (inline-of-data data)
			       (db-read (stamp-of-data data) type)))
	      ;;(format t "make ~a import ~%" (data-flag-inline-p d))

	      d)
	    nil))))

;;;;	
;;;;	data_persist could cache oids for bind collection.
;;;;	 - oids 
;;;;	 - data persists
;;;;	
;;;;	 - optionally data persist should contain oids of all direct oids.
;;;;	  by direct either explicit occurence in term or in closure of data persists of term.
;;;;	
;;;;	Invariant : data persists are acyclic.
;;;;	
;;;;	
;;;;	
(define-primitive |!blob_nameinfo| ((s . name) (s . type)))
(define-primitive |!blob_proxy| () (stamp nameinfo))
;;;;	if type is BLOB, then there is no term. There is a file but it contains
;;;;	arbitrary data. It cannot contain obids or other data_persist terms, ie
;;;;	it is a leaf.
;;;;	
;;;;	  - when term asked for:
;;;;	      * return !BLOB()?
;;;;	      * fail ? why ask for term that can not exist.
;;;;	    either/or depending on circumstance
;;;;	

(define-primitive |!data_persist| ((token . type)) (stamp))
(define-primitive |!data_persist_inline| ((token . type)) (stamp term))

(defun term-to-data (term)
  ;;(setf -term term) (break "ttd")
  (unless (ivoid-term-p term)
    (or (mark-value term 'data)
	(if (idata-persist-inline-term-p term)
	    (let ((stampt (stamp-of-idata-persist-term term)))
	      (inline-data
	       ;; !void() used to indicate data produced ephemerally and not
	       ;; associated with stamp in generating process.
	       (if (ivoid-term-p stampt)
		   (progn (break "ttd !void")
			  (new-transaction-stamp))
		   (term-to-stamp stampt))
	       (type-of-idata-persist-term term)
	       (term-of-idata-persist-inline-term term)))
	    (data (term-to-stamp (stamp-of-idata-persist-term term))
		  (type-of-idata-persist-term term))))))

(defun term-of-idata-persist-term (term)
		      
  (if (idata-persist-inline-term-p term)
      (term-of-idata-persist-inline-term term)
      (if (idata-persist-term-p term)
	  (db-read (term-to-stamp (stamp-of-idata-persist-term term))
		   (type-of-idata-persist-term term))
	  (raise-error '(error-message '(term data persist not) term)))))


(defvar *term-of-data-ephemeral-p* nil)
;; data export will write to data base if data contains non-persistent references.
;; shouldn't happen, but it may be possible that inlined data gets written to disk,
;; due to change of tuning parameters, ie persist-op-count-threshold.
(defun term-of-data (data &optional ephp)
  (if (data-provided-p data)
      (let ((*term-of-data-ephemeral-p* ephp))
	(data-export data (ivoid-term)))
      (if (typep data 'inline-data)
	  (inline-of-data data)
	  (db-read (stamp-of-data data) (type-of-data data)))))

(defvar *persist-op-count-threshold* 32)

;;;;	
;;;;	Worry : !data_persist sent but data not on disk because of inline.
;;;;	  Either provided or not.
;;;;	    If not provided then (typep data 'data-inline) t iff not on disk.
;;;;		if (typep data 'data-inline) is t then persist not sent
;;;;		if (typep data 'data-inline) is nil then data on disk since we had !data persist (induction hyp)
;;;;	    If provided then if check if on disk before return !data_persist.
;;;;	

;;;;	quoting : in some instances an embeded !data_persist may not be 
;;;;	real address
;;;;	
;;;;	data_persist does need a do-not-visit flag for gc for something
;;;;	like metaprl primitive proofs.

;; if data read but not imported and then persist data.
;; then type is 'data

(defun force-inline-export-data (data)
  (let ((type (or (type-of-data data)
		  (type-of data)))
	(stamp (stamp-of-data data)))
	       
    (idata-persist-inline-term type
			       ;;(stamp-to-term stamp)
			       (ivoid-term)
			       (data-export (objc-source data) (ivoid-term)))))


(defmethod persist-data (data &optional provided-required inline-ok)
  ;;(setf a data) (break "pd1")
  ;; if not provided then assume already persistent.
  ;; if data derived from disk then should be able to simply return original !data_persist
  (let ((term
	 (if (null data)
	     (ivoid-term)
	     
	     (let ((type
		    ;; if present type-of-data should be accurate.
		    ;; new constructors may not init though so type-of is backup.
		    (or (type-of-data data)
			(type-of data)))
		   (stamp (stamp-of-data data)))
	       
	       ;;(setf -data data)
	       (if (data-provided-p data)
		   ;; if originally inlined and then made persistent,
		   ;; persist-data will rewrite ie why do we check if inline-p?
		   (if (and (not (eql 'objc type)) ;; flush inlines to log.
			    (data-flag-persistent-p data)
			    (not (data-flag-inline-p data)))
		       (idata-persist-term type (stamp-to-term stamp))
		   
		       ;; perf could skip if not inline and dprevent if 
		       (let ((term (data-export data (ivoid-term))))

			 ;;(setf a data) (break "pd")
			 (if (and (or *term-of-data-ephemeral-p* inline-ok)
				  (or (eql 'objc type)
				      (and *term-of-data-ephemeral-p* (data-flag-inline-p data))
				      (not (term-op-count-exceeds-p term *persist-op-count-threshold*))))

			     (progn
			       ;;(format t "i")
			       (idata-persist-inline-term type (stamp-to-term stamp) term)
			       )
			     
			     (progn
			       (when (eql type 'data)
				 (break "Not expecting type to be data"))
			       ;;(unless (db-persistent-p stamp type) )
			       ;;(break "dbp")
			       (data-flag-set-persistent data t)
			       (db-write-persist stamp type term t)

			       ;;(unless (and nil (db-persistent-p stamp type)))
			       ;;(when (< (term-op-count term) 32) (break "adfa"))
			       ;;(format t "[*~a]~%" (term-op-count term))
			       (idata-persist-term type (stamp-to-term stamp))))))

		   (if provided-required
		       (raise-error (error-message '(data persist provided not)))
		       (if (data-flag-inline-p data) ;;(typep data 'inline-data)
			   ;; fixup inlines which probably should not have been inlined.
			   ;; or were imported.
			   (if (and t (term-op-count-exceeds-p (inline-of-data data) *persist-op-count-threshold*))
			       (if t
				   (progn
				     (if (and nil (db-persistent-p stamp type))
					 (progn
					   ;;(setf -data data) (break "dp!")
					   (format t "inline==persistent[!~a]~%" (term-op-count (inline-of-data data))))
					 (progn
					   ;;(setf -data data) (break "dp")
					   (format t "inline->persistent[~a]~%" (term-op-count (inline-of-data data)))
					   (data-flag-set-inline data nil)
					   (db-write-persist stamp type (inline-of-data data) nil)))
				     (idata-persist-term type (stamp-to-term stamp)))
				   (progn
				     (format t "[~a]~%" (term-op-count (inline-of-data data)))
				     (idata-persist-inline-term type
								(stamp-to-term stamp)
								(inline-of-data data))))
			       (idata-persist-inline-term type
							  (stamp-to-term stamp)
							  (inline-of-data data)))
			   (idata-persist-term type
					       (stamp-to-term stamp)))))))))

    (mark term 'data data)
    term))


(define-primitive |!data| () (term))

(defmethod data-export ((data data) term)
  (idata-term term))
  
(defmethod data-export-aux ((data data) term)
  (idata-term term))
  
(defmethod data-import ((data data) term)
  ;;(setf b term) (break "di")
  (term-of-idata-term term))
  
(defmethod clone (o)
  (clone-copy (make-instance (class-of o)) o))

(defmethod clone ((data data))
  (clone-copy data (make-instance (class-of data) 'stamp (new-transaction-stamp))))

(defmethod clone-copy ((old data) (new data))
  ;;(format t "clone ~a ~a~%" (type-of-data old) (data-flag-inline-p old) )
  (set-data-type (type-of-data old) new)
  (set-data-flags-field (clone-flags (flags-field-of-data old)) new)
  (data-flag-set-persistent new nil)
  (data-flag-set-inline new nil)
  new)


;;;
;;;	 oid-message
;;;


(defstruct (message (:include basic-message))
  (oids nil))

(defun oids-of-message (m)
  (when (message-p m)
    (message-oids m)))

(defun oid-message (oids tags &rest pmsg)
  (make-message :tags (if (consp tags) tags (list tags))
		:pmsg pmsg
		:stamp (transaction-stamp)
		:oids oids))

(defun oid-error-message (oids tags &optional pmsg)
  (unless (listp oids) (break "oem"))
  (oid-message oids (cons 'error tags) pmsg))

(defun oid-warn-message (oids tags &rest pmsg)
  (oid-message oids (cons 'warning tags) pmsg))

(defun oid-inform-message (oids tags &rest pmsg)
  (oid-message oids (cons 'inform tags) pmsg))

(defun oid-type-message (type oids tags &optional pmsg)
  (oid-message oids (cons type tags) pmsg))


;;;
;;;	message -> term.
;;;

(defparameter *imessage* '|!message|)

(defun imessage-term (tags oids stamp pmsg)
  ;;(format t "~%~a~%" tags) (break "imt")
  (instantiate-term (instantiate-operator *imessage*
					  (mapcar #'(lambda (tag)
						      (if (integerp tag)
							  (progn
							    ;;(break "integer should be token")
							    (format t "integer should be token")
							    (instantiate-parameter-r tag *natural-type*))
							  (instantiate-parameter-r tag *token-type*)))
						  tags))
		    (list (instantiate-bound-term oids)
			  (instantiate-bound-term stamp)
			  (instantiate-bound-term pmsg))))

(defun imessage-term-p (term)
  (and (eql *imessage* (id-of-term term))
       (forall-p #'token-parameter-p (parameters-of-term term))
       (let ((bts (bound-terms-of-term term)))
	 (and (null (cdddr bts))
	      (null (bindings-of-bound-term-n (car bts)))
	      (null (bindings-of-bound-term-n (cadr bts)))
	      (null (bindings-of-bound-term-n (caddr bts)))))))

						
(defun tags-of-imessage-term (term)
  (mapcar #'value-of-parameter-r (parameters-of-term term)))

(defun oids-of-imessage-term (term)
  (term-of-bound-term (car (bound-terms-of-term term))))

(defun stamp-of-imessage-term (term)
  (term-of-bound-term (cadr (bound-terms-of-term term))))

(defun transaction-stamp-of-imessage-term (term)
  (term-of-bound-term (cadr (bound-terms-of-term term))))

(defun pmsg-of-imessage-term (term)
  (term-of-bound-term (caddr (bound-terms-of-term term))))


(defun degenerate-term-message (terms)
  (basic-message nil terms))

(defun degenerate-term-message-p (m)
  (and (null (tags-of-message m))
       (term-p (pmsg-of-message m))))


(defun message-to-term (m)
  ;;(setf -mm m)(break "mtt")
  (cond
    ((null m) (imessage-term '(nil message)
			     (ioids-term nil)
			     (stamp-to-term (transaction-stamp))
			     (pmsg-to-term nil)))
    ((degenerate-term-message-p m) (pmsg-of-message m))
    (t (imessage-term (tags-of-message m)
		      (ioids-term (oids-of-message m))
		      (stamp-to-term (stamp-of-message m))
		      (pmsg-to-term (pmsg-of-message m))))))


(defun pmsg-to-term (pmsg)
  (cond

    ((consp pmsg)
     (cond
       ((null (car pmsg))
	(pmsg-to-term (cdr pmsg)))
       ((null (cdr pmsg))
	(pmsg-to-term (car pmsg)))
       (t (itext-cons-term (pmsg-to-term (car pmsg)) (pmsg-to-term (cdr pmsg))))))

    ((basic-message-p pmsg)
     (message-to-term pmsg))

    ((break-control-p pmsg)
     (ibreak-control-term (type-of-break-control pmsg)
			  (pmsg-to-term (text-of-break-control pmsg))))

    (t (text-to-term pmsg))))



(defun print-term-pmsg-item (item depth)

  (cond
    
    ((eql #\space item) (format t  " "))

    ((eql #\newline item) 
     (format t "~%~a" (blank-string depth)))

    ((stringp item) (format t " ~a" item))

    ((symbolp item) (format t " ~a" item))

    ((integerp item) (format t " ~a" item))

    ((term-p item)
     (format t "~%~a" (blank-string depth))
     (print-term item (1+ depth)))
  
    ((basic-message-p item)
     ;;(format t  " print-message-item depth ~a ~%" depth)
     (format t "~%~a" (blank-string depth))
     (print-oid-message item depth))
    
    ((break-control-p item) nil)

    (t (let ((s (princ-to-string item)))
	 ;;(setf -i item)
	 ;;(break "smtt")
	 (message-emit (warn-message '(text term) (list s)))
	 (itext-term s)))))

(defun print-oid-message (m depth)
  ;;(break)
  ;;(format t  " print-oid-message depth ~a ~%" depth)
  (when (< depth (max-print-depth))
    (print-stamp (stamp-of-message m))
    (format t "[~a]:" (tags-of-message m))
    (when (oids-of-message m)
      (format t "[~a oids]" (length (oids-of-message m)))
      ;; TODO twould be nice to find mnemonic and print if available.
      ;;(print-object-addresses (object-addresses-of-message m))
      )
    (print-pmsg (pmsg-of-message m) (1+ depth) #'print-term-pmsg-item)
    ))


(setf *print-message-hook* #'print-oid-message)

(defmacro with-oids ((oids &optional tags) &body body)
  `(with-handle-error ((,tags nil #'(lambda (m)
				      (oid-error-message ,oids
							nil m)))
		       (raise-error nil))
    (progn ,@body)))



(defun new-event-description (system version purposes)
  (make-description :system system
		    :version version
		    :purposes purposes
		    ))


;; bcast is !description term and denv is description structure.

;; match-descriptions-p (<term{description}> <description>)
;; description-match (<description> <description>)

(defvar *component-toks* '(library refiner edit))
(defun component-purpose-p (toks)
  (not (null (intersection toks *component-toks*))))

(defun match-descriptions-p (instance template &optional strictp)
  ;;(setf -instance instance -template template) (break "mdp")
  (or ;;(when (null instance) (break));; LAL why null?
   (ivoid-term-p instance)
   (let ((sys (system-of-idescription-term instance)))
     (and (or (eql '!any sys)
	      (and (eql sys (system-of-description template))
		   (equal (map-isexpr-to-list (version-of-idescription-term instance)
					      (icons-op)
					      #'numeral-of-inatural-term)
			  (version-of-description template))))
	  (let* ((ipurposes (map-isexpr-to-list (purposes-of-idescription-term instance)
						(icons-op)
						#'token-of-itext-term))
		 (tpurposes (purposes-of-description template))
		 (pintersection (intersection ipurposes tpurposes)))
	    (and (not (null pintersection))
		 ;; must have major pupose in common.
		 ;;(or (not (component-purpose-p ipurposes))
		 ;;    (not (null (intersection pintersection *component-toks*))))
		 (or (not strictp)
		     (= (length pintersection) (length ipurposes)))))
	    ))))


(defun description-match (template instance)
  (or (null instance)
      (and (not (null template))
	   (eql (system-of-description template) (system-of-description instance))
	   (equal (version-of-description template) (version-of-description instance))
	   (= (length (intersection (purposes-of-description template)
				    (purposes-of-description instance)))
	      (length (purposes-of-description template))))))



;;;;	
;;;;	Dependencies.
;;;;	

(defstruct (dependency (:include marks))
  (data-stamp nil)
  (objc-stamp nil)
  (oid nil)
  
  ;; if oid mapped since term instantiated may want to re-instantiate.
  ;; OTOH Tis ok without re-instantiate.
  ;; maybe should be a mark.
  (term nil))


(defun dependency (oid objc-stamp data-stamp &optional term)
  (make-dependency :oid oid
		   :objc-stamp objc-stamp
		   :data-stamp data-stamp
		   :term term
		   ))

(defun objc-stamp-of-dependency (d) (dependency-objc-stamp d))
(defun data-stamp-of-dependency (d) (dependency-data-stamp d))
(defun oid-of-dependency (d) (dependency-oid d))
(defun term-of-dependency (d) (dependency-term d))



(defstruct dependencies
  tag
  list)


(defun new-dependencies (tag list) (make-dependencies :tag tag :list list))
(defun tag-of-dependencies (d) (dependencies-tag d))

(defstruct (quick-dependencies (:include dependencies))
  ;; adjustable-array check bounds and grow if necessary
  array)

(defun array-of-quick-dependencies (d) (quick-dependencies-array d))

(defun quick-dependency (dependencies i d)
  (let ((a (array-of-quick-dependencies dependencies)))
    (unless (< i (length a))
      (setf (quick-dependencies-array dependencies)
	    (adjust-array a (+ (length a) 8))))
    
    (or (aref a i)
	(setf (aref a i) d)))

  (values))

(defun new-quick-dependencies (tag table)
  (make-quick-dependencies
   :tag tag
   :list nil
   :array (make-array (+ 8 (quickdepref-index-of-definition-table table)) :initial-element nil)))

;; works both ways 
(defun list-of-dependencies (d)
  (let ((l (dependencies-list d)))
    ;;(setf -d d -l l) (break "lod")
    (if (quick-dependencies-p d)
	(let ((acc nil)
	      (a (array-of-quick-dependencies d)))
	  (dotimes (i (length a))
	    (let ((d (aref a i)))
	      (when d (push d acc))))
	  (if l (nconc acc l) acc))
	(dependencies-list d))))

(defun dependency-to-string (d)
  (with-output-to-string (stream)
    (print-stamp (data-stamp-of-dependency d))))

(defun remove-duplicate-dependencies (sexpr &optional oidp)
  ;;(setf -sexpr sexpr) (break "rdd")
  (let ((acc nil))
    ;;(setf -cds (cdr ds)) 
    ;; flatten and remove dups.
    (walk #'(lambda (d)
	      ;; (when -fu (setf -d d -acc acc) (break "nda"))
	      (if oidp
		  (let ((oid (oid-of-dependency d)))
		    (unless (member-oid oid acc)
		      (push oid acc)))
		  (unless (member d acc :test #'equal-dependencies-p)
		    (push d acc))))
	  sexpr)

    (show-telemetry "~%rdd ~a ~a" (length sexpr) (length acc))

    acc))

;; maybe not proper to be data as not expected to reside independently
;; in database.
(defstruct stamp-dependencies
  stamp
  list)

(defun stamp-of-stamp-dependencies (s) (stamp-dependencies-stamp s))
(defun list-of-stamp-dependencies (s) (stamp-dependencies-list s))


(defstruct (environment-dependencies (:include stamp-dependencies)))

(defun environment-dependencies (d l)
  (make-environment-dependencies :stamp d :list l))

(defun environment-dependencies-normal (d l)
  (make-environment-dependencies :stamp d :list (normalize-dependencies l)))


(defstruct (event-dependencies (:include stamp-dependencies))
  (description nil))

(defun description-of-event-dependencies (d) (event-dependencies-description d))

(defun event-dependencies (s desc l)
  (make-event-dependencies :stamp s :list l :description desc))


(defclass dependency-store (data)
  (
   (list :reader list-of-dependency-store
	 :writer set-dependency-store-list
	 :initarg list
	 :initform nil
	 )

   (normal-list :reader normal-list-of-dependency-store
		:initarg normal-list
		:writer set-dependency-store-normal-list
		:initform nil
		)
   ))


(defun dependency-store-add-dependencies (store deps)

  (let ((l (list-of-dependency-store store))
	(nl (normal-list-of-dependency-store store)))

    (set-dependency-store-list (cons deps l) store)

    (when (or nl (null l))
      (set-dependency-store-normal-list (cons deps nl) store)) ))


(defun dependency-store (l)
  ;;(setf -l l) (break "ds")
  (make-instance 'dependency-store 'list l))

(defun normal-dependency-store (nl)
  (make-instance 'dependency-store 'list nl 'normal-list nl))


(defun term-to-event-dependencies-list (term)
  (map-isexpr-to-list term
		      (ievent-dependencies-cons-op)
		      #'term-to-environment-dependencies))

(defun term-to-dependency-store (term)
  (unless (ivoid-term-p term)
    (normal-dependency-store (term-to-dependencies-list
			      (list-of-idependency-store-term term)))))


;; wouldn't it be better to save in unnormalized form.
;; normalizing loses some infomation.
(defun dependency-store-to-term (store)
  (if (null store)
      (ivoid-term)
      (idependency-store-term
       (dependencies-sexpr-to-term
	(dependencies-of-dependency-store store)))))
	

(defmethod data-import ((ds dependency-store) super)
  (let ((term (call-next-method ds super)))

    (let ((lterm (unless (ivoid-term-p term) (list-of-idependency-store-term term))))
      (if (equal-operators-p (operator-of-term lterm) (ievent-dependencies-cons-op))
	  (progn
	    (format t "Warn : old style dependency store term~%")
	    (set-dependency-store-list (list (event-dependencies (transaction-stamp)
								 (new-event-description *system* '(5 0) '(summary))
								 (term-to-event-dependencies-list lterm)))
				       ds))

	  (let ((l (unless (ivoid-term-p term)
		     (term-to-dependencies-list (list-of-idependency-store-term term)))))
      
	    (set-dependency-store-normal-list l ds)))
      ))

  (values))

(defmethod data-export ((ds dependency-store) term)
  (declare (ignore term))
  (call-next-method ds (dependency-store-to-term ds)))



;;;;	data-export : method - data may contain other data.
;;;;	 when exporting such contained data should be exported
;;;;	 and the stamp should be included in callers export.
;;;;	


;;      (db-write stamp 'inf-tree-proxy (inf-tree-to-iinf-tree-term inf-tree nil))
;;      (db-write stamp 'source (proof-source-to-term inf-tree nil))


;;;
;;;	Dependency <-> term.
;;;


(define-primitive |!dependency_cons| () (car cdr))

(define-primitive |!dependencies| ((token . tag)) (list))
(define-primitive |!dependencies_cons| () (car cdr))

(define-primitive |!environment_dependencies| () (stamp list))
(define-primitive |!environment_dependencies_cons| () (car cdr))

;; must have stamp to read from db so maybe stamp not necessary in term.
;; or have alternate term for db without stamp.
(define-primitive |!event_dependencies| () (stamp description list))
(define-primitive |!event_dependencies_cons| () (car cdr))

(define-primitive |!dependency_store| () (list))


(defun dependency-to-term (d)
  (or (term-of-dependency d)
      (setf (dependency-term d)
	    (idependency-term (oid-of-dependency d)
			      (stamp-to-term (objc-stamp-of-dependency d))
			      (stamp-to-term (data-stamp-of-dependency d))))))


(defun term-to-dependency (term)
  (unless (idependency-term-p term)
    (raise-error (error-message '(dependency term) term)))

  (dependency (oid-of-idependency-term term)
	      (term-to-stamp (objc-of-idependency-term term) t)
	      (term-to-stamp (data-of-idependency-term term) t)
	      term))

(defun dependency-list-to-term (l)
  (when (and nil
	     (> (length l) 128))
    (format t "lots of deps ~a~%" (length l))
    (break "dltt"))
  (map-sexpr-to-ilist l
		      (idependency-nil-term)
		      #'dependency-to-term))

(defun term-to-dependency-list (term)
  (map-isexpr-to-list term
		      (idependency-cons-op)
		      #'term-to-dependency))



;; returns sublist of l with no dup members, order not preserved
(defun unique-sublist (l)
  (let ((table (make-hash-table :test #'equal))
	(r nil))
    (dolist (item l)
      (let ((os (list (stamp-of-oid (dependency-oid item)) (dependency-objc-stamp item) (dependency-data-stamp item))))
	(unless (gethash os table)
	  (setf (gethash os table) t) (push item r))))
    r))

(defun dependencies-to-term (d)
  (idependencies-term (tag-of-dependencies d)
		      (dependency-list-to-term (unique-sublist (list-of-dependencies d)))))

(defun term-to-dependencies (term)
  (new-dependencies (tag-of-idependencies-term term)
		    (term-to-dependency-list (list-of-idependencies-term term))))


(defun dependencies-sexpr-to-term (l)
  (map-sexpr-to-ilist l
		      (idependencies-nil-term)
		      #'dependencies-to-term))

(defun term-to-dependencies-list (term)
  (map-isexpr-to-list term
		      (idependencies-cons-op)
		      #'term-to-dependencies))

(defun environment-dependencies-to-term (env-d)
  (if env-d
      (ienvironment-dependencies-term (stamp-to-term (stamp-of-stamp-dependencies env-d))
				      (map-sexpr-to-ilist (list-of-stamp-dependencies env-d)
							  (idependencies-nil-term)
							  #'dependencies-to-term))
      (ivoid-term)))

(defun environment-dependencies-sexpr-to-term (l)
  (map-sexpr-to-ilist l
		      (ienvironment-dependencies-nil-term)
		      #'environment-dependencies-to-term))


(defun walk-event-dependencies (f evtdeps)
  (unless (null evtdeps)
    (dolist (envdep (list-of-stamp-dependencies evtdeps))
      (dolist (deps (list-of-stamp-dependencies envdep))
	;;(setf wed (list-of-dependencies deps)) (break "wed")
	(walk-p #'(lambda (d)
		    (funcall f d)
		    t)
		(list-of-dependencies deps))))))

(defun null-event-dependencies-p (evtdeps)
  (or (null evtdeps)
      (forall-p #'(lambda (envdep)
		    (let ((p t))
		      (forall-p #'(lambda (deps)
				    (walk-p #'(lambda (d)
						(declare (ignore d))
						(setf p nil) ;indicate one found.
						;; return nil stops walk-p
						nil)
					    (list-of-dependencies deps))
				    p)
				(list-of-stamp-dependencies envdep))))
		(list-of-stamp-dependencies evtdeps))))

(defun term-to-environment-dependencies (term)
  (unless (ivoid-term-p term)
    (environment-dependencies (term-to-stamp (stamp-of-ienvironment-dependencies-term term))
			      (map-isexpr-to-list (list-of-ienvironment-dependencies-term term)
						  (idependencies-cons-op)
						  #'term-to-dependencies))))


(defun event-dependencies-to-term (evt-d)
  (if evt-d
      (ievent-dependencies-term (stamp-to-term
				 (stamp-of-stamp-dependencies evt-d))
				(description-to-term
				 (description-of-event-dependencies evt-d))
				(map-sexpr-to-ilist (list-of-stamp-dependencies evt-d)
						    (ienvironment-dependencies-nil-term)
						    #'environment-dependencies-to-term))
      (ivoid-term)))


(defun term-to-event-dependencies (term)
  (unless (ivoid-term-p term)
    (event-dependencies (term-to-stamp
			 (stamp-of-ievent-dependencies-term term))
			(term-to-description
			 (description-of-ievent-dependencies-term term))
			(map-isexpr-to-list (list-of-ievent-dependencies-term term)
					    (ienvironment-dependencies-cons-op)
					    #'term-to-environment-dependencies))))


(defun equal-dependencies-p (a b)

  ;;(setf -a a -b b) (break "edp")
  (let ((dsa (data-stamp-of-dependency a))
	(dsb (data-stamp-of-dependency b)))

    (if (and dsa dsb)
	(equal-stamps-p dsa dsb)

	(let ((osa (objc-stamp-of-dependency a))
	      (osb (objc-stamp-of-dependency b)))

	  (if (and osa osb)
	      (equal-stamps-p osa osb)

	      (let ((oida (oid-of-dependency a))
		    (oidb (oid-of-dependency b)))

		(if (and oida oidb)
		    (equal-oids-p oida oidb)

		    nil)))))))



;; (and (walk-p #'(lambda (d) (walk-p b #'(lambda (e) (equal-dependencies-p e d)))) a)
;;      (walk-p #'(lambda (d) (walk-p a #'(lambda (e) (equal-dependencies-p e d)))) b)) 

(defun equal-dependency-sexprs-p (a b)
  (or (eq a b)
      (let ((aa nil)
	    (bb nil)
	    (aa-len 0)
	    (bb-len 0))
	(walk-p #'(lambda (d) (unless (member d aa :test #'equal-dependencies-p)
				  (push d aa)
				  (incf aa-len)
				  t))
		a)
	(and (walk-p #'(lambda (d) (unless (member d bb :test #'equal-dependencies-p)
				       (push d bb)
				       (incf bb-len)
				       (member d aa :test #'equal-dependencies-p)))
		     b)
	     ;; Know: no duplicates in aa or bb, and bb is subset of aa.
	     (= aa-len bb-len)))))

;; may assume no duplicates in lists and that lists are nil terminated.
(defun equal-dependency-lists-p (a b)
  (or (eq a b)
      (let ((a-len 0))
	(and (walk-p #'(lambda (d)
			   (when (member d b :test #'equal-dependencies-p)
			     (incf a-len)
			     t))
		     a)
	     ;; Know: a is subset of b.
	     (= a-len (length b))))))





;; assumes stores is nil terminated list.
;; returns sorted list of stores where there are no
;;  - no duplicate store tags.
;;  - dependencies in stores are nil terminated lists with no duplicates.
;;  - dependencies in stores are not null.

;;;;	Ignore environment and event and combine and reduce lists to normal-list.
;;;;	
;;;;	combine and reduce within env and then combine and reduce amongst all envs.
;;;;	likely to be single env.
;;;;	
;;;;	

;;;;	 <dependencies-list> : (<tok{tag}> . <dependency> sexpr) list
;;;;
;;;;	 <n-dependencies-list> : (<tok{tag}> . <dependency> list) list
;;;;	    - tag does not occur more than once.
;;;;	
;;;;	normalize-dependencies (<dependencies-list>) 	: <n-dependencies-list>
;;;;	  * in result :
;; perf todo : ability to mark dependencies would help here. Worry about MTT with marks though.

(defun normalize-dependencies (dependencies)
  (let (;; sort dependencies by type.
	(sorted (sort dependencies
		      #'string-lessp
		      :key #'(lambda (s) (string (tag-of-dependencies s))))))

    ;; combine duplicate stores
    (let ((acc nil))	
      (dolist (d sorted)
	(cond
	  ((null acc) (push (cons (tag-of-dependencies d)
				  (list-of-dependencies d))
			    acc))
	  ((eql (caar acc) (tag-of-dependencies d))
	   (push (list-of-dependencies d) (cdar acc))
	   ;;(append (list-of-dependencies d) (dependencies-list (car acc)))
	   )

	  (t (push (cons (tag-of-dependencies d)
			 (list-of-dependencies d))
		   acc))))
      
      ;;(setf -hacc acc) (break "h")
      ;; remove duplicate dependencies.
      (let ((acc2 nil))
	(dolist (ds acc)
	  (let ((dependencies (remove-duplicate-dependencies  (cdr ds))))
	    (when dependencies
	      (push (new-dependencies (car ds) dependencies)
		    acc2))))

	acc2))))

;; normalized store contains extra field containing <n-dependencies-list>
;; of all dependencies in store.
(defun normalize-dependency-store (store)
  
  (let ((acc nil))
    
    (dolist (evt-d (list-of-dependency-store store))
      (let ((event-dependencies-list (list-of-stamp-dependencies evt-d)))
	(cond
	  ((null event-dependencies-list)
	   nil)

	  ((and nil (null (cdr event-dependencies-list)))
	   (let ((env-d (car event-dependencies-list)))
	     (dolist (d (setf (stamp-dependencies-list env-d)
			      (normalize-dependencies (list-of-stamp-dependencies env-d))))
	       (push d acc))))

	  (t 
	   (dolist (env-d event-dependencies-list)
	     ;;n(setf -ll (list-of-stamp-dependencies env-d)) (break "nds1")
	     (dolist (d (setf (stamp-dependencies-list env-d)
			      (normalize-dependencies (list-of-stamp-dependencies env-d))))
	       (push d acc)))
	   ))))

    ;;(setf -acc acc) (break "nds")
    (set-dependency-store-normal-list (if (or (null acc) (null (cdr acc)))
					  acc
					  (normalize-dependencies acc))
				      store)))



(defun dependencies-of-dependency-store (store)
  (or (normal-list-of-dependency-store store)
      (normalize-dependency-store store)))

(defun tagged-dependencies-of-dependency-store (tag store)
  (find tag (dependencies-of-dependency-store store)
	:key #'tag-of-dependencies))



;; expects normalized stores.
(defun equal-dependency-stores-p (a b)
  (apply-predicate-to-list-pair (dependencies-of-dependency-store a)
				(dependencies-of-dependency-store b)
				#'(lambda (a b)
				    (or (eq a b)
					(and (eql (car a) (car b))
					     (equal-dependency-lists-p (cdr a)
								       (cdr b)))))))


(defvar *dependency-event*)
(defvar *dependency-event-tags*)  
(defvar *dependency-event-description*)  
(defvar *dependency-environment*)  

(defmacro with-dependency-event ((tags &optional desc) &body body)
  `(let ((*dependency-event* nil)
	 (*dependency-event-tags* ,tags)
	 (*dependency-event-description* ,desc))
    ,@body))

(defun dependency-note-environment (env-dep)
  (if (boundp '*dependency-event*)
      (push env-dep *dependency-event*)
      (message-emit (warn-message '(dependency environment event not)))))

(defun flush-dependencies ()
  (when (and (boundp '*dependencies*) (not (eql 'not *dependencies*)))
    (prog1
	(dependency-note-dependencies *dependencies*)
      (setf *dependencies* nil))))

(defun flush-dependency-environment (&optional notnotep nil-too)

  (when (and (boundp '*dependencies*) *dependencies*)
    (flush-dependencies))

  (when (and (boundp `*dependency-environment*) (or nil-too *dependency-environment*))
    (let ((deps (environment-dependencies-normal (stamp-of-environment (current-environment))
						 *dependency-environment*)))
      (setf *dependency-environment* nil)
      (if notnotep
	  deps
	  (dependency-note-environment deps)))))



(defun event-dependencies-collected (desc stamp)
  
  (when (boundp '*dependency-environment*)
    (flush-dependency-environment))
  
  (if (boundp '*dependency-event*)
      (prog1 (event-dependencies stamp desc *dependency-event*)
	(setf *dependency-event* nil))
      (message-emit (warn-message '(dependency event not)))))


(defun any-dependencies-collected-p ()
  (or (and (boundp '*dependencies*)
	   *dependencies*)

      (and (boundp '*dependency-environment*)
	   *dependency-environment*)

      (and (boundp '*dependency-event*)
	   (let ((evt-d *dependency-event*))
	     (and evt-d
		  (exists-p #'(lambda (env-d)
				(and env-d
				     (not (null (list-of-stamp-dependencies env-d)))))
			    evt-d))))))


(defun collecting-environment-dependencies-p ()
  (or (boundp '*dependencies*)
      (boundp '*dependency-environment*)))


;; flushes dependencies and returns. Does not allow dependencies
;; to perc up to event collector.
(defun maybe-environment-dependencies-collected (&optional nil-too)
  ;;(setf -a desc) (break "medc")
  (when (collecting-environment-dependencies-p)
    (flush-dependency-environment t nil-too)))

(defun environment-dependencies-collected (stamp &optional nil-too)
  (let ((env-d (maybe-environment-dependencies-collected nil-too)))
    (when env-d
      (environment-dependencies stamp (list-of-stamp-dependencies env-d)))))


(defun maybe-event-dependencies-collected (desc stamp)
  ;;(setf -a desc) (break "medc")
  (when (any-dependencies-collected-p)
    (event-dependencies-collected desc stamp)))


(defun dependency-note-dependencies (dependencies)
  (if (boundp '*dependency-environment*)
      (setf *dependency-environment*
	    (append dependencies *dependency-environment*))
      (progn
	;;(when (noting-dependencies))
	;;(setf -d dependencies)
	(break "dnd")
	(message-emit (warn-message '(dependency dependencies environment not))))))

(defmacro with-dependency-environment (&body body)
  `(let ((*dependency-environment* nil))
    (prog1 (progn ,@body)
      (when *dependency-environment*
	(dependency-note-environment
	 (environment-dependencies-normal (stamp-of-environment (current-environment))
					  *dependency-environment*))))))


;; (funmlcall (ml-text "view_oid") (mapcar #'oid-of-dependency (lastn 5 (dependencies-list (cadr (environment-dependencies-list -envd))))))

(defun dependencies ()
  (if (boundp '*dependencies*)
      *dependencies*
      (message-emit (warn-message '(dependencies not)))))

(defvar *ephemeral-dependencies-stack* nil)

(defmacro ephemeral-dependencies-p ()
  `(and *ephemeral-dependencies-stack* t))

(defun ephemeral-dependencies  ()
  (if (consp *ephemeral-dependencies-stack*)
      (car *ephemeral-dependencies-stack*)
      (progn (break)
      (raise-error (error-message '(ephemeral-dependencies not))))))

(defmacro with-ephemeral-dependencies (&body body)
  `(let ((*ephemeral-dependencies-stack* (cons nil *ephemeral-dependencies-stack*)))
    ,@body))

(defun add-dependencies (dependencies tag deps)
  (let ((a (find-first #'(lambda (dp)
			   (when (eql (tag-of-dependencies dp) tag)
			     dp))
		       dependencies)))
    (if a
	(progn
	  (when (and nil (> (length (dependencies-list a)) 1000))
	    ;;(setf -a a)
	    (break "dnr"))
	  (setf (dependencies-list a) (append deps (dependencies-list a)))
	  dependencies)
	(cons (new-dependencies tag deps) dependencies))))

;; method of exporting dependencies under a new name to outer contexts.
(defun note-ephemeral-dependencies (tag deps)
  ;;(format t "dnr ~a~%" tag)
  ;;(when (eql tag 'statement) (setf -tag tag -d d) (break "dnr"))
  (when *ephemeral-dependencies-stack*
    (setf *ephemeral-dependencies-stack*
	  (cons (car *ephemeral-dependencies-stack*)
		(mapcar #'(lambda (ed)
			    (add-dependencies ed tag deps))
			(cdr *ephemeral-dependencies-stack*))))))	      

;;;;	
;;;;	Quick dep ref : 
;;;;	
;;;;	  definition contains an array index
;;;;	  no definition contains same index. ie each insertion increments.
;;;;	  in tables with a lot of activity there may be a lot of holes. 
;;;;	
(defun add-dependency (dependencies tag d)
  (let ((a (find-first #'(lambda (dp)
			   (when (eql (tag-of-dependencies dp) tag) dp))
		       dependencies)))
    (if a
	(progn
	  (when (and nil
		     (> (length (dependencies-list a)) 1000))
	    ;;(setf -a a)
	    (break "dnr")
	    )
	  (push d (dependencies-list a))
	  dependencies)
	(cons (new-dependencies tag (list d)) dependencies))))  

(defun dependency-note-quick-reference (tag def)
  (when (boundp '*dependencies*)
    (unless (eql 'not *dependencies*)
      (let ((a (or (find-first #'(lambda (dp)
				   (when (eql (tag-of-dependencies dp) tag) dp))
			       *dependencies*)
		   (let ((a (new-quick-dependencies tag (table-containing-of-definition def))))
		     (setf  *dependencies* (cons a *dependencies*))
		     a))))
	(quick-dependency a
			  (quickdepref-index-of-definition def)
			  (dependency-of-definition def))))))


(defun dependency-note-reference (tag d)
  ;;(format t "dnr ~a~%" tag)
  ;;(when (eql tag 'statement) (setf -tag tag -d d) (break "dnr"))
  (if (boundp '*dependencies*)
      (unless (eql 'not *dependencies*)
	;;(without-dependencies
	;;(when (and (eql tag 'access-definition-object-id)
	;;   (let ((oid (oid-of-dependency d)))
	;;   (and oid (eql 'INF (kind-of-oid oid)))))
	;;(break "dnr")
	;;))
	(setf *dependencies* (add-dependency *dependencies* tag d))
	(setf *ephemeral-dependencies-stack*
	      (mapcar #'(lambda (ed)
			  (add-dependency ed tag d))
		      *ephemeral-dependencies-stack*)))	      
      (progn
	;;(setf -tag tag -d d) (break "dnr")
	(message-emit (warn-message '(dependencies not note) tag)))))

(defmacro with-dependencies (&body body)
  `(let ((*dependencies* (cond
			   ((not (boundp `*dependencies*)) nil)
			   ((eql 'not *dependencies*) 'not)
			   (t nil))))
    (multiple-value-prog1 (progn ,@body)
      (when (and *dependencies* (not (eql 'not *dependencies*)))
	(dependency-note-dependencies *dependencies*)))))



;;;;
;;;;	current dependency : during import the dependency of the item being imported.
;;;;

(defvar *current-dependency*)

(defmacro with-dependency ((d tags) &body body)
  `(let ((*current-dependency* ,d))
    (with-handle-error ((,tags nil #'(lambda (m)
				       (oid-message (list (oid-of-dependency *current-dependency*))
						   nil
						   m)))
			(raise-error nil))
      ,@body)))

(defun current-dependency ()
  (unless (boundp '*current-dependency*)
    ;;(format t "DU") ;;(break)
    (system-error (error-message '(current-dependency unbound))))

  *current-dependency*)

(defun current-object-id ()
  (oid-of-dependency (current-dependency)))


(defun environment-dependencies-collected-term ()
  ;;(break "edct")
  (let ((env-d (maybe-environment-dependencies-collected)))
    (if env-d
	(environment-dependencies-to-term env-d)
	(ivoid-term))))

;; takes snapshot of current collection
(defun event-dependencies-collected-term ()
  ;;(break "edct")
  (let ((evt-d (maybe-event-dependencies-collected
		(or *dependency-event-description*
		    (new-event-description *system* *version* *dependency-event-tags*))
		(transaction-stamp))))
    (if evt-d
	(event-dependencies-to-term evt-d)
	(ivoid-term))))
				     
	  
(defmacro with-dependencies-vertical ((tags) &body body)
  `(with-dependency-event (,tags)
    (with-dependency-environment
	(with-dependencies
	    ,@body))))


(defmacro with-default-dependencies ((tok &optional suppressp) &body body)
  `(with-dependencies-vertical ('(,tok))
    (multiple-value-prog1 (with-dependencies (progn ,@body))
      (when (and (not ,suppressp) (any-dependencies-collected-p))
	(let ((evt-d-term (event-dependencies-collected-term)))
	  ;;(setf -edt evt-d-term) (break "wdd2")
	  (message-emit-asynch
	   (inform-message (list 'dependencies 'default ,tok)
			   evt-d-term)))))))



;; oa as prim type ? not here not now.
;;  (add-primitive-type '|object_address|
;;		      #'(lambda (oa) (object-address-to-string oa))
;;		      :member-p #'(lambda (oa) (object-address-p oa))
;;		      :eq-func #'(lambda (a b) (equal-object-address-p a b)))

(define-primitive |!property| ((token . tag)) (term))

(defun properties-to-term (props)
  (map-list-to-isexpr props
		      (inil-term)
		      #'(lambda (prop)
			  (iproperty-term (car prop)
					  (cdr prop)))))

(defun term-to-properties (term)
  (map-isexpr-to-list term
		      (icons-op)
		      #'(lambda (iprop)
			  (cons (tag-of-iproperty-term iprop)
				(term-of-iproperty-term iprop)))))


;; proxy for shared terms on disk.

(defclass term-data (data)
  ((term :reader term-of-term-data
	 :writer set-term-data-term
	 :initform nil
	 :initarg term)))

(define-primitive |!term_data| () (term))

(defmethod data-import ((term-data term-data) super)
  (let ((term (call-next-method term-data super)))
    (set-term-data-term term term-data)
    ))

(defmethod data-export ((term-data term-data) sub)
  (declare (ignore sub))
  (iterm-data-term (term-of-term-data term-data)))


(defun term-data (term)
  (make-instance 'term-data 'term term))



(defclass substance (data)
  (
   ;; source-reduced source.
   (term :reader term-of-substance
	 :writer set-substance-term
	 :initform nil
	 :initarg term)

   ;; dependencies made in second translation stage.
   (dependencies :reader dependencies-of-substance
		 :writer set-substance-dependencies
		 :initarg dependencies
		 )

   (properties :reader properties-of-substance
	       :writer set-substance-properties
	       :initform nil
	       :initarg properties)
  ))

(define-primitive |!substance| () (term dependencies properties sub))

(defun property-of-substance (sub tag)
  (cdr (assoc tag (properties-of-substance sub))))

(defun name-property-of-substance (sub)
  (let ((p (property-of-substance sub 'name)))
    (when p
      (if (itoken-term-p p)
	  (token-of-itoken-term p)
	  (progn (message-emit (warn-message '(substance property name !token not) p))
		 nil)))))
  
(defmethod data-import ((substance substance) super)
  (let ((term (call-next-method substance super)))
    (set-substance-term (term-of-isubstance-term term) substance)
    (set-substance-dependencies (term-to-event-dependencies
				 (dependencies-of-isubstance-term term))
				substance)
    (set-substance-properties (term-to-properties
			       (properties-of-isubstance-term term))
			      substance)

    (sub-of-isubstance-term term)))

(defmethod data-export ((substance substance) sub)
  ;;(when (eql (type-of substance) 'proof-substance) (setf -a substance -b sub) (break "psp"))
  (call-next-method substance
		    (isubstance-term (term-of-substance substance)
				     (event-dependencies-to-term
				      (dependencies-of-substance substance))
				     (properties-to-term
				       (properties-of-substance substance))
				     sub)))


(defun substance (term)
  (make-instance 'substance 'term term))

(defun prop-substance (term properties)
  (make-instance 'substance 'term term 'properties properties))



;;;;	producer state snap
;;;;	

;; snap func should be a symbol so that recompilation is caught by function-symbol.
(defvar *producer-state-snap-alist* nil)

(defun add-producer-state-snap-func (purpose f)
  (setf *producer-state-snap-alist* (acons purpose f *producer-state-snap-alist*)))

(defun lookup-producer-state-snap-funcs (purposes)
  (let ((acc nil))
    (dolist (p purposes)
      (let ((snap (cdr (assoc p *producer-state-snap-alist*))))
	(when snap (push (symbol-function snap) acc))))
    acc))


(defstruct refine-stats
  time

  ;; dynamically observed
  tactic-calls	;; primitives attempted.
  rule-calls	;; primitives attempted.
  allocation	;; sequent/proof structures allocated.

  ;; statically computed. 
  tactic	;; count of tactic nodes in result proof.
  total		;; total proof nodes in result proof.

  deepest	;; deepest tactic tree (wrt proof-top). 
  largest	;; largest embedded.
  longest 	;; longest assumption list
  )

(defun time-of-refine-stats (r) (refine-stats-time r))
(defun tactic-calls-of-refine-stats (r) (refine-stats-tactic-calls r))
(defun rule-calls-of-refine-stats (r) (refine-stats-rule-calls r))
(defun allocation-of-refine-stats (r) (refine-stats-allocation r))
(defun tactic-of-refine-stats (r) (refine-stats-tactic r))
(defun total-of-refine-stats (r) (refine-stats-total r))
(defun largest-of-refine-stats (r) (refine-stats-largest r))
(defun longest-of-refine-stats (r) (refine-stats-longest r))
(defun deepest-of-refine-stats (r) (refine-stats-deepest r))

(defun new-refine-stats (time rule-count tactic-count node-count)
  (make-refine-stats :time time
		     :tactic-calls tactic-count
		     :rule-calls rule-count
		     :allocation node-count))

(defun new-refine-stats-acc ()
  (make-refine-stats :time (new-time-stats 0 0 0 0 0 0 0)
		     :tactic-calls 0 :rule-calls 0 :allocation 0
		     :tactic 0 :total 0 :largest 0 :longest 0 :deepest 0
		     ))

(defun update-refine-stats (rstats total tactic largest longest deepest)
  (setf (refine-stats-total rstats)	total
	(refine-stats-tactic rstats)	tactic
	(refine-stats-largest rstats)	largest
	(refine-stats-longest rstats)	longest
	(refine-stats-deepest rstats)	deepest))

(defun time-stats-sum (a b)
  (make-time-stats
   :etime	(+ (elapsed-of-time-stats a) (elapsed-of-time-stats b))
   :cputime	(+ (cpu-of-time-stats a) (cpu-of-time-stats b))
   :pfaults	(+ (page-faults-of-time-stats a) (page-faults-of-time-stats b))
   :dyncons	(+ (dynamic-consing-of-time-stats a) (dynamic-consing-of-time-stats b))
   :dyngc	(+ (dynamic-gc-of-time-stats a) (dynamic-gc-of-time-stats b))
   :ephcons	(+ (ephemeral-consing-of-time-stats a) (ephemeral-consing-of-time-stats b))
   :ephgc	(+ (ephemeral-gc-of-time-stats a) (ephemeral-gc-of-time-stats b))
   ))


(defun refine-stats-sum (a b)
  (make-refine-stats
   :time (time-stats-sum (time-of-refine-stats a) (time-of-refine-stats b))
   :tactic-calls (+ (tactic-calls-of-refine-stats a) (tactic-calls-of-refine-stats b))
   :rule-calls (+ (rule-calls-of-refine-stats a) (rule-calls-of-refine-stats b))
   :allocation (+ (allocation-of-refine-stats a) (allocation-of-refine-stats b))
   :tactic (+ (tactic-of-refine-stats a) (tactic-of-refine-stats b))
   :total (+ (total-of-refine-stats a) (total-of-refine-stats b))
   :largest (max (largest-of-refine-stats a) (largest-of-refine-stats b))
   :longest (max (longest-of-refine-stats a) (longest-of-refine-stats b))
   :deepest (max (deepest-of-refine-stats a) (deepest-of-refine-stats b))))


(defun time-stats-acc (acc b)
  (incf (time-stats-etime acc) (elapsed-of-time-stats b))
  (incf	(time-stats-cputime acc) (cpu-of-time-stats b))
  (incf	(time-stats-pfaults acc) (page-faults-of-time-stats b))
  (incf	(time-stats-dyncons acc) (dynamic-consing-of-time-stats b))
  (incf	(time-stats-dyngc acc) (dynamic-gc-of-time-stats b))
  (incf (time-stats-ephcons acc) (ephemeral-consing-of-time-stats b))
  (incf (time-stats-ephgc acc) (ephemeral-gc-of-time-stats b)))

(defun refine-stats-acc (acc b)
  (time-stats-acc (time-of-refine-stats acc) (time-of-refine-stats b))
  (incf (refine-stats-tactic-calls acc) (tactic-calls-of-refine-stats b))
  (incf (refine-stats-rule-calls acc) (rule-calls-of-refine-stats b))
  (incf (refine-stats-allocation acc) (allocation-of-refine-stats b))
  (incf (refine-stats-tactic acc) (tactic-of-refine-stats b))
  (incf (refine-stats-total acc) (total-of-refine-stats b))
  (setf (refine-stats-largest acc) (max (largest-of-refine-stats acc) (largest-of-refine-stats b))
	(refine-stats-longest acc) (max (longest-of-refine-stats acc) (longest-of-refine-stats b))
	(refine-stats-deepest acc) (max (deepest-of-refine-stats acc) (deepest-of-refine-stats b))))


;; need to be able to sum stats.
;; either elapsed/cpu need to be converted to strings or micro/milli seconds.
(define-primitive !time_stats ((natural . elapsed)
			       (natural . cputime)
			       (natural . pfaults)
			       (natural . dyncons)
			       (natural . dyngc)
			       (natural . ephcons)
			       (natural . ephgc)))

(defun time-stats-to-term (tstats)
  (itime-stats-term (elapsed-of-time-stats tstats)
		    (cpu-of-time-stats tstats)
		    (page-faults-of-time-stats tstats)
		    (dynamic-consing-of-time-stats tstats)
		    (dynamic-gc-of-time-stats tstats)
		    (ephemeral-consing-of-time-stats tstats)
		    (ephemeral-gc-of-time-stats tstats)
  ))

(defun term-to-time-stats (term)
  (make-time-stats :etime (elapsed-of-itime-stats-term term)
		   :cputime (cputime-of-itime-stats-term term)
		   :pfaults (pfaults-of-itime-stats-term term)
		   :dyncons (dyncons-of-itime-stats-term term)
		   :dyngc (dyngc-of-itime-stats-term term)
		   :ephcons (ephcons-of-itime-stats-term term)
		   :ephgc (ephgc-of-itime-stats-term term)))

(define-primitive !refine_stats ((natural . tactic-calls)
				 (natural . rule-calls)
				 (natural . allocation)
				 (natural . tactic)
				 (natural . total)
				 (natural . largest)
				 (natural . longest)
				 (natural . deepest))
  (time))

(defun refine-stats-to-term (rstats)
  (irefine-stats-term
   (tactic-calls-of-refine-stats rstats)
   (rule-calls-of-refine-stats rstats)
   (allocation-of-refine-stats rstats)
   (tactic-of-refine-stats rstats)
   (total-of-refine-stats rstats)
   (largest-of-refine-stats rstats)
   (longest-of-refine-stats rstats)
   (deepest-of-refine-stats rstats)

   (time-stats-to-term (time-of-refine-stats rstats))))
		      

(defun term-to-refine-stats (term)
  (make-refine-stats :time (term-to-time-stats (time-of-irefine-stats-term term))
		     :tactic-calls (tactic-calls-of-irefine-stats-term term)
		     :rule-calls (rule-calls-of-irefine-stats-term term)
		     :allocation (allocation-of-irefine-stats-term term)
		     :tactic (tactic-of-irefine-stats-term term)
		     :total (total-of-irefine-stats-term term)
		     :largest (largest-of-irefine-stats-term term)
		     :longest (longest-of-irefine-stats-term term)
		     :deepest (deepest-of-irefine-stats-term term)
		     ))

(defun num-nodes-to-string (num) (num-to-string num "N"))

(defun report-refine-stats (stream stats &optional (descriptor))

  (let ((tstats (time-of-refine-stats stats))
	(total (total-of-refine-stats stats)))

    (report-time-stats stream tstats descriptor)

    (format stream
	    "  Total nodes : ~a, Tactic nodes : ~a, Allocation : ~a.~%"
	    (num-nodes-to-string (total-of-refine-stats stats))
	    (num-nodes-to-string (tactic-of-refine-stats stats))
	    (num-nodes-to-string (allocation-of-refine-stats stats)))

    (format stream
	    "  Rule calls :  ~a, Tactic calls : ~a.~%"
	    (num-to-string (rule-calls-of-refine-stats stats))
	    (num-to-string (tactic-calls-of-refine-stats stats)))

    (unless (zerop total)
      (format stream
	      "  Largest primitive tree : ~a, Longest assumption list : ~a,~%"
	      (num-nodes-to-string (largest-of-refine-stats stats))
	      (num-to-string (longest-of-refine-stats stats)))

      (format stream
	      "  Deepest Recursive Tactic Tree ~a.~%"
	      (num-to-string (deepest-of-refine-stats stats))))

    (let ((n (- total (tactic-of-refine-stats stats))))
      (when (> n 0)
	(format stream
		"~%  Consing per primitive ~a, CPU Time per primitive ~a.~%~%"
		(num-bytes-to-string (round (/ (+ (dynamic-consing-of-time-stats tstats)
						  (ephemeral-consing-of-time-stats tstats))
					       n)))
		(microseconds-to-string (round (/ (cpu-of-time-stats tstats)
						  n))))))))

(defun report-hdr (s descriptor1 descriptor2)
  (format s "~%~%  ****  ~%  ****  ~a : ~a.~%  ****  ~%  ****  ~a~%  ****  ~%~%"
	  descriptor1 descriptor2 (datetime-string (get-universal-time))))





(defconstant *library-object-content-types*  '(abs com rule code inf prf stm prec disp term))

(defun ichar->char (ich)
  (cond ((not (numberp ich))
	 #\space)
	((eql ich inewline)
	 #\newline)
	((eql ich itab)
	 #\tab)
	((code-char ich))))

(defun implode-to-string (x)
  (let* ((l (mapcan #'(lambda (y)
			(if (integerp y)
			    (list (ichar->char y))
			    (coerce (string y) 'list)))
		    x))
	 (len (length l))
	 (s (make-string len)))

      (dotimeslist (i c l)
	(setf (aref s i) c))
      s))


(defun edit-fix-istring (istr)
  (mapcan #'(lambda (ich) (if (< ich 32)
			      (progn
				(format t "~%Removing ~a from ~a~%" ich (implode-to-string istr))
				;; or could \0010 
				nil)
			      (list ich)))
	  istr))

(defun term-fix-text (term)
  (term-walk-ops term
		 #'(lambda (term)
		     (maybe-instantiate-term
		      term
		      (instantiate-operator
		       (id-of-term term)
		       (mapcar #'(lambda (p)
				   ;;(setf -p p) (break "tft")
				   (if (and (real-parameter-p p)
					    (or (token-parameter-p p)
						(string-parameter-p p)))
				       (instantiate-parameter-s
					(implode-to-string (edit-fix-istring
							    (istring
							     (value-of-parameter-r p))))
					(type-of-parameter p))
				       p))
			       (parameters-of-term term)))
		      (bound-terms-of-term term)))))
	     


;;;;	
;;;;	
;;;;	
;;;;	
;;;;	


(defstruct layer-dag
  flags
  oid
  refs					; referenced oids. Consider pointer to other dags.
					;  ...  but they may change.
  ref-dags
  stamp
  index					; height of layer.
  )
  
(defun new-layer-dag (oid oids)
  (make-layer-dag :oid oid :refs oids :index nil :ref-dags nil))

(defun oid-of-layer-dag (d) (layer-dag-oid d))
(defun refs-of-layer-dag (d) (layer-dag-refs d))
(defun ref-dags-of-layer-dag (d) (layer-dag-ref-dags d))
(defun stamp-of-layer-dag (d) (layer-dag-stamp d))
(defun index-of-layer-dag (d) (layer-dag-index d))

(define-flags (layer-dag)
    ((undoable nil t)
     (stale nil t)))

(defstruct (def-dag (:include layer-dag))
  definition
  )
  
(defun new-layers (init-f &optional ignore-undoable)
  (let ((i 0)
	(table (make-hash-table :test #'equal))
	(todo nil)
	(undoable nil) ;; references inaccessible object (directly or indirectly).
	(progress-p nil)
	)

    ;; init
    (funcall init-f
	     #'(lambda (dag)
		 (hashoid-set table (oid-of-layer-dag dag) dag)
		 
		 (if (null (refs-of-layer-dag dag))
		     (progn (setf progress-p t)
			    (setf (layer-dag-index dag) 0)
			    (setf (layer-dag-ref-dags dag) nil))
		     (progn (setf (layer-dag-index dag) nil)
			    (push dag todo) ))))
			   
    ;; tis possible that a reference will resolve to nil, ie there is no associated layer dag.
    ;; could happen due to some referenced object being inactive thus we would desire
    ;; that that the referencers be left out of the layers. Ie, dependents of inactive objects
    ;; are not layered.

    ;;(setf -todo todo -table table) (break "todo")
    (setf todo (if ignore-undoable
		   (mapcan #'(lambda (dag)
			       (format t "#")
			       (layer-dag-flag-set-stale dag nil)
			       (layer-dag-flag-set-undoable dag nil)
			       (let ((dags (setf (layer-dag-ref-dags dag)
						 (mapcan #'(lambda (oid)
							     (let ((d (hashoid-get table oid)))
							       (when d (list d))))
							 (refs-of-layer-dag dag)))))
				 (format t ",")
				 (if dags
				     (list dag)
				     (progn
				       (setf progress-p t)
				       (setf (layer-dag-index dag) 0)
				       nil))))
			   todo)
		   (mapcan #'(lambda (dag)
			       (layer-dag-flag-set-stale dag nil)
			       (setf (layer-dag-ref-dags dag)
				     (mapcar #'(lambda (oid) (hashoid-get table oid))
					     (refs-of-layer-dag dag)))
			       (if (exists-p #'null (ref-dags-of-layer-dag dag))
				   (progn
				     (setf progress-p t)
				     (layer-dag-flag-set-undoable dag t)
				     (push dag undoable)
				     nil)
				   (progn
				     (layer-dag-flag-set-undoable dag nil)
				     (list dag))))
			   todo)))

    (do ()
	((not progress-p))

      (incf i)
      (setf progress-p nil)

      (format t ";;; layer ~a, todo ~a, undoable ~a, total ~a.~%" i (length todo) (length undoable) (hash-table-count table))
      ;;(setf -todo todo -table table -undoable undoable) (break "layer")

      (let ((ntodo nil))
	(dolist (dag todo)
	  (cond
	    
	    ((forall-p #'(lambda (rdag)
			   (let ((j (index-of-layer-dag rdag)))
			     (or (and j (< j i))
				 ;; don't count self reference as cycle???
				 (eql rdag dag))))			
		       (ref-dags-of-layer-dag dag))
	     (setf progress-p t)
	     (setf (layer-dag-index dag) i))

	    ((exists-p #'layer-dag-flag-undoable-p
		       (ref-dags-of-layer-dag dag))
	     (layer-dag-flag-set-undoable dag t)	     
	     (setf progress-p t)
	     (push dag undoable))
	    
	    (t (push dag ntodo))))

	(setf todo ntodo)))
	
    (values table todo undoable)))


;; removes edges where dependency is not present in graph
;; and reflexive dependencies.
(defun graph-reduce-proxy (equal g)
  (let ((dependents (mapcar #'caar g))
	(subs (mapcan #'(lambda (e)
			  (let* ((a (caar e))
				 (proxies (filter #'(lambda (b) (not (funcall equal b a))) (cdar e))))
			    (when proxies
			      (mapcar #'(lambda (x) (cons x a))
				      proxies))))
		      g)))

    ;;(setf -subs subs) (break "grp")
    (mapcar #'(lambda (e)
		(let ((a (caar e)))
		  (cons a
			(mapcan #'(lambda (d)
				    (let ((b (or (cdr (assoc d subs :test equal))
						 d)))
				      ;;(setf -b b -d d -dependents dependents)
				      (when (and (not (funcall equal b a))
						 (member b dependents :test equal))
					(list b))))
				(cdr e)))))
	    g)))



(defun graph-layers (g)
  (mlet* (((table cycles) (new-layers #'(lambda (acc)
					  (dolist (e g)
					    (format t ".")
					    (funcall acc (new-layer-dag (car e) (cdr e)))))
				    t)))

	 ;;(setf -table table -cycles cycles)
	 ;;(break "gl i")
	 (let ((ll nil))
	   (maphash #'(lambda (k dag)
			(declare (ignore k))
			(let ((i (index-of-layer-dag dag)))
			  (when i
			    (let ((l (assoc i ll)))
			      (if l
				  (setf (cdr l) (cons (oid-of-layer-dag dag) (cdr l)))
				  (setf ll (acons i (list (oid-of-layer-dag dag)) ll)))))))

		    table)
	   
	   (values (mapcar #'cdr (sort ll #'< :key #'car))
		   (mapcar #'oid-of-layer-dag cycles)))))



(defun split-graph (p g)
  (let ((a nil)
	(b nil))
    
    ;; split
    (dolist (e g)
      (if (funcall p (car e))
	  (push e a)
	(push e b)))

    (cons a b)))

(defun new-oid-table ()
  (make-hash-table :test #'equal))

(defun oid-graph-path (g ghash start endp &optional stopoids)
  (let ((ohash (or ghash
		   (let ((tt (new-oid-table)))
		     (dolist (e g)
			     (hashoid-set tt (car e) (cdr e)))
		     tt)))
	(vhash (new-oid-table)))

    (dolist (o stopoids)
      (hashoid-set vhash o t))
    (labels ((visit (o p)
		    (unless (hashoid-get vhash o)
		      (hashoid-set vhash o t)
		      (let ((pp (cons o p)))
			(if (funcall endp o)
			    pp
			  (some #'(lambda (x) (visit x pp)) (hashoid-get ohash o)) ))))
	     )

	    (visit start nil))))


(defun graph-cycle (g)
  (mlet* (((l c) (graph-layers g) (declare (ignore l))))

	 (labels ((visit (a ancestors)
		    (let ((e (assoc a g :test #'equal-oids-p)))
		      (let ((na (cons (car e) ancestors)))
			(when (member-oid (car e) (cdr e))
			  (format t "self-reference~a"))
			(dolist (n (cdr e))
			  (let ((r (member-oid n ancestors)))
			    (when r
			      ;; (cdr r) are not involved.
			      (return-from graph-cycle (member-oid n (nreverse na)))))

			  (when (member-oid n c)
			    (visit n na) ))))))
	     
	   ;;(setf -l l -c c) (break)
	   (unless c
	     (raise-error (error-message '(graph-cycle none))))

	   (dolist (a c)
	     (format t "try~%")
	     (visit a nil)) 

	   ;; c contains a cycle, find it.
	   )))


#|(defun old-graph-closure (g seeds &optional filter)

  ;; collect then sort by layering.
  (let ((table (new-oid-table)))
    (dolist (e g)
	    (hashoid-set table (car e) (cdr e)))

    (let ((acc (copy-list seeds))
	  (done nil)
	  (new seeds))

      ;;(setf -g g -new new -acc acc) (break "gc")
      (do ()
	  (done)
	(setf new
	       (nset-difference
		(delete-duplicates
		 (mapcan #'(lambda (oid)
			     (copy-list (hashoid-get table oid)))
			 new)
		 :test #'equal-oids-p)
		acc))
      
	;;(setf -new new -acc acc) (break "gc2")
	(if new
	    (setf acc (append new acc)
		  ;;remaining (nset-difference remaining new :test #'equal-oids-p)
		  )
	    (setf done t)))

      ;;(setf -g g -new new -acc acc -seeds seeds) (break "gc2d")
      ;; PERF might be worth placing in hash table
      ;; sort
      acc
      )))|#

(defun graph-closure (g ghash seeds)

  ;; collect then sort by layering.
  (let ((table (or ghash
		   (let ((tt (new-oid-table)))
		     (dolist (e g)
			     (hashoid-set tt (car e) (cdr e)))
		     tt))))

    (let ((r (new-oid-table)))
      (dolist (o seeds) (hashoid-set r o o))
      
      (let ((done nil)
	    (new seeds))

      ;;(setf -g g -new new -acc acc) (break "gc")
      (do ()
	  (done)

	  (let ((nnew nil))
	    (dolist (o new)
		    (dolist (oo (hashoid-get table o))
			    (unless (hashoid-get r oo)
			      (push oo nnew)
			      (hashoid-set r oo oo))))
	    (if nnew
		(setf new nnew)
	      (setf done t))))
			    
      ;;(setf -g g -new new -acc acc -seeds seeds) (break "gc2d")
      (let ((acc nil))
	(maphash #'(lambda (k o) (declare (ignore k)) (push o acc))
		 r)
	acc) ))))

(defun graph-sort (g l &optional (cycle-ok-p t))
  (mlet* (((ll c) (oid-list-layers g l #'(lambda (a) (declare (ignore a)) t))))

	 (when (and c (not cycle-ok-p))
	   (break "gsc")
	   (raise-error (error-message '(graph-sort cycle) (icut-oid-list c))))

	 (nconc (mapcan #'(lambda (x) x) ll) c)
	 ))

;; 
(defun make-graph-dependency-closure-func (g)
  ;;(setf -g g) (break "mgdcf")

  (let ((table (let ((gtable (new-oid-table)))
		 (dolist (e g)
		   (hashoid-set gtable (car e) (cdr e)))
		 gtable)))

    (mlet* (((layers cycle) (graph-layers g)))

      (when cycle (raise-error (error-message '(graph closure cycle))))
	    
      #'(lambda (filter seeds)
	  ;; collect then sort by layering.
	  (let ((acc (let ((a (new-oid-table)))
		       (dolist (oid seeds)
			 (hashoid-set a oid oid))
		       a))
		(new seeds))

	    ;;(setf -g g -new new -acc acc) (break "gc")
	    (do ()
		((null new))
	      (let ((nn nil))
		(dolist (n new)
		  (dolist (d (hashoid-get table n))
		    (unless (hashoid-get acc d)
		      (hashoid-set acc d d)
		      (push d nn))))
	      
		;;(setf -nn nn -acc acc) (break "gc2")
		(setf new nn)
		))

	    ;;(setf -acc acc -layers layers -g g -seeds seeds -filter filter) (break "gc2")
	    ;; sort
	    (mapcan #'(lambda (l)
			(mapcan #'(lambda (a)
				    (when (and (funcall filter a)
					       (hashoid-get acc a))
				      (list a)))
				l))
		    layers))))))

(defun graph-dependency-closure (g filter seeds)
  (funcall (make-graph-dependency-closure-func g) filter seeds))

;; stable wrt ll.
;; assumes graph is complete ie all in ll must occure in g. 
(defun oid-list-layers (g ll filter)
  (mlet* (((layers cycles) (graph-layers g)))
	 (values (mapcan #'(lambda (l)
			     (let ((nn (mapcan #'(lambda (a)
						   (when (and (funcall filter a)
							      (member-oid a l))
						     (list a)))
					       ll)))
			       (when nn (list nn))))
			 layers)
		 cycles)))


(defun poto-inconsistents (po to items)
  (let ((tol (sort (copy-list items) to)))

    ;;(setf -tol tol)

    ;; pairwise check po
    (let ((inconsistent nil))
      (do ((r tol (cdr r))
	   ;; may avoid consing by looking forward from car r, but then check for later in PO
	   ;; maybe could queue earlier but then dups possible?? 
           (l nil (cons (car r) l)))
	  ((null r))

	;; inconsistent when there exists an a <t b and b <p a.
	(let ((a (car r)))
	  (format t "POTO ~a ~a~%" (length r) (length l))
	  (when (exists-p #'(lambda (b)
			      (when (funcall po a b)
				;;(setf -a a -b b)
				;;(break "poto")
				t))
			  l)
	    (push a inconsistent))))
      inconsistent
      )))

(defun duplicate-p (eq l)
  (do ((r l (cdr r)))
      ((or (null r)
	   (let ((a (car r)))
	     (exists-p #'(lambda (b) (funcall eq a b))
		       (cdr r))))
       (not (null r)))))

;; reduced graph.
;; local comparisons only.
(defun inconsistent-objects-po (g)
  (let ((pohash (new-oid-table)))

    (dolist (e g)
      (let ((a (car e)))
	(hashoid-set pohash a (cdr e))))

    #'(lambda (a b)
	(member-oid a (hashoid-get pohash b)))
    ))

;; l - ll
(defun fast-diff-oids (l ll)
  (let ((ohash (new-oid-table)))
    (dolist (o ll) (hashoid-set ohash o t))

    (let ((acc nil))
      (dolist (o l) (unless (hashoid-get ohash o) (push o acc)))
      (nreverse acc))))

(defun fast-remove-duplicate-oids (l)
  ;;(format t "remove_duplicate_oids called~%")
  (let ((ohash (new-oid-table)))

    (let ((acc nil))
      (dolist (oid l)
	(unless (hashoid-get ohash oid)
	  (push oid acc)
	  (hashoid-set ohash oid t)))

      (nreverse acc))))


(define-primitive |!negative| () ((0 . nat)))

(defun iint-term (i)
  (if (< i 0)
      (inegative-term (inatural-term (abs i)))
      (inatural-term (abs i))))

(defun int-of-iint-term (iint)
  (cond
    ((inegative-term-p iint)
     (- (int-of-iint-term (nat-of-inegative-term iint))))
	
    ((inatural-term-p iint)
     (numeral-of-inatural-term iint))

    (t (raise-error
	(error-message '(int !int not) iint)))))



(defun get-config-data (a)
  (let ((e (assoc a (mapcar #'(lambda (d) (cons (intern-system (string (car d)))
						(cdr d)))
			    (config-data)))))

    (when e
      (labels ((visit (s)
		 (cond
		   ((consp s)
		    (cons (visit (car s))
			  (visit (cdr s))))
		   ((symbolp s)
		    (intern-system (string s))		 
		    )
		   (t s))))
	
	(visit (cdr e))))))

(defun set-config-data (cd)
  (let ((cluser (find-package "CL-USER")))
    (labels ((aux (i)
	       (if (symbolp i)
		   (intern (string i) cluser)
		   (if (or (stringp i) (integerp i))
		       i
		       (if (consp i)
			   (cons (aux (car i)) (aux (cdr i)))
			   (progn (setf -i i)
				  (raise-error (error-message '(set-config-data type unexpected)))))))))


    (setf cl-user:*system-config*
	  (append (mapcar #'(lambda (d)
			      (cons
			       (if (symbolp (car d))
				   (intern (string (car d)) cluser)
				   (raise-error (error-message '(set-config-data type key token not))))
			       (aux (cdr d))))
			  cd)
		  cl-user:*system-config*)))))

(defun config-data-all ()
  (labels ((visit (s)
	     (cond
	       ((consp s)
		(cons (visit (car s))
		      (visit (cdr s))))
	       ((symbolp s)
		(intern-system (string s)))
	       (t s))))

    (mapcar #'visit (config-data))))

(defun print-config-data ()
  (format t "~%;;~%;; ConfigData : ~%;;~%")
  (mapc #'(lambda (e)
	    ;;(format t "~a~%" e)
	    (format t ";;  ~a~20,8T~a~%" (car e)
		    (if (and (consp (cdr e)) (null (cddr e)))
			(cadr e)
			(cdr e)))
	    )
	(reverse (config-data-all)))
  nil)




;;(define-primitive |!tree| () (node children))

;; walks tree, but does not accumulate.

;; addresses start at 1.
(defun walk-itree (f itree)
  (labels ((visit (addr itree)
	     (funcall f addr itree)
	     (dotimeslist (i bt (cdr (bound-terms-of-term itree)))
		  (visit (cons (1+ i) addr) (term-of-bound-term bt)))))

    (visit nil itree))
  (values))

(defun map-itree (f g itree)
  (labels ((visit (addr itree)
	     (let ((node (funcall f addr itree))
		   (children (dotimeslist (i bt (cdr (bound-terms-of-term itree)))
					  (visit (cons (1+ i) addr) (term-of-bound-term bt)))))
	       (funcall g node children))))
    (visit nil itree)))

(defun map-cons-tree (f g ctree)
  (labels ((visit (addr ctree)
	     (let ((node (funcall f addr ctree))
		   (children (dotimeslist (i subtree (cdr ctree))
					  (visit (cons (1+ i) addr) subtree))))
	       (funcall g node children))))
    (visit nil ctree)))


(defun char->ichar (ch &optional (newline-ichar inewline))
  (cond ((char= ch #\newline)
	 newline-ichar)
;	((char= ch #\return)
;	 newline-ichar)
	((char= ch #\tab)
	itab)
	((char-code ch))))

(defun ichar (ch) (char->ichar ch))

(defun istring (x)
  (map 'list #'char->ichar (if (stringp x)
			       x
			       (princ-object-to-string x))))




(defun sanitize-name-string (namet)
  (let ((name (string namet)))
    (cond
      ((string= name "last") "last_RENAMED")
      ((let ((l (length name)))
	 (do ((i 0 (1+ i)))
	     ((or (= i l) (member (char name i)
				  '(#\? #\! #\@ #\# #\$ #\% #\& #\* #\` #\' #\SPACE #\-)))
	      (not (= i l)))))
       (implode-to-string
	(mapcan #'(lambda (ich)
		    (cond 
		      ((eql ich iquestion) (istring "_QUESTIONMARK_"))
		      ((eql ich iexclaimation) (istring "BANG"))
		      ((eql ich iat) (istring "AT"))
		      ((eql ich isplat) (istring "SPLAT"))
		      ((eql ich idollar) (istring "DOLLAR"))
		      ((eql ich ipercent) (istring "PERCENT"))
		      ((eql ich iampersand) (istring "AMPERSAND"))
		      ((eql ich istar) (istring "STAR"))
		      ((eql ich ispace) (istring "_SPACE_"))
		      ((eql ich iquote) (istring "_quote_"))
		      ((eql ich ibackquote) (istring "_backquote_"))
		      ((eql ich idash) (istring "_DASH_"))
		      (t (list ich))))
		(istring name))))				   
      (t (string name)))))

