
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;
;;;; -docs- (mod edd)

;;;;
;;;;	GC : garbage collection
;;;;
;;;;	 The garbage collector deletes, collects, and archives data
;;;;	 appropriately (and safely!).  Various statistics are maintained.
;;;;
;;;;	<db>	: (<collected{directory}> <process{directory}>) list
;;;;	<collected>	: (<archived{directory}> <collection{directory}>) list
;;;;	
;;;;	<collection>	: <collected-process{directory}> list
;;;;	  * collection directory is named "old/"
;;;;	<collected-process>	: <log{file}> list <data{file}> list 
;;;;	  * all files are compressed ascii files
;;;;	  ** data files are not referenced by any active logs
;;;;	<process>	: <log{file}> list <data{file}> list
;;;;	  * files are mathbus files (sequence.type)
;;;;	<archived>
;;;;	  * archived are tarred and zipped ascii compressed (sequence.type.ac) files
;;;;
;;;;	Five states of logs:
;;;;	  - open : open when collection started.
;;;;	  - active : not collected or being collected.
;;;;	  - passive : being collected.
;;;;	  - old : previously collected. 
;;;;	  - archived : previously collected and possibly moved to removable media (tape/cd).
;;;;	
;;;;
;;;;	A log file is a candidate for collection when it has satified some criteria	
;;;;	about its descendents, ie time since last opened for reading > y & all branches
;;;;	rooted at log greater than length x. Common case will be a linear list . 
;;;;	  - other log characteristics may be identified such as gc stats, or
;;;;	    how log produced, ie as result of close, checkpoint or unbind-gc.
;;;;
;;;;	Passive logs satisfy the above criteria, reside in closed processes, and are
;;;;	 not open for reading by any process.  (those open for writing are inherently
;;;;	 disqualified since they reside in an open process)
;;;;   
;;;;	A process is considered closed if none of its logs are open for writing.  This
;;;;	 is determined through advisory file locking.  When a process opens a log for
;;;;	 writing, it acquires a write lock. (only first byte is locked to allow reading) 
;;;;	 This lock is released upon closing the log and also
;;;;	 when the process itself terminates (for any reason).  Thus, a process without
;;;;	 read locks on any of its logs is either running with no open write-logs, or has
;;;;	 terminated, and in either case, available for possible collection.
;;;;
;;;;	Once the gc has determined a process is closed, the process may not open a new log
;;;;	 for writing.  To ensure this, gc creates and locks closed.gc( locks stat) file in the process
;;;;	 directory and deletes (unlocks) when done (double check is done to satisfy race)
;;;;
;;;;	Similarly, locks are used to keep account of logs open for reading. When a
;;;;	 process opens a log for reading, it acquires a read lock (multiple read locks may
;;;;	 exist for a given file). This lock is released upon closing the log and also
;;;;	 when the process itself terminates (for any reason).  Thus, a log without a
;;;;	 write lock is not open for reading by any alive process, and hence,
;;;;	 available for possible collection. (logs open for reading must be considered
;;;;	 to avoid declaring them passive and moving them to old directory in the event
;;;;	 log was not accessed recently)
;;;;
;;;;	When a log is deemed passive, it is moved to old,  preventing it from then
;;;;	 being opened for reading.  However race conditions must be considered.
;;;;	 After a log is deemed passive, but before moved, a process has opportunity to
;;;;	 open log for reading.  To avoid this, with each open, process waits, closes log, then
;;;;	 opens again- if still there then ok (file access time gets updated with open, so
;;;;	 race will not be repeated)
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	Three flavors of garbage collection :
;;;;	  - delete : identifies files never persistently referenced and thus
;;;;		deletable.
;;;;	  - collect : identifies files referenced by passive logs only and moves
;;;;		them to old.
;;;;	  - unbind : identifies object ids no longer strongly referenced and
;;;;		thus candidiates for unbinding. Once unbound they will be available
;;;;		for collection in later gc's.
;;;;
;;;;	
;;;;	Delete :
;;;;   
;;;;	It is possible for files in a passive directory to be unreferenced.
;;;;	They can be deleted (maybe moved to a delete dir FTTB).
;;;;	
;;;;	Main:
;;;;	  Let accumulator = Empty
;;;;	    For each closed (how do we know, file locking) process.
;;;;	      For each active or passive log 
;;;;	        For each log record visit term
;;;;	
;;;;	Visit Term :
;;;;	  If !data_persist accumulate stamp.type, visit stamp.type
;;;;	  Else foreach subterm visit term
;;;;	
;;;;	Cleanup : for each file, if not accumulated then delete.
;;;;	
;;;;	
;;;;	Collect :
;;;;
;;;;	perform delete
;;;;
;;;;	For each closed process containing a passive log.
;;;;	  Find or create old/process dir.
;;;;
;;;;	For each closed process (closed during delete)
;;;;	  For each active log
;;;;	    For each log record
;;;;	      Visit term
;;;;	
;;;;	Cleanup :
;;;;	  Move passive logs to old.
;;;;	  For each passive dir (closed and containing passive log)
;;;;	    For each file
;;;;	      If not accumulated then move to old
;;;;
;;;;	
;;;;	Unbind Log : 
;;;;
;;;;	An object_id is Unbindable:
;;;;	  - not activated
;;;;	  - collectable
;;;;	  - there does not exist a path from a non-collectable
;;;;	    object_id to the object_id.
;;;;	
;;;;	run log and build table indicating binding, activated and collectable
;;;;	(usual oid-table) for each oid.
;;;;	
;;;;	accumulate all bound oids accessible from activated or non-collectable :
;;;;	  for each oid accumulate and read objc and visit term
;;;;	visit-term
;;;;	  visit-parms : if oid accumulate
;;;;	  visit subterms (need to expand !data_persists)
;;;;
;;;;	cleanup : 
;;;;	  unbind all bound-oids not accumulated.
;;;;	  write log, indicate that created via unbind.
;;;;	
;;;;	Could be done as part of open, ie read old log then do unbinds, then
;;;;	dump to new log. Problem is that it could significantly slow open. So
;;;;	better to do batch method.
;;;;
;;;;	??? could cache oids of objects at persist-time. ie persist-term contains list of oids.
;;;;	??? thus no need to read term to find oids. Could get big. or term pointed to by persist
;;;;	??? has oid list at top then need to read one level but not others. so when scanning log
;;;;	??? need to read only persistent terms in log to do bind collection. Need to be careful
;;;;	??? to insure oids are available at write time so as to avoid having to read contained 
;;;;	??? persistent terms to produce oid list for term being written.
;;;;	??? or have persist term contain list of oids and list of other persist terms for which
;;;;	??? oids not collected. see notes in com-defs about !data_persist.
;;;;	
;;;;	
;;;;	
;;;;	
;;;;  -page-
;;;;
;;;;	
;;;;	db-delete ()	: NULL
;;;;	  * deletes useless data
;;;;	db-collect ()	: NULL
;;;;	  * collects data not likely to be referenced again
;;;;	db-unbind ()	: NULL
;;;;	  * ??
;;;;
;;;;	delete-accumulate (closed-process-list)	: NULL
;;;;	  * builds table of persistent data files in each active or passive log record of each
;;;;	    closed process
;;;;	
;;;;	delete-cleanup (closed-process-list)		: NULL
;;;;	  * deletes non accumulated files
;;;;	 
;;;;
;;;;	collect-accumulate (process list)	: NULL
;;;;	  * builds table of persistent data files in each active log record of each
;;;;	    closed process
;;;;	
;;;;	collect-cleanup ((passive process . passive log list) list)
;;;;	  * for each passive process, moves non-accumulated files
;;;;	    to old/process
;;;;
;;;;	collect-passive-processes (process list)		: NULL
;;;;	  * builds log tree, finds each process containing a passive log,
;;;;	    creates old/process directory (if does not already exist)
;;;;	  ** returns 2 values-- passive process list & passive log list
;;;;	  *** as passive log found, moves to old directory--
;;;;	      could write stamp to process/passive.logs and move at cleanup,  not necessary
;;;;
;;;;	collect-file (pathname)		: NULL
;;;;	  * moves mathbus file to compressed ascii file in old/
;;;;
;;;;  -page-
;;;;
;;;;	 gc statistics:
;;;;
;;;;	  For each collection, a collection term containing (1) a new transaction stamp
;;;;	 (process-id and time may be useful) and (2) a stats term is written to
;;;;	 collection.log file in master directory
;;;;
;;;;	 <collect-record>	:!db_collection(<stamp-term>; <stats>)
;;;;	 <stats>		:!stats (<type-counts> <criteria>)
;;;;	 <type-counts>		:!types{(type:t deleted:n collected:n remaining:n):pl list}
;;;;	 <criteria>		:!criteria{time-span:n ancestors:n}
;;;;
;;;;	 a collection term is also written to a collection.log file in each process directory
;;;;	(counts in master directory are sum of those in processes)
;;;;
;;;;	walk-db (process-list term-f &optional log-f)	: NULL
;;;;	  * performs term-f on terms in record of log-f logs in process-list
;;;;	  ** log-f determines which logs to visit ie.active, passive-default all
;;;;	walk-db-logs (logs term-f)	: NULL
;;;;	walk-log-hdrs (logs stream-f)	: NULL
;;;;	  * performs stream-f on stream of hdr file of logs
;;;;
;;;;	db-visit-term (term)		: NULL
;;;;	  * adds term (or subterms) to persistent-data-table if persistent
;;;;
;;;;	process-open-p (<process>)	: bool
;;;;	wlog-open-p (log)		: bool
;;;;	rlog-open-p (log)		: bool
;;;;
;;;;  -page-
;;;;
;;;;
;;;;	hand-waving type of arguments:
;;;;
;;;;	 1. All terms referenced as data persist in a process's log either 
;;;;	 reside in that process's directory or
;;;;	 are referenced as data persist in some other process's log.
;;;;
;;;;	 2. All data in a process directory is either data persist in 1 of
;;;;	   that process's logs or is not data persist in any log
;;;;
;;;;	 3. Once a log open for writing is closed, it does not change
;;;;
;;;;	 We know that delete does not remove referenced data because
;;;;	 * visits all logs in closed processes
;;;;	 ** only deletes data from closed process directories
;;;;	 *** rule No. 2
;;;;
;;;;	 We know that collect does not move unreferenced data because
;;;;	  * delete called first, removing all unreferenced data
;;;;	 We know that collect does not move data referenced by active logs
;;;;	  * accumulates and only moves data from closed processes
;;;;	  ** passive logs are moved so they may not be opened by process as active,
;;;;	      and race condition for this is handled as described above
;;;;	
;;;; -doce-
;;;;	


;; collection criteria
;; file will be collected if (1) more than *maximum-time-span* has passed since
;; last reference and (2) log has children, all of which are > *minimum-ancestor-count* deep

(defvar *process-directory-contents* (make-hash-table :test #'equal))
(defvar *minimum-ancestor-count* 3)
;;(defvar *maximum-time-span*  (* 60 24)) ;; in minutes-ie 1 day
(defvar *maximum-time-span* 0) ;; time is not being computed correctly so set to 0
(defvar *minimum-access-time* (- (get-universal-time) *maximum-time-span*))
(defvar *debug-gc* nil)  ;;if t don't alter db
(defvar *ignore-processes* nil)

;; process directory contents
(defstruct pdc 
  (logs nil)
  (data nil)
  (time 0)  ;;universal time
  (process nil)
  (open-p nil)
  )

(defun show-process-directory-contents ()
  (let ((acc nil))
    (maphash #'(lambda (k v)
		 (declare (ignore k))
		 (push v acc)
		 ;;(setf -k k -v v)
		 (when (logs-of-pdc v) (break)))
	     *process-directory-contents*)

    (dolist (v acc) ; sort?
      (format t "~4a ~a ~a ~a~%"
	      (open-p-of-pdc v)
	      (sortable-datetime-string (time-of-pdc v))
	      (namestring (process-of-pdc v))
	      (length (data-of-pdc v))
	      )
      (dolist (l (logs-of-pdc v))
	(format t "       ~a~%" (namestring l)))
      )))

(defun logs-of-pdc (pdc) (pdc-logs pdc))
(defun data-of-pdc (pdc) (pdc-data pdc))
(defun time-of-pdc (pdc) (pdc-time pdc))
(defun open-p-of-pdc (pdc) (pdc-open-p pdc))
(defun process-of-pdc (pdc) (pdc-process pdc))

(defvar *log-state* 'a) ;; a for active, p passive

(defun id-of-process (process)
  (intern-system (car (last (pathname-directory process)))))

(defun equal-process-dir (p1 p2)
  (string= (car (last (pathname-directory p1)))
	   (car (last (pathname-directory p2)))))

(defun decode-tm-time (time)
  (mlet* (((x m) (floor time 60))
	  ((y h) (floor x 24))
	  ((z d) (floor y 31))
	  ((p mo) (floor z 12)))
	 (values m h (if (zerop d) 31 d) mo (+ p 1900))))

(defun tmtime-string (utime)
  (mlet* (((m h date mo yr) (decode-tm-time utime))
	  ((month year) (if (zerop mo) (values 12 (1- yr)) (values mo yr))))
	 (if (>= h 12)
	     (format-string "~a:~2,'0DPM ~a/~a/~a" (if (= h 12) 12 (- h 12)) m month date year)
	   (format-string "~a:~2,'0DAM ~a/~a/~a" (if (zerop h) 12 h) m month date year))))
	    
(defun tm-time (time)
  (mlet* (((s m h date mo yr day b z) (decode-universal-time time)
	   (declare (ignore z b day s))))
	 (+ (* 60 (+ (* 24 (+ (* (- yr 1900) 12 31) (* mo 31) date)) h)) m)))

;; type is an additional file extension added to pathname
(defun passive-pathname (pathname &optional type)
  (let ((dirs (pathname-directory pathname)))
    (make-pathname :name (if type
			     (file-namestring pathname)
			   (pathname-name pathname))
		   :type (or type (pathname-type pathname))
		   :directory (append (butlast dirs)
				      (cons *passive-dir* (last dirs))))))

;; type is an additional file extension added to pathname
(defun passive-datafile (pathname &optional type)
  (let ((dirs (pathname-directory pathname)))
    (make-pathname :name (if type
			     (file-namestring pathname)
			   (pathname-name pathname))
		   :type (or type (pathname-type pathname))
		   :directory (append (butlast (butlast dirs))
				      (cons *passive-dir* (append (last (butlast dirs))
							  (last dirs)))))))

(defun collected-logs-of-process (process-directory)
  (filter #'log-file-p (directory-listing (passive-pathname process-directory))))					

(defvar *gc-assoc-list* nil);; type*type-stats
(defvar *gc-process-assoc-list* nil);;process*assoc-list

(defun log-write-date (filename)
  (file-write-date filename))

(defun empty-p (pathname &optional data-p collection-p)
  (when collection-p
    ;; TODO derivations process.LOG collection.LOG  should be moved to.
    ;; closed.gc shouldn't exists but aborting gc can leave them.
    (let ((file (make-pathname :name "collection"
			       :type 'log
			       :directory (pathname-directory pathname))))
      (when (file-exists-p file)
	(rename-file file (passive-pathname file 'ac)))))
  (when (null (filter #'(lambda (x)
			  (let ((name (pathname-name x))
				(type (pathname-type x)))
			    (not (or (eql name nil)
				     (string= name ".")
				     (and data-p (string= name "old"))
				     (and data-p (string= name "derivations"))
				     (and data-p (string= name *data-dir*))
				     (and data-p (string= name "process") (string= type "LOG"))
				     (and data-p (string= name "collection") (string= type "LOG"))
				     (and data-p (string= name "closed") (string= type "gc")))
				 )))
		      (directory-listing pathname)))
    t))

(defun empty-process-p (pathname &optional data-p collection-p)
  (and (empty-p pathname data-p collection-p)
       (empty-p (data-directory pathname))))
       
;; determine if log open for reading to prevent deeming it passive
(defun wlog-open-p (log)
  (write-lock-p log))

(defun rlog-open-p (log)
  (read-lock-p log))

(defun process-open-p (process)
  (let* ((logs (logs-of-pdc (gethash (id-of-process process) *process-directory-contents*)))
	 (closed-p (forall-p #'(lambda (x) (not (wlog-open-p x))) logs)))
    ;;(setf c closed-p lg logs) (break)
    (if closed-p
	;; LAL create this file outside of this function-optional
	(if *debug-gc*
	    nil
	    (let ((filename (make-pathname :name "closed.gc";; "lock"
					   :directory (pathname-directory process)))
		  (umask (libc-umask *umask-mode*)))
	      (with-open-file (intermediate-stream
			       filename
			       :direction :io
			       :element-type *character-type*
			       :if-does-not-exist :create
			       :if-exists nil))
	      (libc-umask umask)
	      ;; (sleep 1)
	      ;; check for race
	      (unless (forall-p #'(lambda (x) (not (wlog-open-p x)))
				(db-logs-of-process process))
		(progn (when (file-exists-p filename) (delete-file filename)) t))))
	t)))

(define-primitive |!db_collection| () (stamp stats))
(define-primitive |!stats| () (types criteria))
(define-primitive |!data_dcr| 
  ((token . type) (natural . deleted) (natural . collected) (natural . remaining)) ())
(define-primitive |!types_cons| () (head tail))
(define-primitive |!criteria| ((string . time) (natural . ancestor-count)) ())
 
(defvar *type-table* (make-hash-table :test #'equal))
  
(defstruct type-stats
  (deleted 0)
  (collected 0)
  (remaining 0))

(defun deleted-of-type-stats (s) (type-stats-deleted s))
(defun collected-of-type-stats (s) (type-stats-collected s))
(defun remaining-of-type-stats (s) (type-stats-remaining s))

(defun stats-to-term (alist)
  (istats-term (term-list-to-term (mapcar #'stats-to-types alist))
	       (icriteria-term (tmtime-string *minimum-access-time*) *minimum-ancestor-count*)))

(defun stats-to-param (pair) 
  (let ((stats (cdr pair)))
    ;;parameter-list-parameter
    (list (token-parameter (car pair))
	  (natural-parameter (deleted-of-type-stats stats))
	  (natural-parameter (collected-of-type-stats stats))
	  (natural-parameter (remaining-of-type-stats stats)))))
  
(defun stats-to-types (pair) 
  (let ((stats (cdr pair)))
    ;;parameter-list-parameter
    (idata-dcr-term (car pair)
		    (deleted-of-type-stats stats)
		    (collected-of-type-stats stats)
		    (remaining-of-type-stats stats))))
  
(defstruct db-tree 
  (parent nil)
  (children nil)
  (ancestor-count 0)
  (stamp nil)
  (active-p nil)
  (reference nil)
  (owner nil)
  (hdr nil))

(defun stamp-of-db-tree (tree) (db-tree-stamp tree))
(defun ancestor-count-of-db-tree (tree) (db-tree-ancestor-count tree))
(defun children-of-db-tree (tree) (db-tree-children tree))
(defun parent-of-db-tree (tree) (db-tree-parent tree))
(defun active-p-of-db-tree (tree) (db-tree-active-p tree))
(defun reference-of-db-tree (tree) (db-tree-reference tree))


;; process pathname -> file list
(defvar *directory-table* (make-hash-table :test #'equal))

;; key =  stamp{log}, val = tree
(defvar *log-tree* (make-hash-table :test #'equal))

;;(type . (sequence . process-id)) -> bool 
(defvar *persistent-data-table* (make-hash-table :test #'equal))

(defun readable-directories (directories)
  (let ((l nil))
    (mapcar #'(lambda (x)
		(let ((filename (car (db-logs-of-process x))))
		  (when filename (with-db-input-file (stream (header-of-log filename))
				   (when stream
				     (setf l (cons x l)))))))
	    directories)
    (reverse l))) 

(defun walk-db (term-f &optional log-f)
  (labels ((walk-process (pdc)
	     (if log-f
		 (mapc #'walk-log
		       (filter log-f
			       (logs-of-pdc pdc)))
		 (mapc #'walk-log
		       (logs-of-pdc pdc))))
	   (walk-log (log)
	     (with-db-input-file (intermediate-stream log t)
	       (do ((term (db-read-term intermediate-stream)
			  (db-read-term intermediate-stream))
		    (l nil l))
		   ((null term) l)
		 (funcall term-f term)))))
    (maphash #'(lambda (id pdc)
		 ;;(declare (ignore id))
		 (with-handle-error (('(walk db))
				     (progn (format t "F");;(break "ff")
					    (unless (member id *ignore-processes*)
					      (push id *ignore-processes*))
					    nil))
		   (walk-process pdc)))
	     *process-directory-contents*)))

(defun walk-db-logs (logs term-f)
  (labels ((walk-log (log)
	     (format t ".")
	     (with-handle-error (('(walk db logs))
				 (progn (format t "FF")
					(let ((id (id-of-process log)))
					  (unless (member id *ignore-processes*)
					    (push id *ignore-processes*)))
					nil))
	       (with-db-input-file (intermediate-stream log t)
		 (do ((term (db-read-term intermediate-stream)
			    (db-read-term intermediate-stream))
		      (l nil l))
		     ((null term) l)
		   (funcall term-f term))))))
    (mapc #'walk-log logs)))

(defun walk-hdrs (hdrs stream-f)
  (labels ((walk-hdr (hdr)
	     (with-db-input-file (intermediate-stream hdr)
	       (when intermediate-stream
		 (with-handle-error (('(db log header))
				     ())
		   (funcall stream-f intermediate-stream))))))
    (mapcan #'walk-hdr hdrs)))

(defvar *db-visit-table* (make-hash-table :test #'equal))

(defun db-visit-term-eg (term)
  (if (idata-persist-term-p term)
      (let* ((stamp (term-to-stamp (stamp-of-idata-persist-term term)))
	     (type (type-of-idata-persist-term term))
	     (key (key-of-data stamp type)))
	;;(setf k key) (break)
	
	(unless (eq (gethash key *persistent-data-table*) 'a)
	  (setf (gethash key *persistent-data-table*) *log-state*))
	
	(unless (gethash key *db-visit-table*)
	  (let ((filename (stamp-to-pathname stamp type t)))
	    (setf (gethash key *db-visit-table*) t)
	    (when (file-exists-p filename);;remove after testing???
	      (db-visit-term (db-read stamp type))))))
    
      (mapc #'db-visit-term (mapcar #'term-of-bound-term-f (bound-terms-of-term term))))
  nil)


(defvar *db-visit-op-count* 0)

(defun db-visit-term-incf () 
  (incf *db-visit-op-count*)
  (when (zerop (mod *db-visit-op-count* 100000))
    (format t "~%~a : visited ~a ops ~%"
	    (datetime-string (get-universal-time))
	    (num-to-string *db-visit-op-count*))

    ;; sleep some to give save chance to interrupt.
    (when (zerop (mod *db-visit-op-count* 5000000))
      (let ((interval 5))
	(room)
	(format t "begin ~a sec sleep~%" interval)
	(sleep interval)
	(format t "end sleep~%")
	(format t "begin ~a sec sleep~%" interval)
	(sleep interval)
	(format t "end sleep, continuing~%")
	))))

(defvar *db-visit-stamp-count* 0)

(defun db-visit-stamp-incf ()
  (incf *db-visit-stamp-count*)
  (when (zerop (mod *db-visit-stamp-count* 4096))
    (format t "~%~a : visited ~a stamps ~%"
	    (datetime-string (get-universal-time))
	    (num-to-string *db-visit-stamp-count*)) ))

(defun db-visit-term-aux (term stampf)

  (labels ((visit (term)
	     (when term
	       (db-visit-term-incf)

	       (cond
		 ((idata-persist-term-p term)
		  (let ((stamp-term (stamp-of-idata-persist-term term)))
		    ;; check so that gc goes smoothly even if got in some bad state
		    (when (istamp-term-p stamp-term)
		      (db-visit-stamp-incf)
		      (let ((stamp (term-to-stamp stamp-term))
			    (type (type-of-idata-persist-term term)))
				   
			(let ((contp (funcall stampf stamp type)))
			  (when contp
			    (let ((filename (stamp-to-pathname stamp type t)))
			      (if (file-exists-p filename) ;; should always exist? 
				  (visit (db-read stamp type))
				  (format t "~%GC-visit-term file-missing ~a~%" filename)
				  ))))))))

		 ((iblob-proxy-term-p term)
		  (let ((stamp-term (stamp-of-idata-persist-term term)))
		    (when (istamp-term-p stamp-term)
		      (db-visit-stamp-incf)
		      (let ((stamp (term-to-stamp stamp-term)))
			(funcall stampf stamp 'blob)))))

		 (t (mapc #'visit (mapcar #'term-of-bound-term-f (bound-terms-of-term term))))))))

    (visit term)
    nil))

(defun gc-visit-table-log (stamp)

  (when (eql 'a *log-state*) ;; don't bother to move logs to old.
    (unless (gethash (key-of-data stamp 'log) *db-visit-table*)

      ;; want to prevent log files from being collected but do not want to
      ;; walk files.
      (let ((stamps (with-ignore (collect-table-log-stamps stamp))))
	;;(setf -stamps stamps) (when stamps (break "gcvtl"))
	(dolist (s stamps)
	  (let ((k (key-of-data s 'log)))
	    ;; note that we've been here.
	    (setf (gethash k *db-visit-table*) t)

	    (setf (gethash k *persistent-data-table*) 'a)

	    )))))
  
  ;; must return nil to prevent caller from reading this file.
  nil)

(defun db-visit-term (term)

  (db-visit-term-aux term
		     #'(lambda (stamp type)
			 (if (eql type 'log)
			     (gc-visit-table-log stamp)
			     (let ((key (key-of-data stamp type)))
			       (unless (eq (gethash key *persistent-data-table*) 'a)
				 (setf (gethash key *persistent-data-table*) *log-state*))

			       (unless (gethash key *db-visit-table*)
				 (let ((filename (stamp-to-pathname stamp type t)))
				   (setf (gethash key *db-visit-table*) t)
				   t))))))
  nil)


(defun key-of-data (stamp type)
  (let ((process-id (process-id-of-stamp stamp))
	(sequence (sequence-of-stamp stamp)))
    (cons type (cons (intern-system (princ-to-string sequence))
		     (string process-id)))))

(defun db-delete ()
  (read-process-contents)
  (delete-accumulate)
  (delete-cleanup))

(defun delete-accumulate ()
  (walk-db #'db-visit-term))

(defun delete-accumulate-logs (logs)
  (walk-db-logs logs #'db-visit-term))

(defun delete-cleanup ()
  (if (string= "y" (cl-user:prompt-and-read :string (format-string "You must enter the single lower-case character y then return to continue,~% continue with deletes? ")))
      (maphash #'(lambda (id pdc)
		   ;;(declare (ignore id))
		   ;;(if (equal id '|8888629_eef_ba449d40|) (progn (break) nil)
		   (if (member id *ignore-processes*);;probably can go ahead and cleanup these dirs anyway
		       (progn (format t "~%IP: ~s " id)
			      nil)
		       (unless (open-p-of-pdc pdc)
			 (let* ((process (process-of-pdc pdc))
				;;(dir (db-extend-pathname (append (last (pathname-directory process))
				;; (list "deleted"))))
				(files (data-of-pdc pdc))
				(alist nil))
		       
			   (mapc #'(lambda (x)
				     (let* ((key (key-of-file x))
					    (type (car key))
					    (stats (or (cdr (assoc type alist :test #'equal))
						       (let ((st (make-type-stats)))
							 (setf alist (acons type st alist))
							 st))))
				       (if (gethash key *persistent-data-table*)
					   (setf (type-stats-remaining stats)
						 (1+ (remaining-of-type-stats stats)))
					   (progn
					     (io-echo "d")
					     (unless *debug-gc* (delete-file x))
					     ;; update assoc list
					     (setf (type-stats-deleted stats)
						   (1+ (deleted-of-type-stats stats)))))))
				 files)
			   (when alist (setf *gc-process-assoc-list*
					     (acons (car (last (pathname-directory process)))
						    alist
						    *gc-process-assoc-list*)))))))
	       *process-directory-contents*)
      (format t "gc deletes skipped ~%"))
  nil)
	
(defun create-passive-directory (process)
  (let ((pathname (passive-pathname process)))
    (create-directory (directory-namestring pathname) nil)
    (create-directory (directory-namestring (data-directory pathname)) nil)))

(defun collect-file (file ofile)
  (when (file-exists-p file)

    ;;if (db-ascii-p)

	(rename-file file ofile)

     #|(progn (with-db-input-file (input-stream file t)
		 (let ((pathname ofile);;test not read lock, lock entire
		       (umask (libc-umask *umask-mode*)))
		  
		   (with-open-file (output-stream
				    pathname
				    :direction :output
				    :element-type #+:lucid '(unsigned-byte 8)
				    #+:allegro 'character
				    :if-exists nil
				    :if-does-not-exist :create)
		  
		     (unless output-stream
		       (raise-error (error-message
				     '(collect file not exist)
				     file)))
		     (let ((prl-stream (new-prl-out-stream
					output-stream
					#'lisp-stream-write;;#'(lambda (byte s)
					;; (write-byte byte s))
					)))
		       (do ((term (db-read-term input-stream)
				  (db-read-term input-stream))
			    (l nil l))
			   ((null term))
			 (walk-term-ascii
			  term
			  #'(lambda (byte)
			      (prl-stream-write byte prl-stream))))))
		   (libc-umask umask)))
	       (delete-file file))|#

	))

(defun collect-init ()
  (create-directory (directory-namestring (db-extend-pathname (list *passive-dir*))))
  (clrhash *process-directory-contents*)
  (clrhash *log-tree*)
  (clrhash *db-visit-table*)
  (setf *gc-process-assoc-list* nil)
  (setf *gc-assoc-list* nil)
  (clrhash *persistent-data-table*))

(defun update-collection-log (process)
  (let ((alist (cdr (assoc (car (last (pathname-directory process)))
			   *gc-process-assoc-list*
			   :test #'equal))))
	
    (when alist
      (with-db-output-file (intermediate-stream
			    (make-pathname :name "collection"
					   :type 'log
					   :directory (pathname-directory process))
			    nil nil
			    t)
			   
	(db-write-term (idb-collection-term
			(stamp-to-term (new-transaction-stamp))
			(stats-to-term alist))
		       intermediate-stream)))
    nil))
	   
(defun update-alist ()
  (let ((alist nil))
    (mapcar #'(lambda (pl)
		(mapcar #'(lambda (pair)
			    (let* ((type (car pair))
				   (ts (cdr pair))
				   (ap (or (assoc type alist :test #'equal)
					   (let ((v (cons type (make-type-stats))))
					     (setf alist (cons v alist)) v)))
				   (val (cdr ap)))
			      
			      (setf (type-stats-collected (cdr ap))
				    (+ (collected-of-type-stats val)
				       (collected-of-type-stats ts)))
			      (setf (type-stats-deleted (cdr ap))
				    (+ (deleted-of-type-stats val) 
				       (deleted-of-type-stats ts)))
			      (setf (type-stats-remaining (cdr ap))
				    (+ (remaining-of-type-stats val)
				       (remaining-of-type-stats ts)))))
			pl))
	    
	    (mapcar #'(lambda (x) (cdr x)) *gc-process-assoc-list*))
    alist))

(defun show-gc-alist ()
  (mapc #'(lambda (e)
	    (format t
		    "~30a ~8a ~8a ~8a~%"
		    (car e)
		    (type-stats-deleted (cdr e))
		    (type-stats-collected (cdr e))
		    (type-stats-remaining (cdr e))
		    ))
	(sort (update-alist)
	      #'>
	      :key #'(lambda (e) (type-stats-remaining (cdr e))))))

(defun update-db-collection-log ()
  (let ((term (idb-collection-term (stamp-to-term (new-transaction-stamp))
				   (stats-to-term (update-alist)))))
    (progn (unless *debug-gc* (with-db-output-file (intermediate-stream (db-extend-pathname nil "collection" 'log)
						     nil nil ;;t
						     t)
				(db-write-term term intermediate-stream)))
	   term)))

(defun collect-accumulate (logs)
  (walk-db-logs logs #'db-visit-term))

(defun collect-cleanup (process-list)
  (mapc #'(lambda (process)
	    ;;(if (equal (id-of-process process) '|8888629_eef_ba449d40|) (progn (break) nil))
	    (if (member (id-of-process process) *ignore-processes*)
		(progn (format t "~%IP: ~s " (id-of-process process));;(break)
		       nil)
		(let* ((pdc (gethash (id-of-process process) *process-directory-contents*))
		       (files (data-of-pdc pdc))
		       (alist (cdr (assoc (car (last (pathname-directory process)))
					  *gc-process-assoc-list*
					  :test #'equal))))
		  ;; (setf p process f files a alist) (break)
		  (unless (open-p-of-pdc pdc)
		    (mapc #'(lambda (x)
			      (let* ((key (key-of-file x))
				     (type (car key))
				     (stats (cdr (assoc type alist :test #'equal)))
				     (val (gethash key *persistent-data-table*)))
				(unless (or (null val)
					    (eq val 'a))
				  (progn
				    (io-echo "c")
				    (unless *debug-gc* 
				      (collect-file x (passive-datafile x 'ac)));;move to ASCII compressed file in old
				    ;; update p assoc list
				    (setf (type-stats-remaining stats)
					  (1- (remaining-of-type-stats stats)))
				    (setf (type-stats-collected stats)
					  (1+ (collected-of-type-stats stats)))))))
			  files)))))         
	process-list))

#|(defun io-echo (chx &optional oops)

  (let ((ch (if (stringp chx) chx (or oops "?"))))

    (let ((stat (assoc ch *io-echo-stats*)))
      (if stat
	  (incf (cdr stat))
	  (setf *io-echo-stats* (acons ch 1 *io-echo-stats*))))
    
    (when *io-echo-p*
      (format t "~a" ch))

    (incf *io-echo-count*)
    (when (> *io-echo-count* 4096)
      (format t "IO-echo line wrap[4096] ~%")
      (setf *io-echo-count* 0))))|#

(defun make-pathname-with-type (pathname type)
  (make-pathname :name (pathname-name pathname)
		 :type type
		 :directory (pathname-directory pathname)))

(defun collect-passive-log (log)

  (let ((process (make-pathname :directory (pathname-directory log))))
    (create-passive-directory process)

    ;; move passive log      
    (let* ((new-log (passive-pathname log 'ac))
	   (hdr (make-pathname-with-type log 'hdr))
	   (new-hdr (passive-pathname hdr 'ac)))

      (format t " M ")
      (collect-file log new-log)
      (collect-file hdr new-hdr))

    process))

(defun pathname-member-p (pathname list)
  (let ((name (pathname-name pathname))
	(type (pathname-type pathname))
	(process (car (last (pathname-directory pathname)))))
    (labels ((test-p
		 (l)
	       (when l
		 (let ((p (car l)))
		   (or (and (string= name (pathname-name p))
			    (string= type (pathname-type p))
			    (string= process (car (last (pathname-directory p)))))
		       (test-p (cdr l)))))))
      (test-p list))))

(defun collectable (log &optional references-p) 
  ;;(setf -a log ) (break "dbc")
  (and (>= *minimum-access-time* (libc-atime (namestring log)))
       (not (or (wlog-open-p log) (rlog-open-p log)))
       (with-db-input-file (intermediate-stream (header-of-log log) t)
			   (let* ((header (read-log-header intermediate-stream))
				  (reference (reference-of-log-header header)))
				  
			     ;;should be a reference, some refs may be old (w/o owner-p)remove later
			     (or (not reference)
				 (with-unwind-error
				  (t) 
				  (and (or (not references-p)
					   (and (= (numeral-of-inatural-term
						    (cdr (property-of-term '|count| reference))))
						0)
					   (not (bool-of-ibool-term
						 (cdr (property-of-term '|owner| reference)))))
				       (not (bool-of-ibool-term
					     (cdr (property-of-term '|permanent| reference)))))))))))

(defvar *gc-override-refcount* nil)

(defun collectable-p (reference count owner) 
  
  (with-unwind-error
   (t) 
    (and (or *gc-override-refcount*
	     (<= (- (numeral-of-inatural-term
		     (cdr (property-of-term '|count| reference)))
		    count)
		 0))
	 (or owner
	     (not (bool-of-ibool-term
		   (cdr (property-of-term '|owner| reference))))))))
    
(defvar *ignore-owner-p* nil)

;;update(reread) log properties if t
(defun collect-logs-aux (&optional move-p update-props-p reset-active-p rebuild-log-tree-p) 

  (setf *minimum-access-time* (- (tm-time (get-universal-time)) *maximum-time-span*))
  (when rebuild-log-tree-p
    (clrhash *log-tree*)
    (build-log-tree))
  
  ;; must be seperate pass as might be done after set on since has children.
  (when reset-active-p
    (maphash #'(lambda (stamp tree)
		 (declare (ignore stamp))
		 (setf (db-tree-active-p tree) nil))
	     *log-tree*))

  (maphash #'(lambda (stamp tree)
	       (let ((log (stamp-to-pathname stamp 'log))
		     (garbage-p nil))
		   
		 (progn
		   ;; can't do this here as might be done after set on since has children.
		   ;;(when reset-active-p (setf (db-tree-active-p tree) nil))
		     
		   ;;reread headers since user may have adjusted log properties
		   (when (and t update-props-p)
		     (setf (db-tree-reference tree)
			   (with-db-input-file (intermediate-stream (header-of-log log) t)
			     (let ((header (read-log-header intermediate-stream)))
			       (when header
				 (setf (db-tree-hdr tree) header)
				 (or (reference-of-log-header header)
				     (properties-to-term *log-properties*)))))))

		   (let ((reference (or (db-tree-reference tree)
					(properties-to-term *log-properties*))))

		     (setf garbage-p (bool-of-ibool-term
				      (cdr (property-of-term '|garbage| reference))))
		       
		     (format t "Active? ~a garbage? ~a"
			     (active-p-of-db-tree tree)
			     (bool-of-ibool-term
			      (cdr (property-of-term '|garbage| reference))))

		     ;;set active by ancester count if not garbage
		     ;;(setf -tree tree)
		     ;;(when (equal (cdr -rsystem8) (db-tree-stamp tree)) (break "r8"))
		     (unless (children-of-db-tree tree)
		       (do ((i 0 (1+ i))
			    (node tree (parent-of-db-tree node)))
			   ;; stop when we've traveled min dist  
			   ((or (not node) (= i *minimum-ancestor-count*)))
			 ;;(when (equal (cdr -rsystem8) (db-tree-stamp node)) (setf -i i -node node) (break "r88"))
			   
			 (when (and node (not (= i *minimum-ancestor-count*)))
			   (unless (bool-of-ibool-term
				    (cdr (property-of-term '|garbage| (reference-of-db-tree node))))
			     (format t  "~%ACTIVE ~a~%" (db-tree-addr node))
			     (setf (db-tree-active-p node) t)))))

		     (format t "Act1 ~a " (active-p-of-db-tree tree))
		   
		     ;;set active by access time if not garbage
		     (unless (active-p-of-db-tree tree)
		       (setf (db-tree-active-p tree)
			     (or
			      ;;set active by time if not garbage
			      (and (not (bool-of-ibool-term (cdr (property-of-term '|garbage| reference))))
				   (< *minimum-access-time* (libc-atime (namestring log))))
			      ;;set active if log open
			      (gethash (key-of-file log) *open-wlog-table*)
			      (gethash (key-of-file log) *open-rlog-table*)
			      ;;set active if permanent
			      (bool-of-ibool-term
			       (cdr (property-of-term '|permanent| reference))))))

		     (format t "Act2 ~a g:~a t:~a w:~a r:~a p:~a"
			     (or (active-p-of-db-tree tree) 'f)
			     (or (bool-of-ibool-term (cdr (property-of-term '|garbage| reference))) 'f)
			     (or (not (< *minimum-access-time* (libc-atime (namestring log)))) 'f)
			     (if (gethash (key-of-file log) *open-wlog-table*) t 'f)
			     (if (gethash (key-of-file log) *open-rlog-table*) t 'f)
			     (or (bool-of-ibool-term (cdr (property-of-term '|permanent| reference))) 'f)
			     )
		   
		     ;;set active if ref count = 0 and owner doesn't want it	 
		     (unless (active-p-of-db-tree tree)
		       (let* ((myuid (getuid))
			      (owner-p (or *ignore-owner-p* (= (or (db-tree-owner tree) 0) myuid)))
			      (length (length
				       (filter #'(lambda (x)
						   (or *ignore-owner-p*
						       (= (or (db-tree-owner x)
							      (progn (break "own") 0))
							  myuid)))
					       (children-of-db-tree tree)))))
			 (when (and move-p (not *debug-gc*))
			   (when owner-p (update-reference-owner-p stamp))
			   (decrement-reference-count stamp length))
			 (format t " R:~a L:~a O:~a o:~a C:~a "
				 (numeral-of-inatural-term (cdr (property-of-term '|count| reference)))
				 length
				 (or owner-p 'f)
				 (or (bool-of-ibool-term (cdr (property-of-term '|owner| reference))) 'f)
				 (or (collectable-p reference length owner-p) 'f))
			 (setf (db-tree-active-p tree)
			       (not (collectable-p reference length owner-p)))))

		     ;; gather values to return, maybe move log
		     (format t " ~a Active? ~a ~%" (active-p-of-db-tree tree)
			     (let ((es (filter-db-environments-aux #'(lambda (a s) (equal stamp s)))))
			       (when es (caar es))))

		     (when (and nil garbage-p (active-p-of-db-tree tree))
		       ;;(setf -tree tree -reference reference)
		       ;;(setf -stamp stamp -log log)
		       (break "gandact"))
		     ))))
				       
	   *log-tree*)
    
  (let ((passive-logs nil)
	(passive-processes nil)
	(active-logs nil))

    (maphash #'(lambda (stamp tree)
		 (let ((log (stamp-to-pathname stamp 'log)))
		   (if (active-p-of-db-tree tree)
		       (progn 
			 (setf active-logs (cons (list* log stamp tree) active-logs)))
		       (let ((process (if (and move-p (not *debug-gc*))
					  (collect-passive-log log)
					  (make-pathname :directory (pathname-directory log)))))
			 (unless (pathname-member-p process passive-processes)
			   (push process passive-processes))
			 (setf passive-logs
			       (cons (if (and move-p (not *debug-gc*))
					 (list* (passive-pathname log 'ac) stamp tree)
					 (list* log stamp tree))
				     passive-logs))))))
	     *log-tree*)

    (setf a active-logs p passive-logs);;(break "cl")
    (values active-logs passive-logs passive-processes)))

(defun collect-logs  (&optional move-p update-props-p)
  (mlet* (((act pass pproc) (collect-logs-aux move-p update-props-p)))
	 ;;(setf -act act -pass pass -pproc pproc)
	 (values  (mapcar #'car act) (mapcar #'car pass) pproc)))


(defun logs-with-properties (props &optional update-props-p) 
  (let ((logs nil))
    ;;(setf p props) (break "P")
    (maphash #'(lambda (stamp tree)
		 (let ((log (stamp-to-pathname stamp 'log)))
		   ;;reread headers since user may have adjusted log properties
		   (progn
		     (when update-props-p
		       (setf (db-tree-reference tree)
			     (with-db-input-file (intermediate-stream (header-of-log log) t)
			       (let ((header (read-log-header intermediate-stream)))
				 (when header
				   (or (reference-of-log-header header)
				       (properties-to-term *log-properties*)))))))
		     ;;set active by ancester count if not garbage
		     (let ((reference (or (db-tree-reference tree)
					  (properties-to-term *log-properties*))))
		       (when (or (and (member '|permanent| props)
				      (bool-of-ibool-term
				       (cdr (property-of-term '|permanent| reference))))
				 (and (member '|garbage| props)
				      (bool-of-ibool-term
				       (cdr (property-of-term '|garbage| reference))))
				 (and (member '|owner| props)
				      (bool-of-ibool-term
				       (cdr (property-of-term '|owner| reference))))
				 (and (member '|count| props)
				      (= (numeral-of-inatural-term
					  (cdr (property-of-term '|count| reference)))
					 0)))
			 (setf logs (cons log logs)))))))
				       
	     *log-tree*)
    logs))

(defun view-log-tree ()
  (maphash #'(lambda (stamp tree)
	       (let ((log (stamp-to-pathname stamp 'log)))
		 (format t "~s children:~s ancestors:~s active-p:~s state: ~s ~%"
			 stamp (length (children-of-db-tree tree)) (ancestor-count-of-db-tree tree)
			 (db-tree-active-p tree)
			 (if (wlog-open-p log)
			     (if (rlog-open-p log) "open for reading and writing" "open for writing")
			   (if (rlog-open-p log) "open for reading" "closed")))))
	   *log-tree*))

(defun db-tree-addr (dbt)
  (tokens-of-itokens-term
   (address-of-ienvironment-term
    (environment-of-ilog-description-term
     (description-of-log-header (db-tree-hdr dbt))))))

(defun show-log-tree (&optional quickp)
  (let ((h (make-hash-table :test #'equal)))
    (let ((acc nil))
      (maphash #'(lambda (stamp tree)
		   (declare (ignore stamp))
		   (let ((r (do ((n tree (db-tree-parent n)))
				((or (gethash (db-tree-stamp n) h) (null (db-tree-parent n)))
				 (unless (gethash (db-tree-stamp n) h)
				   (setf (gethash (db-tree-stamp n) h) t)
				   n))
			      (setf (gethash (db-tree-stamp n) h) t))))
		     (when r
		       (when (member r acc :key #'car :test #'equal)
			 ;;(setf -r r -acc acc -stamp stamp -tree tree)
			 (break "rr"))
		       (push (cons r tree) acc))))
	       *log-tree*)

      (setf acc (mapcar #'car acc))

      (filter-db-environments-aux #'(lambda (addr stamp)
				      (setf (gethash stamp h) addr)
				      nil))

      (let ((strings (let ((a (make-array 1000)))
		       (dotimes (i 1000)
			 (setf (aref a i) (make-string (* 2 i) :initial-element #\.))
			 )
		       a)))

	(labels ((print-tree (tree depth)
		   ;;(setf -tree tree)
		   (let ((stamp (db-tree-stamp tree)))
		     (let ((log (stamp-to-pathname stamp 'log))
			   (addr (tokens-of-itokens-term
				  (address-of-ienvironment-term
				   (environment-of-ilog-description-term
				    (description-of-log-header (db-tree-hdr tree)))))))
		       (if quickp
			   (format t "~4a ~a~a~a~a ~%"
				   (db-tree-active-p tree)
				   depth
				   (aref strings depth)
				   addr
				   (gethash stamp h))
			   (format t "~a ~4a ~a~a~a~a ~%"
				   (if (wlog-open-p log)
				       (if (rlog-open-p log) "wr" "w ")
				       (if (rlog-open-p log) "r " "c "))
				   (db-tree-active-p tree)
				   depth
				   (aref strings depth)
				   addr
				   (gethash stamp h)))
		       ))

		   (mapc #'(lambda (c) (print-tree c (1+ depth))) (db-tree-children tree))
		 
		   ))

	  (format t "sizes ~a ~a~%" (length acc) (hash-table-count h))
		 
	  (dolist (r acc) (print-tree r 0) (terpri)))))))
	       

#|
do we really want this, could be dangerous to let user set this.
pass list into finish so can clear log tree
(defun set-log-active (desc)
  (maphash #'(lambda (stamp tree)
	       (declare (ignore stamp))
	       (let ((log (stamp-to-pathname stamp 'log))
		     (l (active-p-of-db-tree tree)))
		     ;;(key )
		(setf (db-tree-active-p tree) (if (member '|user| l)
						 l
					       (cons '|user| l)))))
	   *log-tree*))
|#

(defun build-log-tree ()
  
  (labels ((visit-children
	       (children count)
	     (mapc #'(lambda (x)
		       (incf (db-tree-ancestor-count x) count)
		       (visit-children (children-of-db-tree x) count))
		   children))
	   
	   (accumulate (log)
	     (format t "       ~a~%" (namestring log))
	     (let* ((key (key-of-file log t))
		    (log-header (when (file-exists-p log)
				  (or	; ;(gethash key *db-cache*) ;;ref props may have changed
				   (setf (gethash key *db-cache*)
					 (with-db-input-file
					     (intermediate-stream (header-of-log log))
					   (when intermediate-stream
					     (read-log-header intermediate-stream))))))))
	       ;; twould be simpler to just add stamp to table then make a second pass do the parent/ancestor stuff.
	       (when log-header
		 (let* ((parent (parent-of-log-header log-header))
			(stamp (stamp-of-log-header log-header))
			(parent-value (when parent (gethash parent *log-tree*)))
			(stamp-value (gethash stamp *log-tree*))
			(reference (or (reference-of-log-header log-header)
				       (properties-to-term *log-properties*);;nil
				       ))
			(owner (owner-uid (namestring (header-of-log log)))));;hdr instead so atime
		   (format t "       ~a~a ~a~%" (if stamp-value "v" "n") (if parent-value "p" "n") stamp)
		      
		   (if (and (null parent-value) (null stamp-value))
		       (let ((dbt (make-db-tree :stamp stamp
						:reference reference
						:owner owner
						:hdr log-header)))
			 (setf (gethash stamp *log-tree*) dbt)
			 (format t "nn set~%")
			 (when (and parent (db-log-p parent));; not void or archived
			   (setf (db-tree-ancestor-count dbt) 1)
			   (let ((pdbt (make-db-tree :children
						     (list (gethash stamp *log-tree*))
						     :stamp parent)))
			     (setf (gethash parent *log-tree*) pdbt)
			     (setf (db-tree-parent dbt) pdbt))))
			       
		       (if (null stamp-value) ; parent in table
			   (let ((dbt (make-db-tree :stamp stamp
						    :parent parent-value
						    :reference reference
						    :owner owner
						    :ancestor-count (1+ (ancestor-count-of-db-tree parent-value))
						    :hdr log-header)))
			     (setf (gethash stamp *log-tree*) dbt)
			     (format t "np set~%")
			     (setf (db-tree-children parent-value)
				   (cons dbt (children-of-db-tree parent-value))))

			   (if (null parent-value) ; stamp in table
			       (let ((pdbt (when (and parent (db-log-p parent)) ; not void or archived
					     (let ((pdbt (make-db-tree :stamp parent
								       :children (list stamp-value))))
					       (setf (gethash parent *log-tree*) pdbt)
					       pdbt))))
				 (setf (db-tree-parent stamp-value) pdbt
				       (db-tree-owner stamp-value) owner
				       (db-tree-reference stamp-value) reference
				       (db-tree-hdr stamp-value) log-header
				       (db-tree-ancestor-count stamp-value) 1)
				 (format t "vn set~%")
				 (visit-children (children-of-db-tree stamp-value) 1))
				   
			       ;; both in table
			       (unless (parent-of-db-tree stamp-value)
				 ;;(setf -parent parent -stamp stamp -parent-value parent-value -stamp-value stamp-value)
				 (format t "nilparent~%")
				 (setf (db-tree-parent stamp-value) parent-value
				       (db-tree-owner stamp-value) owner
				       (db-tree-hdr stamp-value) log-header
				       (db-tree-reference stamp-value) reference)
				 (format t "vp set~%")
				 (incf (db-tree-ancestor-count stamp-value)
				       (1+ (ancestor-count-of-db-tree parent-value)))
				 (visit-children (children-of-db-tree stamp-value)
						 (ancestor-count-of-db-tree stamp-value))
				 (setf (db-tree-children parent-value)
				       (cons stamp-value
					     (children-of-db-tree parent-value))))))))))))
	  
    (maphash #'(lambda (key v)
		 (declare (ignore key))
		 (format t "~4a ~a ~a ~a~%"
			 (open-p-of-pdc v)
			 (sortable-datetime-string (time-of-pdc v))
			 (namestring (process-of-pdc v))
			 (length (data-of-pdc v))
			 )
		 (mapc #'accumulate (logs-of-pdc v)))
	     *process-directory-contents*)))

(defun rehash-pdc ()
  (clrhash *process-directory-contents*)
  (read-process-contents))
  
(defun open-process-list ()
  (let ((acc nil))
    (maphash #'(lambda (k v)
		 (when (open-p-of-pdc v)
		   (push (process-of-pdc v) acc)))
	     *process-directory-contents*)

    (cons (length acc) acc)))

(defun small-process-list (size)
  (let ((acc nil))
    (maphash #'(lambda (k v)
		 (when (<= (length (data-of-pdc v)) size)
		   (when (open-p-of-pdc v) (break "open small???"))
		   (push (process-of-pdc v) acc)))
	     *process-directory-contents*)

    (cons (length acc) acc)))

(defun empty-process-list ()
  (let ((l (filter #'(lambda (x) (empty-p x t nil)) (cdr (small-process-list 0)))))
    (cons (length l) l)))

(defun delete-empty-process-list ()
  (dolist (d (cdr (EMPTY-PROCESS-LISTS)))
    (format t "~a~%" d)
    (maybe-delete-directory d nil)))
	     

(defun read-process-contents-aux (l)
  (mapc #'(lambda (process)
	    (let ((time (get-universal-time))
		  (logs (db-logs-of-process process))
		  (data (db-files-of-process process)))
	      ;;(setf -process process -logs logs -data data) (break "rpca")
	      (setf (gethash (id-of-process process) *process-directory-contents*)
		    (make-pdc :logs logs
			      :data data
			      :time time
			      :open-p nil
			      :process process))
	
	      (when (process-open-p process)
		(setf (pdc-open-p (gethash (id-of-process process) *process-directory-contents*))
		      t))))

	l))


(defun read-process-contents ()
  (read-process-contents-aux
   (db-process-list nil *personal-db*)))

(defun read-old-process-contents ()
  (read-process-contents-aux
   (db-process-list *old-master-pathname*)))


(defun update-process-collection-logs ()
  (gc-cleanup t))

(defun abort-collect ()
  (gc-cleanup nil))

(defun gc-cleanup (update-p)
  (maphash #'(lambda (id pdc)
	       (declare (ignore id))
	       (let ((process (process-of-pdc pdc)))
		 (unless (open-p-of-pdc pdc)
			 ;; write collection stats to process log
			 (when update-p (update-collection-log process))
			 (let ((filename (make-pathname :name "closed"
							:type "gc"
							:directory (pathname-directory process))))
			   (when (file-exists-p filename)
				 (delete-file filename))))))
	   *process-directory-contents*)
  nil)


(defun show-passive-logs (update-logs reset-act)
  (mlet* (((act pass procs) (collect-logs-aux nil update-logs (or update-logs reset-act))
	   (declare (ignore procs act))))
	 ;;(setf -act act -pass pass -procs procs) (break "spl1")
	 (filter-db-environments-aux #'(lambda (addr stamp)
					 (declare (ignore addr))
					 ;;(setf -addr addr -stamp stamp) (break "spl")
					 (member stamp pass :key #'cadr :test #'equal)))))



(defun db-collect (&optional delete-p)
  
  (collect-init)
  (read-process-contents);; build process table
  (build-log-tree);; build log table
  	
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs t)))
	 ;;(setf p passive-logs l passive-processes ll active-logs) (break)
	 
	 ;;acumulate and collect data referenced only by passive logs
	 (if delete-p
	     ;;accumulate and delete non-persistant data
	     (progn
	       (setf *log-state* 'a)
	       (delete-accumulate-logs active-logs)
	       (setf *log-state* 'p)
	       (delete-accumulate-logs passive-logs)
	       (delete-cleanup))
	   (progn
	     (setf *log-state* 'a)
	     (collect-accumulate active-logs)))
	 
	 (collect-cleanup passive-processes))

  ;; write collection stats to process log
  (update-process-collection-logs)
  ;;(mapc #'maybe-delete-directory passive-processes)
  ;; write collect stats to master
  (update-db-collection-log))

(defun last-term-of-file (filename)
  (with-db-input-file (intermediate-stream filename)
		      (if intermediate-stream
			  (do* ((term2 (db-read-term intermediate-stream)
				       (db-read-term intermediate-stream))
				(term1 term2 term1))
			       ((not term2) term1)
			       (setf term1 term2))
			nil)))

(defun collections-of-process (process)
  (when (file-exists-p (make-pathname :name "collection"
				      :type 'log
				      :directory (pathname-directory process)))
	(last-term-of-file (make-pathname :name "collection"
					  :type 'log
					  :directory (pathname-directory process)))))


(defun last-collection ()
  (last-term-of-file (db-extend-pathname nil "collection" 'log)))

#| ;; directory mode tests
(defvar *test-count* 0)

(setf *directory-mode*  501)
(setf *test-count* 0)

(defun directory-test1  ()
  (incf *test-count*)
  (create-directory (directory-namestring
		     (make-pathname :directory (append (pathname-directory *master-pathname*)
						       (list (princ-to-string *test-count*)))))))    

;; 69 59 mkdir 71    
(defun directory-test  ()
  (do () ((= *directory-mode* 500))
      (incf *test-count*)
      (create-directory (directory-namestring
			 (make-pathname :directory
					(append (pathname-directory *master-pathname*)
						(list (princ-to-string *test-count*))))))
      (incf *directory-mode*)))


|#

(defun set-ancestor-count (num)
  (let ((old *minimum-ancestor-count*))
    (setf *minimum-ancestor-count* num)
    old))


;; builds log tree, returns passive logs
(defun start-collect ()
  (end-log-buffering)
  (collect-init)
  (read-process-contents)
  (build-log-tree)
  
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs nil nil)
	   (declare (ignore active-logs passive-processes))))
	 (term-list-to-term
	  (log-sort (mapcar #'log-info-of-log passive-logs)))))

(defun finish-collect (&optional delete-p)
  (maphash #'(lambda (stamp tree)
	       (declare (ignore stamp))
	       (setf (db-tree-active-p tree) ;;(if (member '|user| (active-p-of-db-tree tree))
						 ;;(list '|user|))
					       nil))
	   *log-tree*)
  
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs t t)))
	 ;;accumulate and delete non-persistant data
	 (if delete-p
	     (progn
	       (setf *log-state* 'a)
	       (format t "~~%%begin delete accumulate active~%~%")
	       (delete-accumulate-logs active-logs)
	       (setf *log-state* 'p)
	       (format t "~%~%begin delete accumulate passive~%~%")
	       (delete-accumulate-logs passive-logs)
	       (format t "~%~%begin delete cleanup~%~%")
	       (delete-cleanup))
	   (progn
	     (setf *log-state* 'a)
	     (collect-accumulate active-logs)))
	 
	 (format t "begin collect cleanup")
	 (collect-cleanup passive-processes)  
	 (gc-cleanup (not *debug-gc*))
	 (mapc #'maybe-delete-directory passive-processes))
  
  ;; write collect stats to master
  (update-db-collection-log))

(defun maybe-delete-directory (process &optional collection-p)
  (when (empty-process-p process t collection-p);; use stats to determine instead of reading directory

    ;; remove data
    (libc-rmdir (namestring (data-directory process)))

    ;; remove process.log
    (let ((plog (make-pathname :directory (pathname-directory process)
			       :name "process"
			       :type 'log)))
      (when (probe-file plog)
	(delete-file plog)))
    
    ;; remove process derivations
    (let ((derivationsdir (make-pathname :directory (append (pathname-directory process) (list *derivation-dir*)))))
      (when (probe-file derivationsdir)
	(dolist (file (directory-listing derivationsdir))
	  (when (string= "DER" (pathname-type file))
	    (delete-file file)))

	(libc-rmdir (namestring derivationsdir))))

    (libc-rmdir (namestring process))
    
    (format t "rmdir:~s~%" (namestring process))))


(defun log-info-of-log (log)
  (let* ((key (key-of-file log))
	 (atime (tmtime-string (libc-atime (namestring log))))
	 (mtime (tmtime-string (tm-time (log-write-date (namestring log)))))
	 (name (header-of-log log))
	 (header (when (file-exists-p name)
		   (or;;(gethash key *db-cache*) 
		    (setf (gethash key *db-cache*)
			  (with-db-input-file
			      (intermediate-stream name)
			    (read-log-header intermediate-stream)))))))
	 
    (ilog-info-term
     atime
     mtime
     (if header
	 (token-list-to-term
      	  (tokens-of-itokens-term
	   (address-of-ienvironment-term (environment-of-ilog-description-term
					  (description-of-log-header header)))))
	 (itok-nil-term)))))
  
(defun view-collect ()
  (maphash #'(lambda (stamp tree)
	       (declare (ignore stamp))
	       (setf (db-tree-active-p tree) ;;(if (member '|user| (active-p-of-db-tree tree))
		     ;;(list '|user|))
		     nil))
	   *log-tree*)
  
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs nil t)
	   (declare (ignore active-logs passive-processes))))
	 (term-list-to-term
	  (log-sort (mapcar #'log-info-of-log passive-logs)))))

(defun view-remaining ()
  (maphash #'(lambda (stamp tree)
	       (declare (ignore stamp))
	       (setf (db-tree-active-p tree) ;;(if (member '|user| (active-p-of-db-tree tree))
						 ;;(list '|user|))
					       nil))
	   *log-tree*)
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs nil t)
	   (declare (ignore passive-logs passive-processes))))
	 (term-list-to-term (log-sort (mapcar #'log-info-of-log active-logs)))))

(defun view-active-logs ()
  (maphash #'(lambda (stamp tree)
	       (declare (ignore stamp))
	       (setf (db-tree-active-p tree) nil))
	   *log-tree*)
  (mlet* (((active-logs passive-logs passive-processes)
	   (collect-logs)
	   (declare (ignore passive-logs passive-processes))))
	 (term-list-to-term (log-sort (mapcar #'log-info-of-log active-logs)))))

(defun term-list-to-term (list)
  (if (null list)
      (inil-term)
    (let ((term (car list)))
      (ilog-cons-term term (term-list-to-term (cdr list))))))

(defun token-list-to-term (list)
  (if (null list)
      (inil-term)
    (let ((token (car list)))
      (itok-cons-term token (token-list-to-term (cdr list))))))

(defun tokens-to-term (list)
  (if (null list)
      (itok-nil-term)
    (let ((token (car list)))
      (icons-term token (tokens-to-term (cdr list))))))

(defun term-to-tokens-db (term)
  (if (compare-terms-p term (inil-term))
      nil
    (let ((tail (tail-of-itok-cons-term term)))
      (cons (head-of-itok-cons-term term)
	    (term-to-tokens-db tail)))))

(defun term-to-tokens (term)
  (if (compare-terms-p term (inil-term))
      nil
      (let ((tail (cdr-of-icons-term term)))
	(cons (token-of-itoken-term (car-of-icons-term term))
	      (term-to-tokens tail)))))

(defun view-logs-of-process (process)
  (term-list-to-term
   (sort (mapcar #'log-info-of-log (db-logs-of-process process))
	 #'string<
	 :key  #'(lambda (i) (string (nth 2 (term-to-tokens-db (address-of-ilog-info-term i))))))))
			

(defun view-sorted-logs ()
  (term-list-to-term
   (sort (mapcar #'log-info-of-log (flatten (mapcar #'db-logs-of-process
						    (readable-directories
						     (db-process-list nil *personal-db*)))))
	 #'(lambda (x y) (or (string> (car x) (car y)) (string> (cdr x) (cdr y))))
	 :key  #'(lambda (i) (cons (string (nth 2 (term-to-tokens-db (address-of-ilog-info-term i))))
				   (string (nth 3 (term-to-tokens-db (address-of-ilog-info-term i)))))))))

(defun log-info-sort (x y)  
  (or (string< (car x) (car y))
      (and (string= (car x) (car y))
	   (<= (read-from-string (cdr x)) (read-from-string (cdr y))))))
	       
(defun log-info-key (i)
  (let ((addr (term-to-tokens-db (address-of-ilog-info-term i))))
    (cons (string (nth 2 addr))
	  (string (nth 3 addr)))))
  
(defun log-sort (l)
  (sort l #'log-info-sort :key  #'log-info-key))

(defun token-term-sort (l)
  (sort l #'log-info-sort :key  #'(lambda (x) (let ((addr (term-to-tokens-db x)))
						(cons (string (nth 2 addr))
						      (string (nth 3 addr)))))))

(defun view-logs-test ()
  (term-list-to-term
   (log-sort (mapcar #'log-info-of-log
			       (or *logs*
				   (setf *logs*
					 (flatten (mapcar #'db-logs-of-process
							  (readable-directories
							   (db-process-list nil *personal-db*))))))))))
	  
(defun view-logs ()
  (term-list-to-term
   (log-sort (mapcar #'log-info-of-log
		     (flatten (mapcar #'db-logs-of-process
				      (db-process-list nil *personal-db*)))))))
	  
(defun view-logs-with-properties (p)
  (term-list-to-term
   (log-sort (mapcar #'log-info-of-log (logs-with-properties p)))))
	  
(defun set-perm (tags)
  (let* ((stamps (or (mapcar #'cdr (match-db-environment-all tags))
		     (raise-error (error-message '(set perm log not found) tags))))
	 (header nil))
	
    ;;(setf a e-addr b stamp ) (break "S")
    (find-first #'(lambda (stamp)
		    (with-db-input-file
		     (intermediate-stream (stamp-to-pathname stamp 'hdr))
		     (if intermediate-stream
			 (with-handle-error (('(db log header)) ())
					    (setf header (read-log-header intermediate-stream)))))

		    (let ((reference (reference-of-log-header header)))
		      (when (and reference (not (bool-of-ibool-term
						 (cdr (property-of-term '|permanent| reference)))))
			    (update-reference-root-p stamp t) t)))
		(sort stamps 
		      #'>
		      :key #'time-of-stamp))))

(defun permanent-logs-aux (&optional master)
  (let ((hdrs (mapcar #'get-log-header (get-hdr-files (db-process-list master *personal-db*)))))
    (mapcan #'(lambda (x)
		(let* ((reference (reference-of-log-header x))
		       (property (when reference (property-of-term '|permanent| reference))))
		  (when (and x reference property
			     (bool-of-ibool-term (cdr property)))
		    (list (tokens-of-itokens-term (address-of-ienvironment-term
						   (environment-of-ilog-description-term
						    (description-of-log-header x))))))))
	    hdrs)))

(defun permanent-logs (&optional master)
  (let ((plogs (permanent-logs-aux master)))
    (term-list-to-term
     (token-term-sort (mapcar #'token-list-to-term plogs)))))

(defun unset-perm (tags)
  (let ((stamps (or (mapcar #'cdr (match-db-environment-all tags))
		    (raise-error (error-message '(unset perm log not found) tags))))
	(header nil))
    (find-first #'(lambda (stamp)
		    (with-db-input-file
			(intermediate-stream (stamp-to-pathname stamp 'hdr))
		      (if intermediate-stream
			  (with-handle-error (('(db log header)) ())
			    (setf header (read-log-header intermediate-stream)))))
		    (let ((reference (reference-of-log-header header)))
		      (when (and reference (bool-of-ibool-term
					    (cdr (property-of-term '|permanent| reference))))
			(update-reference-root-p stamp nil) t)))
		(sort stamps 
		      #'<
		      :key #'time-of-stamp))))

(defun perm-p (tags)
  (let* ((e-addr (match-db-environment tags))
	 (stamp (or (cdr e-addr) (raise-error (error-message '(perm log not found) tags)))))
    (permanent-p stamp)))
				       
(defun permanent-p (stamp)
  (let* ((header-file (stamp-to-pathname stamp 'hdr))
	 (header nil))
    (with-db-input-file (intermediate-stream header-file t)
			(setf header (read-log-header intermediate-stream)))
    (let ((reference (reference-of-log-header header)))
      (with-unwind-error (nil)
			 (bool-of-ibool-term (cdr (property-of-term '|permanent| reference)))))))
			       
(defun garbage-p (stamp)
  (let* ((header-file (stamp-to-pathname stamp 'hdr))
	 (header nil))
    (with-db-input-file (intermediate-stream header-file t)
			(setf header (read-log-header intermediate-stream)))
    (let ((reference (reference-of-log-header header)))
      (with-unwind-error (nil)
			 (bool-of-ibool-term (cdr (property-of-term '|garbage| reference)))))))
			       
(defun set-garbage (tags &optional filter ghostp)
  (let ((stamps (or (mapcar #'cdr (match-db-environment-all tags))
		    (raise-error (error-message '(set garbage log not found) tags)))))
    (mapcar #'(lambda (stamp)
		(unless (permanent-p stamp)
		  (format t "Set Garbage ~a ~a~%" (sortable-datetime-string (cddr stamp)) (cdar stamp))
		  (unless ghostp
		    ;;(break "sg")
		    (set-log-property stamp (cons '|garbage| (ibool-term t))))))
	    (if filter (funcall filter stamps) stamps)))
  nil)

(defun set-garbage-but1 (tags)
  (set-garbage tags #'cdr))

(defun set-garbage-but2 (tags)
   (set-garbage tags #'cddr))

(defun set-garbage-any (tags)
  (let ((stamps (or (mapcar #'cdr (match-db-environment-all-sub tags))
		    (raise-error (error-message '(set garbage log not found) tags)))))
    (mapcar #'(lambda (stamp) (unless (permanent-p stamp)
				      (set-log-property stamp (cons '|garbage| (ibool-term t)))))
	    stamps)))

(defun unset-garbage (tags)
  (let ((stamps (or (filter #'garbage-p (mapcar #'cdr (match-db-environment-all tags)))
		   (raise-error (error-message '(unset garbage log not found) tags)))))
    (set-log-property (car stamps) (cons '|garbage| (ibool-term nil)))))

(defun unset-garbage-all (tags)
  (let ((stamps (or (mapcar #'cdr (match-db-environment-all tags))
		   (raise-error (error-message '(unset garbage log not found) tags)))))
    (mapcar #'(lambda (stamp) (set-log-property stamp (cons '|garbage| (ibool-term nil))))
	     stamps)))

(defun unset-permanent-all (tags)
  (let ((stamps (or (mapcar #'cdr (match-db-environment-all tags))
		   (raise-error (error-message '(unset garbage log not found) tags)))))
    (mapcar #'(lambda (stamp) (set-log-property stamp (cons '|garbage| (ibool-term nil))))
	     stamps)))


(defvar *old-master-pathname* (concatenate 'string *master-pathname* *passive-dir* "/"))

(defun old-file-p (pathname)
  (string= "AC" (pathname-type pathname)))
		       
(defun old-files-of-process (process)
  (filter #'old-file-p (directory-listing process)))

(defun undo-passive-pathname (pathname &optional type)
  (let ((dirs (pathname-directory pathname)))
    (make-pathname :name (if type
			     (file-namestring pathname)
			     (pathname-name pathname))
		   :directory (append (butlast (butlast dirs))
				      (last dirs)))))
(defun undo-collect-file ()
  (let* ((processes (mapcar #'pathname-to-directory (directory-listing *old-master-pathname*)))	 
	 (oldfiles (flatten (mapcar #'old-files-of-process processes))))
    (mapcar #'(lambda (x) (rename-file (namestring x) (namestring (undo-passive-pathname x))))
	    oldfiles)))

(defun undo-again (processes)
  (let* ((dirs (mapcar #'(lambda (x) (append (butlast (butlast (pathname-directory x)))
					     (last (pathname-directory x))))
		       processes))
	 (files (flatten (mapcar #'(lambda (x) (filter #'old-file-p (directory-listing x)))
				 (mapcar #'(lambda (y) (make-pathname :directory y)) dirs)))))
    
    (mapcar #'(lambda (x) (rename-file x (namestring (make-pathname :name (pathname-name x)
								    :directory 
								    (pathname-directory  x)))))
	    (mapcar #'namestring  files))))

(defun chmod-data ()
  (let* ((dirs (mapcar #'(lambda (x) (make-pathname :directory (append (pathname-directory x)
								       (list *data-dir*))))	     
		       (db-process-list nil t)))
	 (files (flatten (mapcar #'(lambda (x) (filter #'(lambda (y)
							   (not (member (pathname-name y)
									*db-extras* :test #'string=)))
						       (directory-listing x))) dirs))))
    (mapcar #'(lambda (x) (libc-chmod (namestring x))) files)))

(defunml (|view_logs| (unit) :declare ((declare (ignore unit)))) 
    (unit -> term)
  
  (view-logs))


(defunml (|show_log_envs| (update reset buildp))
    (bool -> (bool -> (bool -> (((int |#| (tok list)) list) |#| ((int |#| (tok list)) list)))))

  (mlet* (((act pass procs) (collect-logs-aux nil update (or update reset) buildp)
	   (declare (ignore procs))))
	 ;;(setf -act act -pass pass -procs procs) (break "spl1")
	 (cons
	  (mapcar #'(lambda (dbe) (cons (time-of-stamp (cdr dbe)) (car dbe)))
		  (filter-db-environments-aux #'(lambda (addr stamp)
						  (declare (ignore addr))
						  ;;(setf -addr addr -stamp stamp) (break "spl")
						  (member stamp pass :key #'cadr :test #'equal))))
	  (mapcar #'(lambda (dbe) (cons (time-of-stamp (cdr dbe)) (car dbe)))
		  (filter-db-environments-aux #'(lambda (addr stamp)
						  (declare (ignore addr))
						  ;;(setf -addr addr -stamp stamp) (break "spl")
						  (member stamp act :key #'cadr :test #'equal)))))))


(defun stamps-of-dbes (paddr)
  (let ((dbes (filter-db-environments paddr)))
    (mapcar #'(lambda (dbe) (cdr dbe))
	    dbes)))


(defunml (|start_collect| (unit) :declare ((declare (ignore unit)))) 
    (unit -> term)  
  (start-collect))

(defunml (|view_collect| (unit)  :declare ((declare (ignore unit)))) 
    (unit -> term) 
  (view-collect))

(defunml (|view_remaining| (unit) :declare ((declare (ignore unit)))) 
    (unit -> term)  
  (view-remaining))

(defunml (|view_logs_with_properties| (props) :declare ((declare (ignore unit))))
    ((tok list) -> term)  
  (view-logs-with-properties props))

(defunml (|abort_collect| (unit) :declare ((declare (ignore unit)))) 
    (unit -> term)  
  (abort-collect))

(defunml (|finish_collect| (unit) :declare ((declare (ignore unit)))) 
    (unit -> term)  
  (finish-collect t))

(defunml (|set_ancestor_count| (num))
    (int -> int)
  
  (let ((old *minimum-ancestor-count*))
    (setf *minimum-ancestor-count* num)
    old))

(defunml (|set_time_span| (minutes))
    (int -> int)
  (let ((old *maximum-time-span*))
    (setf *maximum-time-span* minutes)
    old))

(defunml (|last_collection| (unit)  :declare ((declare (ignore unit))))
    (void -> term)
  (last-collection))

(defunml (|set_permanent| (tags))
    ((tok list) -> void)
  (set-perm tags))

(defunml (|unset_permanent| (tags))
    ((tok list) -> void)
  (unset-perm tags))

(defunml (|permanent_log_p| (tags))
    ((tok list) -> bool)
  (perm-p tags))

(defunml (|permanent_logs| (unit) :declare ((declare (ignore unit))))
    (void -> term )
  (permanent-logs))

(defunml (|set_garbage| (tags))
    ((tok list) -> void)
  (set-garbage tags)
  nil)

(defunml (|set_garbage_but| (tags))
    ((tok list) -> void)
  (set-garbage-but1 tags))

(defunml (|set_garbage_but2| (tags))
    ((tok list) -> void)
  (set-garbage-but2 tags))

(defunml (|set_garbage_any| (tags))
    ((tok list) -> void)
  (set-garbage-any tags))

(defunml (|unset_garbage| (tags))
    ((tok list) -> void)
  (unset-garbage tags))

(defunml (|unset_garbage_all| (tags))
    ((tok list) -> void)
  (unset-garbage-all tags))

;;;;	
;;;;	
;;;;	
;;;;	start_collect ();; 
;;;;	
;;;;	# for debugging to reset log tree.
;;;;	(clrhash *log-tree*)
;;;;	(build-log-tree)
;;;;
;;;;	(show-log-tree t)
;;;;
;;;;	(setf *gc-override-refcount* t);;
;;;;	(setf *ignore-owner-p* t);;
;;;;	set_time_span 0;;
;;;;
;;;;	let forceg n = unset_permanent n; set_garbage n;;
;;;;	set_garbage ``rmarkb_test``;;
;;;;
;;;;	# reads props, so should show updates.
;;;;	let (collect,remain) = (show_log_envs false true true) in tty_print "collect"; print_environments collect; tty_print "remain"; print_environments remain;;
;;;;	
;;;;	# Does not read props, but fast
;;;;	let (collect,remain) = (show_log_envs false false false) in tty_print "collect"; print_environments collect; tty_print "remain"; print_environments remain;;
;;;;	
;;;;	db_envs_print ``rmarkb``;;
;;;;	
;;;;    let collect_stats = finish_collect ();;
;;;;	
;;;;	(show-gc-alist)
;;;;	........................     deleted   collected remaining
;;;;	
;;;;	local gc
;;;;	
;;;;	


(defun process-directory (pid) (db-extend-pathname (list (string pid))))

(defun process-headers (pid)
  (mapcar #'get-log-header
	  (get-process-hdr-files (process-directory pid))))

(defun process-header-stamps (pid)
  (mapcar #'stamp-of-log-header
	  (mapcar #'get-log-header
		  (get-process-hdr-files (process-directory pid)))))

(defun process-derivations (pid)
  (mapcar #'(lambda (der)
	      (cons (term-to-stamp (parent-of-iderivation-term der))
		    (term-to-stamp (child-of-iderivation-term der))))
	  (mapcar #'db-read-file
		  (filter #'(lambda (p) (string= "DER" (pathname-type p)))
			  (directory-listing (db-derivation-path-pid pid))))))    

(defun process-log-name (pid) (db-extend-pathname (list (string pid)) "process" 'log))

(defun process-log (pid) 
  (db-read-list (process-log-name pid)))

(defun process-log-p (pid)
  (and (probe-file (process-log-name pid)) t))

;; closed and not checkpoint logs.
;; 
(defun non-checkpoint-logs (plog)
  (let ((alist (mapcar #'(lambda (plog) (cons (kind-of-iloglog-term plog)
					      (term-of-iloglog-term plog)))
		       plog)))
    (let ((checkpoints (filter #'(lambda (a) (eql 'checkpoint (car a))) alist))
	  (closed (filter #'(lambda (a) (eql 'close (car a))) alist)))

      (mapcar #'(lambda (s) (term-to-stamp (cdr s)))
	      (filter #'(lambda (c) (not (member (cdr c) checkpoints :key #'cdr :test #'compare-terms-p)))
		      closed)))))



;; any log without a derivation, ie a leaf.
;; any log with a non-local derivation.
(defun locally-persistent-logs (logs derivations)
  (append
   ;; no derivations : ie leafs.
   (filter #'(lambda (log) (null (member log derivations :key #'car :test #'equal))) logs)

   ;;
   (filter #'(lambda (log) (member log derivations
				   :key #'cdr
				   :test #'(lambda (lstamp pstamp)
					     ;;(setf -l lstamp -p pstamp) (break)
					     (not (eql (process-id-of-stamp lstamp)
						       (process-id-of-stamp pstamp))))))
	   logs))
  )


(defun local-gc-logs (pid)
  (let ((hdrs (process-header-stamps pid)))
    (if (not (process-log-p pid))
	(values nil hdrs)
	(let ((gc (set-difference (non-checkpoint-logs (process-log pid))
				  (locally-persistent-logs hdrs
							   (process-derivations pid))
				  :test #'equal)))
	  (values gc
		  (filter #'(lambda (stamp)
			      (probe-file (stamp-to-filename-aux stamp nil 'log)))
			  (set-difference hdrs gc :test #'equal)))))))
    
(defun local-key-of-file (pathname)
  (list*

   (or (let ((ptype (pathname-type pathname)))
	 (when ptype (intern-system ptype)))
       '|no-type|)
   
   (let ((seq (read-from-string (pathname-name pathname))))
     (unless (integerp seq)
       (break "key-of-file-aux"))
     seq)

   ))

(defun pid-of-process-path (pathname)
  
  (or (let ((term (with-ignore (let ((plogname (db-extend-pathname (list (namestring pathname)) "process" 'log)))
				 (when (probe-file plogname)
				   (db-read-file plogname))))))
	(when term
	  (when (eql 'pid (kind-of-iloglog-term term))
	    (token-of-itoken-term (term-of-iloglog-term term)))))
	      
      (intern-system (car (last (pathname-directory pathname))))))


	  
;; doesn't attempt deletion only moves to old.
(defun local-gc (inpid &optional ghostp)
  (let ((pid (intern-system inpid)))

    (when ghostp (terpri))
  
    (create-directory (directory-namestring (pid-to-path-aux pid (list "old"))))
    (create-directory (directory-namestring (pid-to-path-aux pid (list "old" "data"))))

    ;; collect data files prior to walking logs so that files added after a log walked will not be visible
    ;; for collection. problem: file written but log not yet updated. Then could be moved when not appropriate.
    ;; either allow only when process dead or limit collection of files to those seen in collected logs.
    ;; but it may be that a ref is created dynamically by process later without prior record but still assumes
    ;; on disk. Essentially active process can not rely on persist data actually being persistent.
    ;; Until process is dead we can never be certain there is not a core memory reference to a file.

    (format t ";;;;	Local GC ~a~%" pid)

    ;; move gc logs, leave headers.
    (mlet* (((gclogs persistent-logs) (local-gc-logs pid)))

	   (dolist (l gclogs) (format t ";;;;	Collecting ~a~%" (sequence-of-stamp l)))
	   (dolist (l persistent-logs) (format t ";;;;	Not Collecting ~a~%" (sequence-of-stamp l)))

	   (unless (and nil;; still possible to move unreferenced files?
			(null gclogs))

	     (let ((datafiles (make-hash-table :test #'equal)))
	       (mapc #'(lambda (fname)
			 (setf (gethash (local-key-of-file fname) datafiles)
			       (cons nil fname)))
		     (db-files-of-process (pid-to-path-aux pid nil)))

	       ;;(setf -datafiles datafiles)

	       (dolist (gclog gclogs)
		 (if ghostp
		     (format t ";;;;	Would move ~a ~%   ~a ~%"
			     (stamp-to-pathname gclog 'log) (stamp-to-filename-aux gclog (list "old") 'log))

		     (rename-file (stamp-to-pathname gclog 'log) (stamp-to-filename-aux gclog (list "old") 'log))))


	       ;; mark referenced files
	       (let ((stamptable (make-hash-table :size 10000 :test #'equal)))

		 ;;(setf -stamptable stamptable)
		 
		 ;; visit remaining logs.
		 (walk-db-logs (mapcar #'(lambda (s) (stamp-to-pathname s 'log)) persistent-logs)
			       #'(lambda (term)
				   (db-visit-term-aux term
						      #'(lambda (stamp type)
							  ;;(setf -stamp stamp -type type) (break "lgc")
							  (when (eql pid (process-id-of-stamp stamp))
							    (let ((key (cons type stamp)))

							      (unless (gethash key stamptable)
								(setf (gethash key stamptable) t)
				   
								;; continue since not previously visited.
								t)))))))
				 
		 (maphash #'(lambda (key v)
			      (declare (ignore v))

			      (let ((stamp (cdr key)))

				(let ((fent (gethash (list* (car key)
							    (sequence-of-stamp stamp) )
						     datafiles)))
				  (if fent
				      (setf (car fent) t)
				      (break "stamp but no file ???")
				      ))))

			  stamptable)

		 (when ghostp (terpri))

		 ;; move unrefd.
		 (maphash #'(lambda (key fent)
			      (when (null (car fent))
				;; move
				(if ghostp
				    (format t ";;;;	Would move ~a ~%   ~a ~%"
					    (cdr fent)
					    (pid-to-filename-aux pid
								 (list "old" "data")
								 (princ-to-string (cdr key))
								 (car key)))
				    (rename-file (cdr fent)
						 (pid-to-filename-aux pid
								      (list "old" "data")
								      (princ-to-string (cdr key))
								      (car key)))
				    )))
			  datafiles)
		 ))))))


#|(defun local-gc-all (&optional (ghostp t))
  (mapc #'(lambda (pid) (local-gc pid ghostp))
	
	))
|#

