
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;;
;;;; -docs- (mod com)
;;;;
;;;;	Theorems : a theorem may define two definitions: a statement and a proof.
;;;;
;;;;	There are two definition tables related to theorems. The first contains
;;;;	the theorem statement effects. The second the theorem proof effects.
;;;;	The statement table only allows lookup by object-address, while the
;;;;	proof table allows lookup by termof instance or object-address.
;;;;
;;;;	Proof reference : when a termof is expanded revealing the extract, a reference
;;;;	to the proof is considered to be made.
;;;;
;;;;	Statement reference : when a statement is injected into a proof via invocation 
;;;;	of a lemma or extract rule, a reference to the statement is considered to be made.
;;;;
;;;;	For purposes of cycle detection, the statement may reference the appropriate proofs.
;;;;
;;;;	It would be a simple matter to allow for a statement to reference more than one 
;;;;	proof and also for a proof to be referenced by more than one statement, though 
;;;;	the latter seems of limited utility.
;;;;
;;;;	Statement and proof references of a proof are segregated. There seems to be limited
;;;;	benefit to this, however it costs nothing.
;;;;
;;;;	Termof : Eventually multiple new syntaxes for the termof will be supported. It is 
;;;;	possible for termofs and abstraction instances to clash. This is detected and a
;;;;	warning is issued. In contexts where it is ambiguous, the termof has precedence.
;;;;
;;;;  -page-
;;;;
;;;;	<termof>		:<id{object name}>{<a:l> list}()
;;;;	
;;;;	<references>		: ( <dependency> sexpr{statement references}
;;;;				  . <dependency> sexpr{proof references} )
;;;;	
;;;;	<statement>		: statement[<definition>
;;;;					    <term{termof}>
;;;;					    <term{lemma}>
;;;;					    <dependency{proof}> list]
;;;;
;;;;	<proof>			: proof[<definition>
;;;;					<term{termof}> list
;;;;					<term{extract}>
;;;;					<expansion>	
;;;;					<references>]
;;;;
;;;;
;;;;	<expansion>	: expansion['termof]
;;;;
;;;;	 ** a proof definition is very similar to an abstraction.
;;;;
;;;;	Proof references are optional, if they are not present then related
;;;;	functions fail when called.
;;;;
;;;;  -page-
;;;;
;;;;	lemma-of-statement(<statement>)			: <term>
;;;;	termof-of-statement(<statement>)		: <term>
;;;;	proof-references-of-statement(<statement>)	: <dependency> list
;;;;
;;;; -doct- (mod com data)
;;;;
;;;;	<stm-export-term>	: !statement(<term{termof}>;
;;;;					     <term{lemma}>;
;;;;					     <dependencies-term>)
;;;; -doct- (mod com)
;;;;
;;;;	export-statement(<statement>)			: <term>
;;;;	import-statement(<term>)			: <statement>
;;;;
;;;;	statement-lookup(<oa>)				: <statement>
;;;;	reference-lemma(<oa>)				: (<term{termof}> <term{lemma}>)
;;;;
;;;;
;;;;	termof-of-proof(<proof>)			: <term>
;;;;	extract-of-proof(<proof>)			: <term>
;;;;	expansion-of-proof(<proof>)			: <expansion>
;;;;
;;;;	references-of-proof(<proof>)			: <dependency> sexpr
;;;;	proof-references-of-proof(<proof>)		: <dependency> list
;;;;	statement-references-of-proof(<proof>)		: <dependency> list
;;;;
;;;; -doct- (mod com data)
;;;;
;;;;	<prf-export-term>	: !proof(<term{termof}>;
;;;;					 <prf-extract>;
;;;;					 <prf-references-term>)
;;;;	
;;;;	<prf-extract>		: !extract(<term>)
;;;;				| !extract()
;;;;	
;;;;	<prf-references-term>	: !dependency_cons(<dependency-term-list>;
;;;;						   <dependency-term-list>)
;;;;				| !void()
;;;; -doct- (mod com)
;;;;
;;;;
;;;;	export-proof(<proof>)				: <term>
;;;;	import-proof(<term>)				: <proof>
;;;;
;;;;	proof-lookup(<oa>)				: <proof>
;;;;	;;proof-of-termof(<term>)				: <proof>
;;;;
;;;;	expand-termof(<term>)				: <term>
;;;;	 ** returns input if no termof.
;;;;
;;;;	termof syntax:
;;;;
;;;;	<termof>	: <id{object id}>{<a:l> list}()
;;;;
;;;;	termof object id may refer to a STM or PRF object.
;;;;
;;;;	Clash of termofs is similar to that for abstractions.
;;;;	Match is similar as well.
;;;;
;;;;  -page-
;;;;
;;;;	ML:
;;;; -doct- (mod com ml)
;;;;
;;;;	statement_lookup	: object_id -> term
;;;;	termof_of_statement	: object_id -> term
;;;;
;;;;	proof_lookup		: object_id -> term{termof} # term{extract}
;;;;	extract_of_proof	: object_id -> term
;;;;	termof_of_proof		: object_id -> term
;;;;						    
;;;;	proof_of_termof		: term -> object_id
;;;;
;;;;	 * extract terms have !extract wrappers to distinquish proofs with no extract.
;;;;
;;;;	expand_termof		: term -> term
;;;;	is_termof		: term -> bool
;;;;	
;;;;	map_statements		: (object_id -> unit) -> unit
;;;;	map_proofs		: (object_id -> unit) -> unit
;;;;
;;;;
;;;;
;;;; -doce-






;; RLE NB  proofs with no extract should be allowed. Then can distribute references
;;  even if proof not complete.
;; RLE NB refiner should use object address in lemma and extract rules.


;; termof is used to locate extract, thus it should be part of the proof_object.

;; substance term is lemma.
;; proof dependencies are in substance dependencies.


;; proofs should be in order that they were in source.
;; thus they should have own field in substance, ie looking at
;; proof dependencies is not adequate.

(defclass statement-substance (substance)
  ((proofs :reader proofs-of-statement-substance
	       :writer set-statement-substance-proofs
	       :initform nil
	       :initarg proofs)
   (extract :reader extract-of-statement-substance
	    :writer set-statement-substance-extract
	    :initform nil
	    :initarg extract)))

;; statement substance orig:
;; (define-primitive |!statement_substance| () (proofs sub))
;; sub should be first so that adding new fields does not change position.
;; Expect many instances of old form so maintain indefinitely.

;;(define-primitive |!statement_substance| () (sub proofs extract))
(define-primitive |!statement_substance| () () t)


(defun sub-of-istatement-substance-term (term)
  (let ((bts (bound-terms-of-term term)))
    (let* ((aterm (term-of-bound-term (car bts)))
	   (btsb (cdr bts))
	   (bterm (term-of-bound-term (car btsb)))
	   (btsc (cdr btsb)))
      
      (cond
	((null btsc) bterm)
	(t aterm)))))

(defun proofs-of-istatement-substance-term (term)
  (let ((bts (bound-terms-of-term term)))
    (let* ((aterm (term-of-bound-term (car bts)))
	   (btsb (cdr bts))
	   (bterm (term-of-bound-term (car btsb)))
	   (btsc (cdr btsb)))
      
      (cond
	((null btsc) aterm)
	(t bterm)))))

(defun extract-of-istatement-substance-term (term)
  (let ((bts (bound-terms-of-term term)))
    (let ((btsc (cddr bts)))
      (cond
	((null btsc) nil)
	(t (term-of-bound-term (car btsc)))))))

(defun istatement-substance-term (sub proofs ext)
  (instantiate-term (istatement-substance-op)
		    (list* (instantiate-bound-term sub)
			   (instantiate-bound-term proofs)
			   (when ext
			     (list (instantiate-bound-term ext))))))

(defun istatement-substance-term-p (term)
  (istatement-substance-term-op-p term))


(defmethod data-import ((substance statement-substance) super)
  (let ((term (call-next-method substance super)))
    (set-statement-substance-proofs (oids-of-ioids-term
				     (proofs-of-istatement-substance-term term))
				    substance)
    (let ((ext (extract-of-istatement-substance-term term)))
      (when ext
	(set-statement-substance-extract (term-to-extract ext) substance)))
    
    (sub-of-istatement-substance-term term)))

(defmethod data-export ((substance statement-substance) sub)
  (call-next-method substance
		    (istatement-substance-term sub
					       (ioids-term (proofs-of-statement-substance substance))
					       (extract-to-term (extract-of-statement-substance substance)))))

(defun new-statement-substance (term props proofs &optional ext)
  (make-instance 'statement-substance 'term term 'properties props 'proofs proofs 'extract ext))



;; cacheing proofs in def would allow substance to be abbreviated.
(defstruct (statement (:include abstraction-super))
  ;; le		;; level expr derived from statement.
  (termof nil)
  (ref-dag)
  )

(defun ref-dag-of-statement (s) (statement-ref-dag s))

(defun statement-visible-p (s tv)
  (definition-visible-p s tv))

(defun proofs-of-statement (s)
  (proofs-of-statement-substance (substance-of-definition s 'statement-substance)))

(defun proof-of-statement (s)
  (or (car (proofs-of-statement s))
      (raise-error (error-message (list (oid-of-definition s)) '(statement proof not)))))


(defun lemma-of-statement (s)
  (term-of-substance (substance-of-definition s 'statement-substance)))

(defun termof-of-statement (s)
  (or (statement-termof s)
      (setf (statement-termof s)
	    (termof-of-term (oid-of-definition s)
			    (lemma-of-statement s)))))

(defun termof-of-statement-? (s)
  (or ;;(statement-termof s)
      (setf (statement-termof s)
	    (name-termof-of-term (statement-name s)
			    (lemma-of-statement s)))))


(defun import-statement (term)
  (let ((substance (provide-data (term-to-data term) 'statement-substance)))
    (let ((statement (make-statement :substance substance
				     :ref-dag (new-ref-dag (current-object-id)
							   substance
							   (when (proofs-of-statement-substance substance) 0)))))

      #| (let ((ext (extract-of-statement-substance substance)))
	(when ext
	  (import-termof statement
			 (lemma-of-statement statement)
			 (extract-of-statement-substance substance)
			 (statement-name statement)))) |#

      statement)))
      
    

(defmacro statement-table (stamp tag &rest keys &key &allow-other-keys)
  `(define-definition-table
      ,stamp
      (list 'statement ,tag)
    nil
    :import-f #'import-statement
    
    ,@keys))


;; if import-f then get name-table else plain def-table.
(defun new-statement-table (stamp tag &optional import-f)
  (if import-f
      (name-table stamp (list 'statement tag) nil
		  :import-f import-f
		  :visibility (new-table-visibility `|visible_statements| "get_visible_statements"))
      (statement-table stamp tag :import-f #'import-statement
		       :visibility (new-table-visibility `|visible_statements| "get_visible_statements"))))

(defun lookup-statement-def (oid &optional nil-ok-p dont-note)
  (if (resource-p 'statements)
      (let ((statements (resource 'statements)))
	(let ((def (definition-lookup-by-oid statements oid nil-ok-p (current-transaction-stamp) dont-note)))
	  (let ((tv (get-definition-table-visibility statements)))
	    (unless (statement-visible-p def tv)
	      (unless (and nil  ;; fttb to catch improper lookups.
			   nil-ok-p)
		(setf -def def)
		;;(break "lookup-statement-def not visible")
		(raise-error (oid-error-message (cons (oid-of-definition def) (when tv (list tv)))
						'(statements lookup visible not)
						(list (statement-name def)))))))
	  def))
      (raise-error (oid-error-message (list oid) '(statements lookup not)))))

(defun statement-lookup (oid)
  (let ((def (lookup-statement-def oid nil nil)))
    (lemma-of-statement def)))

(defun lemma-reference (oid)
  (let ((statement (lookup-statement-def oid nil t)))
    (dependency-note-reference 'statement (dependency-of-definition statement))
    statement))


(defvar *ref-validity-check* nil)

(defun ref-validity-check-index (i)
  (if (eql t *ref-validity-check*)
      t
      (< i *ref-validity-check*)))

(defvar *recent-ref-validity-failures* nil)

(defun check-ref-validity (statement)
  (let ((rdag (ref-dag-of-statement statement)))
    (setf -rdag rdag)
    
    (let ((tv (get-definition-table-visibility (resource 'statements))))
      (unless (statement-visible-p statement tv)
	;;(setf -tv tv -statement statement) (break "crvv")
	(fooa (oid-error-message (cons (oid-of-definition statement) (when tv (list tv)))
				 '(ref validity check visible not)
				 (list (statement-name statement))))))
    (when *ref-validity-check*
      (and 
       (let ((i (index-of-ref-dag rdag)))
	 (unless (and i (ref-validity-check-index i))
	   ;;(break "crvh")
	   (pushnew (statement-name (lookup-statement-def (oid-of-ref-dag rdag) nil t)) *recent-ref-validity-failures*)
	   (format t "check_ref_validity ~a~%" (statement-name (lookup-statement-def (oid-of-ref-dag rdag) nil t)))
	   (fooa  (oid-error-message (list (oid-of-ref-dag rdag)) '(ref validity check) (list (statement-name (lookup-statement-def (oid-of-ref-dag rdag) nil t)) i)))
	   ;;(raise-error (oid-error-message (list (oid-of-ref-dag rdag)) '(ref validity check) (list (statement-name (lookup-statement-def (oid-of-ref-dag rdag) nil t)) i)))
	   ))

       ;;(break "crv")
       t
       ))))




(defun lemma-use (oid)
  (let ((statement (lemma-reference oid)))
    (check-ref-validity statement)
    (list (termof-of-statement statement) (lemma-of-statement statement)) ))

(defun lemma-use-named (oid)
  (let* ((statement (lemma-reference oid))
	 (termof (termof-of-statement statement)))
    (check-ref-validity statement)
    (list (named-termof-term (statement-name statement) termof)
	  (lemma-of-statement statement))))

  

(defunml (|statement_lookup| (oid))
    (object_id -> term)
  (statement-lookup oid)
  )


(defunml (|termof_of_statement| (oid))
    (object_id -> term)
  (termof-of-statement (lookup-statement-def oid)))

(defunml (|proofs_of_statement| (oid))
    (object_id -> (object_id list))

  (proofs-of-statement (lookup-statement-def oid)))



;; RLE NAP ability to cache def on term (in compression table) for import.

;; RLE TODO refresh of statement refresh proof reference. 


;;;
;;; (defstruct (proof 
;;;   references)
;;;
;;; would be better and expand-termof becomes expand.
;;;

(defstruct (proof (:include definition))
  ref-dag ;; ref-dag
  )

;;LAL
(defun termof-of-proof (p) (lhs-of-abstraction p))
(defun extract-of-proof (p) (rhs-of-abstraction p))

;;(defun named-termof-of-proof (p) (named-termof-term (lhs-of-abstraction p)) (proof-name p))
(defun ref-dag-of-proof (p) (proof-ref-dag p))
       

;;;;	
;;;;	ref-dag : list of prf or stm object-ids directly referenced by containing object.
;;;;	  - if object is stm then is simply list of stm proofs.
;;;;	    maybe could map of stm proofs and produce union of proofs' refs? but why.
;;;;	
;;;;	

;;;;	
;;;;	ref-dag : dag of oids.
;;;;	
;;;;	 only subset of lemma table listed in ref dag prior to some member is 
;;;;	 available during refinement.
;;;;	Either
;;;;	 - eagerly, produce that subset
;;;;	 - lazily, at reference check if member of subset.
;;;;	
;;;;	so eager wins
;;;;
;;;;	Construct from stm, and prf table.
;;;;	  - produce linear stm list assign indices for comparison.
;;;;	  - have index lookup check for rehash required.
;;;;	  
;;;;	  - if stm has no prf place in distinct class from those with proofs.
;;;;	     * bimodal call which optionally disallow stm with no proofs.
;;;;	
;;;;	
;;;;	Map over stm table producing rdag list
;;;;
;;;;	<rdag>	: (<oid>,<tok{type:prf|stm}>) list, <nat{index}>
;;;;	<rdag>	: <oid> list, <nat{index}>
;;;;	  * type 
;;;;	  
;;;;	
;;;;	
;;;;	Iteratively create grouping :
;;;;	
;;;;	<rgrp>	: <rdag> list
;;;;	
;;;;	 group (<grp> list <rdag> list{remainder})
;;;;	  foreach rdag if all stm-prfs occur in grp list then add to new grp. otherwise lea
;;;;	
;;;;	
;;;;	suppose prf a refs prf b via termof expansion.
;;;;	  - stm A uses prf a and stm B uses prf b. 
;;;;	
;;;;	  Thus A depends on b refs without depending on B
;;;;	
;;;;	set references of statement at import to be proofs
;;;;	at import set references from  proof substance-dependencies.
;;;;	
;;;;	then make stm grps
;;;;	  - stm -> prf list
;;;;	  - prf -> stm|prf list
;;;;	
;;;;	
;;;;	grps include prfs
;;;;	
;;;;	
;;;;	then grp0 is all stms whose prfs referenc no stms or proofs.
;;;;	 grp1 is all stms whose prfs directly refence only stms of grp 1
;;;;	    or stms closure of pr
;;;;	
;;;;	validity : allows eager failure of cycles. 
;;;;	  - does not guarantee absence of cycles lib must still check for ref cycles.
;;;;	    but if properly called it would cause cycle failures during tactic call
;;;;	    allowing tactic to find diff proof.

;;;;
;;;;	TODO : adapt to using DDG layering functions.
;;;;


(defstruct ref-dag

  oid
  refs
  index

  ;; tis possible that validity is suspect thus would like to segregate according to judgement of
  ;; validity.
  ;; fttb nil means don't know and 0 means ok. 
  (validity nil)
  )

(defun validity-of-ref-dag (rdag) (ref-dag-validity rdag))
(defun oid-of-ref-dag (rdag) (ref-dag-oid rdag))
(defun refs-of-ref-dag (rdag) (ref-dag-refs rdag))
(defun index-of-ref-dag (rdag) (ref-dag-index rdag))

(defun new-ref-dag (oid substance &optional validity)
  (let ((refs (mapcan #'(lambda (envdeps)
			  (mapcan #'(lambda (deps)
				      (when (member (tag-of-dependencies deps)
						    '(statement proof))
					(mapcar #'oid-of-dependency
						(list-of-dependencies deps))))
				  (list-of-stamp-dependencies envdeps)))
		      (list-of-stamp-dependencies
		       (dependencies-of-substance substance))))) 
    (make-ref-dag
     :oid oid
     :refs refs
     :validity validity
     )))


(defun ref-dag-of-proof (p)
  (let ((refs (proof-ref-dag p)))
    (when (null refs)
      (raise-error (oid-error-message (list (oid-of-definition p))
				      '(proof ref-dag))))
    refs))

#|
(defun proof-references-of-proof (p)
  (cdr (assoc 'proof (references-of-proof p))))

(defun statement-references-of-proof (p)
  (cdr (assoc 'lemma (references-of-proof p))))
|#

;; proof substance is main goal of inf-tree and/or extract. Be extracting goal
;; may be able to avoid read of inf-tree is some circumstances..
(defclass proof-substance (substance)
  ((goal :reader goal-of-proof-substance
	 :writer set-proof-substance-goal
	 :initform nil
	 :initarg goal)

   (extract :reader extract-of-proof-substance
	    :writer set-proof-substance-extract
	    :initform nil
	    :initarg extract)
   ))

(defun new-proof-substance (properties goal extract)
  (make-instance 'proof-substance
		 'term (ivoid-term)
		 'properties properties
		 'goal goal
		 'extract extract))

(define-primitive |!proof_substance| () (goal extract))

(defmethod data-import ((substance proof-substance) super)
  (let ((term (call-next-method substance super)))
    
    (set-proof-substance-goal (iinf-goal-term-to-goal (goal-of-iproof-substance-term term)) substance)
    ;;(setf ttt term) (break "s")
    (set-proof-substance-extract (term-to-extract (extract-of-iproof-substance-term term))
				 substance)

  (values)))

(defmethod data-export ((substance proof-substance) sub)
  (call-next-method substance
		    (iproof-substance-term (goal-to-term
					    (goal-of-proof-substance substance))
					   (extract-to-term
					    (extract-of-proof-substance substance)))))


(define-primitive |!extract| () (term))


;; !extract(<extract>)
;; !extract()
(defun extract-to-term (ext)
  (if ext
      (iextract-term ext)
      (instantiate-term (iextract-op) nil)))

(defun term-to-extract (term)
  (let ((bts (bound-terms-of-term term)))
    (when bts
      (term-of-bound-term (car bts)))))




;;;;	extract/termof :
;;;;	
;;;;	quick review : level expressions in termof come from examining goal of stm.
;;;;	
;;;;	termof{<stm>:o,\v:l<other level expressions>}()
;;;;	  - stm caches first availble extract from substance proof list if extract-required property
;;;;	    true. 
;;;;	     * less ephemeral.
;;;;	     * setting pr/extract may touch stm and thus termof expansion only if extract different.
;;;;	     * avoids broadcast prfs.
;;;;	     * avoids visibility of prfs.
;;;;	
;;;;	   - termof variants :
;;;;	       * might have other termof by which to access extract.
;;;;	           - ie symbolic by name. 
;;;;	
;;;;	Lazy extracts : since replay now feasible. compute extracts on demand (and cache)?
;;;;	  - possiblely eagerly extract if have some evidence extract required.
;;;;	  - any change to goal/prf list invalidates ext cache.
;;;;

;; for a while created all the instantiated all the termofs created by lemma/extract rules
;; with two special variable if the ref'd lemma contained a termof in its extract.
;; this of course is somewhat dangerous but fttb avoids have to recompute all the extracts.
;; which I will do and then this should be removed.
;; but also used in migration to convert termof instances.
(defun fixup-extract (term)
  (let ((statements (resource 'statements)))
    (term-walk-ops
     ;; at one time had extra !extract wrapper in stm-substance.
     (if (iextract-term-p term)
	 (term-of-iextract-term term)
	 term)
     ;; had extra \\v special parm. also covert id{<rest>} -> TERMOF{<oid{id}>, <rest>}
     #'(lambda (term)

	 (let ((parms (parameters-of-term term)))
	   ;;(when (and (car parms) (level-expression-parameter-p (car parms))) (setf -term term ) (break))
	   (let ((aparm (car parms))
		 (bparm (cadr parms)))
	     (if (and aparm (special-level-parameter-p aparm))
		 ;;bparm (special-level-parameter-p bparm)
		 (let ((sdefs (name-table-lookup statements (id-of-term term)
						 (current-transaction-stamp))))
		   (if (and sdefs (null (cdr sdefs)))
		       (instantiate-term (instantiate-operator
					  'termof
					  (cons (oid-parameter (oid-of-definition (car sdefs)))
						(if (and bparm (special-level-parameter-p bparm))
						    (cdr parms)
						    parms)))
					 (bound-terms-of-term term))
		       term))
		 term)))))))

(defunml (|fixup_extract| (term))
    (term -> term)

  (fixup-extract term))



(defun make-termof-expansion (extract-p)
  ;;(break)
  (if extract-p 
      (new-expansion 'parameter nil 'proof)
      (new-expansion 'error '(proof termof expand not))))


(defun named-termof-term (name termof)
  (instantiate-term
   (instantiate-operator name
			 (level-expressions-of-termof-term termof))))
		    
(defun import-termof (def goal extract name)
  (let ((termof (termof-of-term (oid-of-definition def) goal)))

    (setf (abstraction-lhs def) termof
	  (abstraction-rhs def) (when extract
				  ;;(setf -e extract) (break "hello")
				  (fixup-extract extract))
	  (abstraction-expansion def) (make-termof-expansion (and extract t))
	  (abstraction-other-models def) (when name (list (named-termof-term name termof)))
	  ))
    def)


#|
(defun import-proof (data)

  ;;(setf -data data) (break "ip")
  
  (let* ((def (make-proof :substance data))
	 (substance (ephemeral-substance-of-definition def 'proof-substance))
	 (goal (sequent-of-goal (goal-of-proof-substance substance)))
	 (extract (extract-of-proof-substance substance)))
    
    (setf -a substance -b def -e extract);;(break "ip")
    ;;(setf -data data -substance substance -def def) (break "ip")
  
    (let* ((name (name-property-of-substance substance))
	   (termof (termof-of-term (oid-of-dependency (current-dependency)) goal)
	    ;;(termof-of-term name goal)
	     ))

      ;; some controversy over which style of termof is in vogue, easy answer is allow lookup
      ;; to work for multiple styles. 
      (setf (proof-lhs def) termof
	    ;;(named-termof-term name termof)
	    )
	    
      (setf (proof-rhs def) (when (iextract-term-p extract) (fixup-extract (term-of-iextract-term extract)))
	    (proof-name def) name
	    (proof-expansion def) (make-termof-expansion (iextract-term-p extract))
	    ;; removed in preference to ref dag.
	    ;;(proof-references def) (let ((refs (dependency-store
		;;				(list (dependencies-of-substance substance)))))
		;;		     (dependencies-of-dependency-store refs))
	    (proof-other-models def) (when name (list (named-termof-term name termof))))

      (setf (proof-ref-dag def) (new-ref-dag (current-object-id) substance 0))
      )

    def))
|#

		 
;;;;	
;;;;	FTTB, we support a bastardized termof lookup.
;;;;	The migrated v4 objects have termofs whose opids are 
;;;;	v4 object name.
;;;;	
;;;;	if prf has name property and the termof otherwise matches
;;;;	and the abstraction at that name is a proof then it is returned as termof def.
;;;;	

#|
(defun lookup-proof-by-opid (term)
  (when (null (bound-terms-of-term term))
    (find-first #'(lambda (def)
		    (when (and (proof-p def)
			       (let ((termof (termof-of-proof def)))
				 (null (bound-terms-of-term termof))
				 (forall-p #'abstraction-match-parameter-p
					   (level-expressions-of-termof-term termof)
					   (parameters-of-term term))))
		      def))
		(name-table-lookup (resource 'abstractions)
				   (id-of-term term)
				   (current-transaction-stamp)))))

;; proof is a subclass of abstraction. thus to check for abstraction but not proof
;; must check for not proof, or subclass abstraction with abstraction'.


(defun proof-lookup (oid)
  (let ((def (definition-lookup-by-oid (resource 'abstractions) oid)))
    (if (typep def 'proof)
	def
	(raise-error (oid-error-message (list oid) '(proof lookup oid not))))))
|#

;; RLE   when we allow multiple termofs how to detect clash? Check all against all.
;; RLE   when we allow multiple termofs how to match? any.
;;;; to accomplish change:
;;;;	** merge-key-f(<definition{insert}>)		: <term{key}>
;;;;  ->
;;;;	** merge-key-f(<definition{insert}>)		: <term{key}> sexpr

;;;;	clash is not possible because each termof will contain the obid of the proof.
;;;;	but then what if abstraction contains obid 
;;;;	
;;;;	termof : to be compat with v4.2 need <lemma-name>{<le>:l} 
;;;;	  - admits clash
;;;;	
;;;;	<termof>	: TERMOF{<oid>:o, <le>:l} 
;;;;	  - milling may be worthwhile in this case.





;; fu{\v:l}()
(defun definition-of-termof (term &optional nil-ok-p)

  (let ((def (abstraction-super-of-term term nil t)))
    (if (typep def 'statement)
	def
	(unless nil-ok-p
	  (raise-error (error-message '(termof lookup term not) term))))))


;; extract (statement abstraction) visibility ?!
;;  - no  : this check if is a termof, if actual expansion occurs then will fail if not visible.
;;  - yes : assume this is true but not visible then may change behaviour of tactics even
;;	      without eventual failure since may not attempt expansion.
;;  YES.
(defun termof-p (term)
  (let ((def (definition-of-termof term t)))
    (and def
	 (statement-visible-p def (get-definition-table-visibility (resource 'statements)))
	 )))

(defun expand-termof (term)
  (let ((def (definition-of-termof term t)))

    (unless def
      (raise-error (error-message '(definition termof not) term)))

    (dependency-note-reference 'extract (dependency-of-definition def))

    (expand-term-aux term def (expansion-of-abstraction def))))


(defun map-proof-table (f)
  (format t "map abstraction table ")
  (without-dependencies 
   (definition-table-map (resource 'abstractions)
       (current-transaction-stamp)
     #'(lambda (oid def)
	 (when (typep def 'proof)
	   (funcall f oid def)))))

  (values))


;;;;	TODO
;;;;	expand should work on abs and termofs and compute should work on primitives.
;;;;	
;;;;	Suppose you have a term and if it is an abs or termof you want to expand, but
;;;;	if computable you do not want to compute.  You must check yourself before computing
;;;;	if it is primitive. Want primitives in lib. Ie computation is defined in an object.
;;;;	extend direct-computation-table to be a distributed table.



;;;; RLE NB *abstractions* must be meaningfully bound when proofs inserted, and vice versa.
;; intertable clash not a possibility as all termofs uniquely identified by obid? no since
;; one could define an abs that looks like a termof???

;; could put proofs and abstractions in same table! yes, done.

;;(defun proof-intertable-clash-tables () (list (environment-resource 'abstractions)))

#|
(defunml (|proof_lookup| (oid))
    (object_id -> (term |#| term))
  ;;(break "f")
  (let ((proof (proof-lookup oid)))
    (cons (termof-of-proof proof)
	  (extract-of-proof proof))))

(defunml (|extract_of_proof| (oid))
    (object_id -> term)
  (break "f") ; need proof_visibility ?!
  (let ((proof (proof-lookup oid)))
    (if proof
	(extract-of-proof proof)
	(breakout evaluation "extract_of_proof"))))

#|(defunml (|termof_of_proof| (oid))
    (object_id -> term)
  (break "f") ; need proof_visibility ?!
  (let ((proof (proof-lookup oid)))
    (if proof
	(termof-of-proof proof)
	(breakout evaluation "termof_of_proof"))))|#


#|(defunml (|proof_of_termof| (term));; remove this ew?
    (term -> object_id)
  (break "f") ; need proof_visibility ?!
  (let ((proof (proof-of-termof term t)));; put nil here, else loops sometimes?
    (if proof
	(oid-of-definition proof)
	(breakout evaluation "proof_of_term"))))|#


(defunml (|map_proofs| (f))
    ((object_id -> unit) -> unit)

  (definition-table-map  (resource 'proofs)
      (current-transaction-stamp)
     #'(lambda (oid def)
	 (declare (ignore oid))
	 (when (typep def 'proof)
	   (ap f (oid-of-definition def)))))
    nil)


|#

;;;;	expand_termof		: term -> term
;;;;	

(defunml (|expand_termof| (term) :error-wrap-p nil)
    (term -> term)

  (expand-termof term))

(defunml (|is_termof| (term) :error-wrap-p nil);;ew
    (term -> bool)
  (termof-p term))

(defun map-statement-table (f)
  (without-dependencies
   (definition-table-map (resource 'statements)
       (current-transaction-stamp)
     f))

  (values))

(defunml (|map_statements| (f))
    ((object_id -> unit) -> unit)

  ;;(format t "map_statements")
  (let ((tv (get-definition-table-visibility (resource 'statements))))
    (map-statement-table
     #'(lambda (oid def)
	 (declare (ignore oid))
	 (when (statement-visible-p def tv)
	   (ap f (oid-of-definition def)))))))



(defunml (|map_statement_names| (pattern f))
    (string -> ((tok -> (object_id -> unit)) -> unit))

  ;; should be weeding calls to this out. not necessarily some use ok.
  ;;(break "map_statement_names ")
  (format t "map_statement_names ")

    (without-dependencies
     (let ((statements (resource 'statements)))
       (let ((tv (get-definition-table-visibility statements)))
	 (name-table-search (resource 'statements)
			    (current-transaction-stamp)
			    pattern
			    #'(lambda (name oid)
				(let ((def (definition-lookup-by-oid statements oid t (current-transaction-stamp) t)))
				  (when (and def
					     (statement-visible-p def tv))
				    (ap f name oid))))))))
  nil)



;; let find_statement_names pattern = letref m = nil in map_statment_names pattern (\name oid. m := (name,oid) . m) in m
;; functions for rolling up proofs :

;;;;	top-proof-to-iinf-top : rule -> term
;;;;	  * calls top-proof-to-iinf-tree on proof-top of rule
;;;	  * calls top-proof-to-iinf-top on rule
;;;;	  
;;;;	proof -> term
;;;;	  - top-proof-to-iinf-tree 
;;;;	  - interior-proof-to-iinf-tree 
;;;;	      * calls top-proof-to-iinf-top on rule
;;;;	      * calls interior on children
;;;;	  - exterior-proof-to-iinf-tree
;;;;	      * calls top-proof-to-iinf-top on rule
;;;;	      * calls exterior on children
;;;;	
;;;;	refinement-to-inf-tree
;;;	  * calls top on proof of rule
;;;;	  * calls exterior on children
;;;;	
;;;;	  - top is most exterior.
;;;;	
;;;;	
;;;;	

