%
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2003                                *
;;;                                                                       *
;;;                                                                       *
;;;                Formal Digital Library System                          *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the FDL group, Department of Computer Science,         *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify FDL provided this notice    *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;                                                                       *
;;;************************************************************************
%

%[
***************************************************************************
    General purpose ml functions.
***************************************************************************

***************************************************************************
Type definitions.
***************************************************************************
]%

lettype void = . ;;
lettype unit = . ;;

%[
***************************************************************************
Obselete
***************************************************************************
]%

% keep these round for compatibility, just as long as is necessary %

let itlist f l x = rev_itlist f (rev l) x;;

% [x1; ...; xn]   --->   (fun x1 ... (fun (xn-1) xn)...)   for n>0 %
let end_itlist fun l =
    if null l then failwith `end_itlist`
    else (let last.rest = rev l in
          rev_itlist fun rest last)
;;


let eqfst x (y,z) = (x=y)
and eqsnd x (y,z) = (x=z);;

let assoc x = find (eqfst x) ;;
let rev_assoc x = find (eqsnd x) ;;
  
letrec find_position atom list =
  if atom = hd list then 1
  else find_position atom (tl list) + 1
;;

letrec find_position_p atom list equal =
  if equal atom (hd list) then 1
  else find_position_p atom (tl list) equal + 1
;;

%[
***************************************************************************
Format translation.
***************************************************************************
]%

let pair x y = (x,y);;

let list_to_pair [x;y] = x,y
and list_to_triple [x;y;z] = x,y,z
and list_to_quad  [v;x;y;z] = v,x,y,z
;;

let pair_to_list (x,y) = [x;y]
and triple_to_list (x,y,z) = [x;y;z] 
and quad_to_list (w,x,y,z) = [w;x;y;z]
;;

let curry f x y = f (x,y) ;;
let three_curry f x y z = f (x,y,z) ;;
let four_curry f x y z a = f (x,y,z,a) ;;
let uncurry f (x,y) = f x y ;;
let three_uncurry f (x,y,z) = f x y z ;;
let four_uncurry f (x,y,z,a) = f x y z a ;;

%[
***************************************************************************
Basic
***************************************************************************
]%

let id x = x ;;
let ap x y = x y ;;

let isr x = not (isl x);;

%[
***************************************************************************
Combinators
***************************************************************************
]%

let I x = x;;
let K x y = x;;
let KI = K I;;  % Dual of K; K and KI are analogs of fst and snd%
let C f x y = f y x         %  the permutator  %
and W f x   = f x x         %  the duplicator  %
and B f g x = f (g x)       %  the compositor  %  %curried form of "o"%
and S f g x = f x (g x);;
%
 S, K, I permit the definition of lambda-abstraction
 \x.x = I      actually unnecessary, since I = S K K)    
 \x.A = K A    where A is a constant or a variable other than x
 \x.(A B) = S (\x.A) (\x.B)                               
%

%[
***************************************************************************
Function composition
***************************************************************************
]%

ml_paired_infix `o`;;
ml_paired_infix `#`;;
ml_paired_infix `oo`;;

let $o (f,g)x = f(g x) ;;
let $# (f,g) (x,y) = (f x, g y);;
%composition for a function that takes a paired argument%
let $oo (f,(g,h)) x = f(g x, h x);;


ml_paired_infix `Co`;;
let $Co (f,g) x y = f (g y) x;;    %     permutation-composition          %
                                   % Ainsi nomme car  $Co (f,g) = C (f o g) %


%sequencing operators for functions%

ml_curried_infix `thenf` ;;
ml_curried_infix `orelsef` ;;

%apply f and then g%
let $thenf f g x = g(f x);;

%apply f or else g%
let $orelsef f g x = f x ? g x;;

let tryf f x = f x ? x ;;

let progressf f x = 
  let x' = f x in if (alpha_equal_terms x' x) then failwith `progressf` else x' ;;

let progresseqf f x = 
  let x' = f x in if (eq_terms x' x) then failwith `progresseqf` else x' ;;

let all_fun x = x;;
let no_fun x = failwith `no_fun`;;

% fail with failure message of last function, (if length fl > 0) %

letrec first_fun fl x =
  if null fl then failwith `first_fun`
  if null (tl fl) then
    (hd fl) x
  else
    ((hd fl) x 
     ? first_fun (tl fl) x)
;;


let apply_to_one f l =
  letrec aux l =
    if null l then fail
    else (f (hd l) ? aux (tl l))  in
  aux l
;;

let first_value f arg_list =
  letrec aux l = 
    if null l then failwith `first_value`
    if null (tl l) then f (hd l)
    else (f (hd l) ? aux (tl l)) in
  aux arg_list
;;

letrec every_fun fl x =
  if null fl then x
  else
     (hd fl) (every_fun (tl fl) x) ;;

letrec repeatf f x = ((progressf f) thenf (repeatf f)) x ? x;;


%[
***************************************************************************
Boolean 
***************************************************************************
]%

ml_curried_infix `implies`;;
let $implies a b = not a or b ;;

ml_curried_infix `xor`;;
let $xor a b = (not a & b) or (a & not b) ;;

ml_curried_infix `iff`;;
let $iff a b = (a & b) or (not a & not b) ;;


% defined as builtin.
let all (P: *->bool) (l: * list) =
  letrec aux l = if null l then true else P (hd l) & aux (tl l) in
  aux l
;;

let exists (P: *->bool) (l: * list) =
  letrec aux l = if null l then false else P (hd l) or aux (tl l) in
  aux l
;; 
%
let all = forall ;;
let every l P = all P l ;;
let some l P = exists P l ;;

let satisfies_one P_list x =
  letrec f l = if null l then [] else if hd l x then fail else f (tl l)  in
  (f P_list; false) ? true
;;

letrec exists_pair P l =
  if null l then false
  else exists (P (hd l)) (tl l)
       or exists_pair P (tl l)
;;

let can f x = (f x ; true) ? false ;;


let both (P: *->bool) (x:*) (y:*) = P x & P y ;;


%[
***************************************************************************
List: Folding and unfolding
***************************************************************************
]%
% A folding function reduces a list to a single value. 

  list = [x1;...;xn]
  accumulator = a
%

% ai = f ai-1 xi %

letrec accumulate f a l =
  if null l then a
  else accumulate f (f a (hd l)) (tl l)
;;

% ai = f xi ai+1 %

letrec reduce f a l =
  if null l then a 
  else f (hd l) (reduce f a (tl l))
;;

% :(*->*->*)->* list->*.  [x1;...;xn] ---> f(...(f(f x1 x2)(x3))....)xn 
  for n>1, [x] ---> x, [] ---> failure. 
%
let accumulate1 f args =
  letrec aux accumulation args =
    (let (x.l) = args in aux (f accumulation x) l)
    ?
    accumulation  in
  if null args then failwith `accumulate1`
  if length args = 1 then hd args
  else aux (hd args) (tl args)
;;



% :(*->*->*)->* list->*.  [x1;...;xn] ---> f x1 (f x2 (... (f x(n-1) xn)...)
  for n>1, [x] ---> x, [] ---> failure.
%
let reduce1 f args =

  letrec aux args =
        (let [x;y] = args in f x y)
        ?
        (let (x.l) = args in f x (aux l))  in
  if null args then failwith `reduce1`
  if length args = 1 then hd args
  else aux args
;;

% similar to reduce1, but only reduces sections of input for which
  the reducing function doesn't fail
%

let multi_reduce1 f as =

  letrec aux as =
    if null as then []
    else let as' = aux (tl as) in

    if null as' then [hd as]
    else
      f (hd as) (hd as') . (tl as')
      ?
      (hd as) . as'
  in
    aux as
;;


letrec unreduce destructor obj =
  (let x,y = destructor obj in
   let l,z = unreduce destructor y in 
   (x.l),z)
  ? [], obj
;;

letrec unreduce1 destructor obj =
  (let x,y = destructor obj in x . unreduce1 destructor y)
  ? [obj]
;;

let accumulate_and_combine f init_value l =
  letrec aux accumulation revd_combination_so_far l_tail =
    if null l_tail then revd_combination_so_far
    else aux (f accumulation (hd l_tail))
             ((hd l_tail, accumulation) . revd_combination_so_far)
             (tl l_tail)      in
  rev (aux init_value [] l)
;;

%[
***************************************************************************
Arithmetic
***************************************************************************
]%

ml_curried_infix `LE` ;;
ml_curried_infix `GE` ;;
let $LE i j = not j < i ;;
let $GE i j = not i < j ;;


let max x y =
  if x>y then x else y;;

let min x y = if x<y then x else y ;;

let sum = accumulate (\x y. x+y) 0 ;;

let max_value f = accumulate (\n x. max n (f x)) 0 ;;

let list_max = max_value id ;;

let abs x = max x (-x) ;;

let sign x = x < 0 => (-1)
           |x = 0 => 0
           | 1
;;

let gcd i j = 
  let i' = abs i and j' = abs j in
  let a = max i' j' and b = min i' j' in
  letrec gcd' m n =
    let q = m/n in
    let r = m-q*n in
    if r = 0 then n
    else gcd' n r
  in
  if b = 0 then 
    a
  else
    gcd' a b
;;

let lcm i j = (i*j)/gcd i j ;;

% gives floor of real value of log base 2 of i. log2 0 gives 0. %

letrec log2 i = i < 2 => 0 | 1 + log2 (i/2) ;;

%[
***************************************************************************
List: Basic
***************************************************************************
]%
% Content independent operations. %

%
From lisp: 

rev
length

%

letrec nth n l = if null l then failwith `nth`
                 else if n=1 then hd l
                 else nth (n-1) (tl l)
;;


%
letrec select position list =
  if position = 1 then hd list
  else select (position-1) (tl list);;
%
let select = nth ;;

let append x y = x @ y;;
%let flatten = reduce $@ [] ;;%
let flatten ll = 
 letrec auxx l r = 
   if null l then r
   else (hd l) . auxx (tl l) r in

 letrec aux ll = 
  let l.ll' = ll in
   if null ll' then l else
   if (null l) then aux ll' 
   else auxx l (aux ll') in
 if null ll then nil else aux ll
;;

let first l = select 1 l ;;
let second l = select 2 l ;;
let third l = select 3 l ;;
let fourth l = select 4 l ;;
let fifth l = select 5 l ;;


% split n [a1,...,an,...am] = ([a1,...,an], [a(n+1),...,am] %

let split n l =
    letrec s l1 i l2 = 
           if i=0 then (rev l1, l2)
           else if not null l2 then (let h.t = l2 in s (h.l1) (i-1) t)
           else failwith `split`
    in s [] (max n 0) l 
;;

let firstn n = fst o split n ;;

let nthtl n = snd o split n ;;

let split_lastn n l =
  let len = length l in
  if len < n then failwith `split_lastn` else
    split (len - n) l
;;

let lastn n = snd o split_lastn n ;;
let last = hd o lastn 1 ;;

letrec remove_nth n l =  
  if n = 1 then tl l else hd l. remove_nth (n-1) (tl l)
;;


let remove_lastn n = fst o split_lastn n ;;
let remove_last = remove_lastn 1 ;;


% A list u of lists which concatenate to l and such that (map length u) =
  block_sizes. %
let partition block_sizes l =
   letrec p blks ns l =
     if null ns & null l then rev blks
     else if not null ns then
             (let x,y = split (hd ns) l in p (x.blks) (tl ns) y)
          else failwith `partition`  in
    p [] block_sizes l 
;;

let rotate_left (a.l) = l @ [a]
and rotate_right l =
    let ra.rl = rev l in ra . (rev rl)
;;

let replace_nth_by n list new_item =
  letrec f n lis =
    let h.t = lis in
    if n = 1 then new_item.t
    else h . f (n-1) t
  in 
    f n list
;;

let update_nth g list n =
  letrec f n lis =
    let h.t = lis in
    if n = 1 then g h.t
    else h . f (n-1) t
  in 
    f n list
;;


% tails [1;2;3;4] --> [[2;3;4];[3;4];[4];[]] %

letrec tails as = 
  if null as then []
  else (tl as) . tails (tl as)
;;


%[
***************************************************************************
List: Searching
***************************************************************************
]%

let member element list =
  letrec aux x l =
    let h.t = l in
       if x=h then true
       else aux x t
  in
  (aux element list) ? false;;

let member_p element list equal =
  letrec aux x l =
    let h.t = l in
       if equal x h then true
       else aux x t
  in
  (aux element list) ? false;;

let memberp eqp element list = member_p element list eqp;;

% 1-based %
% find 1st position in list at which P is true %

let search_list P = 
  letrec f i l = 
         if null l then failwith `search_list`
         if P (hd l) then i
         else f (i+1) (tl l) in
  f 1
;;


% find last position in list at which P is true %

let rev_search_list P =
  letrec f i l = if null l then fail
                 else (f (i+1) (tl l) ? if P (hd l) then i else fail)  in
  (f 1) ? failwith `rev_search_list` 
;;



% The index (starting at 1) of the first occurrence of x in l. %
let position x l = 
  search_list ($= x) l
;;

let split_p p l =
 split (search_list p l) l
;; 

let find_preceding p l = last (fst (split ((search_list p l) - 1) l));;

% The index (starting at 1) of the first occurrence of x in term list, l. %
let term_position x l = 
  search_list (\t. alpha_equal_terms t x) l
;;


% find 1st element which satisfies P %

%%% following is already a lisp primitive thus no need to have here and it keeps
    confusing the compiler to have the name clash.
%
%
let find P l =
  letrec f l' = 
    if null l' then failwith `find`
    else
    let h.t = l' in
    if P h then h else f t
  in
  f l
;;
%

let find2 (R: *->**->bool) (l: * list) (l': ** list) : *#** =
  letrec aux l =
    hd l, find (R (hd l)) l'  ?   aux (tl l)  in
  aux l
;;


%[
***************************************************************************
List: maps. 
***************************************************************************
]%
% all mapping functions output a list of same length as input list. %


letrec map2 f as bs = 
 (if null as & null bs then [] 
  if not null as & not null bs then
    f (hd as) (hd bs) . map2 f (tl as) (tl bs) 
  else
    failwith `map2`
 )
;;

let map2_p p as bs =
 letrec aux as bs =
 (if null as & null bs then true 
  if not null as & not null bs then
    (p (hd as) (hd bs) & aux (tl as) (tl bs)) 
  else
    failwith `map2_p`
 )
  in aux as bs
;;

letrec map3 f as bs cs = 
 if null as & null bs & null cs then [] 
 else
    f (hd as ? failwith `map3`) 
      (hd bs ? failwith `map3`)
      (hd cs ? failwith `map3`) 
    . map3 f (tl as) (tl bs) (tl cs)
;;

%
      x1    x2    x3  ...  xn

  c0  f  c1 f  c2 f ...    f   cn

      y1    y2    y3       yn


  yi,ci = f xi c(i-1)

%

let map_with_carry f cin l =
  letrec g l ci_mns_1 =
    if null l then [],ci_mns_1 
    else
    let yi,ci = f (hd l) ci_mns_1 in
    let ys,cn = g (tl l) ci in
      yi.ys , cn
  in
    g l cin
;;


%
apply f to each term of l left to right. If f fails draw replacement from
repairs.
%

let map_with_repair f repairs l =
  let l',cout = 
     map_with_carry 
      (\x cin. f x, cin ? hd cin, tl cin 
               ? failwith `map_with_repair: too few repairs`)
      repairs
      l
  in
    if null cout then l' else failwith `map_with_repair: too many repairs`
;;

% 
Try to map f on each element of the list, leaving original element if f fails.
Fail if all applications of f fail.

as': * list = map_on_some  (f : * -> *) (as : * list)
%

let map_on_some f  l =
  letrec do_it l =
   if null l then
     [],false 
   else
     let l',b = do_it (tl l) and a = hd l  in
      ((f a).l', true) ? (a.l', b)
  in
  let res, b = do_it l  in
    b => res | failwith `map_on_some: no successful applications`
;;


% For mapping a function on elements of multidimensional lists %

let map_2d f =
  map (map f) 
;;

let map_3d f =
  map (map (map f))
;;

%[
***************************************************************************
List: Filters.
***************************************************************************
]%

let divide_list p l =
    reduce (\x (yes,no). if p x then (x.yes),no else yes, (x.no))
     ([],[]) l;;

%let filter p l = fst (divide_list p l) ;;%
% defined by lisp 
let filter p l = 
 letrec aux l = 
  if null l then []
  else let a.l' = l in
        if p a then a . aux l'
        else aux l'
  in
  aux l
;;
%%let remove_if p l = snd (divide_list p l) ;;%
let remove_if p l = 
 letrec aux l = 
  if null l then []
  else let a.l' = l in
        if p a then aux l'
        else a . (aux l')
  in
  aux l
;;
let remove_one_if p l = 
 letrec aux l = 
  if null l then []
  else let a.l' = l in
        if p a then l'
        else a . (aux l')
  in
  aux l
;;


% Like map, except for the omission of members where f fails. %

letrec map_omitting_failures f l =
  if null l then [] else f (hd l) . map_omitting_failures f (tl l) 
  ? map_omitting_failures f (tl l)
;;

let longest_prefix P l = firstn (search_list ($not o P) l - 1) l ? l 
and longest_suffix P l = nthtl (rev_search_list ($not o P) l) l ? l
;;


% [x1; ...; x(m+n)]   --->  [y1; ...; ym], [z1; ...; zn]
where the y's are all x's that satisfy p, the z's all other x's. Preserve
order of lists.
%


%[
***************************************************************************
List: As sets
***************************************************************************
]%


let empty = [] ;;

let singleton x = [x] ;;

let mk_set l =
    accumulate (\s x. if member x s then s else x.s) [] l ;;

let mk_set_p l equal =
    accumulate (\s x. if member_p x s equal then s else x.s) [] l ;;

let insert x l = if member x l then l else x.l ;;

let insert_p x l equal = if member_p x l equal then l else x.l ;;

let intersection l l' =
  filter (\x. member x l) l'
;;

let intersection_p p l l' =
  filter (\x. member_p x l p) l'
;;

let diff l l' = filter ($not o C member l') l ;;
let diff_p p l l' = filter ($not o (\e. member_p e l' p)) l ;;

let union l1 l2 = l1 @ (diff l2 l1) ;;
let union_p p l1 l2 = l1 @ (diff_p p l2 l1) ;;

let collect f = accumulate (\ acc x. union (f x) acc) empty  ;;

letrec remove x l = 
  if null l then []
  else let y.l = l in if x=y then l else y . remove x l
;;

let subset l l' = all (C member l') l ;;

% alpha equality check for term lists%
let subset_p l l' equal = all (C (\x y. member_p x y equal) l') l ;;


%Check if the elements of `l` are all distinct%
letrec all_distinct l = 
    (null l) or
    (not (member (hd l) (tl l)) & all_distinct (tl l))
;;

letrec remove_prior_duplicates l =
  if null l then []
  if member (hd l) (tl l) then remove_prior_duplicates (tl l)
  else (hd l) . (remove_prior_duplicates (tl l))   
;;

let remove_prior_duplicates_p eqp l =
 letrec aux l =
    if null l then []
    if memberp eqp (hd l) (tl l) then aux (tl l)
    else (hd l) . (aux (tl l))
  in aux l
;;

let remove_later_duplicates l =
  rev (remove_prior_duplicates (rev l))
;;

let remove_later_duplicates_p eq l =
  rev (remove_prior_duplicates_p eq (rev l))
;;

%[
***************************************************************************
List: Creation
***************************************************************************
]%

%mk the list [x; x; ...; x] of length n%
let replicate x n =
    if n<0 then failwith `replicate`
    else
      letrec aux x n r =
         if n=0 then r
         else aux x (n-1) (x.r) in
      aux x n [];;

% generation ends when iterator fails. %
let generate_sequence first_member iterator =
  letrec aux members_so_far =
    aux (iterator (hd members_so_far) . members_so_far) 
    ? members_so_far  in
  rev (aux [first_member])
;;

%mk the list [from; from+1; ...; to]%
let upto from to =
    letrec aux from to result =
       if to < from then result
       else aux from (to - 1) (to . result) in
    aux from to [];;
      
% (int#*) list -> * list.  Construct a list of * from l using the integers
  as repetition factors.
%
letrec build_list l =
        letrec ncons n x l = if n<1 then l else x.(ncons (n-1) x l)  in
        (let (n,x).tl = l  in  ncons n x (build_list tl))
        ?
        []
;;

%[
***************************************************************************
List: Zipping and unzipping
***************************************************************************
]%

let zip  = map2 pair ;;

let zip3 as bs cs = 
  let b_c_pairs = zip bs cs in
    zip as b_c_pairs
;;

ml_curried_infix `com`;;
let $com = zip ;;


letrec unzip pr_list = 
  if null pr_list then [],[]
  else
    let (a,b).rest = pr_list in
    let as,bs = unzip rest in
      a.as, b.bs
;;

letrec unzip3 trip_list =
  let as,bs_cs_pair = unzip trip_list in
    as,unzip bs_cs_pair
;;

% [[1;2;3]; [4;5;6]; [7;8;9]]   --->   [1; 5; 9]        %
letrec diagonal ll =
    if null ll then []
    else hd (hd ll) . (diagonal (map tl (tl ll)));;


let zip_tails as = zip as (tails as) ;;

%[
***************************************************************************
List: Interleaving
***************************************************************************
Take lists as and bs and interleave, starting with first element of a.
]%

letrec interleave as bs = 
  if null as then bs else let a.as' = as 
  in
  a . interleave bs as'
;;


    

%[
***************************************************************************
List: Numbering
***************************************************************************
]%

let number l = zip (upto 1 (length l)) l
;;


%[
***************************************************************************
List: sorting
***************************************************************************
]%

let quicksort le_test l =
  letrec sort l' =
    if null l' or null (tl l') then l'
    else
    let h.t = l' in
    let less_eq_set,greater_set =
      divide_list (\i.le_test i h) t 
    in
      (sort less_eq_set) @ [h] @ (sort greater_set)
  in
    sort l
;;

% returns lis1,lis2 such that |lis1| = | lis2| or |lis1| = |lis2| + 1 %

let divide_in_half l =
  letrec aux input out1 out2 =
    if null input then
     out1,out2
    else
      aux (tl input) (hd input.out2) out1
  in
    aux l [] []
;;

letrec merge le_test as bs =
  if null as then bs
  if null bs then as
  else
  if le_test (hd as) (hd bs) then
    (hd as) . merge le_test (tl as) bs
  else
    (hd bs) . merge le_test as (tl bs)
;;

letrec mergesort le_test l =
  if null l or null (tl l) then 
    l
  else
    let a,b = divide_in_half l in
       merge le_test (mergesort le_test a) (mergesort le_test b)
;;

% 
Insertion sort. 

Gives topological sort when le is transitive relation. 

(A list xs is considered topologically sorted here just when 
lt(xs[i], xs[j]) implies that i < j, lt being the strict part of le.)
%

let topolsort le as = 
  letrec insert x bs = 
    if null bs then [x] else let b.bs' = bs in
    if le x b then x.bs
    else b . insert x bs'
  in
  letrec sort bs = 
    if null bs then [] else let b.bs' = bs in
    insert b (sort bs')
  in
    sort as
;;


%[
***************************************************************************
List: Fancy
***************************************************************************
]%

%
Combinatorial functions for forward chaining tactic.

all_permutations [[1;2;3];[4];[1;2]] --->
[[1; 4; 1]; [1; 4; 2]; [2; 4; 1]; [2; 4; 2]; [3; 4; 1]; [3; 4; 2]]
  
all_distinct_permutations  [[1;2;3];[4];[1;2]] --->
[[1; 4; 2]; [2; 4; 1]; [3; 4; 1]; [3; 4; 2]] 
%

letrec all_permutations input =
  if null input then []
  if null (tl input) then
    map (\x.[x]) (hd input)
  else
    let partial_output = all_permutations (tl input) in
    flatten
    ( map
      (\y. map ($. y) partial_output)
      (hd input)
    )
;;
letrec all_distinct_permutations input =
  if null input then []
  if null (tl input) then
    map (\x.[x]) (hd input)
  else
    let partial_output = all_distinct_permutations (tl input) in
    flatten
    ( map
      (\y. map_omitting_failures
             (\zs. if member y zs then fail else y . zs)
             partial_output)
      (hd input)
    )
;;




%[
***************************************************************************
List: Comparison
***************************************************************************
]%

letrec is_prefix l1 l2 =
  if null l1 then true 
  if null l2 then false
  if hd l1 = hd l2 then is_prefix (tl l1) (tl l2)
  else false
;;

let is_suffix l1 l2 =
  is_prefix (rev l1) (rev l2)
;;

letrec remove_prefix prefix l =
 (if null prefix then l 
  if hd prefix = hd l then remove_prefix (tl prefix) (tl l)
  else fail
 ) ? failwith `remove_prefix`
;;

let remove_suffix suffix l =
  rev (remove_prefix (rev suffix) (rev l))
;;


% get index of start of first occurrence of subl in l %

let split_at_infix infix l = 
  letrec aux revleft right = 
  ( revleft,remove_prefix infix right
    ?
    aux (hd right .revleft) (tl right)
    ?
    failwith `split_at_infix`
  )
  in
  let revleft,right = aux [] l
  in (rev revleft) , right
;;

let is_sublist as bs = 
  (split_at_infix as bs ; true) ? false
;;

%[
***************************************************************************
Association list 
***************************************************************************
]%


let add_to_alist_start i v l = 
  (i,v).l
;;
 
let add_to_alist_end i v l =
  l @ [i,v]
;;

letrec apply_alist l x = 
  if null l then failwith `apply_alist`
  else let (y,v).t = l in if x=y then v else apply_alist t x
;;
  
letrec apply_alist_p l x equal = 
  if null l then failwith `apply_alist_p`
  else let (y,v).t = l in if equal x y then v else apply_alist_p t x equal
;;

letrec rev_apply_alist l x = 
  if null l then failwith `rev_apply_alist`
  else let (v,y).t = l in if x=y then v else rev_apply_alist t x
;;

letrec rev_apply_alist_p l x equal = 
  if null l then failwith `rev_apply_alist_p`
  else let (v,y).t = l in if equal x y then v else rev_apply_alist_p t x equal
;;

% fails if not found OR if found but value is nil%
letrec apply_alist_v l x = 
  if null l then failwith `apply_alist_v`
  else let (y,v).t = l in
  if x=y then
   (if (null v ? false) then failwith `apply_alist_v`
    else v)
  else apply_alist_v t x
;;

let bound x alist = exists (\y,(). y=x) alist ;;

let is_bound = bound ;;

% Remove duplicate bindings, fail if there is an inconsistent pair. %
let condense_alist =
  accumulate (\l (x,v). (if bound x l then
                           (if v = apply_alist l x then l
                            else failwith `condense_alist` )
                         else (x,v).l))
             []
;;

let condense_alist_p eqfst eqsnd =
  accumulate (\l (x,v). (if bound x l then
                           (if eqsnd v (apply_alist_p l x eqfst) then l
                            else failwith `condense_alist` )
                         else (x,v).l))
             []
;;

% if not present places at end of list. %
let update_alist l i v =
  letrec f l = 
    if null l then [i, v]
    else let a.l' = l in
         if (fst a)=i then (i,v).l' 
         else a . (f l') in
  f l
;;

% if not present places at end of list. %
let update_alist_p l i v equal =
  letrec f l = 
    if null l then [i, v]
    else let a.l' = l in 
         if equal (fst a) i then (i,v).l' 
         else a . (f l') in
  f l
;;

% if not present places at head of list. %
let update_insert_alist l i v =
  letrec f l = 
    if null l then fail
    else let a.l' = l in 
         if (fst a)=i then (i,v).l' 
         else a . f l' in
  f l ? (i, v). l
;;

% if not present places at head of list. %
let update_insert_alist_p eqp l i v =
  letrec f l = 
    if null l then fail
    else let a.l' = l in 
         if (eqp (fst a) i) then (i,v).l' 
         else a . f l' in
  f l ? (i, v). l
;;

% do updates in left to right order %

letrec multi_update_alist l i_v_prs  =
  if null i_v_prs then l
  else
    let (i,v).rest = i_v_prs in
      (multi_update_alist (update_alist l i v) rest)
;;

% presumes list is longer than update. NOT STABLE wrt list. %
let fast_update_alist p updates l  =
  updates 
   @ (filter (\e. not (member_p (fst e) updates (\li u. p li (fst u)))) l)
;;

% if (i,x) is in l, replace it with (i, g x). If no entry with index i, fail %

let modify_alist_entry g l i =
  letrec f l = 
    if null l then failwith `modify_alist_entry: entry not found`
    else let e.l' = l in 
         if (fst e)=i then (i, g (snd e)) . l' 
         else e. f l' in
  f l
;;

let modify_or_add_alist_entry g v l i =
  modify_alist_entry g l i 
  ? add_to_alist_start i v l 
;;


%
Doesn't fail if no entry found. Assumes only one entry
%

let remove_alist_entry l i =
  letrec f l = 
    if null l then []
    else let e.l' = l in 
         if (fst e)=i then l' 
         else e . f l' in
  f l
;;

let remove_alist_entry_p eqp l i =
  letrec f l = 
    if null l then []
    else let e.l' = l in 
         if (eqp (fst e) i) then l' 
         else e . f l' in
  f l
;;

let split_off_alist_entry l i = 
  apply_alist l i, remove_alist_entry l i
;;


let totally_remove_alist_entry l i =
  letrec f l = 
    if null l then []
    else let (i',v').l' = l in 
         if i'=i then f l' 
         else (i',v'). f l' in
  f l
;;

% removes occurrences of all entries in l with indices is %

let remove_alist_entries l is =
  letrec f l = 
    if null l then []
    else let (i',v').l' = l in 
         if member i' is then f l' 
         else (i',v'). f l' in
  f l
;;

let move_alist_entry_to_end l i =
 let v = apply_alist l i
 in add_to_alist_end i v (remove_alist_entry l i)
;;

% 
Assumes l1 and l2 don't have duplicated entries, although the same
binding can occur in both lists. In the latter case, remove one of
the bindings from the resulting list. 
%

letrec merge_alists l1 l2 =
  if null l1 then l2 else
  let h1.t1 = l1 in
  let is_dup,dups_match =
  (( true , apply_alist l2 (fst h1) = (snd h1) 
   ) ?
     false, false
  )
  in
  if (is_dup & dups_match) then
    merge_alists t1 l2
  if not is_dup then
    h1 . merge_alists t1 l2
  else
    failwith `merge_alists: lists incompatible`
;;
    
letrec merge_alists_p l1 l2 equal =
  if null l1 then l2 else
  let h1.t1 = l1 in
  let is_dup,dups_match =
  (( true , equal (apply_alist l2 (fst h1)) (snd h1) 
   ) ?
     false, false
  )
  in
  if (is_dup & dups_match) then
    merge_alists_p t1 l2 equal
  if not is_dup then
    h1 . merge_alists_p t1 l2 equal
  else
    failwith `merge_alists_p: lists incompatible`
;;

% merges l1 and l2. Discards bindings in l2 for which there exist bindings
  in l1.
%
% differs from (multi_update_alist l2 l1) in that the order of elements in
  the result differs.
  in either case l1 is the prefered list.
  in this case the non-conflicting elements of l2 are appended to l1.
  in the update case then non-conflicting elements are appended l1 are
   appended to l2 and the conflicting element replace those of l2.
%

   
let priority_merge_alists l1 l2 =
  l1
  @ 
  accumulate
    (\a (i,v). remove_alist_entry a i)
    l2
    l1
;;

% 
Returns an alist where all entries with the same index are grouped into
a single entry.
Groups ordered by first occurrence of index, and order of entries within
each group is preserved.
%

let group_alist_entries alist =
  letrec aux as bs =
    if null as then bs
    else let (i,a).as' = as 
    in let bs' = modify_or_add_alist_entry (\x. a.x) [a] bs i
    in aux as' bs'
  in
    map (id # rev) (rev (aux alist []))
;;

%
Takes alist and creates new alist with new_keys as list of keys.

nth occurrence of key k in new_keys gets value of nth occurrence
of k in alist. If no nth occurence in alist, then uses default_val
instead.
%

let reorder_alist alist new_keys default_val =
  letrec aux alist ks = 
    if null ks then [] else
    let k.ks' = ks in
    let v,alist' = split_off_alist_entry alist k ? default_val,alist in
      (k,v) . aux alist' ks'
  in
    aux alist new_keys
;;


%[
***************************************************************************
2D Association list 
***************************************************************************
]%

let update_2d_alist l i j v =
  letrec f l = 
   if null l then [i, [j,v]] 
   else let (i',v').l' = l in 
        if i'=i then (i, update_alist v' j v) . l' 
        else (i',v') . f l'  in
  f l
;;

% PERF: could be done more efficiently %
let multi_update_2d_alist l updates = 
 letrec aux l updates = 
  if null updates then l
  else let (i,j,v). rest = updates in
         aux (update_2d_alist l i j v) rest
 in aux l updates
;;

let remove_2d_alist_entry l i j =
  letrec f l = 
   if null l then []
   else let (i',v).l' = l in 
        if i'=i then (i, remove_alist_entry v j) . l' 
        else (i',v) . f l'  in
  f l
;;

let apply_2d_alist l i j = apply_alist (apply_alist l i) j ;;

%[
***************************************************************************
Multisets
***************************************************************************
* Multiset  is (* # int) list
]%

let mset_addn n x mset = 
  modify_or_add_alist_entry (\x.x+n) n mset x ;;
  
let mset_add = mset_addn 1 ;;

let mset_inj x = [x,1] ;;

let mset_union ms1 ms2 = 
  reduce (\(x,n) acc.mset_addn n x acc) ms1  ms2
;;

% rank high to low %
let mset_sort mset = mergesort (\((),m) ((),n).m > n) mset 
;;


%[
***************************************************************************
Token manipulation
***************************************************************************
]%

let whitespace_toks = 
  map int_to_char 
  [32  % space %
  ;13  % carraige-return %
  ;10  % line-feed %
  ]
;;

%Use the character `sep` to split the token into non-empty words
words2 `/` `abc//d/ef/`  -->  [`abc`; `d`; `ef`]
%
let words2 sep string =
    snd (itlist (\ch (chs,tokl). 
             if ch = sep then
                if null chs then [],tokl
                else [], (implode chs . tokl)
             else (ch.chs), tokl)
    (sep . explode string)
    ([],[]));;

%words `are you there`  -->  [`are`; `you`; `there`]    %
let words = words2 ` `;;

let ids = words ;; 

%maptok f `are you there` = [f `are`; f `you`; f `there`]       %
let maptok f tok = map f (words tok);;

%Token concatenation%
%for speed, these should be implemented in lisp%

let concat tok1 tok2 = implode( explode tok1 @ explode tok2) ;;
let concatl tokl =
    implode (itlist append (map explode tokl) nil);;

ml_curried_infix `^`;;
let $^ = concat;;

let empty_token = implode [] ;;

%  Following are some functions for a version of new_id, 
   called undeclared_id.
%
let number_suffixing_letter tok =
   int_of_tok (implode (tl (explode tok)))  ?  -1
;;

let number_suffixing_given_letter letter tok =
( let letter' . rest = explode tok in
  if letter' = letter then int_of_tok (implode rest) else fail
) ? failwith `number_suffixing_given_letter`
;;

let n_ids n = map (\i. `x`^(tok_of_int i)) (upto 1 n)
;;

let longest_number_suffix tok =
  implode (longest_suffix (C member ``0 1 2 3 4 5 6 7 8 9``)
                          (explode tok))
;;

let append_underscore tok =
  implode (explode tok @ [`_`]) 
;;

let remove_underscore tok =
  let l = explode tok in
  if last l = `_` then implode (remove_last l) 
  else failwith `remove_underscore`
;;

let int_to_tok = tok_of_int ;;
let tok_to_int = int_of_tok ;;

let message tok = print_string tok; print_newline();;

let extend_name name extension =
  name ^ `_` ^ extension ;;


let lower_upper_case_tab = 
  [`a`,`A`
  ;`b`,`B`
  ;`c`,`C`
  ;`d`,`D`
  ;`e`,`E`
  ;`f`,`F`
  ;`g`,`G`
  ;`h`,`H`
  ;`i`,`I`
  ;`j`,`J`
  ;`k`,`K`
  ;`l`,`L`
  ;`m`,`M`
  ;`n`,`N`
  ;`o`,`O`
  ;`p`,`P`
  ;`q`,`Q`
  ;`r`,`R`
  ;`s`,`S`
  ;`t`,`T`
  ;`u`,`U`
  ;`v`,`V`
  ;`w`,`W`
  ;`x`,`X`
  ;`y`,`Y`
  ;`z`,`Z`
]
;;


let upcase_char c = 
  apply_alist lower_upper_case_tab c ? c
;;

let downcase_char c =  
  rev_apply_alist lower_upper_case_tab c ? c
;;

let upper_case_char_p c =
  member c (snd (unzip lower_upper_case_tab))
;;

let lower_case_char_p c =
  member c (fst (unzip lower_upper_case_tab))
;;

%[
***************************************************************************
String manipulation
***************************************************************************
The ml type "string" is a lifting of the lisp string type. Tokens are a 
lifting of lisp symbols. token operations other than equality are rather 
costly, so strings should be used for character manipulation intensive 
operations such as formatting output messages.

Defined in lisp:
  tok_to_string : tok -> string
  toks_to_string : tok list -> strings
  string_to_tok : string -> tok
 
  concatenate_strings : string list -> string
  explode_string : string -> tok list

  int_to_string : int -> string
  string_to_int : string -> int

  variable_to_string : variable -> string
  string_to_variable : string -> variable

  term_to_print_string : term -> int -> string
  rule_to_print_string : rule -> int -> string
]%

let append_strings s1 s2 = 
  concatenate_strings [s1;s2]
;;

ml_curried_infix `J`;;
let $J = append_strings ;;

let string_to_toks = explode_string ;;
let empty_string = toks_to_string [] ;;

let upcase_string s = 
  toks_to_string
    (map upcase_char (string_to_toks s))
;;

let char_replace old new s = 
  toks_to_string
    (map (\c. if c = old then new else c)
	 (string_to_toks s))
;;

let undash_string = char_replace `-` `_`;;
let unquestion_string = char_replace `?` `_QuestionMark_`;;

let downcase_string s =
  toks_to_string
    (map downcase_char (string_to_toks s))
;;

let reverse_string str = toks_to_string (rev (string_to_toks str)) ;;

let strip_string_prefix pre_s s =
  let pre_ts = string_to_toks pre_s 
  in let ts = string_to_toks s 
  in
    toks_to_string (remove_prefix pre_ts ts)
;;

let strip_string_suffix suf_s s =
  let suf_ts = string_to_toks suf_s 
  in let ts = string_to_toks s 
  in
    toks_to_string (rev (remove_prefix (rev suf_ts) (rev ts)))
;;


% string divided into sequences delimited by characters (toks) which 
satisfy P
%

let divide_string P str = 
  letrec aux toks seq = 
    if null toks then 
    ( if null seq then [] else [toks_to_string (rev seq)])
    else let tok.toks' = toks 
    in if P tok then
       (if null seq then 
          aux toks' [] 
        else 
          toks_to_string (rev seq) . aux toks' []
       )
       else
         aux toks' (tok.seq)
  in
    aux (explode_string str) []
;;

let string_to_words str = divide_string (\x.member x whitespace_toks) str
;;

% replace with direct calls to lisp functions.
let is_substring as bs = 
  is_sublist (string_to_toks as) (string_to_toks bs)
;;
let is_string_prefix as bs = 
  is_prefix (string_to_toks as) (string_to_toks bs)
;;
let is_string_suffix as bs = 
  is_suffix (string_to_toks as) (string_to_toks bs)
;;
%
%[
**********************************************************************
Failure control 
**********************************************************************

(f' : *->**),(id_if_lucky:***->***) = nths_time_lucky (ns:int list) (f:*->**) 

f' is like f, but it fails more often. Specifically, when f' is called with 
argument x, f' tries evaluating f x and notes whether the evaluation succeeds
or fails. f' keeps a count of the number of successes since it was created.
On the ith success f' returns f x if i is in ns. Otherwise it fails.
id_if_lucky is the identity if f has succeeded on every time listed in ns.
Otherwise id_if_lucky fails.


]%

let nths_time_lucky ns f =
  letref count = 0 in
  let maxn = accumulate max 0 ns in
  (\x. if count < maxn then 
         let y = f x in
           count := count + 1 ;
           if member count ns then y 
           else failwith `nths_time_lucky: too bad. count not in ns`
       else
         failwith `nths_time_lucky: too bad. count >= max ns`
  ) 
  , (\z. if count < maxn then
           failwith `nths_time_lucky: still waiting for last success`
         else z)
;;

let nth_time_lucky n = nths_time_lucky [n]
;;



%
(f' : *->**),(id_if_success:***->***) = add_success_monitor (f:*->**) 

f' behaves just like f. 
id_if_success is the identity only if f' has succeeded at least once.
%

let add_success_monitor f = 
  letref successful = false in
  (\x. let y = f x in 
         successful := true ;
         y
  ) ,
  (\z. if successful then z 
       else failwith `add_success_monitor: no success so far`)
;;




let with_default x f = \y. f y ? x ;;

let fail_with token fn = \t. fn t ? failwith token ;;


%
Provides a simple backtrace, since it prefixes a token to the previous
failure token.  Warning:  this
  (1)  slows down failure propagation.
  (2)  works only with the innermost lambda of a curried function.
%
let set_fail_prefix tok fun arg =
    fun arg ?\tail failwith (concatl [tok; ` -- `; tail]);;

let fails_p e a = 
  (e a ; false) ? true 
;;

% emulate lisp macro %
  
let unwind_protect bodyf cleanupf =  
  let val = inl(bodyf()) ?\x inr(x) in
  ( cleanupf () ;
    if isl val then outl val else failwith (outr val)
  )
;;

%[
For catching and discriminating on failures of function calls
]%


let maybe f x = inl (f x) ? inr () ;;
let successful val = isl val ;;
let value val = outl val ;;


%[
***************************************************************************
Miscellaneous
***************************************************************************
]%

letrec dotimes n f =
  if n=0 then () else (f (); dotimes (n-1) f)
;;


let assert P x = if P x then x else failwith `assert` ;;


%[
***************************************************************************
Graphs with Labelled Vertices
***************************************************************************
Assume graph is represented by list of triples 

(a,b),l

each such triple denoting an edge from node a to node b with label l.

The functions here are path finding functions.

An O(n*n) algorithm if there are n edges in graph.

]%
  
letrec explore_graph 
         is_acyclic 
         graph
         from_nodes
         to_node
         from_labels
       = 
  let from_node = hd from_nodes
  in
  letrec walk_entries graph' = 
    let ((a,b),lab).graph'' = graph'
    in 
    ( if a = from_node 
         & (is_acyclic or not member b from_nodes)
      then 
      ( if b = to_node then
          lab.from_labels
        else
          explore_graph
            is_acyclic
            graph
            (b.from_nodes)
            to_node
            (lab.from_labels)
      )
      else
        fail
    )
    ? 
    walk_entries graph''
  in
    walk_entries graph
;;


let find_graph_path_aux is_acyclic graph from to = 
  if from = to then []
  else
    rev (explore_graph is_acyclic graph [from] to [])
;;

let find_graph_path = find_graph_path_aux false ;;
let find_path_in_acyclic_graph = find_graph_path_aux true ;;


let test_graph = 
 [(1,2),`a`
 ;(2,3),`b`
 ;(3,4),`c`
 ;(4,1),`d`
 ;(2,5),`e`
 ;(7,3),`f`
 ]
;;


%[
***************************************************************************
Unlabelled Graphs.
***************************************************************************
Fast path checking functions with no consing.
]%

% 
loops if don't have dag, so good idea to do is_acyclic_graph
check when updating dag 
%

letrec exists_positive_path_in_dag dag from to = 
  letrec walk_dag dag' = 
    if null dag' then 
      false 
    if from = fst (hd dag') then
      to = snd (hd dag')
      or 
      exists_positive_path_in_dag dag (snd (hd dag')) to
      or
      walk_dag (tl dag')
    else
      walk_dag (tl dag')
  in
    walk_dag dag
;;

let is_acyclic_graph graph = 
  all 
    (\i,j.not exists_positive_path_in_dag graph i i)
    graph
;;

let exists_path_in_dag dag from to = 
  from = to or exists_positive_path_in_dag dag from to
;;

    
let test_dag = 
 [1,2
 ;3,2
 ;3,4
 ;4,1
 ;2,5
 ;7,3
 ]
;;

let bad_dag = 
 [1,2
 ;2,3
 ;3,4
 ;4,1
 ;2,5
 ;7,3
 ]
;;


%[
***************************************************************************
Discrimination Tree
***************************************************************************
types:

  indices: * list
  values : **
]%


absrectype (*,**) dtree = 
  (** + unit) # (* # (*,**) dtree) list
  with
      mk_dtree lab alist = abs_dtree (lab,alist)
  and dest_dtree t = rep_dtree t
;;

let null_dtree = mk_dtree (inr ()) [] ;;

letrec update_dtree dt is val = 
  let lab,alist = dest_dtree dt
  in
  if null is then
    mk_dtree (inl val) alist
  else
  let i.is' = is 
  in let alist' = 
     modify_or_add_alist_entry 
     (\dt. update_dtree dt is' val)
     (update_dtree null_dtree is' val)
     alist
     i
  in
    mk_dtree lab alist'
;;

letrec apply_dtree dt is = 
  if null is then
  ( outl (fst (dest_dtree dt)) ? failwith `apply_dtree: no value`)
  else
    let i.is' = is 
    in let (),alist = dest_dtree dt
    in let dt' = apply_alist alist i ? failwith `apply_dtree: invalid indices`
    in 
      apply_dtree dt' is'
;;


%[
***************************************************************************
Ordering functions
***************************************************************************
strict order relations for strings and tokens and lists.
]%

% chars are single character tokens % 

let char_lt a b = char_to_int a < char_to_int b ;;


% lexicographic: dictionary order 
  assume built in equality is OK (i.e. a<b or a=b or a>b)
  and is faster than lt, so want to minimize calls to lt fun.
%

let list_lex_lt el_lt as bs = 
  letrec comp as bs = 
    if null bs then false
    if null as then true
    if (hd as) = (hd bs) then comp (tl as) (tl bs)
    if el_lt (hd as) (hd bs) then true
    else
      false  
  in
    comp as bs
;;

let list_lex_lt_p eqp el_lt as bs = 
  letrec comp as bs = 
    if null bs then false
    if null as then true
    if eqp (hd as) (hd bs) then comp (tl as) (tl bs)
    if el_lt (hd as) (hd bs) then true
    else
      false  
  in
    comp as bs
;;


% redefined as call to lisp.
let string_lt a b = 
  list_lex_lt char_lt (string_to_toks a) (string_to_toks b)
;;
%

let tok_lt a b = string_lt (tok_to_string a) (tok_to_string b);;
      
let mk_string_list null_str pre_str in_str post_str strs = 
  letrec join strs = 
    if null strs then post_str
    else let str.strs' = strs
    in
      concatenate_strings [in_str;str;join strs'] 
  in
  if null strs then 
    null_str
  else
    concatenate_strings [pre_str;hd strs;join (tl strs)]
;;


%[
***************************************************************************
Leftist Heaps
***************************************************************************
See Tarjan, Data Structures and Network Algorithms p38 for details.

LHeaps store sets of (index,val) pairs where indices are totally 
ordered. They support operations:

lheap_insert i v h => h'

lheap_splitmin h => (i,v).h'

lheap_merge h1 h2 => h

lheap_mergelist [h1;...;hn] => h

null_lheap 


All operations maintain invariants:

rank(null_lheap) = 0
rank(h) = 1 + min(rank(lchild),rank(rchild))  if h not null
rank(lchild) GE rank(rchild) 

(index h) le (index lchild)
(index h) le (index rchild)

where le is order on indices.

It follows that rightmost paths from root down always
have length at most log_2 (n) where n is number of entries,
and that root node has a minimum index.
]%

absrectype (*,**) lheap = 
  unit + * # ** # int # (*,**) lheap # (*,**) lheap
  with
  % constructors %

  null_lheap = abs_lheap (inl())
  and mk_lheap i v r lh rh =
       abs_lheap (inr(i,v,r,lh,rh))
  
  % analyzer %

  and  is_null_lheap h = isl (rep_lheap h)

  % destructors %

  and  lheap_index h = fst (outr (rep_lheap h))
  and  lheap_value h = fst (snd (outr( rep_lheap h)))
  and  lheap_rank h = if isr (rep_lheap h) then fst (snd (snd (outr(rep_lheap h))))
                      else 0
  and  lheap_lchild h = fst (snd (snd (snd (outr (rep_lheap h)))))
  and  lheap_rchild h = snd (snd (snd (snd (outr (rep_lheap h)))))
;;

% create new node st rank(lchild) GE rank(rchild) 
Calculate rank of node

rank = 1 + min(rank(lchild),rank(rchild))

Ideally, put this in abstype defn then hide rank, but this is easier
to write.
%

let mk_lheap_node i v h1 h2 = 
  let r1 = lheap_rank h1 in
  let r2 = lheap_rank h2 in
  if r1 > r2 then
     mk_lheap i v (r2+1) h1 h2
  else
     mk_lheap i v (r1+1) h2 h1
;;

let mk_basic_lheap i v = 
  mk_lheap_node i v null_lheap null_lheap ;;
  
letrec lheap_merge lt ha hb = 
  if is_null_lheap ha then hb
  if is_null_lheap hb then ha
  if lt (lheap_index ha) (lheap_index hb) then
    mk_lheap_node 
      (lheap_index ha)
      (lheap_value ha)
      (lheap_lchild ha)
      (lheap_merge lt (lheap_rchild ha) hb)
  else
    mk_lheap_node 
      (lheap_index hb)
      (lheap_value hb)
      (lheap_lchild hb)
      (lheap_merge lt (lheap_rchild hb) ha)
;;

let lheap_insert lt i v h = 
  lheap_merge lt (mk_basic_lheap i v) h
;;

let lheap_splitmin lt h = 
  if is_null_lheap h then failwith `lheap_splitmin: heap empty`
  else
    lheap_index h
    , lheap_value h
    , lheap_merge lt (lheap_lchild h) (lheap_rchild h)
;;



%[
***************************************************************************
An example of a mutually-recursive datatype definition.
***************************************************************************
Type definition for odd and even length lists of elements of type *.  
]%

absrectype 
 * olist = * # * elist
 and
 * elist = * # * olist + unit
 with
   enil = abs_elist (inr())
   and econs x ol = abs_elist (inl (x , ol))
   and enull el = isr (rep_elist el)
   and ehd el = fst (outl (rep_elist el))
   and etl el = snd (outl (rep_elist el))
   and ocons x el = abs_olist (x , el) 
   and ohd ol = fst (rep_olist ol)
   and otl ol = snd (rep_olist ol)
;;


