
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;;;
;;;  proofs
;;;

(defunml (|make_proof_node| (assumptions concl))
    ((assumption list) -> (term -> proof))
  (instantiate-proof-node-r assumptions concl))

(dml |hypotheses| 1 assumptions-of-proof-node (proof -> (assumption list)))
(dml |conclusion| 1 conclusion-of-proof-node (proof -> term))

(defunml (|children| (proof) :error-wrap-p nil)
    (proof -> (proof list))
  (cond
    ((refined-proof-node-p proof)
     (children-of-proof-node proof))
    (t (breakout evaluation "children"))))

(defunml (|refinement| (proof) :error-wrap-p nil)
    (proof -> rule)
  (cond
    ((refined-proof-node-p proof)
     (rule-of-proof-node proof))
    (t (breakout evaluation "refinement"))))

(defunml (|refined_proof_p| (proof) :error-wrap-p nil)
    (proof -> bool)

  (refined-proof-node-p proof))



;;;;
;;;;   Annotating Proofs.
;;;;

;; replaces any current annotation
(defunml (|annotate_proof| (proof annotation) :error-wrap-p nil)
    (proof -> (* -> proof))

  ;;(setf -aa annotation pp proof) (when (or t (eq (caar aa) '|sub|)) (format t "ap: ~s~%" aa))
  ;;(when (eq (caar aa) '|sub|)(break))

  (when (null annotation)
    (breakout evaluation '|annotate_proof_nil|))
  
  ;;(mark-proof-node proof 'ml-annotation annotation)
  (mark-proof-node proof 'ml-annotation annotation)

  )

;; returns annotation or fails if no annotation.
(defunml (|annotation_of_proof| (proof) :error-wrap-p nil)
    (proof -> *)
	  
  (let* ((marks (or (marks-of-proof-node proof)
		    (breakout evaluation '|annotation_of_proof_not|)));;(null-annotation)
	 (value (mark-value marks 'ml-annotation)))
    ;;(setf m marks) (setf p proof) (break) 
	
    (when value
      ;;(setf v value) (break)
      (if (listp value)
	  value
	  (term-to-ml-annotation value)));; LAL should always be listp?
           
    ))
      
(defunml (|clear_annotation_of_proof| (proof) :error-wrap-p nil)
    (proof -> proof)
  (break "caop") (copy-of-proof-node proof nil 'ml-annotation))



;;;
;;;	Assumptions
;;;


(defunml (|destruct_assumption| (assum) :error-wrap-p nil)
    (assumption -> (variable |#| (term |#| bool)))
  (cons (id-of-assumption assum)
	(cons (type-of-assumption assum)
	      (hidden-assumption-p assum))))

(dml |id_of_assumption|		1	id-of-assumption	(assumption -> variable))
(dml |type_of_assumption|	1	type-of-assumption	(assumption -> term))
(dml |hidden_assumption_p|	1	hidden-assumption-p	(assumption -> bool))

(defunml (|nth_assumption| (i p) :error-wrap-p nil)
    (int -> (proof -> assumption))

  (let ((assums (assumptions-of-proof-node p)))
    (let ((l (length assums)))
      (let ((j (if (< i 0)
		   (+ l i)
		   (1- i))))

	(unless (< -1 j l)
	  (breakout evaluation '|nth_decl: index out of range|))

	(nth j assums)))))

    

(defunml (|make_assumption| (id type hidden-p))
    (variable -> (term -> (bool -> assumption)))
  (instantiate-assumption-r id type hidden-p))




;;;
;;; Rule args
;;;


(defun construct-argument (value syntax-directive)
  (make-argument :syntax (cond 
			   ((eql 'assumption-index syntax-directive) '|int|)
			   ((eql *natural-typeid* syntax-directive) '|int|)
			   ((eql 'binding syntax-directive) *variable-id-typeid*)
			   ((eql 'term syntax-directive) '|term|)
			   ((eql 'bound-term syntax-directive) '|bound_term|)
			   ((eql 'assumption-list syntax-directive) '|assumption_list|)
			   (t syntax-directive)) ; variable token
		 :value value))


;; RLE PERF: this is nfg. Change refiner to accept arguments directly.
;;   (check args supplied match expected args at rule instantiation.
;; ie rule interpreter can rely on correct number and types of args,
;; rule args should be argument struct list in rule instance.

;;
;; Constructors
;;


(defunml (|make_assumption_index_argument| (i) :error-wrap-p nil)
    (int -> argument)
  (make-argument :value i :syntax '|assumption_index|))

(defunml (|make_variable_argument| (v)  :error-wrap-p nil)
    (variable -> argument)
  (make-argument :value v :syntax '|variable|))

(defunml (|make_parameter_argument| (parameter) :error-wrap-p nil)
	  (parameter -> argument)
  (make-argument :value parameter :syntax '|parameter|))

(defunml (|make_term_argument| (term) :error-wrap-p nil)
	  (term -> argument)
  (make-argument :value term :syntax '|term|))

(defunml (|make_bound_term_argument| (vars term) :error-wrap-p nil)
	  ((variable list) -> (term -> argument))
  (make-argument :value (instantiate-bound-term term vars)
		 :syntax '|bound_term|))

(defunml (|make_assumption_list_argument| (assums) :error-wrap-p nil)
    ((assumption list) -> argument)
  (make-argument :value assums
		 :syntax '|assumption_list|))

(defunml (|make_substitution_list_argument| (psubs tsubs) :error-wrap-p nil)
    (((tok |#| parameter) list) -> (((variable |#| term) list) -> argument))
  (make-argument :value (append psubs tsubs)
		 :syntax '|substitution_list|))

(defunml (|make_proof_argument| (proof) :error-wrap-p nil)
    (proof -> argument)
  (make-argument :value proof
		 :syntax '|proof|))

(defunml (|make_permute_map_argument| (m) :error-wrap-p nil)
    ((int list) -> argument)
  (make-argument :value m
		 :syntax '|permute_map|))


;;;
;;; Rules:
;;;

;; 8/00 RLE assumption lists are not being coerced to terms
;;   but rule interpreter apparently expects this.
;;   the idea of converting to a term list here is somewhat suspect so
;;   re-examine later.
(defun argument-to-term-list (arg)
  (cond
    ((eql '|assumption_index| (argument-syntax arg))
     (list (assumption-index-term (argument-value arg))))
    ((eql *variable-id-typeid* (argument-syntax arg))
     (list (variable-term (argument-value arg))))
    ((eql '|parameter| (argument-syntax arg))
     (list (iparameter-term (argument-value arg))))
    ((eql '|term| (argument-syntax arg))
     (list (argument-value arg)))
    ((eql '|bound_term| (argument-syntax arg))
     (nconc (mapcar #'variable-term
		    (bindings-of-bound-term-r (argument-value arg)))
	    (list (term-of-bound-term (argument-value arg)))))
    ((eql '|assumption_list| (argument-syntax arg))
     (list (argument-value arg)))
    ((eql '|substitution_list| (argument-syntax arg))
     (list (map-list-to-ilist (argument-value arg)
			      (isub-nil-term)
			      #'(lambda (sub)
				  (if (parameter-p (cdr sub))
				      (iparameter-sub-term (if (level-expression-parameter-p (cdr sub))
							       (car sub)
							       (get-abstraction-meta-variable-id (car sub)))
							   (cdr sub))
				      (iterm-sub-term (car sub) (cdr sub)))))))
    ;;((eql *level-expression-typeid* (argument-syntax arg))
    ;;(list (level-expression-term (argument-value arg))))
    ;;((eql '|ifname| (argument-syntax arg))
    ;;(list (ifid-term (argument-value arg))))
    ;;((eql '|opname| (argument-syntax arg))
    ;;(list (opid-term (argument-value arg))))
    (t (system-error (error-message '(argument-to-term) (argument-syntax arg))))))


(defun term-to-argument (term)
  (let ((type (type-of-iargument-term term)))
    (case type

      (|assumption_index| (make-argument :syntax type
					 :value (value-of-parameter-r
						 (parameter-of-iparameter-term
						  (value-of-iargument-term term)))))

      (|variable| (make-argument :syntax type
				 :value (value-of-parameter-r
					 (parameter-of-iparameter-term
					  (value-of-iargument-term term)))))

      (|parameter| (make-argument :syntax type
				  :value (parameter-of-iparameter-term
					  (value-of-iargument-term term))))
      
      (|term|  (make-argument :syntax type
			      :value (value-of-iargument-term term)))

      (|bound_term|  (make-argument :syntax type
				    :value (bound-term-of-ibound-id-term
					    (value-of-iargument-term term))))


      (|make_assumption_list_argument|
       (make-argument
	:syntax type
	:value (map-isexpr-to-sexpr (value-of-iargument-term term)
				    (iassum-cons-op)
				    #'(lambda (term)
					(instantiate-assumption (id-of-iassumption-term term)
								(type-of-iassumption-term term)
								(hidden-of-iassumption-term term))))))

      (|make_substitution_list_argument|
       (make-argument :syntax type
		      :value (map-isexpr-to-sexpr (value-of-iargument-term term)
						  (isub-cons-op)
						  #'(lambda (term)
						      (if (iparameter-sub-term-p term)
							  (cons (variable-of-iparameter-sub-term term)
								(value-of-iparameter-sub-term term))
							  (cons (id-of-iterm-sub-term term)
								(term-of-iterm-sub-term term)))))))

      (otherwise (raise-error (error-message '(term argument)))))))

(defun argument-to-term (arg)
  (let ((type (syntax-of-argument arg)))
    (case type

      (|assumption_index| (iargument-term type
					(iparameter-term
					 (instantiate-parameter-r (value-of-argument arg)
								  *natural-type*))))

      (|variable| (iargument-term type
				  (iparameter-term
				   (instantiate-parameter-r (value-of-argument arg)
							    *variable-type*))))

      (|parameter| (iargument-term type
				   (iparameter-term (value-of-argument arg))))

      (|term| (iargument-term type (value-of-argument arg)))

      (|bound_term| (iargument-term type
				    (bound-term-to-ibound-id-term (value-of-argument arg))))

      (|assumption_list|
       (iargument-term type
		       (map-sexpr-to-isexpr (value-of-argument arg)
					    (iassum-nil-term)
					    #'(lambda (a)
						(iassumption-term (hidden-assumption-p a)
								  (id-of-assumption a)
								  (type-of-assumption a))))))

      (|substitution_list|
       (iargument-term type
		       (map-sexpr-to-isexpr (value-of-argument arg)
					    (isub-nil-term)
					    #'(lambda (sub)
						(if (parameter-p (cdr sub))
						    (iparameter-sub-term (car sub) (cdr sub))
						    (iterm-sub-term (car sub) (cdr sub)))))))

      (otherwise (raise-error (error-message '(argument term)))))))


(defunml (|term_to_argument| (term))
    (term -> argument)
  (term-to-argument term))


(defunml (|argument_to_term| (arg))
    (argument -> term)
  (argument-to-term arg))



(defunml (|make_primitive_rule| (tok args))
    (tok -> ((argument list) -> rule))
  ;;(when (member nil (mapcan #'argument-to-term-list args)) (break "mpr"))
  (make-primitive-rule :definition (rule-id-table-lookup tok)
		       :args (mapcan #'argument-to-term-list args)))


(defunml (|make_tactic_rule| (term) :error-wrap-p nil)
    (term -> rule)
    ;;(break "tr")
  (nml-tactic-rule term))


(defunml (|type_of_rule| (rule) :error-wrap-p nil)
    (rule -> tok)
  (type-of-rule rule))

(defunml (|id_of_rule| (rule) :error-wrap-p nil )
    (rule -> tok)
  (id-of-rule rule))

(defunml (|args_of_primitive_rule| (rule))
    (rule -> (term list))

  (if (primitive-rule-p rule)
      (filter #'term-p (args-of-primitive-rule rule))
      (raise-error (error-message '(args_of_primitive_rule primitive not)))))


(defunml (|is_prim_rule| (rule))
    (rule -> bool)
  
  (primitive-rule-p rule))

(defunml (|tactic_rule_to_term| (rule))
    (rule -> term)

  (text-of-tactic-rule tactic rule))



(defunml (|proof_of_tactic_rule| (rule))
  (rule -> proof)
  
  (cond
    ((nml-tactic-rule-p rule)
     (proof-of-tactic-rule rule))
    ;;((prl-rule-p rule)
    ;;(if (prl-rule-proof-top rule)
    ;;(prl-rule-proof-top rule)
    ;;(breakout evaluation "Proof_of_TacticRule: prl tactic has no proof.")))
    (t (breakout evaluation "Proof_of_TacticRule: rule is not a tactic."))))



(defunml (|equal_sequents| (p1 p2))
    (proof -> (proof -> bool))
  (equal-sequents-p p1 p2))

(defunml (|lex_equal_sequents| (p1 p2))
    (proof -> (proof -> bool))
  (lex-equal-sequents-p p1 p2))

(defunml (|alpha_equal_sequents| (p1 p2))
    (proof -> (proof -> bool))
  (alpha-equal-sequents-p p1 p2))

(defunml (|alpha_equal_sequents_aux| (ylongok x-assums x-concl y-assums y-concl))
    (bool -> ((assumption list) -> (term -> ((assumption list) -> (term -> bool)))))
  (alpha-equal-sequents-aux-p x-assums x-concl y-assums y-concl nil ylongok))

(defunml (|lex_equal_assumptions_p| (a1 a2))
    (assumption -> (assumption -> bool))
  (lex-equal-assumptions-p a1 a2))

(defunml (|alpha_equal_assumptions_p| (a1 a2))
    (assumption -> (assumption -> bool))
  (equal-assumptions-p a1 a2))

;;; RLE ??? should this be a recursive copy??
;;; what could possibley be the point of this? 
(defunml (|copy_proof| (proof) :error-wrap-p nil)
    (proof -> proof)
  (copy-proof-node proof))



	      
;;;;
;;;; Proof Cache
;;;;


;; LAL some of this not used

(defvar *current-proof-cache* nil)
(defvar *current-object* nil)
(defvar *current-object-name* nil)

(defun current-object ()
  *current-object*)

(defun current-object-name ()
   (if *current-object*
      (object-name *current-object*)
      '||))

(defun current-cache (&optional create-p)
  (or *current-proof-cache*
      (when (and create-p *current-object* (eql 'thm (object-kind *current-object*)))
	(setf *current-proof-cache* (or (theorem-object-cache *current-object*)
					(setf (theorem-object-cache *current-object*)
					      (make-proof-cache)))))))

(defun current-proof-object-id ()
  (or (when (boundp '*ref-current-objects*)
	(car *ref-current-objects*))
      (current-object-id)
      (process-err "NoCache"))
  )


(defun current-cache-object-name ()
  (if *current-proof-cache*
      (when *current-object* (object-name *current-object*))
    (if (and *current-object* (eql 'thm (object-kind *current-object*)))	
	(object-name *current-object*)
      ;;(process-err "NoCache")
      ;; (or *current-object-name* (process-err "NoCache"))
      (if (boundp '*ref-current-objects*)
	  (or (car *ref-current-objects*) (process-err "NoCache"))
	(process-err "NoCache"))
      )))



(defunml (|current_pcache| (unit) :declare ((declare (ignore unit))))
  (void -> object_id)
  (current-proof-object-id)) 

(defunml (|set_current_object| (name))
  (tok -> void)
(setf *current-object-name* name)) 

(defun new-proof-var (v p &optional (maybep t))
  (let ((assums (assumptions-of-proof-node p)))

    ;;(setf -p p -v v) (break "npv")
    (if (and maybep
	     (not (exists-p #'(lambda (assum) (eql v (id-of-assumption assum))) assums)))
	v
	(let ((olds (string v)))
	  (when (zerop (length olds))
	    (breakout evaluation '|HD|))
	  
	  (let ((oldch (char olds 0))
		(max-index 0))

	    (declare (integer max-index))

	    (dolist (h assums)
	      (setf max-index
		    (max max-index
			 (new-var-index-aux oldch (string (id-of-assumption h))))))

	    (new-var-aux oldch max-index))))))

(defunml (|maybe_new_proof_var| (v p))
    (variable -> (proof -> variable))

  (new-proof-var v p))

(defunml (|new_proof_var| (v p))
    (variable -> (proof -> variable))

  (new-proof-var v p nil))



