
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)


;; ******************************************************************
;; ******************************************************************
;; ******************************************************************
;;
;; Auxilliary Functions Used by the Main Monotonicity Rule Functions
;;
;; ******************************************************************
;; ******************************************************************
;; ******************************************************************

;; Returns the (obvious) list of pairs of elements (e1 e2),
;;   ei froom listi.  Length is lenth of shorter of the two lists.
(defun pairup (list1 list2)
  (cond
    ((null list1)  nil)			 
    ((null list2)  nil)
    (t 
      (cons  (list (car list1)
		   (car list2))
	     (pairup (cdr list1)
		     (cdr list2))))))

(define-primitive |!monot-error|)

;;; The monotonicity rule.

(defparameter valid-monot-opkinds '(+ - * /)
  "The valid operator kinds for monotonicity")

(defparameter monot-pattern-vars (list (get-variable-id '|t1|) (get-variable-id '|t2|)
				       (get-variable-id '|t3|) (get-variable-id '|t4|))
  "The variables used in the pattern-terms in the monotonicity tables")


(defun id-to-var (id)
  (variable-term (get-variable-id id)))

;;; Tables associating the op-kind atoms to the numbers
;;;   used as indices into the monotonicity tables.
(defvar relation-codes-plus
	(pairup '(GREATER GEQ EQ NEQ) '(0 1 2 3)))
(defvar relation-codes-minus
	(pairup '(GREATER GEQ EQ NEQ) '(0 1 2 3)))
(defvar relation-codes-mult2
	(pairup '(GEQ GREATER EQ NEQ) '(0 1 2 3)))
(defvar relation-codes-mult1
	(pairup '(GREATER GEQ EQ LEQ LESS NEQ) '(0 1 2 3 4 5)))
(defvar relation-codes-div1
	(pairup '(GREATER LESS NEQ) '(0 1 2)))
(defvar relation-codes-div2
	(pairup '(GREATER GEQ EQ NEQ) '(0 1 2 3)))

;;; The arrays.
(defvar monot-table-plus
	(make-array 
	  '(4 4) 
	  :initial-contents
	  (list
	    (list
	      ;; (t1+t3 < t2+t4+2) -> void
	      (not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (add-term (id-to-var '|t2|)
				       (add-term (id-to-var '|t4|) (natural-number-term 2))))

	      ;; (t1+t3 < t2+t4+1) -> void
	      (not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (add-term (id-to-var '|t2|)
				       (add-term (id-to-var '|t4|) (natural-number-term 1))))
	  
	      ;; ((t1+t3 < t2+t4+1) -> void) # ((t1+t4 < t2+t3+1) -> void)
	      (independent-product-term 
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t2|)
					 (add-term (id-to-var '|t4|)
						   (natural-number-term 1))))
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (add-term (id-to-var '|t2|)
					 (add-term (id-to-var '|t3|)
						   (natural-number-term 1)))))

	      ;; 'error
	      (imonot-error-term))

	    (list
	      ;; (t1+t3 < t2+t4+1) -> void
	      (not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (add-term (id-to-var '|t2|)
				       (add-term (id-to-var '|t4|) (natural-number-term 1))))
	  
	      ;; (t1+t3 < t2+t4) -> void
	      (not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (add-term (id-to-var '|t2|) (id-to-var '|t4|)))

	  
	      ;; ((t1+t3 < t2+t4)->void) # ((t1+t4 < t2+t3) -> void)
	      (independent-product-term 
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t2|) (id-to-var '|t4|)))
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (add-term (id-to-var '|t2|) (id-to-var '|t3|))))
	  
	      ;; 'ERROR
	      (imonot-error-term))

	    (list
	      ;; ((t1+t3 < t2+t4+1)->void) # ((t2+t3 < t1+t4+1) -> void)
	      (independent-product-term 
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t2|)
					 (add-term (id-to-var '|t4|)
						   (natural-number-term 1))))
		(not-less-than-term (add-term (id-to-var '|t2|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t1|)
					 (add-term (id-to-var '|t4|)
						   (natural-number-term 1)))))

	      ;; ((t1+t3 < t2+t4)->void) # ((t2+t3 < t1+t4) -> void)
	      (independent-product-term 
		(not-less-than-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t2|) (id-to-var '|t4|)))
		(not-less-than-term (add-term (id-to-var '|t2|) (id-to-var '|t3|))
			       (add-term (id-to-var '|t1|) (id-to-var '|t4|))))

	      ;; (t1+t3 = t2+t4 in int) # (t1+t4 = t2+t3 in int)
	      (independent-product-term
		(int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
				(add-term (id-to-var '|t2|) (id-to-var '|t4|)))
		(int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t4|))
				(add-term (id-to-var '|t2|) (id-to-var '|t3|))))

	      ;; ((t1+t3 = t2+t4 in int)->void) # ((t1+t4 = t2+t3 in int) -> void)
	      (independent-product-term 
		(not-int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
				    (add-term (id-to-var '|t2|) (id-to-var '|t4|)))
		(not-int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t4|))
				    (add-term (id-to-var '|t2|) (id-to-var '|t3|)))))


	    (list
	      ;; 'ERROR
	      (imonot-error-term)

	      ;; 'ERROR
	      (imonot-error-term)

	      ;; ((t1+t3 = t2+t4 in int)->void) # ((t1+t4 = t2+t3 in int) -> void)
	      (independent-product-term 
		(not-int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t3|))
				    (add-term (id-to-var '|t2|) (id-to-var '|t4|)))
		(not-int-equal-term (add-term (id-to-var '|t1|) (id-to-var '|t4|))
				    (add-term (id-to-var '|t2|) (id-to-var '|t3|))))

	      ;; 'ERROR
	      (imonot-error-term)))))

(defvar monot-table-minus
	(make-array
	  '(4 4) 
	  :initial-contents
	  (list
	    (list
	      ;; (t1-t4 < (t2-t3)+2) -> void
	      (not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			     (add-term (subtract-term (id-to-var '|t2|) 
						 (id-to-var '|t3|))
				       (natural-number-term 2)))

	      ;; (t1-t4 < (t2-t3)+1) -> void
	      (not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			     (add-term (subtract-term (id-to-var '|t2|)
						 (id-to-var '|t3|))
				       (natural-number-term 1)))

	      ;; ((t1-t4 < (t2-t3)+1)->void) # ((t1-t3 < (t2-t4)+1) -> void)
	      (independent-product-term 
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (add-term (subtract-term (id-to-var '|t2|)
						   (id-to-var '|t3|))
					 (natural-number-term 1)))
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (add-term (subtract-term (id-to-var '|t2|)
						   (id-to-var '|t4|))
					 (natural-number-term 1))))

	      ;; 'ERROR
	      (imonot-error-term))

	    (list
	      ;; (t1-t4 < (t2-t3)+1) -> void
	      (not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			     (add-term (subtract-term (id-to-var '|t2|)
						 (id-to-var '|t3|))
				       (natural-number-term 1)))

	      ;; (t1-t4 < t2-t3) -> void
	      (not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			     (subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))

	      ;; ((t1-t4 < t2-t3)->void) # ((t1-t3 < t2-t4) -> void)
	      (independent-product-term
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t3|))
			       (subtract-term (id-to-var '|t2|) (id-to-var '|t4|))))

	      ;; 'error
	      (imonot-error-term))

	    (list
	      ;; ((t1-t4 < (t2-t3)+1)->void) # ((t2-t4 < (t1-t3)+1) -> void)
	      (independent-product-term
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (add-term (subtract-term (id-to-var '|t2|)
						   (id-to-var '|t3|))
					 (natural-number-term 1)))
		(not-less-than-term (subtract-term (id-to-var '|t2|) (id-to-var '|t4|))
			       (add-term (subtract-term (id-to-var '|t1|)
						   (id-to-var '|t3|))
					 (natural-number-term 1))))

	      ;; ((t1-t4 < t2-t3)->void) # ((t2-t4 < t1-t3) -> void)
	      (independent-product-term 
		(not-less-than-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
			       (subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))
		(not-less-than-term (subtract-term (id-to-var '|t2|) (id-to-var '|t4|))
			       (subtract-term (id-to-var '|t1|) (id-to-var '|t3|))))

	      ;; (t1-t4 = t2-t3 in int) # (t2-t4 = t1-t3 in int)
	      (independent-product-term 
		(int-equal-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
				(subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))
		(int-equal-term (subtract-term (id-to-var '|t2|) (id-to-var '|t4|))
				(subtract-term (id-to-var '|t1|) (id-to-var '|t3|))))

	      ;; ((t1-t4 = t2-t3 in int)->void) # ((t1-t3 = t2-t4 in int) -> void)
	      (independent-product-term
		(not-int-equal-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
				    (subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))
		(not-int-equal-term (subtract-term (id-to-var '|t1|) (id-to-var '|t3|))
				    (subtract-term (id-to-var '|t2|) (id-to-var '|t4|)))))

	    (list
	      ;; 'error
	     (imonot-error-term)

	      ;; 'error
	     (imonot-error-term)

	      ;; ((t1-t4 = t2-t3 in int)->void) # ((t1-t3 = t2-t4 in int) -> void)
	      (independent-product-term 
		(not-int-equal-term (subtract-term (id-to-var '|t1|) (id-to-var '|t4|))
				    (subtract-term (id-to-var '|t2|) (id-to-var '|t3|)))
		(not-int-equal-term (subtract-term (id-to-var '|t1|) (id-to-var '|t3|))
				    (subtract-term (id-to-var '|t2|) (id-to-var '|t4|))))

	      ;; 'error
	      (imonot-error-term) ))))

(defvar monot-table-mult
	(make-array 
	  '(6 4) 
	  :initial-contents
	  (list
	    (list
	      ;; (t1*t2 < t1*t3) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t3 < t1*t2)
	      (less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))
			 (multiply-term (id-to-var '|t1|) (id-to-var '|t2|)))

	      ;; (t1*t2 =  t1*t3 in int)
	      (int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			      (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 = t1*t3 in int) -> void
	      (not-int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				  (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))))

	    (list
	      ;; (t1*t2 < t1*t3) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 < t1*t3) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 =  t1*t3 in int)
	      (int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			      (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; 'error
	      (imonot-error-term))

	    (list
	      ;; (t1*t2 = t1*t3 in int) # (t1*t3 = 0 in int)
	      (independent-product-term 
		(int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				(multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))
		(equal-zero-term (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))))

	      ;; (t1*t2 = t1*t3 in int) # (t1*t2 = 0 in int)
	      (independent-product-term
		(int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				(multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))
		(equal-zero-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))))

	      ;; (t1*t2 = t1*t3 in int) # (t1*t2 = 0 in int)
	      (independent-product-term 
		(int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				(multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))
		(equal-zero-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))))

	      ;; (t1*t2 = t1*t3 in int) # (t1*t2 = 0 in int)
	      (independent-product-term 
		(int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				(multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))
		(equal-zero-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|)))))

	    (list
	      ;; (t1*t3 < t1*t2) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t2|)))

	      ;; (t1*t3 < t1*t2) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t2|)))

	      ;; (t1*t2 =  t1*t3 in int)
	      (int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			      (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; 'error
	      (imonot-error-term))

	    (list
	      ;; (t1*t3 < t1*t2) -> void
	      (not-less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))
			     (multiply-term (id-to-var '|t1|) (id-to-var '|t2|)))

	      ;; (t1*t2 < t1*t3)
	      (less-than-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			 (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 =  t1*t3 in int)
	      (int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			      (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 = t1*t3 in int) -> void
	      (not-int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				  (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))))

	    (list
	      ;; 'error
	      (imonot-error-term)

	      ;; (t1*t2 = t1*t3 in int) -> void
	      (not-int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				  (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))

	      ;; (t1*t2 =  t1*t3 in int)
	      (int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
			      (multiply-term (id-to-var '|t1|) (id-to-var '|t3|)))


	      ;; (t1*t2 = t1*t3 in int) -> void
	      (not-int-equal-term (multiply-term (id-to-var '|t1|) (id-to-var '|t2|))
				  (multiply-term (id-to-var '|t1|) (id-to-var '|t3|))) ))))


(defvar monot-table-div
	(make-array 
	  '(3 4) 
	  :initial-contents
	  (list
	    (list
	      ;; (t2 < t1)
	      (less-than-term (id-to-var '|t2|) (id-to-var '|t1|))

	      ;; (t1 < t2) -> void
	      (not-less-than-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; (t1 = t2 in int)
	      (int-equal-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; (t1 = t2 in int) -> void
	      (not-int-equal-term (id-to-var '|t1|) (id-to-var '|t2|)))

	    (list
	      ;; (t1 < t2)
	      (less-than-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; (t2 < t1) -> void
	      (not-less-than-term (id-to-var '|t2|) (id-to-var '|t1|))

	      ;; (t1 = t2 in int)
	      (int-equal-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; (t1 = t2 in int) -> void
	      (not-int-equal-term (id-to-var '|t1|) (id-to-var '|t2|)))

	    (list
	      ;; (t1 = t2 in int) -> void
	      (not-int-equal-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; 'error
	      (imonot-error-term)

	      ;; (t1 = t2 in int)
	      (int-equal-term (id-to-var '|t1|) (id-to-var '|t2|))

	      ;; (t1 = t2 in int) -> void
	      (not-int-equal-term (id-to-var '|t1|) (id-to-var '|t2|)))) ))



#+nuprl-compatability
(defun refine-monot () 
 (let*  
   ((hnum1    (hypnum1-of-monot-rule ref-rule))
    (hnum2    (hypnum2-of-monot-rule ref-rule))
    (opkind   (op-of-monot-rule ref-rule))
    (decl1    (get-assum$ hnum1 ref-assums))
    (decl2    (get-assum$ hnum2 ref-assums))
    (hyp1     (type-of-assumption decl1))
    (hyp2     (type-of-assumption decl2)))

   (handle-error 'monot-tag
		 #'ref-error

     (setf ref-children
	   (list
	     (make-child
	       (append
		 ref-assums
		 (list (declaration nil nil (do-monot$ opkind hyp1 hyp2))))

	       ref-concl))))
   nil))



;; Checks opkind and the hypothesis-terms for correct format,
;;  and if correct, returns the new hypotheses-term (representing
;;  an inference using one instance of non-trivial monotonicity
;;  of integer arithmetic.)

#+nuprl-compatability
(defun do-monot$ (opkind hyp1 hyp2)
  ;; Returns the new hypothesis term representing an inference using
  ;; one instance of non-trivial monotonicity.
  (case opkind
    (+
     (lookup-and-make-literal$
       monot-table-plus (process-hyps-plus$ hyp1 hyp2)))
    (-
     (lookup-and-make-literal$
       monot-table-minus (process-hyps-minus$ hyp1 hyp2)))
    (*
     (lookup-and-make-literal$
       monot-table-mult (process-hyps-mult$ hyp1 hyp2)))
    (/
     (lookup-and-make-literal$
       monot-table-div (process-hyps-div$ hyp1 hyp2)))
    (otherwise
     (throw 'monot-tag "Improper operation given for monotonicity rule"))))

;; Returns the term which will be the new hypoth generated by
;;   monot rule.  Uses the indices (obtained from the relation-kind
;;   of the input hypoths), and the terms (primary subterms of
;;   the input hypoths) to find the right pattern-term in the monot
;;   tables and then perform the appropriate substitution.

(defun lookup-and-make-literal$  (table  indices-and-terms)
  
  (let*
    ((indices  (car indices-and-terms))
     (terms    (cadr indices-and-terms))
     (row      (car indices))
     (column   (cadr indices))
     (pattern-term  (aref table row column)))
    (cond
      ((imonot-error-term-p pattern-term)
       (throw 'monot-tag "Wrong relation-kind in a monot hypothesis."))
      (t
       (make-new-monot-literal pattern-term terms)))))

;; Pair up the variables from the pattern-term with
;;   the user-supplied terms.  Perform the substitution.
(defun make-new-monot-literal (pattern terms)
 (let*
   ((vars monot-pattern-vars)
    (subst-list  (pairup vars terms)))
   (substitute pattern subst-list)))

;;**********************************************************
;; The destruct-...-monot functions:
;;   Analyze a monot hypothesis, check for correct form,
;;   and return     1. relation-kind  ('EQ, 'LESS, etc.)
;;                  2. The two subterms
;;
;;   The correct format depends on which kind of monotonicity 
;;     (addition, mult, subt, div), and which of the two necessary
;;     hypoths is being processed.
;;
;;   There are six of these functions:
;;       addition     1
;;       subtraction  1
;;       multiplic.   2
;;       division     2
;;
;;**********************************************************
(defun destruct-plus-monot (hyp-term)

  (cond

    ((less-than-term-p hyp-term)
     (list  'GREATER
	    (rightterm-of-less-than-term hyp-term)
	    (leftterm-of-less-than-term hyp-term)))

    ((equal-term-p hyp-term)
       (let*  ((tms (terms-of-equal-term hyp-term)))
	 (cond
	   ((= (length tms) 1)
	    (list  'EQ
		   (car tms)
		   (car tms)))

	   (t
	    (list  'EQ
		   (car tms)
		   (cadr tms))))))

    ((not-term-p hyp-term)
     (let*  ((ltm (lefttype-of-function-term hyp-term)))
       (cond
	 ((less-than-term-p ltm)
	  (let*
	    ((left (leftterm-of-less-than-term ltm))
	     (right (rightterm-of-less-than-term ltm)))
	    (list  'GEQ
		   left
		   right)))

	 ((equal-term-p ltm)
	  (let* ((tms (terms-of-equal-term ltm)))
	    (cond
	      ((= (length tms) 2)
	       (list  'NEQ
		      (car tms)
		      (cadr tms)))
	      (t (throw 'monot-tag "monot: Bad hypothesis, too many terms in equal term")))))

       
	 (t (throw 'monot-tag "monot: Bad format for hypothesis. ")))))

     (t (throw 'monot-tag "monot: Bad format for hypothesis."))))


(defun destruct-minus-monot (hyp-term)
  (destruct-plus-monot hyp-term))


(defun destruct-mult2-monot (hyp-term)
  (destruct-plus-monot hyp-term))


(defun destruct-mult1-monot (hyp-term)

  (cond

    ((less-than-term-p hyp-term)
     (let*
       ((left (leftterm-of-less-than-term hyp-term))
	(right (rightterm-of-less-than-term hyp-term)))
       (cond 
	 ((zero-term-p left)
	  (list  'GREATER
		 right
		 left))
	 ((zero-term-p right)
	  (list  'LESS
		 left
		 right))
	 (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

    ((equal-term-p hyp-term)
       (let*  ((tms (terms-of-equal-term hyp-term)))
	 (cond
	   ((= (length tms) 2)
	    (cond 
	      ((zero-term-p (car tms))
	       (list  'EQ
		      (cadr tms)
		      (car tms)))
	      ((zero-term-p (cadr tms))
	       (list  'EQ
		      (car tms)
		      (cadr tms)))
	      (t  (throw 'monot-tag "monot: Bad format for hypothesis."))))
	   (t  (throw 'monot-tag "monot: Bad format for hypothesis.")))))

	  

    ((not-term-p hyp-term)
     (let*  ((ltm (lefttype-of-function-term hyp-term)))
       (cond
	 ((less-than-term-p ltm)
	  (let*
	    ((left (leftterm-of-less-than-term ltm))
	     (right (rightterm-of-less-than-term ltm)))
	    (cond 
	      ((zero-term-p left)
	       (list  'LEQ
		      right
		      left))
	      ((zero-term-p right)
	       (list  'GEQ
		      left
		      right))
	      (t  (throw 'monot-tag "monot: Bad format for hypothesis.")))))

	 ((equal-term-p ltm)
	  (let* ((tms (terms-of-equal-term ltm)))
	    (cond
	      ((= (length tms) 2)
	       (cond 
		 ((zero-term-p (car tms))
		  (list  'NEQ
			 (cadr tms)
			 (car tms)))
		 ((zero-term-p (cadr tms))
		  (list  'NEQ
			 (car tms)
			 (cadr tms)))
		 (t  (throw 'monot-tag "monot: Bad format for hypothesis."))))

	      (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

       
	 (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

     (t (throw 'monot-tag "monot: Bad format for hypothesis."))))

(defun destruct-div2-monot (hyp-term)
  (destruct-plus-monot  hyp-term))

(defun destruct-div1-monot (hyp-term)

  (cond

    ((less-than-term-p hyp-term)
     (let*
       ((left (leftterm-of-less-than-term hyp-term))
	(right (rightterm-of-less-than-term hyp-term)))
       (cond 
	 ((zero-term-p left)
	  (list  'GREATER
		 right
		 left))
	 ((zero-term-p right)
	  (list  'LESS
		 left
		 right))
	 (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

	  

    ((not-term-p hyp-term)
     (let*  ((ltm (lefttype-of-function-term hyp-term)))
       (cond
	 ((equal-term-p ltm)
	  (let* ((tms (terms-of-equal-term ltm)))
	    (cond
	      ((= (length tms) 2)
	       (cond 
		 ((zero-term-p (car tms))
		  (list  'NEQ
			 (cadr tms)
			 (car tms)))
		 ((zero-term-p (cadr tms))
		  (list  'NEQ
			 (car tms)
			 (cadr tms)))
		 (t  (throw 'monot-tag "monot: Bad format for hypothesis."))))

	      (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

       
	 (t (throw 'monot-tag "monot: Bad format for hypothesis.")))))

     (t (throw 'monot-tag "monot: Bad format for hypothesis."))))

;; Process the user's two hypotheses given to
;;  the monotonicity-addition rule.
;; Return list (rels  tms), where
;;   rels is list of the two relation-codes which will
;;   index into the monotonicity-addition table, and
;;   tms is list of the terms given for substitution into
;;   the new, rule-generated hypothesis term.
;;
(defun process-hyps-plus$ (hyp1 hyp2)
  (let*
    ((results1  (destruct-plus-monot hyp1))
     (results2  (destruct-plus-monot hyp2))
     (rel1  (cadr (assoc (car results1) relation-codes-plus)))
     (rel2  (cadr (assoc (car results2) relation-codes-plus)))
     (t1  (cadr results1))
     (t2  (caddr results1))
     (t3  (cadr results2))
     (t4  (caddr results2)))
    (list
      (list rel1 rel2)
      (list t1 t2 t3 t4))))

;; Process the user's two hypotheses given to
;;  the monotonicity-subtraction rule.
;; Return list (rels  tms), where
;;   rels is list of the two relation-codes which will
;;   index into the monotonicity-subtraction table, and
;;   tms is list of the terms given for substitution into
;;   the new, rule-generated hypothesis term.
;;
(defun process-hyps-minus$ (hyp1 hyp2)
  (let*
    ((results1  (destruct-minus-monot hyp1))
     (results2  (destruct-minus-monot hyp2))
     (rel1  (cadr (assoc (car results1) relation-codes-minus)))
     (rel2  (cadr (assoc (car results2) relation-codes-minus)))
     (t1  (cadr results1))
     (t2  (caddr results1))
     (t3  (cadr results2))
     (t4  (caddr results2)))
    (list
      (list rel1 rel2)
      (list t1 t2 t3 t4))))

;; Process the user's two hypotheses given to
;;  the monotonicity-multiplication rule.
;; Return list (rels  tms), where
;;   rels is list of the two relation-codes which will
;;   index into the monotonicity-multiplication table, and
;;   tms is list of the terms given for substitution into
;;   the new, rule-generated hypothesis term.
(defun process-hyps-mult$ (hyp1 hyp2)
  (let*
    ((results1  (destruct-mult1-monot hyp1))
     (results2  (destruct-mult2-monot hyp2))
     (rel1  (cadr (assoc (car results1) relation-codes-mult1)))
     (rel2  (cadr (assoc (car results2) relation-codes-mult2)))
     (t1  (cadr results1))
    
     (t2  (cadr results2))
     (t3  (caddr results2)))
    (list
      (list rel1 rel2)
      (list t1 t2 t3))))

;; Process the user's two hypotheses given to
;;  the monotonicity-division rule.
;; Return list (rels  tms), where
;;   rels is list of the two relation-codes which will
;;   index into the monotonicity-division table, and
;;   tms is list of the terms given for substitution into
;;   the new, rule-generated hypothesis term.
(defun process-hyps-div$ (hyp1 hyp2)
  (declare (ignore hyp1 hyp2))
  (ref-error "The monotonicity rule is not yet implemented for division.  Sorry.")
  #|
  (let*
    ((results1  (destruct-div1-monot hyp1))
     (results2  (destruct-div2-monot hyp2))
     (rel1  (cadr (assoc (car results1) relation-codes-div1)))
     (rel2  (cadr (assoc (car results2) relation-codes-div2)))
     (t1  (cadr results1))
    
     (t2  (cadr results2))
     (t3  (caddr results2))
     (d1  (simplify (poly-div t2 t1)))
     (d2  (simplify (poly-div t3 t1))))
    (list
       (list rel1 rel2)
       (list d1 d2)))
  |#)



