
;;;************************************************************************
;;;                                                                       *
;;;    Copyrighted Cornell University 2000                                *
;;;                                                                       *
;;;                                                                       *
;;;                Nuprl Proof Development System                         *
;;;                ------------------------------                         *
;;;                                                                       *
;;;   Developed by the Nuprl group, Department of Computer Science,       *
;;;   Cornell University, Ithaca NY.  See the release notes for a list    *
;;;   of the members of the group.                                        *
;;;                                                                       *
;;;   Permission is granted to use and modify Nuprl provided this notice  *
;;;   is retained in derived works.                                       *
;;;                                                                       *
;;;************************************************************************

#+cltl2(in-system-package)
#-cltl2(in-package *system-package-name*)

;;;; -docs- (mod ref)
;;;;
;;;;	what is a simplified term ?
;;;;
;;;;	my best guess is
;;;;
;;;;	if ([c0 +] a1 + a2 + ... + an) 
;;;;	   then simplified if 
;;;;		c0 is a constant
;;;;		there are no constants in a1 ... an
;;;;		a1 to an are in lexicographic order
;;;;		there are no ai ai+1 which are lexicographically equal.
;;;;            there are no ai which are add-terms
;;;;            
;;;;	        sum all constants so that result is
;;;;               (c + ai + ... + am) where ai is not an integer.
;;;;
;;;;
;;;;  if ([c0 *] a1**e1 * ... * an**en )
;;;;
;;;;     then simplified if
;;;;
;;;;		c0 is a constant
;;;;		there are no constants in a1 ... an.
;;;;		a1 to an are in lexicographic order
;;;;            there are no two ai ai+1 which are lexicographically equal.
;;;;		no ai is an add term.
;;;;		no ai is a minus term
;;;;               
;;;;		(a + b) * c * d -> (a * c * d) + (b * c * d)
;;;;
;;;;  -page-
;;;; -doct- (mod ref ml)
;;;;
;;;;	arith_simplify_complete_term	: term -> term
;;;;	arith_simplify_term		: term -> term
;;;;
;;;; -doce-


;;; some things we can say

;; if coeff of s-term is zero then (zero-term-p s-term) is t.
;; if coeff of s-term is -1 then (minus-term-p of s-term is t.
;; if term of s-term is nil then (integer-term-p s-term) is t.
;; multiplicands of a multiply term can not be add term or a zero-term.
;; s-term is a constant iff term-of s-term is nil.


(defun simplified-term (coeff term)

  (cond
    
    ((null term) (integer-term coeff))
    
    ((zerop coeff) (zero-term))

    ((onep coeff) term)

    ((neg-onep coeff) (minus-term term))

    (t (multiply-term (integer-term coeff) term))))


(defmacro constant-simplified-term-p (term)
  `(integer-term-p ,term))

(defmacro zero-simplified-term-p (term)
  `(and (integer-term-p ,term)
	(zerop (integer-of-integer-term ,term))))


(defmacro constant-simplified-term (coeff)
  `(integer-term ,coeff))

(defun coefficient-of-simplified-term (term)

  (cond

    ((integer-term-p term)	(integer-of-integer-term term))
    
    ((minus-term-p term)	-1)

    ((not (multiply-term-p term))	1)

    ((integer-term-p (leftterm-of-multiply-term term)) 
     (integer-of-integer-term (leftterm-of-multiply-term term)))
    
    (t 1)))
  

(defun term-of-simplified-term (term)

  (cond 

    ((integer-term-p term)	nil)

    ((minus-term-p term)	(term-of-minus-term term))

    ((and (multiply-term-p term)
	  (integer-term-p (leftterm-of-multiply-term term)))
     (rightterm-of-multiply-term term))

    (t term)))


(defun multiplicands-of-term-of-simplified-term (term)
  (if (multiply-term-p term)
      (cons (leftterm-of-multiply-term term)
	    (multiplicands-of-term-of-simplified-term (rightterm-of-multiply-term term)))
      (list term)))
   

(defun multiplicands-to-term-of-simplified-term (multiplicands)
  (if (null (cdr multiplicands))
      (car multiplicands)
      (multiply-term (car multiplicands)
		(multiplicands-to-term-of-simplified-term (cdr multiplicands)))))


(defun addends-of-simplified-term (term)
  (if (add-term-p term)
      (cons (leftterm-of-add-term term)
	    (addends-of-simplified-term (rightterm-of-add-term term)))
      (list term)))
                                                                          

(defun addends-to-simplified-term (addends)
  (cond
    ((null addends)		(zero-term))
    ((null (cdr addends))	(car addends))
    (t				(add-term (car addends)
					  (addends-to-simplified-term (cdr addends))))))


(defun simplify (expr)    
  
  (cond
         
    ((add-term-p expr)
     (simplify-addends (list (simplify (leftterm-of-binary-term expr))
			     (simplify (rightterm-of-binary-term expr)))))
    
    ((multiply-term-p expr)
     (simplify-multiplicands (list (simplify (leftterm-of-binary-term expr))
				   (simplify (rightterm-of-binary-term expr)))))
         
    ((divide-term-p expr)
     (let ((dividend (simplify (leftterm-of-binary-term expr)))
	   (divisor  (simplify (rightterm-of-binary-term expr))))
       (if (and (integer-term-p dividend)
		(integer-term-p divisor)
		(not (zero-term-p divisor)))
	   (integer-term (truncate (integer-of-integer-term dividend)
				   (integer-of-integer-term divisor)))
	   (divide-term dividend divisor))))

    ((subtract-term-p expr)
     (simplify-addends (list (simplify (leftterm-of-binary-term expr))
			     (negate-simplified-term
			       (simplify (rightterm-of-binary-term expr))))))
    
    ((minus-term-p expr)
     (negate-simplified-term (simplify (term-of-minus-term expr))))

    (t (mapcr-on-subterms #'(lambda (term)
			      (if (null (bound-terms-of-term term))
				  term
				  (simplify term)))
			  (expr)))))



(defun simplify-top (expr)    
  
  (cond
         
    ((add-term-p expr)
     (simplify-addends (list (simplify (leftterm-of-binary-term expr))
			     (simplify (rightterm-of-binary-term expr)))))
    
    ((multiply-term-p expr)
     (simplify-multiplicands (list (simplify (leftterm-of-binary-term expr))
				   (simplify (rightterm-of-binary-term expr)))))
         
    ((divide-term-p expr)
     (let ((dividend (simplify (leftterm-of-binary-term expr)))
	   (divisor  (simplify (rightterm-of-binary-term expr))))
       (if (and (integer-term-p dividend)
		(integer-term-p divisor)
		(not (zero-term-p divisor)))
	   (integer-term (truncate (integer-of-integer-term dividend)
				   (integer-of-integer-term dividend)))
	   (divide-term dividend divisor))))

    ((subtract-term-p expr)
     (simplify-addends (list (simplify (leftterm-of-binary-term expr))
			     (negate-simplified-term
			       (simplify (rightterm-of-binary-term expr))))))
    
    ((minus-term-p expr)
     (negate-simplified-term (simplify (term-of-minus-term expr))))

    (t expr)))


;;
;; produces simplified term from simplified term.
;;
(defun negate-simplified-term (term)
  (if (add-term-p term)
      (add-term (negate-simplified-term (leftterm-of-binary-term term))
		(negate-simplified-term (rightterm-of-binary-term term)))
      (simplified-term (- 0 (coefficient-of-simplified-term term))
		       (term-of-simplified-term term))))


;;
;; produces simplified term from list of simplified addends.
;;

(defun simplify-addends (addends)

  (if (null (cdr addends))

      (car addends)

      (addends-to-simplified-term
	(merge-lists-by-comparator 
	  (addends-of-simplified-term (car addends))
	  (addends-of-simplified-term (simplify-addends (cdr addends)))
	  #'(lambda (terma termb)
	      (let* ((term1 (term-of-simplified-term terma))
		     (term2 (term-of-simplified-term termb))
		     (lower-term (lexicographically-compare-terms term1 term2)))
		(cond 
	 ((eql t lower-term)		; ie, terms are equal
		   (let ((coeff (+ (coefficient-of-simplified-term terma)
				   (coefficient-of-simplified-term termb))))
		     (unless (zerop coeff)
		       (simplified-term coeff term1))))
		  
		  ;; constants always first.
		  ((constant-simplified-term-p terma)
		   (if (zerop (coefficient-of-simplified-term terma))
		       'skip-a
		       terma))
		       
		  ((constant-simplified-term-p termb)
		   (if (zerop (coefficient-of-simplified-term termb))
		       'skip-b
		       termb))

		  ;; variables second
		  ((and (variable-p term1) (not (variable-p term2)))
		   terma)

		  ((and (variable-p term2) (not (variable-p term1)))
		   termb)

		  ;; all else lexicographical.
		  ((eq term1 lower-term) 
		   terma)

		  (t termb))))))))

;;
;; produces simplified term from list of simplified multiplicands.
;;

(defun simplify-multiplicands (multiplicands)
  (if (null (cdr multiplicands))
      (car multiplicands)
      (symbolic-multiply (car multiplicands) 
			 (simplify-multiplicands (cdr multiplicands)))))


(defun symbolic-multiply (multiplicanda multiplicandb)

  (cond

    ;; (a1 + ... + an) * b  ->  (a1b + ... anb)
    ((add-term-p multiplicanda)
     (simplify-addends (mapcar #'(lambda (addend) 
				   (symbolic-multiply addend multiplicandb))
			       (addends-of-simplified-term multiplicanda))))

    ;; a * (b1 + ... + bn)  ->  (ab1 + ... abn)
    ((add-term-p multiplicandb)
     (simplify-addends (mapcar #'(lambda (addend)
				   (symbolic-multiply multiplicanda addend))
			       (addends-of-simplified-term multiplicandb))))

    ;; i * b  ->  ib
    ((constant-simplified-term-p multiplicanda)
     (simplified-term (* (coefficient-of-simplified-term multiplicanda)
			 (coefficient-of-simplified-term multiplicandb))
		      (term-of-simplified-term multiplicandb)))

    ;; a * i  ->  ia
    ((constant-simplified-term-p multiplicandb)
     (simplified-term (* (coefficient-of-simplified-term multiplicanda)
			 (coefficient-of-simplified-term multiplicandb))
		      (term-of-simplified-term multiplicanda)))


    ;; (a1 * ... * an) * (b1 * ... * bm)  ->  (c1 * ... * cm+n) : ci lexically< cj for i<j.
    (t (simplified-term 
	 (* (coefficient-of-simplified-term multiplicanda)
	    (coefficient-of-simplified-term multiplicandb))
	 (multiplicands-to-term-of-simplified-term
	   (merge-lists-by-comparator
	     (multiplicands-of-term-of-simplified-term (term-of-simplified-term multiplicanda))
	     (multiplicands-of-term-of-simplified-term (term-of-simplified-term multiplicandb))
	     #'(lambda (terma termb)
		 (let ((lower-term (lexicographically-compare-terms terma termb)))
		   (cond
		     ((eql t lower-term) t)
		     ((and (variable-p terma) (variable-p termb))
		      lower-term)
		     ((variable-p terma) terma)
		     ((variable-p termb) termb)
		     (t lower-term)))))))) ))


;; comparator compares two elements and returns t, nil, one of those passed in, or a new one.
(defun merge-lists-by-comparator (lista listb comparator)

  (labels 
    ((merge-lists (lista listb)
       (cond

	 ((null lista) listb)

	 ((null listb) lista)

	 (t (let ((comparison (funcall comparator (car lista) (car listb))))
	      (cond
		
		((eq comparison t) (cons (car lista)
					 (cons (car listb)
					       (merge-lists (cdr lista) (cdr listb)))))

		((null comparison)
		 (merge-lists (cdr lista) (cdr listb)))

		((eq comparison 'skip-a)
		 (merge-lists (cdr lista) listb))

		((eq comparison 'skip-b)
		 (merge-lists lista (cdr listb)))

		((eq comparison (car lista))
		 (cons comparison
		       (merge-lists (cdr lista) listb)))

		((eq comparison (car listb))
		 (cons comparison
		       (merge-lists lista (cdr listb))))
		
		(t (cons comparison (merge-lists (cdr lista) (cdr listb))) )))))))

    (merge-lists lista listb)))





;;;;
;;;;  simplify
;;;;

(defunml (|arith_simplify_complete_term| (term))
    (term -> term)
  (simplify term))

(defunml (|arith_simplify_term| (term))
    (term -> term)
  (simplify-top term))

