%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*****************************************************************************
Building Objects for Recursive Definitions.
*****************************************************************************

Usage:

add_rec_def 'id{ps}(s1;...;sn) y1 ... ym'  'g'

Where:

  o id{ps}(...) is the new abstraction to be created.
  o s1,...,sn are each of form vsi.xi[vsi] where vsi is possibly empty.
  o x1,...,xn y1 ... ym are variables.  m,n >= 0
  o g contains free occurrences of xs and ys, and occurrences of id{ps}(...)

Each xi is either a constant parameter of the recursion, or an argument
on which the def is recursing. An argument xi must have an empty vsi.
The list of argument xi is xa1..xaq.

Applications of add_rec_def should always be installed in an ML object with 
name id_ml.

If they don't already exist, add_rec_def adds three new objects to 
the library.

(disp) id_df:      id{ps}(x1;...;xn)==id{ps}(x1;...;xn)
(ab)   id:         id{ps}(x1;...;xn)==
                        Y (\idv xa1 ... xaq y1 ... ym.
                               g[idv ta1 ... taq/id{ps}(t1;...;tn)])
                        xa1 ... xaq
(ml)   id_ml:      id{ps}(x1;...;xn) y1 ... ym   ==>>  g  
(thm)  id_wf:      _

g[idv ta1 ... taq/id{ps}(t1;...;tn)] means for all t1;...;tn 
occurrences of id{ps}(t1;...;tn) are replaced by idv ta1 ... taq
where the correspondence between the taj and the ti is the same as
that between the xaj and the xi.

Whether or not the objects already exists, add_rec_def initializes
caches so that the RecUnfold and RecFold tactics recognize the 
recursive definition.

Since (ab) is not really of interest, it is usually hidden.
]%


% untyped definition %
let add_def_content add_wf_lemma_p lhs rhs = 
  let id = tok_to_string (opid_of_term lhs)
  in let dfname = id J "_df"
  in let abname = id 
  in let wfname = id J "_wf"
  in let floating_vars = diff (free_vars rhs) (free_vars lhs) 
  in if not null floating_vars then
    failwith 
      (`add_def: found vars on rhs but not lhs:`
       ^ implode (map (\x.` ` ^ var_to_tok x) floating_vars)
      )
  else
        [ create_disp_content_for_new_def lhs dfname abname
        ; create_ab_content_for_new_def lhs rhs abname
        ; create_thm_obj_data ivoid_term ivoid_term wfname
        ]
;;


%[
*****************************************************************************
Processing Typed Definitions
*****************************************************************************
]%

let proc_typed_def tdef = 
  let xAs,Bs,lrT = dest_simple_formula tdef in
  let xs = map fst xAs in
  let T,tl,tr = dest_equal lrT in
  let floating_vars = diff (free_vars tr) (free_vars tl) in
  if not null floating_vars then
  ( let var_str = concatenate_strings
           (map (\x." " J (tok_to_string (var_to_tok x))) floating_vars)
    in
      failwith 
        (`proc_typed_def: found vars on rhs but not lhs:`
         ^ string_to_tok var_str
        )
  )
  else
  let ablterm = so_ize_so_aps_of_vars xs tl in
  let abrterm = so_ize_so_aps_of_vars xs tr in
  let wf_goal = mk_simple_formula xAs Bs (mk_member_term T tl) 
  in
    ablterm,abrterm,wf_goal
;;


%[ 
-----------------------------------------------------------------------------
Object / Conversion Creation Function 
-----------------------------------------------------------------------------
This should always be installed in an ML object, since it updates
conversion caches as well.
]%

% Checks occurrences of lhs in body. Returns a list of numbers of those
subterms on which recursion is done. (i.e. those that are not simply 
parameters).
%

let analyze_rec_def_body lhs_tm body = 
  let (id,ps),bts = dest_term lhs_tm
  in let rhs_tms = 
    map fst (find_subterms_with_addrs (\vs,t.is_term id t) body)
  in
  if null rhs_tms then 
    failwith `add_rec_def: definition not recursive`
  else
  % bts_status is list of booleans, 1 per btm. 
    true means that exists a corresponding btm in some rhs_tm 
    which is different.
  %
  let bts_status = 
    accumulate
      (\status t'.
          let (),bts' = dest_term t'
          in let diffs = map2 (\x y. not (alpha_equal_bterms x y)) bts bts'
          in 
            map2 (\x y.x or y) status diffs
      )
      (map (\t.false) bts)        % initial status %
      rhs_tms
  in
  % check no binding in non parameter position %
  if 
    exists (\st,(vs,t).st & not null vs) (zip bts_status bts)
  then
    failwith `add_rec_def: cannot handle binding var in rec argument`
  else
    mapfilter (\st,i.if st then i else fail) 
              (zip bts_status (upto 1 (length bts_status)))      
;;

let select_subterm_subseq subseq t = 
  map (\i.get_addressed_subterm [i] t) subseq
;;

%
let status_of_object id = `COMPLETE`
;;
%

%
let ref_add_recdef_additions (uf,eta) index edges = 
  ref_add_RecUnfoldFold_additions uf index edges;
  ref_add_RecEta_additions eta index edges
 ; ()
;;

let build_recdef_additions rddata =
 let (opid, [idp]), [[],lhs; [], rhs] = destruct_term rddata in
 let id = (destruct_token_parameter idp) in

  let UnfoldC,FoldC = RecUnrollRollC id lhs rhs 
     ? failwith (`add_rec_def: could not add conversions for: ` ^ id)
   in
     ([id, UnfoldC, FoldC],  (build_RecEta_convs lhs))
;;
%

let ref_add_recdef_foldunfold_additions uf index edges = 
  ref_add_RecUnfoldFold_additions uf index edges
 ; ()
;;
let ref_add_recdef_receta_additions eta index edges = 
  ref_add_RecEta_additions eta index edges
 ; ()
;;

%
;;;;	
;;;;	FoldUnfold/RecEta updates : one or two updates to convs may
;;;;	be produced. If two then the second must run in the scope of the first!
;;;;	  - produce two update objects. Second uses first.  
;;;;	  
;;;;	
%  
let build_recdef_unfoldfold_additions rddata =
 let (opid, [idp]), [[],lhs; [], rhs] = destruct_term rddata in
 let id = (destruct_token_parameter idp) in

  [(id, (RecUnrollRollC id lhs rhs))]
  ? failwith (`add_rec_def: could not add conversions for: ` ^ id)
;;

%For compatibility with old code: %
let build_recdef_foldunfold_additions =  build_recdef_unfoldfold_additions;;

let build_recdef_receta_additions rddata =
 let op, (([],lhs) . r) = destruct_term rddata in (build_RecEta_convs lhs)
;;

let recdef_update_terms opid lhs rhs =
 ((tok_to_string opid) J "_RecUnfoldFold_conv",
  mk_ref_update_RecUnfoldFold_term "build_recdef_unfoldfold_additions"
     (make_term (`args`,[make_token_parameter opid]) [[], lhs; [],rhs]))
 . 
 if can_build_RecEta_convs lhs
   then [(tok_to_string opid) J "_RecEta_conv",
          mk_ref_update_RecEta_conv_term "build_recdef_receta_additions"
             (make_term (`args`,[]) [[], lhs])]
   else []
;;

let create_recdef_updates id lhs rhs =
 let aux (n,t) = 
  create_ml_obj_data 
     (include_properties_term [`reference environment additions`, itoken_term `update`]
        t)
     n in
  map aux (recdef_update_terms id lhs rhs)
;;
     
%
let add_rec_def_content oacc lhs rhs= 
  let ab_lhs.y_tms = dest_iterated_apply lhs in
  let (id,ps),bts = dest_term ab_lhs in
  let id_str = tok_to_string id
  in 
  
    let arg_indices = analyze_rec_def_body ab_lhs rhs
    in let xas = map dest_var (select_subterm_subseq arg_indices ab_lhs)
      ? failwith `add_rec_def: expecting active lhs subterms to be vars`
    in let ys = map dest_var y_tms
      ? failwith `add_rec_def: expecting all var args`
    in let idv = tok_to_var id  
    in let g' = 
      higher_map
        (\t'.if is_term id t' then
               mk_iterated_apply 
                 (mk_var_term idv . select_subterm_subseq arg_indices t')
             else
               fail
        )
        rhs
    in let ab_rhs = 
      mk_iterated_apply 
        (mk_simple_term `ycomb` []
         . mk_iterated_lambda (idv.xas @ ys) g'
         . map mk_var_term xas)

    in let dfname = id_str J "_df"
    in let abname = id_str 
    in
      (if (isr oacc) then
       ((outr oacc)
        [ create_disp_content_for_new_def ab_lhs dfname abname
        ; create_ab_content_for_new_def ab_lhs ab_rhs abname
        ; create_recdef_addition id lhs rhs
        ]))
;;
%

let add_rec_def_content_aux f lhs rhs = 
  let ab_lhs.y_tms = dest_iterated_apply lhs in
  let (id,ps),bts = dest_term ab_lhs in
  
    let arg_indices = analyze_rec_def_body ab_lhs rhs
    in let xas = map dest_var (select_subterm_subseq arg_indices ab_lhs)
      ? failwith `add_rec_def: expecting active lhs subterms to be vars`
    in let ys = map dest_var y_tms
      ? failwith `add_rec_def: expecting all var args`
    in let idv = tok_to_var id  
    in let g' = 
      higher_map
        (\t'.if is_term id t' then
               mk_iterated_apply 
                 (mk_var_term idv . select_subterm_subseq arg_indices t')
             else
               fail
        )
        rhs
    in let ab_rhs = 
      mk_iterated_apply 
        (mk_simple_term `ycomb` []
         . mk_iterated_lambda (idv.xas @ ys) g'
         . map mk_var_term xas)

     in (f id ab_lhs ab_rhs)
;;						   

let rec_def_content createp lhs rhs =
 % if createp is false then just run for side_effects
   at this point seems like there should be no side effects.
 %
 add_rec_def_content_aux
 (\id ab_lhs ab_rhs.

  let id_str = tok_to_string id

    in let dfname = id_str J "_df"
    in let abname = id_str 
    in
      (if createp then
        ( create_disp_content_for_new_def ab_lhs dfname abname
        . create_ab_content_for_new_def ab_lhs ab_rhs abname
        . create_recdef_updates id lhs rhs
        ) 
      else []))
  lhs rhs
;;

let add_rec_def_content oacc lhs rhs = 
 let createp = isr oacc in
 let content = rec_def_content createp lhs rhs in
  if createp then ((outr oacc) content)
;;

%let add_rec_def_content oacc lhs rhs = 
 add_rec_def_content_aux
 (\id ab_lhs ab_rhs.

  let id_str = tok_to_string id

    in let dfname = id_str J "_df"
    in let abname = id_str 
    in
      (if (isr oacc) then
       ((outr oacc)
        [ create_disp_content_for_new_def ab_lhs dfname abname
        ; create_ab_content_for_new_def ab_lhs ab_rhs abname
        ; create_recdef_addition id lhs rhs
        ])))
  lhs rhs
;;
%

let add_rec_def_addition lhs rhs dir place = 
  let ab_lhs.y_tms = dest_iterated_apply lhs in
  let (id,ps),bts = dest_term ab_lhs in

  create_lib_objects_at_wrechain dir place
   (map odt (create_recdef_updates id lhs rhs))
;;

%
Function for invoking in ML top loop. 
1. Create and check ML object with add_rec_def invocation. 
2. set goal of wf lemma according to typed def. 

Fancier one takes care of trying to run tactic as well.
%

%
Fix up wf_goal to take account of possible currying of definition.
%

let cleanup_dep_types tm = 
  let f t = 
    if is_terms ``function product`` t then
      let op,[[],A;[x],B] = dest_std_term t in
      if not (member x (free_vars B)) then 
        mk_std_term op [[],A;[null_var],B]
      else
        fail
    else
      fail
  in
    sweep_up_map f tm
;;

let fix_rec_wf_goal goal = 
  let xAs,Bs,t_in_T = dest_simple_formula goal in
  let T,t = dest_member t_in_T in
  let ltm.arg_tms = dest_iterated_apply t in
  let argvs = map dest_var arg_tms 
    ? failwith `rec_def_prep: curried args should be all vars`
  in
  let xAs' = remove_alist_entries xAs argvs in
  let T' = mk_iterated_function (map (\v.v,apply_alist xAs v) argvs) T in
  let T'' = cleanup_dep_types T' 
  in
    mk_simple_formula xAs' Bs (mk_member_term T'' ltm)
;;

let mk_tactic_wrapper wrap_id body = 
  mk_text_seq `!ml_text_cons` [mk_simple_term wrap_id [body]] 
;;


let mk_rec_def_wf_tac rec_ap = 
  let recab.args = dest_iterated_apply rec_ap in
  let opid_str = tok_to_string (opid_of_term recab) in
  let s1 = 
    if length args > 0 then
      "RepD THENM RWH (RecEtaExpC `"
      J opid_str
      J "`) 0 THENM RepeatMFor "  
      J int_to_string (length args)
      J " MemCD"
    else
      "RepD"
  in
    mk_text_seq `!ml_text_cons` [mk_simple_term `aux_auto` [mk_text_term s1]] 
;;


let rec_def_prep typed_def = 
  let  deflterm,defrterm,wf_goal = proc_typed_def typed_def in
  let wf_goal' = fix_rec_wf_goal wf_goal in
  let wftac = mk_rec_def_wf_tac deflterm in
  let opid = opid_of_term (hd (dest_iterated_apply deflterm)) in
  let addrecdef_ml_tm = 
    mk_text_seq `!ml_text_cons` 
      [mk_simple_term `addrecdef` [deflterm;defrterm]] 
  in
    opid,addrecdef_ml_tm,wf_goal',wftac
;;


%
;;;;	
;;;;	add_rec_def adds disp and abs and wf thm with no goal or prf.
;;;;	
;;;;	recdef adds ml object calling add_rec_def checks it and then
;;;;	adds proof to wf object.
;;;;	
;;;;	To emulate this, we will add abs, disp, thm and ml from recdef, 
;;;;	the ml check should no-op since the objects will be present.
;;;;	
%
 
let recdef_content oacc ptyped_def = 

  let deflterm,defrterm,wf_goal = ptyped_def in
  let opid = opid_of_term (hd (dest_iterated_apply deflterm)) in

  let wf_goal' = fix_rec_wf_goal wf_goal in
  let wftac = mk_rec_def_wf_tac deflterm in
  let addrecdef_ml_tm = 
    mk_text_seq `!ml_text_cons` 
      [mk_simple_term `addrecdef` [deflterm;defrterm]] 
  in


  add_rec_def_content oacc deflterm defrterm;

  if (isr oacc) then
     ((outr oacc)
	[ create_ml_obj_data addrecdef_ml_tm (tok_to_string opid J "_ml")
	; create_thm_obj_data wf_goal' wftac (tok_to_string opid J "_wf")
	])

  ; ()
;;


%[
*****************************************************************************
Adding definitions with wf goals
*****************************************************************************

Usage:

add_def2  'All xs:As Bs =>  lhs = rhs in T' "<position>" ;;

lhs should be a new term <opid> (with so_applies used when necessary).


Creates

1. <opid>_df    lhs df
2. <opid>       lhs == rhs
3. <opid>_wf    All xs:As Bs =>  lhs in T' 

and attempts to prove 3 by unfolding <opid> then using Auto.
]%


let add_def2_prep def = 
  let  ablterm,abrterm,wf_goal = proc_typed_def def in
  let opid_str = tok_to_string (opid_of_term ablterm) in
  let s1 = "Unfold `" J opid_str J "` 0" in
  let tactic_term = 
      mk_text_seq `!ml_text_cons` [mk_simple_term `auto` [mk_text_term s1]] 
  in
    ablterm,abrterm,wf_goal,tactic_term
;;

let add_def2_content_aux defstuff = 
  let lhs,rhs,goal,Tac = defstuff in
  let id = tok_to_string (opid_of_term lhs) in 
  let dfname = id J "_df" in 
  let abname = id in 
  let wfname = id J "_wf" in

    [ create_disp_content_for_new_def lhs dfname abname
    ; create_ab_content_for_new_def lhs rhs abname
    ; create_thm_obj_data goal Tac wfname
    ]
;;

let add_def2_content oacc defstuff = 

  if (isr oacc) then
     ((outr oacc)
      (add_def2_content_aux defstuff))
  ;()
     % 12/20/00
     (((outr oacc)
      [ create_disp_content_for_new_def lhs dfname abname
      ; create_ab_content_for_new_def lhs rhs abname
      ])
     ;((outr oacc)
      [ create_thm_obj_data goal Tac wfname
      ]))
     %
;;


% creates inclusion ml object and properties lemma as well %

% assumes rhs of form {y:S|P_y}

Adds two objects:

<lhs-opid>_ml_inc:

  add_set_inclusion_info
   `<lhs-opid>` `<S-opid>` AbSetDForInc Auto;;

<lhs-opid>_properties:

All xAs Bs => y:S. P_y

Assumes that P_y is squash stable
%

let ProvePropertiesLemma p =
  (UnivCD 
   THENM (Repeat (UnfoldTopAb (-1)) THEN SetHD (-1))
   THEN Auto
  ) p
;;  


let set_inc_build sidata =
  let ((opid, [ida; idb]), []) = destruct_term sidata in
    [ ( destruct_token_parameter ida
      , destruct_token_parameter idb
      , AbSetDForInc
      , Auto)
    ]
;;

% needs to be updated to do updates.
  however apparently little used so FTTB ignore %
let set_inc_additions_term ida idb =
 failwith `set_inc_additions_term needs recoding see rec-def.ml`
%  build_and_apply_additions_term "ref_add_set_inc_additions" "set_inc_build"
   ( make_term (`args`, [ make_token_parameter ida
			; make_token_parameter idb])
	   [])
%;;


let setdef_prep def = 
  let xAs,Bs,lrT = dest_simple_formula def in
  let xs = map fst xAs in
  let T,tl,tr = dest_equal lrT in
  let y,S,Py = dest_set tr in
  let properties_goal = mk_simple_formula xAs Bs (mk_all_term y tl Py) in
  let properties_tac = mk_text_term "Try ProvePropertiesLemma" in
    properties_goal,properties_tac,
     (include_properties_term [`reference environment additions`, ibool_term true]
       (set_inc_additions_term (opid_of_term tl) (opid_of_term S)))
;;


let setdef_content oacc deftm defstuff = 
  let lhs,rhs,wfgoal,wftac = defstuff in
  let propgoal,proptac,ml_inc_fun = setdef_prep deftm in
  let id = tok_to_string (opid_of_term lhs) in 
  let dfname = id J "_df" in 
  let abname = id in 
  let wfname = id J "_wf" in
  let incname = id J "_ml_inc" in
  let propname = id J "_properties"
  in

  if (isr oacc) then
     ((outr oacc)
      [ create_disp_content_for_new_def lhs dfname abname
      ; create_ab_content_for_new_def lhs rhs abname 
      ; create_thm_obj_data wfgoal wftac wfname
      ; create_ml_obj_data ml_inc_fun incname
      ; create_thm_obj_data propgoal proptac propname
      ])
;;



letrec flatten_compound_pred Q = 
  if (has_ab_attr_t `COMPOUND` Q) then
    flatten
    ( map flatten_compound_pred
      (dest_conjunction (unfold_ab Q))
    )
  if is_term `and` Q then
    flatten
    ( map flatten_compound_pred
      (dest_conjunction Q)
    )
  else
    [Q]
;;

let ProvePropertyLemma id n = 
  Try (
  SeqOnM 
   [RepeatMFor n (D 0)
   ;AddProperties (-1)
   ;RepeatAndHD (-1)
   ;FlattenCompounds
   ;All (RW (AUnfoldsC ["basic"]))
   ;Trivial
   ]
  )
;;


let setdef_pred_prep def = 
  let xAs,Bs,lrT = dest_simple_formula def in
  let xs = map fst xAs in
  let T,tl,tr = dest_equal lrT in
  let y,S,Py = dest_set tr in
  let basic_preds = number (flatten_compound_pred Py) in
  let n = length xAs + length Bs + 1 in
  let nstr = int_to_string n in
  let mk_tactic id = 
      mk_text_seq `!ml_text_cons` 
        [mk_simple_term `auto` 
           [mk_text_term 
              ("ProvePropertyLemma `" J tok_to_string id J "` " J nstr)]] 
  in
  let condition_pred t = apply_conv (AUnfoldsC ["basic"]) t in
  let mk_pred_goal (i,P) = 
  ( if has_ab_attr_t `BASIC` P then
      (tok_to_string (opid_of_term P))
      , mk_simple_formula xAs Bs (mk_all_term y tl (unfold_ab P))
      , mk_tactic (opid_of_term P) 
    else
      int_to_string i
      , mk_simple_formula xAs Bs (mk_all_term y tl (condition_pred P))
      , mk_tactic `none`
  )
  in
    map mk_pred_goal basic_preds
;;


% assumes that there already exists an _properties lemma %

%
let add_seperate_property_lemmas opid = 
  let nam_t_T_trips = setdef_pred_prep (mk_typed_def opid) in  
  let id_str = tok_to_string opid J "_" in
  let full_nams = map (\n,(),().id_str J n) nam_t_T_trips in
  let pos_nams = remove_last ((id_str J "properties") . full_nams) in

  let add_lem (pos,nam,g,Tac) = add_new_thm_and_run nam g Tac ("+" J pos) 
  in
    do map add_lem (zip pos_nams (zip full_nams (map snd nam_t_T_trips)))
;;
%  


let add_inc_content oacc deftm defstuff = 
  let lhs,rhs,wfgoal,wftac = defstuff in
  let propgoal,proptac,ml_inc_fun = setdef_prep deftm in
  let id = tok_to_string (opid_of_term lhs) in 
  let wfname = id J "_wf" in
  let incname = id J "_ml_inc" in
  let propname = id J "_properties"
  in

  if (isr oacc) then
     ((outr oacc)
      [ create_ml_obj_data ml_inc_fun incname
      ; create_thm_obj_data propgoal proptac propname
      ])

 ; ()
;;


% Create a guarded version of a theorem %

let add_guarded_theorem_content oacc name ext addr =
  let thm = raw_main_goal_of_theorem name in
  let thm',(),() = AddrC addr GuardC null_env thm 
  in

  if (isr oacc) then
    ((outr oacc)
     [create_thm_obj_data thm'
      (itext_term
        (concatenate_strings
          ["Unfold `guard` 0 THEN Lemma "
          ;"`"
          ;tok_to_string name
          ;"`"
          ]))
      ("name" J (tok_to_string ext))
      ])
 ; ()
;;

