/***************************************************************************
                          cgraph.cpp  -  description
                             -------------------
    begin                : Thu Jul 24 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cgraph.h"
#include "cprogram.h"
#include "cnode.h"
#include "crule.h"
#include "coperator.h"
#include "caggregation.h"
#include "print.h"
#include "ctodo.h"

#include <unistd.h>
#include <iostream>

namespace NS_NOMORE {

template <class T>
struct NodeSelector {
  typedef typename std::vector<T*>::size_type index_size_t;
  typedef typename std::vector<T*>::iterator iterator;
  typedef std::vector<T*> NodeList;
};

template <class T>
bool InsertTypedNode(T* node, std::vector<T*>& index_cont, std::vector<T*>& cont) {
  typename NodeSelector<T>::index_size_t index = node->GetId() - 1;       
  
  if(index >= index_cont.size()) {
    index_cont.resize(index + 1, static_cast<T*>(0));
  }
  
  if(index_cont[index] == 0) {
    typename NodeSelector<T>::iterator insPos = lower_bound(cont.begin(), cont.end(), node,
      CLessIdDeref<CNode>());

//    assert(insPos == cont.end() || (*insPos)->GetId() != node->GetId());
    cont.insert(insPos, node);
    index_cont[index] = node;
  } else {
    // node already there
    return true;
  }       
        
  return true;
}

/***************************************************************************
  class CGraph
****************************************************************************/
CGraph::CGraph() : _uncolored_nodes(0), _lookahead(false) {

  _panswer_set = new TAtomSet;

  _pcolornode = new CColorNode();

  _last_bodynode_id = 0;
  _last_headnode_id = 0;  
  _can_be_prop_by_supported_node = false;
  
  CREATEOBJECT("CGraph")
  
}

CGraph::CGraph(CProgram* prg, CTodoLists *todo) : _uncolored_nodes(0), _lookahead(false) {

  CHECK_POINTER("CGraph::CGraph(prg)" , prg);
  CHECK_POINTER("CGraph::CGraph(todo)", todo);

  _panswer_set = new TAtomSet;

  _pcolornode = new CColorNode();
  _last_bodynode_id = 0;
  _last_headnode_id = 0;  
  _can_be_prop_by_supported_node = false;

  CREATEOBJECT("CGraph")

  GenGraph(prg, todo);
  
  this->_can_be_propagated_bodynodes_plus.resize(_last_bodynode_id+1, 0);  
  this->_can_be_propagated_bodynodes_minus.resize(_last_bodynode_id+1, 0);  
  this->_can_be_propagated_headnodes.resize(_last_headnode_id+1, 0);
}

void CGraph::InsertPossibleSupportedBodies(TnBodySet bodies, CProgram* prg, CTodoLists *todo) {
  TnBodySetIterator currentBody = bodies.begin();
  TnBodySetIterator endBody = bodies.end();
  TnBodySetIterator save;
  
  TIdSet atoms;
  CStack notNeeded;
  
  bool pendingStartOver = false;
  
  // The OR-Part is only considered if currentBody is at the end of the bodies.
  // It not only checks if a start over is necessary but also resets 
  // pendingStartOver and currentBody in that case. 
  // This could better be done in the last else-clause but 
  // measurements proved this obfuscation to be superior since it leads to fewer comparisons
  while(currentBody != endBody || 
      (pendingStartOver && (currentBody = bodies.begin()) != endBody && !(pendingStartOver = false))) {
    save = currentBody;
    
    TIdSetIterator bkp_i = atoms.begin();
    TIdSetIterator bkp_e = atoms.end();
    
    while ((*currentBody)->Supported(atoms)) {
    
      // process all bodies supported by atoms
      CBodyNode* bodyNode = InsertBodyNode(*currentBody);
      const CBody* b = bodyNode->GetBody();
      const TRuleVector& rules = b->GetRules();
      for(TRuleVectorIterator ruleIt = rules.begin(); ruleIt != rules.end(); ++ruleIt) {
        TId id = (*ruleIt)->GetHead()->GetId();
        atoms.insert(id);
        CAtom* head = prg->GetAtom(id);
        CHeadNode* headNode = InsertHeadNode(head, &notNeeded, todo);
        bodyNode->InsertSuccessor(headNode);
      }
                        
      if(++currentBody == endBody) {
        break;
      }
    }
                
    // we found some supported bodies.
    // Once a body is supported it will always be supported, therefore
    // we don't need to bother with those supported bodies again. 
    if(currentBody != save) {
      bodies.erase(save, currentBody);
      
      // Since the bodies are ordered we may continue forward scanning 
      // as long as following bodies don't have more B+ atoms than there are 
      // elements in atoms.
      // Nevertheless it will be necessary to start over once we reache
      // the end of the bodies, because previously unsupported bodies
      // may now be supported.
      // Performance measurements proved this 
      // "scan-forward-as-long-as-possible-then-restart"-Tactic to be
      // superior to the old "if-something-changed-start-over"-approach
      if(currentBody != endBody && (*currentBody)->GetPosAtoms().size() <= atoms.size()) {
        pendingStartOver = true;
      } else {
        // If on the otherhand currentBody has more B+ atoms than there are
        // elements in atoms, no following body can be supported by atoms.
        // Forward scanning no longer makes sense, thus start over.
        currentBody = bodies.begin();
      }
    } else 
    /*if((*currentBody)->GetPosAtomSize() > atoms.size()) {
      cout << "too many atoms"<<endl;
      if (currentBody == bodies.begin())
        return;
      currentBody = bodies.begin();
      pendingStartOver = false;
    } else */{
      ++currentBody;
    }
  }
}

void CGraph::GenGraph(CProgram* prg, CTodoLists *todo) {
  CHECK_POINTER("CGraph::GenGraph(prg)", prg);
  CHECK_POINTER("CGraph::GenGraph(todo)", todo);

  InsertPossibleSupportedBodies(prg->GetBodies(), prg, todo);

  TBodyNodeVectorIterator bodyIt = _body_nodes.begin();
  TBodyNodeVectorIterator bodyEnd = _body_nodes.end();

  for (; bodyIt != bodyEnd; ++bodyIt) {
    const TIdSet& positives = (*bodyIt)->GetBody()->GetPosAtoms();
    TIdSetIterator atomEnd = positives.end();
    for (TIdSetIterator atomIt = positives.begin(); atomIt != atomEnd; ++atomIt) {
      if (CHeadNode *h = GetHeadNode(*atomIt)) {
        h->InsertZeroSuccessor(*bodyIt);
      } else {
        cerr << "pos body atom of rule does not exist as head " << *atomIt << endl;
        exit(2);
      }
    }
                
    const TIdSet& negatives = static_cast<CBodyNode*>(*bodyIt)->GetBody()->GetNegAtoms();
    atomEnd = negatives.end();
    for (TIdSetIterator atomIt = negatives.begin(); atomIt != atomEnd; ++atomIt) {
      if (CHeadNode * h = GetHeadNode(*atomIt)) {
        h->InsertOneSuccessor(*bodyIt);
      }
    }
  }
 
  // establish invariant namely that our Node-Lists are ordered.
  sort(_head_nodes.begin(), _head_nodes.end(), CLessIdDeref<CNode>());
  sort(_body_nodes.begin(), _body_nodes.end(), CLessIdDeref<CNode>());
}

CGraph::~CGraph() {
  for_each(_head_nodes.begin(), _head_nodes.end(), DerefDelete());
  for_each(_body_nodes.begin(), _body_nodes.end(), DerefDelete());

  _head_node_index.clear();
  _head_nodes.clear();
  _body_node_index.clear();
  _body_nodes.clear();
    
  delete _panswer_set;
  delete _pcolornode;
  
  DELETEOBJECT("CGraph");
}

bool CGraph::operator==(CGraph& graph) {
  return graph._head_nodes == _head_nodes && graph._body_nodes == _body_nodes;
}

bool CGraph::operator!=(CGraph& graph) {
  return !(operator==(graph));
}

bool CGraph::BuildAnswerSet() {
  if (!_panswer_set->empty())
    _panswer_set->clear();

  THeadNodeVectorIterator it = _head_nodes.begin();
  THeadNodeVectorIterator end = _head_nodes.end();

  for(; it!=end; it++) {
    CHeadNode* node = *it;
    if(node->GetColor() == color_plus) {
      CAtom *atom = node->GetHead();
      (*_panswer_set)[atom->GetId()] = atom;
    }
  }
  return true;
}

void CGraph::GenHeuristics(CHeuristic* heu) {
  CHECK_POINTER("CGraph::GenHeuristics(heu)", heu);

  TBodyNodeVector *nodes = GetBodyNodes();
  TBodyNodeVectorIterator ite;
  for(ite=nodes->begin();ite!=nodes->end();ite++) {
    CNode *node = *ite;
    // create new heuristic object
    CHeuristic* nodeheu = heu->GenerateInstance(node);
    // set heuristic
    node->SetHeuristic(nodeheu);
  }

  THeadNodeVector *hnodes = GetHeadNodes();
  THeadNodeVectorIterator hite;
  for(hite=hnodes->begin();hite!=hnodes->end();hite++) {
    CNode *node = *hite;
    // create new heuristic object
    CHeuristic* nodeheu = heu->GenerateInstance(node);
    // set heuristic
    node->SetHeuristic(nodeheu);
  }
}

bool CGraph::InsertNode(CNode* node) {
  CHECK_POINTER("CGraph::InsertNode(node)", node);

  switch(node->GetType()) {

    case type_cheadnode:
      InsertTypedNode<CHeadNode>((CHeadNode*)node, this->_head_node_index, this->_head_nodes);
      break;

    case type_cbodynode:
      InsertTypedNode<CBodyNode>((CBodyNode*)node, this->_body_node_index, this->_body_nodes);
      break;

    default:
      break;
  }

  if(node->GetColor() == color_none || node->GetColor() == color_weak_plus)
    _uncolored_nodes++;

  return true;
}

/*TPNodeList CGraph::GetNodes(TColorEnum color, TNodeType node_type) {

  TPNodeList list;
  list.reset(new TNodeList());

  TNodeSet::iterator ite;
  TNodeSet *nodes;

  switch(node_type) {

    case type_cheadnode :
      nodes = _phead_nodes;
      break;

    case type_cbodynode :
      nodes = _pbody_nodes;
      break;

    default :
      return list;
      
  }

  for(ite=nodes->begin();ite!=nodes->end();ite++) {
    
    CNode* node = ite->second;

    // is color is preferred put into list
    if(node->GetColor() & color)
      list.get()->push_back(node);
      
  }

  return list;
  
}*/

TRetOperator CGraph::ColorNodes(TColorEnum color, TNodeType node_type, TColor to_color, CStack *stack,
              CTodoLists *todo, CStatusInformation* status, bool secure) {
  CHECK_POINTER("CGraph::ColorNodes(stack)", stack);
  CHECK_POINTER("CGraph::ColorNodes(todo)" , todo);

  THeadNodeVectorIterator hite = _head_nodes.begin();
  THeadNodeVectorIterator hend = _head_nodes.end();
  TBodyNodeVectorIterator bite = _body_nodes.begin();
  TBodyNodeVectorIterator bend = _body_nodes.end();
  
  TRetOperator value = return_unchanged;
  switch(node_type) {
    case type_cheadnode :
      for(hite=_head_nodes.begin();hite!=_head_nodes.end();hite++) {
        CHeadNode* node = *hite;
    
        TColor old_color = node->GetColor();
        // is color is preferred put into list
        if(old_color & color) {
          if(!ColorNode(node, to_color, stack, false, color_none, todo, status, secure))
            return return_colorerror;
          value = return_changed;
        }
      }
      break;

    case type_cbodynode :
      for(bite=_body_nodes.begin();bite!=_body_nodes.end();bite++) {
        CBodyNode* node = *bite;
    
        TColor old_color = node->GetColor();
        // is color is preferred put into list
        if(old_color & color) {
          if(!ColorNode(node, to_color, stack, false, color_none, todo, status, secure))
            return return_colorerror;
          value = return_changed;
        }
      }
      break;

    default :
      break;
  }

  return value;  
}

bool CGraph::ColorNode(CNode *node, TColor color, CStack *stack, bool choice, TColor choice_color,
      CTodoLists *todo, CStatusInformation* status, bool secure) {
  CHECK_POINTER("CGraph::ColorNode(node)" , node);
  CHECK_POINTER("CGraph::ColorNode(stack)", stack);
  CHECK_POINTER("CGraph::ColorNode(todo)" , todo);

  TColor old_color = node->GetColor();

  if(color == old_color)
    return true;

  if(!secure ||
         // test is deactivated
     old_color == color_none ||
         // old color is uncolored
     old_color == color_ignore ||
         // old color is ignored
     color == color_ignore ||
         // new color is ignored
     (old_color == color_weak_plus && color == color_plus)) {
         // new color is plus and old color was weak-plus

    if(!node->ColorNode(color, stack, choice, choice_color))
      return false;
            
    if(status != NULL && node->GetType() == type_cheadnode) {
      status->ModifyAllColored(1);
      if(((CHeadNode*)node)->GetOneSuccessors().size() != 0 && (old_color == color_none)) {
        status->ModifyHardColored(1);
      }
    }

    if((old_color == color_none || old_color == color_weak_plus) &&
       (color == color_plus || color == color_minus || color == color_ignore)) {
      _uncolored_nodes--;
    }

    if(color == color_minus)
      todo->PushMinus(node);

    if(node->GetType() == type_cheadnode) {
      if(!_pcolornode->HandleHeadZeroSuccessor((CHeadNode*)node, old_color, color, this, stack, todo, status, secure))
        return false;

      if(!_pcolornode->HandleHeadOneSuccessor((CHeadNode*)node, old_color, color, this, stack, todo, status, secure))
        return false;

      if(!_pcolornode->HandleHeadPredecessor((CHeadNode*)node, old_color, color, this, stack, todo, status, secure))
        return false;
    } else
    if(node->GetType() == type_cbodynode) {
      if(!_pcolornode->HandleBodySuccessor((CBodyNode*)node, old_color, color, this, stack, todo, status, secure))
        return false;        
    }
    
    this->SetCanBePropagated(node->GetId(), node->GetType(), color, true);
    return true;
  }

  return (old_color == color+30);
}

bool CGraph::LoadState(TStackElement *stack_element, CStack *stack, CTodoLists* todo) {

  CHECK_POINTER("CGraph::LoadState(stack_element)", stack_element);
  CHECK_POINTER("CGraph::LoadState(stack)"        , stack);
  CHECK_POINTER("CGraph::LoadState(todo)"         , todo);

  CNode* node = stack_element->_pNode;
  
  TColor old_color = node->GetColor();
  if (!node->LoadState(stack_element, stack))
    return false;
  TColor new_color = node->GetColor();

  if(new_color == color_minus)
      todo->PushMinus(node);

  if((new_color == color_none || new_color == color_weak_plus) &&
     (old_color == color_plus || old_color == color_minus || old_color == color_ignore)) {
    _uncolored_nodes++;
  }
  
  return true;
}

void CGraph::BeginLookahead() {
  this->_can_be_propagated_bodynodes_minus.assign(_last_bodynode_id+1, 0);  
  this->_can_be_propagated_bodynodes_plus.assign(_last_bodynode_id+1, 0);  
  this->_can_be_propagated_headnodes.assign(_last_headnode_id+1, 0);
    
  this->_lookahead = true;
}
  
void CGraph::EndLookahead() {
  this->_lookahead = false;
}
  
bool CGraph::IsLookahead() {
  return this->_lookahead;
}

void CGraph::SetCanBePropagated(TId id, TNodeType type, TColor color, bool value) {
  if(!this->_lookahead)
    return;
       
  if(color == color_ignore)
    return;
    
  if(type == type_cbodynode) {
    int val = value;
    if(value)
      val += _can_be_prop_by_supported_node;
      
    if(color == color_minus) {
      _can_be_propagated_bodynodes_minus[id] = val;
    } else {
      _can_be_propagated_bodynodes_plus[id] = val;
    }
  } else {
    if(color == color_minus) {
      _can_be_propagated_headnodes[id] = value;
    }    
  }
}

bool CGraph::GetCanBePropagatedOfBodyPlus(TId id, bool supported) {
  if(!this->_lookahead)
    return false;
  
  return _can_be_propagated_bodynodes_plus[id] > supported;
}
  
bool CGraph::GetCanBePropagatedOfBodyMinus(TId id, bool supported) {
  if(!this->_lookahead)
    return false;
  
  return _can_be_propagated_bodynodes_minus[id] > supported;
}
  
bool CGraph::GetCanBePropagatedOfHeadMinus(TId id) {
  if(!this->_lookahead)
    return false;
  
  return _can_be_propagated_headnodes[id];
}
  
bool CGraph::GetCanBePropBySupportedNode() {
  return this->_can_be_prop_by_supported_node;
}
  
void CGraph::SetCanBePropBySupportedNode(bool value) {
  this->_can_be_prop_by_supported_node = value;
}

CBodyNode* CGraph::InsertBodyNode(CBody* b) {
  TBodyNodeVector::size_type index = b->GetId() - 1;
        
  if(index >= _body_node_index.size()) {
    _body_node_index.resize(index + 1, static_cast<CBodyNode*>(0));
    _last_bodynode_id=index+1;
  }
  
  if(_body_node_index[index] == 0) {
    CBodyNode *t = new CBodyNode(b->GetId(), b);
//    t->SetMaxSupport(true);
    // insert to body nodes and its index
    _body_nodes.push_back(t);
    _body_node_index[index] = t;

    if(t->GetColor() == color_none || t->GetColor() == color_weak_plus) {                         
      ++_uncolored_nodes;
    }
    return t;
  }
        
  return _body_node_index[index];
}

CHeadNode* CGraph::InsertHeadNode(CAtom* h, CStack* s, CTodoLists *todo) {
  THeadNodeVector::size_type index = h->GetId() - 1;

  if(index >= _head_node_index.size()){
    _head_node_index.resize(index + 1, static_cast<CHeadNode*>(0));
    _last_headnode_id=index+1;
  }
        
  if(_head_node_index[index] == 0) {
    CHeadNode* t = new CHeadNode(h->GetId(), h);
//    t->SetMaxSupport(true);
    _head_nodes.push_back(t);
    _head_node_index[index] = t;    
    ++_uncolored_nodes;
    
    if (!ColorNode(t, h->GetColor(), s, false, color_none, todo)) {
      cerr << "CGraph::CGraph(CProgram*): Unable to color node!" << endl;
      exit(1);
    }

    // insert in backward propagation queue only if colored plus or minus
    if (h->GetColor() >= 16) {
      todo->PushBackProp(t);
    }
    return t;
  }
        
   return _head_node_index[index];
}

CHeadNode* CGraph::GetHeadNode(TId id) {
  if(id <= _head_node_index.size())
    return _head_node_index[id-1];
  else
    return NULL;
}

CBodyNode* CGraph::GetBodyNode(TId id) {
  if(id <= _body_node_index.size())
    return _body_node_index[id-1];
  else
    return NULL;
}

void WriteColoring(CGraph* graph) {

  CHECK_POINTER("WriteColoring(graph)", graph);

  TBodyNodeVector *nodes = graph->GetBodyNodes();
  TBodyNodeVectorIterator ite;

  std::cout << endl << "Body-Nodes : " << endl;
  for(ite=nodes->begin();ite!=nodes->end();ite++) {
    std::cout << "     node " << (*ite)->GetId() << " is " << (*ite)->GetColor() << " colored " << endl;
  }

  THeadNodeVector *hnodes = graph->GetHeadNodes();
  THeadNodeVectorIterator hite;

  std::cout << "Head-Nodes : " << endl;
  for(hite=hnodes->begin();hite!=hnodes->end();hite++){
    std::cout << "     node " << (*hite)->GetId() << " is " << (*hite)->GetColor() << " colored " << endl;
  }
}

} // end of namespace NS_NOMORE
