/***************************************************************************
                          cgraph.h  -  description
                             -------------------
    begin                : Thu Jul 24 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cgraph.h
		Contain Definition of CGraph. */

#ifndef CGRAPH_H
#define CGRAPH_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <iostream>
#include "misc.h"
#include "cnode.h"
#include "crule.h"
#include "cstatusinformation.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class dependencies                                                      *
 *                                                                         *
 ***************************************************************************/
class CProgram;
class CColorNode;
class CTodoLists;

/***************************************************************************
 *                                                                         *
 * class CGraph                                                            *
 *                                                                         *
 ***************************************************************************/
/*! \class CGraph
 *  \brief Defines the graph which consists of simple body and head nodes.
 *  \author Andre Neumann
 *  \date Thu Jul 24 2003
 *
 *  This class computes the graph of a corresponding program (CProgram) and
 *  handles the node in simple sets.
 */
class CGraph {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CGraph& graph);
	
private:
  //! The current answer set.
  TAtomSet *_panswer_set; 

  //! Set of all body nodes.
  TBodyNodeVector _body_nodes;

  //! Set of all head nodes.
  THeadNodeVector _head_nodes;

  //! Set of all body nodes.
  TBodyNodeVector _body_node_index;

  //! Set of all head nodes.
  THeadNodeVector _head_node_index;

  //! Object to optional elements like todo, ignore and jumping.
  CColorNode* _pcolornode;

  //! Count of uncolored nodes.
  long _uncolored_nodes;
  
  //! sequence container which holds informationen weather a node can be propagated by another or not (bodynodes).
  /*! Values: 0 corresponds to false, 1 corresponds to true but not supported and 2 corresponds
   *          to true and supported. */
  vector<char> _can_be_propagated_bodynodes_plus;
  
  //! sequence container which holds informationen weather a node can be propagated by another or not (bodynodes).
  /*! Values: 0 corresponds to false, 1 corresponds to true but not supported and 2 corresponds
   *          to true and supported. */
  vector<char> _can_be_propagated_bodynodes_minus;
  
  //! sequence container which holds informationen weather a node can be propagated by another or not (headnodes).
  vector<char> _can_be_propagated_headnodes;
  
  //! coloring of nodes in lookahead or not.
  bool _lookahead;
  
  bool _can_be_prop_by_supported_node;
  
  TId _last_bodynode_id;
  TId _last_headnode_id;
  
protected:
  //! Creates the graph by generating the maximal support graph.
	/*! \pre  No.
   *  \post No.
   *  \param prg  Pointer to an CProgram object which contains the heads
   *              and the bodies of the program.
   *  \param todo Contains information about all todo lists.
   */
  void GenGraph(CProgram *prg, CTodoLists *todo);

public: 
  //! Constructor.
  CGraph();

  //! Constructor initializes the graph by a program.
  CGraph(CProgram* prg, CTodoLists *todo);

  //! Destructor.
  virtual ~CGraph();

  //! Returns the type of the graph.
  inline virtual TType GetType() const {
    return type_cgraph;
  }
  
  //! Equality Operator.
  /*! Checks the equality of the head and the body nodes.
   */
  bool operator==(CGraph& graph);

  //! Inequality Operator.
  /*! Checks the inequality of the head and the body nodes.
   */
  bool operator!=(CGraph& graph);

  //! Returns the answer set corresponding to the colored graph.
  /*! \pre  The graph must be totally colored.
   */
  inline TAtomSet* GetAnswerSet() {
    if (BuildAnswerSet())
      return _panswer_set;
    else 
      exit(1);
  }

  //! Inserts a head or a body node into the graph.
  /*! \param node Pointer to a head or a body node which is copied into the
   *              corresponding set of nodes with their id.
   *  \return Returns true if the node is successful inserted else false.
   */
  bool InsertNode(CNode *node);

  //! Return the status of the coloring.
  /*! \return If all nodes colored plus, minus or ignore the method returns
   *          true else false.
   */
  inline bool TotalColoring() {
    INC_COUNT(methode_totalcoloring);
    return 0 == GetUncoloredCounter();
  }

  //! Put the heads of all plus colored head nodes to _panswer_set.
  /*! \pre  The graph must be totally colored.
   *  \post The member _panswer_set contains the answer set.
   */
  bool BuildAnswerSet();

  //! Generates a body node with the corresponding body object and body id.
  inline CNode* GenerateBodyNode(CBody *body) const {
    CHECK_POINTER("CGraph::GenerateBodyNode(body)", body);
    return (CNode*)new CBodyNode(body->GetId(), body);
  }

  //! Generates a head node with the corresponding head object and head id.
  inline CNode* GenerateHeadNode(CAtom *atom) const {
    CHECK_POINTER("CGraph::GenerateHeadNode(atom)", atom);
    return (CNode*)new CHeadNode(atom->GetId(),atom);
  }

/*  //! Returns plus colored body nodes.
  inline TPNodeList GetPlusColoredBodies() {
    
  	return GetNodes(color_plus, type_cbodynode);
   
  }

	//! Returns weak plus colored body nodes.
  inline TPNodeList GetWeakPlusColoredBodies() {
    
  	return GetNodes(color_weak_plus, type_cbodynode);
   
  }

	//! Returns minus colored body nodes.
  inline TPNodeList GetMinusColoredBodies() {
    
  	return GetNodes(color_minus, type_cbodynode);
   
	}

	//! Returns uncolored body nodes.
  inline TPNodeList GetUncoloredBodies() {
    
  	return GetNodes(color_none, type_cbodynode);
   
  }

	//! Returns plus colored head nodes.
  inline TPNodeList GetPlusColoredHeads() {
    
  	return GetNodes(color_plus, type_cheadnode);
   
  }

	//! Returns weak plus colored head nodes.
  inline TPNodeList GetWeakPlusColoredHeads() {
    
  	return GetNodes(color_weak_plus, type_cheadnode);
   
  }

	//! Returns minus colored head nodes.
  inline TPNodeList GetMinusColoredHeads() {
    
  	return GetNodes(color_minus, type_cheadnode);
   
	}

	//! Returns uncolored head nodes.
  inline TPNodeList GetUncoloredHeads() {
    return GetNodes(color_none, type_cheadnode);
  }
*/
  //! Generates the heuristics for all body and head nodes.
  /*! First a new heuristic object is created for a node by calling
   *  CHeuristic::GenNodeHeuristic(CNode*). After generating the heuristic
   *  the methode call CHeuristic::GenHeuristic() for computing static
   *  heuristics.
   *  \param heu Reference heuristic which is needed to create a new
   *             heuristic object for a specified node.
   */
  void GenHeuristics(CHeuristic* heu);

  //! Returns the counter for uncolored nodes.
  inline long GetUncoloredCounter() {
    return _uncolored_nodes;
  }

  //! Returns the CColorNode object.
  inline CColorNode* GetColorNode() {
    return _pcolornode;
  }

  //! Returns a list of nodes.
  /*! \param color     Specifies the color of the nodes, use | connector
   *                   for return nodes with different colors.
   *  \param node_type Specifies the type of the node like type_bodynode or
   *                   type_headnode.
   *  \return An auto pointer to a TNodeList is returned and must be
   *          destroyed after using.
   */
  //TPNodeList GetNodes(const TColorEnum color, const TNodeType node_type);

  //! Colors all nodes with a specified color to another specifies color.
  /*! \param color     Specifies the color of the nodes, use | connector
   *                   for color nodes with an other color.
   *  \param node_type Specifies the type of the node like type_bodynode
   *                   or type_headnode.
   *  \param to_color  The new color of the choosen nodes.
   *  \param stack     Stack for saving state of nodes.
   *  \param todo      Contains information about all todo lists.
   *  \param status    Contains information about how many nodes are
   *                   colored.
   *  \param secure    Test for valid color changing?
   *  \return If no error occured the methode returns true else false.
   */
  TRetOperator ColorNodes(const TColorEnum color, const TNodeType node_type, const TColor to_color,
      CStack *stack, CTodoLists* todo, CStatusInformation* status = NULL, const bool secure = false);

  //! Colors a single node by executing the CColorNode methods.
  /*! First this method test for valid color changing then
   *  CNode::COlorNode() is call and this method handle the successors and
   *  predecessors for counters, todo-lists, etc.
   *  \pre  The node is part of the graph.
   *  \post The node is colored (if possible).
   *  \param node         Node which should be colored.
   *  \param color        The new color of the node.
   *  \param stack        Stack for saving state informations.
   *  \param choice       Is this node a choice point?
   *  \param choice_color More informations for choice points.
   *  \param todo         Contains information about all todo lists.
   *  \param status       Contains information about how many nodes are
   *                      colored.
   *  \param secure       Test for valid color changing?
   *  \return If no error occured the method returns true else if the node
   *          can't be colored false is returned.
   */
  bool ColorNode(CNode *node, const TColor color, CStack *stack, const bool choice,
      const TColor choice_color, CTodoLists* todo,CStatusInformation* status = NULL,
      const bool secure = true);
  
  //! Returns the node set of the body nodes.
  inline TBodyNodeVector* GetBodyNodes() {
    return &_body_nodes;
  }

  //! Returns the head node corresponds to the input id.
  CHeadNode *GetHeadNode(TId id);

  //! Returns the body node corresponds to the input id.
  CBodyNode *GetBodyNode(TId id);

  //! Returns the node set of the head nodes.
  inline THeadNodeVector* GetHeadNodes() {
    return &_head_nodes;
  }

  //! Load the state of a single node.
  /*! First this method loads the node using CNode::LoadState() and then
   *  handle the successors and predecessors for counters.
   *  \pre  The node is part of the graph.
   *  \post The node is uncolored (or recolored) if no error occures.
   *  \param stack_element Contains the information for loading the node.
   *  \param stack         Stack for saving stack element.
   *  \param todo          Contains information about all todo lists.
   *  \return If no error occured the method returns true else if the node
   *          can't be loaded false is returned.
   */
  bool LoadState(TStackElement *stack_element, CStack *stack, CTodoLists* todo);
                 
  //! Starts the lookahead mode.
  void BeginLookahead();
  
  //! Stops the lookahead mode.
  void EndLookahead();
  
  //! Returns the status of the lookahead mode.
  bool IsLookahead();
  
  //! Sets a node as can be propagated.
  void SetCanBePropagated(TId id, TNodeType type, TColor color, bool value);
  
  //! Returns the status weather a node is can be propagated or not.
  bool GetCanBePropagatedOfBodyPlus(TId id, bool supported);
  
  //! Returns the status weather a node is can be propagated or not.
  bool GetCanBePropagatedOfBodyMinus(TId id, bool supported);
  
  //! Returns the status weather a node is can be propagated or not.
  bool GetCanBePropagatedOfHeadMinus(TId id);
  
  bool GetCanBePropBySupportedNode();
  
  void SetCanBePropBySupportedNode(bool value);
  
  //! Returns the status weather a node is can be propagated or not.
  //bool GetCanBePropagated(TId id, TNodeType type, TColor color);

  void InsertPossibleSupportedBodies(TnBodySet bodies, CProgram* prg, CTodoLists *todo);

  CBodyNode* InsertBodyNode(CBody* b);
  CHeadNode* InsertHeadNode(CAtom* h, CStack* s, CTodoLists *todo);
};

#ifdef _DEBUG
  //! Writes the coloring of a graph to stdout.
  void WriteColoring(CGraph* graph);
#endif

} // end of namespace NS_NOMORE


#endif
