/***************************************************************************
                          cheuristic.h  -  description
                             -------------------
    begin                : Thu Oct 16 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cheuristic.h
 *  Contains Definitions of CHeuristic, CHeuristicNone, CHeuristicS1,
 *  CHeuristicS2, CHeuristicSR, CHeuristicD1 and CHeuristicDR.
 */

#ifndef CHEURISTIC_H
#define CHEURISTIC_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "print.h"
#include "misc.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class dependencies                                                      *
 *                                                                         *
 ***************************************************************************/
class CNode;
class CBodyNode;
class CHeadNode;

/***************************************************************************
 *                                                                         *
 * class CHeuristic                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristic
 *  \brief Defines the abstract class for static and dynamic heuristics
 *         for a node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *  Represents the interface for static and dynamic heuristics. CNode
 *  contains a private element of this Class which is used for determining
 *  position in priority queue of TODO-CPs. It is also used during
 *  lookahead, but this will be removed. Which Heuristic will be used is
 *  determined when initializing a CGraph. To write a new heuristic
 *  function you must overwrite the methodes CalculateHeuristicValue(),
 *  GenerateInstance(), GetType().
 */
class CHeuristic {

protected:
  //! Node for heuristic.
  CNode* _pnode;

  //! Constructs the object.
	CHeuristic();

public:
  //! Destructs the object.
  virtual ~CHeuristic();

  //! Returns the heuristic value of a node.
  /*! Returns the heuristic value of a node. In case of a dynamic
   *  heuristic, this method will always call CalculateHeuristicValue,
   *  otherwise it will calculate only once and then return the stored
   *  value.
   *  \pre  The heuristic is precompiled (if static heuristic).
   *  \return Returns the heuristic value. The higher this value the more
   *          important the node.
   */
  virtual int GetHeuristicValue() = 0;

  //! Calculates the heuristic value of a node.
  virtual int CalculateHeuristicValue() = 0;

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node) = 0;

  //! Returns a heuristic specific id which e.g. is used for equality check.
  virtual TType GetType() const = 0;

  //! Tests for the simple smodels heuristics.
  /*! \todo longer explanation
   */
  virtual bool LookaheadTest(CLookaheadInformation *l_old,
                             CLookaheadInformation *l_new);

  //! Tests for the simple smodels heuristics.
  /*! This method computes the priority of a node with informations about
   *  its effects for the deterministic operator.
   *  \param info Contains information of the effects from the
   *              deterministic operators.
   *  \return Returns an integer value which represents the priority of the
   *          node (100*color_effects_on_plus+color_effects_on_minus).
   */
  virtual int Lookahead(CLookaheadInformation *info);

  //! Returns the node.
  inline CNode* GetNode() {
    
    return _pnode;
    
  }

  //! Sets the node.
  inline virtual void SetNode(CNode* n) {

    CHECK_POINTER("CHeuristic::SetNode(n)", n);
    
    _pnode = n;
    
  }

  //! Compare two heuristics. If the return value is true the current heuristic value is better.
  /*! \attention Overwrite this method for new heuristic objects!
   */
  virtual bool operator<(CHeuristic &heu) = 0;
  
};


/***************************************************************************
 *                                                                         *
 * class CStaticHeuristic                                                  *
 *                                                                         *
 ***************************************************************************/
/*! \class CStaticHeuristic
 *  \brief Defines the interface for static heuristics for a node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *  Static Heuristics. To write a new heuristic function you must overwrite
 *  the methodes CalculateHeuristicValue(), GenerateInstance(), GetType().
 */
class CStaticHeuristic : public CHeuristic {

protected:
  //! Heuristic value.
	int _value;

  //! Constructs the object.
	CStaticHeuristic();

public:
  //! Destructs the object.
  virtual ~CStaticHeuristic();

  //! Returns the heuristic value of a node.
  /*! Returns the heuristic value of a node. In case of a dynamic
   *  heuristic, this method will always call CalculateHeuristicValue,
   *  otherwise it will calculate only once and then return the stored
   *  value.
   *  \return Returns the heuristic value. The higher this value the more
   *          important the node.
   */
  inline virtual int GetHeuristicValue() {
    
  	return _value;
   
  }

  //! Sets the node.
  inline virtual void SetNode(CNode* n) {

    CHECK_POINTER("CStaticHeuristic::SetNode(n)", n);
    
    _pnode = n;
    _value = CalculateHeuristicValue();
    
  }

  //! Compare two heuristics. If the return value is true the current heuristic value is better.
  inline virtual bool operator<(CHeuristic &heu) {
    
    return GetHeuristicValue() < heu.GetHeuristicValue();
    
  }

};

/***************************************************************************
 *                                                                         *
 * class CDynamicHeuristic                                                 *
 *                                                                         *
 ***************************************************************************/
/*! \class CDynamicHeuristic
 *  \brief Defines the interface for Dynamic heuristics for a node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *  Dynamic Heuristics. To write a new heuristic function you must
 *  overwrite the methodes CalculateHeuristicValue(), GenerateInstance(),
 *  GetType().
 */
class CDynamicHeuristic : public CHeuristic{

public:
  //! Constructs the object.
	CDynamicHeuristic();

  //! Destructs the object.
  virtual ~CDynamicHeuristic();

  //! Returns the heuristic value of a node.
  /*! Returns the heuristic value of a node. In case of a dynamic
   *  heuristic, this method will always call CalculateHeuristicValue,
   *  otherwise it will calculate only once and then return the stored
   *  value.
   *  \return Returns the heuristic value. The higher this value the more
   *          important the node.
   */
  inline virtual int GetHeuristicValue() {
    
  	return CalculateHeuristicValue();
   
  }

  //! Compare two heuristics. If the return value is true the current heuristic value is better.
  inline virtual bool operator<(CHeuristic &heu) {
    
    return GetHeuristicValue() < heu.GetHeuristicValue();
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CHeuristicNode                                                    *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicNone
 *  \brief Defines the static heuristic for a node which returns always the
 *         priority 0.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 */
class CHeuristicNone : public CStaticHeuristic {
  
public:
  //! Constructs the object.
	CHeuristicNone();

  //! Destructs the object.
  virtual ~CHeuristicNone();

  //! Calculates the heuristic value of a node.
  inline virtual int CalculateHeuristicValue() {

  	return 0;
   
  };

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_cheuristic_none;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CHeuristicS1                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicS1
 *  \brief Define a simple static heuristic for a body node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *   The priority of a body node computes itself out of
 *   256*(1-successors) + (1-predecessors). 1-successors means all
 *   1-successors of the body node successors head. For head nodes the
 *   priority 0 is returned.
 */
class CHeuristicS1 : public CStaticHeuristic {

public:
  //! Constructs the object.
	CHeuristicS1();

  //! Destructs the object.
  virtual ~CHeuristicS1();

  //! Calculates the heuristic value of a node.
  virtual int CalculateHeuristicValue();

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_cheuristic_s1;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CHeuristicS2                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicS2
 *  \brief Define a simple static heuristic for a body node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *  The priority of a body node computes itself out of
 *  (0-successors) + (1-successors). 0-successors and 1-successors means
 *  all 0- and 1-successors of the body node successors head. For head
 *  nodes the priority 0 is returned.
 */
class CHeuristicS2 : public CStaticHeuristic {

public:
  //! Constructs the object.
	CHeuristicS2();

  //! Destructs the object.
  virtual ~CHeuristicS2();

  //! Calculates the heuristic value of a node.
  virtual int CalculateHeuristicValue();

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_cheuristic_s2;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CHeuristicSR                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicSR
 *  \brief Define a random static heuristic for a body node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *   The return value for the node priority is a random number between 0
 *   and 1023.
 */
class CHeuristicSR : public CStaticHeuristic {

public:
  //! Constructs the object.
	CHeuristicSR();

  //! Destructs the object.
  virtual ~CHeuristicSR();

  //! Calculates the heuristic value of a node.
  inline virtual int CalculateHeuristicValue() {

    INC_COUNT(heuristics_sr_call);
        
    return rand()%1024;
    
  }
   
  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const{
    
    return type_cheuristic_sr;
    
  }

};


/***************************************************************************
 *                                                                         *
 * class CHeuristicD1                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicD1
 *  \brief Define a simple dynamic heuristic for a body node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *   The priority of a body node computes itself out of
 *   256*(1-successors) + (1-predecessors). 1-successors means all
 *   1-successors of the body node successors head. Only colored successors
 *   and predecessors are used to compute the priority. For head nodes the
 *   priority 0 is returned.
 */
class CHeuristicD1 : public CDynamicHeuristic {

public:
  //! Constructs the object.
	CHeuristicD1();

  //! Destructs the object.
  virtual ~CHeuristicD1();

  //! Calculates the heuristic value of a node.
  virtual int CalculateHeuristicValue();

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_cheuristic_d1;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CHeuristicDR                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeuristicDR
 *  \brief Define a random dynamic heuristic for a body node.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *   The return value for the node priority is a random number between 0
 *   and 1023.
 */
class CHeuristicDR : public CDynamicHeuristic {

public:
  //! Constructs the object.
	CHeuristicDR();

  //! Destructs the object.
  virtual ~CHeuristicDR();

  //! Calculates the heuristic value of a node.
  inline virtual int CalculateHeuristicValue() {
    
    INC_COUNT(heuristics_dr_call);

    return rand()%1024;
    
  }
  
  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
      \return Returns a new heuristic object for the input node.*/
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const{
    
    return type_cheuristic_dr;
    
  }
  
};

/*! \class CHeuristicAlphabetical
 *  \brief Define a heuristic which sorts the node alphabetical.
 *  \author Christian Anger, Andre Neumann
 *  \date Thu Oct 16 2003
 *
 *   The return value for the node priority is unimportant. Use only
 *   the operator<()!
 */
class CHeuristicAlphabetical : public CDynamicHeuristic {
  
public:
  //! Constructs the object.
	CHeuristicAlphabetical();

  //! Destructs the object.
  virtual ~CHeuristicAlphabetical();

  //! Calculates the heuristic value of a node.
  inline virtual int CalculateHeuristicValue() {
    
    return 0;
    
  }

  //! Sets the node.
  virtual void SetNode(CNode* n);

  //! Generates a new heuristic object for another node.
  /*! \param node CNode for which the new heuristic object is created.
   *  \return Returns a new heuristic object for the input node.
   */
  virtual CHeuristic* GenerateInstance(CNode* node);

  //! Returns a heuristic specific id which is used for testing of equality.
  inline virtual TType GetType() const{
    
    return type_cheuristic_alpha;
    
  }

  //! Compare two heuristics by comparing its body strings.
  virtual bool operator<(CHeuristic &heu);
  
};

} // end of namespace NS_NOMORE

#endif
