/***************************************************************************
                          cprogram.h  -  description
                             -------------------
    begin                : Mon Jul 21 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cprogram.h
 *  Declaration of class CProgram.
 */

#ifndef CPROGRAM_H
#define CPROGRAM_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "misc.h"
#include "crule.h"
#include <algorithm>
#include <iostream>

using std::pair;

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class CProgram                                                          *
 *                                                                         *
 ***************************************************************************/
/*! \class CProgram cprogram.h
 *  \brief Defines internal representation of a logic program.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *
 *  CProgram consists of a set of atoms, a set of heads, a set of bodies
 *  and a set of rules which are read from stdin by Read(). Only lparse
 *  output can be read at this time.
 */
class CProgram {

  //! output operator.
  friend std::ostream& operator<<(std::ostream& os, CProgram& program);

private:
  //! All atoms of the logic program.
  TAtomVector _atoms;

  //! All heads of the logic program.
  TIdSet _heads;

  //! All bodies of the logic program.
  TnBodySet _bodies;

  //! The rules of the logic program .
  TRuleVector _rules; 
  
  int _body_counter;

public:
  //! Constructor.
  CProgram();

  //! Destructor deletes all pointers in atom-, body- and rule-set.
  virtual ~CProgram();

  //! Returns the set of atoms of the program.
  inline TAtomVector& GetAtoms() {
    return _atoms;
  };

  //! Returns the set of heads of the program.
  inline TIdSet& GetHeads() {
    return _heads;
  };

  //! Returns the set of bodies of the program.
  inline TnBodySet& GetBodies() {
    return _bodies;
  };

  //! Returns the set of rules of the program.
  inline TRuleVector& GetRules() {
    return _rules;
  };

  //! Read the output of lparse.
  /*! \post The rules, heads, bodies and atoms are set.
   *  \param f Specifies the stream from which the program is read (file
   *           or stdin).
   *  \return If the syntax is correct and the program is read the methode
   *          returns true else false.
   */
  bool Read(std::istream &f);

  //! Adds a basic rule to the rule set (one lparse line).
  /*! \post If no error occurs the basic rule is inserted.
   *  \param f Specifies the stream from which the basic rule is read (file
   *           or stdin).
   *  \return If the syntax is correct and the basic rule is read the
   *          methode returns true else false.
   *  \todo read more different (lparse) rule types.
   */
  bool AddBasicRule(std::istream &f);

  //! Reads the body part of an lparse decoded rule.
  /*! \post The body is read and added to the instance member if no error
   *        occured.
   *  \param f        Specifies the stream from which the body is read
   *                  (file or stdin).
   *  \param size     Count of positive and negative body literals for read.
   *  \param size_neg Count of negative body literals for read.
   *  \param rule     Rule for which the body is to set.
   *  \return If the syntax is correct and the body is read the methode
   *          returns true else false.
   */
  bool ReadBody(std::istream &f, long size, long size_neg, CRule* rule);

  //! Equality operator used for testing.
  bool operator==(const CProgram& prg) const;

  //! Inequality operator used for testing.
  bool operator!=(const CProgram& prg) const;

  //! Insert a rule into the map of rules by copying the pointer address.
  /*! \attention If the destructor is called the pointer is deleted.
   */
  CRule* InsertRule(CRule* rule);

  //! Insert an atom into the map of atoms by copying the pointer address.
  /*! \attention If the destructor is called the pointer is deleted.
   */
  CAtom* InsertAtom(TId id);

  //! Insert a head into the map of heads by copying the pointer address.
  /*! \attention If the destructor is called the pointer is deleted.
   */
  CAtom* InsertHead(TId id);

  //! Insert a body into the map of bodies by copying the pointer address.
  /*! \attention If the destructor is called the pointer is deleted.
   */
  CBody* InsertBody(CBody* body);

  //! Returns a rule specified by ID.
  /*! \param id Specifies the id of the rule which should be returned.
   *  \return If the rule with the specified ID does not exist the methode
   *          returns NULL.
   */
  //CRule* GetRule(const TId id);

  //! Returns an atom specified by ID.
  /*! \param id Specifies the id of the atom which should be returned.
   *  \return If the atom with the specified ID does not exist the methode
   *          returns NULL.
   */
  CAtom* GetAtom(const TId id);
  
};

} // end of NS_NOMORE

#endif
