/***************************************************************************
                          misc.h  -  description
                             -------------------
    begin                : Thu Jun 19 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file misc.h
 *  Declaration of different types and class CIdGenerator, CStack, CLess,
 *  CLess, CLookaheadInformation and CPQueueLookahead.
 */

#ifndef __misc_h__
#define __misc_h__

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <map>
#include <list>
#include <vector>
#include <queue>
#include <stack>
#include "print.h"
#include "cset.h"
#include <memory>

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class dependencies                                                      *
 *                                                                         *
 ***************************************************************************/
class CNode;
class CBody;
class CAtom;
class CRule;
class COperator;
class CLessNode;
class CStatusInformation;
class CLookaheadInformation;
class CAtomComparison;

/***************************************************************************
 *                                                                         *
 * Typedefinitions                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \enum TColor
 *  \brief Defines different possible colors for a node.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  The color is defined as bit-mask for the get-methode CColorStructur
 *  for multi-color-output.
 */
typedef enum {
	color_none      =  1, // -> 00000001
	color_weak_plus =  2, // -> 00000010
	color_ignore    =  4, // -> 00000100
	color_unknown   =  8, // -> 00001000
	color_minus     = 16, // -> 00010000
	color_plus      = 32  // -> 00100000

} TColor;

typedef enum {
  queuetype_propagation,
  queuetype_backwardpropagation,
  queuetype_minus,
  queuetype_possiblechoices
  
} TQueueType;

/*! \enum TColorEnum
 *  \brief Defines different possible colors as bit-mask enumeration.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  The color is defined as bit-mask for the get-methode CColorStructur for
 *  multi-color-output.
 */
typedef int TColorEnum;

/*! \enum TRetOperator
 *  \brief Defines different possible return values of an operator.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef enum {
	return_unchanged  = 0,
	return_changed    = 1,
	return_colorerror = 2,
  return_inputerror = 3,
  return_unknown    = 4

} TRetOperator;

/*! \enum TType
 *  \brief Defines different possible types of classes. Used to determine
 *         the type of a particular instance.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef enum {
  // types for queue's
  type_none                  =        0,
  type_cn_todo               =        1,  // 2^0
  type_cn_jumping            =        2,  // 2^1
  type_cn_ignore             =        4,  // 2^2
  type_cn_none               =        8,  // 2^3

  // types for deterministic operators
  type_coperator_bp          =       16,  // 2^4
	type_coperator_ms          =       32,  // 2^5
	type_coperator_p           =       64,  // 2^6
  type_coperator_ps          =      128,  // 2^7
  type_coperator_u           =      256,  // 2^8
  type_coperator_v           =      512,  // 2^9
  type_coperator_v2          =     1024,  // 2^10
  type_coperator_v3          =     2048,  // 2^11
  type_coperator_v4          =     4096,  // 2^12
  type_coperator_v5          =     8192,  // 2^13
  type_coperator_n           =    16384,  // 2^14
  type_coperator_nsecure     =    32768,  // 2^15
  type_coperator_pre         =    65536,  // 2^16
  type_coperator_labh        =   131072,  // 2^17
  type_coperator_lab         =   262144,  // 2^17

  // types for aggregation
  type_caggregation_error    =   524288,  // 2^18
  type_cdetoperator_none,
  type_caggregation_star,
  type_caggregation_sequence,

  // types for non-deterministic operators
  type_coperator_c,
  type_coperator_cq,
  type_coperator_d,
  type_coperator_dq,
  type_coperator_dh,
  type_coperator_lafaire,
  type_coperator_lacomplete,

  // types for heuristics
  type_cheuristic_none,
  type_cheuristic_s1,
  type_cheuristic_s2,
  type_cheuristic_sr,
  type_cheuristic_d1,
  type_cheuristic_dr,
  type_cheuristic_alpha,

  // various types
  type_cgraph,

  type_unknown

} TType;

/*! \enum TTypeEnum
 *  \brief Defines different possible types as bit-mask enumeration.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  The type is defined as bit-mask for the enable-methode CColorNode.
 */
typedef int TTypeEnum;

/*! \enum TRuleType
 *  \brief From SModels! Defines different rule types for reading the
 *         lparse output.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef enum {
  ENDRULE,
  BASICRULE,
  CONSTRAINTRULE,
  CHOICERULE,
  GENERATERULE,
  WEIGHTRULE,
  OPTIMIZERULE

} TRuleType;

/*! \typedef TId
 *  \brief Type for identifiers.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef long int TId;

/*! \typedef TBoolSet
 *  \brief A std::map containing simple boolean values for an identifier.
 *         Used for saving information if a node is in a queue or not.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, bool> TBoolSet;

/*! \typedef TNodeSet
 *  \brief A std::map containing pointers to CNode. The Id of a Node is
 *         used for indexing.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, CNode*> TNodeSet;

/*! \typedef TNodeList
 *  \brief A std::list containing pointers to CNode.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CList<CNode*> TNodeList;

/*! \typedef TPNodeList
 *  \brief An auto pointer to a std::list containing pointers to CNode.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef auto_ptr<TNodeList> TPNodeList;

/*! \typedef TAtomSet
 *  \brief A std::map containing pointers to CAtom. The Id of an Atom is
 *         used for indexing.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, CAtom*> TAtomSet;

/*! \typedef TAtomList
 *  \brief A std::list containing pointers to CAtom.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CList<CAtom*> TAtomList;

/*! \typedef TBodySet
 *  \brief A std::map containing pointers to CBody. The Id of a body is
 *         used for indexing.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, CBody*> TBodySet;

/*! \typedef TSetOfAtomSets
 *  \brief A std::map containing atom sets. The Id of an AtomSet is used
 *         for indexing.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, TAtomSet> TSetOfAtomSets;

/*! \typedef TRuleSet
 *  \brief A std::map containing pointers to CRule. The Id of a Rule is
 *         used for indexing.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef CSet<TId, CRule*> TRuleSet;

/*! \struct TStackElement
 *  \brief A struct containing TId, the data and the size of the data block.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef struct{

  //! Pointer to node.
  CNode* _pNode;
  
  //! Is this node a choice point?
  bool _ChoicePoint;

  //! Is the next color for recolor the choice point.
  /*! If this value is color_none both colors are tested.
   */
  TColor _ChoicePointColor;

  //! Saved color before changing the color of this node.
  TColor _SavedColor;

} TStackElement;

/*! \typedef TQueue
 *  \brief A std::queue for a todo-list, etc.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef std::queue<CNode*> TQueue;

/*! \typedef TPQueue
 *  \brief A std::priority_queue for choice-point-list, etc. using the
 *         CLess class for inserting a node.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
typedef std::priority_queue<CNode*,
                            vector<CNode*>,
                            CLessNode>       TPQueue;

/***************************************************************************
 *                                                                         *
 * Helper functions                                                        *
 *                                                                         *
 ***************************************************************************/
//! Returns for a specified color the string value.
/*! \param color TColor value for translating into string value.
 */
string TranslateColor(const TColor color);

/***************************************************************************
 *                                                                         *
 * Class definitions                                                       *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 * class CStack                                                            *
 *                                                                         *
 ***************************************************************************/
/*! \class CStack
 *  \brief A std::stack for saving state informations of nodes.
 *         Additionally can hold information about current backtracking
 *         depth and if an error occured the backtracking depth of the
 *         error node.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *  \attention The backtracking depth feature is not available yet because
 *             of performance.
 */
class CStack : public std::stack<TStackElement> {

public:
  //! Constructs the object.
  CStack();

  //! Destructs the object.
  virtual ~CStack();

};

/***************************************************************************
 *                                                                         *
 * class CAtomComparision                                                  *
 *                                                                         *
 ***************************************************************************/
//! Compares two atom-pointers.
/*! \class CAtomComparison
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
class CAtomComparison {

public:
  //! Comparions of two atoms.
  bool operator()(const CAtom* a1,
                  const CAtom* a2) const;

};

/***************************************************************************
 *                                                                         *
 * class CLessAtom                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \class CLessAtom
 *  \brief Comparision class for sorting atom in bodies.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
class CLessAtom : public binary_function<CAtom*, CAtom*, bool> {

public:
  //! Comparison of two atoms.
  bool operator()(CAtom* a1,
                  CAtom* a2);
    
};

/***************************************************************************
 *                                                                         *
 * class CLessNode                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \class CLessNode
 *  \brief Comparision class for the priority-queue container operating on
 *         the heuristics of a node.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
class CLessNode : public binary_function<CNode*, CNode*, bool> {

public:
  //! Comparison of two nodes.
  bool operator()(CNode* n1,
                  CNode* n2);

};

/***************************************************************************
 *                                                                         *
 * class CLessLookahead                                                    *
 *                                                                         *
 ***************************************************************************/
/*! \class CLessLookahead
 *  \brief Comparision class for the priority-queue container operating on
 *         the heuristics of a node for look ahead.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
class CLessLookahead : public binary_function<CLookaheadInformation*,
                                              CLookaheadInformation*,
                                              bool> {

public:
  //! Comparison of two lookahead info objects.
  bool operator()(CLookaheadInformation* n1,
                  CLookaheadInformation* n2);
    
};

/***************************************************************************
 *                                                                         *
 * class TPQueueLookahead                                                  *
 *                                                                         *
 ***************************************************************************/
/*! \class TPQueueLookahead
 *  \brief A std::priority_queue for choice-point-list, etc. using the
 *         CLess class for inserting a node.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 */
class CPQueueLookahead
        : public std::priority_queue<CLookaheadInformation*,
                                     vector<CLookaheadInformation*>,
                                     CLessLookahead> {

public:
  //! Constructs the object.
  CPQueueLookahead();

  //! Destructs the object. Calls Delete().
  virtual ~CPQueueLookahead();

  //! Deletes all pointers in queue.
  void Delete();
    
};

/***************************************************************************
 *                                                                         *
 * class CIdGenerator                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CIdGenerator
 *  \brief Generates unique Ids.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  CIdGenerator contains a static reference to the only allowed instance
 *  of its class.
 */
class CIdGenerator {

private:
  //! Contains the last generated Id.
  static TId _id; 

public:
  //! Constructor.
  CIdGenerator();

  //! Destructor.
	~CIdGenerator();

	//! Generates an unique id
	static inline TId GetId() {
    
		return ++_id;
    
  };

  //! Sets the unique id to zeo.
  static inline void Reset() {
    
    _id = 0;
    
  }

};

/***************************************************************************
 *                                                                         *
 * class CLookaheadInformation                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CLookaheadInformation
 *  \brief Stores look ahead information.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  If a node is selected for look ahead, the Lookahead() method of the
 *  non-deterministic operator calls the deterministic operator and stores
 *  all effects on other nodes (bodies and heads) in CStatusInformation. You
 *  get two objects of CStatusInformation for color possible choice node to
 *  plus and minus. Additionaly this class stores the heuristic value of the
 *  node.
 */
class CLookaheadInformation {

private:
  //! Possible choice point.
  CNode* _pnode;

  //! Coloring informations for if choice point is colored plus.
  CStatusInformation *_pstatus_plus;

  //! Coloring informations for if choice point is colored minus.
  CStatusInformation *_pstatus_minus;

  //! Heuristic value for priority queue.
  int _heuristic;

public:
  //! Constructs the object. (Sets the node)
  CLookaheadInformation(CNode* node = NULL);

  //! Destructs the object.
  virtual ~CLookaheadInformation();

  //! Returns the node.
  inline CNode* GetNode() {
    
    return _pnode;
    
  }

  //! Returns the status information for the plus colored node.
  inline CStatusInformation* GetPlusStatus() {
    
    return _pstatus_plus;
    
  }

  //! Returns the status information for the minus colored node.
  inline CStatusInformation* GetMinusStatus() {
    
    return _pstatus_minus;
    
  }

  //! Sets the heuristic value.
  inline void SetHeuristic(int value) {
    
    _heuristic = value;
    
  }

  //! Returns the heuristic value.
  inline int HeuristicValue() const {
    
    return _heuristic;
    
  }

}; 

} // end of NS_NOMORE

#endif
