/***************************************************************************
                          coperator.h  -  description
                             -------------------
    begin                : Tue Jul 29 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file coperator.h
 *  Declaration of method GenerateOperator and the classes COperator,
 *  CDetOperator, COperatorP, COperatorPs, COperatorV, COperatorV2,
 *  COperatorV4, COperatorV5, COperatorN, COperatorPre and CDetOperatorNone.
 */

#ifndef COPERATOR_H
#define COPERATOR_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "cgraph.h"
#include "ctodolists.h"
#include "cstatusinformation.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class COperator                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \class COperator coperator.h
 *  \brief Interface for defining several deterministic and non-
 *         deterministic operators.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperator is the base class for all deterministic and non-deterministic
 *  operators.
 */
class COperator {
private:
  static bool _use_simple_choice_operator;

public:
  //! Constructor.
	COperator();

  //! Destructor.
	virtual ~COperator();
	
  //! Returns an operator specific id which is used for testing of equality.
  virtual TType GetType() const = 0;

  //! Equality operator.
  virtual bool operator==(COperator& op);

  //! Inequality operator.
  virtual bool operator!=(COperator& op);

  static  bool UseSimpleChoiceOperator() {
    return _use_simple_choice_operator;
  }
  
  static  void SetSimpleChoiceOperator(bool value) {
    _use_simple_choice_operator = value;
  }
  
};
  
/***************************************************************************
 *                                                                         *
 * class CDetOperator                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CDetOperator coperator.h
 *  \brief Interface for defining deterministic operators like COperatorP.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 * CDetOperator is the base class for all deterministic operators. All
 * these operators must be implements the Call() methode.
 */
class CDetOperator : public COperator {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CDetOperator& op);

public:
  //! Constructor.
	CDetOperator();

  //! Destructor.
	virtual ~CDetOperator();

  //! Starts the deterministic computation.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If any error occured the computation the method returns
   *          return_colorerror else the method returns the state of the
   *          computation (return_changed or return_unchanged).
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL) = 0;

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  virtual TTypeEnum GetOperatorID() = 0;

};

/***************************************************************************
 *                                                                         *
 * CDetOperatorNone                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class CDetOperatorNone coperator.h
 *  \brief This Operator does not change anything. The Call() method
 *         returns return_unchanged!
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 */
class CDetOperatorNone : public CDetOperator{

public:
  //! Constructor.
	CDetOperatorNone();

  //! Destructor.
	virtual ~CDetOperatorNone();

  //! Do nothing and return return_unchanged.
  /*! \param graph Graph on which the computation is started.
   *  \param stack Stack on which the changes are stored.
   *  \param todo Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return Returns return_unchanged.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_cdetoperator_none;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_cdetoperator_none;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorBP                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorBP coperator.h
 *  \brief Back-propagation operator.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  This operator back-propagates all nodes in the back-propagation queue
 *  of the input object of class CTodoLists. The operator is divided into
 *  two parts. Part one tries to back propagate a head node. The table
 *  shows how the method works with head nodes.
 *  <TABLE BORDER=1 CELLSPACING=3 CELLPADDING=3 WIDTH=70%>
 *  <TR>
 *    <TH>color of the head node                                      </TH>
 *    <TH>what does this operator do                                  </TH>
 *  </TR>
 *  <TR>
 *    <TD>color_plus                                                  </TD>
 *    <TD>if all except one of the predecessors are colored minus then
 *        try to color this one to weak_plus and put the node into the
 *        back-propagation queue                                      </TD>
 *  </TR>
 *  <TR>
 *    <TD>color_weak_plus                                             </TD>
 *    <TD>see color_plus                                              </TD>
 *  </TR>
 *  <TR>
 *    <TD>color_minus                                                 </TD>
 *    <TD>color all predecessors to minus and put these nodes into
 *        back-propagation queue                                      </TD>
 *  </TR>
 *  </TABLE><br>
 *  The second part tries to color a body node.
 *  <TABLE BORDER=1 CELLSPACING=3 CELLPADDING=3 WIDTH=70%>
 *  <TR>
 *    <TH>color of the body node                                     </TH>
 *    <TH>what does this operator do                                 </TH>
 *  </TR>
 *  <TR>
 *    <TD>color_plus                                                 </TD>
 *    <TD>color all 0- and 1-predecessors to weak_plus and put these
 *        nodes into back-propagation queue                          </TD>
 *  </TR>
 *  <TR>
 *    <TD>color_weak_plus                                            </TD>
 *    <TD>see color_plus                                             </TD>
 *  </TR>
 *  <TR>
 *    <TD>color_minus                                                </TD>
 *    <TD>
 *        -# if node is supported
 *          - if all except one of the 1-predecessors are colored
 *            minus then try to color this one weak_plus and put
 *            this node into back-propagation queue
 *        -# if node is unblocked
 *          - if all except one of the 0-predecessors are colored
 *            plus then try to color this one minus and put this
 *            node into back-propagation queue                      </TD>
 *  </TR>
 *  </TABLE>
 */
class COperatorBP : public CDetOperator {

public:
  //! Constructor.
	COperatorBP();

  //! Destructor.
	virtual ~COperatorBP();

  //! Starts the back-propagation.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Back-propagation on a head node.
  /*! \param graph  Graph for back-propagation.
   *  \param stack  Stack for saving state informations of the colored nodes.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \param node   Node for back-propagation.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  TRetOperator BackPropHeadNode(CGraph* graph,
                                CStack* stack,
                                CHeadNode *node,
                                CTodoLists* todo,
                                CStatusInformation* status);

  //! Back-propagation on a body node.
  /*! \param graph  Graph for back-propagation.
   *  \param stack  Stack for saving state informations of the colored nodes.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \param node   Node for back-propagation.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  TRetOperator BackPropBodyNode(CGraph* graph,
                                CStack* stack,
                                CBodyNode *node,
                                CTodoLists* todo,
                                CStatusInformation* status);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_bp;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_bp;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorMS                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorMS coperator.h
 *  \brief Colors all supported nodes to plus.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  Generates the maximal support graph by coloring all supported nodes in
 *  TODO list to plus and put successors into todo list. Computation in
 *  four steps:
 *
 *  -# if propagation queue is empty then return
 *  -# take a body node from propagation queue (also used by OperatorPs)
 *  -# if this node is supported (all 0-predecessors are colored plus) then
 *     color it to plus
 *      - automatically color successor head node to plus
 *      - put 0-successors of head node into propagation queue
 *  -# goto step 1
 */
class COperatorMS : public CDetOperator {

public:
  //! Constructor.
	COperatorMS();

  //! Destructor.
	virtual ~COperatorMS();

  //! Colors all supported nodes to plus.
  /*! \param graph G raph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_ms;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_ms;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorP                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorP coperator.h
 *  \brief Deterministic operator P defined in paper.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperatorP computes the deterministic consequences of the colored BHG.
 *  The operator tests all body nodes if there are
 *  - supported and unblocked then color the node to plus
 *    - automatically color successor head node to plus
 *  - unsupported or blocked the color the node to minus
 *    - automatically color successor head node to minus IF all its
 *      predecessors are minus
 */
class COperatorP : public CDetOperator {

public:
  //! Constructor.
	COperatorP();

  //! Destructor.
	virtual ~COperatorP();

  //! Starts the simple forward propgation.
  /*! \param graph  Graph on which the computation is started.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \param stack  Stack on which the changes are stored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_p;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_p;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorPs                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorPs coperator.h
 *  \brief Implements an improvement of the operator P.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperatorPs computes the deterministic consequences of the colored BHG.
 *  The methode checks all propagation queue members if the node is
 *  <TABLE BORDER=1 CELLSPACING=3 CELLPADDING=3 WIDTH=70%>
 *  <TR>
 *    <TH>status of the body node                                     </TH>
 *    <TH>description                                                 </TH>
 *  </TR>
 *  <TR>
 *    <TD>blocked                                                     </TD>
 *    <TD>color node to minus
 *          - automatically color successor head node to minus IF all
 *            its predecessors are minus
 *          - put all 0- and 1-successors of head node IF head node is
 *            colored                                                 </TD>
 *  </TR>
 *  <TR>
 *    <TD>unsupported                                                 </TD>
 *    <TD>color node to minus
 *          - automatically color successor head node to minus IF all
 *            its predecessors are minus
 *          - put all 0- and 1-successors of head node IF head node is
 *            colored                                                 </TD>
 *  </TR>
 *  <TR>
 *    <TD>supported and unblocked                                     </TD>
 *    <TD>color node to plus
 *          - automatically color successor head node to plus
 *          - put all 0- and 1-successors of head node                </TD>
 *  </TR>
 *  <TR>
 *    <TD>weak_supported (one 0-predecessor is colored weak_plus)
 *        and unblocked                                               </TD>
 *    <TD>color node to weak_plus
 *          - automatically color successor head node to weak_plus
 *          - put all 0- and 1-successors of head node                </TD>
 *  </TR>
 *  <TR>
 *    <TD>supported and not unblocked                                 </TD>
 *    <TD>put node into choice point queue (possible choice point)    </TD>
 *  </TR>
 *  <TR>
 *    <TD>weak_supported and not unblocked                            </TD>
 *    <TD>put node into choice point queue (possible choice point)    </TD>
 *  </TR>
 *  </TABLE>
 */
class COperatorPs : public CDetOperator{

public:
  //! Constructor.
	COperatorPs();

  //! Destructor.
	virtual ~COperatorPs();

  //! Starts the operatorP which performs simple cyclic forward-propagation.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_ps;

  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {

    return type_coperator_ps;

  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorLABH                                                     *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorLABH coperator.h
 *  \brief Implements an improvement of the lookahead as operator.
 *  \author Andre Neumann
 *  \date Mon Dec 29 2004
 *
 */
class COperatorLABH : public CDetOperator {

public:
  //! Constructor.
	COperatorLABH();

  //! Destructor.
	virtual ~COperatorLABH();

  //! Starts the operatorLABH which performs lookahead as operator.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_labh;

  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {

    return type_coperator_labh;

  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorLABH                                                     *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorLAB coperator.h
 *  \brief Implements an improvement of the lookahead as operator.
 *  \author Andre Neumann
 *  \date Thu Jan 06 2005
 *
 */
class COperatorLAB : public CDetOperator {

public:
  //! Constructor.
	COperatorLAB();

  //! Destructor.
	virtual ~COperatorLAB();

  //! Starts the operatorLABH which performs lookahead as operator.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {

    return type_coperator_lab;

  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {

    return type_coperator_lab;

  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorN                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorN coperator.h
 *  \brief Deterministic operator N defined in paper.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperatorN colors all uncolored and weak_plus colored nodes (head and
 *  body nodes) to minus.
 */
class COperatorN : public CDetOperator {

public:
  //! Constructor.
	COperatorN();

  //! Destructor.
	virtual ~COperatorN();

  //! Starts the operatorN for color all uncolored nodes to minus.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_n;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_n;
    
  }

};

/***************************************************************************
 *                                                                         *
 * class COperatorNSecure                                                  *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorNSecure coperator.h
 *  \brief Deterministic operator N defined in paper. But do not color
 *         weak_plus node to minus. This will be give an error.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperatorNSecure colors all uncolored nodes (head and body nodes) to
 *  minus.
 */
class COperatorNSecure : public CDetOperator {

public:
  //! Constructor.
	COperatorNSecure();

  //! Destructor.
	virtual ~COperatorNSecure();

  //! Starts the operatorN colors all uncolored nodes to minus.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_nsecure;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_n;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorU                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorU coperator.h
 *  \brief Deterministic operator U defined in paper.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  The operator color all nodes which are not in the maximal support
 *  graph w.r.t. to the current coloring to minus.
 */
class COperatorU : public CDetOperator{

public:
  //! Constructor.
	COperatorU();

  //! Destructor.
	virtual ~COperatorU();

  //! Starts the operatorU.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_u;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_u;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorV                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorV coperator.h
 *  \brief Deterministic operator V defined in paper.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  The operator color all nodes which are not in the maximal support graph
 *  w.r.t. to the current coloring to minus.
 *
 *  -# Divide the body node set into three parts
 *     - all plus colored body and head nodes (PSET)
 *     - all minus colored body nodes (MSET)
 *     - all uncolored nodes (USET)
 *  -# iterate through all uncolored nodes (USET) until nothing more changes
 *     - if one node is supported relative to the plus colored set (PSET)
 *       then put this node and its successor head nodes into PSET and
 *       delete this body node from the uncolored set (USET)
 *  -# color all body nodes not in plus colored set (PSET) to minus
 *     - automatically color successor head nodes to minus IF all its
 *       predecessors are minus
 */
class COperatorV : public CDetOperator{
  
public:
  //! Constructor.
	COperatorV();

  //! Destructor.
	virtual ~COperatorV();

  //! Starts the operatorV.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_v;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_v;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorV2                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorV2 coperator.h
 *  \brief Deterministic operator for computing the maximal support graph
 *         like V.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  The operator color all nodes which are not in the maximal support graph
 *  w.r.t. to the current coloring to minus.
 *
 *  -# save all possible choices (choice point queue)
 *  -# call the OperatorMS for generating the maximal support graph (all
 *     now uncolored nodes not supported w.r.t. to the current coloring)
 *     - save color information on a new stack
 *  -# use OperatorN to color all uncolored and weak_plus colored node
 *     (body and head nodes) to minus
 *  -# restore the new stack used by OperatorMS
 */
class COperatorV2 : public CDetOperator{
  
public:
  //! Constructor.
	COperatorV2();

  //! Destructor.
	virtual ~COperatorV2();

  //! Starts the OperatorV2.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   *  \attention If you want to use this operator with lookahead out all
   *             possible choice points into todo queue for lookahead with
   *             plus and minus.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_v2;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_v2;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorV3                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorV3 coperator.h
 *  \brief Deterministic operator for computing the maximal support graph
 *         like V and V2.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  \todo Computes wrong answer sets for hamkX and queens examples.
 *  \todo Detailed comments.
 */
class COperatorV3 : public CDetOperator{

public:
  //! Constructor.
	COperatorV3();

  //! Destructor.
	virtual ~COperatorV3();

  //! Starts the OperatorV3.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_v3;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_v3;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorV4                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorV4 coperator.h
 *  \brief Deterministic operator for computing the maximal support graph
 *         like V and V2 (idea by Martin Gebser).
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  \todo Detailed comments.
 */
class COperatorV4 : public CDetOperator{

public:
  //! Constructor.
	COperatorV4();

  //! Destructor.
	virtual ~COperatorV4();

  //! Starts the OperatorV4.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_v4;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_v4;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorV5                                                       *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorV5 coperator.h
 *  \brief Deterministic operator for computing the maximal support graph
 *         like V, V2 and V5 (idea by Martin Gebser).
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  \todo Detailed comments.
 */
class COperatorV5 : public CDetOperator{

public:
  //! Constructor.
	COperatorV5();

  //! Destructor.
	virtual ~COperatorV5();

  //! Starts the OperatorV5.
  /*! \param graph  Graph on which the computation is started.
   *  \param stack  Stack on which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_v5;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_v5;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class COperatorPre                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class COperatorPre coperator.h
 *  \brief Deterministic operator Pre.
 *  \author Andre Neumann, Christian Anger
 *  \date Tue Jul 29 2003
 *
 *  COperatorPre iterates through all body nodes and
 *  -# colors all selfblocker to minus and
 *     - automatically color successor head nodes to minus IF all its
 *       predecessors are minus
 *  -# colors all facts (body nodes without 0- and 1-predecessors) to plus.
 *     - automatically color successor head nodes to plus
 *  -# put supported body nodes into choice point queue (possible choice
 *     points)
 */
class COperatorPre : public CDetOperator{

public:
  //! Constructor.
	COperatorPre();

  //! Destructor.
	virtual ~COperatorPre();

  //! Colors all self-blocker minus and all facts plus. 
  /*! \post Stack is empty.
   *  \param graph  Graph on which the computation is started.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \param stack  Stack on which the changes are stored.
   *  \return If a node is colored with two different colors the methode
   *          returns return_colorerror else if the operator doesn't
   *          changed anything the methode returns return_unchanged else
   *          return_changed.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_coperator_pre;
    
  }

  //! Returns the operator id like GetType. Important for conflict testing.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   */
  inline virtual TTypeEnum GetOperatorID() {
    
    return type_coperator_pre;
    
  }
  
};


/***************************************************************************
 *                                                                         *
 * class dependencies                                                      *
 *                                                                         *
 ***************************************************************************/
class CNdOperator;
class CHeuristic;

/***************************************************************************
 *                                                                         *
 * generic stuff                                                           *
 *                                                                         *
 ***************************************************************************/
//! Generates the operators.
/*! Generates the aggregation tree and the other deterministic and
 *  non-deterministic operator out of the input string. Also a specified
 *  heuristic is returned.
 *  \param op            Input string which specifies the operators
 *                       ("Pre:Choice:Det:Closure:Heuristic").
 *  \param lookahead_op  Input string which specifies the deterministic
 *                       operator for lookahead.
 *  \param det_out       Deterministic output operator.
 *  \param pre_out       Deterministic output operator for pre processing.
 *  \param ndet_out      Non-deterministic output operator.
 *  \param end_out       Deterministic output operator for generating a
 *                       total coloring.
 *  \param heu_out       Heuristics output for sorting the choice point
 *                       list.
 *  \param lookahead_out Deterministic output operator for computing
 *                       lookahead.
 *  \return If the input string syntax is ok and the created operators are
 *          not in conflict the methode return true else false and all
 *          output parameters are set to NULL.
 */
bool GenerateOperator(string op,
                      string lookahead_op,
                      CDetOperator*& det_out,
                      CDetOperator*& pre_out,
                      CNdOperator*& ndet_out,
                      CDetOperator*& end_out,
                      CHeuristic*& heu_out,
                      CDetOperator*& lookahead_out);

//! Tests if the input operators are in conflict.
/*! \todo tests for other ndet-operators
 */
bool ConflictOperators(TTypeEnum det_ops,
                       CNdOperator *ndet,
                       CDetOperator *end);

} // end of namespace NS_NOMORE

#endif
