/***************************************************************************
                          crule.h  -  description
                             -------------------
    begin                : Thu Jul 10 2003
    copyright            : (C) 2003 by nomoe-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file crule.h
 *  Contains Definitions of CAtom, CBody and CRule.
 */

#ifndef CRULE_H
#define CRULE_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <map>
#include <iostream>
#include "misc.h"
#include "print.h"
#include "typedefinitions.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE{

class CProgram;

/***************************************************************************
 *                                                                         *
 * class CAtom                                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CAtom
 *  \brief Defines atoms.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jul 10 2003
 *
 *  An atom consists of a string (the name) and an id.
 */
class CAtom {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CAtom& atom);

private:
  //! Contains the name of the atom as found in the logic program.
  string _name;

  //! Contains an id to uniquely identify each atom.
  TId _id;

  //! color after parsed by lparse.
  TColor _color; 

public:
  //! Constructs the atom and initialize the name.
  CAtom(const TId id,
        const string name = "");

  //! Constructs the atom and initialize the name.
  CAtom(CIdGenerator *ig,
        const string name = "");

  //! Destroys the object.
  ~CAtom();

  //! Equality operator on atoms (tests for equal id's).
  inline bool operator==(const CAtom& atom) const {
    
    return _id == atom._id;
    
  }

  //! Inequality operator on atoms (tests for equal id's).
  inline bool operator!=(const CAtom& atom) const {
    
    return !((*this) == atom);
    
  }

  //! Comparison operator on atoms for alphabetical sorting.
  /*! \pre The name of the atom is set.
   */
  inline bool operator<(const CAtom& atom) const {
    
    return _name < atom._name;
    
  }

  //! Sets name attribute.
  /*! \param name Sets the new name of the atom.
   */
  inline void SetName(const string name) {

    _name = name;
    
  }

	//! Returns name attribute.
  /*! \pre The name is set previously.
   */
  inline string GetName() const {

    return _name;
    
  };

	//! Sets id attribute.
  /*! \param id The new id.
   */
  inline void SetId(const TId id) {

    _id = id;
    
  }

	//! Returns id attribute.
  inline TId GetId() const {
    
 	 return _id;
   
  };

  //! Returns the color generated by lparse.
  inline TColor GetColor() const {
    
  	return _color;
   
  }

  //! Sets the color generated by lparse.
  /*! \param col Color which is generated by lparse in B+ or B- section.
   */
  inline void SetColor(TColor col) {
    
  	_color = col;
   
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CBody                                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CBody
 *  \brief Defines a body for a rule consisting of two atom set.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jul 10 2003
 *
 *  The positve part corresponds to the first element of the std::pair and
 *  the negative part corresponds to the second element of the std::pair.
 */
class CBody {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CBody& body);

private:
  //! Identifier of the body.
  TId _id;
  
  //! Positive atoms.
  TIdSet _positives;
  
  //! Negative atoms.
  TIdSet _negatives;
  
  //! References to rules with this body.
  TRuleVector _rules;
  
  //! Reference to the program.
  CProgram *_pprogram;

public:
  //! Constructes the body and sets the id.
  CBody(const TId id, CProgram* prg);

  //! Constructes the body and sets the id by the id-generator.
  CBody(CIdGenerator *ig, CProgram* prg);

  //! Destroys the object.
  virtual ~CBody();

  //! Tests if the body is supported by the input set of atoms.
  /*! This methode is used to generate the maximal support-graph of the
   *  program.
   *  \param atoms Set of atoms to which the rule is supported.
   *  \return Returns true if this body is supported by the input atoms.
   */
  bool Supported(const TIdSet &atoms);

  //! Returns the pointer to the positive part of the body.
  const TIdSet& GetPosAtoms() const {
    return _positives;
  }
  
  TIdSet::size_type GetPosAtomSize() {
    return _positives.size();
  }

  //! Returns the pointer to the negative part of the body.
  const TIdSet& GetNegAtoms() const {
    return _negatives;
  }

  TIdSet::size_type GetNegAtomSize() {
    return _negatives.size();
  }
  
  //! Inserts a positive atom by copying the pointer.
  void InsertPosAtom(TId atom) {
    _positives.insert(atom).second;
  }

  //! Inserts a negative atom by copying the pointer.
  void InsertNegAtom(TId atom) {
    _negatives.insert(atom).second;
  }

  //! Equality operator on bodies.
  inline bool operator==(const CBody& body) const {
    return _positives == body._positives && _negatives == body._negatives;       
  }

  //! Inequality operator on bodies.
  inline bool operator!=(const CBody& body) const {
    return !(*this == body);
  }

  //! Comparison operator on bodies for alphabetical sorting.
  bool operator<(const CBody& body) const {
    return _positives < body._positives ||
           !(body._positives < _positives) && _negatives < body._negatives;
  }

  //! Returns the id.
  inline TId GetId() const {
    
  	return _id;
   
  }

  //! Sets the id.
  inline void SetId(const TId id) {
    
  	_id = id;
   
  }
  
  void SetProgram(CProgram* prg) {
    _pprogram = prg;
  }
  
  CProgram* GetProgram() {
    return _pprogram;
  }
  
  void AddRule(CRule *r) {
    _rules.push_back(r);
  }
  
  const TRuleVector& GetRules() const {
    return _rules;
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CRule                                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CRule
 *  \brief Defines a simple rule of the form
 *           (form) head :- a1, .. , an, not an+1, .. , not am
 *         using CBody's.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jul 10 2003
 */
class CRule {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CRule& rule);

private:
  //! Set of all heads of the rule.
  CAtom *_phead;

  //! Set of all bodies of the rule.
  CBody *_pbody;

  //! Identifier of the rule.
  TId  _id;

public:
  //! Constructor the rule by setting the id.
  CRule(const TId id);

  //! Constructor the rule by setting the id.
  CRule(CIdGenerator *ig);

  //! Destructor.
  ~CRule();

  //! Equality operator on rules.
  inline bool operator==(const CRule& rule) const {
    return *_phead == *_phead && *_pbody == *_pbody;
  }

  //! Inequality operator on rules.
  inline bool operator!=(const CRule& rule) const {
    return !(operator==(rule));
  }

  //! Inserts an atom to heads.
  /*! \param atom Inserts the address of the atom to the set of heads.
   */
  inline void SetHead(CAtom* atom) {
    CHECK_POINTER("CRule::InsertHead(atom)", atom);
    _phead = atom;    
  }

  //! Inserts a body to the body set.
  /*! \param body Inserts the address of the body.
   */
  inline void SetBody(CBody* body) {
    CHECK_POINTER("CRule::InsertBody(body)", body);
    _pbody = body;
  }

  //! Returns the ID of the rule.
  inline TId GetId() const {
    return _id;
  };

  //! Sets the ID of the rule.
  inline void SetId(TId id) {
    _id = id;
  };

  //! Returns the set of all heads.
  inline CAtom* GetHead() {
    return _phead;
  };

  //! Returns the set of all heads.
  inline CBody* GetBody() {
    return _pbody;
  };
  
};

} // end of namespace NS_NOMORE

#endif
