/***************************************************************************
                          ctodo.h  -  description
                             -------------------
    begin                : Fri Dec 19 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CTODO_H
#define CTODO_H

/*! \file ctodo.h
 *  Contains the definition of CComputeNode, CColorNode, CTodo, CJumping
 *  and CIgnore.
 */

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "ctodolists.h"
#include "misc.h"
#include "cgraph.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class dependencies                                                      *
 *                                                                         *
 ***************************************************************************/
class CBodyNode;
class CHeadNode;
class CGraph;

/***************************************************************************
 *                                                                         *
 * class CComputeNode                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CComputeNode ctodo.h
 *  \brief Base class for CIgnore, CTodo and CJumping.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *
 *  If for example ignore is disabled the CColorNode::_pignore member is
 *  instanciated with this class otherwise it is an instance of the CIgnore
 *  class.
 *  \todo Detailed comments.
 */
class CComputeNode {

public:
  //! Constructor.
	CComputeNode();

  //! Destructor.
	virtual ~CComputeNode();

  //! Do nothing.
  inline virtual bool Compute(CNode* node,
                              TColor old_color,
                              TColor new_color,
                              CGraph* grp,
                              CStack* stack,
                              CTodoLists* todo,
                              CStatusInformation *status = NULL,
                              bool secure = true) {

    CHECK_POINTER("CComputeNode::Compute(node)" , node);
    CHECK_POINTER("CComputeNode::Compute(grp)"  , grp);
    CHECK_POINTER("CComputeNode::Compute(stack)", stack);
    CHECK_POINTER("CComputeNode::Compute(todo)" , todo);

    return true;
    
  }

  //! Returns the type of the object.
  inline virtual TType GetType() const {

    return type_cn_none;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CTodo                                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CTodo ctodo.h
 *  \brief Class for putting all 0- and 1-successors of a head node into
 *         the TodoList.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *  \todo Detailed comments.
 */
class CTodo : public CComputeNode {
  
public:
  //! Constructor.
	CTodo();

  //! Destructor.
	virtual ~CTodo();

  //! Inserts the node into the todo list.
  inline virtual bool Compute(CNode* node,
                              TColor old_color,
                              TColor new_color,
                              CGraph* grp,
                              CStack* stack,
                              CTodoLists* todo,
                              CStatusInformation *status = NULL,
                              bool secure = true) {

    CHECK_POINTER("CTodo::Compute(node)" , node);
    CHECK_POINTER("CTodo::Compute(grp)"  , grp);
    CHECK_POINTER("CTodo::Compute(stack)", stack);
    CHECK_POINTER("CTodo::Compute(todo)" , todo);

    INC_COUNT(queue_todo_call);

//    todo->PushProp(node);
    return true;
      
  }

  //! Returns the type of the object.
  inline virtual TType GetType() const {
    
    return type_cn_todo;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CJumping                                                          *
 *                                                                         *
 ***************************************************************************/
/*! \class CJumping ctodo.h
 *  \brief Class for inserting a node into back propagation list.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *  \todo Detailed comments.
 */
class CJumping : public CComputeNode  {

public:
  //! Constructor.
	CJumping();

  //! Destructor.
	virtual ~CJumping();

  //! Insert the node into the back-propagation list.
  inline virtual bool Compute(CNode* node,
                              TColor old_color,
                              TColor new_color,
                              CGraph* grp,
                              CStack* stack,
                              CTodoLists* todo,
                              CStatusInformation *status = NULL,
                              bool secure = true) {

    CHECK_POINTER("CJumping::Compute(node)" , node);
    CHECK_POINTER("CJumping::Compute(grp)"  , grp);
    CHECK_POINTER("CJumping::Compute(stack)", stack);
    CHECK_POINTER("CJumping::Compute(todo)" , todo);

    INC_COUNT(queue_jumping_call);

    todo->PushBackProp(node);
    return true;
    
  }

  //! Returns the type of the object.
  inline virtual TType GetType() const {
    
    return type_cn_jumping;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CIgnore                                                           *
 *                                                                         *
 ***************************************************************************/
/*! \class CIgnore ctodo.h
 *  \brief Class for color some rules to the ignore color.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *  \todo Detailed comments.
 */
class CIgnore : public CComputeNode  {

public:
  //! Constructor.
	CIgnore();

  //! Destructor.
	virtual ~CIgnore();

  //! Try to color a with the ignore color.
  inline virtual bool Compute(CNode* node,
                              TColor old_color,
                              TColor new_color,
                              CGraph* grp,
                              CStack* stack,
                              CTodoLists* todo,
                              CStatusInformation *status = NULL,
                              bool secure = true) {

    CHECK_POINTER("CIgnore::Compute(node)" , node);
    CHECK_POINTER("CIgnore::Compute(grp)"  , grp);
    CHECK_POINTER("CIgnore::Compute(stack)", stack);
    CHECK_POINTER("CIgnore::Compute(todo)" , todo);

    if(grp->ColorNode(node,
                      color_ignore,
                      stack,
                      false,
                      color_none,
                      todo,
                      status,
                      secure)) {

      INC_COUNT(count_ignore);
      return true;
      
    }

    return false;
    
  }

  //! Returns the type of the object.
  inline virtual TType GetType() const {
    
    return type_cn_ignore;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CColorNode                                                        *
 *                                                                         *
 ***************************************************************************/
/*! \class CColorNode ctodo.h
 *  \brief Class for handle successors and predecessors of a colored node.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Jul 21 2003
 *  \todo Detailed comments.
 */
class CColorNode {

private:
  //! pointer to a ignore object.
  CComputeNode *_pignore;

public:
  //! Constructor.
	CColorNode();

  //! Destructor.
	virtual ~CColorNode();

  //! Enables the todo, jumping and/or ignore object.
  void Enable(TTypeEnum _type);

  //! Disables the todo, jumping and/or ignore object.
  void Disable(TTypeEnum _type);

  //! Get enabled status.
  TTypeEnum GetEnabled() const;

  //! Returns the ignore object.
  inline CComputeNode* GetIgnore() {
    
    return _pignore;
    
  }

  //! Computes the 0-successors of the node. 
  /*! This method take all 0-successors of a head node and try to do
   *  something:
   *  -# insert successor body node into todo queue used for operator Ps
   *  -# if head node is colored plus decrease the supported counter of the
   *     successor body node
   *     - if the old color was undefined also decrease the weak_supported
   *       counter
   *  -# if head node is colored plus (or weak_plus) and the successor body
   *     node is colored minus insert the successor node into the jumping
   *     queue (Jumping)
   *  -# if head node is colored weak_plus decrease the weak_supported
   *     counter of the successor body node
   *  -# if head node is colored minus decrease the unsupported counter of
   *     the successor body node
   *     - if the old color of the head node was weak_plus increase the
   *       weak_supported counter of the successor body node
   *  -# if head node is colored ignore and the successor body node is
   *     color weak_plus increase the weak_supported counter of the
   *     successor body node
   */
  bool HandleHeadZeroSuccessor(CHeadNode* node,
                               TColor old_color,
                               TColor new_color,
                               CGraph* grp,
                               CStack* stack,
                               CTodoLists* todo,
                               CStatusInformation *status = NULL,
                               bool secure = true);

  //! Computes the 1-successors of the node. 
  /*! This method take all 1-successors of a head node and try to do
   *  something:
   *  -# insert successor body node into todo queue used for operator Ps
   *  -# if head node is colored plus (or weak_plus) decrease the blocked
   *     counter of the successor body node
   *  -# if head node is colored minus decrease the unblocked counter of
   *     the successor body node
   *     - if the old color of the head node was weak_plus increase the
   *       blocked counter of the successor body node
   *  -# if head node is colored minus and the successor body node is color
   *     minus insert the successor node into jumping queue (Jumping)
   */
  bool HandleHeadOneSuccessor(CHeadNode* node,
                              TColor old_color,
                              TColor new_color,
                              CGraph* grp,
                              CStack* stack,
                              CTodoLists* todo,
                              CStatusInformation *status = NULL,
                              bool secure = true);

  //! Computes the predecessors of the node. 
  /*! This method take all predecessors of a head node and try to do
   *  something:
   *  -# if the head node is colored plus or minus decrease the
   *     all_succs_colored counter used for Ignore
   *     - if predecessor body node is can be ignored (all_succs_colored
   *       counter is zero) try to color to ignore
   */
  bool HandleHeadPredecessor(CHeadNode* node,
                             TColor old_color,
                             TColor new_color,
                             CGraph* grp,
                             CStack* stack,
                             CTodoLists* todo,
                             CStatusInformation *status = NULL,
                             bool secure = true);

  //! Computes the 0-predecessors of the node. 
  /*! This method take all 0-predecessors of a body node and try to do
   *  something:
   *  -# do nothing/not used
   */
  bool HandleBodyZeroPredecessor(CBodyNode* node,
                                 TColor old_color,
                                 TColor new_color,
                                 CGraph* grp,
                                 CStack* stack,
                                 CTodoLists* todo,
                                 CStatusInformation *status = NULL,
                                 bool secure = true);

  //! Computes the 1-predecessors of the node. 
  /*! This method take all 1-predecessors of a body node and try to do
   *  something:
   *  -# do nothing/not used
   */
  bool HandleBodyOnePredecessor(CBodyNode* node,
                                TColor old_color,
                                TColor new_color,
                                CGraph* grp,
                                CStack* stack,
                                CTodoLists* todo,
                                CStatusInformation *status = NULL,
                                bool secure = true);

  //! Computes the successors of the node. 
  /*! This method take all successors of a body node and try to do
   *  something:
   *  -# if body node is colored plus (or weak_plus) try to color successor
   *     (head node) to plus (or weak_plus)
   *  -# if body node is colored minus decrease unsupported counter and if
   *     this counter is zero color the successor head node to minus
   *  -# if body node is colored minus and the successor head node is plus
   *     or weak_plus insert successor node into jumping queue (Jumping)
   */
  bool HandleBodySuccessor(CBodyNode* node,
                           TColor old_color,
                           TColor new_color,
                           CGraph* grp,
                           CStack* stack,
                           CTodoLists* todo,
                           CStatusInformation *status = NULL,
                           bool secure = true);
                           
};

} // NS_NOMORE

#endif
