/***************************************************************************
                          ctodolists.h  -  description
                             -------------------
    begin                : Fri Dec 19 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CTODOLISTS_H
#define CTODOLISTS_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "misc.h"
#include "typedefinitions.h"
#include "cnode.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * CTodoLists                                                              *
 *                                                                         *
 ***************************************************************************/
/*! \class CTodoLists ctodolists.h
 *  \brief Class which holds different lists like todo list for P*
 *         operator or lists for back-propagation.
 *  \author Andre Neumann
 *  \date Wed Nov 10 2004
 */
class CTodoLists {

  //! Output operator.
	friend std::ostream& operator<<(std::ostream& os, CTodoLists& todo);

private:
  //! Todo list for COperatorPs.
  TBodyNodeQueue _prop;

  //! Back-propagation list for COperatorBP.
  TQueue _backprop;

  //! Minus colored list for COperatorV3.
  TQueue _minus;

  //! List of possible choice points for supportdriven non-deterministic operators.
  TQueue _choices;

  //! List of possible choice points for non-deterministic operators.
  TQueue _c_choices;

public:
  //! Constructor.
	CTodoLists();

  //! Destructor.
	virtual ~CTodoLists();

  //! Clears all lists.
  void Clear();

  bool EmptyProp() {
    return _prop.empty();
  }

  bool EmptyBackProp() {
    return _backprop.empty();
  }

  bool EmptyMinus() {
    return _minus.empty();
  }

  bool EmptyPossChoices() {
    return _choices.empty();
  }

  bool EmptyCPossChoices() {
    return _c_choices.empty();
  }

  void ClearProp()  {
    while(!EmptyProp()) {
      PopProp();
    }
  }

  void ClearBackProp()  {
    while(!EmptyBackProp()) {
      PopBackProp();
    }
  }

  void ClearMinus()  {
    while(!EmptyMinus()) {
      PopMinus();
    }
  }

  void ClearPossChoices()  {
    while(!EmptyPossChoices()) {
      PopPossChoices();
    }
  }

  void ClearCPossChoices()  {
    while(!EmptyCPossChoices()) {
      PopCPossChoices();
    }
  }

  CBodyNode* PopProp() {
    if(EmptyProp())
      return NULL;
    CBodyNode *n = _prop.front();
    _prop.pop();
    n->SwitchIsInPropagationQueue(false);
    return n;
  }

  CNode* PopBackProp() {
    if(EmptyBackProp())
      return NULL;
    CNode *n = _backprop.front();
    _backprop.pop();
    n->SwitchIsInBackwardPropagationQueue(false);
    return n;
  }

  CNode* PopMinus() {
    if(EmptyMinus())
      return NULL;
    CNode *n = _minus.front();
    _minus.pop();
    n->SwitchIsInMinusQueue(false);
    return n;
  }

  CNode* PopPossChoices() {
    if(EmptyPossChoices())
      return NULL;
    CNode *n = _choices.front();
    _choices.pop();
    n->SwitchIsInPossibleChoicesQueue(false);
    return n;
  }

  CNode* PopCPossChoices() {
    if(EmptyCPossChoices())
      return NULL;
    CNode *n = _c_choices.front();
    _c_choices.pop();
    n->SwitchIsInCPossibleChoicesQueue(false);
    return n;
  }

  void PushProp(CBodyNode* n) {
    if(!n->IsInPropagationQueue()) {      
      _prop.push(n);
      n->SwitchIsInPropagationQueue(true);
    }
  }

  void PushBackProp(CNode* n) {
    if(!n->IsInBackwardPropagationQueue()) {
      _backprop.push(n);
      n->SwitchIsInBackwardPropagationQueue(true);
    }
  }

  void PushMinus(CNode* n) {
    if(!n->IsInMinusQueue()) {
      _minus.push(n);
      n->SwitchIsInMinusQueue(true);
    }
  }

  void PushPossChoices(CNode* n) {
    if(!n->IsInPossibleChoicesQueue()) {
      _choices.push(n);
      n->SwitchIsInPossibleChoicesQueue(true);
    }
  }

  void PushCPossChoices(CNode* n) {
    if(!n->IsInCPossibleChoicesQueue()) {
      _c_choices.push(n);
      n->SwitchIsInCPossibleChoicesQueue(true);
    }
  }

/*  void PushProp(TNodeSet* nodeset) {
    TNodeSet::iterator ite;
    for(ite=nodeset->begin();ite!=nodeset->end();ite++) {
      PushProp(ite->second);
    }
  }*/

  void PushBackProp(TNodeSet* nodeset) {
    TNodeSet::iterator ite;
    for(ite=nodeset->begin();ite!=nodeset->end();ite++) {
      PushBackProp(ite->second);
    }
  }

  void PushMinus(TNodeSet* nodeset) {
    TNodeSet::iterator ite;
    for(ite=nodeset->begin();ite!=nodeset->end();ite++) {
      PushMinus(ite->second);
    }
  }

  void PushMinus(THeadNodeSet* nodeset) {
    THeadNodeSetIterator ite;
    THeadNodeSetIterator end = nodeset->end();    
    for(ite=nodeset->begin();ite!=end;ite++) {
      PushMinus(*ite);
    }
  }

  void PushMinus(TBodyNodeSet* nodeset) {
    TBodyNodeSetIterator ite;
    TBodyNodeSetIterator end = nodeset->end();    
    for(ite=nodeset->begin();ite!=end;ite++) {
      PushMinus(*ite);
    }
  }

  void PushPossChoices(TNodeSet* nodeset) {
    TNodeSet::iterator ite;
    for(ite=nodeset->begin();ite!=nodeset->end();ite++) {
      PushPossChoices(ite->second);
    }
  }

  void PushCPossChoices(TNodeSet* nodeset) {
    TNodeSet::iterator ite;
    for(ite=nodeset->begin();ite!=nodeset->end();ite++) {
      PushCPossChoices(ite->second);
    }
  }

  TBodyNodeQueue *GetPropQueue() {
    return &_prop;
  }

  TQueue *GetBackPropQueue() {
    return &_backprop;
  }

  TQueue *GetMinusQueue() {
    return &_minus;
  }

  TQueue *GetPossChoicesQueue() {
    return &_choices;
  }

  TQueue *GetCPossChoicesQueue() {
    return &_c_choices;
  }

};

}; // NS_NOMORE

#endif
