/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Thu Jun 19 15:32:55 MET DST 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define MAIN
//#define SHOW_ALPHABETICAL_SORT
// also enable ASOUTPUT in print.h
#ifdef MAIN


/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <iostream>
#include <fstream>
#include <stdio.h>
#include <unistd.h>
#include "algorithm.h"
#include "cgraph.h"
#include "cprogram.h"
#include "timer.h"
#include "print.h"
#include "io.h"
#include "ctodo.h"
#include "ctodolists.h"

/***************************************************************************
 *                                                                         *
 * Help functions                                                          *
 *                                                                         *
 ***************************************************************************/
//! Converts boolean into string (true/false).
std::string Boolean2String(bool b) {
  if(b)
    return std::string("true");
  else
    return std::string("false");
}

//! Enables the alphabetical output of the body nodes.
#ifdef SHOW_ALPHABETICAL_SORT
void ShowOrdering(CGraph *grp) {

  CHECK_POINTER("ShowOrdering(grp)", grp);
  
  TNodeSet *bnodes = grp->GetBodyNodes();
  TNodeSet::iterator ite;

  CPriorityQueue queue;

  // put all nodes into priority queue for sorting
  for(ite=bnodes->begin();ite!=bnodes->end();ite++)
    queue.Insert(ite->second);

  while(!queue.Empty()) {
    
    CBodyNode* node = (CBodyNode*)queue.GetHead();

    cout << *node << endl;
    
  }
  
}
#endif

/***************************************************************************
 *                                                                         *
 * Usage output                                                            *
 *                                                                         *
 ***************************************************************************/
//! print out usage of noMoRe++.
void usage () {
  std::cerr << "Usage: nomore [number] [--nolookahead] [--lookahead-b] [--nobackprop]"       << std::endl
            << "              [--novoperator] [--choiceoperator] [--show-settings"           << std::endl
            << "              [--operators \"nd:d:post\"]"                                   << std::endl
            << "  The number determines the number of stable models computed."               << std::endl
            << "  A zero indicates all."                                                     << std::endl            
            << "    --help | -h            This ouput."                                      << std::endl
            << "    --nolookahead | -l     Do not use any lookahead."                        << std::endl
            << "    --lookahead-b | -lb    Use lookahead only for bodynodes instead for "    << std::endl
            << "                           body- and headnodes."                             << std::endl
            << "    --nobackprop | -b      Do not use backward propagation."                 << std::endl
            << "    --novoperator | -v     Do not use V-operator for maximal supportedness." << std::endl
            << "    --choiceoperator | -c  Use the C-operator for choosing any undecided "   << std::endl
            << "                           bodynodes instead of the D-operator which choose" << std::endl
            << "                           only supported undecided bodynodes."              << std::endl
            << "    --show-settings | -s   Shows the settings for the nomore system used for"<< std::endl
            << "                           current computation."                             << std::endl
            << "    --operators | -op      Enables choosing of an own operator where "       << std::endl
            << "                           nd    is the nondeterministic operator out of the"<< std::endl
            << "                                 following:  {C, D}."                        << std::endl
            << "                           d     is the deterministic operator out of the "  << std::endl
            << "                                 following: {P, B, V, U, LAB, LABH, None} or"<< std::endl
            << "                                 an aggregation of the above using "         << std::endl
            << "                                 sequence (,) and iteration (*)."            << std::endl
            << "                           post  is the postprocessing operator out of the " << std::endl
            << "                                 following: {N, None}."                      << std::endl
            << "                           if obmitted, D:(((P,B)*,V)*,LABH):None is"        << std::endl
            << "                           used. This option overrides all previous declared"<< std::endl
            << "                           options."                                         << std::endl;
}

/***************************************************************************
 *                                                                         *
 * MAIN                                                                    *
 *                                                                         *
 ***************************************************************************/
//! Main application as command line program.
int main(int argc, char *argv[]) {

  // get randomized (current time in seconds) seed for CHeuristic{S/D}R
  int seed = time(NULL); 

/*
correctness for:

   1. -c -l -b 
   2.    -l -b -v
   3. -c -l   
   4. -c    -b 
   5.    -l -b 
   6.    -l    -v 
   7.       -b -v
   8. -c   
   9.    -l 
  10,       -b 
  11.          -v 
  12.         
  
*/

//  1: OK std::string operators = "Pre:C:(P,U)*:None:None";
//  2: OK std::string operators = "Pre:D:P:N:None";
//  3: OK std::string operators = "Pre:C:((P,B)*,U)*:None:None";
//  4: OK std::string operators = "Pre:C:((P,U)*,LABH):None:None";
//  5: OK std::string operators = "Pre:D:(P,V)*:None:None";
//  6: OK std::string operators = "Pre:D:(P,B)*:N:None";
//  7: ERROR for clumpy_4_4_3.lp (Vermutung: V im lookahead) : std::string operators = "Pre:D:(P,LABH):N:None";
//  8: OK std::string operators = "Pre:C:(((P,B)*,U)*,LABH):None:None";
//  9: OK std::string operators = "Pre:D:((P,B)*,V)*:None:None";
// 10: OK std::string operators = "Pre:D:((P,V)*,LABH):None:None";
// 11: ERROR for hamk6.lp (Vermutung: V im lookahead) : std::string operators = "Pre:D:((P,B)*,LABH):N:None";
// 12: OK std::string operators = "Pre:D:(((P,B)*,V)*,LABH):None:None";

  // operators used as standard
  std::string operators = "Pre:D:(((P,B)*,V)*,LABH):None:None";
  std::string lookahead_operators = "((P,B)*,V)*"; // unused
  
  bool nolookahead = false;
  bool lookahead_b = false;
  bool nobackprop = false;
  bool novoperator = false;
  bool choiceoperator = false;
  bool show_settings = false;
  bool overide_operator = false;
  
  // error in arguments found
  bool error = false;
  
  // standard is one answer set
  int as_number = 1;

  
  // create all todo lists (for operator-Ps, operator-BP and for
  // possible choice points)
  NS_NOMORE::CTodoLists *todo_lists = new NS_NOMORE::CTodoLists();

  // get information by arguments
  for(int c=1;c<argc;c++) {
    if (argv[c][0] == '-') {

      if(strcmp(&argv[c][1], "op") == 0 || strcmp(&argv[c][1], "-operators") == 0) {
        c++;
        if(c >= argc) {
          error = true;
          std::cerr << "ERROR: Using option --operators or -op with the follwing operator string!" << std::endl;
          break;
        }
       
        operators = "Pre:" + std::string(argv[c]) + ":None"; // add heuristics and preprocessing type
        overide_operator = true;
        
      } else
      if(strcmp(&argv[c][1], "l") == 0 || strcmp(&argv[c][1], "-nolookahead") == 0) {
        nolookahead = true;
      } else
      if(strcmp(&argv[c][1], "lb") == 0 || strcmp(&argv[c][1], "-lookahead-b") == 0) {
        lookahead_b = true;
      } else
      if(strcmp(&argv[c][1], "b") == 0 || strcmp(&argv[c][1], "-nobackprop") == 0) {
        nobackprop = true;
      } else
      if(strcmp(&argv[c][1], "v") == 0 || strcmp(&argv[c][1], "-novoperator") == 0) {
        novoperator = true;
      } else
      if(strcmp(&argv[c][1], "c") == 0 || strcmp(&argv[c][1], "-choiceoperator") == 0) {
        choiceoperator = true;
      } else
      if(strcmp(&argv[c][1], "s") == 0 || strcmp(&argv[c][1], "-show-settings") == 0) {
        show_settings = true;
      } else 
      if(strcmp(&argv[c][1], "h") == 0 || strcmp(&argv[c][1], "-help") == 0) {
        error = true;
        break;
      } else {
        // undefined option
        std::cerr << "ERROR: Unknown parameter "<< argv[c] << "!" << std::endl;
        error = true;
      }      
    } else
    // set number of arguments (0 = all answer sets)
    if(0 == sscanf(argv[c], "%i", &as_number)) {
      error = true;
      std::cerr << "ERROR: Wrong parameter "<< argv[c] << "!" << std::endl;
      break;        
    }
  }

  // if any error occured show the usage and exit the program
  if(error) {
    usage();
    return 1;
  }

  if(!overide_operator) {

    std::string vo = "V";    
    std::string co = "D";
    if(choiceoperator) {
      vo = "U";
      co = "C";
      novoperator = false;
    }
  
    std::string la = "LABH";
    if(lookahead_b)
      la = "LAB";  
        
    if(nolookahead) {
      if(novoperator) {
        if(nobackprop)          
          operators = "Pre:"+co+":P:N:None";
        else
          operators = "Pre:"+co+":(P,B)*:N:None";
      } else {
        if(nobackprop)          
          operators = "Pre:"+co+":(P,"+vo+")*:None:None";
        else
          operators = "Pre:"+co+":((P,B)*,"+vo+")*:None:None";
      }
    } else {
      if(novoperator) {
        if(nobackprop)          
          operators = "Pre:"+co+":(P,"+la+"):N:None";
        else
          operators = "Pre:"+co+":((P,B)*,"+la+"):N:None";
      } else {
        if(nobackprop)          
          operators = "Pre:"+co+":((P,"+vo+")*,"+la+"):None:None";
        else
          operators = "Pre:"+co+":(((P,B)*,"+vo+")*,"+la+"):None:None";
      }
    }           
  }

  // output for more information of the usage
  if(show_settings) {
    
    std::cout << "--operators (-op)             : " << operators << std::endl;
    std::cout << "--nolookahead (-l)            : " << Boolean2String(nolookahead) << std::endl;
    std::cout << "--lookahead-b (-lb)           : " << Boolean2String(lookahead_b) << std::endl;
    std::cout << "--novoperator (-v)            : " << Boolean2String(novoperator) << std::endl;
    std::cout << "--nobackprop (-b)             : " << Boolean2String(nobackprop) << std::endl;
    std::cout << "--choiceoperator (-c)         : " << Boolean2String(choiceoperator) << std::endl;   
    std::cout << "number of answer sets to find : " << as_number << std::endl;
  }

  // set seed
  srand(seed);

  // start timer
  CTimer timer;
  timer.Start();
  
  std::cout << PACKAGE << " " << VERSION << " Reading ... ";  
  // read program from stdin	
  NS_NOMORE::CProgram *program = new NS_NOMORE::CProgram();
  if (!program->Read(/*file*/std::cin)) {
    std::cerr << "Something wrong with reading lparse output." << std::endl;
    delete program;
    return 1;
  }
  std::cout << "done" << std::endl;

  // set todo, ignore and jumping
  NS_NOMORE::TTypeEnum status = 0;
  status |= NS_NOMORE::type_cn_todo  | NS_NOMORE::type_cn_ignore | NS_NOMORE::type_cn_jumping;
  
  // generate graph out of the program
  NS_NOMORE::CGraph *graph = new NS_NOMORE::CGraph(program, todo_lists);
//  std::cout << *graph << std::endl;
  graph->GetColorNode()->Enable(status);

  // start color the graph  
  if(!ColorGraph(graph, todo_lists, operators, as_number, lookahead_operators, false, choiceoperator)) {
    usage();
    delete graph;
    delete program;
    return 1;
  }

  // stop timer and print summary
  timer.Stop();

  std::cout << "Number of atoms: " << program->GetAtoms().size() << std::endl;
  std::cout << "Number of rules: " << program->GetRules().size() << std::endl;
  std::cout << "Number of body nodes: " << graph->GetBodyNodes()->size() << std::endl;
  std::cout << "Number of head nodes: " << graph->GetHeadNodes()->size() << std::endl;
	std::cout << "Time: " << timer.Print() << std::endl;
	
  delete graph;
  delete program;
  delete todo_lists;

  PRINT_ALL_COUNTER;
}

#endif
