#ifndef LIBNOMORE_OPERATOR_BASED_HEURISTIC_H
#define LIBNOMORE_OPERATOR_BASED_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <heuristic.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
namespace NS_NOMORE {

//! base class for operator based heuristics
/*!
 * An operator based heuristic uses the result of an operator as heuristic value
 * and selects the best choice point based on this value.
 * 
 *
 * \par Template parameter: \n
 * - HeuStrat: The policy (aka strategy) must provide the following types/typedefs:
 * .
 *
 * - EventType: The type of operator-event this heuristic should listen to. EventType
 * must contain a member value_ of type ValueType and a member node_ of Type Node (or a subclasss).
 * - ValueType: The type of the value contained in the event this heuristic listens to.
 * - ComparatorType: A function-type that accepts two values of type ValueType and
 * returns true if the first value is "better" than the second.
 * .
 *
 */
template <class HeuStrat>
class OperatorBasedHeuristic : public Heuristic {
public:
  //! the type of the heuristic value
  typedef typename HeuStrat::ValueType ValueType;
  
  //! the event type this heuristic should handle and that contains a value of type ValueType
  typedef typename HeuStrat::EventType EventType;
  
  //! the type of the binary predicate to be used for value comparisons
  typedef typename HeuStrat::ComparatorType ComparatorType;
  
  //! creates a new object
  /*! 
   * \param c A binary predicate used to compare two values of type ValueType
   */
  OperatorBasedHeuristic(const ComparatorType& c = ComparatorType())
    : graph_(0)
    , comparator_(c) {
  }
  
  /*!
   * registers this as handler for events of type EventType.
   */
  void setGraph(Graph& g) {
    graph_ = &g;
    graph_->getEventManager().getChannelFor(event::Event<EventType>()).connect(*this);
  }

  //! event handler for events of type EventType
  /*!
   * associated the value e.value_ with the node e.node_
   */
  void handle(const EventType& e) {
    assert(graph_);
    if (!e.node_->getValue())
      e.node_->installValue(new ValueHolder<ValueType>(e.value_)); 
    else
      static_cast<ValueHolder<ValueType>*>(e.node_->getValue())->value() = e.value_;
  }
  
  /*!
   * selects the uncolored body node with the best heuristic value 
   * (previously set in the event handler) that satisfies the given constraint.
   *
   * \pre setGraph was called.
   *
   * \complexity linear in the number of uncolored body nodes
   */
  BodyNode* selectBodyNode(const Constraint& c) {
    assert(graph_);
    return getBestNode(graph_->uncoloredBodyNodesBegin(), 
      graph_->uncoloredBodyNodesEnd(), c, (BodyNode*)0);
  }

  /*!
   * selects the uncolored head node with the best heuristic value 
   * (previously set in the event handler) that satisfies the given constraint.
   *
   * \pre setGraph was called.
   *
   * \complexity linear in the number of uncolored head nodes
   */
  HeadNode* selectHeadNode(const Constraint& c) {
    assert(graph_);
    return getBestNode(graph_->uncoloredHeadNodesBegin(), 
      graph_->uncoloredHeadNodesEnd(), c, (HeadNode*)0);
  }
private:
  template <class It, class NT>
  NT* getBestNode(It first, It last, const Constraint& c, NT*) {
    NT* best = 0;
    for (; first != last; ++first) {
      if (c.isSatisfiedBy(**first)) {
        if (best == 0 || comparator_(*first, best))
          best = *first;
      }
    }
    return best;
  }
  typedef HVCompare<ValueType, ComparatorType> CMP;
  
  Graph* graph_;
  CMP comparator_;
};

}
#endif
