/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_DETAIL_FUNCTORS_H
#define LIBNOMORE_DETAIL_FUNCTORS_H
#include <iterator>
namespace NS_NOMORE {

//! Contains private implementation details not meant for general use.
namespace DETAIL {

//! DeleteObject from Scott Meyers' "Effective STL - Item 7"
/*! Allows for the use of std::for_each to delete all objects in a container
 * of pointers.
 **/
struct DeleteObject {
  template <class T>
  void operator()(const T* p) const {
    delete p;
  }
};

//! Compares the values of two pointers using operator<.
struct LessDeref {
  /*!
   * \pre p1 and p2 are dereferenceable
   */
  template <class Ptr>
  bool operator()(Ptr p1, Ptr p2) const {
    return *p1 < *p2;
  }
};

//! Compares the identifiers of two pointers.
struct LessDerefId {
  /*!
   * \pre p1 and p2 are dereferenceable
   */
  template <class Ptr>
  bool operator()(Ptr p1, Ptr p2) const {
    return p1->getId() < p2->getId();
  }
};

//! Compares the values of two head node pointers using the atom name.
struct LessLexOrderHeadPtr {
  /*!
   * \pre p1 and p2 are dereferenceable and head nodes
   */
  template <class HeadPtr>
  bool operator()(HeadPtr p1, HeadPtr p2) const {
    return p1->getAtom().getName() < p2->getAtom().getName();
  }
};

//! An output iterator object that inserts elements into a container using the container's put-method.
/*!
 * The container must define:
 * - The member type value_type, which is the type of an element of the sequence controlled by the container. 
 * - The member function push(value_type c), which adds a new element with value c to the container
 */
template<class Cont>
class PushInsertIterator 
	: public std::iterator<std::output_iterator_tag, typename Cont::value_type> {
public:
  typedef Cont container_type;
  typedef typename Cont::value_type value_type;
  explicit PushInsertIterator(Cont& x)
		: container_(x) {
	}
  PushInsertIterator& operator=(const value_type& val) {
		container_.push(val);
		return *this;
  }
  PushInsertIterator& operator*() {
		return *this;
  }
  PushInsertIterator& operator++() {
		return *this;
  }
  PushInsertIterator operator++(int) {
		return *this;
  }
private:
	Cont& container_;
};

template <class C>
PushInsertIterator<C> pushInserter(C& c) {
	return PushInsertIterator<C>(c);
}


}	// end namespace DETAIL

} // end namespace NS_NOMORE

#endif

