/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#define for if (0); else for
#endif
#include <object_factory.h>
#include <operator_decorator_factory.h>
#include <heuristic.h>
#include <operator_string_parser.h>
#include <operators/forward_prop.h>
#include <operators/backward_prop.h>
#include <operators/unfounded_set.h>
#include <operators/null_operator.h>
#include <operators/hybrid_lookahead.h>
#include <operators/body_lookahead.h>
#include <operators/select_uncolored.h>
#include <operators/select_supported.h>
#include <operators/sequence.h>
#include <heuristics/hybrid_lookahead_heuristic.h>
#include <heuristics/body_lookahead_heuristic.h>
#include <heuristics/static_lex_ordering.h>
#include <heuristics/neg_precond_count.h>
#include <heuristics/neg_arcs_count.h>
#include <heuristics/random.h>
#include <graph.h>
#include <cassert>
#include <set>
#include <stack>
#include <memory>
namespace NS_NOMORE { namespace {

template <class T>
struct Creator {
  static std::auto_ptr<Operator> createOp(Graph& g) {
    return std::auto_ptr<Operator>(new T(g));
  }
  static std::auto_ptr<Operator> createChoiceOp(Graph& g, Heuristic* h) {
    return std::auto_ptr<Operator>(new T(g, h));
  }
  static std::auto_ptr<Heuristic> createHeuristic() {
    return std::auto_ptr<Heuristic>(new T());
  }
};

std::string& trim(std::string& str) {
  std::string::size_type pos = str.find_last_not_of(" \t");
  if (pos != std::string::npos)
    str.erase(pos + 1, std::string::npos);
  pos = str.find_first_not_of(" \t");
  if (pos != 0)
    str.erase(0, pos);
  return str;
}

std::vector<std::string > explode(const std::string& separator, const std::string& str) {
  std::string::size_type len = separator.length();
  std::vector<std::string> result;
  if (len == 0) {
    result.push_back(str);
    return result;
  }

  std::string::size_type startPos = 0;
  std::string::size_type endPos = str.find(separator, startPos);
  while (endPos != std::string::npos) {
    if (endPos - startPos)
      result.push_back(std::string(str, startPos, endPos - startPos));

    startPos = endPos;
    if (startPos != std::string::npos)
      startPos += len;
    endPos = str.find(separator, startPos);
  }
  if (startPos != std::string::npos) {
    result.push_back(std::string(str, startPos, std::string::npos));
  }
  return result;
}

template <class V>
std::vector<std::string> asVector(const std::map<std::string, V>& m) {
  typename std::map<std::string, V>::const_iterator it = m.begin();
  typename std::map<std::string, V>::const_iterator end = m.end();
  std::vector<std::string> r;
  for (; it != end; ++it)
    r.push_back(it->first);
  return r;
}

std::pair<std::string, std::string> parseLookaheadString(const std::string& str) {
  std::string::size_type posBegin = str.find("[");
  std::string::size_type posEnd = str.find("]");
  if (posBegin == std::string::npos && posEnd == std::string::npos)
    return std::pair<std::string, std::string>(str, "");
  else if (posBegin != std::string::npos && posEnd != std::string::npos)
    return std::pair<std::string, std::string>(std::string(str, 0, posBegin),
      std::string(str, posBegin + 1, posEnd - (posBegin + 1)));
  
  throw FormatError("Illegal String Format! Expected: LOp[DetOp]");
}

///////////////////////////////////////////////////////////////////////////////
// class DetOpParser
///////////////////////////////////////////////////////////////////////////////
class DetOpParser : public OperatorStringParser {
public:
  DetOpParser(Graph& g, Operator* opSet, ObjectFactory* opFactory)
	: opSet_(opSet)
    , factory_(opFactory)
    , graph_(g)
    , result_(0)
    , ownsResult_(false) {	
  }
  Operator* getResult();
  std::set<std::string>& getOperatorNames() {
    return opNames_;
  }
  void clearResult() {
    if (ownsResult_)
      delete getResult();
    result_ = 0;
    while (!seqs_.empty()) seqs_.pop();
    opNames_.clear();
  }

  bool ownsResult() const {
    return ownsResult_;
  }
private:
  void startSequence();
  void endSequence(bool closure);
  void addOperator(const std::string& name);
  Operator* opSet_;
  ObjectFactory* factory_;
  Graph& graph_;
  Operator* result_;
  std::stack<Sequence*> seqs_;
  bool ownsResult_;
  std::set<std::string> opNames_;
  
};

void DetOpParser::startSequence() {
  seqs_.push(new Sequence);
  ownsResult_ = true;
}

void DetOpParser::endSequence(bool cl) {
  Sequence* top = seqs_.top();
  top->setClosure(cl);
  if (seqs_.size() > 1) {
    seqs_.pop();
    seqs_.top()->add(top);
  }
  result_ = seqs_.top();
}

void DetOpParser::addOperator(const std::string& name) {
  if (opNames_.insert(name).second) {
    Operator* o = opSet_ ? opSet_->extractOperator(name) : 0;
    bool newOp = false;
    if (!o) {
      o = factory_->createOperator(graph_, name).release();
      newOp = true;
    }
    if (seqs_.empty()) {  
      result_ = o;
      ownsResult_ = newOp;
    }
    else {
      seqs_.top()->add(o, newOp);
    }
  }
}

Operator* DetOpParser::getResult() {
  while (seqs_.size() > 1) {
    Sequence* top = seqs_.top();
    seqs_.pop();
    seqs_.top()->add(top);
    result_ = seqs_.top();
  }
  return result_;
}

} // end anonymous-namespace

///////////////////////////////////////////////////////////////////////////////
// ObjectFactory::Operators
///////////////////////////////////////////////////////////////////////////////
ObjectFactory::Operators::~Operators() {
  /* intentionally left blank */
}

///////////////////////////////////////////////////////////////////////////////
// ObjectFactory
///////////////////////////////////////////////////////////////////////////////
ObjectFactory::OpMap ObjectFactory::operators_s = ObjectFactory::initOpMap();
ObjectFactory::ChoiceOpMap ObjectFactory::choiceOps_s = ObjectFactory::initChoiceOpMap();
ObjectFactory::HeuristicsMap ObjectFactory::heuristics_s = ObjectFactory::initHeuristicsMap();
ObjectFactory::OpMap ObjectFactory::lookaheadOps_s = ObjectFactory::initLookaheadOpMap();

ObjectFactory::ObjectFactory() 
  : opDecorator_(new OperatorDecoratorFactory())
  , owner_(true) {
}
  
ObjectFactory::ObjectFactory(OperatorDecoratorFactory* opd) 
  : opDecorator_(opd)
  , owner_(true) {
  assert(opDecorator_);
}

ObjectFactory::ObjectFactory(OperatorDecoratorFactory& opd)
  : opDecorator_(&opd)
  , owner_(false) {

}

ObjectFactory::~ObjectFactory() {
  if (owner_)
    delete opDecorator_;
}

ObjectFactory::OpMap ObjectFactory::initOpMap() {
  OpMap ret;
  ret[ForwardPropagator::getOpName()]    = &Creator<ForwardPropagator>::createOp;
  ret[BackwardPropagator::getOpName()]   = &Creator<BackwardPropagator>::createOp;
  ret[UnfoundedSetOperator::getOpName()] = &Creator<UnfoundedSetOperator>::createOp;
  ret[NullOperator::getOpName()]         = &Creator<NullOperator>::createOp;
  ret["None"]                            = &Creator<NullOperator>::createOp;
  return ret;
}

ObjectFactory::OpMap ObjectFactory::initLookaheadOpMap() {
  OpMap ret;
  ret[HybridLookahead::getOpName()]      = &Creator<HybridLookahead>::createOp;
  ret[BodyLookahead::getOpName()]        = &Creator<BodyLookahead>::createOp;
  return ret;
}

ObjectFactory::ChoiceOpMap ObjectFactory::initChoiceOpMap() {
  ChoiceOpMap ret;
  ret[SelectUncolored::getOpName()] = &Creator<SelectUncolored>::createChoiceOp;
  ret[SelectSupported::getOpName()] = &Creator<SelectSupported>::createChoiceOp;
  return ret;
}

ObjectFactory::HeuristicsMap ObjectFactory::initHeuristicsMap() {
  HeuristicsMap ret;
  ret["NegPrec"] = &Creator<NegativePreconditionCount>::createHeuristic;
  ret["NegArcs"] = &Creator<NegativeWeightArcsCount>::createHeuristic;
  ret["SRand"] = &Creator<StaticRandom>::createHeuristic;
  ret["None"] = &Creator<SelectFirst>::createHeuristic;
  ret["Null"] = &Creator<SelectFirst>::createHeuristic;
  ret["HybridLA"] = &Creator<HybridLookaheadHeuristic>::createHeuristic;
  ret["BodyLA"] = &Creator<BodyLookaheadHeuristic>::createHeuristic;
  ret["LexOrder"] = &Creator<LexOrdering>::createHeuristic;
  ret["DRand"] = &Creator<DynamicRandom>::createHeuristic;
  return ret;
}

std::auto_ptr<Operator> ObjectFactory::createOperator(Graph& g, const std::string& name) const {
  OpMap::iterator it = operators_s.find(name);
  if (it != operators_s.end())  
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second(g).release() ));
  throw UnknownObject(name, ObjectFactory::DET_OP);
}

std::auto_ptr<Operator> ObjectFactory::createLookaheadOperator(Graph& g, const std::string& name) const {
  OpMap::iterator it = lookaheadOps_s.find(name);
  if (it != lookaheadOps_s.end())  
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second(g).release() ));
  throw UnknownObject(name, ObjectFactory::LOOKAHEAD_OP);
}

std::auto_ptr<Operator> ObjectFactory::createChoiceOperator(Graph& g, const std::string& name, const std::string& heu) const{
  std::auto_ptr<Heuristic> h(createHeuristic(g, heu));
  ChoiceOpMap::iterator it = choiceOps_s.find(name);
  if (it != choiceOps_s.end())
    return std::auto_ptr<Operator>(opDecorator_->decorate( it->second(g, h.release()).release() ));
  throw UnknownObject(name, ObjectFactory::CHOICE_OP);
}

std::auto_ptr<Heuristic> ObjectFactory::createHeuristic(Graph&, const std::string& name) const {
  HeuristicsMap::iterator it = heuristics_s.find(name);
  if (it != heuristics_s.end())  
    return it->second();
  throw UnknownObject(name, ObjectFactory::HEURISTIC);
}

ObjectFactory::Operators ObjectFactory::createOperators(Graph& g, const std::string& objStr, bool validate) {
  std::vector<std::string> str = explode(":", objStr);
  const unsigned int parts = 4;
  if (str.size() != parts)
    throw FormatError("Illegal String Format! Expected: ChoiceOp:DetOp:LOp[DetOp]:Heuristic");
  
  for (std::vector<std::string>::size_type i = 0; i != parts; ++i) {
    if (trim(str[i]).empty())
      throw FormatError("Illegal String Format! Expected: ChoiceOp:DetOp:LOp[DetOp]:Heuristic");
  }
  Operators o;
  o.choiceOp_ = createChoiceOperator(g, str[CHOICE_OP], str[HEURISTIC]);
  
  DetOpParser parser(g, 0, this);
  try {
    parser.parse(str[DET_OP]);
  }
  catch(const std::runtime_error&) {
    parser.clearResult();
    throw;
  }
  o.detOp_ = std::auto_ptr<Operator>(parser.getResult());
  if (validate) {
    std::set<std::string>& ops = parser.getOperatorNames();
    ops.insert(o.choiceOp_->getName());
    if (!o.choiceOp_->validate(ops) || !o.detOp_->validate(ops))
      throw InvalidOperatorCombination();
  }
  
  o.detOp_ = addLookahead(g, o.detOp_, str[LOOKAHEAD_OP], str[DET_OP]);

  return o;
}

std::auto_ptr<Operator> ObjectFactory::addLookahead(Graph& g,
                                                    std::auto_ptr<Operator> detOp,
                                                    const std::string& lopStr,
                                                    const std::string& detOpStr) {
  if (lopStr == "None")
    return detOp;
  
  std::pair<std::string, std::string> lopPair = parseLookaheadString(lopStr);
  std::auto_ptr<Operator> lop = createLookaheadOperator(g, lopPair.first);
  
  // safe since createLookaheadOperator creates only Lookahead-Ops.
  // But since the operator is potentially decorated we can't downcast
  // the returned operator directly. Instead we must access the lookahead
  // operator indirectly using extractOperator.
  Lookahead* l = static_cast<Lookahead*>(lop->extractOperator(lopPair.first));
  
  if (lopPair.second == "" || lopPair.second == detOpStr) {
    l->setPropagationOperator(*detOp);
    return std::auto_ptr<Operator>(new Sequence(detOp.release(), lop.release()));
  }
  else {
    DetOpParser parser(g, detOp.get(), this);
    try {
      parser.parse(lopPair.second);
    }
    catch(...) {
      parser.clearResult();
      throw;
    }
    l->setPropagationOperator(parser.getResult(), parser.ownsResult());
    return std::auto_ptr<Operator>(new Sequence(detOp.release(), lop.release()));
  }
}

std::vector<std::string> ObjectFactory::getKnownObjects(ObjectType ot) {
  if (ot == DET_OP) {
    return asVector(operators_s);
  }
  else if (ot == CHOICE_OP) {
    return asVector(choiceOps_s);
  }
  else if (ot == HEURISTIC) {
    return asVector(heuristics_s);
  }
  else if (ot == LOOKAHEAD_OP) {
    return asVector(lookaheadOps_s);
  }
  return std::vector<std::string>();
}

}

