#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/body_lookahead.h>
#include <heuristics/body_lookahead_heuristic.h>
#include <operators/forward_prop.h>
#include <constraint.h>
#include <graph.h>
#include "test_common.h"
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class BodyLookaheadProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(BodyLookaheadProp);
  CPPUNIT_TEST(testSuccess);
  CPPUNIT_TEST(testWithDefaultHeuristic);
  CPPUNIT_TEST(testException);
  CPPUNIT_TEST(testFailure1);
  CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    grp = new Graph();

    body1 = grp->insertBodyNode();
    body2 = grp->insertBodyNode();
    body3 = grp->insertBodyNode();
    head1 = grp->insertHeadNode(1);
    head2 = grp->insertHeadNode(2);

    det = new ForwardPropagator(*grp);
    lookahead = new BodyLookahead(*grp, *det, con, heu);
  }

  void tearDown() {
    delete lookahead;
    delete det;
    delete grp;
  }

  void testSuccess() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    CPPUNIT_ASSERT_EQUAL(ColorOperation::unchanged, (*lookahead)());

    CPPUNIT_ASSERT_EQUAL(grp->countBodyNodes(), heu.values_.size());
    
    CPPUNIT_ASSERT( heu.valueFor(body1, BodyLookahead::HeuristicValueType(1, 1)) );
    CPPUNIT_ASSERT( heu.valueFor(body2, BodyLookahead::HeuristicValueType(0, 0)) );
    CPPUNIT_ASSERT( heu.valueFor(body3, BodyLookahead::HeuristicValueType(0, 0)) );
    
    CPPUNIT_ASSERT( heu.noValueFor( head1 ) );
    CPPUNIT_ASSERT( heu.noValueFor( head2 ) ) ;
    
    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
  }

  void testWithDefaultHeuristic() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    BodyLookaheadHeuristic heu;
    heu.setGraph(*grp);
    lookahead->setHeuristic(heu);
    
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL(static_cast<Node*>(body1), heu.selectNode(NoConstraint()));

    
  }

  void testException() {
    HeadNode *head3 = grp->insertHeadNode(3);
  
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    body3->insertSuccessor(*head3);
    head3->insertZeroSuccessor(*body1);
    
    CPPUNIT_ASSERT_EQUAL( ColorOperation::failed, (*lookahead)());
    CPPUNIT_ASSERT_EQUAL( true, grp->hasConflict() );
  }

  void testFailure1() {    
    body1->insertSuccessor(*head1);
    head1->insertZeroSuccessor(*body2);
    body2->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    
    grp->color(*head2, Color::minus);
    grp->color(*body2, Color::minus);
    
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL( Color::minus, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
  }
  
private:
  struct FakeHeuristic : public BodyLAHeuristicBase {
  public:
    BodyNode* selectBodyNode(const Constraint& c) {return 0;} 
    HeadNode* selectHeadNode(const Constraint& c) {return 0;}
    Node* selectNode(const Constraint& c) {return 0;}
    void addHeuristicValue(Node* n, const HeuristicValueType& v) {
      values_.push_back(NVPair(n, v));
    }
    bool valueFor(Node* n, const BodyLookahead::HeuristicValueType& v) const {
      for (size_t i = 0; i != values_.size(); ++i) {
        if (values_[i].first == n && values_[i].second.max_ == v.max_ 
          && values_[i].second.min_ == v.min_)
          return true;
      }
      return false;
    }
    bool noValueFor(Node* n) const {
      for (size_t i = 0; i != values_.size(); ++i) {
        if (values_[i].first == n)
          return false;
      }
      return true;
    }
    typedef std::pair<Node*, HeuristicValueType> NVPair;
    std::vector<NVPair> values_;
  };

  Graph *grp;
  Operator *det;
  BodyLookahead *lookahead;
  SuppConstraint con;
  FakeHeuristic heu;

  BodyNode *body1;
  BodyNode *body2;
  BodyNode *body3;
  HeadNode *head1;
  HeadNode *head2;
};

CPPUNIT_TEST_SUITE_REGISTRATION(BodyLookaheadProp);

} // end namespace NS_NOMORE_TESTS
