#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/lookahead.h>
#include <operators/forward_prop.h>
#include <graph.h>
#include <heuristics/neg_precond_count.h>
#include "test_common.h"

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class LookaheadProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(LookaheadProp);
  CPPUNIT_TEST(testColorBodyNode);
  CPPUNIT_TEST(testColorHeadNode);
  CPPUNIT_TEST(testColorNodeException);
  CPPUNIT_TEST(testOwnership);
	CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    grp = new Graph();

    body1 = grp->insertBodyNode();
    body2 = grp->insertBodyNode();
    body3 = grp->insertBodyNode();
    head1 = grp->insertHeadNode(1);
    head2 = grp->insertHeadNode(2);

    det = new ForwardPropagator(*grp);
    SuppConstraint con;
    lookahead = new FakeLookahead(*grp, *det, con);
  }

  void tearDown() {
    delete lookahead;
    delete det;
    delete grp;
  }

  void testColorBodyNode() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    lookahead->checkColor(*body1, Color::plus);

    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
  }

  void testColorHeadNode() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    lookahead->checkColor(*head2, Color::minus);

    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
  }

  void testColorNodeException() {
    HeadNode *head3 = grp->insertHeadNode(3);
    head1->insertOneSuccessor(*body1);
    body1->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    body2->insertSuccessor(*head3);
    head3->insertOneSuccessor(*body3);
    body3->insertSuccessor(*head1);
    
    CPPUNIT_ASSERT_EQUAL( false, lookahead->checkColor(*head2, Color::minus) );

    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head3->getColor() );
  }

  void testOwnership() {
    bool del = false;
    FakeOp o(del);
    {
      SuppConstraint con;
      FakeLookahead l(*grp, o, con); 
    }
    CPPUNIT_ASSERT_EQUAL(false, del);
    {
      SuppConstraint con;
      FakeLookahead l(*grp, new FakeOp(del), con);
    }
    CPPUNIT_ASSERT_EQUAL(true, del);

    bool del2 = false;
    bool del3 = false;
    SuppConstraint con;
    FakeLookahead l(*grp, new FakeOp(del2), con);
    FakeOp f(del3);
    l.setPropagationOperator(f);
    CPPUNIT_ASSERT_EQUAL(true, del2);
  }
private:
  
  class FakeOp : public Operator {
  public:
    FakeOp(bool& trackDel)
      : Operator("F")
      , del_(trackDel) {
    }
    ColorOpResult operator()() {
      return ColorOperation::unchanged;
    }
    ~FakeOp() {
      del_ = true;
    }
    bool& del_;
  };
  class FakeLookahead : public Lookahead {
  public:
    FakeLookahead(Graph& grp, Operator& op, const Constraint& con) : Lookahead("FLA", grp, op, con) {
    }
    FakeLookahead(Graph& grp, Operator* op, const Constraint& con) : Lookahead("FLA", grp, op, con) {
    }
    bool checkColor(Node& n, Color::ColorValue c) {
      return Lookahead::checkColor(n, c);
    }
    void setHeuristic(Heuristic& h) {}
  private:
    void execute() {}

  };

  Graph *grp;

  Operator *det;
  FakeLookahead *lookahead;
  NegativePreconditionCount heu;

  BodyNode *body1;
  BodyNode *body2;
  BodyNode *body3;
  HeadNode *head1;
  HeadNode *head2;
};

CPPUNIT_TEST_SUITE_REGISTRATION(LookaheadProp);

}	// end namespace NS_NOMORE_TESTS
