/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include "nomore_app.h"
#include "nomore.h"
#ifdef KDEVELOP  
  #include "value.h"  
#else
  #include "program_opts/value.h"  
#endif

#include <graph.h>
#include <lparsereader.h>
#include <object_factory.h>
#include <choice_operator.h>
#include <operators/backward_prop.h>
#include <operators/lookahead.h>
#include <printer/null_printer.h>
#include <printer/stdout_printer.h>
#include <statistics/null_statistic.h>
#include <statistics/full_statistic.h>
#include <statistics/std_statistic.h>
#include <string>
#include <iostream>
#include <istream>
#include <fstream>
#include <ostream>
#include <cctype>
using namespace ProgramOptions;
using namespace std;
using namespace NS_NOMORE;

namespace {

typedef std::map<std::string, std::string> OperatorMap;
OperatorMap initOperatorMap() {
  OperatorMap ret;
  ret["B"]    = "B";
  ret["P"]    = "P";
  ret["U"]    = "(P,U)*";
  ret["BP"]   = "(P,B)*";
  ret["BU"]   = "(B,U)*";
  ret["PU"]   = "(P,U)*";
  ret["BPU"]  = "((P,B)*,U)*";
  return ret;
}

OperatorMap operatorMap_g = initOperatorMap();
bool isAlpha(char c) {
  return isalpha(static_cast<unsigned char>(c)) != 0;
}
std::string lookup(const string& opStr) {
  std::string alphaStr;
  for (std::string::size_type i = 0; i != opStr.size(); ++i)
    if (isAlpha(opStr[i]))
      alphaStr += opStr[i];
  
  sort(alphaStr.begin(), alphaStr.end());
  OperatorMap::iterator it = operatorMap_g.find(alphaStr);
  if (it != operatorMap_g.end()) {
    return it->second;
  }
  throw CommandLineError("'" + opStr + "' no such propagation operator");
}
 
istream& getInputStream(const string& fn) {
  static ifstream file;
  if (!fn.empty()) {
    file.open(fn.c_str());
    if (!file.is_open())
      throw CommandLineError(fn + " no such file");
    return file;
  }
  return cin;
}

void errorExit(const ObjectFactory& f, const UnknownObject& u) {
  string typeAsString;
  switch(u.getObjectType()) {
  case ObjectFactory::DET_OP:
    typeAsString = "operator";
    break;
  case ObjectFactory::LOOKAHEAD_OP:
    typeAsString = "lookahead-operator";
    break;
  case ObjectFactory::CHOICE_OP:
    typeAsString = "choice operator";
    break;
  case  ObjectFactory::HEURISTIC:
    typeAsString = "heuristic";
    break;
  default:
    typeAsString = "object";
  }
  string msg = string("'") + u.what() + "' unknown " + typeAsString + "!\n";
  msg += "known " + typeAsString + "s are:\n";
  vector<string> ko = f.getKnownObjects(u.getObjectType());
  for (vector<string>::size_type i = 0; i != ko.size(); ++i)
    msg += ko[i] + "\n";
  throw CommandLineError(msg);
}

}
NomoreApp::NomoreApp(int& argc, char** argv) 
  : options_((string("nomore++ ") + string(VERSION) + 
              string("\nusage: nomore [number] [options]\n")).c_str())
  , backtrackingMode_(NS_NOMORE::BacktrackingMode::symmetric) {
  
  OptionGroup generalOps("general options:");
  generalOps.addOptions()
    ("file,f", value<string>()->defaultValue(""), "read from arg instead of from stdin")
    ("version,V",	bool_switch(),	"Print version and exit")
		("help,h",		bool_switch(),	"Print help and exit")
    ("dlv-strategy", bool_switch(), "Using the dlv strategy for choosing nodes and backtracking")
  ;

  OptionGroup opOps("operator options:");
  opOps.addOptions()     
    ("choice-op", value<string>()->defaultValue("D"), "choice operator to use")
    ("prop-op", value<string>()->defaultValue("PBU"), "list of propagation operators to use (e.g. PBU)")
    ("lookahead-op", value<string>(), "list of propagation operators to use in lookahead (e.g. PBU)")
    ("heuristic", value<string>(), "heuristic to use")
    ("no-heuristic", bool_switch(), "do not use a heuristic")
    ("no-lookahead", bool_switch(), "do not use lookahead")
    ("body-lookahead", bool_switch(), "use body lookahead instead of hybrid lookahead")
    ("no-jumping", bool_switch(), "disbale jumping in B-Operator")
    ("no-ignore", bool_switch(), "do not use ignore-mode of B-Operator")
  ;

  OptionGroup outOps("output options:");
  outOps.addOptions()
    ("info,i", bool_switch(), "displays the values of selected options")
    ("no-output", bool_switch(), "do not print answer sets")
    ("no-statistics", bool_switch(), "do not print statistics")
    ("full-statistics", bool_switch(), "print all statistics")
  ;
  
  options_.addOptions(generalOps);
  options_.addOptions(opOps);
  options_.addOptions(outOps);

  OptionGroup allOpts;
  allOpts.addOptions(options_);
  allOpts.addOptions()
    ("number", value<int>()->defaultValue(1))
    ("operators,o", value<string>(), "op-string with format: ChoiceOp:DetOp:PostOp:LOp[DetOp]:Heu")
  ;

  try {
    values_.store(parseCommandLine(argc, argv, allOpts, false, "number"));
  }
  catch(const std::exception& e) {
    throw CommandLineError(e.what());
  } 

}

void NomoreApp::printHelp() const {
  cout << options_ << endl;
	cout << "Number is the number of answer sets to compute. A zero indicates all." << endl;
}

void NomoreApp::printVersion() const {
  cout << "NoMore++ - Version: " << VERSION << endl;
  cout << "Copyright (C) 2003-2005 NoMoRe Developing Group" << endl;
  cout << "This software is provided \"as is\" without expressed or implied warranty,\n"
       << "and with no claim as to its suitability for any purpose." << endl;
}

int NomoreApp::run() {
  if (values_.count("help")) {
    printHelp();
    return 0;
  }
  if (values_.count("version")) {
    printVersion();
    return 0;
  }
  validateArguments();
  
  std::istream& input = getInputStream(value_cast<string>(values_["file"]));
  Graph g;
  std::auto_ptr<Statistics> stats = createStatisticObject(g);
  std::auto_ptr<Solver> solver(createSolver(g, *stats, buildOpString()));
  if (values_.count("info") != 0)
    printInfo(*solver);
  LparseReader r;
  cout << "Reading program..." << flush;
  stats->startTimer();
  r.readProgram(input, g);
  cout << "done" << endl;
  std::cout << "solve ... " << std::endl;  
  if(solver->solve(g))
    std::cout << "YES" << std::endl;  
  else
    std::cout << "NO" << std::endl;  
  stats->stopTimer();
  stats->printReport(std::cout);
  return 0;
}

void NomoreApp::validateArguments() const {
  if (value_cast<int>(values_["number"]) < 0)
    throw CommandLineError("number can not be negative.");

  if (values_.count("full-statistics") && values_.count("no-statistics"))
    throw CommandLineError("options 'full-statistics' and 'no-statistics' are in conflict!");

  if (values_.count("no-lookahead") 
    && (values_.count("lookahead-op") || values_.count("body-lookahead"))) {
    throw CommandLineError("option 'no-lookahead' can't be used together with 'lookahead-op' or 'body-lookahead'!");
  }

  if (values_.count("no-heuristic") && values_.count("heuristic")) {
    throw CommandLineError("option 'no-heuristic' can't be used together with 'heuristic'!");
  }
}

string NomoreApp::buildOpString() const {
  if (values_.count("operators")) {
    return value_cast<string>(values_["operators"]);
  }

  std::string opString = value_cast<string>(values_["choice-op"]);
  opString += ":" + lookup(value_cast<string>(values_["prop-op"]));
  if (values_.count("no-lookahead") != 0) {
    opString += ":None:";
  }
  else {
    std::string lop = values_.count("body-lookahead") ? ":BLA" : ":HLA";
    if (values_.count("lookahead-op"))
      lop += "[" + lookup(value_cast<string>(values_["lookahead-op"])) + "]:";
    else
      lop += ":";
    opString += lop;
  }
  return addHeuristicName(opString);
}

std::auto_ptr<Solver> NomoreApp::createSolver(NS_NOMORE::Graph& g, NS_NOMORE::Statistics& stats,
                               const std::string& opString) const {
  std::auto_ptr<Solver> p(new Solver);
  p->setNrOfAnswerSetsToCompute(value_cast<int>(values_["number"]));
  p->setAnswerSetPrinter(
    values_.count("no-output") 
      ? static_cast<Printer*>(new NullPrinter()) 
      : new StdoutPrinter()
  );

  ObjectFactory factory(stats);
  try {
    ObjectFactory::Operators o = factory.createOperators(g, opString, true);
    bool nj = values_.count("no-jumping") != 0;
    bool ni = values_.count("no-ignore") != 0;
    if ( nj || ni ) {
      if (BackwardPropagator* b = static_cast<BackwardPropagator*>(o.detOp_->extractOperator("B"))) {
        if (nj) b->disableJumping();
        if (ni) b->disableIgnore();
      }
      else {
        cerr << "Warning: Options 'no-jumping' and 'no-ignore' are superfluous. B-Operator is not active!" << endl;
      }
    }
    p->setPropagationOperator(o.detOp_.release());
    p->setChoiceOperator(o.choiceOp_.release());
  }
  catch(const UnknownObject& e) {    
    errorExit(factory, e);
  }
  catch (const FormatError& e) {    
    throw CommandLineError(string("Option 'operators': ").append(e.what()));
  }
  catch (const InvalidOperatorCombination& e) {    
    throw CommandLineError(string("Option 'operators': ").append(e.what()));
  }
  return p;
}


std::auto_ptr<Statistics> NomoreApp::createStatisticObject(Graph& g) const {
  std::auto_ptr<Statistics> s;

  if (values_.count("no-statistics"))
    s.reset(new NullStatistics);
  else if (values_.count("full-statistics"))
    s.reset(new FullStatistics);
  else
    s.reset(new StdStatistics);

  s->registerWith(g.getEventManager());
  return s;
}

void NomoreApp::printInfo(const Solver& p) const {
  Operator* det = p.getPropagationOperator();
  cout 
    << "Propagation-Operators         : "  << det->getName() << "\n"
    << "Choice-Operator               : " << p.getChoiceOperator()->getName() << "\n"
    << "Lookahead                     : ";
  if (values_.count("no-lookahead") != 0)
    cout << "Disabled" << "\n";
  else {
    std::string lop = values_.count("body-lookahead") ? "BLA" : "HLA";
    cout << (lop == "BLA" ? "Bodies" : "Hybrid") << "\n";
    cout << "Lookahead-Propagation         : ";
    cout << static_cast<Lookahead*>(det->extractOperator(lop))->getPropagationOperator().getName() << "\n";
  }
  cout
    << "Backpropagtion                : " << (det->extractOperator("B") != 0 ? "Enabled" : "Disabled") << "\n"
    << "U-Operator                    : " << (det->extractOperator("U") != 0 ? "Enabled" : "Disabled") << "\n"
    << "Number of answer sets to find : ";
  if (p.getNrOfAnswerSetsToCompute() == 0)
    cout << "all";
  else
    cout << p.getNrOfAnswerSetsToCompute();
  cout << endl;
}

std::string NomoreApp::addHeuristicName(std::string opStr) const {
  if (values_.count("no-heuristic") != 0) {
    opStr += "None";
  }
  else if (values_.count("heuristic") != 0) {
    std::string heu = value_cast<std::string>(values_["heuristic"]);
    if ( (heu == "BodyLA" && !values_.count("body-lookahead"))
      || (heu == "HybridLA" && values_.count("hybrid-lookahead"))
      || (values_.count("no-lookahead") != 0 && (heu == "BodyLA" || heu == "HybridLA"))) {
      cerr << "warning: Heuristic is incompatible with lookahead. Heuristic is ignored." << endl;
      heu = "None";
    }
    opStr += heu;
  }
  else if (values_.count("no-lookahead") == 0) {
    opStr += values_.count("body-lookahead") ? "BodyLA" : "HybridLA";
  }
  else {
    opStr += "None";
  }
  return opStr;
}
