/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_CHOICE_OPERATOR_H
#define NOMORE_CHOICE_OPERATOR_H

#include <operator.h>
#include <color.h>

namespace NS_NOMORE {

class Node;
class Heuristic;
class Constraint;

//! Base class for all non deterministic choice-operators.
/*! ChoiceOperator implements the operator()() of the base class Operator. 
 *  Concrete sub-classses must override the method selectChoice for choosing 
 *  the next node to be colored as choice point. 
 *  
 *  There are two ways to use a concrete choice-operator.
 *
 *  - <b>Method 1:</b> Use the choice-operator just like any other operator
 *    by calling its function-operator. This way an uncolored node is selected
 *    (using the installed heuristics) and automatically colored as choice point. 
 *    \code
 *      ColorOpResult result = anChoiceOperator();
 *      if (result == ColorOperation::succeeded) {
 *        // choice point found and successfully colored
 *      }
 *      else if (result == ColorOperation::failed) {
 *        // choice point found but color error while coloring the node
 *      }
 *      else {
 *        // no choice point found
 *      }
 *    \endcode
 *    
 *  - <b>Method 2:</b> Select and color a node manually. 
 *  To do this call the selectChoice method then coolor the returned node (e.g.
 *  by calling graph::colorChoicePoint).
 *  selectChoice returns a null-ptr if no choices are left.
 *    \code
 *      node = anChoiceOperator.selectChoice();
 *      if(node == NULL) {
 *        // no choice point found
 *      } 
 *      else {
 *        graph.colorChoicePoint(*node, myFirstColor, mySecondColor);
 *      }
 *    \endcode
 */
class ChoiceOperator : public Operator {
public:
  //! Creates the choice operator.
  /*! 
   * \param name The name of this choice operator
   * \param grp The graph the operator should work on. 
   * \param h The heuristic to use for valuating possible choice points
   */
  ChoiceOperator(const std::string& name, Graph& grp, Heuristic* h);
  ~ChoiceOperator();

  //! Executes this choice-operator.
  /*!
   * This base class implements the following algorithm:
   * \code
   *  ColorOpResult result = ColorOperation::unchanged;
   *  Node* n = selectNode();
   *  if (n != 0) {
   *    if (graph_->colorChoicePoint(n, n->preferredChoicePointColor(), n->alternativeChoicePointColor()))
   *      result = ColorOperation::succeeded;
   *    else
   *      result = ColorOperation::failed;
   *  }
   *  return result;
   *
   * \endcode
   *
   * \note 
   *  concrete subclasses shall not override this method. Instead they should
   *  override the hook-method ChoiceOperator::selectChoice
   */
  ColorOpResult operator()();

  
  //! Selects a node that can be colored as choice point.
  /*! Returns an uncolored node which can be colored as choice point. If no such node is 
   *  found the method returns a null pointer.
   *  \return A node which can be colored as choice point, a null pointer if no node is
   *          found.
   */
  virtual Node* selectChoice() = 0;

  virtual const Constraint& getConstraint() const = 0;

  const Heuristic& getHeuristic() const {return *heuristic_;}
  Heuristic& getHeuristic() {return *heuristic_;}

  
private:
  Graph* graph_;
  Heuristic* heuristic_;
};

}

#endif
