/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#include <solver.h>
#include <printer.h>
#include <graph.h>
#include <operator.h>
#include <choice_operator.h>
#include <stdexcept>
namespace NS_NOMORE {

Solver::Solver() 
  : as_(0)
  , det_(0)
  , choice_(0)
  , printer_(0) {  
}  

Solver::Solver(int as, Operator *det, Operator *choice, Printer *printer) 
  : as_(as)
  , det_(det)
  , choice_(choice)
  , printer_(printer) {
}

Solver::Solver(const Solver& other) 
  : as_(other.as_)
  , det_(other.det_.release())
  , choice_(other.choice_.release())
  , printer_(other.printer_.release()) {  
}  
  
Solver::~Solver() {  
}

void Solver::setAnswerSetPrinter(Printer* printer) {
  assert(printer);
  printer_ = std::auto_ptr<Printer>(printer);
}

void Solver::setChoiceOperator(Operator* choice) {
  assert(choice);
  choice_ = std::auto_ptr<Operator>(choice);
  assert(getChoiceOperator());
}

ChoiceOperator* Solver::getChoiceOperator() const {
  assert( !choice_.get() 
    || dynamic_cast<ChoiceOperator*>(choice_->extractOperator(choice_->getName())));
  if (!choice_.get())
    return 0;
  
  // The indirection through extractOperator is necessary to allow for decorated
  // operators. I.e. choice_ may be decorated by an InvocationDecorator, in that
  // case choice is not a ChoiceOperator but transparently forwards to one.
  // extractOperator in that case yields the underlying ChoiceOperator.
  return dynamic_cast<ChoiceOperator*>(choice_->extractOperator(choice_->getName()));
}


void Solver::setNrOfAnswerSetsToCompute(int as) {
  as_ = as;
}

int Solver::getNrOfAnswerSetsToCompute() const {
  return as_;
}


void Solver::setPropagationOperator(Operator* det) {
  assert(det);
  det_ = std::auto_ptr<Operator>(det);
}

Operator* Solver::getPropagationOperator() const {
  return det_.get();
}

bool Solver::solve(Graph& grp) {
  grp.finishConstruction();

  if (as_ == 0)
    as_ = -1;
  
  bool restoreOK = true;
  for(int asNr = 0; asNr != as_ && restoreOK; ) {
    if ((*det_)() == ColorOperation::failed) {
      det_->reset();
      restoreOK = grp.recolorChoicePoint();
    }
    else if (grp.totalColoring()) {
      (*printer_)(++asNr, grp);
      restoreOK = grp.recolorChoicePoint();
    }
    else {
      ColorOpResult choiceOp = (*choice_)();
      if (choiceOp == ColorOperation::unchanged) {
        // no choices left
        ColorOpResult res = grp.totalize();
        if (res == ColorOperation::failed) {
          restoreOK = grp.recolorChoicePoint();
        }
        else if (res == ColorOperation::unchanged) {
          assert("error in solver.cpp(70)" && false);
          throw std::runtime_error("internal error detected in solver.cpp (70)");
        }
        else {
          assert(grp.totalColoring());
        }      
      }
      else if (choiceOp == ColorOperation::failed) {
        // illegal choice
        restoreOK = grp.recolorChoicePoint();
      }
    }
  }
  return restoreOK;
}

}
