#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/backward_prop.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include <strong_components.h>

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestBackwardProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestBackwardProp);
  CPPUNIT_TEST( testBodyMinus1HeadMinusBlocked );
  CPPUNIT_TEST( testBodyMinus1HeadMinusUnsupported );
//  CPPUNIT_TEST( testBodyWeakPlus1HeadMinus );
//  CPPUNIT_TEST( testBodyPlus1HeadMinus );

  CPPUNIT_TEST( testBodyMinus0HeadWeakPlusUnsupported );
  CPPUNIT_TEST( testBodyMinus0HeadWeakPlusBlocked );
  CPPUNIT_TEST( testBodyMinus0HeadPlusUnsupported );
  CPPUNIT_TEST( testBodyMinus0HeadPlusBlocked );

  CPPUNIT_TEST( testHeadWeakPlusBodyMinus);
  CPPUNIT_TEST( testHeadPlusBodyMinus);
	CPPUNIT_TEST_SUITE_END();

public:
  /* Generates the following graph/program with b[1..3] (body) and h[1..4] (head)
   * 
   * arcs: x <-0- y   (zero-arc from y to x)
   *       x <-1- y   (one-arc from y to x)
   *       x <--- y   (unlabeled-arc from y to x)
   * 
   *         h1 <--- b1
   * b3 <-0- h1 <--- b2
   * b3 <-0- h2
   * b3 <-1- h3
   * b3 <-1- h4
   */
  void setUp() {
    grp = new Graph();
    head1 = grp->insertHeadNode(1);
    head2 = grp->insertHeadNode(2);
    head3 = grp->insertHeadNode(3);
    head4 = grp->insertHeadNode(4);

    body1 = grp->insertBodyNode();
    body2 = grp->insertBodyNode();
    body3 = grp->insertBodyNode();

    body1->insertSuccessor(*head1);
    body2->insertSuccessor(*head1);
    head1->insertZeroSuccessor(*body3);
    head2->insertZeroSuccessor(*body3);
    head3->insertOneSuccessor(*body3);
    head4->insertOneSuccessor(*body3);
    
    
    b = new BackwardPropagator(*grp);
  }

  void tearDown() {
    delete grp;
    delete b;
  }

  /*
checking all cases with [*] (error cases causes an error, unimportant cases are not 
checked with the backward propagator)
B <-0- H: + <-0- +  [* necessary?]
          + <-0- +w [* necessary?]
          + <-0- -  [error]
          +w<-0- +  [* necessary?]
          +w<-0- +w [* necessary?]
          +w<-0- -  [error]
          - <-0- +  [*] (two cases -> supported or unblocked)
          - <-0- +w [*] (two cases -> supported or unblocked)
          - <-0- -  [unimportant]
B <-1- H: + <-1- +  [error]
          + <-1- +w [error]
          + <-1- -  [* necessary?]
          +w<-1- +  [error]
          +w<-1- +w [error]
          +w<-1- -  [* necessary?]
          - <-1- +  [unimportant]
          - <-1- +w [unimportant]
          - <-1- -  [*] (two cases -> supported or unblocked)
H <--- B: + <--- +  [unimportant]
          + <--- +w [unimportant]
          + <--- -  []
          +w<--- +  [unimportant]
          +w<--- +w [unimportant]
          +w<--- -  []
          - <--- +  [error]
          - <--- +w [error]
          - <--- -  [* necessary?]
  */

  /* pre: head3 and head4 are minus
   *      body3 is minus (body3 is unblocked but not supported)
   * -> Now color the head2 to weak plus and handle the node. The last head node (head1)
   *    must be colored to minus for unsupporting the body node (body3). 
   * -> additional check: body1 and body2 must be colored to minus because of the color
   *    of head1.*/
	void testBodyMinus0HeadWeakPlusUnsupported() {
    // prepare colorings
    grp->color(*head3, Color::minus);
    grp->color(*head4, Color::minus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );

    // check
    grp->color(*head2, Color::weak_plus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );

    // additional check (body1 and body2 must be colored to minus)
    CPPUNIT_ASSERT_EQUAL( Color::minus, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );

    CPPUNIT_ASSERT_EQUAL(0, identifyStrongComponents(*grp));
	}

  /* pre: head1 is plus and head3 is minus
   *      body3 is minus (body3 is supported but not unblocked)
   * -> Now color the head2 to weak plus and handle the node. The last head node (head4)
   *    must be colored to weak plus for blocking the body node (body3). */
	void testBodyMinus0HeadWeakPlusBlocked() {
    // prepare colorings
    grp->color(*head1, Color::plus);
    grp->color(*head3, Color::minus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head4->getColor() );

    // check
    grp->color(*head2, Color::weak_plus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head4->getColor() );
	}


  /* pre: head3 and head4 are minus
   *      body3 is minus (body3 is unblocked but not supported)
   * -> Now color the head2 to plus and handle the node. The last head node (head1)
   *    must be colored to minus for unsupporting the body node (body3). 
   * -> additional check: body1 and body2 must be colored to minus because of the color
   *    of head1.*/
	void testBodyMinus0HeadPlusUnsupported() {
    // prepare colorings
    grp->color(*head3, Color::minus);
    grp->color(*head4, Color::minus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );

    // check
    grp->color(*head2, Color::plus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );

    // additional check (body1 and body2 must be colored to minus)
    CPPUNIT_ASSERT_EQUAL( Color::minus, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );
	}

  /* pre: head1 is plus and head3 is minus
   *      body3 is minus (body3 is supported but not unblocked)
   * -> Now color the head2 to plus and handle the node. The last head node (head4)
   *    must be colored to weak plus for blocking the body node (body3). */
	void testBodyMinus0HeadPlusBlocked() {
    // prepare colorings
    grp->color(*head1, Color::plus);
    grp->color(*head3, Color::minus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head4->getColor() );

    // check
    grp->color(*head2, Color::plus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head4->getColor() );
	}

  /* pre: head1 and head2 are plus
   *      body3 is minus (body3 is supported but not blocked)
   * -> Now color the head3 to minus and handle the node. The last head node (head4)
   *    must be colored to plus for blocking the body node (body3). */
	void testBodyMinus1HeadMinusBlocked() {
    // prepare colorings
    grp->color(*head1, Color::plus);
    grp->color(*head2, Color::plus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head3->getColor() );

    // check
    grp->color(*head3, Color::minus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head4->getColor() );
	}

  /* pre: head1 is plus and head3 are minus
   *      body3 is minus (body3 is unblocked but not supported)
   * -> Now color the head1 to plus and handle the node. The last head node (head2)
   *    must be colored to minus for unsupporting the body node (body3). */
	void testBodyMinus1HeadMinusUnsupported() {
    // prepare colorings
    grp->color(*head1, Color::plus);
    grp->color(*head3, Color::minus);
    grp->color(*body3, Color::minus);
    (*b)();
    // head is uncolored
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );

    // check
    grp->color(*head4, Color::minus);
    (*b)();
    // head is colored
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
	}

  /* pre: body3 is weak plus
   * -> Now color the head3 to minus and handle the node. All zero predecessors
   *    colored to weak plus and all one predecessors colored to minus so that
   *    the body node (body3) is weak supported and unblocked. */
	void testBodyWeakPlus1HeadMinus() {
    // prepare colorings
    grp->color(*body3, Color::weak_plus);

    // check
    grp->color(*head3, Color::minus);
    (*b)();
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head4->getColor() );
	}

  /* pre: body3 is plus
   * -> Now color the head3 to minus and handle the node. All zero predecessors
   *    colored to weak plus and all one predecessors colored to minus so that
   *    the body node (body3) is supported and unblocked. */
	void testBodyPlus1HeadMinus() {
    // prepare colorings
    grp->color(*body3, Color::plus);

    // check
    grp->color(*head3, Color::minus);
    (*b)();
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head4->getColor() );
	}

  /* pre: head1 is weak plus
   * -> Now color the body1 to minus and handle the node. The last predecessor (body2)
   *    must be colored to weak plus. */
	void testHeadWeakPlusBodyMinus() {
    // prepare colorings
    grp->color(*head1, Color::weak_plus);

    // check
    grp->color(*body1, Color::minus);
    (*b)();
    CPPUNIT_ASSERT_EQUAL( Color::plus, body2->getColor() );
	}

  /* pre: head1 is plus
   * -> Now color the body1 to minus and handle the node. The last predecessor (body2)
   *    must be colored to weak plus. */
	void testHeadPlusBodyMinus() {
    // prepare colorings
    grp->color(*head1, Color::plus);

    // check
    grp->color(*body1, Color::minus);
    (*b)();
    CPPUNIT_ASSERT_EQUAL( Color::plus, body2->getColor() );
	}

private:
  HeadNode* head1;
  HeadNode* head2;
  HeadNode* head3;
  HeadNode* head4;

  BodyNode* body1;
  BodyNode* body2;
  BodyNode* body3;

  Graph* grp;
  
  BackwardPropagator* b;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestBackwardProp);

}	// end namespace NS_NOMORE_TESTS
