#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/sequence.h>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestSequenceOp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestSequenceOp);
  CPPUNIT_TEST(testBothFalse);
	CPPUNIT_TEST(testOneTrue);
  CPPUNIT_TEST(testOneError);
	CPPUNIT_TEST(testBothTrue);
	CPPUNIT_TEST(testClosure);
  CPPUNIT_TEST(testName);
  CPPUNIT_TEST(testExtract);
  CPPUNIT_TEST(testOwnership);
  CPPUNIT_TEST(testReset);
	CPPUNIT_TEST_SUITE_END();

public:
	void setUp() {
		o1_ = new FakeOperator("F");
		o2_ = new FakeOperator("F2");
		seqOp_ = new Sequence();
    seqOp_->add(o1_);
    seqOp_->add(o2_);
	}
	void tearDown() {
		delete seqOp_;
	}
	void testBothFalse() {
    CPPUNIT_ASSERT_EQUAL(ColorOperation::unchanged, (*seqOp_)());
	}
	void testOneTrue() {
		o1_->result_ = true;
    CPPUNIT_ASSERT_EQUAL(ColorOperation::succeeded, (*seqOp_)());
		o1_->result_ = false;
		o2_->result_ = true;
    CPPUNIT_ASSERT_EQUAL(ColorOperation::succeeded, (*seqOp_)());
	}

  void testOneError() {
    Sequence s;
    s.add(new FakeOperator("F"));
    s.add(new ErrorOp("Op2"));
		CPPUNIT_ASSERT_EQUAL(ColorOperation::failed, s());
    
  }
	void testBothTrue() {
		o1_->result_ = true;
		o2_->result_ = true;
    CPPUNIT_ASSERT_EQUAL(ColorOperation::succeeded, (*seqOp_)());
	}

	void testClosure() {
		o1_->result_ = true;
		o1_->maxRepeat_ = 3;
		seqOp_->setClosure(true);
    CPPUNIT_ASSERT_EQUAL(ColorOperation::succeeded, (*seqOp_)());
		CPPUNIT_ASSERT_EQUAL(-1, o1_->maxRepeat_);
	}
  void testName() {
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2)"), seqOp_->getName() );

    seqOp_->setClosure(true);
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2)*"), seqOp_->getName() );

    seqOp_->add(new FakeOperator("NF"));
    CPPUNIT_ASSERT_EQUAL( std::string("(F,F2,NF)*"), seqOp_->getName() );
  }
  void testExtract() {
    CPPUNIT_ASSERT_EQUAL( (Operator*)o1_, seqOp_->extractOperator("F") );
    CPPUNIT_ASSERT_EQUAL( (Operator*)o2_, seqOp_->extractOperator("F2") );
    CPPUNIT_ASSERT_EQUAL( (Operator*)0, seqOp_->extractOperator("F22") );
  }

  void testOwnership() {
    bool f1 = false;
    bool f2 = false;
    FakeOperator o1("F1", 1, &f1);
    {
      Sequence s;
      s.add(o1);
      s.add(new FakeOperator("F2", 1, &f2));
    }
    CPPUNIT_ASSERT_EQUAL(false, f1);
    CPPUNIT_ASSERT_EQUAL(true, f2);

  }

  void testReset() {
    Sequence s;
    ErrorOp *op1, *op2;
    s.add( op1 = new ErrorOp("Op1"));
    s.add( op2 = new ErrorOp("Op2"));
    CPPUNIT_ASSERT_EQUAL(ColorOperation::failed, s() );
    s.reset();
    CPPUNIT_ASSERT_EQUAL(true, op1->reset_);
    CPPUNIT_ASSERT_EQUAL(true, op2->reset_);
  }
private:
	Sequence* seqOp_;
  class ErrorOp : public Operator {
  public:
    ErrorOp(const std::string& name) 
      : Operator(name)
      , reset_(false) 
    {}

    ColorOpResult operator()() {
      return ColorOperation::failed;
    }
    void reset() {
      reset_ = true;
    }
    bool reset_;
  };
  class FakeOperator : public Operator {
	public:
    explicit FakeOperator(const std::string& name, int maxRepeat = 1, bool* trackDeletion = 0)
			: Operator(name)
			, result_(false)
			, maxRepeat_(maxRepeat)
      , trackDeletion_(trackDeletion){
		}
    ~FakeOperator() {
      if (trackDeletion_)
        *trackDeletion_ = true;
    }
    ColorOpResult operator()() {
			if (result_ && maxRepeat_--)
        return ColorOperation::succeeded;
      return ColorOperation::unchanged;
		}
    bool result_;
		int maxRepeat_;
    bool* trackDeletion_;
  } *o1_, *o2_;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestSequenceOp);

}	// end namespace NS_NOMORE_TESTS
