/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_HEADNODE_H
#define LIBNOMORE_HEADNODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <node.h>
#include <vector>
#include <string>
#include <cassert>


namespace NS_NOMORE {

class BodyNode;

//! A head node represents a rule's head of a logic program in nomore's body-head-graph.
/*!
 * \see Graph
 */
class HeadNode : public Node {
public:
  //! Set of body nodes used as predecessors and successors.
  typedef std::vector<BodyNode*> BodyNodeSet;
	
	//! Iterator type that can be used to iterate over a set of body nodes.
	typedef BodyNodeSet::const_iterator BodyNodeIterator;

  //! Constructs a new node representing the atom with the given atomId
  /*!
   * \param id the unique id to be used for the new node
   * \param atomId the id of the atom that is represented by this node (normally
   * this id comes from lparse).
   */
  HeadNode(long id, long atomId);
  HeadNode(long id, long atomId, const std::string& name);

  //! Destroys this head node and its atom.
	~HeadNode();

  //! returns the id of the atom represented by this head node.
  long getAtomId() const;
  
  //! returns the name of the atom represented by this head node
  const std::string& getAtomName() const;
  
  //! sets the name of the atom represented by this head node
  void setAtomName(const std::string& name);
  

  //! Returns an iterator that points to the first predecessor of this head node.
	/*! If this head node does not have any predecessors the end-iterator is returned. */
	BodyNodeIterator predecessorsBegin() const {
		return predecessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the predecessors of this head node.
	BodyNodeIterator predecessorsEnd() const {
		return predecessors_.end();
	}
  
	//! Returns an iterator that points to the first zero-successor of this head node.
	/*! If this head node does not have any successors the end-iterator is returned. */
	BodyNodeIterator zeroSuccessorsBegin() const {
		return zeroSuccessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the zero-successors of this head node.
	BodyNodeIterator zeroSuccessorsEnd() const {
		return zeroSuccessors_.end();
	}

	//! Returns an iterator that points to the first zero-successor of this head node.
	/*! If this head node does not have any successors the end-iterator is returned. */
	BodyNodeIterator oneSuccessorsBegin() const {
		return oneSuccessors_.begin();
	}

	//! Returns an iterator that points just beyond the end of the zero-successors of this head node.
	BodyNodeIterator oneSuccessorsEnd() const {
		return oneSuccessors_.end();
	}

  //! Updates internal counters to reflect the coloring of a predecessor.
  /*! \param oldColor The old color of the predecessor.
   *  \param newColor The new color of the predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by predecessors of this node.
   */
  virtual void predecessorColored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Updates internal counters to reflect the fact that a predecessor was restored.
  /*! \param oldColor The old color of the predecessor.
   *  \param newColor The new color of the predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by predecessors of this node.
   */
  virtual void predecessorRestored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Returns the unsupported state of this node.
  /*! Unsupported means all predecessors are colored \ref Color::minus "minus".
   *  \return True if the node is unsupported else false. */
  bool isUnsupported() const {
    assert(unsupported_ >= 0);
    return unsupported_ == 0;
  }

	//! Returns the number of predecessors not colored minus.
	long countPredecessorsNotMinus() const {
		return unsupported_;
	}

  
  //! adds a predecessor to this node.
  /*! Modifies unsupported_ counter.
   *  \param b The node to be added as predecessor to this node.
   */
  void insertPredecessor(BodyNode& b);

  //! adds a zero-successor to this node and calls BodyNode::insertZeroPredecessor().
  /*!  
   *  \param b The node to be added as zero-successor to this node. 
   *  \post this node was added as a zero-predecessor to b.
   */
  void insertZeroSuccessor(BodyNode& b);

  //! adds a one-successor to this node and calls BodyNode::insertOnePredecessor().
  /*!  
   *  \param b The node to be added as one-successor to this node. 
   *  \post this node was added as a one-predecessor to b.
   */
  void insertOneSuccessor(BodyNode& b);

  //! Returns the number of one-successors this node has.
  size_t countOneSuccesors() const {
    return oneSuccessors_.size();
  }

  //! Returns the color which preserves the recursive support used for the V operator.
  /*!
   * \return always Color::weak_plus
   */
  virtual Color::ColorValue recursiveSupportColor() const {
    return Color::weak_plus;
  }

  //! Returns the preferred choice point color of this node.
  /*!
   * The preferred choice point color for head nodes is Color::minus.
   */
  virtual Color::ColorValue preferredChoicePointColor() const {
    return Color::minus;
  }

  //! Returns the alternative choice point color for this node.
  /*!
   * The alternative choice point color for head nodes is Color::weak_plus.
   */
  virtual Color::ColorValue alternativeChoicePointColor() const {
    return Color::weak_plus;
  }

  /*!
   * returns true if all zero-successors and all predecessors have a component number
   * differenet to this node's.
   */
  bool trivialComponent() const;

  void propagate(Graph& g);

  virtual double getTightnessFactor(int gradient) const;
protected:
  //! Notifies predecessors and successors of this node about the color change.
  /*! Calls for all zero-predecessors the BodyNode::zeroPredecessorColored() method,
   *  for all one-predecessors the BodyNode::onePredecessorColored() method and
   *  for all successors the BodyNode::successorColored() method.
   *  \param oldColor The old color before coloring this node. 
   */
  virtual void doSetColor(Color::ColorValue oldColor);

  //! Notifies predecessors and successors of this node about the restoring of a color.
  /*! Calls for all zero-predecessors the BodyNode::zeroPredecessorRestored() method,
   *  for all one-predecessors the BodyNode::onePredecessorRestored() method and
   *  for all successors the BodyNode::successorRestored() method.
   *  \param oldColor The old color before restoring this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor);

  //! Fires an event of type \ref HeadNodeColored. 
  /*! \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor);

  //! Fires an event of type \ref HeadNodeColoredChoicePoint.
  /*! \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, bool recolored);

private:
  long atomId_;
  std::string atomName_;

  //! Unsupported counter initialized by the number of uncolored predecessors.
  /*! A node is unsupported if this counter is zero. All predecessors are colored
   *  Color::minus. */
  long unsupported_;

  //! Set of predecessors of this node.
  BodyNodeSet predecessors_;

  //! Set of zero successors of this node.
  BodyNodeSet zeroSuccessors_;

  //! Set of one successors of this node.
  BodyNodeSet oneSuccessors_;
};

//! Event type used to transfer information about the fact that a head node was colored.
/*! \see HeadNode::fireColorEvent */
struct HeadNodeColored {
	HeadNodeColored(HeadNode* h, Color::ColorValue oldColor)
		: node_(h)
		, oldColor_(oldColor) {
	}
	HeadNode*					node_;      /*!< The head node that was colored */
	Color::ColorValue	oldColor_;  /*!< The head node's previous color */
};
//! Event type used to transfer information about the fact that a Head node was colored as choice point
/*! \see HeadNode::fireColorChoicePointEvent */
struct HeadNodeColoredChoicePoint {
	HeadNodeColoredChoicePoint(HeadNode* h, Color::ColorValue oldColor, bool recolored)
		: node_(h)
		, oldColor_(oldColor) 
    , recolored_(recolored) {
	}

	HeadNode*					node_;      /*!< The head node that was colored */
	Color::ColorValue oldColor_;  /*!< The head node's previous color */
  bool              recolored_; /*!< True if the choice point was recolored. */
};

} // NS_NOMORE

#endif
