/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_HYBRID_LOOKAHEAD_OPERATOR_H
#define NOMORE_HYBRID_LOOKAHEAD_OPERATOR_H

#include <operators/lookahead.h>
#include <heuristic.h>
#include <set>
#include <cassert>

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
struct HeadNodeColored;
struct HeadNodeColoredChoicePoint;
struct BodyNodeColored;

class HybridLAHeuristicBase;

//! Lookahead operator that works on both body- and head nodes
/*! 
 *  The hybrid lookahead operator assigns a color to uncolored head- and body node
 *  whenever the opposite assignment would lead to a conflict.
 *
 *  In contrast to the body-lookahead-operator this operator uses only
 *  one-way checks, i.e. Head nodes are checked with Color::minus and body nodes with
 *  Color::plus. 
 *  Refer to the Nomore-Paper for a rationale for this behaviour.
 *  
 *  Additionally this operator produces information that can be used as
 *  heuristic value. An event of type HybridLookahead::EventType is fired
 *  for each node for which no conflict was found. 
 *  The value-member of this event type contains a counter that store the number
 *  of nodes colored because of the coloring of the lookahead-node.
 *  
 *  The algorithm looks like the following code fragment:
 *  \code
 *    check-queue contains all uncolored nodes;
 *    while(check-queue not empty) {
 *      node := check-queue.pop();
 *      if (propagation with node's preferred choice color leads to conflict) {
 *          assign(node, alternative choice color);
 *      }
 *      else { 
 *        store heuristic value for node   
 *      }
 *    }
 *  \endcode
 *
 *  Technically the operator does not check every uncolored node but
 *  only those nodes that don't get a color as a result of propagating other nodes. 
 *  That is if assigning (and propagating) color c1 to node n1 leads to the coloring
 *  of n2 to the color c1 and n1 is supported then n2 is removed from the
 *  lookahead-set. The restriction that n1 must be supported is needed in order
 *  to get better heuristic-values for supported-driven choice-operators.
 *
 */
class HybridLookahead : public Lookahead {
public:
  typedef long HeuristicValueType;
  
  //! Creates the operator.
  /*! 
   * \param grp The graph the operator should work on. 
   * \param op Operator to be used for propagation
   */ 
  HybridLookahead(Graph& grp, Operator& op, const Constraint& con, Heuristic& heu);  
  explicit HybridLookahead(Graph& grp);

  //! event handler for HeadNodeColored events.
  void handle(const HeadNodeColored& e);

  //! event handler for HeadNodeColoredChoicePoint events
  void handle(const HeadNodeColoredChoicePoint& e);

  //! event handler for BodyNodeColored events.
  void handle(const BodyNodeColored& e);
  
  //! Returns the name of the operator.
  const char* getName() const;

  virtual void setHeuristic(Heuristic& h);
protected:
  //! Resets the look ahead counter.
  void resetCounter();
  
  //! Sets the counters for the current lookahead node as heuristic value.
  void storeHeuristic();

  //! Checks the current lookahead node by calling Lookahead::checkColor.
  /*! 
   * If the color check fails (a conflict is found) the node is colored to
   * the opposite color.
   */
  bool checkCurrentNode();

private:  
  typedef std::set<long> LookaheadInfoMap;
  typedef LookaheadInfoMap::iterator LookaheadInfoMapIterator;

  //! executes this operator
  virtual void execute();
  
  Node*               currentNode_;
  HeuristicValueType  currentValue_;
  
  LookaheadInfoMap lookaheadInfos_;

  HybridLAHeuristicBase* heuristic_;
};

//! base interface for heuristics based on the result of the hybrid-lookahead operator,
class HybridLAHeuristicBase : public Heuristic {
public:
  typedef HybridLookahead::HeuristicValueType HeuristicValueType;
  virtual void addHeuristicValue(Node* n, const HeuristicValueType& v) = 0;
};


}

#endif
