#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <object_factory.h>
#include <operator_decorator_factory.h>
#include <graph.h>
#include <operators/forward_prop.h>
#include <operators/hybrid_lookahead.h>
#include <operators/sequence.h>
#include <operators/select_supported.h>
#include <heuristic.h>
#include <typeinfo>
#include <algorithm>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestObjectFactory : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestObjectFactory);
  CPPUNIT_TEST(testEmpty);
  CPPUNIT_TEST(testRegisterOperator);
  CPPUNIT_TEST(testCreateOperator);
  CPPUNIT_TEST(testCreateChoiceOperator);

  CPPUNIT_TEST(testDecoration);
  CPPUNIT_TEST(testRegisterKnownClasses);

  CPPUNIT_TEST(testCreateOperatorsMissingComponent);
  CPPUNIT_TEST(testCreateOperatorsToManyComponents);

  CPPUNIT_TEST(testMissingP);
  
  CPPUNIT_TEST(testCreateOperators);
  
  CPPUNIT_TEST(testCreateWithLookahead);
  CPPUNIT_TEST(testInvalidLookaheadStr);

  CPPUNIT_TEST_SUITE_END();

  static Operator* createPOperator(Graph& g) {
    return new ForwardPropagator(g);
  }
  static ChoiceOperator* createDOperator(Graph& g, Heuristic* h) {
    return new SelectSupported(g, h);
  }
  static Heuristic* createSelectFirst() {
    return new SelectFirst();
  }
  static Lookahead* createHLAOperator(Graph& g) {
    return new HybridLookahead(g);
  }
public:
  void setUp() {
    factory_ = new ObjectFactory();
  }
  void tearDown() {
    delete factory_;
  }

  void testEmpty() {
    std::vector<std::string> o = factory_->getRegisteredClasses(ObjectFactory::ALL_OBJECTS);
    CPPUNIT_ASSERT_EQUAL(true, o.empty());

    CPPUNIT_ASSERT_THROW(factory_->getDescription("Foo"), UnknownObject);
  }

  void testRegisterOperator() {
    std::string pName = getName<ForwardPropagator>(0);
    factory_->registerClass(ObjectFactory::OpInfo(pName, createPOperator));
    std::vector<std::string> o = factory_->getRegisteredClasses(ObjectFactory::DET_OP);
    CPPUNIT_ASSERT_EQUAL(size_t(1), o.size());
    CPPUNIT_ASSERT_EQUAL(pName, o[0]);
    std::string empty = "";
    CPPUNIT_ASSERT_EQUAL(empty, factory_->getDescription(pName));
  }

  void testCreateOperator() {
    std::string pName = getName<ForwardPropagator>(0);
    factory_->registerClass(ObjectFactory::OpInfo(pName, createPOperator));
    Graph g;
    std::auto_ptr<Operator> p(factory_->createOperator(g, pName));
    CPPUNIT_ASSERT_EQUAL(pName, std::string(p->getName()));
    CPPUNIT_ASSERT( typeid(ForwardPropagator) == typeid( *p ));

    CPPUNIT_ASSERT_THROW(factory_->createOperator(g, "Foo"), UnknownObject);
  }

  void testCreateChoiceOperator() {
    std::string suppName = getName<SelectSupported>(0);
    CPPUNIT_ASSERT_EQUAL(true, 
      factory_->registerClass(ObjectFactory::ChoiceOpInfo(suppName,  createDOperator))
    );
    CPPUNIT_ASSERT_EQUAL(true, 
      factory_->registerClass(ObjectFactory::HeuristicInfo("SF", createSelectFirst))
    );
    Graph g;
    std::auto_ptr<Operator> d(factory_->createChoiceOperator(g, suppName, "SF"));
    CPPUNIT_ASSERT_EQUAL(suppName, std::string(d->getName()));
    CPPUNIT_ASSERT( dynamic_cast<SelectSupported*>(d->extractOperator(suppName)) != 0 );
    ChoiceOperator* c = static_cast<ChoiceOperator*>(d->extractOperator(suppName));
    CPPUNIT_ASSERT(typeid(SelectFirst) == typeid( c->getHeuristic() ));

    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, suppName, "Foo"), UnknownObject);
    CPPUNIT_ASSERT_THROW(factory_->createChoiceOperator(g, "Foo", "SF"), UnknownObject);
  }
  
  void testDecoration() {
    ObjectFactory f(new FakeDecoratorFactory);
    std::string pName = getName<ForwardPropagator>(0);
    std::string hlaName = getName<HybridLookahead>(0);
    std::string suppName = getName<SelectSupported>(0);
    f.registerClass(ObjectFactory::OpInfo(pName, createPOperator));
    f.registerClass(ObjectFactory::LookOpInfo(hlaName, createHLAOperator));
    f.registerClass(ObjectFactory::ChoiceOpInfo(suppName, createDOperator));
    f.registerClass(ObjectFactory::HeuristicInfo("SF", createSelectFirst));

    Graph g;
    std::auto_ptr<Operator> o1(f.createOperator(g, pName));
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o1));

    std::auto_ptr<Operator> o2(f.createChoiceOperator(g, suppName, "SF"));
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o2));

    ObjectFactory::Operators o = f.createOperators(g, "D:P:HLA:SF");
    
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o.detOp_));
    CPPUNIT_ASSERT(typeid(FakeDecoratorFactory::NullDecorator) == typeid(*o.lookOp_));
  }

  void testRegisterKnownClasses() {
    factory_->registerKnownClasses();
    std::vector<std::string> classes = factory_->getRegisteredClasses(ObjectFactory::ALL_OBJECTS);
    using std::find;
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "P") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "B") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "U") != classes.end());

    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "D") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "C") != classes.end());

    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "HybridLA") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "BodyLA") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "SRand") != classes.end());
    CPPUNIT_ASSERT(find(classes.begin(), classes.end(), "DRand") != classes.end());
  }

  void testCreateOperators() {
    Graph g;
    factory_->registerKnownClasses();
    ObjectFactory::Operators o = factory_->createOperators(g, "D:((P, B)*,U)*:None:HybridLA");
    CPPUNIT_ASSERT(o.choiceOp_->getName() == std::string("D"));
    CPPUNIT_ASSERT_EQUAL(std::string("((P,B)*,U)*"), std::string(o.detOp_->getName()));
  }

  void testCreateOperatorsMissingComponent() {
    Graph g;
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "P:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, ":P:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D::None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:None:HybridLA"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HybridLA"), FormatError);
  }

  void testCreateOperatorsToManyComponents() {
    Graph g;
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:N:None:HybridLA:Foo"), FormatError);
  }
  
  void testMissingP() {
    Graph g;  
    factory_->registerKnownClasses();
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:B:None:HybridLA"), InvalidOperatorCombination);
  }

  
  void testCreateWithLookahead() {
    Graph g;
    factory_->registerKnownClasses();
    ObjectFactory::Operators o = factory_->createOperators(g, "D:P:HLA[(P,B)]:None");
    Lookahead* hla = dynamic_cast<Lookahead*>(o.lookOp_.get());
    CPPUNIT_ASSERT(hla != 0);
    Operator* pInHla = hla->getPropagationOperator().extractOperator("P");
    Operator* pInDet = o.detOp_->extractOperator("P");
    CPPUNIT_ASSERT_EQUAL(pInHla, pInDet);
  }

  void testInvalidLookaheadStr() {
    Graph g;
    factory_->registerKnownClasses();
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA(P,B)]:None"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA[(P,B):None"), FormatError);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:HLA(P,B):None"), UnknownObject);
    CPPUNIT_ASSERT_THROW(factory_->createOperators(g, "D:P:Foo[(P,B)]:None"), UnknownObject);
  }
private:
  class FakeDecoratorFactory : public OperatorDecoratorFactory {
  public:
    virtual Operator* decorate(Operator* op) {return new NullDecorator(op);}
    struct NullDecorator : public Operator {
    public:
      NullDecorator(Operator* o)
        : Operator()
        , op_(o) {
      }
      ~NullDecorator() {
        delete op_;
      }
      const char* getName() const {
        return op_->getName();
      }
      ColorOpResult operator()() {
        return (*op_)();
      }
      Operator* extractOperator(const std::string& name) {
        return op_->extractOperator(name);
      }
    private:
      Operator* op_;
    };
  };
  ObjectFactory* factory_;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestObjectFactory);

}	// end namespace NS_NOMORE_TESTS
