/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_BODYNODE_H
#define LIBNOMORE_BODYNODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <vector>
#include <cassert>
#include <algorithm>
#include <node.h>

namespace NS_NOMORE {

class HeadNode;

//! A body node represents a rule's body of a logic program in nomore's body-head-graph.
/*!
 * \see Graph
 */
class BodyNode : public Node {
  friend class Graph;

public:
  //! Set of head nodes used as predecessors and successors.
  typedef std::vector<HeadNode*> HeadNodeSet;

  //! Size type of the head node set.
  typedef HeadNodeSet::size_type size_type;
	
	//! Iterator type that can be used to iterate over a set of head nodes.
	typedef HeadNodeSet::const_iterator HeadNodeIterator;

  //! Constructs a new node with the given id.
  /* \param id The id for this body node.
   */
  explicit BodyNode(long id);

  //! Returns an iterator that points to the first successor of this body node.
	/*! If this body node does not have any successors the end-iterator is returned. 
   * \see successorsEnd
   */
	HeadNodeIterator successorsBegin() const {
		return successors_.begin();
	}
	
	//! Returns an iterator that points just beyond the end of the successors of this body node.
	HeadNodeIterator successorsEnd() const {
		return successors_.end();
	}

	//! Returns an iterator that points to the first zero-predecessor of this body node.
	/*! If this body node does not have any zero-predecessors the end-iterator is returned. 
   * \see zeroPredecessorsEnd
   */
	HeadNodeIterator zeroPredecessorsBegin() const {
		return zeroPredecessors_.begin();
	}
	
	//! Returns an iterator that points just beyond the end of the zero-predecessors of this body node.
	HeadNodeIterator zeroPredecessorsEnd() const {
		return zeroPredecessors_.end();
	}

	//! Returns an iterator that points to the first one-predecessor of this body node.
	/*! If this body node does not have any one-predecessors the end-iterator is returned. 
   * \see onePredecessorsEnd
   */
	HeadNodeIterator onePredecessorsBegin() const {
		return onePredecessors_.begin();
	}
	
	//! Returns an iterator that points just beyond the end of the one-predecessors of this body node.
	HeadNodeIterator onePredecessorsEnd() const {
		return onePredecessors_.end();
	}

	//! Updates internal counters to reflect the coloring of a zero-predecessor.
  /*! \param oldColor The old color of the zero predecessor.
   *  \param newColor The new color of the zero predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by zero-predecessors of this node.
   */
  virtual void zeroPredecessorColored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Updates internal counters to reflect the coloring of a one-predecessor.
  /*! \param oldColor The old color of the one predecessor.
   *  \param newColor The new color of the one predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by one-predecessors of this node.
   */
  virtual void onePredecessorColored(Color::ColorValue oldColor, Color::ColorValue newColor);

  
  //! Updates internal counters to reflect the fact that a zero-predecessor was restored.
  /*! \param oldColor The old color of the zero-predecessor.
   *  \param newColor The new color of the zero-predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by zero-predecessors of this node.
   */
  virtual void zeroPredecessorRestored(Color::ColorValue oldColor, Color::ColorValue newColor);

  //! Updates internal counters to reflect the fact that a one-predecessor was restored.
  /*! \param oldColor The old color of the one-predecessor.
   *  \param newColor The new color of the one-predecessor.
   *  \note this method is public for testing reasons only. Normally this method 
   *  should only be called by zero-predecessors of this node.
   */
  virtual void onePredecessorRestored(Color::ColorValue oldColor, Color::ColorValue newColor);

  
  //! Returns the supported state of the node.
  /*! Supported means all zero-predecessors are colored \ref Color::plus "plus".
   *  \return True if the node is supported else false. 
   */
  bool isSupported() const {
    assert(supported_ >= 0);
    return supported_ == 0;
  }

  //! Returns the weak supported state of the node.
  /*! Weak supported means all zero-predecessors are colored \ref Color::plus "plus"
   *  or \ref Color::weak_plus "weak_plus".
   *  \return True if the node is weak supported else false. */
  bool isWeakSupported() const {
    assert(weakSupported_ >= 0);
    return weakSupported_ == 0;
  }

	//! Returns the number of zero-predecessors not colored Color::plus or Color::weak_plus.
	long countZeroPredecessorsNotPlusOrWeakPlus() const {
		return weakSupported_;
	}

  //! Returns the unsupported state of the node.
  /*! Unsupported means at least one zero-predecessors is colored \ref Color::minus "minus".
   *  \return True if the node is unsupported else false. */
  bool isUnsupported() const {
    assert(unsupported_ <= 0);
    return unsupported_ < 0;
  }

  //! Returns the blocked state of the node.
  /*! Blocked means at least one one-predecessors is colored \ref Color::plus "plus"
   *  or \ref Color::weak_plus "weak_plus".
   *  \return True if the node is blocked else false. */
  bool isBlocked() const {
    assert(blocked_ <= 0);
    return blocked_ < 0;
  }

  //! Returns the unblocked state of the node.
  /*! Unblocked means all one-predecessors are colored \ref Color::minus "minus".
   *  \return True if the node is unblocked else false. */
  bool isUnblocked() const {
    assert(unblocked_ >= 0);
    return unblocked_ == 0;
  }

	//! Returns the number of one-predecessors not colored Color::minus.
	long countOnePredecessorsNotMinus() const {
		return unblocked_;
	}

  //! adds a zero-predecessor to this node.
  /*! Modifies supported and weak supported counter.
   *  \param h The node to be added as zero-predecessor to this node. */
  void insertZeroPredecessor(HeadNode &h);

  //! adds a one-predecessor to this node.
  /*! Modifies unblocked counter.
   *  \param h The node to be added as one-predecessor to this node. */
  void insertOnePredecessor(HeadNode &h);

  //! adds a successor to this node and calls HeadNode::insertPredecessor().
  /*! Modifies the uncoloredSuccessors_ counter.
   *  \param h The node to be added as successor to this node. 
   *  \post this node was added as a predecessor to h */
  void insertSuccessor(HeadNode &h);

  //! Returns the number of positive and negative atoms in the body.
  /*! This number corresponds to the number of zero- and one-predecessors of the 
   *  body node.*/
  size_type countBodyAtoms() const {
    return zeroPredecessors_.size() + onePredecessors_.size();
  }

  //! Returns the number of negative atoms in the body (corresponds to the number of one-predecessors).
  size_type countNegBodyAtoms() const {
    return onePredecessors_.size();
  }

  //! returns true if this node is a self-blocker.
  /*!
   * A body node is a self-blocker if one of its successors is also a
   * one-predecessors of this node.
   *
   * A self-blocker can never become positive.
   */
  bool isSelfBlocker() const {
    for(HeadNodeIterator it = successorsBegin(); it != successorsEnd(); ++it) {
      if(std::find(onePredecessorsBegin(), onePredecessorsEnd(), *it) 
        != onePredecessors_.end())
        return true;
    }
    return false;
  }

  
  //! Returns the color which preserves the recursive support used for the U-operator.
  /*! 
   *  \return Plus if the node is supported else weak_plus. 
   */
  virtual Color::ColorValue recursiveSupportColor() const {
    return trivialComponent() || isSupported() ? Color::plus : Color::weak_plus;
  }

  //! Returns the preferred choice point color of this node.
  /*! 
   *  \return Returns the recursive support state of the node as color. 
   *  \see recursiveSupportColor()*/
  virtual Color::ColorValue preferredChoicePointColor() const {
    return recursiveSupportColor();
  }

  //! Returns the alternative choice point color of this node.
  /*!
   *  \return Always minus. 
   */
  virtual Color::ColorValue alternativeChoicePointColor() const {
    return Color::minus;
  }

  virtual double getTightnessFactor(int gradient) const;
protected:
  //! Notifies predecessors of this node about the color change.
  /*! Calls for all predecessors the HeadNode::predecessorColored() method.
   *  \param oldColor The old color before coloring this node.
   *  \post All predecessors were informed about the coloring of this node. */
  virtual void doSetColor(Color::ColorValue oldColor);

  //! Notifies predecessors of this node about the restoring of a color.
  /*! Calls for all predecessors the HeadNode::predecessorRestored() method.
   *  \param oldColor The old color before restoring this node.
   *  \post All predecessors were informed about the color change of this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor);

  //! Fires an event of type \ref BodyNodeColored. 
  /*! \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor);

  //! fires an event of type \ref BodyNodeColoredChoicePoint
  /*! \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, ColorOpType::Value type);
private:
  //! Supported counter initialized by the number of uncolored zero predecessors.
  /*! A node is supported if this counter is zero. All zero predecessors are colored
   *  \ref Color::plus "plus". */
  long supported_;

  //! Weak supported counter initialized by the number of uncolored zero predecessors.
  /*! A node is weak supported if this counter is zero. All zero predecessors are
   *  colored \ref Color::plus "plus" or \ref Color::weak_plus "weak_plus". */
  long weakSupported_;

  //! Unsupported counter initialized by zero.
  /*! A node is unsupported if this counter is less than zero. At least one zero
   *  predecessors is colored \ref Color::minus "minus". */
  long unsupported_;

  //! Blocked counter initialized by zero.
  /*! A node is blocked if this counter is less than zero. At least one one
   *  predecessors is colored \ref Color::plus "plus". */
  long blocked_;

  //! Unblocked counter initialized by the number of uncolored one predecessors.
  /*! A node is unblocked if this counter is zero. All one predecessors are colored
   *  \ref Color::minus "minus". */
  long unblocked_;

  //! Set of zero predecessors of this node.
  HeadNodeSet zeroPredecessors_;

  //! Set of one predecessors of this node.
  HeadNodeSet onePredecessors_;

  //! Set of successors of this node.
  HeadNodeSet successors_;
};

//! Event type used to transfer information about the fact that a body node was colored.
/*! \see BodyNode::fireColorEvent */
struct BodyNodeColored {
	BodyNodeColored(BodyNode* h, Color::ColorValue oldColor)
		: node_(h)
		, oldColor_(oldColor) {
	}
	BodyNode*					node_;
	Color::ColorValue	oldColor_;
};

//! Event type used to transfer information about the fact that a Body node was colored as choice point
/*! \see BodyNode::fireColorChoicePointEvent */
struct BodyNodeColoredChoicePoint {
	BodyNodeColoredChoicePoint(BodyNode* h, Color::ColorValue oldColor, ColorOpType::Value  type)
		: node_(h)
		, oldColor_(oldColor)
    , type_(type) {
	}

	BodyNode*					node_;
	Color::ColorValue oldColor_;
  ColorOpType::Value  type_;      /*!< The type of the coloring */
};


} // NS_NOMORE

#endif
